/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// cl_tent.c -- client side temporary entities

#include "quakedef.h"
#include "gl_decals.h"


int			num_temp_entities;
entity_t	cl_temp_entities[MAX_TEMP_ENTITIES];
beam_t		cl_beams[MAX_BEAMS];

sfx_t			*cl_sfx_wizhit;
sfx_t			*cl_sfx_knighthit;
sfx_t			*cl_sfx_tink1;
sfx_t			*cl_sfx_ric1;
sfx_t			*cl_sfx_ric2;
sfx_t			*cl_sfx_ric3;
sfx_t			*cl_sfx_r_exp3;

/*
=================
CL_ParseTEnt
=================
*/
void CL_InitTEnts (void)
{
	cl_sfx_wizhit = S_PrecacheSound ("wizard/hit.wav");
	cl_sfx_knighthit = S_PrecacheSound ("hknight/hit.wav");
	cl_sfx_tink1 = S_PrecacheSound ("weapons/tink1.wav");
	cl_sfx_ric1 = S_PrecacheSound ("weapons/ric1.wav");
	cl_sfx_ric2 = S_PrecacheSound ("weapons/ric2.wav");
	cl_sfx_ric3 = S_PrecacheSound ("weapons/ric3.wav");
	cl_sfx_r_exp3 = S_PrecacheSound ("weapons/r_exp3.wav");
}

/*
=================
CL_ParseBeam
=================
*/
void CL_ParseBeam (model_t *m)
{
	int		ent;
	vec3_t	start, end;
	beam_t	*b;
	int		i;
	dlight_t	*dl;
	
	ent = MSG_ReadShort ();
	
	start[0] = MSG_ReadCoord ();
	start[1] = MSG_ReadCoord ();
	start[2] = MSG_ReadCoord ();
	
	end[0] = MSG_ReadCoord ();
	end[1] = MSG_ReadCoord ();
	end[2] = MSG_ReadCoord ();

	dl = CL_AllocDlight (0);
	VectorCopy (start, dl->origin);
	dl->radius = 200;
	dl->die = cl.time + 0.1;
	dl->decay = 300;
	dl->colour[0] = 0.1;
	dl->colour[1] = 0.1;
	dl->colour[2] = 0.5;

	dl = CL_AllocDlight (0);
	VectorCopy (end, dl->origin);
	dl->radius = 200;
	dl->die = cl.time + 0.1;
	dl->decay = 300;
	dl->colour[0] = 0.1;
	dl->colour[1] = 0.1;
	dl->colour[2] = 0.5;

	// override any beam with the same entity
	for (i=0, b=cl_beams ; i< MAX_BEAMS ; i++, b++)
		if (b->entity == ent)
		{
			b->entity = ent;
			b->model = m;
			b->endtime = cl.time + 0.2;
			VectorCopy (start, b->start);
			VectorCopy (end, b->end);
			return;
		}

	// find a free beam
	for (i=0, b=cl_beams ; i< MAX_BEAMS ; i++, b++)
	{
		if (!b->model || b->endtime < cl.time)
		{
			b->entity = ent;
			b->model = m;
			b->endtime = cl.time + 0.2;
			VectorCopy (start, b->start);
			VectorCopy (end, b->end);
			return;
		}
	}

	Con_DPrintf ("beam list overflow!\n");	
}

/*
=================
CL_ParseTEnt

Most of these occur inside the radius of a dlight, so there is no additional speed loss from 
having to rebuild the lightmap for stains.
=================
*/
void R_RunWizSpikeEffect (vec3_t org, vec3_t dir, int color, int count);
void R_RunKnightSpikeEffect (vec3_t org, vec3_t dir, int color, int count);

void CL_ParseTEnt (void)
{
	int		type;
	vec3_t	pos;
	dlight_t	*dl;
	int		rnd;
	int		colorStart, colorLength;
	vec3_t staincolour;
	msurface_t *hit;

	DecalEffect_t dEff;

	type = MSG_ReadByte ();

	switch (type)
	{
	case TE_WIZSPIKE:			// spike hitting wall
		pos[0] = MSG_ReadCoord ();
		pos[1] = MSG_ReadCoord ();
		pos[2] = MSG_ReadCoord ();

		// quake c doesn't send directions in the server message so we get the normal of the surface
		// which was hit and use that instead.  if it hit nothing don't do the effect
		//if (!(hit = SurfContainsPoint (pos, 5.0))) return;

		//R_RunGunshotEffect2 (pos, hit->planeangles, 3, 63);
		R_RunGunshotEffect (pos, 20, 63);

		dl = CL_AllocDlight (0);
		VectorCopy (pos, dl->origin);
		dl->radius = 300;
		dl->die = cl.time + 0.1;
		dl->decay = 300;
		dl->colour[0] = 0.1;
		dl->colour[1] = 0.5;
		dl->colour[2] = 0.1;

		S_StartSound (-1, 0, cl_sfx_wizhit, pos, 1, 1);

		staincolour[0] = 0.01f;
		staincolour[1] = 0.98f;
		staincolour[2] = 0.01f;

		CL_AddStain (pos, staincolour, 15, false);

		break;
		
	case TE_KNIGHTSPIKE:			// spike hitting wall
		pos[0] = MSG_ReadCoord ();
		pos[1] = MSG_ReadCoord ();
		pos[2] = MSG_ReadCoord ();

		// quake c doesn't send directions in the server message so we get the normal of the surface
		// which was hit and use that instead.  if it hit nothing don't do the effect
		//if (!(hit = SurfContainsPoint (pos, 5.0))) return;

		//R_RunGunshotEffect2 (pos, hit->planeangles, 5, 107);
		R_RunGunshotEffect (pos, 20, 107);

		dl = CL_AllocDlight (0);
		VectorCopy (pos, dl->origin);
		dl->radius = 300;
		dl->die = cl.time + 0.1;
		dl->decay = 300;

		S_StartSound (-1, 0, cl_sfx_knighthit, pos, 1, 1);

		staincolour[0] = 0.98f;
		staincolour[1] = 0.51f;
		staincolour[2] = 0.01f;

		CL_AddStain (pos, staincolour, 15, true);

		break;
		
	case TE_SPIKE:			// spike hitting wall
		pos[0] = MSG_ReadCoord ();
		pos[1] = MSG_ReadCoord ();
		pos[2] = MSG_ReadCoord ();

		// quake c doesn't send directions in the server message so we get the normal of the surface
		// which was hit and use that instead.  if it hit nothing don't do the effect
		//if (!(hit = SurfContainsPoint (pos, 5.0))) return;

		//R_RunGunshotEffect2 (pos, hit->planeangles, 2, 107);
		R_RunGunshotEffect (pos, 2, 107);

		if ( rand() % 5 )
			S_StartSound (-1, 0, cl_sfx_tink1, pos, 1, 1);
		else
		{
			rnd = rand() & 3;
			if (rnd == 1)
				S_StartSound (-1, 0, cl_sfx_ric1, pos, 1, 1);
			else if (rnd == 2)
				S_StartSound (-1, 0, cl_sfx_ric2, pos, 1, 1);
			else
				S_StartSound (-1, 0, cl_sfx_ric3, pos, 1, 1);
		}

		staincolour[0] = 0.11f;
		staincolour[1] = 0.11f;
		staincolour[2] = 0.11f;

		CL_AddStain (pos, staincolour, 10, false);

		break;

	case TE_SUPERSPIKE:			// super spike hitting wall
		pos[0] = MSG_ReadCoord ();
		pos[1] = MSG_ReadCoord ();
		pos[2] = MSG_ReadCoord ();

		// quake c doesn't send directions in the server message so we get the normal of the surface
		// which was hit and use that instead.  if it hit nothing don't do the effect
		//if (!(hit = SurfContainsPoint (pos, 5.0))) return;

		//R_RunGunshotEffect2 (pos, hit->planeangles, 4, 107);
		R_RunGunshotEffect (pos, 4, 107);

		if ( rand() % 5 )
			S_StartSound (-1, 0, cl_sfx_tink1, pos, 1, 1);
		else
		{
			rnd = rand() & 3;
			if (rnd == 1)
				S_StartSound (-1, 0, cl_sfx_ric1, pos, 1, 1);
			else if (rnd == 2)
				S_StartSound (-1, 0, cl_sfx_ric2, pos, 1, 1);
			else
				S_StartSound (-1, 0, cl_sfx_ric3, pos, 1, 1);
		}

		staincolour[0] = 0.11f;
		staincolour[1] = 0.11f;
		staincolour[2] = 0.11f;

		CL_AddStain (pos, staincolour, 20, false);

		break;
		
	case TE_GUNSHOT:			// bullet hitting wall
		// QC spawns up to 14 of these ents per gunshot!!!
		pos[0] = MSG_ReadCoord ();
		pos[1] = MSG_ReadCoord ();
		pos[2] = MSG_ReadCoord ();

		// quake c doesn't send directions in the server message so we get the normal of the surface
		// which was hit and use that instead.  if it hit nothing don't do the effect
		//if (!(hit = SurfContainsPoint (pos, 5.0))) return;

		//R_RunGunshotEffect2 (pos, hit->planeangles, 4, 107);
		R_RunGunshotEffect (pos, 10, 107);

		staincolour[0] = 0.11f;
		staincolour[1] = 0.11f;
		staincolour[2] = 0.11f;

		CL_AddStain (pos, staincolour, 5 + rand () & 10, false);
		//R_DefaultDecal (&dEff);
		//R_SpawnDecalEffect (pos, &dEff);

		break;
		
	case TE_EXPLOSION:			// rocket explosion
		pos[0] = MSG_ReadCoord ();
		pos[1] = MSG_ReadCoord ();
		pos[2] = MSG_ReadCoord ();

		// increase the fudge for these
		// is it valid?  explosions could happen in mid air...
		//if (!(hit = SurfContainsPoint (pos, 20.0))) return;

		R_ParticleExplosion (pos);

		dl = CL_AllocDlight (0);
		VectorCopy (pos, dl->origin);
		dl->radius = 350;
		dl->die = cl.time + 0.5;
		dl->decay = 300;

		S_StartSound (-1, 0, cl_sfx_r_exp3, pos, 1, 1);

		staincolour[0] = 0.31f;
		staincolour[1] = 0.31f;
		staincolour[2] = 0.31f;

		CL_AddStain (pos, staincolour, 30 + rand () & 30, false);

		break;
		
	case TE_TAREXPLOSION:			// tarbaby explosion
		pos[0] = MSG_ReadCoord ();
		pos[1] = MSG_ReadCoord ();
		pos[2] = MSG_ReadCoord ();

		// don't bother sending this through SurfContainsPoint cos it always comes from an entity
		R_BlobExplosion (pos);

		dl = CL_AllocDlight (0);
		VectorCopy (pos, dl->origin);
		dl->radius = 350;
		dl->die = cl.time + 0.5;
		dl->decay = 300;
		dl->colour[0] = 0.5;
		dl->colour[1] = 0.1;
		dl->colour[2] = 0.5;

		S_StartSound (-1, 0, cl_sfx_r_exp3, pos, 1, 1);

		staincolour[0] = 0.51f;
		staincolour[1] = 0.01f;
		staincolour[2] = 0.51f;

		CL_AddStain (pos, staincolour, 30 + rand () & 30, false);

		break;

	case TE_LIGHTNING1:				// lightning bolts
		CL_ParseBeam (Mod_ForName("progs/bolt.mdl", true));
		break;
	
	case TE_LIGHTNING2:				// lightning bolts
		CL_ParseBeam (Mod_ForName("progs/bolt2.mdl", true));
		break;
	
	case TE_LIGHTNING3:				// lightning bolts
		CL_ParseBeam (Mod_ForName("progs/bolt3.mdl", true));
		break;
	
// PGM 01/21/97 
	case TE_BEAM:				// grappling hook beam
		CL_ParseBeam (Mod_ForName("progs/beam.mdl", true));
		break;
// PGM 01/21/97

	case TE_LAVASPLASH:	
		pos[0] = MSG_ReadCoord ();
		pos[1] = MSG_ReadCoord ();
		pos[2] = MSG_ReadCoord ();

		dl = CL_AllocDlight (0);
		VectorCopy (pos, dl->origin);
		dl->radius = 500;
		dl->die = cl.time + 0.5;
		dl->decay = 300;
		dl->colour[0] = 0.5;
		dl->colour[1] = 0.1;
		dl->colour[2] = 0.1;

		R_LavaSplash (pos);
		break;

	case TE_TELEPORT:
		pos[0] = MSG_ReadCoord ();
		pos[1] = MSG_ReadCoord ();
		pos[2] = MSG_ReadCoord ();

		R_TeleportSplash (pos);
		break;

	case TE_EXPLOSION2:				// color mapped explosion
		pos[0] = MSG_ReadCoord ();
		pos[1] = MSG_ReadCoord ();
		pos[2] = MSG_ReadCoord ();
		colorStart = MSG_ReadByte ();
		colorLength = MSG_ReadByte ();

		// increase the fudge
		//if (!(hit = SurfContainsPoint (pos, 20.0))) return;

		R_ParticleExplosion2 (pos, colorStart, colorLength);
		dl = CL_AllocDlight (0);
		VectorCopy (pos, dl->origin);

		// we'll leave these as white for now, but will do it proper later
		dl->colour[0] = 0.5;
		dl->colour[1] = 0.5;
		dl->colour[2] = 0.5;

		dl->radius = 350;
		dl->die = cl.time + 0.5;
		dl->decay = 300;

		S_StartSound (-1, 0, cl_sfx_r_exp3, pos, 1, 1);

		staincolour[0] = 0.31f;
		staincolour[1] = 0.31f;
		staincolour[2] = 0.31f;

		CL_AddStain (pos, staincolour, 30 + rand () & 30, false);

		break;
		
	default:
		// do nothing
		return;
		//Sys_Error ("CL_ParseTEnt: bad type");
	}
}


/*
=================
CL_NewTempEntity
=================
*/
entity_t *CL_NewTempEntity (void)
{
	entity_t	*ent;

	if (cl_numvisedicts == MAX_VISEDICTS)
		return NULL;
	if (num_temp_entities == MAX_TEMP_ENTITIES)
		return NULL;
	ent = &cl_temp_entities[num_temp_entities];
	memset (ent, 0, sizeof(*ent));
	num_temp_entities++;
	cl_visedicts[cl_numvisedicts] = ent;
	cl_numvisedicts++;

	ent->colormap = vid.colormap;
	return ent;
}


/*
=================
CL_UpdateTEnts
=================
*/
void CL_UpdateTEnts (void)
{
	int			i;
	beam_t		*b;
	vec3_t		dist, org;
	float		d;
	entity_t	*ent;
	float		yaw, pitch;
	float		forward;

	num_temp_entities = 0;

// update lightning
	for (i=0, b=cl_beams ; i< MAX_BEAMS ; i++, b++)
	{
		if (!b->model || b->endtime < cl.time)
			continue;

	// if coming from the player, update the start position
		if (b->entity == cl.viewentity)
		{
			VectorCopy (cl_entities[cl.viewentity].origin, b->start);
		}

	// calculate pitch and yaw
		VectorSubtract (b->end, b->start, dist);

		if (dist[1] == 0 && dist[0] == 0)
		{
			yaw = 0;
			if (dist[2] > 0)
				pitch = 90;
			else
				pitch = 270;
		}
		else
		{
			yaw = (int) (atan2(dist[1], dist[0]) * 180 / M_PI);
			if (yaw < 0)
				yaw += 360;
	
			forward = sqrt (dist[0]*dist[0] + dist[1]*dist[1]);
			pitch = (int) (atan2(dist[2], forward) * 180 / M_PI);
			if (pitch < 0)
				pitch += 360;
		}

	// add new entities for the lightning
		VectorCopy (b->start, org);
		d = VectorNormalize(dist);
		while (d > 0)
		{
			ent = CL_NewTempEntity ();
			if (!ent)
				return;
			VectorCopy (org, ent->origin);
			ent->model = b->model;
			ent->angles[0] = pitch;
			ent->angles[1] = yaw;
			ent->angles[2] = rand()%360;

			for (i=0 ; i<3 ; i++)
				org[i] += dist[i]*30;
			d -= 30;
		}
	}
	
}


