/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// draw.c -- this is the only file outside the refresh that touches the
// vid buffer

#include "quakedef.h"
#include "glext.h"
#include "resource.h"

cvar_t		gl_nobind = {"gl_nobind", "0"};
cvar_t		gl_picmip = {"gl_picmip", "0"};

qpic_t		*draw_disc;
qpic_t		*draw_backtile;

int			translate_texture;
int			char_texture = 0;

byte		conback_buffer[sizeof(qpic_t) + sizeof(glpic_t)];
qpic_t		*conback = (qpic_t *)&conback_buffer;

int logoTex = -1;
int particlefont = -1;
int crosshairfont = -1;


int		texels;


// gl_draw.c needs these prototyped
image_t *GL_LoadTexture (char *identifier, int width, int height, byte *data, qboolean mipmap, qboolean alpha);
image_t *GL_LoadPicTexture (qpic_t *pic, char *picname);
void GL_Upload8 (byte *data, int width, int height, qboolean mipmap, qboolean alpha);
void GL_Upload32 (unsigned *data, int width, int height, qboolean mipmap, qboolean alpha);
void GL_ResampleTexture (unsigned *texIn, int inwidth, int inheight, unsigned *texOut, int outwidth, int outheight);
void WriteTGA (char *name, int width, int height, int bpp, byte *data);


/*
=============================================================================

  scrap allocation

  Allocate all the little status bar obejcts into a single texture
  to crutch up stupid hardware / drivers

=============================================================================
*/

// increased from 2 scraps in the event of more complex game mods
#define	MAX_SCRAPS		4
#define	SCRAP_WIDTH		256
#define	SCRAP_HEIGHT	256

int			scrap_allocated[MAX_SCRAPS][SCRAP_WIDTH];
byte		scrap_texels[MAX_SCRAPS][SCRAP_WIDTH*SCRAP_HEIGHT*4];
qboolean	scrap_dirty;
int			scrap_texnum;


// get rid of that stupid warning
#pragma warning (disable : 4715)


// returns a texture number and the position inside it
int Scrap_AllocBlock (int w, int h, int *x, int *y)
{
	int		i, j;
	int		best, best2;
	int		bestx;
	int		texnum;

	for (texnum=0 ; texnum<MAX_SCRAPS ; texnum++)
	{
		best = SCRAP_HEIGHT;

		for (i=0 ; i<SCRAP_WIDTH-w ; i++)
		{
			best2 = 0;

			for (j=0 ; j<w ; j++)
			{
				if (scrap_allocated[texnum][i+j] >= best)
					break;
				if (scrap_allocated[texnum][i+j] > best2)
					best2 = scrap_allocated[texnum][i+j];
			}
			if (j == w)
			{	// this is a valid spot
				*x = i;
				*y = best = best2;
			}
		}

		if (best + h > SCRAP_HEIGHT)
			continue;

		for (i=0 ; i<w ; i++)
			scrap_allocated[texnum][*x + i] = best + h;

		return texnum;
	}

	Sys_Error ("Scrap_AllocBlock: full");
}


void Scrap_Upload (void)
{
	int		texnum;

	for (texnum = 0; texnum < MAX_SCRAPS; texnum++)
	{
		glBindTexture (GL_TEXTURE_2D, scrap_texnum + texnum);
		GL_Upload8 (scrap_texels[texnum], SCRAP_WIDTH, SCRAP_HEIGHT, false, true);
	}

	scrap_dirty = false;
}

//=============================================================================
/* Support Routines */

typedef struct cachepic_s
{
	char		name[MAX_QPATH];
	qpic_t		pic;
	byte		padding[32];	// for appended glpic
} cachepic_t;

#define	MAX_CACHED_PICS		128
cachepic_t	menu_cachepics[MAX_CACHED_PICS];
int			menu_numcachepics;

byte		menuplyr_pixels[4096];

int		pic_texels;
int		pic_count;


qpic_t *Draw_PicFromWad (char *name)
{
	qpic_t	*p;
	glpic_t	*gl;

	int w;
	int h;

	p = W_GetLumpName (name);

	// filthy hack to reuse the same memory instead of allocating more...
	gl = (glpic_t *)p->data;

	// load little ones into the scrap
	if (p->width < 64 && p->height < 64)
	{
		int		x, y;
		int		i, j, k;
		int		texnum;

		texnum = Scrap_AllocBlock (p->width, p->height, &x, &y);

		scrap_dirty = true;

		for (i = 0, k = 0; i < p->height; i++)
		{
			for (j = 0; j < p->width; j++, k++)
			{
				scrap_texels[texnum][(y + i) * SCRAP_WIDTH + x + j] = p->data[k];
			}
		}

		texnum += scrap_texnum;
		gl->texnum = texnum;

		gl->sl = (x + 0.01) / (float) SCRAP_WIDTH;
		gl->sh = (x + p->width - 0.01) / (float) SCRAP_WIDTH;
		gl->tl = (y + 0.01) / (float) SCRAP_WIDTH;
		gl->th = (y + p->height - 0.01) / (float) SCRAP_HEIGHT; // was: SCRAP_WIDTH;

		pic_count++;
		pic_texels += p->width * p->height;
	}
	else
	{
		// if it's not a power of 2 pad it with alpha to make it one
		for (w = 1; w < p->width; w *= 2);
		for (h = 1; h < p->height; h *= 2);

		if (p->width >= 512 || p->height >= 512)
		{
			gl->texnum = (GL_LoadTexture (name, p->width, p->height, p->data, false, true))->texnum;

			gl->sl = 0;
			gl->sh = 1;
			gl->tl = 0;
			gl->th = 1;
		}
		else if (w == p->width && h == p->height)
		{
			// texture is a power of 2 - cool
			gl->texnum = (GL_LoadPicTexture (p, name))->texnum;

			gl->sl = 0;
			gl->sh = 1;
			gl->tl = 0;
			gl->th = 1;
		}
		else
		{
			// if the picture isn't a power of 2, pad it with alpha pixels to make it one...
			byte picTemp1[512][512];
			byte picTemp2[512 * 512];

			int i, j, k, l;

			// initialize to all alpha (do all this fast instead of sending through memset 
			// and/or memcpy cos it can happen at runtime)
			for (i = 0, k = 0; i < 512; i++)
			{
				for (j = 0; j < 512; j++, k++)
				{
					picTemp1[i][j] = 255;
					picTemp2[k] = 255;
				}
			}

			for (i = 0, k = 0; i < p->height; i++)
			{
				for (j = 0; j < p->width; j++, k++)
				{
					picTemp1[i][j] = p->data[k];
				}
			}

			for (i = 0, k = 0; i < h; i++)
			{
				for (j = 0; j < w; j++, k++)
				{
					picTemp2[k] = picTemp1[i][j];
				}
			}

			gl->texnum = (GL_LoadTexture (name, w, h, picTemp2, false, true))->texnum;

			// scale the tex-coords to the new size
			// keep the old sizes in the pic->pic struct otherwise we'll interfere with
			// glVertex2f - we only wanna interfere with glTexCoord2f here!!!
			gl->sl = 0;
			gl->sh = 1.0 * (float) p->width / (float) w;
			gl->tl = 0;
			gl->th = 1.0 * (float) p->height / (float) h;
		}
	}

	return p;
}


void R_ClearMenuCachePics (void)
{
	cachepic_t	*pic;
	int			i;

	for (pic = menu_cachepics, i = 0; i < MAX_CACHED_PICS; pic++, i++)
	{
		// clear the name
		pic->name[0] = '\0';
	}

	// set to no pics cached
	menu_numcachepics = 0;
}


/*
================
Draw_CachePic
================
*/
qpic_t *Draw_CachePic (char *path)
{
	cachepic_t	*pic;
	int			i;
	qpic_t		*dat;
	glpic_t		*gl;

	int w;
	int h;

	for (pic = menu_cachepics, i = 0; i < menu_numcachepics; pic++, i++)
		if (!strcmp (path, pic->name))
			return &pic->pic;

	if (menu_numcachepics == MAX_CACHED_PICS)
		Sys_Error ("menu_numcachepics == MAX_CACHED_PICS");

	menu_numcachepics++;
	strcpy (pic->name, path);

	// load the pic from disk
	dat = (qpic_t *) COM_LoadTempFile (path);
	if (!dat) Sys_Error ("Draw_CachePic: failed to load %s", path);

	SwapPic (dat);

	// HACK HACK HACK --- we need to keep the bytes for
	// the translatable player picture just for the menu
	// configuration dialog
	// strcmp at runtime!!!  tut tut mr. carmack...
	// (admittedly it's in a menu so it doesn't really matter...)
	if (!strcmp (path, "gfx/menuplyr.lmp"))
		memcpy (menuplyr_pixels, dat->data, dat->width * dat->height);

	pic->pic.width = dat->width;
	pic->pic.height = dat->height;

	gl = (glpic_t *) pic->pic.data;

	for (w = 1; w < dat->width; w *= 2);
	for (h = 1; h < dat->height; h *= 2);

	if (dat->width >= 512 || dat->height >= 512)
	{
		// too big, so send it straight through GL_LoadTexture
		gl->texnum = (GL_LoadTexture (path, dat->width, dat->height, dat->data, false, true))->texnum;

		gl->sl = 0;
		gl->sh = 1;
		gl->tl = 0;
		gl->th = 1;
	}
	else if (w == dat->width && h == dat->height)
	{
		gl->texnum = (GL_LoadPicTexture (dat, path))->texnum;

		gl->sl = 0.12;
		gl->sh = 0.88;
		gl->tl = 0.12;
		gl->th = 0.88;
	}
	else
	{
		// if the picture isn't a power of 2, pad it with alpha pixels to make it one...
		// pictures are interfered with enough by scaling to the appropriate resolution
		// without any pain caused by going through GL_ResampleTexture as well!!!
		// accuracy is more important with the 2D stuff here
		byte picTemp1[512][512];
		byte picTemp2[512 * 512];

		int i, j, k, l;

		// initialize to all alpha (do all this fast instead of sending through memset 
		// and/or memcpy cos it can happen at runtime)
		for (i = 0, k = 0; i < 512; i++)
		{
			for (j = 0; j < 512; j++, k++)
			{
				picTemp1[i][j] = 255;
				picTemp2[k] = 255;
			}
		}

		for (i = 0, k = 0; i < dat->height; i++)
		{
			for (j = 0; j < dat->width; j++, k++)
			{
				picTemp1[i][j] = dat->data[k];
			}
		}

		for (i = 0, k = 0; i < h; i++)
		{
			for (j = 0; j < w; j++, k++)
			{
				picTemp2[k] = picTemp1[i][j];
			}
		}

		gl->texnum = (GL_LoadTexture (path, w, h, picTemp2, false, true))->texnum;

		// scale the tex-coords to the new size
		// keep the old sizes in the pic->pic struct otherwise we'll interfere with
		// glVertex2f - we only wanna interfere with glTexCoord2f here!!!
		gl->sl = 0;
		gl->sh = 1.0 * (float) dat->width / (float) w;
		gl->tl = 0;
		gl->th = 1.0 * (float) dat->height / (float) h;
	}

	return &pic->pic;
}


/*
=============
DrawParticleFont

Draw the font to the screen for testing purposes.  Not used.
=============
*/
void DrawParticleFont (void)
{
	glDisable (GL_ALPHA_TEST);
	glEnable (GL_BLEND);

	glBindTexture (GL_TEXTURE_2D, particlefont);

	glBegin (GL_QUADS);

	glTexCoord2f (0, 0);
	glVertex2f (0, 0);

	glTexCoord2f (1, 0);
	glVertex2f (384, 0);

	glTexCoord2f (1, 1);
	glVertex2f (384, 384);

	glTexCoord2f (0, 1);
	glVertex2f (0, 384);

	glEnd ();

	glDisable (GL_BLEND);
	glEnable (GL_ALPHA_TEST);
}


void GL_LoadCharset (void)
{
	int		i, j, k, l;
	byte *draw_chars;
	byte temp[256][256];

	draw_chars = W_GetLumpName ("conchars");

	for (i = 0; i < 256 * 64; i++)
		if (draw_chars[i] == 0)
			draw_chars[i] = 255;	// proper transparent color

	// resample it up to 256 * 256
	for (i = 0, k = 0; i < 128; i++)
	{
		for (j = 0; j < 128; j++, k++)
		{
			temp[i * 2][j * 2] = draw_chars[k];
			temp[i * 2 + 1][j * 2] = draw_chars[k];
			temp[i * 2 + 1][j * 2 + 1] = draw_chars[k];
			temp[i * 2][j * 2 + 1] = draw_chars[k];
		}
	}

	if (!char_texture)
	{
		char_texture = texture_extension_number++;
	}

	// now turn them into textures
	glBindTexture (GL_TEXTURE_2D, char_texture);

	GL_Upload8 ((byte *) temp, 256, 256, false, true);
}


void GL_TexmanInit (void);
void GL_LoadMapTexture (texture_t *tx);
extern texture_t *r_white_texture;

/*
===============
Draw_Init
===============
*/
void Draw_Init (void)
{
	GL_TexmanInit ();

	// lock 2D stuff at 640 x 480
	vid.conwidth = vid.width = 640;
	vid.conheight = vid.height = 480;

	Cvar_RegisterVariable (&gl_nobind);
	Cvar_RegisterVariable (&gl_picmip);

	GL_LoadCharset ();

	// save a texture slot for translated picture
	translate_texture = texture_extension_number++;

	// save slots for scraps
	scrap_texnum = texture_extension_number;
	texture_extension_number += MAX_SCRAPS;

	// get the other pics we need
	draw_disc = Draw_PicFromWad ("disc");
	draw_backtile = Draw_PicFromWad ("backtile");

	LoadResourceTextures ();

	// set this up as a proper texture
	GL_LoadMapTexture (r_notexture_mip);
	GL_LoadMapTexture (r_white_texture);
}



/*
================
Draw_Character

Draws one 8*8 graphics character with 0 being transparent.
It can be clipped to the top of the screen to allow the console to be
smoothly scrolled off.

only the deathmatch overlays go through this
================
*/
void Draw_Character (int x, int y, int num)
{
	int				row, col;
	float			frow, fcol, size;

	// space
	if (num == 32) return;

	num &= 255;

	// totally off screen
	if (y <= -8) return;

	row = num >> 4;
	col = num & 15;

	frow = row * 0.0625;
	fcol = col * 0.0625;
	size = 0.0625;

	glBindTexture (GL_TEXTURE_2D, char_texture);

	glBegin (GL_QUADS);
	glTexCoord2f (fcol, frow);
	glVertex2f (x, y);
	glTexCoord2f (fcol + size, frow);
	glVertex2f (x + 8, y);
	glTexCoord2f (fcol + size, frow + size);
	glVertex2f (x + 8, y + 8);
	glTexCoord2f (fcol, frow + size);
	glVertex2f (x, y + 8);
	glEnd ();
}


void Draw_SingleCharacter (int x, int y, int num)
{
	int				row, col;
	float			frow, fcol, size;

	// don't draw spaces
	if (num == 32) return;

	// 0 to 255 range
	num &= 255;

	// totally off screen
	if (y <= -8) return;

	row = num >> 4;
	col = num & 15;

	frow = row * 0.0625;
	fcol = col * 0.0625;
	size = 0.0625;

	glTexCoord2f (fcol, frow);
	glVertex2f (x, y);
	glTexCoord2f (fcol + size, frow);
	glVertex2f (x + 8, y);
	glTexCoord2f (fcol + size, frow + size);
	glVertex2f (x + 8, y + 8);
	glTexCoord2f (fcol, frow + size);
	glVertex2f (x, y + 8);
}


/*
================
Draw_String
================
*/
void Draw_String (int x, int y, char *str)
{
	glBindTexture (GL_TEXTURE_2D, char_texture);

	glBegin (GL_QUADS);

	while (*str)
	{
		Draw_SingleCharacter (x, y, *str);
		str++;
		x += 8;
	}

	glEnd ();
}


void Draw_ColourString (int x, int y, char *str, byte r, byte g, byte b)
{
	glColor3ub (r, g, b);

	Draw_String (x, y, str);

	glColor3f (1, 1, 1);
}


/*
================
Draw_DebugChar

Draws a single character directly to the upper right corner of the screen.
This is for debugging lockups by drawing different chars in different parts
of the code.
================
*/
void Draw_DebugChar (char num)
{
}

/*
=============
Draw_AlphaPic
=============
*/
void Draw_AlphaPic (int x, int y, qpic_t *pic, float alpha)
{
	byte			*dest, *source;
	unsigned short	*pusdest;
	int				v, u;
	glpic_t			*gl;

	if (scrap_dirty)
		Scrap_Upload ();

	gl = (glpic_t *)pic->data;
	glDisable(GL_ALPHA_TEST);
	glEnable (GL_BLEND);
//	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
//	glCullFace(GL_FRONT);
	glColor4f (1,1,1,alpha);
	glBindTexture (GL_TEXTURE_2D, gl->texnum);
	glBegin (GL_QUADS);
	glTexCoord2f (gl->sl, gl->tl);
	glVertex2f (x, y);
	glTexCoord2f (gl->sh, gl->tl);
	glVertex2f (x+pic->width, y);
	glTexCoord2f (gl->sh, gl->th);
	glVertex2f (x+pic->width, y+pic->height);
	glTexCoord2f (gl->sl, gl->th);
	glVertex2f (x, y+pic->height);
	glEnd ();
	glColor4f (1,1,1,1);
	glEnable(GL_ALPHA_TEST);
	glDisable (GL_BLEND);
}


/*
=============
Draw_Pic
=============
*/
void Draw_Pic (int x, int y, qpic_t *pic)
{
	byte			*dest, *source;
	unsigned short	*pusdest;
	int				v, u;
	glpic_t			*gl;

	if (!pic) return;

	if (scrap_dirty)
		Scrap_Upload ();

	gl = (glpic_t *)pic->data;
	glColor4f (1,1,1,1);
	glBindTexture (GL_TEXTURE_2D, gl->texnum);
	glBegin (GL_QUADS);
	glTexCoord2f (gl->sl, gl->tl);
	glVertex2f (x, y);
	glTexCoord2f (gl->sh, gl->tl);
	glVertex2f (x+pic->width, y);
	glTexCoord2f (gl->sh, gl->th);
	glVertex2f (x+pic->width, y+pic->height);
	glTexCoord2f (gl->sl, gl->th);
	glVertex2f (x, y+pic->height);
	glEnd ();
}


/*
=============
Draw_TransPic
=============
*/
void Draw_TransPic (int x, int y, qpic_t *pic)
{
	if (x < 0 || (unsigned)(x + pic->width) > vid.width || y < 0 || (unsigned)(y + pic->height) > vid.height)
	{
		// Sys_Error ("Draw_TransPic: bad coordinates");
		// just don't bother drawing it...
		return;
	}

	Draw_Pic (x, y, pic);
}


void Draw_TransPicAlpha (int x, int y, qpic_t *pic, float alpha)
{
	if (x < 0 || (unsigned)(x + pic->width) > vid.width || y < 0 || (unsigned)(y + pic->height) > vid.height)
	{
		// Sys_Error ("Draw_TransPic: bad coordinates");
		// just don't bother drawing it...
		return;
	}

	Draw_AlphaPic (x, y, pic, alpha);
}


/*
================
Draw_ConsoleBackground

MH - modified to a rotating quad damage model on full-screen consoles, half-screen
consoles are just a shaded band with a bottom border
================
*/
void R_DrawAliasModelSBar (entity_t *e, float Light, int skin, float trans, float scale);
model_t *conQuad = NULL;


void doLogo (int left, int top)
{
	glBindTexture (GL_TEXTURE_2D, logoTex);

	glBegin (GL_QUADS);

	glTexCoord2f (0, 0);
	glVertex2f (left, top);

	glTexCoord2f (0, 1);
	glVertex2f (left, top + 64);

	glTexCoord2f (1, 1);
	glVertex2f (left + 256, top + 64);

	glTexCoord2f (1, 0);
	glVertex2f (left + 256, top);

	glEnd ();
}


void Draw_ConsoleBackground (int lines)
{
	int y = (vid.height * 3) >> 2;
	float conlines;

	entity_t e;

	float bobjrotate = anglemod (100 * Sys_FloatTime ());
	float conwidth = vid.width;

	qpic_t *pic = Draw_CachePic ("gfx/box_bm.lmp");
	glpic_t *gl;

	float quadorg[3];
	float quadang[3];

	int quadframe;

	if (scr_con_current != vid.height)
	{
		// half screen
		conlines = vid.height / 2;

		// background tile
		glDisable (GL_ALPHA_TEST);
		glEnable (GL_BLEND);
		glDisable (GL_TEXTURE_2D);
		glColor4ub (0, 0, 0, 128);

		glBegin (GL_QUADS);

		glVertex2f (0, 0);
		glVertex2f (0, conlines);
		glVertex2f (conwidth, conlines);
		glVertex2f (conwidth, 0);

		glEnd ();

		glEnable (GL_TEXTURE_2D);

		// now draw a bottom border
		gl = (glpic_t *) pic->data;

		glColor4ub (255, 255, 255, 128);

		glBindTexture (GL_TEXTURE_2D, gl->texnum);

		glBegin (GL_QUADS);

		glTexCoord2f (0, 0);
		glVertex2f (0, conlines);

		glTexCoord2f (0, 0.75);
		glVertex2f (0, conlines + 8);

		glTexCoord2f ((conwidth / 16), 0.75);
		glVertex2f (conwidth, conlines + 8);

		glTexCoord2f ((conwidth / 16), 0);
		glVertex2f (conwidth, conlines);

		glEnd ();

		// finally, do the MHQuake logo
		doLogo (conwidth - 240, conlines - 80);

		glColor4f (1, 1, 1, 1);
		glEnable (GL_ALPHA_TEST);
		glDisable (GL_BLEND);

		// no quad in game, it's too slow
		return;
	}

	// full screen
	conlines = vid.height;

	// attempt to load the quad model if necessary
	// the model used depends on the mod
	if (!conQuad)
	{
		/*if (hipnotic)
			conQuad = Mod_ForName ("progs/empathy.mdl", false);
		else if (rogue)
			conQuad = Mod_ForName ("progs/dragon.mdl", false);
		else */
			conQuad = Mod_ForName ("progs/quaddama.mdl", false);

		// do zerlogo.mdl for zer...
	}

	if (!conQuad) return;

	// fixme - hipnotic and rogue mdls are CRAP!!!
/*	if (hipnotic)
	{
		quadorg[0] = 50;
		quadorg[1] = 0;
		quadorg[2] = -41;

		quadang[0] = 0;
		quadang[1] = bobjrotate;
		quadang[2] = 0;

		quadframe = 0;
	}
	else if (rogue)
	{
		// best i can get at this time.  if anyone wants a go at positioning it better, do
		// feel free - i'd like to hear.
		quadorg[0] = 100;
		quadorg[1] = -25;
		quadorg[2] = -15;

		quadang[0] = 0;
		quadang[1] = 200;
		quadang[2] = 0;

		quadframe = 2 + bobjrotate / 30;
	}
	else */
	{
		quadorg[0] = 30;
		quadorg[1] = 0;
		quadorg[2] = -21;

		quadang[0] = 0;
		quadang[1] = bobjrotate;
		quadang[2] = 0;

		quadframe = 0;
	}

	// create a 3D viewport for the Quad - this is gonna fail on crap hardware - too bad
	glViewport (glx, gly, glwidth, glheight);

	// initialize our matrices
	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();

	// we don't really need a huge far clipping plane here
	gluPerspective (r_refdef.fov_y, (float) glwidth / (float) glheight, 4, 700);

	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity ();

	glEnable (GL_DEPTH_TEST);
	glClear (GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
	gldepthmin = 0;
	gldepthmax = 1;
	glDepthFunc (GL_LEQUAL);
	glDepthRange (gldepthmin, gldepthmax);

	// put Z going up and X going deep to match Quakes main render
	// Y remains at right to left, Z becomes top to bottom and X becomes near to far
	glRotatef (-90, 1, 0, 0);
	glRotatef (90, 0, 0, 1);

	// draw the quad
	memset (&e, 0, sizeof (e));
	e.model = conQuad;

	e.origin[0] = quadorg[0];	// always positive, higher numbers are deeper
	e.origin[1] = quadorg[1];	// positive = to the left, negative = to the right
	e.origin[2] = quadorg[2];	// negative = further down, positive = higher up

	e.angles[0] = quadang[0];
	e.angles[1] = quadang[1];
	e.angles[2] = quadang[2];

	e.frame = quadframe;

	glEnableClientState (GL_VERTEX_ARRAY);
	glVertexPointer (3, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[0]);

	glEnableClientState (GL_TEXTURE_COORD_ARRAY);
	glClientActiveTexture (GL_TEXTURE0);
	glTexCoordPointer (2, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[3]);

	glEnableClientState (GL_COLOR_ARRAY);
	glColorPointer (4, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[5]);

	R_DrawAliasModelSBar (&e, 0.5, 0, 1.0, 0.0);

	glDisableClientState (GL_VERTEX_ARRAY);
	glDisableClientState (GL_TEXTURE_COORD_ARRAY);
	glDisableClientState (GL_COLOR_ARRAY);

	// switch back to 2D for the console text
	glViewport (glx, gly, glwidth, glheight);

	glMatrixMode(GL_PROJECTION);
    glLoadIdentity ();
	glOrtho  (0, 640, 480, 0, -1, 1);

	glMatrixMode(GL_MODELVIEW);
    glLoadIdentity ();

	glDisable (GL_DEPTH_TEST);
	glDisable (GL_CULL_FACE);
	glDisable (GL_BLEND);
	glEnable (GL_ALPHA_TEST);

	glColor4f (1,1,1,1);

	// finally, do the MHQuake logo
	doLogo (conwidth - 240, conlines - 80);
}


/*
=============
Draw_TileClear

This repeats a 64*64 tile graphic to fill the screen around a sized down
refresh window.
=============
*/
void Draw_TileClear (int x, int y, int w, int h)
{
	glColor3f (1,1,1);
	glBindTexture (GL_TEXTURE_2D, *(int *)draw_backtile->data);
	glBegin (GL_QUADS);
	glTexCoord2f (x/64.0, y/64.0);
	glVertex2f (x, y);
	glTexCoord2f ( (x+w)/64.0, y/64.0);
	glVertex2f (x+w, y);
	glTexCoord2f ( (x+w)/64.0, (y+h)/64.0);
	glVertex2f (x+w, y+h);
	glTexCoord2f ( x/64.0, (y+h)/64.0 );
	glVertex2f (x, y+h);
	glEnd ();
}


/*
=============
Draw_Fill

Fills a box of pixels with a single color
=============
*/
void Draw_Fill (int x, int y, int w, int h, int c)
{
	glDisable (GL_TEXTURE_2D);
	glColor3f (host_basepal[c*3] * ONEOVER255,
		host_basepal[c*3+1] * ONEOVER255,
		host_basepal[c*3+2] * ONEOVER255);

	glBegin (GL_QUADS);

	glVertex2f (x,y);
	glVertex2f (x+w, y);
	glVertex2f (x+w, y+h);
	glVertex2f (x, y+h);

	glEnd ();
	glColor3f (1,1,1);
	glEnable (GL_TEXTURE_2D);
}
//=============================================================================

/*
================
Draw_FadeScreen

================
*/
void Draw_FadeScreen (void)
{
	// hexen 2 fade screen
	int bx,by,ex,ey;
	int c;

	// now do the flashy bits
	glAlphaFunc(GL_ALWAYS, 0);

	glEnable (GL_BLEND);
	glDisable (GL_TEXTURE_2D);

	glColor4f (208.0 * ONEOVER255, 180.0 * ONEOVER255, 80.0  * ONEOVER255, 0.2);

	glBegin (GL_QUADS);
	glVertex2f (0,0);
	glVertex2f (vid.width, 0);
	glVertex2f (vid.width, vid.height);
	glVertex2f (0, vid.height);
	glEnd ();

	glColor4f (208.0 * ONEOVER255, 180.0 * ONEOVER255, 80.0 * ONEOVER255, 0.035);

	for(c=0;c<40;c++)
	{
		bx = rand() % vid.width-20;
		by = rand() % vid.height-20;

		ex = bx + (rand() % 40) + 20;
		ey = by + (rand() % 40) + 20;

		if (bx < 0) bx = 0;
		if (by < 0) by = 0;

		if (ex > vid.width) ex = vid.width;
		if (ey > vid.height) ey = vid.height;

		glBegin (GL_QUADS);
		glVertex2f (bx,by);
		glVertex2f (ex, by);
		glVertex2f (ex, ey);
		glVertex2f (bx, ey);
		glEnd ();
	}

	glColor4f (1,1,1,1);
	glDisable (GL_BLEND);
	glEnable (GL_TEXTURE_2D);

	glAlphaFunc(GL_GREATER, 0.666);

	Sbar_Changed();
}

//=============================================================================

/*
================
Draw_BeginDisc

Draws the little blue disc in the corner of the screen.
Call before beginning any disc IO.
================
*/
void Draw_BeginDisc (void)
{
	if (!draw_disc)
		return;
	glDrawBuffer  (GL_FRONT);
	Draw_Pic (vid.width - 24, 0, draw_disc);
	glDrawBuffer  (GL_BACK);
}


/*
================
Draw_EndDisc

Erases the disc icon.
Call after completing any disc IO
================
*/
void Draw_EndDisc (void)
{
}

/*
================
GL_Set2D

Setup as if the screen was 640 x 480
================
*/

void GL_Set2D (void)
{
	glViewport (glx, gly, glwidth, glheight);

	glMatrixMode(GL_PROJECTION);
    glLoadIdentity ();

	// always draw at 640 x 480
	glOrtho  (0, 640, 480, 0, -1, 1);

	glMatrixMode(GL_MODELVIEW);
    glLoadIdentity ();

	glDisable (GL_DEPTH_TEST);
	glDisable (GL_CULL_FACE);
	glDisable (GL_BLEND);
	glEnable (GL_ALPHA_TEST);

	glColor4f (1,1,1,1);

	// on some cards, not doing this will cause the stuff to draw but not to be
	// acclerated!!!  ouch!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	glActiveTexture (GL_TEXTURE0);
	glEnable (GL_TEXTURE_2D);
}


