/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// models.c -- model loading and caching

// models are the only shared resource between a client and server running
// on the same machine.


#include "quakedef.h"
#include "glext.h"

model_t	*loadmodel;
char	loadname[32];	// for hunk tags

void Mod_LoadSpriteModel (model_t *mod, void *buffer);
void Mod_LoadBrushModel (model_t *mod, void *buffer);
void Mod_LoadAliasModel (model_t *mod, void *buffer);
model_t *Mod_LoadModel (model_t *mod, qboolean crash);

byte	mod_novis[MAX_MAP_LEAFS/8];
byte	*decompressed_vis;

#define	MAX_MOD_KNOWN	512
model_t	mod_known[MAX_MOD_KNOWN];
int		mod_numknown;

cvar_t gl_subdivide_size = {"gl_subdivide_size", "10240"};

qboolean WorldModelLoaded = false;
int worldtype = -1;



void GL_LoadMapTexture (texture_t *tx);



char *Mod_FindWorldSpawnEntry (model_t *mod, char *KeyToFind)
{
	char *entlump = mod->entities;
	char WSQuotedKey[128];
	static char WSValue[128];
	int seekOffset;
	int i;

	// build the key we wanna find and get the seekoffset
	// we add the quotes around it to ensure that we get an exact match, i.e. so that light for example
	// (not that you'd have light in worldspawn, but you know what i mean) isn't reported as light_blah
	sprintf (WSQuotedKey, "\"%s\"", KeyToFind);
	seekOffset = strlen (WSQuotedKey) + 1;

	// clear to NULLs
	memset (WSValue, 0, sizeof (WSValue));

	while (1)
	{
		// end of the worldspawn entity
		if (*entlump == '}') return WSValue;

		if (!strncmp (entlump, WSQuotedKey, seekOffset - 1))
			goto Eat_Your_Heart_Out_Bono;

		entlump++;

		continue;

Eat_Your_Heart_Out_Bono:;
		// I've just found what I'm looking for!!!!
		// advance by the seekOffset
		entlump += seekOffset;

		// skip whitespace
		while (1)
		{
			if (*entlump == '\"') break;
			entlump++;
		}

		// skip the quote we just found
		entlump++;

		// copy to WSValue
		for (i = 0; i < 127; i++)
		{
			if (*entlump == '\"') break;

			WSValue[i] = *entlump;
			entlump++;
		}

		return WSValue;
	}

	// never gets here
	return WSValue;
}


/*
===============
Mod_Init

Caches the data if needed
===============
*/
void *Mod_Extradata (model_t *mod)
{
	void	*r;
	
	r = Cache_Check (&mod->cache);
	if (r)
		return r;

	Mod_LoadModel (mod, true);
	
	if (!mod->cache.data)
		Sys_Error ("Mod_Extradata: caching failed");
	return mod->cache.data;
}

/*
===============
Mod_PointInLeaf
===============
*/
mleaf_t *Mod_PointInLeaf (vec3_t p, model_t *model)
{
	mnode_t		*node;
	float		d;

	if (!model || !model->nodes)
		Sys_Error ("Mod_PointInLeaf: bad model");

	node = model->nodes;

	while (1)
	{
		if (node->contents < 0)
			return (mleaf_t *) node;

		switch (node->plane->type)
		{
		case PLANE_X:
			d = p[0] - node->plane->dist;
			break;
		case PLANE_Y:
			d = p[1] - node->plane->dist;
			break;
		case PLANE_Z:
			d = p[2] - node->plane->dist;
			break;
		default:
			d = DotProduct (p, node->plane->normal) - node->plane->dist;
			break;
		}

		if (d > 0)
			node = node->children[0];
		else
			node = node->children[1];
	}

	// never reached
	return NULL;
}


msurface_t *hitsurf;

// fudge the distance cos it's not quite spot on.  5 seems to catch most (if not all).  if you check the
// particle origins you'll see that they're not QUITE on the surf, but a few units out from it
float DIST_FUDGE;

void SCPTraverse (vec3_t p, mnode_t *node)
{
	msurface_t *surf;
	float dist;
	int c;
	int sidebit;
	float fdist;

	// didn't hit anything
	if (node->contents < 0) return;

	// we have a hit so no more checking
	if (hitsurf) return;

	// distance stuff
	dist = DotProduct (p, node->plane->normal) - node->plane->dist;

	// allow fudging by DIST_FUDGE units
	if (dist > DIST_FUDGE)
	{
		SCPTraverse (p, node->children[0]);
		return;
	}

	// we have a hit so no more checking
	if (hitsurf) return;

	// allow fudging by DIST_FUDGE units
	if (dist < -DIST_FUDGE)
	{
		SCPTraverse (p, node->children[1]);
		return;
	}

	// we have a hit so no more checking
	if (hitsurf) return;

	// find out which side of the node we are on
	if (dist >= 0)
		sidebit = 0;
	else
		sidebit = SURF_PLANEBACK;

	// do the surface thing
	for (c = node->numsurfaces, surf = cl.worldmodel->surfaces + node->firstsurface; c ; c--, surf++)
	{
		//if (surf->flags & (SURF_DRAWTURB | SURF_DRAWSKY)) continue;
		if (surf->flags & (SURF_DRAWTURB)) continue;
		if ((surf->flags & SURF_PLANEBACK) != sidebit) continue;

		fdist = DotProduct (p, surf->plane->normal) - surf->plane->dist;

		if (fdist > DIST_FUDGE || fdist < -DIST_FUDGE) continue;

		if ((surf->flags & SURF_PLANEBACK)) fdist *= -1;

		if ((DIST_FUDGE - fabs (fdist) < 0)) continue;

		// gotcha!!!
		hitsurf = surf;
		return;
	}

	// we have a hit so no more checking
	if (hitsurf) return;

	// go down both sides
	SCPTraverse (p, node->children[0]);

	// we have a hit so no more checking
	if (hitsurf) return;

	SCPTraverse (p, node->children[1]);
}

// old test code - checks every surf in a brush model
void BModSCP (model_t *mod, vec3_t p)
{
	msurface_t *surf;
	float fdist;
	int i;

	surf = &mod->surfaces[mod->firstmodelsurface];

	for (i = 0; i < mod->nummodelsurfaces; i++, surf++)
	{
		if (surf->flags & (SURF_DRAWSKY | SURF_DRAWTURB)) continue;

		fdist = DotProduct (p, surf->plane->normal) - surf->plane->dist;

		if (fdist > DIST_FUDGE || fdist < -DIST_FUDGE) continue;

		if ((surf->flags & SURF_PLANEBACK)) fdist *= -1;

		if ((DIST_FUDGE - fabs (fdist) < 0)) continue;

		// gotcha!!!
		hitsurf = surf;
		return;
	}
}


msurface_t *SurfContainsPoint (vec3_t pt, float fudge)
{
	int i;
	model_t *mod;
	vec3_t mod_point;

	DIST_FUDGE = fudge;

	// start with nothing hit
	hitsurf = NULL;

	// do the world
	SCPTraverse (pt, cl.worldmodel->nodes);

	// are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?
	// are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?
	// are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?
	// are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?
	// are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?
	// are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?
	// are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?  are we there yet?
	if (!hitsurf)
	{
		// check the models
		for (i = 0; i < cl_numvisedicts; i++)
		{
			mod = cl_visedicts[i]->model;

			// ensure we have a valid model for checking
			if (!mod) break;
			if (mod->name[0] != '*') continue;
			if (mod->type != mod_brush) continue;

			// save out the model point and offset by entity origin to make the check valid
			mod_point[0] = pt[0] + cl_visedicts[i]->origin[0];
			mod_point[1] = pt[1] + cl_visedicts[i]->origin[1];
			mod_point[2] = pt[2] + cl_visedicts[i]->origin[2];

			// traverse the model nodes
			SCPTraverse (mod_point, mod->nodes + mod->hulls[0].firstclipnode);

			// alternate version - checks every surf in the bmodel.
			// is more successful on bmodels where the node check doesn't work, although the result is
			// messed up plane normals.  my guess is that bad plane normals in the original bsp are
			// at the root of all of this.
			//BModSCP (mod, mod_point);

			// hit something
			if (hitsurf) break;
		}
	}

	/*
	checking
	if (hitsurf)
	{
		Con_Printf ("Hit texture %s and normal [%f] [%f] [%f]\n", hitsurf->texinfo->texture->name,
			hitsurf->plane->normal[0], hitsurf->plane->normal[1], hitsurf->plane->normal[2]);
	}
	*/

	// check for this before checking flags
	if (!hitsurf) return NULL;

	// don't return sky
	if (hitsurf->flags & SURF_DRAWSKY) return NULL;

	// return the surf that was hit
	return hitsurf;
}


/*
===================
Mod_DecompressVis
===================
*/
byte *Mod_DecompressVis (byte *visin, model_t *model)
{
	static byte	decompressed[MAX_MAP_LEAFS / 8];
	int		c;
	byte	*visout;
	int		row;

	memset (decompressed, 0, sizeof (decompressed));
	row = (model->numleafs+7)>>3;
	visout = decompressed;

	if (!visin)
	{
		// no vis info, so make all visible
		while (row)
		{
			*visout++ = 0xff;
			row--;
		}

		return decompressed;		
	}

	do
	{
		if (*visin)
		{
			*visout++ = *visin++;
			continue;
		}
	
		c = visin[1];
		visin += 2;

		while (c)
		{
			*visout++ = 0;
			c--;
		}
	} while (visout - decompressed < row);
	
	return decompressed;
}


// get a basic PVS not including any cached visleafs
byte *Mod_SimpleLeafPVS (mleaf_t *leaf, model_t *model)
{
	if (leaf == model->leafs)
		return mod_novis;
	return Mod_DecompressVis (leaf->compressed_vis, model);
}


byte *Mod_LeafPVS (mleaf_t *leaf, model_t *model)
{
	if (leaf == model->leafs) return mod_novis;
	if (leaf->decompressed) return leaf->decompressed;

	return Mod_SimpleLeafPVS (leaf, model);
}


/*
===================
Mod_ClearAll
===================
*/
void Mod_ClearAll (void)
{
	int		i;
	model_t	*mod;
	
	for (i=0 , mod=mod_known ; i<mod_numknown ; i++, mod++)
		if (mod->type != mod_alias)
			mod->needload = true;
}


/*
==================
Mod_FindName

==================
*/
model_t *Mod_FindName (char *name)
{
	int		i;
	model_t	*mod;
	
	if (!name[0])
		Sys_Error ("Mod_ForName: NULL name");
		
	// search the currently loaded models
	for (i=0 , mod=mod_known ; i<mod_numknown ; i++, mod++)
		if (!strcmp (mod->name, name) )
			break;

	if (i == mod_numknown)
	{
		if (mod_numknown == MAX_MOD_KNOWN)
			Sys_Error ("mod_numknown == MAX_MOD_KNOWN");
		strcpy (mod->name, name);
		mod->needload = true;
		mod_numknown++;
	}

	return mod;
}


/*
==================
Mod_TouchModel

==================
*/
void Mod_TouchModel (char *name)
{
	model_t	*mod;
	
	mod = Mod_FindName (name);
	
	if (!mod->needload)
	{
		if (mod->type == mod_alias)
			Cache_Check (&mod->cache);
	}
}


/*
==================
Mod_LoadModel

Loads a model into the cache
==================
*/
void Sbar_Preload3DModels (void);
qboolean SbarPreload = false;

model_t *Mod_LoadModel (model_t *mod, qboolean crash)
{
	void	*d;
	unsigned *buf;
	byte	stackbuf[1024];		// avoid dirtying the cache heap

	if (!mod->needload)
	{
		if (mod->type == mod_alias)
		{
			d = Cache_Check (&mod->cache);

			if (d) return mod;
		}
		else return mod;		// not cached at all
	}

	// load the file
	buf = (unsigned *) COM_LoadStackFile (mod->name, stackbuf, sizeof(stackbuf));

	if (!buf)
	{
		if (crash)
			Sys_Error ("Mod_NumForName: %s not found", mod->name);
		return NULL;
	}
	
	// allocate a new model
	COM_FileBase (mod->name, loadname);
	
	loadmodel = mod;

	// fill it in
	// call the apropriate loader
	mod->needload = false;

	mod->PulseEffect = false;

	switch (LittleLong(*(unsigned *)buf))
	{
	case IDPOLYHEADER:
		Mod_LoadAliasModel (mod, buf);
		break;

	case IDSPRITEHEADER:
		Mod_LoadSpriteModel (mod, buf);
		break;
	
	default:
		Mod_LoadBrushModel (mod, buf);
		break;
	}

	// load them here cos loading BSPs at runtime messes with the worldmodel lighting
	// make sure we only load them once...!
	if (!SbarPreload)
	{
		SbarPreload = true;
		Sbar_Preload3DModels ();
	}

	return mod;
}


/*
==================
Mod_ForName

Loads in a model for the given name
==================
*/
model_t *Mod_ForName (char *name, qboolean crash)
{
	model_t	*mod;
	
	mod = Mod_FindName (name);

	return Mod_LoadModel (mod, crash);
}


/*
===============
Mod_Init
===============
*/
void Mod_Init (void)
{
	Cvar_RegisterVariable (&gl_subdivide_size);
	memset (mod_novis, 0xff, sizeof(mod_novis));
}

/*
===============================================================================

					BRUSHMODEL LOADING

===============================================================================
*/

byte	*mod_base;


/*
=================
Mod_LoadTextures
=================
*/
texture_t *sky_tex;

void Mod_LoadTextures (lump_t *l)
{
	int			i, j, pixels, num, tmax, altmax;
	miptex_t	*mt;
	texture_t	*tx, *tx2;
	texture_t	*anims[10];
	texture_t	*altanims[10];
	dmiptexlump_t *m;

	if (!l->filelen)
	{
		loadmodel->textures = NULL;
		return;
	}

	m = (dmiptexlump_t *) (mod_base + l->fileofs);

	m->nummiptex = LittleLong (m->nummiptex);

	loadmodel->numtextures = m->nummiptex;
	loadmodel->textures = Hunk_AllocName (m->nummiptex * sizeof (*loadmodel->textures), loadname);

	for (i = 0; i < m->nummiptex; i++)
	{
		m->dataofs[i] = LittleLong(m->dataofs[i]);

		if (m->dataofs[i] == -1)
			continue;

		mt = (miptex_t *) ((byte *) m + m->dataofs[i]);
		mt->width = LittleLong (mt->width);
		mt->height = LittleLong (mt->height);

		for (j = 0; j < MIPLEVELS; j++)
			mt->offsets[j] = LittleLong (mt->offsets[j]);

		if ((mt->width & 15) || (mt->height & 15))
			Sys_Error ("Texture %s is not 16 aligned", mt->name);

		// change this cos GL doesn't need the miplevels
		pixels = mt->width * mt->height / 64 * 85;

		tx = Hunk_AllocName (sizeof (texture_t) + pixels, loadname);
		loadmodel->textures[i] = tx;

		memcpy (tx->name, mt->name, sizeof (tx->name));
		tx->width = mt->width;
		tx->height = mt->height;

		for (j = 0; j < MIPLEVELS; j++)
			tx->offsets[j] = mt->offsets[j] + sizeof (texture_t) - sizeof (miptex_t);

		// HACK HACK HACK
		if (!strcmp (mt->name, "shot1sid") && mt->width == 32 && mt->height == 32 && CRC_Block ((byte *)(mt + 1), mt->width * mt->height) == 65393)
		{
			// This texture in b_shell1.bsp has some of the first 32 pixels painted white.
			// They are invisible in software, but look really ugly in GL. So we just copy
			// 32 pixels from the bottom to make it look nice.
			memcpy (mt + 1, (byte *) (mt + 1) + 32 * 31, 32);
		}

		// the pixels immediately follow the structures
		memcpy (tx + 1, mt + 1, pixels);

		if (!Q_strncmp (mt->name,"sky",3))
		{
			// this causes stack overflows when called from here, so save out the texture
			// instead and init it later
			// R_InitSky (tx);
			sky_tex = tx;
		}
		else
		{
			//Con_DPrintf ("Texture %s has %i unique colours\n", mt->name, CountUniqueColours ((byte *)(tx + 1), pixels));
			GL_LoadMapTexture (tx);
		}
	}

	// sequence the animations
	for (i = 0; i < m->nummiptex ; i++)
	{
		tx = loadmodel->textures[i];

		// NULL texture or not animating
		if (!tx || tx->name[0] != '+') continue;

		// already sequenced
		if (tx->anim_next) continue;

		// find the number of frames in the animation
		memset (anims, 0, sizeof (anims));
		memset (altanims, 0, sizeof (altanims));

		tmax = tx->name[1];
		altmax = 0;

		if (tmax >= 'a' && tmax <= 'z') tmax -= 'a' - 'A';

		if (tmax >= '0' && tmax <= '9')
		{
			tmax -= '0';
			altmax = 0;
			anims[tmax] = tx;
			tmax++;
		}
		else if (tmax >= 'A' && tmax <= 'J')
		{
			altmax = tmax - 'A';
			tmax = 0;
			altanims[altmax] = tx;
			altmax++;
		}
		else Sys_Error ("Bad animating texture %s", tx->name);

		for (j = i + 1; j < m->nummiptex; j++)
		{
			tx2 = loadmodel->textures[j];

			// NULL texture or not animating
			if (!tx2 || tx2->name[0] != '+') continue;

			// not in this sequence babe
			if (strcmp (tx2->name + 2, tx->name + 2)) continue;

			num = tx2->name[1];

			if (num >= 'a' && num <= 'z') num -= 'a' - 'A';

			if (num >= '0' && num <= '9')
			{
				num -= '0';
				anims[num] = tx2;

				if (num + 1 > tmax) tmax = num + 1;
			}
			else if (num >= 'A' && num <= 'J')
			{
				num = num - 'A';
				altanims[num] = tx2;

				if (num + 1 > altmax) altmax = num + 1;
			}
			else Sys_Error ("Bad animating texture %s", tx->name);
		}
		
#define	ANIM_CYCLE	2

		// link them all together
		for (j = 0; j < tmax; j++)
		{
			tx2 = anims[j];

			if (!tx2) Sys_Error ("Missing frame %i of %s", j, tx->name);

			tx2->anim_total = tmax * ANIM_CYCLE;
			tx2->anim_min = j * ANIM_CYCLE;
			tx2->anim_max = (j + 1) * ANIM_CYCLE;
			tx2->anim_next = anims[(j + 1) % tmax];

			if (altmax) tx2->alternate_anims = altanims[0];
		}

		for (j = 0; j < altmax; j++)
		{
			tx2 = altanims[j];

			if (!tx2) Sys_Error ("Missing frame %i of %s", j, tx->name);

			tx2->anim_total = altmax * ANIM_CYCLE;
			tx2->anim_min = j * ANIM_CYCLE;
			tx2->anim_max = (j + 1) * ANIM_CYCLE;
			tx2->anim_next = altanims[(j + 1) % altmax];

			if (tmax) tx2->alternate_anims = anims[0];
		}
	}
}


/*
=================
Mod_LoadLighting
=================
*/
// for restoring stainmaps
extern char lmloadname[256];
extern qboolean reloadlm;


void Mod_LoadLighting (lump_t *l)
{
	int	i;				// loop control
	byte *light_base;	// base offset of light data
	byte *light_buffer;

	FILE		*LITFile;
	char		LITFileName[1024];
	litheader_t	LITHeader;

	int bufferavg;
	int j;

	// have we any light data to begin with?
	if (!l->filelen)
	{
		loadmodel->lightdata = NULL;
		loadmodel->staindata = NULL;
		loadmodel->lightdatasize = 0;
		return;
	}

	// load the light data from the BSP first and expand to 3 components
	loadmodel->lightdata = Hunk_AllocName (l->filelen * 3, loadname);
	light_base = mod_base + l->fileofs;

	for (i = 0; i < l->filelen; i++)
	{
		// unnecessary brackets improve legibility and clarity
		loadmodel->lightdata[(i * 3) + 0] = light_base[i];
		loadmodel->lightdata[(i * 3) + 1] = light_base[i];
		loadmodel->lightdata[(i * 3) + 2] = light_base[i];
	}

	// now load it from the lit file.  most people have lit files generated by my old
	// mhcolour program - unfortunately, that's based on tyrlight which didn't generate
	// the exact same lightmap sizes as id's original light had.  i'm not knocking
	// tyrann - his work was great - the mistake was mine to use his code cos it just
	// wasn't suitable for the purpose i tried to use it for.  (basically i just got
	// lazy and wanted a good light executable that already supported coloured light
	// without having to make my own.)

	strcpy (LITFileName, loadmodel->name);

	// we can't just search for '.' cos there may be one in a folder name!!!
	for (i = 0; LITFileName[i] != '\0'; i++);

	LITFileName[i - 3] = 'l';
	LITFileName[i - 2] = 'i';
	LITFileName[i - 1] = 't';

	// see is a LIT file present - try to open it (correct game dir only, otherwise we'll
	// have problems when we try to open a lit file for a map that has the same name as
	// one of the id1 maps but belongs to a different game - start, for example)
	LITFile = fopen (va ("%s/%s", com_gamedir, LITFileName), "rb");

	if (LITFile)
	{
		// read and validate the header
		fread (&LITHeader, sizeof (LITHeader), 1, LITFile);

		if (LITHeader.version == 1 && LITHeader.ident[0] == 'Q' && LITHeader.ident[1] == 'L' && LITHeader.ident[2] == 'I' && LITHeader.ident[3] == 'T')
		{
			// it's valid
			light_buffer = (byte *) malloc (l->filelen * 3);

			// read it in
			fread (light_buffer, l->filelen * 3, 1, LITFile);

			// now compare with the light from the BSP - incorrect lightmap sizes in
			// mhcolour can cause some weird effects in some areas - the nailgun room
			// in e1m1 is a perfect example.  this is a hack - the real solution
			// would be to rewrite mhcolour to fix this.  too late for that though...
			for (i = 0; i < l->filelen; i++)
			{
				for (j = 0; j < 3; j++)
				{
					if (light_buffer[(i * 3) + j] == loadmodel->lightdata[(i * 3) + j]) continue;
					if (light_buffer[(i * 3) + j] == 0) continue;
					if (loadmodel->lightdata[(i * 3) + j] == 0) continue;
					if (light_buffer[(i * 3) + j] == 255) continue;
					if (loadmodel->lightdata[(i * 3) + j] == 255) continue;

					loadmodel->lightdata[(i * 3) + j] = light_buffer[(i * 3) + j];
				}
			}

			free (light_buffer);
		}

		fclose (LITFile);
	}

	// make the stain data
	loadmodel->staindata = Hunk_AllocName (l->filelen * 3, loadname);

	// set to all fullbright
	memset (loadmodel->staindata, 255, l->filelen * 3);

	loadmodel->lightdatasize = l->filelen * 3;

	// nothing to reload
	if (!reloadlm) return;

	// make sure it doesn't reload next time round unless explicitly instructed to do so
	reloadlm = false;

	// we had to get the original lightdata anyway so we could save it out to staindata:
	// now we load lightdata from the lit file in the game save directory.  The names are
	// misleading as "staindata" is actually nothing to do with stains, but is just used
	// for R_LightPoint on alias models, whereas lightdata contains the stains: they're
	// the wrong way round from what you'd expect!!!
	Con_DPrintf ("Restoring stain data from %s\n", lmloadname);

	LITFile = fopen (lmloadname, "rb");

	if (!LITFile)
	{
		// this can only happen if there is no stainmap data file to load, i.e. none
		// was saved.  It's a harmless condition and no need to crap out.
		Con_DPrintf ("Could not open stain data in %s\n", lmloadname);
		return;
	}

	// read and validate the header
	fread (&LITHeader, sizeof (LITHeader), 1, LITFile);

	if (LITHeader.version == 1 && LITHeader.ident[0] == 'Q' && LITHeader.ident[1] == 'L' && LITHeader.ident[2] == 'I' && LITHeader.ident[3] == 'T')
	{
		// it's valid so read it in
		fread (loadmodel->staindata, l->filelen * 3, 1, LITFile);
		Con_DPrintf ("Successfully restored stain data from %s\n", lmloadname);
	}
	else
	{
		// another benign error condition - we can just use the original light data
		Con_DPrintf ("Bad Lightdata save file\n");
	}

	fclose (LITFile);
}


/*
=================
Mod_LoadVisibility
=================
*/
void Mod_LoadVisibility (lump_t *l)
{
	if (!l->filelen)
	{
		loadmodel->visdata = NULL;
		return;
	}

	loadmodel->visdata = Hunk_AllocName ( l->filelen, loadname);	
	memcpy (loadmodel->visdata, mod_base + l->fileofs, l->filelen);
}


/*
=================
Mod_LoadEntities
=================
*/
void Mod_LoadEntities (lump_t *l)
{
	if (!l->filelen)
	{
		loadmodel->entities = NULL;
		return;
	}

	// add 1 to allow for NULL termination
	loadmodel->entities = Hunk_AllocName ( l->filelen + 1, loadname);

	// clear the string
	memset (loadmodel->entities, 0, l->filelen + 1);

	// copy it in
	memcpy (loadmodel->entities, mod_base + l->fileofs, l->filelen);

	// save out the size
	loadmodel->entLumpSize = l->filelen;

	// load the lights (worldmodel only)
	// if (!WorldModelLoaded) R_LoadLights (loadmodel->entities);
}


/*
=================
Mod_LoadVertexes
=================
*/
void Mod_LoadVertexes (lump_t *l)
{
	dvertex_t	*vertin;
	mvertex_t	*vertout;
	int			i, count;

	vertin = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*vertin))
		Sys_Error ("MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*vertin);
	vertout = Hunk_AllocName ( count*sizeof(*vertout), loadname);	

	loadmodel->vertexes = vertout;
	loadmodel->numvertexes = count;

	for ( i=0 ; i<count ; i++, vertin++, vertout++)
	{
		vertout->position[0] = LittleFloat (vertin->point[0]);
		vertout->position[1] = LittleFloat (vertin->point[1]);
		vertout->position[2] = LittleFloat (vertin->point[2]);
	}
}

/*
=================
Mod_LoadSubmodels
=================
*/
void Mod_LoadSubmodels (lump_t *l)
{
	dmodel_t	*submodelin;
	dmodel_t	*submodelout;
	int			i, j, count;

	submodelin = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*submodelin))
		Sys_Error ("MOD_LoadBmodel: funny lump size submodelin %s",loadmodel->name);
	count = l->filelen / sizeof(*submodelin);
	submodelout = Hunk_AllocName ( count*sizeof(*submodelout), loadname);	

	loadmodel->submodels = submodelout;
	loadmodel->numsubmodels = count;

	for ( i=0 ; i<count ; i++, submodelin++, submodelout++)
	{
		for (j=0 ; j<3 ; j++)
		{	// spread the mins / maxs by a pixel
			submodelout->mins[j] = LittleFloat (submodelin->mins[j]) - 1;
			submodelout->maxs[j] = LittleFloat (submodelin->maxs[j]) + 1;
			submodelout->origin[j] = LittleFloat (submodelin->origin[j]);
		}
		for (j=0 ; j<MAX_MAP_HULLS ; j++)
			submodelout->headnode[j] = LittleLong (submodelin->headnode[j]);
		submodelout->visleafs = LittleLong (submodelin->visleafs);
		submodelout->firstface = LittleLong (submodelin->firstface);
		submodelout->numfaces = LittleLong (submodelin->numfaces);
	}
}

/*
=================
Mod_LoadEdges
=================
*/
void Mod_LoadEdges (lump_t *l)
{
	dedge_t *edgein;
	medge_t *edgeout;
	int 	i, count;

	edgein = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*edgein))
		Sys_Error ("MOD_LoadBmodel: funny lump size edgein %s",loadmodel->name);
	count = l->filelen / sizeof(*edgein);
	edgeout = Hunk_AllocName ( (count + 1) * sizeof(*edgeout), loadname);	

	loadmodel->edges = edgeout;
	loadmodel->numedges = count;

	for ( i=0 ; i<count ; i++, edgein++, edgeout++)
	{
		edgeout->v[0] = (unsigned short)LittleShort(edgein->v[0]);
		edgeout->v[1] = (unsigned short)LittleShort(edgein->v[1]);
	}
}

/*
=================
Mod_LoadTexinfo
=================
*/
void Mod_LoadTexinfo (lump_t *l)
{
	texinfo_t *texinfoin;
	mtexinfo_t *texinfoout;
	int 	i, j, count;
	int		miptex;
	float	len1, len2;

	texinfoin = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*texinfoin))
		Sys_Error ("MOD_LoadBmodel: funny lump size texinfoin %s",loadmodel->name);
	count = l->filelen / sizeof(*texinfoin);
	texinfoout = Hunk_AllocName ( count*sizeof(*texinfoout), loadname);	

	loadmodel->texinfo = texinfoout;
	loadmodel->numtexinfo = count;

	for ( i=0 ; i<count ; i++, texinfoin++, texinfoout++)
	{
		for (j=0 ; j<8 ; j++)
			texinfoout->vecs[0][j] = LittleFloat (texinfoin->vecs[0][j]);

		len1 = Length (texinfoout->vecs[0]);
		len2 = Length (texinfoout->vecs[1]);

		len1 = (len1 + len2)/2;

		if (len1 < 0.32)
			texinfoout->mipadjust = 4;
		else if (len1 < 0.49)
			texinfoout->mipadjust = 3;
		else if (len1 < 0.99)
			texinfoout->mipadjust = 2;
		else
			texinfoout->mipadjust = 1;

		miptex = LittleLong (texinfoin->miptex);
		texinfoout->flags = LittleLong (texinfoin->flags);
	
		if (!loadmodel->textures)
		{
			texinfoout->texture = r_notexture_mip;	// checkerboard texture
			texinfoout->flags = 0;
		}
		else
		{
			if (miptex >= loadmodel->numtextures)
				Sys_Error ("miptex >= loadmodel->numtextures");

			texinfoout->texture = loadmodel->textures[miptex];

			// this only happens in e2m3, which is corrupt in other ways too.
			if (!texinfoout->texture)
			{
				texinfoout->texture = r_notexture_mip; // texture not found
				texinfoout->flags = 0;
			}
		}
	}
}

/*
================
CalcSurfaceExtents

Fills in s->texturemins[] and s->extents[]
================
*/
void CalcSurfaceExtents (msurface_t *s)
{
	float	mins[2], maxs[2], val;
	int		i,j, e;
	mvertex_t	*v;
	mtexinfo_t	*tex;
	int		bmins[2], bmaxs[2];

	mins[0] = mins[1] = 999999;
	maxs[0] = maxs[1] = -99999;

	tex = s->texinfo;
	
	for (i=0 ; i<s->numedges ; i++)
	{
		e = loadmodel->surfedges[s->firstedge+i];
		if (e >= 0)
			v = &loadmodel->vertexes[loadmodel->edges[e].v[0]];
		else
			v = &loadmodel->vertexes[loadmodel->edges[-e].v[1]];
		
		for (j=0 ; j<2 ; j++)
		{
			val = v->position[0] * tex->vecs[j][0] + 
				v->position[1] * tex->vecs[j][1] +
				v->position[2] * tex->vecs[j][2] +
				tex->vecs[j][3];
			if (val < mins[j])
				mins[j] = val;
			if (val > maxs[j])
				maxs[j] = val;
		}
	}

	for (i=0 ; i<2 ; i++)
	{	
		bmins[i] = floor(mins[i]/16);
		bmaxs[i] = ceil(maxs[i]/16);

		s->texturemins[i] = bmins[i] * 16;
		s->extents[i] = (bmaxs[i] - bmins[i]) * 16;
		if ( !(tex->flags & TEX_SPECIAL) && s->extents[i] > 512 /* 256 */ )
			Sys_Error ("Bad surface extents");
	}
}


/*
=================
Mod_LoadFaces
=================
*/
void GL_BuildPolygonFromSurface (model_t *mod, msurface_t *fa);
void GL_SubdivideSurface (msurface_t *fa);


// we need some info before we can begin loading lightdata
void Mod_PreCheckFaces (lump_t *l)
{
	dface_t		*facein;
	int			count;

	facein = (void *)(mod_base + l->fileofs);

	if (l->filelen % sizeof(*facein))
		Sys_Error ("MOD_LoadBmodel: funny lump size facein %s",loadmodel->name);

	count = l->filelen / sizeof(*facein);

	loadmodel->numsurfaces = count;
}


void Mod_LoadFaces (lump_t *l)
{
	dface_t		*facein;
	msurface_t 	*faceout;
	int			i, count, surfnum;
	int			planenum, side;
	int			j;

	facein = (void *)(mod_base + l->fileofs);

	if (l->filelen % sizeof(*facein))
		Sys_Error ("MOD_LoadBmodel: funny lump size facein %s",loadmodel->name);

	count = l->filelen / sizeof(*facein);
	faceout = Hunk_AllocName ( count*sizeof(*faceout), loadname);	

	loadmodel->surfaces = faceout;
	loadmodel->numsurfaces = count;

	for (surfnum = 0; surfnum < count; surfnum++, facein++, faceout++)
	{
		faceout->firstedge = LittleLong(facein->firstedge);
		faceout->numedges = LittleShort(facein->numedges);		
		faceout->flags = 0;

		planenum = LittleShort(facein->planenum);
		side = LittleShort(facein->side);

		if (side) faceout->flags |= SURF_PLANEBACK;			

		faceout->plane = loadmodel->planes + planenum;

		// normals in quake aren't exactly negative/positive to properly 
		// refect the surface winding, so hack the normals if it planebacks
		if (side)	// same as (faceout->flags & SURF_PLANEBACK)
		{
			faceout->plane->normal[0] *= -1;
			faceout->plane->normal[1] *= -1;
			faceout->plane->normal[2] *= -1;
		}

		// moved from mod_loadplanes - see the comment in the msurface_s struct for the reason for this.
		// convert plane normals to angles so we don't have to do it at run time.
		faceout->planeangles[0] = acos (faceout->plane->normal[2]) / M_PI * 180;

		if (faceout->plane->normal[0])
			faceout->planeangles[1] = atan2 (faceout->plane->normal[1], faceout->plane->normal[0]) / M_PI * 180;
		else if (faceout->plane->normal[1] > 0)
			faceout->planeangles[1] = 90;
		else if (faceout->plane->normal[1] < 0)
			faceout->planeangles[1] = 270;
		else
			faceout->planeangles[1] = 0;

		// restore the normals
		if (side)	// same as (faceout->flags & SURF_PLANEBACK)
		{
			faceout->plane->normal[0] *= -1;
			faceout->plane->normal[1] *= -1;
			faceout->plane->normal[2] *= -1;
		}

		faceout->texinfo = loadmodel->texinfo + LittleShort (facein->texinfo);

		CalcSurfaceExtents (faceout);

		// lighting info
		faceout->staticmap = true;

		for (i = 0; i < MAXLIGHTMAPS; i++)
		{
			faceout->styles[i] = facein->styles[i];

			// wait until the server sends the lightstyles before building these
			// we do animating lightstyles as well cos they're going to have to
			// be rebuilt the first time they're seen anyway, owing to the
			// animation.
			if (faceout->styles[i] >= 1 && faceout->styles[i] != 255)
			{
				faceout->staticmap = false;
			}
		}

		i = LittleLong(facein->lightofs);

		if (i == -1)
		{
			faceout->samples = NULL;
			faceout->stainsamples = NULL;
			faceout->lightofs = -1;
		}
		else
		{
			faceout->samples = loadmodel->lightdata + (i * 3);
			faceout->stainsamples = loadmodel->staindata + (i * 3);
			faceout->lightofs = (i * 3);
		}

		// initial state
		faceout->stained = false;
		faceout->cached_dlight = false;
		faceout->dlightbits = 0;
		faceout->dlightframe = -1;
		faceout->drawChain = NULL;
		faceout->vertlightchain = NULL;
		faceout->fbChain = NULL;
		faceout->lqChain = NULL;
		faceout->visframe = -1;

		faceout->inClientPVS = false;

		// set the drawing flags flag
		faceout->smax = (faceout->extents[0] >> 4) + 1;
		faceout->tmax = (faceout->extents[1] >> 4) + 1;

		// auto water trans matching
		faceout->surfNum = surfnum;
		faceout->match = NULL;

		// underwater fog and caustics
		faceout->caustic = NULL;

		if (!Q_strncmp(faceout->texinfo->texture->name,"sky",3))	// sky
		{
			faceout->flags |= (SURF_DRAWSKY | SURF_DRAWTILED);

			continue;
		}

		// precalculate the surface warp indexes (do it here so that it's not done for sky)
		switch (faceout->plane->type)
		{
		case PLANE_Y:
			// vertical in line with the world grid
			faceout->warpindex[0] = 0;
			faceout->warpindex[1] = 2;
			break;

		case PLANE_X:
			// vertical in line with the world grid
			faceout->warpindex[0] = 2;
			faceout->warpindex[1] = 1;
			break;

		case PLANE_ANYX:
		case PLANE_ANYY:
			// vertical at an angle to the world grid
			faceout->warpindex[0] = 2;
			faceout->warpindex[1] = 0;
			break;

		case PLANE_Z:
			// flat
		default:
			// same as flat
			faceout->warpindex[0] = 0;
			faceout->warpindex[1] = 1;
			break;
		}

		if (faceout->texinfo->texture->name[0] == '*')		// turbulent
		{
			faceout->flags |= (SURF_DRAWTURB | SURF_DRAWTILED);

			// identify different types of liquid - faster than strncmp
			// oh you evil little bastard!!!  i had the 2 "else if"s as "if"s...
			if (faceout->texinfo->texture->name[1] == 'l' &&
				faceout->texinfo->texture->name[2] == 'a' &&
				faceout->texinfo->texture->name[3] == 'v' &&
				faceout->texinfo->texture->name[4] == 'a')
				faceout->flags |= SURF_DRAWLAVA;
			else if (faceout->texinfo->texture->name[1] == 't' &&
				faceout->texinfo->texture->name[2] == 'e' &&
				faceout->texinfo->texture->name[3] == 'l' &&
				faceout->texinfo->texture->name[4] == 'e')
				faceout->flags |= SURF_DRAWTELE;
			else if (faceout->texinfo->texture->name[1] == 's' &&
				faceout->texinfo->texture->name[2] == 'l' &&
				faceout->texinfo->texture->name[3] == 'i' &&
				faceout->texinfo->texture->name[4] == 'm' &&
				faceout->texinfo->texture->name[5] == 'e')
				faceout->flags |= SURF_DRAWSLIME;
			else faceout->flags |= SURF_DRAWWATER;

			for (i=0 ; i<2 ; i++)
			{
				faceout->extents[i] = 16384;
				faceout->texturemins[i] = -8192;
			}

			GL_SubdivideSurface (faceout);	// cut up polygon for warps

			// MH don't subdivide sky

			continue;
		}

		GL_BuildPolygonFromSurface (loadmodel, faceout);
	}
}


/*
=================
Mod_SetParent
=================
*/
void Mod_SetParent (mnode_t *node, mnode_t *parent)
{
	node->parent = parent;

	if (node->contents < 0) return;

	Mod_SetParent (node->children[0], node);
	Mod_SetParent (node->children[1], node);
}

/*
=================
Mod_LoadNodes
=================
*/
void Mod_LoadNodes (lump_t *l)
{
	int			i, j, count, p;
	dnode_t		*nodein;
	mnode_t 	*nodeout;

	int			c;
	glpoly_t	*poly;
	float		*v;
	float		numverts;

	msurface_t	*surf;

	nodein = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*nodein))
		Sys_Error ("MOD_LoadBmodel: funny lump size nodein %s",loadmodel->name);
	count = l->filelen / sizeof(*nodein);
	nodeout = Hunk_AllocName ( count*sizeof(*nodeout), loadname);	

	loadmodel->nodes = nodeout;
	loadmodel->numnodes = count;

	for ( i=0 ; i<count ; i++, nodein++, nodeout++)
	{
		for (j=0 ; j<3 ; j++)
		{
			nodeout->minmaxs[j] = LittleShort (nodein->mins[j]);
			nodeout->minmaxs[3+j] = LittleShort (nodein->maxs[j]);
		}
	
		p = LittleLong(nodein->planenum);
		nodeout->plane = loadmodel->planes + p;

		nodeout->firstsurface = LittleShort (nodein->firstface);
		nodeout->numsurfaces = LittleShort (nodein->numfaces);
		
		nodeout->nodenum = i;

		nodeout->inClientPVS = false;
		nodeout->lightVisFrame = 0;

		for (j=0 ; j<2 ; j++)
		{
			p = LittleShort (nodein->children[j]);

			if (p >= 0)
				nodeout->children[j] = loadmodel->nodes + p;
			else
				nodeout->children[j] = (mnode_t *)(loadmodel->leafs + (-1 - p));
		}

		nodeout->visframe = -1;

		numverts = nodeout->origin[0] = nodeout->origin[1] = nodeout->origin[2] = 0;

		for (j = 0, surf = loadmodel->surfaces + nodeout->firstsurface; j < nodeout->numsurfaces; j++, surf++)
		{
			if (surf->flags & SURF_DRAWTURB)
			{
				for (poly = surf->polys; poly; poly = poly->next)
				{
					for (c = 0, v = poly->verts; c < poly->numverts; c++, v += VERTEXSIZE)
					{
						numverts++;

						nodeout->origin[0] += v[0];
						nodeout->origin[1] += v[1];
						nodeout->origin[2] += v[2];
					}
				}
			}
			else if (surf->flags & SURF_DRAWSKY)
			{
			}
			else
			{
				poly = surf->polys;
				v = poly->verts;

				for (c = 0; c < poly->numverts; c++, v += VERTEXSIZE)
				{
					numverts++;

					nodeout->origin[0] += v[0];
					nodeout->origin[1] += v[1];
					nodeout->origin[2] += v[2];
				}
			}
		}

		if (!numverts) continue;

		nodeout->origin[0] /= numverts;
		nodeout->origin[1] /= numverts;
		nodeout->origin[2] /= numverts;
	}
	
	Mod_SetParent (loadmodel->nodes, NULL);	// sets nodes and leafs
}

/*
=================
Mod_LoadLeafs
=================
*/
void Mod_LoadLeafs (lump_t *l)
{
	dleaf_t 	*leafin;
	mleaf_t 	*leafout;
	int			i, j, count, p;

	int			c;
	glpoly_t	*poly;
	float		*v;
	float		numverts;

	leafin = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*leafin))
		Sys_Error ("MOD_LoadBmodel: funny lump size leafin %s",loadmodel->name);
	count = l->filelen / sizeof(*leafin);
	leafout = Hunk_AllocName ( count*sizeof(*leafout), loadname);	

	loadmodel->leafs = leafout;
	loadmodel->numleafs = count;

	for ( i=0 ; i<count ; i++, leafin++, leafout++)
	{
		for (j=0 ; j<3 ; j++)
		{
			leafout->minmaxs[j] = LittleShort (leafin->mins[j]);
			leafout->minmaxs[3+j] = LittleShort (leafin->maxs[j]);
		}

		p = LittleLong(leafin->contents);
		leafout->contents = p;

		leafout->firstmarksurface = loadmodel->marksurfaces + LittleShort(leafin->firstmarksurface);
		leafout->nummarksurfaces = LittleShort(leafin->nummarksurfaces);
		
		p = LittleLong(leafin->visofs);

		leafout ->leafnum = i;

		if (p == -1)
			leafout->compressed_vis = NULL;
		else
			leafout->compressed_vis = loadmodel->visdata + p;

		leafout->decompressed = NULL;

		leafout->efrags = NULL;

		leafout->inClientPVS = false;
		leafout->lightVisFrame = 0;
		
		for (j=0 ; j<4 ; j++)
			leafout->ambient_sound_level[j] = leafin->ambient_level[j];

		// liquid flag
		leafout->hasLiquid = false;

		for (j = 0; j < leafout->nummarksurfaces; j++)
		{
			if (leafout->firstmarksurface[j]->flags & SURF_DRAWTURB)
			{
				leafout->hasLiquid = true;
				break;
			}
		}

		// volumes
		leafout->volume = -1;
		leafout->chain = NULL;

		// gl underwater warp
		if (leafout->contents != CONTENTS_EMPTY)
			for (j = 0; j < leafout->nummarksurfaces; j++)
				leafout->firstmarksurface[j]->flags |= SURF_UNDERWATER;

		// underwater fog and caustics
		leafout->caustic = NULL;

		// trans water visibility
		leafout->waterVisFrame = -1;

		// leaf origleafin
		// this is just the average of the verts of all the surfs leafin the leaf
		numverts = leafout->origin[0] = leafout->origin[1] = leafout->origin[2] = 0;

		for (j = 0; j < leafout->nummarksurfaces; j++)
		{
			if (leafout->firstmarksurface[j]->flags & SURF_DRAWTURB)
			{
				for (poly = leafout->firstmarksurface[j]->polys; poly; poly = poly->next)
				{
					for (c = 0, v = poly->verts; c < poly->numverts; c++, v += VERTEXSIZE)
					{
						numverts++;

						leafout->origin[0] += v[0];
						leafout->origin[1] += v[1];
						leafout->origin[2] += v[2];
					}
				}
			}
			else if (leafout->firstmarksurface[j]->flags & SURF_DRAWSKY)
			{
			}
			else
			{
				poly = leafout->firstmarksurface[j]->polys;
				v = poly->verts;

				for (c = 0; c < poly->numverts; c++, v += VERTEXSIZE)
				{
					numverts++;

					leafout->origin[0] += v[0];
					leafout->origin[1] += v[1];
					leafout->origin[2] += v[2];
				}
			}
		}

		if (!numverts) continue;

		leafout->origin[0] /= numverts;
		leafout->origin[1] /= numverts;
		leafout->origin[2] /= numverts;
	}
}

/*
=================
Mod_LoadClipnodes
=================
*/
void Mod_LoadClipnodes (lump_t *l)
{
	dclipnode_t *cnin, *cnout;
	int			i, count;
	hull_t		*hull;

	cnin = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*cnin))
		Sys_Error ("MOD_LoadBmodel: funny lump size cnin %s",loadmodel->name);
	count = l->filelen / sizeof(*cnin);
	cnout = Hunk_AllocName ( count*sizeof(*cnout), loadname);	

	loadmodel->clipnodes = cnout;
	loadmodel->numclipnodes = count;

	// player size
	hull = &loadmodel->hulls[1];
	hull->clipnodes = cnout;
	hull->firstclipnode = 0;
	hull->lastclipnode = count-1;
	hull->planes = loadmodel->planes;
	hull->clip_mins[0] = -16;
	hull->clip_mins[1] = -16;
	hull->clip_mins[2] = -24;
	hull->clip_maxs[0] = 16;
	hull->clip_maxs[1] = 16;
	hull->clip_maxs[2] = 32;

	// shambler size
	hull = &loadmodel->hulls[2];
	hull->clipnodes = cnout;
	hull->firstclipnode = 0;
	hull->lastclipnode = count-1;
	hull->planes = loadmodel->planes;
	hull->clip_mins[0] = -32;
	hull->clip_mins[1] = -32;
	hull->clip_mins[2] = -24;
	hull->clip_maxs[0] = 32;
	hull->clip_maxs[1] = 32;
	hull->clip_maxs[2] = 64;

	// crouchcning
	hull = &loadmodel->hulls[3];
	hull->clipnodes = cnout;
	hull->firstclipnode = 0;
	hull->lastclipnode = count-1;
	hull->planes = loadmodel->planes;
	hull->clip_mins[0] = -16;
	hull->clip_mins[1] = -16;
	hull->clip_mins[2] = -24;
	hull->clip_maxs[0] = 16;
	hull->clip_maxs[1] = 16;
	hull->clip_maxs[2] = 32;

	for (i=0 ; i<count ; i++, cnout++, cnin++)
	{
		cnout->planenum = LittleLong(cnin->planenum);
		cnout->children[0] = LittleShort(cnin->children[0]);
		cnout->children[1] = LittleShort(cnin->children[1]);
	}
}


/*
=================
Mod_MakeHull0

Deplicate the drawing hull structure as a clipping hull
=================
*/
void Mod_MakeHull0 (void)
{
	mnode_t		*hullin, *child;
	dclipnode_t *hullout;
	int			i, j, count;
	hull_t		*hull;
	
	hull = &loadmodel->hulls[0];	
	
	hullin = loadmodel->nodes;
	count = loadmodel->numnodes;
	hullout = Hunk_AllocName ( count*sizeof(*hullout), loadname);	

	hull->clipnodes = hullout;
	hull->firstclipnode = 0;
	hull->lastclipnode = count-1;
	hull->planes = loadmodel->planes;

	for (i=0 ; i<count ; i++, hullout++, hullin++)
	{
		hullout->planenum = hullin->plane - loadmodel->planes;
		for (j=0 ; j<2 ; j++)
		{
			child = hullin->children[j];
			if (child->contents < 0)
				hullout->children[j] = child->contents;
			else
				hullout->children[j] = child - loadmodel->nodes;
		}
	}
}

/*
=================
Mod_LoadMarksurfaces
=================
*/
void Mod_LoadMarksurfaces (lump_t *l)
{
	int		i, j, count;
	short		*msin;
	msurface_t **msout;

	msin = (void *)(mod_base + l->fileofs);

	if (l->filelen % sizeof (*msin))
		Sys_Error ("MOD_LoadBmodel: funny lump size msin %s",loadmodel->name);

	count = l->filelen / sizeof (*msin);
	msout = Hunk_AllocName (count * sizeof (*msout), loadname);	

	loadmodel->marksurfaces = msout;
	loadmodel->nummarksurfaces = count;

	for (i = 0; i < count; i++)
	{
		j = LittleShort (msin[i]);

		if (j >= loadmodel->numsurfaces)
			Sys_Error ("Mod_ParseMarksurfaces: bad surface number");

		msout[i] = loadmodel->surfaces + j;
	}
}


/*
=================
Mod_LoadSurfedges
=================
*/
void Mod_LoadSurfedges (lump_t *l)
{	
	int		i, count;
	int		*sein, *seout;
	
	sein = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*sein))
		Sys_Error ("MOD_LoadBmodel: funny lump size sein %s",loadmodel->name);
	count = l->filelen / sizeof(*sein);
	seout = Hunk_AllocName ( count*sizeof(*seout), loadname);	

	loadmodel->surfedges = seout;
	loadmodel->numsurfedges = count;

	for ( i=0 ; i<count ; i++)
		seout[i] = LittleLong (sein[i]);
}


/*
=================
Mod_LoadPlanes
=================
*/
void Mod_LoadPlanes (lump_t *l)
{
	int			i, j;
	mplane_t	*planeout;
	dplane_t 	*planein;
	int			count;
	int			bits;
	
	planein = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*planein))
		Sys_Error ("MOD_LoadBmodel: funny lump size planein %s",loadmodel->name);
	count = l->filelen / sizeof(*planein);
	planeout = Hunk_AllocName ( count*2*sizeof(*planeout), loadname);	
	
	loadmodel->planes = planeout;
	loadmodel->numplanes = count;

	for ( i=0 ; i<count ; i++, planein++, planeout++)
	{
		bits = 0;
		for (j=0 ; j<3 ; j++)
		{
			planeout->normal[j] = LittleFloat (planein->normal[j]);
			if (planeout->normal[j] < 0)
				bits |= 1<<j;
		}

		planeout->dist = LittleFloat (planein->dist);
		planeout->type = LittleLong (planein->type);
		planeout->signbits = bits;
	}
}


/*
======================================================================================

					MH - AUTOMATIC WATER TRANSLUCENCY BEGIN
					---------------------------------------
"The Theory Bit" (tm) - Visibility in Quake is totally controlled by leafs.  Each leaf
has associated vis data, which is just an array of bits, one for every other leaf in
the map.  If a leaf's bit is set to 1 it's visible, if it's set to 0 it's not.  Vis
data is stored in a compressed format (all zeroes are bunched together with the number
of consecutive zeroes recorded) to save on disk space and memory, but these days thats
not such an issue any more, and decompressing it does require extra CPU time.

This method just goes through each leaf in the map, finds a PVS (potentially viewable
or visible set, whichever you prefer - it's just the list of other leafs which are
visible from the main leaf) from the vis data, then determines if there are any water
surfaces in the PVS.

Surfaces can be shared between leafs, but not across content boundaries.  If one leaf
is underwater and another is not, they do not share the same surface data - a surface
must therefore be recorded twice.

However, the basic data in the original MAP file is the same, so it's a reliable bet
that the surfaces, although recorded twice, have similar characteristics.  The number
of polys will be the same, the number of verts in each poly will be the same, the
actual vert data will be the same (but not necessarily in the same order, although
the first vert is always the same) and the textures on the surfaces will be the same.
The surfaces will also be in sequence within the mod->surfaces list, i.e. one will
directly follow the other.

I found this out by dumping a lot of surface data to text files from within the engine
and going through it by hand.  My pain means that you don't have to do it now.

So I have a precalculated list of these matching surfaces, and for each leaf which
contains water in the standard PVS I go through the water surfaces and mark the ones
that match them.  By doing this I now have a list of water surfaces in the map which
are directly underwater from the ones which would normally be visible.

There's no way of finding which leafs contain any given surface from Quakes basic data
(and it can't be done too easily without big static arrays) so now I just do a 
simple loop through all the leafs in the world, identifying the ones which contain
the matched surfaces.  Then I calculate a basic PVS for these leafs and add that PVS
to the original one from back up above.  Since a PVS is just bit data, a bitwise OR
suffices to add them.  The result is a new PVS for the original leaf containing all
underwater surfaces which would be viewable from it.  Then I store this (seperately
from the original vis data so as not to confuse them) in an uncompressed format to
make retrieval faster.

This operation is only valid for the world model - submodels and brush models don't
use it, and the results are "undefined" if you try to run one through it.  I have a
qboolean global variable called WorldModelLoaded with an initial value of false,
which is set to true during any call to R_LoadBrushModel, and is used to control
whether this code gets called.  After all the models are loaded I just set it to
false again (R_NewMap is a good place to do this).

You'll note that in some places I have a for loop starting at 1 and going to
mod->numleafs + 1, which looks plain wrong.  Check the definition of model_s in
gl_model.h - numleafs doesn't count leaf 0.  Leaf 0 is always CONTENTS_SOLID, and I
can't see any real use for it aside from if we ever want to return full map visibility
from Mod_LeafPVS - I guess it was used in earlier versions of Quake and just retained.
This is why we always add 1 to the leaf number when going through the PVS bits :-
adding 1 actually gives the *correct* leaf (leaf 0 can never be returned by going
through the PVS bits).

This code is called from R_LoadBrushModel so that the new underwater PVS can be made
available to both the client and the server, meaning that underwater entities are
correctly sent from the server to the client (which doesn't happen with r_novis 1).

If you want to use this code in your own engine, feel free to.  Pretty much all of
what you need is in this comment block; the only other things you need are a few lines
at the end of R_LoadBrushModel to call it, the necessary modifications to Mod_LeafPVS
(just look for it at the top of this file); also do a global search for the 
Mod_SimpleLeafPVS function, and make sure you build your surface polygon data *before*
calling my code (the relevant function is BuildSurfaceDisplayList).  You'll also need
a few new strucure members - you'll find them if you try to compile my code and it
doesn't work, or you could look in my gl_model.h.  I think that's it.

======================================================================================
*/
int mod_visframecount;
mleaf_t *visleaf;
byte fullpvs[MAX_MAP_LEAFS / 8];


// add the specified visdata to the pvs
void AddVisToPVS (byte *visdata, model_t *mod)
{
	int j;

	for (j = 0; j < ((mod->numleafs + 7) / 8); j++)
		fullpvs[j] |= visdata[j];
}


// this one isn't actually used but it's here anyway...
void RemoveVisFromPVS (byte *visdata, model_t *mod)
{
	int j;

	// or it first, then xor it
	for (j = 0; j < ((mod->numleafs + 7) / 8); j++)
	{
		fullpvs[j] |= visdata[j];
		fullpvs[j] ^= visdata[j];
	}
}


// add a full pvs for a given leaf to the pvs
void AddLeafToPVS (model_t *mod, mleaf_t *leaf)
{
	AddVisToPVS (Mod_SimpleLeafPVS (leaf, mod), mod);
}


void Mod_InitDecompressedVis (model_t *mod)
{
	// adding 7 here prevents rounding errors and makes sure that the 
	// vissize is big enough to accomodate all the leafs in the map
	int vissize = (mod->numleafs + 7) >> 3;
	int totalsize;

	totalsize = vissize * mod->numleafs;

	// put on the hunk
	decompressed_vis = Hunk_AllocName (totalsize, loadname);

	// initialize it to nothing visible
	memset (decompressed_vis, 0, totalsize);
}


void Mod_SaveDecompressedVis (model_t *mod, byte *dec, byte *vis)
{
	int vissize = (mod->numleafs + 7) >> 3;

	// write vis into dec - we wont bother recompressing it...
	memcpy (dec, vis, vissize);
}


qboolean PVSHasWater;


void Mod_DoNodalPath (model_t *mod, mnode_t *node)
{
	int i;
	int j;

	mleaf_t *leaf;

	if (node->contents == CONTENTS_SOLID) return;
	if (node->visframe != mod_visframecount) return;

	if (node->contents < 0)
	{
		// mark the surfaces in the leaf
		leaf = (mleaf_t *) node;

		// mark this leaf in the base PVS
		leaf->waterVisFrame = mod_visframecount;

		// only if it has water
		if (!leaf->hasLiquid) return;

		// flag that the PVS does contain water, as we don't even want to bother
		// with further steps if it doesn't (speeds up load times)
		PVSHasWater = true;

		// mark the matching surfs
		for (j = 0; j < leaf->nummarksurfaces; j++)
			if (leaf->firstmarksurface[j]->match)
				leaf->firstmarksurface[j]->match->visframe = mod_visframecount;

		return;
	}

	Mod_DoNodalPath (mod, node->children[0]);
	Mod_DoNodalPath (mod, node->children[1]);
}


void Mod_MarkSurfaceVisframes (mleaf_t *leaf, int count)
{
	int i;
	msurface_t *surf = leaf->firstmarksurface[0];

	for (i = 0; i < leaf->nummarksurfaces; i++, surf++)
		surf->visframe = count;
}


void Mod_AddLeafs (model_t *mod)
{
	int i, j, k;
	byte *vis;

	if (!PVSHasWater) return;

	for (i = 1; i < mod->numleafs + 1; i++)
	{
		if (mod->leafs[i].waterVisFrame == mod_visframecount) continue;
		if (!mod->leafs[i].hasLiquid) continue;

		// we want leafs with different contents to the base visleaf
		if (mod->leafs[i].contents == CONTENTS_EMPTY && visleaf->contents == CONTENTS_EMPTY) continue;
		if (mod->leafs[i].contents != CONTENTS_EMPTY && visleaf->contents != CONTENTS_EMPTY) continue;

		// go through the surfs for a valid leaf not in the base PVS
		for (j = 0; j < mod->leafs[i].nummarksurfaces; j++)
		{
			// has this surf been matched?
			if (mod->leafs[i].firstmarksurface[j]->visframe == mod_visframecount)
			{
				// this leaf was marked as a matched leaf
				// EXCLUSION POSSIBILITY - this will add a whole shit more leafs to the
				// PVS than are neccessary.  If any leaf added as part of this exercise
				// has a water surf which was *not* in the original base PVS the leaf
				// could be excluded.  this may require ages to load though...
				AddLeafToPVS (mod, &mod->leafs[i]);

				// mark this leaf as having been added (set up for second pass)
				// (knock on effect - speeds up this function)
				mod->leafs[i].waterVisFrame = mod_visframecount;
				Mod_MarkSurfaceVisframes (&mod->leafs[i], mod_visframecount);
				mod->volumes[mod->leafs[i].volume].visframe = mod_visframecount;
				vis = Mod_SimpleLeafPVS (&mod->leafs[i], mod);

				// mark the leafs in it's PVS too (set up for second pass)
				// (knock on effect - speeds up this function)
				for (k = 0; k < mod->numleafs; k++)
				{
					if (vis[k >> 3] & (1 << (k & 7)))
					{
						mod->leafs[k + 1].waterVisFrame = mod_visframecount;
						Mod_MarkSurfaceVisframes (&mod->leafs[k + 1], mod_visframecount);
						mod->volumes[mod->leafs[k + 1].volume].visframe = mod_visframecount;
					}
				}

				// we only need one surf here cos we're dealing with leafs
				break;
			}
		}
	}
}


// ALTERNATE STRATEGY
// ------------------
// check all leafs NOT in the original PVS and NOT added to it, check matching water surfs,
// if there is a match in the added PVS add the leaf.  the leaf CAN NOT be in the same volume
// as the visleaf or as any other leaf in the original PVS - should we look at adding a visframe
// to volumes too for checking this???  if the visleaf is CONTENTS_EMPTY the new leafs also have
// to be, if it's not CONTENTS_EMPTY they must also not be (but they can be different otherwise,
// e.g. water and slime).
void Mod_CheckLeafsPass2_2 (model_t *mod)
{
	int i;
	int j;
	int k;
	msurface_t *surf;
	int newleafs = 0;
	byte *vis;

	if (!PVSHasWater) return;

	return;

	// first pass - mark the matching surfs
	for (i = 1; i < mod->numleafs + 1; i++)
	{
		if (mod->leafs[i].waterVisFrame == mod_visframecount) continue;
		if (!mod->leafs[i].hasLiquid) continue;
		if (visleaf->contents == CONTENTS_EMPTY && mod->leafs[i].contents != CONTENTS_EMPTY) continue;
		if (visleaf->contents != CONTENTS_EMPTY && mod->leafs[i].contents == CONTENTS_EMPTY) continue;
		if (mod->volumes[mod->leafs[i].volume].visframe == mod_visframecount) continue;

		for (j = 0, surf = mod->leafs[i].firstmarksurface[0]; j < mod->leafs[i].nummarksurfaces; j++, surf++)
		{
			if ((surf->flags & SURF_DRAWTURB) && surf->match)
			{
				if (surf->match->visframe == mod_visframecount)
				{
					// leaf contains a surf the match of which was added in 
					// pass 1 - mark it and get out.  -1 to differentiate.
					mod->leafs[i].waterVisFrame = mod_visframecount - 1;
					break;
				}
			}
		}
	}

	// pass 2 - add the new leafs
	for (i = 1; i < mod->numleafs + 1; i++)
	{
		if (mod->leafs[i].waterVisFrame == mod_visframecount - 1)
		{
			AddLeafToPVS (mod, &mod->leafs[i]);
			vis = Mod_SimpleLeafPVS (&mod->leafs[i], mod);

			// invalidate this leaf and all leafs in it's PVS
			mod->leafs[i].waterVisFrame = -1;

			for (k = 0; k < mod->numleafs; k++)
			{
				if (vis[k >> 3] & (1 << (k & 7)))
				{
					mod->leafs[k + 1].waterVisFrame = -1;
					newleafs++;
				}
			}
		}
	}

	if (newleafs) Con_Printf ("%i leafs added\n", newleafs);
}


void Mod_CheckLeafsPass2 (model_t *mod)
{
	int i;
	int j;
	int newleafs = 0;
	mleaf_t *leaf;
	msurface_t *surf;

	if (!PVSHasWater) return;

	return;

	for (i = 1; i < mod->numleafs + 1; i++)
	{
		leaf = &mod->leafs[i];

		// not in either the base PVS nor the added PVS
		if (leaf->waterVisFrame != mod_visframecount) continue;

		// doesn't have water
		if (!leaf->hasLiquid) continue;

		// add matches not in the PVS
		for (surf = leaf->firstmarksurface[0], j = 0; j < leaf->nummarksurfaces; j++, surf++)
			if ((surf->flags & SURF_DRAWTURB) && surf->match)
				if (surf->match->visframe != mod_visframecount)
					surf->match->visframe = mod_visframecount - 1;
	}

	// now find the leafs containing the newly marked surfs and add them to the PVS
	for (i = 1; i < mod->numleafs + 1; i++)
	{
		leaf = &mod->leafs[i];

		for (surf = leaf->firstmarksurface[0], j = 0; j < leaf->nummarksurfaces; j++, surf++)
		{
			if (surf->flags & SURF_DRAWTURB)
			{
				if (surf->visframe == mod_visframecount - 1)
				{
					AddLeafToPVS (mod, leaf);
					newleafs++;
					break;
				}
			}
		}
	}

	if (newleafs) Con_Printf ("%i leafs added\n", newleafs);
}


// build the base PVS
void Mod_BuildBasePVS (model_t *mod)
{
	int i;
	mnode_t *node;

	// decrement by 2 so it doesn't interfere with the second pass
	mod_visframecount -= 2;

	mod->volumes[visleaf->volume].visframe = mod_visframecount;

	for (i = 0; i < mod->numleafs; i++)
	{
		if (fullpvs[i >> 3] & (1 << (i & 7)))
		{
			// leaf is visible, so build a nodal path to it
			// mark the visible nodes and volumes
			mod->volumes[mod->leafs[i + 1].volume].visframe = mod_visframecount;
			node = (mnode_t *) &mod->leafs[i + 1];

			do
			{
				// already added
				if (node->visframe == mod_visframecount) break;

				node->visframe = mod_visframecount;
				node = node->parent;
			}
			while (node);
		}
	}

	PVSHasWater = false;
}


void Mod_BuildVLeafCache (model_t *mod)
{
	int i;

	byte *vis;
	byte *dec;

	// no visibilty data - we can bomb out, water translucency will work anyway...!
	if (!mod->visdata) return;

	// use negative numbers so it doesn't interfere with the draw
	mod_visframecount = -2;

	Con_Printf ("Building water translucency...\n");

	Mod_InitDecompressedVis (mod);

	for (i = 1, dec = decompressed_vis; i < mod->numleafs + 1; i++, dec += ((mod->numleafs + 7) >> 3))
	{
		visleaf = &mod->leafs[i];

		// initial PVS has nothing visible
		memset (fullpvs, 0x00, sizeof (fullpvs));

		// get our base visibility
		vis = Mod_SimpleLeafPVS (&mod->leafs[i], mod);
		AddVisToPVS (vis, mod);

		// get our underwater stuff
		Mod_BuildBasePVS (mod);
		Mod_DoNodalPath (mod, mod->nodes);
		Mod_AddLeafs (mod);

		// pass 2 - get the pool in the start map kinda stuff
		Mod_CheckLeafsPass2 (mod);

		// now store out the new visdata
		Mod_SaveDecompressedVis (mod, dec, fullpvs);
		mod->leafs[i].decompressed = dec;
	}
}
/*
======================================================================================

					MH - AUTOMATIC WATER TRANSLUCENCY END

======================================================================================
*/


/*
======================================================================================

					BRUTE FORCE SURFACE MATCHING BEGIN

Every water surface potentially has a matched surface, owing to the way QBSP carves
up leafs.  We find those surfaces with a brute-force search (i.e. checking everything)
with a little bit of subtlety added, and some knowledge of some quirks of QBSP.

======================================================================================
*/
qboolean Mod_CompareSurfs (msurface_t *surf1, msurface_t *surf2)
{
	glpoly_t	*p1;
	float		*v1;
	glpoly_t	*p2;
	float		*v2;
	int			i;
	int			j;

	// don't forget this!!!
	if (surf1->surfNum == surf2->surfNum) return false;

	// already matched (redundant check)
	if (surf1->match) return false;
	if (surf2->match) return false;

	// it's always either one before or 1 after (qbsp quirk)
	// both searches are going forward, so the logic here is sound.  the first check of surf2
	// will always be the one before surf1, the second check the one after.
	if (surf1->surfNum > surf2->surfNum + 1) return false;
	if (surf2->surfNum > surf1->surfNum + 1) return false;

	// same texture
	if (surf1->texinfo->texture->gl_texture->texnum != surf2->texinfo->texture->gl_texture->texnum) return false;

	// both must be turbulent (redundant check)
	if (!((surf1->flags & SURF_DRAWTURB) && (surf2->flags & SURF_DRAWTURB))) return false;

	// same number of polys
	if (surf1->numPolys != surf2->numPolys) return false;

	// if the number of polys are equal we only need to check for p1 as a terminator
	// we only need to check if each poly has the same number of verts here
	for (p1 = surf1->polys, p2 = surf2->polys; p1; p1 = p1->next, p2 = p2->next)
		if (p1->numverts != p1->numverts) return false;

	return true;
}


void Mod_BruteForceSurfMatch (model_t *mod)
{
	msurface_t *surf1;
	msurface_t *surf2;

	int i;
	int j;

	for (i = 0, surf1 = mod->surfaces; i < mod->numsurfaces; i++, surf1++)
	{
		if (!(surf1->flags & SURF_DRAWTURB)) continue;
		if (surf1->match) continue;

		for (j = 0, surf2 = mod->surfaces; j < mod->numsurfaces; j++, surf2++)
		{
			if (!(surf2->flags & SURF_DRAWTURB)) continue;
			if (surf2->match) continue;

			// try to match them
			if (Mod_CompareSurfs (surf1, surf2))
			{
				// match in both directions
				surf1->match = surf2;
				surf2->match = surf1;
			}
		}
	}
}
/*
======================================================================================

					BRUTE FORCE SURFACE MATCHING END

======================================================================================
*/


/*
======================================================================================

					MH - UNDERWATER FOG AND CAUSTICS BEGIN

Uses the world volume data to derive underwater fog and caustic data.  Any underwater
volume is assigned fog and caustics based on the properties of water surfaces in it.
It's a bit clunky for now cos I'm not storing the volume data in a linked list yet.

======================================================================================
*/
msurface_t *volSurf;

void Mod_MarkVolumeSurfs (model_t *mod, int volumeNum)
{
	int i;
	int j;
	int numSurfs = 0;

	//Con_DPrintf ("Setting Water To: %f %f %f\n", volSurf->texinfo->texture->uwater_fog[0], volSurf->texinfo->texture->uwater_fog[1], volSurf->texinfo->texture->uwater_fog[2]);

	for (i = 1; i < mod->numleafs + 1; i++)
	{
		if (mod->leafs[i].volume != volumeNum) continue;

		// mark the leaf
		mod->leafs[i].caustic = volSurf->texinfo->texture;

		for (j = 0; j < mod->leafs[i].nummarksurfaces; j++)
		{
			// mark the surfs as belonging to this volume
			mod->leafs[i].firstmarksurface[j]->volume = volumeNum;
			mod->leafs[i].firstmarksurface[j]->caustic = volSurf->texinfo->texture;

			numSurfs++;
		}
	}
}


void Mod_FindWaterSurfForVolume (model_t *mod, int volumeNum)
{
	int i;
	int j;

	volSurf = NULL;

	for (i = 1; i < mod->numleafs + 1; i++)
	{
		if (mod->leafs[i].volume != volumeNum) continue;

		for (j = 0; j < mod->leafs[i].nummarksurfaces; j++)
		{
			// store any water surfs (except for teleports...)
			if (mod->leafs[i].firstmarksurface[j]->flags & (SURF_DRAWWATER | SURF_DRAWSLIME | SURF_DRAWLAVA))
			{
				// one is all we need...
				volSurf = mod->leafs[i].firstmarksurface[j];
				return;
			}
		}
	}
}


void Mod_FillInPVSData (model_t *mod, mleaf_t *leaf)
{
	// find all leafs in this PVS
	byte *vis = Mod_SimpleLeafPVS (leaf, mod);
	mleaf_t *markLeaf;
	int i;
	int j;

	// go through them and mark them
	for (i = 0; i < mod->numleafs; i++)
	{
		// in this pvs
		if (vis[i >> 3] & (1 << (i & 7)))
		{
			markLeaf = &mod->leafs[i + 1];

			if (!markLeaf->caustic) continue;

			leaf->caustic = markLeaf->caustic;

			// now do the surfs
			for (j = 0; j < leaf->nummarksurfaces; j++)
				leaf->firstmarksurface[j]->caustic = markLeaf->caustic;

			return;
		}
	}
}


void Mod_LoadUWSurfs (model_t *mod)
{
	// rewrite - uses world volume carving to get the data
	int i;
	int j;

	for (i = 1; i < mod->numleafs + 1; i++)
	{
		// already done or doesn't need to be done
		if (mod->leafs[i].volume < 1) continue;
		if (mod->leafs[i].caustic) continue;
		if (mod->leafs[i].contents == CONTENTS_SOLID) continue;
		if (mod->leafs[i].contents == CONTENTS_SKY) continue;
		if (mod->leafs[i].contents == CONTENTS_EMPTY) continue;

		// first pass - find a water surf in this volume
		Mod_FindWaterSurfForVolume (mod, mod->leafs[i].volume);

		// bad data
		if (!volSurf) continue;

		// mark all other surfs in this volume
		Mod_MarkVolumeSurfs (mod, mod->leafs[i].volume);
	}

	// the above code will get 99% of underwater leafs, but may miss a few in some maps,
	// so we do another pass to fill these in
	for (i = 1; i < mod->numleafs + 1; i++)
	{
		if (mod->leafs[i].contents == CONTENTS_SOLID) continue;
		if (mod->leafs[i].contents == CONTENTS_SKY) continue;
		if (mod->leafs[i].contents == CONTENTS_EMPTY) continue;
		if (mod->leafs[i].volume < 1) continue;

		if (!(mod->leafs[i].caustic))
		{
			// no caustic data for an underwater leaf - find all underwater leafs in it's
			// standard PVS and copy the caustic data from one of them
			Mod_FillInPVSData (mod, &mod->leafs[i]);
			if (!(mod->leafs[i].caustic)) Con_DPrintf ("Leaf %i has no caustic data\n");
		}
	}
}
/*
======================================================================================

					MH - UNDERWATER FOG AND CAUSTICS END

======================================================================================
*/


/*
======================================================================================

					MH - WORLDMODEL VOLUME DIVISION BEGIN
					-------------------------------------
Uses world volume division (or world volume carving) to divide the map into seperate
volumes.  A volume is a collection of leafs all of which have the same contents type
and all of which are normally visible from each other in the normal PVS (with no
water translucency).  Tweaked to also handle maps vised for translucent water.  This
is a trivially simple algorithm, using a recursive function with no return condition.

======================================================================================
*/
int currentVolume;
qboolean transVis;

// in theory there could be any number of volumes in a map, but there will never be more than
// there are leafs.  I ran through all of id1 and it maxed out at 104 - but that was a very rare
// case (e3m6) - aside from that the max was 46.  To be safe, I took the 104, increased it to
// the nearest power of 2, then doubled the result...
volume_t worldvols[MAX_MAP_VOLUMES];


void Mod_AllocVolume (model_t *mod, mleaf_t *leaf)
{
	// find all leafs in this PVS
	byte *vis = Mod_SimpleLeafPVS (leaf, mod);
	mleaf_t *markLeaf;
	int i;

	// go through them and mark them
	for (i = 0; i < mod->numleafs; i++)
	{
		// in this pvs
		if (vis[i >> 3] & (1 << (i & 7)))
		{
			markLeaf = &mod->leafs[i + 1];

			// already marked
			if (markLeaf->volume != -1) continue;

			// bad contents
			if (markLeaf->contents != leaf->contents)
			{
				// this condition also indicates that the map has been vised for
				// translucent water
				transVis = true;
				continue;
			}

			// mark it
			markLeaf->volume = currentVolume;

			// recurse for each leaf in the PVS
			Mod_AllocVolume (mod, markLeaf);
		}
	}

	worldvols[currentVolume].caustic = NULL;
}


void Mod_MarkSurfs (model_t *mod)
{
	int i;
	int j;

	mleaf_t *leaf;
	msurface_t *surf;

	// mark which volume a surf belongs to
	for (i = 1; i < mod->numleafs + 1; i++)
	{
		leaf = &mod->leafs[i];

		if (leaf->volume == -1)
		{
			// this doesn't happen
			Con_DPrintf ("Leafnum %i with contents %i has no volume set\n", i, leaf->contents);
			continue;
		}

		surf = leaf->firstmarksurface[0];

		// if a surf is shared between leafs and the leafs are in different volumes 
		// the volumes should be merged
		for (j = 0; j < leaf->nummarksurfaces; j++, surf++)
			if (surf->volume != -1) surf->volume = leaf->volume;
	}

	// check any remaining surfs
	for (i = 0; i < mod->numsurfaces; i++)
	{
		surf = &mod->surfaces[i];

		if (!(surf->flags & SURF_UNDERWATER))
		{
			// any surf which isn't explicitly underwater should be allocated to volume 0
			// (bugs in certain of the ID1 maps have resulted in certain above water surfs
			// being put in underwater leafs)
			surf->volume = 0;

			// explicitly NULL the caustic texture
			surf->caustic = NULL;
		}

		if (surf->volume == -1)
		{
			// this generally doesn't happen either
			surf->volume = 0;
			surf->caustic = NULL;
		}
	}
}


/*
===================
Mod_MergeVolumes

We needed to keep the volumes seperate for surf identification, but to optimise
the render we now merge any with the same properties
===================
*/
void Mod_MergeVolumes (model_t *mod)
{
	int i;
	int j;
	int firstvol;

	volume_t temp[MAX_MAP_VOLUMES];

	msurface_t *surf;
	volume_t *vol;

	// 1. merge all above water surfs into volume 0
	// this step has already been partially done earlier, now we pick up anything left
	for (i = 0; i < mod->numsurfaces; i++)
	{
		surf = &mod->surfaces[i];

		if (!(surf->volume > 0 && (surf->flags & SURF_UNDERWATER)))
		{
			// explicitly above water
			surf->volume = 0;
			surf->caustic = NULL;
		}
	}

	// 2. starting at volume 1, check the properties
	for (j = 0; j < mod->numVolumes; j++)
	{
		// only if a caustic texture has been marked for this volume
		if (mod->volumes[j].chain->caustic)
		{
			// check all surfs to see if they belong here
			for (i = 0; i < mod->numsurfaces; i++)
			{
				surf = &mod->surfaces[i];

				// marked for above water
				if (!surf->volume || !surf->caustic)
				{
					// remark it
					surf->volume = 0;
					surf->caustic = NULL;
					continue;
				}

				// same properties so add the surf to this volume
				if (surf->caustic == mod->volumes[j].chain->caustic) surf->volume = j;
			}
		}

		mod->volumes[j].caustic = mod->volumes[j].chain->caustic;
		mod->volumes[j].visframe = -2;
	}

	// 3. all volumes are towards the end of the list now aside from volume 0.  to
	// reduce the number of texturechains stored for each texture we move them to the
	// front of the list.  first off, go through the surfs and mark all volumes that
	// are now being used (we'll use a crude hack with the visframes for this)
	for (i = 0; i < mod->numsurfaces; i++)
	{
		surf = &mod->surfaces[i];

		mod->volumes[surf->volume].visframe = -1;
	}

	// clear down the temp buffer
	memset (temp, 0, sizeof (temp));

	for (i = 0; i < MAX_MAP_VOLUMES; i++)
		temp[i].visframe = -2;

	firstvol = 0;

	for (i = 0; i < mod->numVolumes; i++)
	{
		// volume has no surfs
		if (mod->volumes[i].visframe == -2) continue;

		// copy the base data
		temp[firstvol].caustic = mod->volumes[i].caustic;
		temp[firstvol].contents = mod->volumes[i].contents;
		temp[firstvol].visframe = -1;

		// mark the surfs
		for (j = 0; j < mod->numsurfaces; j++)
		{
			surf = &mod->surfaces[j];

			if (surf->volume == i) surf->volume = firstvol;
		}

		// next in the temp buffer
		firstvol++;
	}

	// safety - we only allow up to 16 "drawing volumes" in MHQuake, so if any surf
	// has a volume number higher than that we add it to volume 0.  This is highly
	// unlikely, but possible all the same.  A run through all of ID1 gave a max
	// of 4 so 16 seems safe enough.
	for (j = 0; j < mod->numsurfaces; j++)
	{
		surf = &mod->surfaces[j];

		if (surf->volume > 15) surf->volume = 0;
	}

	// copy the buffer back to the map
	for (i = 0; i < MAX_MAP_VOLUMES; i++)
	{
		mod->volumes[i].caustic = temp[i].caustic;
		mod->volumes[i].contents = temp[i].contents;
		mod->volumes[i].visframe = temp[i].visframe;
	}

	// remove this return to get some map info printed
	return;

	// count the drawing volumes used
	for (i = 0, j = 0; j < mod->numVolumes; j++)
	{
		if (mod->volumes[j].visframe == -2) break;
		i++;
	}

	Con_Printf ("%i drawing volumes\n", i);
}


void Mod_WorldVolCarve (model_t *mod)
{
	int i;
	mleaf_t *leaf;

	mod->leafs[0].volume = currentVolume = 0;
	transVis = false;

	// pick a leaf, any leaf.  we have to go through them all so we might as well
	// take them in order :)
	for (i = 1; i < mod->numleafs + 1; i++)
	{
		// OK, got one!!!  Phew!  I was worried there for a minute...
		leaf = &mod->leafs[i];

		if (leaf->contents == CONTENTS_SOLID) leaf->volume = 0;

		// already allocated
		if (leaf->volume != -1) continue;

		// only interested in these
		if (!(leaf->contents == CONTENTS_EMPTY || leaf->contents == CONTENTS_WATER || 
			leaf->contents == CONTENTS_SLIME || leaf->contents == CONTENTS_LAVA))
			continue;

		// allocate a volume to this leaf and move on
		currentVolume++;
		Mod_AllocVolume (mod, leaf);
	}

	// volumes 1 to currentVolume, inclusive, and volume 0
	mod->numVolumes = currentVolume + 1;

	Con_DPrintf ("Carved %i Volumes\n", currentVolume);
	if (transVis) Con_DPrintf ("Map %s has been vised for translucent water!!!\n", mod->name);

	Mod_MarkSurfs (mod);

	// store the volumes
	// putting this on the hunk trashes things in a fairly ugly fashion for some maps, so
	// we just use a great big static array instead.  some day i might investigate further.
	memset (worldvols, 0, sizeof (worldvols));

	for (i = 0; i < mod->numleafs + 1; i++)
	{
		leaf = &mod->leafs[i];

		// chain them so we can easily access the leafs in a volume
		leaf->chain = worldvols[leaf->volume].chain;
		worldvols[leaf->volume].chain = leaf;

		// set the other data (this will always be the same)
		worldvols[leaf->volume].volumenum = leaf->volume;
		worldvols[leaf->volume].contents = leaf->contents;
		worldvols[leaf->volume].visframe = -1;
	}

	mod->volumes = &worldvols[0];
}
/*
======================================================================================

					MH - WORLDMODEL VOLUME DIVISION END

======================================================================================
*/


/*
=================
RadiusFromBounds
=================
*/
float RadiusFromBounds (vec3_t mins, vec3_t maxs)
{
	int		i;
	vec3_t	corner;

	corner[0] = fabs(mins[0]) > fabs(maxs[0]) ? fabs(mins[0]) : fabs(maxs[0]);
	corner[1] = fabs(mins[1]) > fabs(maxs[1]) ? fabs(mins[1]) : fabs(maxs[1]);
	corner[2] = fabs(mins[2]) > fabs(maxs[2]) ? fabs(mins[2]) : fabs(maxs[2]);

	return Length (corner);
}


/*
=================
Mod_TransWaterVisIdentify

Base identification of whether or not a BSP has been vis'ed for translucent water.  This has deliberately
written to not be dependent on any of my other modifications here.  You'll need to add a qboolean called
TransWaterVis to your model_s structure definition.

Using this info, you can then do stuff like toggle translucent water on or off on a map by map basis rather
than having the poor hassled player set translucency.
=================
*/
void Mod_TransWaterVisIdentify (model_t *mod)
{
	int i;
	int j;
	mleaf_t *leaf;
	mleaf_t *visleaf;
	msurface_t **surf;
	byte *vis;

	// initially false
	mod->TransWaterVis = false;

	// only deal with brush models
	if (mod->type != mod_brush)
		return;

	// no visibility data!  in this case we could set to true, cos we will be able to see through water
	// with this map, but we probably shouldn't do so
	if (!mod->visdata)
		return;

	// check all of the leafs
	for (i = 0; i < mod->numleafs; i++)
	{
		// get a pointer to the current leaf
		leaf = &mod->leafs[i];

		// contents < 0 is valid for checking
		if (leaf->contents >= 0) continue;

		// not interested
		if (leaf->contents == CONTENTS_SOLID) continue;

		// check the surfs in the leaf to see if any have water
		for (j = 0, surf = leaf->firstmarksurface; j < leaf->nummarksurfaces; j++, surf++)
		{
			// we have water
			if ((*surf)->texinfo->texture->name[0] == '*')
			{
				goto LeafVisCheck;
			}
		}

		// explicitly continue unless the goto above jumps us past this line
		continue;

LeafVisCheck:;

		// get the PVS
		vis = Mod_LeafPVS (leaf, mod);

		// do another loop through the leafs checking everything in the PVS
		for (j = 0; j < mod->numleafs; j++)
		{
			// in the PVS
			if (vis[j >> 3] & (1 << (j & 7)))
			{
				visleaf = &mod->leafs[j];

				if (leaf->contents == CONTENTS_EMPTY)
				{
					if (visleaf->contents == CONTENTS_WATER || visleaf->contents == CONTENTS_LAVA || visleaf->contents == CONTENTS_SLIME)
					{
						// a single hit is sufficient
						mod->TransWaterVis = true;

						Con_Printf ("Model %s has been vised for translucent water\n", mod->name);

						return;
					}
				}
				else if (leaf->contents == CONTENTS_WATER || leaf->contents == CONTENTS_LAVA || leaf->contents == CONTENTS_SLIME)
				{
					if (visleaf->contents == CONTENTS_EMPTY)
					{
						// a single hit is sufficient
						mod->TransWaterVis = true;

						Con_Printf ("Model %s has been vised for translucent water\n", mod->name);

						return;
					}
				}
			}
		}
	}

	// if we get to here all of the leafs have been checked without a hit, so the map doesn't have
	// translucent water visibility info
	Con_Printf ("Model %s has NOT been vised for translucent water\n", mod->name);
}


/*
=================
Mod_LoadBrushModel
=================
*/
void Mod_LoadBrushModel (model_t *mod, void *buffer)
{
	int			i, j;
	dheader_t	*header;
	dmodel_t 	*bm;
	model_t		*world;

	Con_DPrintf ("loading %s\n", mod->name);

	loadmodel->type = mod_brush;
	
	header = (dheader_t *)buffer;

	i = LittleLong (header->version);

	if (i != BSPVERSION)
	{
		Con_Printf ("Mod_LoadBrushModel: %s has wrong version number (%i should be %i)\n", mod->name, i, BSPVERSION);
		mod->numvertexes = -1;
		return;
	}

	world = mod;

	// swap all the lumps
	mod_base = (byte *)header;

	for (i=0 ; i<sizeof(dheader_t)/4 ; i++)
		((int *)header)[i] = LittleLong ( ((int *)header)[i]);

	sky_tex = NULL;

	Con_DPrintf ("%s\n", loadname);

	Mod_LoadVertexes (&header->lumps[LUMP_VERTEXES]);
	Mod_LoadEdges (&header->lumps[LUMP_EDGES]);
	Mod_LoadSurfedges (&header->lumps[LUMP_SURFEDGES]);
	Mod_LoadTextures (&header->lumps[LUMP_TEXTURES]);
	Mod_PreCheckFaces (&header->lumps[LUMP_FACES]);
	Mod_LoadLighting (&header->lumps[LUMP_LIGHTING]);
	Mod_LoadPlanes (&header->lumps[LUMP_PLANES]);
	Mod_LoadTexinfo (&header->lumps[LUMP_TEXINFO]);
	Mod_LoadFaces (&header->lumps[LUMP_FACES]);
	Mod_LoadMarksurfaces (&header->lumps[LUMP_MARKSURFACES]);
	Mod_LoadVisibility (&header->lumps[LUMP_VISIBILITY]);
	Mod_LoadLeafs (&header->lumps[LUMP_LEAFS]);
	Mod_LoadNodes (&header->lumps[LUMP_NODES]);
	Mod_LoadClipnodes (&header->lumps[LUMP_CLIPNODES]);
	Mod_LoadEntities (&header->lumps[LUMP_ENTITIES]);
	Mod_LoadSubmodels (&header->lumps[LUMP_MODELS]);

	// MH - doesn't work properly...
	// Mod_TransWaterVisIdentify (mod);

	if (sky_tex) R_InitSky (sky_tex);

	Mod_MakeHull0 ();

	mod->numframes = 2;		// regular and alternate animation

	Con_DPrintf ("loaded %s\n", mod->name);

	// set up the submodels (FIXME: this is confusing)
	for (i = 0; i < mod->numsubmodels; i++)
	{
		bm = &mod->submodels[i];

		mod->hulls[0].firstclipnode = bm->headnode[0];

		for (j = 1; j < MAX_MAP_HULLS; j++)
		{
			mod->hulls[j].firstclipnode = bm->headnode[j];
			mod->hulls[j].lastclipnode = mod->numclipnodes-1;
		}

		mod->firstmodelsurface = bm->firstface;
		mod->nummodelsurfaces = bm->numfaces;

		VectorCopy (bm->maxs, mod->maxs);
		VectorCopy (bm->mins, mod->mins);

		mod->radius = RadiusFromBounds (mod->mins, mod->maxs);

		mod->numleafs = bm->visleafs;

		if (i < mod->numsubmodels-1)
		{	// duplicate the basic information
			char	name[10];

			sprintf (name, "*%i", i+1);
			loadmodel = Mod_FindName (name);
			*loadmodel = *mod;
			strcpy (loadmodel->name, name);
			mod = loadmodel;
		}
	}

	if (!WorldModelLoaded)
	{
		// use "world" in here just to make double triple absolutely sure that we work on the correct model!

		// get the base info
		Mod_BruteForceSurfMatch (world);
		Mod_WorldVolCarve (world);

		// now get the rest of it
		if (!transVis) Mod_BuildVLeafCache (world);
		Mod_LoadUWSurfs (world);

		// now merge the volumes
		// we only merge the surfs in them, not the leafs
		// this is important because each volume that is rendered potentially requires
		// a switch in the fogging mode, which tends to hurt the FPS if it's done a lot.
		Mod_MergeVolumes (world);
	}

	WorldModelLoaded = true;
}

/*
==============================================================================

ALIAS MODELS

==============================================================================
*/

struct aliashdr_s	*pheader;

stvert_t	stverts[MAXALIASVERTS];
mtriangle_t	triangles[MAXALIASTRIS];

// a pose is a single set of vertexes.  a frame may be
// an animating sequence of poses
trivertx_t	*poseverts[MAXALIASFRAMES];
int			posenum;

byte		**player_8bit_texels_tbl;
byte		*player_8bit_texels;

/*
=================
Mod_LoadAliasFrame
=================
*/
int aliasbboxmins[3], aliasbboxmaxs[3];

void * Mod_LoadAliasFrame (void * pin, maliasframedesc_t *frame)
{
	trivertx_t		*pframe, *pinframe;
	int				i, j;
	daliasframe_t	*pdaliasframe;
	
	pdaliasframe = (daliasframe_t *)pin;

	strcpy (frame->name, pdaliasframe->name);
	frame->firstpose = posenum;
	frame->numposes = 1;

	for (i=0 ; i<3 ; i++)
	{
		// these are byte values, so we don't have to worry about
		// endianness
		frame->bboxmin.v[i] = pdaliasframe->bboxmin.v[i];
		frame->bboxmax.v[i] = pdaliasframe->bboxmax.v[i];	// bug - was min in the lvalue

		aliasbboxmins[i] = Qmin (frame->bboxmin.v[i], aliasbboxmins[i]);
		aliasbboxmaxs[i] = Qmax (frame->bboxmax.v[i], aliasbboxmaxs[i]);
	}

	pinframe = (trivertx_t *)(pdaliasframe + 1);

	poseverts[posenum] = pinframe;

	posenum++;

	pinframe += pheader->numverts;

	return (void *)pinframe;
}


/*
=================
Mod_LoadAliasGroup
=================
*/
void *Mod_LoadAliasGroup (void * pin,  maliasframedesc_t *frame)
{
	daliasgroup_t		*pingroup;
	int					i, numframes;
	daliasinterval_t	*pin_intervals;
	void				*ptemp;
	
	pingroup = (daliasgroup_t *)pin;

	numframes = LittleLong (pingroup->numframes);

	frame->firstpose = posenum;
	frame->numposes = numframes;

	for (i=0 ; i<3 ; i++)
	{
		// these are byte values, so we don't have to worry about endianness
		frame->bboxmin.v[i] = pingroup->bboxmin.v[i];
		frame->bboxmax.v[i] = pingroup->bboxmax.v[i];

		aliasbboxmins[i] = Qmin (frame->bboxmin.v[i], aliasbboxmins[i]);
		aliasbboxmaxs[i] = Qmax (frame->bboxmax.v[i], aliasbboxmaxs[i]);
	}

	pin_intervals = (daliasinterval_t *)(pingroup + 1);

	frame->interval = LittleFloat (pin_intervals->interval);

	pin_intervals += numframes;

	ptemp = (void *)pin_intervals;

	for (i=0 ; i<numframes ; i++)
	{
		poseverts[posenum] = (trivertx_t *)((daliasframe_t *)ptemp + 1);
		posenum++;

		ptemp = (trivertx_t *)((daliasframe_t *)ptemp + 1) + pheader->numverts;
	}

	return ptemp;
}

//=========================================================

/*
=================
Mod_FloodFillSkin

Fill background pixels so mipmapping doesn't have haloes - Ed
=================
*/

typedef struct
{
	short		x, y;
} floodfill_t;

extern unsigned d_8to24table[];

// must be a power of 2
#define FLOODFILL_FIFO_SIZE 0x1000
#define FLOODFILL_FIFO_MASK (FLOODFILL_FIFO_SIZE - 1)

#define FLOODFILL_STEP( off, dx, dy ) \
{ \
	if (pos[off] == fillcolor) \
	{ \
		pos[off] = 255; \
		fifo[inpt].x = x + (dx), fifo[inpt].y = y + (dy); \
		inpt = (inpt + 1) & FLOODFILL_FIFO_MASK; \
	} \
	else if (pos[off] != 255) fdc = pos[off]; \
}

void Mod_FloodFillSkin( byte *skin, int skinwidth, int skinheight )
{
	byte				fillcolor = *skin; // assume this is the pixel to fill
	floodfill_t			fifo[FLOODFILL_FIFO_SIZE];
	int					inpt = 0, outpt = 0;
	int					filledcolor = -1;
	int					i;

	if (filledcolor == -1)
	{
		filledcolor = 0;

		// attempt to find opaque black
		for (i = 0; i < 256; ++i)
			if (d_8to24table[i] == (255 << 0)) // alpha 1.0
			{
				filledcolor = i;
				break;
			}
	}

	// can't fill to filled color or to transparent color (used as visited marker)
	if ((fillcolor == filledcolor) || (fillcolor == 255))
	{
		//printf( "not filling skin from %d to %d\n", fillcolor, filledcolor );
		return;
	}

	fifo[inpt].x = 0, fifo[inpt].y = 0;
	inpt = (inpt + 1) & FLOODFILL_FIFO_MASK;

	while (outpt != inpt)
	{
		int			x = fifo[outpt].x, y = fifo[outpt].y;
		int			fdc = filledcolor;
		byte		*pos = &skin[x + skinwidth * y];

		outpt = (outpt + 1) & FLOODFILL_FIFO_MASK;

		if (x > 0)				FLOODFILL_STEP( -1, -1, 0 );
		if (x < skinwidth - 1)	FLOODFILL_STEP( 1, 1, 0 );
		if (y > 0)				FLOODFILL_STEP( -skinwidth, 0, -1 );
		if (y < skinheight - 1)	FLOODFILL_STEP( skinwidth, 0, 1 );
		skin[x + skinwidth * y] = fdc;
	}
}

/*
===============
Mod_LoadAllSkins
===============
*/
void *Mod_LoadAllSkins (int numskins, daliasskintype_t *pskintype)
{
	int		i, j, k;
	char	name[64];
	int		s;
	byte	*copy;
	byte	*skin;
	byte	*texels;
	daliasskingroup_t		*pinskingroup;
	int		groupskins;
	daliasskininterval_t	*pinskinintervals;
	byte	*skinCopy;

	skin = (byte *)(pskintype + 1);

	if (numskins < 1 || numskins > MAX_SKINS)
		Sys_Error ("Mod_LoadAliasModel: Invalid # of skins: %d\n", numskins);

	s = pheader->skinwidth * pheader->skinheight;

	// we need to save a copy of the skin
	// whats the biggest size skin we have?  we could potentially get away with
	// a static array here
	skinCopy = (byte *) malloc (s);

	for (i = 0; i < numskins; i++)
	{
		if (pskintype->type == ALIAS_SKIN_SINGLE)
		{
			texels = Hunk_AllocName(s, loadname);
			pheader->texels[i] = texels - (byte *)pheader;
			memcpy (texels, (byte *)(pskintype + 1), s);

			sprintf (name, "%s_%i", loadmodel->name, i);

			memcpy (skinCopy, (byte *)(pskintype + 1), s);

			// only remove fullbrights from viewmodels
			if (!Q_strncmp (loadmodel->name, "progs/v_", 8))
				RemoveFullBrights (skinCopy, s, 255);

			Mod_FloodFillSkin (skinCopy, pheader->skinwidth, pheader->skinheight);

			pheader->gl_texture[i][0] =
			pheader->gl_texture[i][1] =
			pheader->gl_texture[i][2] =
			pheader->gl_texture[i][3] =
				GL_LoadTexture (name, pheader->skinwidth, 
				pheader->skinheight, skinCopy, true, true);

			if (FindFullbrightTexture ((byte *)(pskintype + 1), s))
			{
				memcpy (skinCopy, (byte *)(pskintype + 1), s);

				// convert any non-fullbright pixel to fully transparent
				ConvertPixels (skinCopy, s, 255);

				Mod_FloodFillSkin (skinCopy, pheader->skinwidth, pheader->skinheight);

				// derive a new name for the texture to prevent mismatches
				sprintf (name, "%s_luma_%i", loadmodel->name, i);

				// and now load it - we need alpha for the non-fullbright components
				pheader->fullbright[i][0] = 
				pheader->fullbright[i][1] = 
				pheader->fullbright[i][2] = 
				pheader->fullbright[i][3] = 
					GL_LoadTexture (name, pheader->skinwidth, 
					pheader->skinheight, skinCopy, true, true);
			}
			else pheader->fullbright[i][0] = 
				 pheader->fullbright[i][1] = 
				 pheader->fullbright[i][2] = 
				 pheader->fullbright[i][3] = 0;

			pskintype = (daliasskintype_t *) ((byte *) (pskintype + 1) + s);
		}
		else
		{
			// animating skin group.  yuck.
			Con_DPrintf ("animating skin group.  yuck.\n");
			pskintype++;
			pinskingroup = (daliasskingroup_t *) pskintype;
			groupskins = LittleLong (pinskingroup->numskins);
			pinskinintervals = (daliasskininterval_t *) (pinskingroup + 1);

			pskintype = (void *) (pinskinintervals + groupskins);

			for (j = 0; j < groupskins; j++)
			{
				if (j == 0)
				{
					texels = Hunk_AllocName (s, loadname);
					pheader->texels[i] = texels - (byte *) pheader;
					memcpy (texels, (byte *) (pskintype), s);
				}

				sprintf (name, "%s_%i_%i", loadmodel->name, i,j);

				memcpy (skinCopy, (byte *) (pskintype), s);

				// only remove fullbrights from viewmodels
				if (!Q_strncmp (loadmodel->name, "progs/v_", 8))
					RemoveFullBrights (skinCopy, s, 255);

				Mod_FloodFillSkin (skinCopy, pheader->skinwidth, pheader->skinheight);

				pheader->gl_texture[i][j & 3] = 
					GL_LoadTexture (name, pheader->skinwidth, 
					pheader->skinheight, skinCopy, true, true);

				if (FindFullbrightTexture ((byte *) (pskintype), s))
				{
					memcpy (skinCopy, (byte *) (pskintype), s);

					// convert any non-fullbright pixel to fully transparent
					ConvertPixels (skinCopy, s, 255);

					Mod_FloodFillSkin (skinCopy, pheader->skinwidth, pheader->skinheight);

					// derive a new name for the texture to prevent mismatches
					sprintf (name, "%s_luma_%i_%i", loadmodel->name, i,j);

					// and now load it - we need alpha for the non-fullbright components
					pheader->fullbright[i][j & 3] = 
						GL_LoadTexture (name, pheader->skinwidth, 
						pheader->skinheight, skinCopy, true, true);
				}
				else pheader->fullbright[i][j & 3] = 0;

				pskintype = (daliasskintype_t *) ((byte *) (pskintype) + s);
			}

			k = j;

			for (; j < 4; j++)
			{
				pheader->gl_texture[i][j & 3] = pheader->gl_texture[i][j - k];
				pheader->fullbright[i][j & 3] = pheader->fullbright[i][j - k];
			}
		}
	}

	free (skinCopy);

	return (void *) pskintype;
}


void *Mod_ReLoadAllSkins (model_t *mod, int numskins, daliasskintype_t *pskintype)
{
	int		i, j, k;
	char	name[64];
	int		s;
	byte	*copy;
	byte	*skin;
	daliasskingroup_t		*pinskingroup;
	int		groupskins;
	daliasskininterval_t	*pinskinintervals;
	byte	*skinCopy;

	skin = (byte *)(pskintype + 1);

	if (numskins < 1 || numskins > MAX_SKINS)
		Sys_Error ("Mod_LoadAliasModel: Invalid # of skins: %d\n", numskins);

	s = pheader->skinwidth * pheader->skinheight;

	// we need to save a copy of the skin
	// whats the biggest size skin we have?  we could potentially get away with
	// a static array here
	skinCopy = (byte *) malloc (s);

	for (i = 0; i < numskins; i++)
	{
		if (pskintype->type == ALIAS_SKIN_SINGLE)
		{
			sprintf (name, "%s_%i", mod->name, i);

			memcpy (skinCopy, (byte *)(pskintype + 1), s);

			// only remove fullbrights from viewmodels
			if (!Q_strncmp (mod->name, "progs/v_", 8))
				RemoveFullBrights (skinCopy, s, 255);

			Mod_FloodFillSkin (skinCopy, pheader->skinwidth, pheader->skinheight);

			pheader->gl_texture[i][0] =
			pheader->gl_texture[i][1] =
			pheader->gl_texture[i][2] =
			pheader->gl_texture[i][3] =
				GL_LoadTexture (name, pheader->skinwidth, 
				pheader->skinheight, skinCopy, true, true);

			if (FindFullbrightTexture ((byte *)(pskintype + 1), s))
			{
				memcpy (skinCopy, (byte *)(pskintype + 1), s);

				// convert any non-fullbright pixel to fully transparent
				ConvertPixels (skinCopy, s, 255);

				Mod_FloodFillSkin (skinCopy, pheader->skinwidth, pheader->skinheight);

				// derive a new name for the texture to prevent mismatches
				sprintf (name, "%s_luma_%i", mod->name, i);

				// and now load it - we need alpha for the non-fullbright components
				pheader->fullbright[i][0] = 
				pheader->fullbright[i][1] = 
				pheader->fullbright[i][2] = 
				pheader->fullbright[i][3] = 
					GL_LoadTexture (name, pheader->skinwidth, 
					pheader->skinheight, skinCopy, true, true);
			}
			else pheader->fullbright[i][0] = 
				 pheader->fullbright[i][1] = 
				 pheader->fullbright[i][2] = 
				 pheader->fullbright[i][3] = 0;

			pskintype = (daliasskintype_t *) ((byte *) (pskintype + 1) + s);
		}
		else
		{
			// animating skin group.  yuck.
			pskintype++;
			pinskingroup = (daliasskingroup_t *) pskintype;
			groupskins = LittleLong (pinskingroup->numskins);
			pinskinintervals = (daliasskininterval_t *) (pinskingroup + 1);

			pskintype = (void *) (pinskinintervals + groupskins);

			for (j = 0; j < groupskins; j++)
			{
				sprintf (name, "%s_%i_%i", mod->name, i,j);

				memcpy (skinCopy, (byte *) (pskintype), s);

				// only remove fullbrights from viewmodels
				if (!Q_strncmp (mod->name, "progs/v_", 8))
					RemoveFullBrights (skinCopy, s, 255);

				Mod_FloodFillSkin (skinCopy, pheader->skinwidth, pheader->skinheight);

				pheader->gl_texture[i][j & 3] = 
					GL_LoadTexture (name, pheader->skinwidth, 
					pheader->skinheight, skinCopy, true, true);

				if (FindFullbrightTexture ((byte *) (pskintype), s))
				{
					memcpy (skinCopy, (byte *) (pskintype), s);

					// convert any non-fullbright pixel to fully transparent
					ConvertPixels (skinCopy, s, 255);

					Mod_FloodFillSkin (skinCopy, pheader->skinwidth, pheader->skinheight);

					// derive a new name for the texture to prevent mismatches
					sprintf (name, "%s_luma_%i_%i", mod->name, i,j);

					// and now load it - we need alpha for the non-fullbright components
					pheader->fullbright[i][j & 3] = 
						GL_LoadTexture (name, pheader->skinwidth, 
						pheader->skinheight, skinCopy, true, true);
				}
				else pheader->fullbright[i][j & 3] = 0;

				pskintype = (daliasskintype_t *) ((byte *) (pskintype) + s);
			}

			k = j;

			for (; j < 4; j++)
			{
				pheader->gl_texture[i][j & 3] = pheader->gl_texture[i][j - k];
				pheader->fullbright[i][j & 3] = pheader->fullbright[i][j - k];
			}
		}
	}

	free (skinCopy);

	return (void *) pskintype;
}


//=========================================================================

void Mod_ReloadAliasModel (model_t *mod)
{
	mdl_t *pinmodel;
	daliasskintype_t *pskintype;
	unsigned *buf;
	byte stackbuf[1024];

	buf = (unsigned *) COM_LoadStackFile (mod->name, stackbuf, sizeof(stackbuf));

	pinmodel = (mdl_t *) buf;
	pheader = (struct aliashdr_s *) Mod_Extradata (mod);

	// load the skins - as the mdl was already successfully loaded, we can assume that it will be again
	// (famous last words)
	pskintype = (daliasskintype_t *) &pinmodel[1];
	pskintype = Mod_ReLoadAllSkins (mod, LittleLong (pinmodel->numskins), pskintype);
}


/*
=================
Mod_LoadAliasModel
=================
*/
void Mod_LoadAliasModel (model_t *mod, void *buffer)
{
	int					i, j;
	mdl_t				*pinmodel;
	stvert_t			*pinstverts;
	dtriangle_t			*pintriangles;
	int					version, numframes, numskins;
	int					size;
	daliasframetype_t	*pframetype;
	daliasskintype_t	*pskintype;
	int					start, end, total;

	start = Hunk_LowMark ();

	pinmodel = (mdl_t *)buffer;

	version = LittleLong (pinmodel->version);

	if (version != ALIAS_VERSION)
		Sys_Error ("%s has wrong version number (%i should be %i)",
				 mod->name, version, ALIAS_VERSION);

	// allocate space for a working header, plus all the data except the frames,
	// skin and group info
	size = sizeof (struct aliashdr_s) + (LittleLong (pinmodel->numframes) - 1) * sizeof (pheader->frames[0]);

	pheader = Hunk_AllocName (size, loadname);

	mod->flags = LittleLong (pinmodel->flags);

	// endian-adjust and copy the data, starting with the alias model header
	pheader->boundingradius = LittleFloat (pinmodel->boundingradius);
	pheader->numskins = LittleLong (pinmodel->numskins);
	pheader->skinwidth = LittleLong (pinmodel->skinwidth);
	pheader->skinheight = LittleLong (pinmodel->skinheight);

	if (pheader->skinheight > MAX_LBM_HEIGHT)
		Sys_Error ("model %s has a skin taller than %d", mod->name,
				   MAX_LBM_HEIGHT);

	pheader->numverts = LittleLong (pinmodel->numverts);

	if (pheader->numverts <= 0)
		Sys_Error ("model %s has no vertices", mod->name);

	if (pheader->numverts > MAXALIASVERTS)
		Sys_Error ("model %s has too many vertices", mod->name);

	pheader->numtris = LittleLong (pinmodel->numtris);

	if (pheader->numtris <= 0)
		Sys_Error ("model %s has no triangles", mod->name);

	pheader->numframes = LittleLong (pinmodel->numframes);
	numframes = pheader->numframes;
	if (numframes < 1)
		Sys_Error ("Mod_LoadAliasModel: Invalid # of frames: %d\n", numframes);

	pheader->size = LittleFloat (pinmodel->size) * ALIAS_BASE_SIZE_RATIO;
	mod->synctype = LittleLong (pinmodel->synctype);
	mod->numframes = pheader->numframes;

	for (i=0 ; i<3 ; i++)
	{
		pheader->scale[i] = LittleFloat (pinmodel->scale[i]);
		pheader->scale_origin[i] = LittleFloat (pinmodel->scale_origin[i]);
		pheader->eyeposition[i] = LittleFloat (pinmodel->eyeposition[i]);
	}


	// load the skins
	pskintype = (daliasskintype_t *)&pinmodel[1];
	pskintype = Mod_LoadAllSkins (pheader->numskins, pskintype);

	// load base s and t vertices
	pinstverts = (stvert_t *)pskintype;

	for (i=0 ; i<pheader->numverts ; i++)
	{
		stverts[i].onseam = LittleLong (pinstverts[i].onseam);
		stverts[i].s = LittleLong (pinstverts[i].s);
		stverts[i].t = LittleLong (pinstverts[i].t);
	}

	// load triangle lists
	pintriangles = (dtriangle_t *)&pinstverts[pheader->numverts];

	for (i=0 ; i<pheader->numtris ; i++)
	{
		triangles[i].facesfront = LittleLong (pintriangles[i].facesfront);

		for (j=0 ; j<3 ; j++)
		{
			triangles[i].vertindex[j] =
					LittleLong (pintriangles[i].vertindex[j]);
		}
	}

	// load the frames
	posenum = 0;
	pframetype = (daliasframetype_t *)&pintriangles[pheader->numtris];

	aliasbboxmins[0] = aliasbboxmins[1] = aliasbboxmins[2] = 99999;
	aliasbboxmaxs[0] = aliasbboxmaxs[1] = aliasbboxmaxs[2] = -99999;

	for (i=0 ; i<numframes ; i++)
	{
		aliasframetype_t	frametype;

		frametype = LittleLong (pframetype->type);

		if (frametype == ALIAS_SINGLE)
		{
			pframetype = (daliasframetype_t *)
					Mod_LoadAliasFrame (pframetype + 1, &pheader->frames[i]);
		}
		else
		{
			pframetype = (daliasframetype_t *)
					Mod_LoadAliasGroup (pframetype + 1, &pheader->frames[i]);
		}
	}

	pheader->numposes = posenum;

	mod->type = mod_alias;

	for (i = 0; i < 3; i++)
	{
		mod->mins[i] = aliasbboxmins[i] * pheader->scale[i] + pheader->scale_origin[i];
		mod->maxs[i] = aliasbboxmaxs[i] * pheader->scale[i] + pheader->scale_origin[i];
	}

	mod->radius = RadiusFromBounds (mod->mins, mod->maxs);

	// build the draw lists
	GL_MakeAliasModelDisplayLists (mod, pheader);

	// special effects - set the defaults
	pheader->bolt = false;
	pheader->flicker = false;
	pheader->missile = false;
	pheader->interpolate = true;
	pheader->pulse = false;
	pheader->shambler = false;
	pheader->torch = false;
	pheader->trans = false;
	pheader->powerup = false;
	pheader->translevel = 1.0;

	// glow effects - set the defaults
	pheader->glow = true;
	pheader->radius = 30;
	pheader->glowColours[0] = 0.4;
	pheader->glowColours[1] = 0.2;
	pheader->glowColours[2] = 0.1;

	mod->MHType = MH_UNKNOWN;

	if (!strncmp (mod->name, "progs/flame",11))
	{
		pheader->torch = true;
		pheader->radius = 19.0;
		mod->MHType = MH_FLAME;
	}
	else if (!strncmp (mod->name, "progs/bolt",10))
	{
		pheader->bolt = true;
		pheader->radius = 27;
		pheader->glowColours[0] = 0.05;
		pheader->glowColours[1] = 0.05;
		pheader->glowColours[2] = 0.2;
		mod->MHType = MH_BOLT;
	}
	else if (!strcmp  (mod->name, "progs/missile.mdl"))
	{
		pheader->missile = true;
		mod->MHType = MH_MISSILE;
	}
	else if (!strcmp  (mod->name, "progs/laser.mdl"))
	{
		pheader->glowColours[0] = 0.25;
		pheader->glowColours[1] = 0.025;
		pheader->glowColours[2] = 0.025;
		mod->MHType = MH_LASER;
	}
	else if (!strcmp  (mod->name, "progs/w_spike.mdl"))
	{
		pheader->glowColours[0] = 0.025;
		pheader->glowColours[1] = 0.25;
		pheader->glowColours[2] = 0.025;
		mod->MHType = MH_WSPIKE;
	}
	else if (!strcmp  (mod->name, "progs/v_spike.mdl"))
	{
		pheader->glowColours[0] = 0.25;
		pheader->glowColours[1] = 0.025;
		pheader->glowColours[2] = 0.25;
		mod->MHType = MH_VSPIKE;
	}
	else if (!strcmp  (mod->name, "progs/k_spike.mdl"))
	{
		mod->MHType = MH_KSPIKE;
	}
	else if (!strcmp  (mod->name, "progs/beam.mdl"))
	{
		mod->MHType = MH_BEAM;
	}
	else if (!strcmp  (mod->name, "progs/candle.mdl"))
	{
		pheader->torch = true;
		pheader->radius = 19.0;
		mod->MHType = MH_CANDLE;
	}
	else if (!strcmp  (mod->name, "progs/lantern.mdl"))
	{
		pheader->torch = true;
		pheader->radius = 19.0;
		mod->MHType = MH_LANTERN;
	}
	else if (!strcmp  (mod->name, "progs/lspike.mdl"))
	{
		pheader->glowColours[0] = 0.25;
		pheader->glowColours[1] = 0.025;
		pheader->glowColours[2] = 0.025;
		mod->MHType = MH_LSPIKE;
	}
	else if (!strcmp  (mod->name, "progs/lasrspik.mdl"))
	{
		pheader->glowColours[0] = 0.25;
		pheader->glowColours[1] = 0.025;
		pheader->glowColours[2] = 0.025;
		mod->MHType = MH_LASRSPK;
	}
	else if (!strcmp  (mod->name, "progs/plasma.mdl"))
	{
		pheader->glowColours[0] = 0.05;
		pheader->glowColours[1] = 0.05;
		pheader->glowColours[2] = 0.2;
		mod->MHType = MH_PLASMA;
	}
	else if (!strcmp  (mod->name, "progs/s_light.mdl"))
	{
		pheader->shambler = true;
		pheader->bolt = true;
		pheader->glowColours[0] = 0.05;
		pheader->glowColours[1] = 0.05;
		pheader->glowColours[2] = 0.2;
		mod->MHType = MH_SLIGHT;
	}
	else if (!strcmp  (mod->name, "progs/quaddama.mdl"))
	{
		pheader->pulse = true;
		pheader->powerup = true;
		pheader->glowColours[0] = 0.05;
		pheader->glowColours[1] = 0.05;
		pheader->glowColours[2] = 0.2;
		mod->MHType = MH_QUAD;
	}
	else if (!strcmp  (mod->name, "progs/invulner.mdl"))
	{
		pheader->pulse = true;
		pheader->powerup = true;
		pheader->glowColours[0] = 0.2;
		pheader->glowColours[1] = 0.05;
		pheader->glowColours[2] = 0.05;
		mod->MHType = MH_PENT;
	}
	else pheader->glow = false;

	pheader->trans = true;

	// probably totally unnecessary
	if (!strcmp (mod->name, "progs/player.mdl"))
	{
		pheader->crouch = true;
		mod->MHType = MH_PLAYER;
	}
	else pheader->crouch = false;

	// MH - check for model transparency
	// i KNOW the proper term is "translucent" here, but hell, there ARE more
	// important things in life than splitting hairs over this kinda crap.
	if (!strcmp (mod->name, "progs/wizard.mdl") || !strcmp (mod->name, "progs/h_wizard.mdl"))
	{
		// wizards are harder to see when transparent, so make them more solid
		pheader->translevel = 0.75;	
		mod->MHType = MH_WIZARD;
	}
	else if (!strcmp (mod->name, "progs/laser.mdl"))
	{
		mod->MHType = MH_LASER;

		// lasers are the only other model i'm concerned with
		pheader->translevel = 0.5;
	}
	else if (!strcmp (mod->name, "progs/oldone.mdl"))
	{
		// oh, and this... looks utterly disgusting and annelidic (as it should) :)
		pheader->translevel = 0.85;
	}
	else
	{
		pheader->translevel = 1.0;
		pheader->trans = false;
	}

	// mh - check for shadow casting by model name
	if (pheader->trans)
		pheader->shadow = false;	// no shadow if transparent
	else if (!strcmp  (mod->name, "progs/v_spike.mdl"))
	{
		// just so we don't get these mixed up
	}
	else if (!strncmp (mod->name, "progs/v_",8))
	{
		// don't use a shadow on viewmodels cos they're not suitable
		pheader->shadow = false;
		mod->MHType = MH_VWEAP;
	}
	else pheader->shadow = true;

	// move the complete, relocatable alias model to the cache
	end = Hunk_LowMark ();
	total = end - start;
	
	Cache_Alloc (&mod->cache, total, loadname);

	if (!mod->cache.data)
		return;

	memcpy (mod->cache.data, pheader, total);

	Hunk_FreeToLowMark (start);
}

//=============================================================================

/*
=================
Mod_LoadSpriteFrame
=================
*/
void * Mod_LoadSpriteFrame (void * pin, mspriteframe_t **ppframe, int framenum)
{
	dspriteframe_t		*pinframe;
	mspriteframe_t		*pspriteframe;
	int					i, width, height, size, origin[2];
	unsigned short		*ppixout;
	byte				*ppixin;
	char				name[64];

	pinframe = (dspriteframe_t *)pin;

	width = LittleLong (pinframe->width);
	height = LittleLong (pinframe->height);
	size = width * height;

	pspriteframe = Hunk_AllocName (sizeof (mspriteframe_t),loadname);

	Q_memset (pspriteframe, 0, sizeof (mspriteframe_t));

	*ppframe = pspriteframe;

	pspriteframe->width = width;
	pspriteframe->height = height;
	origin[0] = LittleLong (pinframe->origin[0]);
	origin[1] = LittleLong (pinframe->origin[1]);

	pspriteframe->up = origin[1];
	pspriteframe->down = origin[1] - height;
	pspriteframe->left = origin[0];
	pspriteframe->right = width + origin[0];

	sprintf (name, "%s_%i", loadmodel->name, framenum);
	pspriteframe->gl_texture = GL_LoadTexture (name, width, height, (byte *)(pinframe + 1), true, true);

	return (void *)((byte *)pinframe + sizeof (dspriteframe_t) + size);
}


/*
=================
Mod_LoadSpriteGroup
=================
*/
void * Mod_LoadSpriteGroup (void * pin, mspriteframe_t **ppframe, int framenum)
{
	dspritegroup_t		*pingroup;
	mspritegroup_t		*pspritegroup;
	int					i, numframes;
	dspriteinterval_t	*pin_intervals;
	float				*poutintervals;
	void				*ptemp;

	pingroup = (dspritegroup_t *)pin;

	numframes = LittleLong (pingroup->numframes);

	pspritegroup = Hunk_AllocName (sizeof (mspritegroup_t) +
				(numframes - 1) * sizeof (pspritegroup->frames[0]), loadname);

	pspritegroup->numframes = numframes;

	*ppframe = (mspriteframe_t *)pspritegroup;

	pin_intervals = (dspriteinterval_t *)(pingroup + 1);

	poutintervals = Hunk_AllocName (numframes * sizeof (float), loadname);

	pspritegroup->intervals = poutintervals;

	for (i=0 ; i<numframes ; i++)
	{
		*poutintervals = LittleFloat (pin_intervals->interval);
		if (*poutintervals <= 0.0)
			Sys_Error ("Mod_LoadSpriteGroup: interval<=0");

		poutintervals++;
		pin_intervals++;
	}

	ptemp = (void *)pin_intervals;

	for (i=0 ; i<numframes ; i++)
	{
		ptemp = Mod_LoadSpriteFrame (ptemp, &pspritegroup->frames[i], framenum * 100 + i);
	}

	return ptemp;
}


/*
=================
Mod_LoadSpriteModel
=================
*/
void Mod_LoadSpriteModel (model_t *mod, void *buffer)
{
	int					i;
	int					version;
	dsprite_t			*pin;
	msprite_t			*psprite;
	int					numframes;
	int					size;
	dspriteframetype_t	*pframetype;
	
	pin = (dsprite_t *)buffer;

	version = LittleLong (pin->version);
	if (version != SPRITE_VERSION)
		Sys_Error ("%s has wrong version number "
				 "(%i should be %i)", mod->name, version, SPRITE_VERSION);

	numframes = LittleLong (pin->numframes);

	size = sizeof (msprite_t) +	(numframes - 1) * sizeof (psprite->frames);

	psprite = Hunk_AllocName (size, loadname);

	mod->cache.data = psprite;

	psprite->type = LittleLong (pin->type);
	psprite->maxwidth = LittleLong (pin->width);
	psprite->maxheight = LittleLong (pin->height);
	psprite->beamlength = LittleFloat (pin->beamlength);
	mod->synctype = LittleLong (pin->synctype);
	psprite->numframes = numframes;

	mod->mins[0] = mod->mins[1] = -psprite->maxwidth/2;
	mod->maxs[0] = mod->maxs[1] = psprite->maxwidth/2;
	mod->mins[2] = -psprite->maxheight/2;
	mod->maxs[2] = psprite->maxheight/2;
	
//
// load the frames
//
	if (numframes < 1)
		Sys_Error ("Mod_LoadSpriteModel: Invalid # of frames: %d\n", numframes);

	mod->numframes = numframes;

	pframetype = (dspriteframetype_t *)(pin + 1);

	for (i=0 ; i<numframes ; i++)
	{
		spriteframetype_t	frametype;

		frametype = LittleLong (pframetype->type);
		psprite->frames[i].type = frametype;

		if (frametype == SPR_SINGLE)
		{
			pframetype = (dspriteframetype_t *)
					Mod_LoadSpriteFrame (pframetype + 1,
										 &psprite->frames[i].frameptr, i);
		}
		else
		{
			pframetype = (dspriteframetype_t *)
					Mod_LoadSpriteGroup (pframetype + 1,
										 &psprite->frames[i].frameptr, i);
		}
	}

	mod->type = mod_sprite;

	// certain sprites are now drawn differently
	if (!strcmp (mod->name, "progs/s_explod.spr"))
		psprite->drawType = SPR_EXPLODE;
	else if (!strcmp (mod->name, "progs/s_light.spr"))
		psprite->drawType = SPR_LIGHT;
	else if (!strcmp (mod->name, "progs/s_bubble.spr"))
		psprite->drawType = SPR_BUBBLE;
	else if (!strcmp (mod->name, "progs/s_rain.spr"))
		psprite->drawType = SPR_RAIN;
	else psprite->drawType = SPR_DRAW;
}

//=============================================================================

/*
================
Mod_Print
================
*/
void Mod_Print (void)
{
	int		i;
	model_t	*mod;

	Con_Printf ("Cached models:\n");
	for (i=0, mod=mod_known ; i < mod_numknown ; i++, mod++)
	{
		Con_Printf ("%8p : %s\n",mod->cache.data, mod->name);
	}
}


