/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#ifndef __MODEL__
#define __MODEL__

#include "modelgen.h"
#include "spritegn.h"



/*
==============================================================================

TEXTURE MANAGEMENT

==============================================================================
*/

typedef struct image_s
{
	// linked list pointers
	struct image_s *prev;
	struct image_s *next;

	// gl texture number
	int texnum;

	char identifier[64];

	int width;
	int height;

	qboolean mipmap;
	qboolean alpha;

	qboolean dyndata;
	byte *data;

	int bpp;

	byte md5digest[16];
} image_t;


image_t *GL_FindImageTexnum (int texnum);
image_t *GL_FindImageName (char *identifier);
image_t *GL_AllocImage (int texnum, char *identifier);
image_t *GL_CreateImage (char *identifier, int texnum, int width, int height, byte *data, qboolean mipmap, qboolean alpha);


/*

d*_t structures are on-disk representations
m*_t structures are in-memory

*/

// entity effects

#define	EF_BRIGHTFIELD			1
#define	EF_MUZZLEFLASH 			2
#define	EF_BRIGHTLIGHT 			4
#define	EF_DIMLIGHT 			8


/*
==============================================================================

BRUSH MODELS

==============================================================================
*/


//
// in memory representation
//
// !!! if this is changed, it must be changed in asm_draw.h too !!!
typedef struct
{
	vec3_t		position;
} mvertex_t;

#define	SIDE_FRONT	0
#define	SIDE_BACK	1
#define	SIDE_ON		2


// plane_t structure
// !!! if this is changed, it must be changed in asm_i386.h too !!!
typedef struct mplane_s
{
	vec3_t	normal;
	float	dist;
	byte	type;			// for texture axis selection and fast side tests
	byte	signbits;		// signx + signy<<1 + signz<<1
	byte	pad[2];
} mplane_t;

typedef struct texture_s
{
	char		name[16];
	unsigned	width, height;
	image_t		*gl_texture;

	// in theory we *could* use a seperate texture chain for each volume,
	// but volume merging and some astutely placed Con_Printfs indicates
	// that no more than 4 drawing volumes (i.e. volumes after merging) are
	// used in ID1.  I've upped it to 16 to be utterly safe.
	struct msurface_s	*texturechain[16];

	// chain for vertex lighting
	struct msurface_s	*vertlightchain;

	int			anim_total;
	int			anim_min, anim_max;
	struct texture_s *anim_next;
	struct texture_s *alternate_anims;
	struct texture_s *animCurrent;
	unsigned	offsets[MIPLEVELS];

	image_t *fullbright;
	float uwater_fog[4];

	// additional chains for further optimised drawing
	struct msurface_s	*fbChain;
	struct msurface_s	*lqChain;

	int flags;
} texture_t;


#define	SURF_PLANEBACK		2
#define	SURF_DRAWSKY		4
#define SURF_DRAWSPRITE		8
#define SURF_DRAWTURB		16
#define SURF_DRAWTILED		32
#define SURF_DRAWBACKGROUND	64
#define SURF_UNDERWATER		128

// extra water surface types
#define SURF_DRAWLAVA		256
#define SURF_DRAWSLIME		512
#define SURF_DRAWTELE		1024
#define SURF_DRAWWATER		2048

// !!! if this is changed, it must be changed in asm_draw.h too !!!
typedef struct
{
	unsigned short	v[2];
	unsigned int	cachededgeoffset;
} medge_t;

typedef struct
{
	float		vecs[2][4];
	float		mipadjust;
	texture_t	*texture;
	int			flags;
} mtexinfo_t;


/*
====================
Current Vertex Usage
--------------------
0/1/2		Verts
3/4			Primary TexCoords
5/6			Lightmap TexCoords (water colour r and g)
7/8			128 x 128 scale (water colour b and a)
9/10/11/12	fog colour
13/14		Caustics and water warps (calculated at run time)

A VERTEXSIZE of 16 is used to make each vertex a multiple of 16, 32, and 64
====================
*/
#define	VERTEXSIZE	16

extern int worldverts;
extern float surfarray[];

typedef struct glpoly_s
{
	struct	glpoly_s	*next;
	struct	glpoly_s	*smChain;
	int		saindex;	// index into surfarray for the polys verts
	int		numverts;
	int		flags;			// for SURF_UNDERWATER
	float	*verts;		// pointer to surface vertex array
	vec3_t	midpoint;
} glpoly_t;


// get rid of anything thats not being used in here some day
typedef struct msurface_s
{
	unsigned int			visframe;		// should be drawn when node is crossed

	mplane_t	*plane;
	int			flags;

	int			firstedge;	// look up in model->surfedges[], negative numbers
	int			numedges;	// are backwards edges
	
	short		texturemins[2];
	short		extents[2];

	int			light_s, light_t;	// gl lightmap coordinates
	int			smax, tmax;
	int			lightofs;

	glpoly_t	*polys;				// multiple if warped
	byte		warpindex[2];

	mtexinfo_t	*texinfo;

	// lighting info
	int			dlightframe;
	int		dlightbits;

	// underwater fog and caustics
	texture_t	*caustic;

	int			lightmaptexturenum;
	byte		styles[MAXLIGHTMAPS];
	int			cached_light[MAXLIGHTMAPS];	// values currently used in lightmap
	qboolean	cached_dlight;				// true if dynamic light in cache

	byte		*samples;		// [numstyles*surfsize]
	byte		*stainsamples;	// likewise

	int			numPolys;
	int			surfNum;

	// matching surfs
	struct msurface_s *match;

	// volume carving
	int volume;

	// where is my lightmap???
	byte *lmbase;

	// and my stainmap???  (i could have done a little maths to figure this instead, but want to avoid
	// that at runtime)
	byte *smbase;

	// chain for drawing by the most efficient path
	struct msurface_s *fbChain;
	struct msurface_s *lqChain;
	struct msurface_s *drawChain;
	struct msurface_s *accumSurf;
	struct msurface_s *detailchain;

	// true if the surf has a standard lightstyle
	qboolean staticmap;

	// used for stainmaps cos cached light won't force an update if the style
	// on the first map is 255
	qboolean stained;

	// record if the surf has ever had a stainmap on it
	qboolean everstained;

	// true if the surf has ever been visible in the current session.  used for the automap
	qboolean inClientPVS;

	// software vertex fog
	float		fogBase;
	float		fogRange;
	float		fogAlphaMax;
	float		fogColour[3];
	struct msurface_s *fogSurfs;

	// vertex light
	qboolean	lightmapmodified;

	// chain for vertex lighting
	struct msurface_s	*vertlightchain;

	vec3_t midpoint;

	// i'd prefer to have this in mplane_s but asm_i386.h becomes unhappy.  becuase i have that file also
	// compiled into my quakeasm.lib i can't touch it here, so planeangles goes into the surf struct instead.
	// also, some surfaces are backplanes so we need to invert the normals before we can use them to calculate
	// angles, so planes aren't really valid for this exercise
	vec3_t planeangles;
} msurface_t;



typedef struct glRect_s
{
	byte l,b,w,h;
} glRect_t;


#define LIGHTMAP_BYTES 4

// only update light at 25FPS - reduce this for more frequent uploads
#define LIGHT_UPLOAD 0.04

// match the increase for the marcher fortress engine.
// i know it's not really called that, but it's a convenient term to refer to it by, so i will.
#define MAX_LIGHTMAPS 96

// make sure these 2 are always the same
#define	BLOCK_WIDTH		128
#define	BLOCK_HEIGHT	BLOCK_WIDTH


typedef struct lightmapinfo_s
{
	qboolean modified;
	qboolean active;
	glRect_t rectchange;
	glpoly_t *polys;

	// keep this dynamic to reduce memory overhead
	byte *mapdata;
} lightmapinfo_t;


extern lightmapinfo_t lightmapinfo[];

extern lightmapinfo_t stainmapinfo[];


typedef struct mnode_s
{
	// common with leaf
	int			contents;		// 0, to differentiate from leafs
	unsigned int			visframe;		// node needs to be traversed if current
	
	float		minmaxs[6];		// for bounding box culling

	struct mnode_s	*parent;

	unsigned int lightVisFrame;

	qboolean inClientPVS;

	// node specific
	mplane_t	*plane;
	struct mnode_s	*children[2];

	unsigned short		firstsurface;
	unsigned short		numsurfaces;

	int			nodenum;

	vec3_t		origin;
	qboolean dlvis;
} mnode_t;


typedef struct mleaf_s
{
	// common with node
	int			contents;		// wil be a negative contents number
	unsigned int			visframe;		// node needs to be traversed if current

	float		minmaxs[6];		// for bounding box culling

	struct mnode_s	*parent;

	unsigned int lightVisFrame;

	qboolean inClientPVS;

	// leaf specific
	byte		*compressed_vis;
	efrag_t		*efrags;

	msurface_t	**firstmarksurface;
	int			nummarksurfaces;
	int			key;			// BSP sequence number for leaf's contents
	byte		ambient_sound_level[NUM_AMBIENTS];

	qboolean	hasLiquid;

	// trans water visibility
	int			waterVisFrame;

	// visibility cache
	byte		*decompressed;

	int			leafnum;

	// underwater fog and caustics
	texture_t	*caustic;

	// leaf origin
	vec3_t		origin;

	int			volume;
	struct mleaf_s		*chain;
} mleaf_t;


// moved these here from client.h cos they need the mleaf_t struct
#define	MAX_DLIGHTS		32

typedef struct dlight_s
{
	vec3_t	origin;
	float	radius;
	float	die;				// stop lighting after this time
	float	decay;				// drop this each second
	float	minlight;			// don't add when contributing less
	int		key;
	qboolean	nodevis;
	float colour[3];
	vec3_t angles;
	qboolean active;
	mleaf_t *viewLeaf;
} dlight_t;

extern	dlight_t		cl_dlights[MAX_DLIGHTS];
dlight_t *CL_AllocDlight (int key);


// !!! if this is changed, it must be changed in asm_i386.h too !!!
typedef struct
{
	dclipnode_t	*clipnodes;
	mplane_t	*planes;
	int			firstclipnode;
	int			lastclipnode;
	vec3_t		clip_mins;
	vec3_t		clip_maxs;
} hull_t;

/*
==============================================================================

SPRITE MODELS

==============================================================================
*/


// FIXME: shorten these?
typedef struct mspriteframe_s
{
	int		width;
	int		height;
	float	up, down, left, right;
	image_t	*gl_texture;
} mspriteframe_t;

typedef struct
{
	int				numframes;
	float			*intervals;
	mspriteframe_t	*frames[1];
} mspritegroup_t;

typedef struct
{
	spriteframetype_t	type;
	mspriteframe_t		*frameptr;
} mspriteframedesc_t;


#define SPR_DRAW		0
#define SPR_EXPLODE		1
#define SPR_LIGHT		2
#define SPR_BUBBLE		3
#define SPR_RAIN		4


typedef struct
{
	int					type;
	int					maxwidth;
	int					maxheight;
	int					numframes;
	float				beamlength;		// remove?
	void				*cachespot;		// remove?

	int					drawType;

	mspriteframedesc_t	frames[1];
} msprite_t;


/*
==============================================================================

ALIAS MODELS

Alias models are position independent, so the cache manager can move them.
==============================================================================
*/

typedef struct
{
	int					firstpose;
	int					numposes;
	float				interval;
	trivertx_t			bboxmin;
	trivertx_t			bboxmax;
	int					frame;
	char				name[16];
} maliasframedesc_t;

typedef struct
{
	trivertx_t			bboxmin;
	trivertx_t			bboxmax;
	int					frame;
} maliasgroupframedesc_t;

typedef struct
{
	int						numframes;
	int						intervals;
	maliasgroupframedesc_t	frames[1];
} maliasgroup_t;

// !!! if this is changed, it must be changed in asm_draw.h too !!!
typedef struct mtriangle_s {
	int					facesfront;
	int					vertindex[3];
} mtriangle_t;


#define	MAX_SKINS	32
typedef struct aliashdr_s
{
	int			ident;
	int			version;
	vec3_t		scale;
	vec3_t		scale_origin;
	float		boundingradius;
	vec3_t		eyeposition;
	int			numskins;
	int			skinwidth;
	int			skinheight;
	int			numverts;
	int			numtris;
	int			numframes;
	synctype_t	synctype;
	int			flags;
	float		size;

	int					numposes;
	int					poseverts;
	int					posedata;	// numposes*poseverts trivert_t
	int					commands;	// gl command list with embedded s/t
	image_t				*gl_texture[MAX_SKINS][4];
	image_t				*fullbright[MAX_SKINS][4];
	int					texels[MAX_SKINS];	// only for player skins

	// this stuff has to go before the last struct member cos that's variable sized...

	// alias model header extensions
	qboolean	interpolate;
	qboolean	glow;
	qboolean	flicker;
	qboolean	pulse;
	qboolean	torch;
	qboolean	missile;
	qboolean	shambler;
	qboolean	bolt;
	qboolean	powerup;
	qboolean	shadow;
	float		radius;
	qboolean	crouch;
	vec3_t		glowColours;
	qboolean	trans;
	float		translevel;

	maliasframedesc_t	frames[1];	// variable sized
} aliashdr_t;

#define	MAXALIASVERTS	1024
#define	MAXALIASFRAMES	256
#define	MAXALIASTRIS	2048
extern	struct aliashdr_s	*pheader;
extern	stvert_t	stverts[MAXALIASVERTS];
extern	mtriangle_t	triangles[MAXALIASTRIS];
extern	trivertx_t	*poseverts[MAXALIASFRAMES];

//===================================================================

//
// Whole model
//

typedef enum {mod_brush, mod_sprite, mod_alias} modtype_t;

#define	EF_ROCKET	1			// leave a trail
#define	EF_GRENADE	2			// leave a trail
#define	EF_GIB		4			// leave a trail
#define	EF_ROTATE	8			// rotate (bonus items)
#define	EF_TRACER	16			// green split trail
#define	EF_ZOMGIB	32			// small blood trail
#define	EF_TRACER2	64			// orange split trail + rotate
#define	EF_TRACER3	128			// purple trail

typedef struct MHTeleports_s
{
	char model[64];
	char targ[64];
	float destorigin[3];
	float destangles[3];
	qboolean active;
} MHTeleports_t;


// volume division support stuff
typedef struct volume_s
{
	mleaf_t *chain;
	int volumenum;
	int contents;
	unsigned int visframe;

	struct texture_s *caustic;
} volume_t;


#define MH_UNKNOWN	0
#define MH_FLAME	1
#define MH_BOLT		2
#define MH_MISSILE	3
#define MH_LASER	4
#define MH_WSPIKE	5
#define MH_VSPIKE	6
#define MH_KSPIKE	7
#define MH_BEAM		8
#define MH_CANDLE	9
#define MH_LANTERN	10
#define MH_LSPIKE	11
#define MH_LASRSPK	12
#define MH_PLASMA	13
#define MH_SLIGHT	14
#define MH_QUAD		15
#define MH_PENT		16
#define MH_PLAYER	17
#define MH_WIZARD	18
#define MH_VWEAP	19

typedef struct model_s
{
	char		name[MAX_QPATH];
	qboolean	needload;		// bmodels and sprites don't cache normally

	modtype_t	type;
	int			numframes;
	synctype_t	synctype;
	
	int			flags;

//
// volume occupied by the model graphics
//		
	vec3_t		mins, maxs;
	float		radius;

//
// solid volume for clipping 
//
	qboolean	clipbox;
	vec3_t		clipmins, clipmaxs;

//
// brush model
//
	int			firstmodelsurface, nummodelsurfaces;

	int			numsubmodels;
	dmodel_t	*submodels;

	int			numplanes;
	mplane_t	*planes;

	int			numleafs;		// number of visible leafs, not counting 0
	mleaf_t		*leafs;

	int			numvertexes;
	mvertex_t	*vertexes;

	int			numedges;
	medge_t		*edges;

	int			numnodes;
	mnode_t		*nodes;

	int			numtexinfo;
	mtexinfo_t	*texinfo;

	int			numsurfaces;
	msurface_t	*surfaces;

	int			numsurfedges;
	int			*surfedges;

	int			numclipnodes;
	dclipnode_t	*clipnodes;

	int			nummarksurfaces;
	msurface_t	**marksurfaces;

	hull_t		hulls[MAX_MAP_HULLS];

	int			numtextures;
	texture_t	**textures;

	byte		*visdata;

	byte		*lightdata;
	int			lightdatasize;		// * 3 for RGB if necessary
	byte		*staindata;

	// water translucency
	qboolean	needDraw;

	char		*entities;

	qboolean	PulseEffect;

	int			numVolumes;
	volume_t	*volumes;
	mleaf_t		*modelleaf;
	vec3_t		origin;
	int			entLumpSize;

	MHTeleports_t *portaldata;

	// accumulated surfaces for the entire model
	msurface_t *accumSurf;

	int			MHType;

	float		minmaxx[2], minmaxy[2], minmaxz[2];
	float		xSize;
	float		ySize;

	struct msurface_s *fogSurfs;

	qboolean TransWaterVis;

	vec3_t			lightspot;
	struct mplane_s *lightplane;

	// additional model data
	cache_user_t	cache;		// only access through Mod_Extradata
} model_t;


// litfile header - darkplaces standard
typedef struct litheader_s
{
	char ident[4];
	int version;
} litheader_t;



//============================================================================

void	Mod_Init (void);
void	Mod_ClearAll (void);
model_t *Mod_ForName (char *name, qboolean crash);
void	*Mod_Extradata (model_t *mod);	// handles caching
void	Mod_TouchModel (char *name);

mleaf_t *Mod_PointInLeaf (float *p, model_t *model);
byte	*Mod_LeafPVS (mleaf_t *leaf, model_t *model);
byte	*Mod_SimpleLeafPVS (mleaf_t *leaf, model_t *model);

extern qboolean emboss;

#endif	// __MODEL__
