/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// r_light.c

#include "quakedef.h"
#include "glext.h"

int		lightmap_textures = 0;
int		stainmap_textures = 0;


unsigned blocklightcolours[3][18*18];
qboolean blocklightinit = false;


// for lightmaps
int	current_lightmap_texture;

int			allocated[BLOCK_WIDTH];


lightmapinfo_t lightmapinfo[MAX_LIGHTMAPS];

lightmapinfo_t stainmapinfo[MAX_LIGHTMAPS];


/*
==================
R_AnimateLight
==================
*/
void R_AnimateLight (void)
{
	int i, j, k, l;

	// light animations
	// 'm' is normal light, 'a' is no light, 'z' is double bright
	i = (int) (cl.time * 10);

	for (j = 0; j < MAX_LIGHTSTYLES; j++)
	{
		if (!cl_lightstyle[j].length)
		{
			// 264 is the proper "normal" lightstyle
			d_lightstylevalue[j] = 264;
			continue;
		}

		k = i % cl_lightstyle[j].length;

		k = cl_lightstyle[j].map[k] - 'a';
		k = k * 22;

		d_lightstylevalue[j] = k;
	}
}


/*
=============================================================================

  DYNAMIC LIGHTS

=============================================================================
*/

/*
=============
R_MarkLights

marks the "possible" hits for each dlight - we won't know for sure until we check the surfs
=============
*/
void R_MarkLights (dlight_t *light, int bit, mnode_t *node, qboolean bmnode)
{
	mplane_t	*splitplane;
	float		dist;
	msurface_t	*surf;
	int			i, sidebit;

	// R_CullBox is going too far here - the amount of checking is more expensive
	// than a few falsely added nodes
	if (node->contents < 0) return;
	if (node->contents == CONTENTS_SOLID) return;

	// if the node isn't visible none of it's children will be either so don't check any
	// further (the exception is brushmodels where we check everything)
	if (!(node->visframe == r_visframecount || bmnode)) return;

	splitplane = node->plane;

	dist = DotProduct (light->origin, splitplane->normal) - splitplane->dist;

	if (dist > light->radius)
	{
		// outside the radius of the dlight
		R_MarkLights (light, bit, node->children[0], bmnode);
		return;
	}

	if (dist < -light->radius)
	{
		// outside the radius of the dlight
		R_MarkLights (light, bit, node->children[1], bmnode);
		return;
	}

	if (node->numsurfaces)
	{
		if (dist >= 0)
			sidebit = 0;
		else sidebit = SURF_PLANEBACK;

		// mark the polygons
		surf = cl.worldmodel->surfaces + node->firstsurface;

		for (i = 0; i < node->numsurfaces; i++, surf++)
		{
			// wrong side of the node
			if ((surf->flags & SURF_PLANEBACK) != sidebit) continue;
			if (surf->flags & (SURF_DRAWTURB | SURF_DRAWSKY)) continue;

			if (surf->dlightframe != r_framecount)
			{
				// hit by this light
				surf->dlightbits = 0;
				surf->dlightframe = r_framecount;
			}

			surf->dlightbits |= bit;
		}
	}

	R_MarkLights (light, bit, node->children[0], bmnode);
	R_MarkLights (light, bit, node->children[1], bmnode);
}


/*
==============
R_PushDlights

mark the possible hits for all dlights - we won't know for sure until we check the surfs
==============
*/
void R_PushDlights (mnode_t *node, qboolean bmnode)
{
	int i;
	dlight_t *dl = cl_dlights;

	for (i = 0; i < MAX_DLIGHTS; i++, dl++)
	{
		if (dl->die < cl.time || !dl->radius) continue;

		R_MarkLights (dl, (1 << i), node, bmnode);
	}
}


/*
=============================================================================

LIGHT SAMPLING

=============================================================================
*/

mplane_t		*lightplane;
vec3_t			lightspot;
vec3_t			pointColour;


// go here
int RecursiveLightPoint (mnode_t *node, vec3_t start, vec3_t end)
{
	int			r;
	float		front, back, frac;
	int			side;
	mplane_t	*plane;
	vec3_t		mid;
	msurface_t	*surf;
	int			s, t, ds, dt;
	int			i;
	mtexinfo_t	*tex;
	byte		*lightmap;
	float		scale;
	int			maps;

	if (node->contents < 0)
		return -1;		// didn't hit anything

	// calculate mid point
	plane = node->plane;

	front = DotProduct (start, plane->normal) - plane->dist;
	back = DotProduct (end, plane->normal) - plane->dist;

	side = front < 0;

	if ((back < 0) == side)
		return RecursiveLightPoint (node->children[side], start, end);
	
	frac = front / (front-back);
	mid[0] = start[0] + (end[0] - start[0]) * frac;
	mid[1] = start[1] + (end[1] - start[1]) * frac;
	mid[2] = start[2] + (end[2] - start[2]) * frac;

	// go down front side	
	r = RecursiveLightPoint (node->children[side], start, mid);

	if (r >= 0)
		return r;		// hit something

	if ((back < 0) == side)
		return -1;		// didn't hit anuthing

	// check for impact on this node
	VectorCopy (mid, lightspot);

	surf = cl.worldmodel->surfaces + node->firstsurface;

	for (i = 0; i < node->numsurfaces; i++, surf++)
	{
		if (surf->flags & (SURF_DRAWTILED | SURF_DRAWTURB | SURF_DRAWSKY))
			continue;	// no lightmaps

		tex = surf->texinfo;

		s = DotProduct (mid, tex->vecs[0]) + tex->vecs[0][3];
		t = DotProduct (mid, tex->vecs[1]) + tex->vecs[1][3];

		if (s < surf->texturemins[0] || t < surf->texturemins[1])
			continue;

		ds = s - surf->texturemins[0];
		dt = t - surf->texturemins[1];

		if (ds > surf->extents[0] || dt > surf->extents[1])
			continue;

		if (!surf->samples)
			return -1;

		ds >>= 4;
		dt >>= 4;

		// MHQuake does stainmaps seperately so that they're not affected by any dynamic light changes
		lightmap = surf->samples;

		pointColour[0] = 0;
		pointColour[1] = 0;
		pointColour[2] = 0;

		if (lightmap)
		{
			lightmap += 3 * (dt * (surf->smax) + ds);

			for (maps = 0 ; maps < MAXLIGHTMAPS && surf->styles[maps] != 255 ; maps++)
			{
				scale = d_lightstylevalue[surf->styles[maps]];

				pointColour[0] += lightmap[0] * scale * ONEOVER255;
				pointColour[1] += lightmap[1] * scale * ONEOVER255;
				pointColour[2] += lightmap[2] * scale * ONEOVER255;

				lightmap += 3 * ((surf->smax) * (surf->tmax));
			}
		}

		// store the plane for angled shadows
		lightplane = surf->plane;

		// store lightspot[2] at the surface midpoint to get the exact hit point
		// no - it fucks with the slopey surf thing
		//lightspot[2] = surf->midpoint[2];

		return 1;
	}

	// go down back side
	return RecursiveLightPoint (node->children[!side], mid, end);
}


void R_LightPoint (vec3_t p, vec3_t colour)
{
	vec3_t		start;
	vec3_t		end;
	int			r;
	int			lnum;
	vec3_t		dist;
	float		add;
	int			x;
	model_t		*mod;
	model_t		*shortest;

	lightplane = NULL;

	if (!cl.worldmodel->lightdata)
	{
		colour[0] = 255;
		colour[1] = 255;
		colour[2] = 255;

		return;
	}

	// set up end point
	// this must be done seperately for each model checked as it's passed as a pointer to
	// RecursiveLightPoint, and therefore may have it's value changed
	end[0] = p[0];
	end[1] = p[1];
	end[2] = p[2] - 2048;

	// do the world first
	r = RecursiveLightPoint (cl.worldmodel->nodes, p, end);

	if (lightplane)
	{
		// copy out the data
		cl.worldmodel->lightplane = lightplane;

		cl.worldmodel->lightspot[0] = lightspot[0];
		cl.worldmodel->lightspot[1] = lightspot[1];
		cl.worldmodel->lightspot[2] = lightspot[2];
	}
	else
	{
		// set to valid data for shortest path testing
		cl.worldmodel->lightplane = NULL;
		cl.worldmodel->lightspot[2] = -1000000;
	}

	// find the shortest path - this will be the model with the highest value in lightspot[2]
	// init it to the worldmodel first
	shortest = cl.worldmodel;

	// now check the brush models
	for (x = 0; x < cl_numvisedicts; x++)
	{
		mod = cl_visedicts[x]->model;

		if (mod->type != mod_brush) continue;

		if (mod->name[0] != '*') continue;

		// set up start point
		// offset the start [0] and [1] points by entity origin to do func_train bmodels
		// this is positioned above the model to account for moving bmodels (note the "-" in [2])
		start[0] = p[0] + cl_visedicts[x]->origin[0];
		start[1] = p[1] + cl_visedicts[x]->origin[1];
		start[2] = p[2] - cl_visedicts[x]->origin[2];

		// set up end point
		// this must be done seperately for each model checked as it's passed as a pointer to
		// RecursiveLightPoint, and therefore may have it's value changed
		// offset the end [0] and [1] points by entity origin to do func_train bmodels
		end[0] = p[0] + cl_visedicts[x]->origin[0];
		end[1] = p[1] + cl_visedicts[x]->origin[1];
		end[2] = p[2] - 2048 + cl_visedicts[x]->origin[2];

		// NULL it again
		lightplane = NULL;

		r = RecursiveLightPoint (mod->nodes + mod->hulls[0].firstclipnode, start, end);

		if (lightplane)
		{
			// copy out the data
			mod->lightplane = lightplane;

			mod->lightspot[0] = lightspot[0];
			mod->lightspot[1] = lightspot[1];

			// add origin[2] cos the bmodel may move!!!
			// it's official.  i'm a stupid twit.  i was using the origin of the *model* originally,
			// not of the *entity*.  this works just fine.
			mod->lightspot[2] = lightspot[2] + cl_visedicts[x]->origin[2];

			// check for shortest path
			if (mod->lightspot[2] > shortest->lightspot[2]) shortest = mod;
		}
		else mod->lightplane = NULL;
	}

	// copy the shortest out to the globals
	lightplane = shortest->lightplane;

	lightspot[0] = shortest->lightspot[0];
	lightspot[1] = shortest->lightspot[1];
	lightspot[2] = shortest->lightspot[2];

	if (!lightplane)
	{
		// hit nothing
		colour[0] = 0;
		colour[1] = 0;
		colour[2] = 0;
	}
	else
	{
		colour[0] = pointColour[0];
		colour[1] = pointColour[1];
		colour[2] = pointColour[2];
	}

	// add dynamic lights
	for (lnum = 0; lnum < MAX_DLIGHTS; lnum++)
	{
		if (cl_dlights[lnum].die >= cl.time)
		{
			VectorSubtract (p, cl_dlights[lnum].origin, dist);

			add = cl_dlights[lnum].radius - Length (dist);

			if (add > 0)
			{
				colour[0] += (cl_dlights[lnum].colour[0] * add);
				colour[1] += (cl_dlights[lnum].colour[1] * add);
				colour[2] += (cl_dlights[lnum].colour[2] * add);
			}
		}
	}
}



/*
=============================================================================

  LIGHTMAP ALLOCATION

=============================================================================
*/
extern model_t *loadmodel;

#define CDIV  200
#define CMULT 200


byte BLTable[262144];


/*
================
InitBLTable

used for scaling the lighting range and gives R_BuildLightmap a lookup table to work from.
To make things brighter increase CMULT above towards 255, to make them darker increase
CDIV towards 255.  Any other operation will probably have weird effects.  In theory
this table *should* be 4 (MAXLIGHTMAPS) * 255 (max value in lightmaps) * 550 (max that a
style can scale by), or 561000, so I'm taking a wee risk with the lower size, but I've
never seen MHQuake crash because of this.  Ideally if there was a short data type that
clamped rather than wrapped I'd use that for blocklights... sigh...
================
*/
void InitBLTable (void)
{
	int i, j;
	float k;

	for (i = 0; i < 262144; i++)
	{
		j = i >> 7;

		if (j > 255) j = 255;

		k = ((float) j) / CDIV;
		k *= k;
		j = k * CMULT;

		if (j > 255) j = 255;

		BLTable[i] = j;
	}
}


/*
===============
R_BuildLightMap

Combine and scale multiple lightmaps into the 8.8 format in blocklights
===============
*/
void R_BuildLightMap (msurface_t *surf, byte *dest, int stride, byte *basedata, qboolean staintrick)
{
	int			t;
	int			i, j, size;
	byte		*lightmap;
	unsigned	scale;
	int			maps;
	int			lightadj[4];
	unsigned	*bl;
	unsigned	*blcr, *blcg, *blcb;
	int			r, g, b, a;
	int			style;

	size = surf->smax * surf->tmax;
	lightmap = basedata;

	// clear to no light
	if (!blocklightinit || staintrick)
	{
		for (i = 0; i < size; i++)
			blocklightcolours[0][i] = 
			blocklightcolours[1][i] = 
			blocklightcolours[2][i] = 0;
	}

	// trick R_BuildLightmap into not respecting lightstyles!!!
	if (staintrick && lightmap)
	{
		// multiplier for normal intensity light
		scale = 264;
		goto ImplementStainTrick;
	}

	// add all the lightmaps
	if (lightmap)
	{
		for (maps = 0; maps < MAXLIGHTMAPS && surf->styles[maps] != 255; maps++)
		{
			style = surf->styles[maps];
			scale = d_lightstylevalue[style];
			surf->cached_light[maps] = scale;	// 8.8 fraction

			// scale can NEVER be 1... (it can be 0 though...)
			if (scale)
			{
ImplementStainTrick:;
				for (i = 0, j = 0; i < size; i++)
				{
					blocklightcolours[0][i] += lightmap[j++] * scale;
					blocklightcolours[1][i] += lightmap[j++] * scale;
					blocklightcolours[2][i] += lightmap[j++] * scale;
				}

				if (staintrick) goto store;
			}

			lightmap += size * 3;	// skip to next lightmap
		}
	}

	// bound, invert, and shift
store:
	stride -= (surf->smax << 2);

	blcr = blocklightcolours[0];
	blcg = blocklightcolours[1];
	blcb = blocklightcolours[2];

	// RGBA lightmaps
	for (i = 0; i < surf->tmax ; i++, dest += stride)
	{
		for (j = 0; j < surf->smax; j++)
		{
			// we're taking a BIIIIG risk here, but in practice they never get
			// higher than 170,000 or so, so we can probably afford to
			dest[0] = BLTable[*blcr++];
			dest[1] = BLTable[*blcg++];
			dest[2] = BLTable[*blcb++];
			dest[3] = 255;

			dest += 4;
		}
	}
}


static void R_MarkActiveLoadedStainmaps (msurface_t *surf, byte *dest, int stride)
{
	int i;
	int j;

	// initial parameters
	surf->everstained = false;
	stainmapinfo[surf->lightmaptexturenum].active = false;

	stride -= (surf->smax << 2);

	// RGBA lightmaps
	for (i = 0; i < surf->tmax ; i++, dest += stride)
	{
		for (j = 0; j < surf->smax; j++)
		{
			if ((dest[0] != 255 && dest[0] != 0) || (dest[1] != 255 && dest[1] != 0) || (dest[2] != 255 && dest[2] != 0))
			{
				// just one hit is all we need.
				surf->everstained = true;
				stainmapinfo[surf->lightmaptexturenum].active = true;

				return;
			}

			dest += 4;
		}
	}
}


static void LM_InitBlock (void)
{
	// clear the block to no light
	memset (allocated, 0, sizeof (allocated));

	// allocate memory on the hunk for the mapdata - both lightmap and stainmap must be done
	lightmapinfo[current_lightmap_texture].mapdata = (byte *) Hunk_AllocName (BLOCK_WIDTH * BLOCK_HEIGHT * LIGHTMAP_BYTES, va ("lm_%02i", current_lightmap_texture));
	stainmapinfo[current_lightmap_texture].mapdata = (byte *) Hunk_AllocName (BLOCK_WIDTH * BLOCK_HEIGHT * LIGHTMAP_BYTES, va ("sm_%02i", current_lightmap_texture));
}


static void LM_UploadBlock (void)
{
	image_t *mht;

	glBindTexture (GL_TEXTURE_2D, lightmap_textures + current_lightmap_texture);

	glTexParameteri (GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);

	glTexImage2D (GL_TEXTURE_2D,
				  0,
				  LIGHTMAP_BYTES,
				  BLOCK_WIDTH,
				  BLOCK_HEIGHT,
				  0,
				  GL_RGBA,
				  GL_UNSIGNED_BYTE,
				  lightmapinfo[current_lightmap_texture].mapdata);

	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	lightmapinfo[current_lightmap_texture].modified = false;

	// so that the surf coords will always be greater or less than as appropriate
	lightmapinfo[current_lightmap_texture].rectchange.l = BLOCK_WIDTH;
	lightmapinfo[current_lightmap_texture].rectchange.b = BLOCK_HEIGHT;
	lightmapinfo[current_lightmap_texture].rectchange.w = 0;
	lightmapinfo[current_lightmap_texture].rectchange.h = 0;

	mht = GL_FindImageTexnum (lightmap_textures + current_lightmap_texture);

	if (!mht) Sys_Error ("Could not find lightmap %i\n", lightmap_textures + current_lightmap_texture);

	mht->data = lightmapinfo[current_lightmap_texture].mapdata;

	// now do the same for the stainmap texture (the texture data in this case will already have been filled
	// in - here we are just uploading it as a texture and setting some parameters for it)
	glBindTexture (GL_TEXTURE_2D, stainmap_textures + current_lightmap_texture);

	glTexParameteri (GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);

	glTexImage2D (GL_TEXTURE_2D,
				  0,
				  LIGHTMAP_BYTES,
				  BLOCK_WIDTH,
				  BLOCK_HEIGHT,
				  0,
				  GL_RGBA,
				  GL_UNSIGNED_BYTE,
				  stainmapinfo[current_lightmap_texture].mapdata);

	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	stainmapinfo[current_lightmap_texture].modified = false;
	stainmapinfo[current_lightmap_texture].active = false;

	stainmapinfo[current_lightmap_texture].polys = NULL;

	// so that the surf coords will always be greater or less than as appropriate
	stainmapinfo[current_lightmap_texture].rectchange.l = BLOCK_WIDTH;
	stainmapinfo[current_lightmap_texture].rectchange.b = BLOCK_HEIGHT;
	stainmapinfo[current_lightmap_texture].rectchange.w = 0;
	stainmapinfo[current_lightmap_texture].rectchange.h = 0;

	mht = GL_FindImageTexnum (stainmap_textures + current_lightmap_texture);

	if (!mht) Sys_Error ("Could not find stainmap %i\n", stainmap_textures + current_lightmap_texture);

	mht->data = stainmapinfo[current_lightmap_texture].mapdata;

	if (++current_lightmap_texture >= MAX_LIGHTMAPS)
		Host_Error ("Error: MAX_LIGHTMAPS exceeded\n");
}


static qboolean LM_AllocBlock (int w, int h, int *x, int *y)
{
	int i, j;
	int best, best2;

	best = BLOCK_HEIGHT;

	for (i = 0; i < BLOCK_WIDTH - w; i++)
	{
		best2 = 0;

		for (j = 0; j < w; j++)
		{
			if (allocated[i + j] >= best) break;

			if (allocated[i + j] > best2)
				best2 = allocated[i + j];
		}

		if (j == w)
		{
			// this is a valid spot
			*x = i;
			*y = best = best2;
		}
	}

	if (best + h > BLOCK_HEIGHT)
		return false;

	for (i = 0; i < w; i++)
		allocated[*x + i] = best + h;

	return true;
}


/*
=================
GL_CreateSurfaceLightmap

Stainmaps use the exact same data structures as lightmaps, so the pointers and map numbers for them are also
the same.  There is no need to calculate them a second time, as the same data as has been calculated for
lightmaps is also valid for them.
=================
*/
void GL_CreateSurfaceLightmap (msurface_t *surf)
{
	byte *base;
	vec3_t vec;
	float s, t;
	int i;
	float *v;

	// is this check redundant???
	if (surf->flags & (SURF_DRAWSKY | SURF_DRAWTURB))
		return;

	// allocate space in a lightmap (also fills in light_s and light_t)
	if (!LM_AllocBlock (surf->smax, surf->tmax, &surf->light_s, &surf->light_t))
	{
		LM_UploadBlock ();
		LM_InitBlock ();

		if (!LM_AllocBlock (surf->smax, surf->tmax, &surf->light_s, &surf->light_t))
			Sys_Error ("Consecutive calls to LM_AllocBlock failed");
	}

	// calculate lightmap texture coordinates
	for (i = 0, v = surf->polys->verts; i < surf->polys->numverts; i++, v += VERTEXSIZE)
	{
		vec[0] = v[0];
		vec[1] = v[1];
		vec[2] = v[2];

		s = DotProduct (vec, surf->texinfo->vecs[0]) + surf->texinfo->vecs[0][3];
		s -= surf->texturemins[0];
		s += surf->light_s * 16;
		s += 8;
		s /= BLOCK_WIDTH * 16;

		t = DotProduct (vec, surf->texinfo->vecs[1]) + surf->texinfo->vecs[1][3];
		t -= surf->texturemins[1];
		t += surf->light_t * 16;
		t += 8;
		t /= BLOCK_HEIGHT * 16;

		v[5] = s;
		v[6] = t;
	}

	// build the lightmap
	surf->lightmaptexturenum = current_lightmap_texture;

	base = lightmapinfo[current_lightmap_texture].mapdata;
	base += (surf->light_t * BLOCK_WIDTH + surf->light_s) * LIGHTMAP_BYTES;

	// save out the base to avoid recalculating it at run time
	surf->lmbase = base;

	// i think i may have done this one up above also, but it makes no odds so what the hey?
	lightmapinfo[current_lightmap_texture].modified = false;

	// set the data
	R_BuildLightMap (surf, surf->lmbase, BLOCK_WIDTH * LIGHTMAP_BYTES, surf->samples, false);

	// and again for the stain data
	base = stainmapinfo[current_lightmap_texture].mapdata;
	base += (surf->light_t * BLOCK_WIDTH + surf->light_s) * LIGHTMAP_BYTES;

	// save out the base to avoid recalculating it at run time
	surf->smbase = base;

	// i think i may have done this one up above also, but it makes no odds so what the hey?
	stainmapinfo[current_lightmap_texture].modified = false;

	R_BuildLightMap (surf, surf->smbase, BLOCK_WIDTH * LIGHTMAP_BYTES, surf->stainsamples, true);

	// check the stain data to see if the stain is active
	R_MarkActiveLoadedStainmaps (surf, surf->smbase, BLOCK_WIDTH * LIGHTMAP_BYTES);
}


void GL_BeginBuildingLightmaps (model_t *mod)
{
	LM_InitBlock ();

	Con_DPrintf ("GL_BeginBuildingLightmaps: %s\n", mod->name);
}


void GL_EndBuildingLightmaps (void)
{
	Con_DPrintf ("GL_EndBuildingLightmaps\n");

	LM_UploadBlock ();
}


int mapLightmaps;


/*
=================
GL_BuildAllLightmaps
=================
*/
void GL_BuildAllLightmaps (void)
{
	int i;
	int j;
	int scount;
	model_t *m;
	msurface_t *surf;
	qboolean BlockUpload = false;
	int nummaps = 0;

	GL_BeginBuildingLightmaps (cl.worldmodel);

	for (i = 1; i < MAX_MODELS; i++)
	{
		m = cl.model_precache[i];

		if (!m) break;
		if (m->name[0] == '*') continue;
		if (m->type != mod_brush) continue;

		for (j = 0, surf = m->surfaces; j < m->numsurfaces; j++, surf++)
		{
			if (surf->flags & (SURF_DRAWTURB | SURF_DRAWSKY)) continue;

			BlockUpload = true;
			GL_CreateSurfaceLightmap (surf);
			nummaps++;
		}
	}

	if (BlockUpload) GL_EndBuildingLightmaps ();

	for (j = 0; ; j++)
	{
		scount = 0;

		for (i = 0, surf = cl.worldmodel->surfaces; i < cl.worldmodel->numsurfaces; i++, surf++)
			if (surf->lightmaptexturenum == j) scount++;

		if (!scount) break;
	}

	mapLightmaps = j;

	// for the next map
	current_lightmap_texture = 0;

	Con_DPrintf ("Total Lightmaps used: %i\n", j);

	// because we may have lightmap modified = true as a hangover from the previous
	// map, here we set it to false for all lightmaps.  Purely a precaution.  Ditto for modified and
	// active stainmaps.  we'll fill in which stainmaps are currently active after we load the
	// actual staindata if we're in a map reload situation.  Again, I think this is an unnecessary
	// precaution primarily brought on by sleep deprivation a coupla months back, but it does no
	// harm so why fret?
	// update - it's not!!!  happens on e1m3 anyway...
	// update again - i have dirty lightmaps after the initial map load, so i refresh them all
	// at this stage.  scratch everything above...
	for (i = 0; i < MAX_LIGHTMAPS; i++)
	{
		lightmapinfo[i].modified = true; // false;
		stainmapinfo[i].modified = false;

		if (stainmapinfo[i].active)
		{
			Con_DPrintf ("Stainmap %i is active\n", i);
			stainmapinfo[i].active = false;
		}
	}
}



/*
====================
GL_InitLightmapTextures

Initialize texture slots for the lightmaps.
====================
*/
void GL_InitLightmapTextures (void)
{
	int i;
	image_t *mht;

	// lightmaps
	lightmap_textures = texture_extension_number;
	texture_extension_number += MAX_LIGHTMAPS;

	for (i = 0; i < MAX_LIGHTMAPS; i++)
	{
		mht = GL_CreateImage (va ("Lightmap_%i", i), lightmap_textures + i, BLOCK_WIDTH, BLOCK_HEIGHT, NULL, true, false);
		mht->bpp = 32;
	}

	// and for stainmaps
	stainmap_textures = texture_extension_number;
	texture_extension_number += MAX_LIGHTMAPS;

	for (i = 0; i < MAX_LIGHTMAPS; i++)
	{
		mht = GL_CreateImage (va ("Stainmap_%i", i), stainmap_textures + i, BLOCK_WIDTH, BLOCK_HEIGHT, NULL, true, false);
		mht->bpp = 32;
	}

	InitBLTable ();

	current_lightmap_texture = 0;
}



/*
==========================================================================================
testing of blend funcs
----------------------
this is a good way to experiment with different blending modes to see what looks
best - just replace a call to glBlendFunc with this.  be careful to only test one
blend at a time (e.g. water/light/caustic/etc) or you may get confused!!!
==========================================================================================
*/
GLenum bFactors[11] =
{
	GL_ZERO,					// 0
	GL_ONE,						// 1
	GL_SRC_COLOR,				// 2
	GL_ONE_MINUS_SRC_COLOR,		// 3
	GL_DST_COLOR,				// 4
	GL_ONE_MINUS_DST_COLOR,		// 5
	GL_SRC_ALPHA,				// 6
	GL_ONE_MINUS_SRC_ALPHA,		// 7
	GL_DST_ALPHA,				// 8
	GL_ONE_MINUS_DST_ALPHA,		// 9
	GL_SRC_ALPHA_SATURATE		// 10
};


cvar_t gl_blend_src = {"gl_blend_src", "0"};
cvar_t gl_blend_dst = {"gl_blend_dst", "0"};

void GL_BlendTest (void)
{
	if (gl_blend_src.value < 0) Cvar_SetValueDirect (&gl_blend_src, 0);
	if (gl_blend_src.value > 10) Cvar_SetValueDirect (&gl_blend_src, 10);

	if (gl_blend_dst.value < 0) Cvar_SetValueDirect (&gl_blend_dst, 0);
	if (gl_blend_dst.value > 9) Cvar_SetValueDirect (&gl_blend_dst, 9);

	glBlendFunc (bFactors[(int) gl_blend_src.value], bFactors[(int) gl_blend_dst.value]);
}


/*
==========================================================================================

LIGHTMAP MODIFICATION UPLOADING

==========================================================================================
*/

/*
================
GL_ReloadLightmap

reload any lightmap in it's entirety - currently only used with resolution changing, where
textures need to be reloaded after destroying the previous GL context.
================
*/
void GL_ReloadLightmap (int lightmapnum)
{
	glBindTexture (GL_TEXTURE_2D, lightmap_textures + lightmapnum);

	glTexImage2D (GL_TEXTURE_2D,
				  0,
				  LIGHTMAP_BYTES,
				  BLOCK_WIDTH,
				  BLOCK_HEIGHT,
				  0,
				  GL_RGBA,
				  GL_UNSIGNED_BYTE,
				  lightmapinfo[current_lightmap_texture].mapdata);

	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	lightmapinfo[lightmapnum].modified = false;
}


// and for stainmaps
void GL_ReloadStainmap (int stainmapnum)
{
	glBindTexture (GL_TEXTURE_2D, stainmap_textures + stainmapnum);

	glTexImage2D (GL_TEXTURE_2D,
				  0,
				  LIGHTMAP_BYTES,
				  BLOCK_WIDTH,
				  BLOCK_HEIGHT,
				  0,
				  GL_RGBA,
				  GL_UNSIGNED_BYTE,
				  stainmapinfo[current_lightmap_texture].mapdata);

	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	stainmapinfo[stainmapnum].modified = false;
}


/*
================
R_SaveLightmaps

Saves the current lightdata (including stains) to a file (standard LIT file format) in the 
game save folder.  Previously I was writing the contents of "lightmaps" (4MB per save) which
had proven unreliable - now I'm doing cl.worldmodel->staindata (e.g. 494K for e1m1, a saving
of some 90%).
================
*/
void R_SaveLightmaps (char *savename)
{
	FILE *f;
	litheader_t lithead;

	f = fopen (savename, "wb");

	// save in LIT file format
	lithead.ident[0] = 'Q';
	lithead.ident[1] = 'L';
	lithead.ident[2] = 'I';
	lithead.ident[3] = 'T';
	lithead.version = 1;

	fwrite (&lithead, sizeof (lithead), 1, f);
	fwrite (cl.worldmodel->staindata, cl.worldmodel->lightdatasize, 1, f);

	fclose (f);

	Con_DPrintf ("Saved stainmaps to %s\n", savename);
}


/*
================
R_LoadLightmaps

Called from Host_Loadgame_f - this doesn't actually *do* any loading!!!  Instead it saves
out the name of the correct lightdata file so that we can load it from Mod_LoadLighting.

Must be called *before* SV_SpawnServer!!!
================
*/
char lmloadname[256];
qboolean reloadlm = false;

void R_LoadLightmaps (char *savename)
{
	sprintf (lmloadname, savename);

	reloadlm = true;
}


extern float lighttime;


/*
====================
R_PrebuildLightmaps

Prebuild all of the lightmaps for a given model in advance of the render.  This is so we
can avoid having to rebuild them on a surf-by-surf basis if we have 3 or 4 TMUs, and so
drastically reduce the number of uploads per frame (down from several hundred to 3 or 4)
(and that's 3 or 4, as in numbers less than 10, *not* 3 or 4 hundred...).  The result:
we can actually take proper advantage of those extra TMUs!!!!

Hot damn this is FAST!!!!!!!!!!!!!!!!!!!!!!!!!!
====================
*/
int uploads;

void R_PrebuildLightmaps (void)
{
	int i;
	glRect_t *theRect;

	uploads = 0;

	for (i = 0; i < mapLightmaps; i++)
	{
		if (!lightmapinfo[i].modified) continue;

		// for the next frame
		// set this here so it always gets false even if we're not using lightmaps
		lightmapinfo[i].modified = false;

		if (gl_vertexlight.value) continue;

		// bleagh.  extra binds.  these shouldn't be necessary but unfortunately
		// they are - we won't have more than 2 or 3 of those per frame though,
		// and the absolute maximum for them is the number of lightmaps in the
		// map; e.g. 10 for e1m1.
		glBindTexture (GL_TEXTURE_2D, lightmap_textures + i);

		theRect = &lightmapinfo[i].rectchange;

		// does this crap out on width cos the data isn't contiguous if you specify a subregion of the width?
		glTexSubImage2D (GL_TEXTURE_2D,
						 0, 
						 0, 
						 theRect->b, 
						 BLOCK_WIDTH, 
						 theRect->h, 
						 GL_RGBA, 
						 GL_UNSIGNED_BYTE, 
						 lightmapinfo[i].mapdata + theRect->b * BLOCK_WIDTH * LIGHTMAP_BYTES);

		// so that the surf coords will always be greater or less than as appropriate
		theRect->l = BLOCK_WIDTH;
		theRect->b = BLOCK_HEIGHT;

		theRect->w = 0;
		theRect->h = 0;

		uploads++;
	}
}


void R_BlendStainmaps (void)
{
	int i;
	int j;
	float *v;
	glRect_t *theRect;
	glpoly_t *p;

	// set the proper blend mode
	glDepthMask (GL_FALSE);
	glEnable (GL_BLEND);
	glBlendFunc (GL_DST_COLOR, GL_ZERO);
	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

	for (i = 0; i < mapLightmaps; i++)
	{
		// inactive stainmap
		if (!stainmapinfo[i].active) goto nextSM;

		// no polys accumulated for this stainmap this frame
		if (!(p = stainmapinfo[i].polys)) goto nextSM;

		// bind the stainmap texture
		glBindTexture (GL_TEXTURE_2D, stainmap_textures + i);

		// upload if necessary
		if (stainmapinfo[i].modified)
		{
			// for the next frame
			stainmapinfo[i].modified = false;

			theRect = &stainmapinfo[i].rectchange;

			// does this crap out on width cos the data isn't contiguous if you specify a subregion of the width?
			glTexSubImage2D (GL_TEXTURE_2D,
							0, 
							0, 
							theRect->b, 
							BLOCK_WIDTH, 
							theRect->h, 
							GL_RGBA, 
							GL_UNSIGNED_BYTE, 
							stainmapinfo[i].mapdata + theRect->b * BLOCK_WIDTH * LIGHTMAP_BYTES);

			// so that the surf coords will always be greater or less than as appropriate
			theRect->l = BLOCK_WIDTH;
			theRect->b = BLOCK_HEIGHT;

			theRect->w = 0;
			theRect->h = 0;
		}

		// and draw it
		for (; p; p = p->smChain)
		{
			glDrawArrays (GL_TRIANGLE_FAN, p->saindex, p->numverts);
		}

nextSM:;
		// NULL the chain after drawing it
		stainmapinfo[i].polys = NULL;
	}

	// back to normal
	glDisable (GL_BLEND);
	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDepthMask (GL_TRUE);
}


/*
====================
R_PrebuildSurfaceLightmap

Prebuild the lightmap for an individual surf
====================
*/
void R_PrebuildSurfaceStainmap (msurface_t *surf)
{
	glRect_t *theRect;

	// set the stainmap parameters
	stainmapinfo[surf->lightmaptexturenum].modified = true;
	stainmapinfo[surf->lightmaptexturenum].active = true;

	// calculate how much of the stainmap has changed so we only upload 
	// as much as we need to.  this has never worked properly.
	theRect = &stainmapinfo[surf->lightmaptexturenum].rectchange;

	if (surf->light_s < theRect->l)
	{
		if (theRect->w) theRect->w += theRect->l - surf->light_s;

		theRect->l = surf->light_s;
	}

	if (surf->light_t < theRect->b)
	{
		if (theRect->h) theRect->h += theRect->b - surf->light_t;

		theRect->b = surf->light_t;
	}

	if ((theRect->l + theRect->w) < (surf->light_s + surf->smax))
		theRect->w = (surf->light_s + surf->smax) - theRect->l;

	if ((theRect->h + theRect->b) < (surf->light_t + surf->tmax))
		theRect->h = (surf->light_t - theRect->b) + surf->tmax;

	// rebuild the stainmap data for this surf
	R_BuildLightMap (surf, surf->smbase, BLOCK_WIDTH * LIGHTMAP_BYTES, surf->stainsamples, true);
}


void R_PrebuildSurfaceLightmap (msurface_t *surf)
{
	int i;
	int j;
	int map = 0;
	glRect_t *theRect;
	dlight_t *dl;
	qboolean cleartonolight = true;

	// dynamic lights modification
	qboolean dlhit;
	float dist;
	float radius;
	float minlight;

	float dlr;
	float dlg;
	float dlb;

	vec3_t impact;
	vec3_t local;

	int s;
	int t;

	int sd;
	int td;

	int sacc;
	int tacc;

	int add;

	int sidebit;

	// initially false
	surf->lightmapmodified = false;

	// build the new lightmap if necessary
	if (lighttime < LIGHT_UPLOAD)
	{
		// don't need to store any of the caches
		return;
	}

	// prebuild the stainmap first as any of the lightmap conditions could cause it to be skipped over
	if (surf->stained)
	{
		R_PrebuildSurfaceStainmap (surf);
	}

	// dynamic this frame
	// this is always checked first cos we need to accumulate the dynamic light modification
	// before we add on anything else.
	// ooooh this one is FAST.  it's amazing how idiotic IDs original code was here...
	if (surf->dlightframe == r_framecount && r_dynamic.value)
	{
		// this is a "possible" dynamic based on the node containing this surf being in
		// range of 1 or more dlights - first thing we gotta do is see if the surf itself
		// is in range of any dlights
		dlhit = false;

		for (i = 0; i < MAX_DLIGHTS; i++)
		{
			// not lit by this light
			if (!(surf->dlightbits & (1 << i))) continue;

			dl = &cl_dlights[i];

			// check for distance to the dlight
			// this can be done in either R_MarkLights or here, it's not really much of a difference
			dist = DotProduct (dl->origin, surf->plane->normal) - surf->plane->dist;

			// out of range
			if (dist > dl->radius || dist < -dl->radius) continue;

			// check for planeback
			if (dist >= 0)
				sidebit = 0;
			else
				sidebit = SURF_PLANEBACK;

			// surf not facing the dlight
			if ((surf->flags & SURF_PLANEBACK) != sidebit) continue;

			radius = dl->radius - fabs (dist);

			// another range check against minlight for the dlight
			if (radius < dl->minlight) continue;

			// now accumulate the lights
			// we're not sure we've got a hit yet... in fact, we may very well not...
			// (this assignment to true is just for comparison testing purposes)
			// dlhit = true;

			minlight = radius - dl->minlight;

			impact[0] = dl->origin[0] - surf->plane->normal[0] * dist;
			impact[1] = dl->origin[1] - surf->plane->normal[1] * dist;
			impact[2] = dl->origin[2] - surf->plane->normal[2] * dist;

			local[0] = DotProduct (impact, surf->texinfo->vecs[0]) + surf->texinfo->vecs[0][3];
			local[1] = DotProduct (impact, surf->texinfo->vecs[1]) + surf->texinfo->vecs[1][3];

			local[0] -= surf->texturemins[0];
			local[1] -= surf->texturemins[1];

			// init the colours
			// this has to be done seperately for each dlight
			dlr = dl->colour[0] * 256;
			dlg = dl->colour[1] * 256;
			dlb = dl->colour[2] * 256;

			for (t = 0, tacc = 0; t < surf->tmax; t++, tacc += 16)
			{
				td = local[1] - tacc;

				if (td < 0) td = -td;

				add = t * surf->smax;

				for (s = 0, sacc = 0; s < surf->smax; s++, sacc += 16)
				{
					sd = local[0] - sacc;

					if (sd < 0) sd = -sd;

					if (sd > td)
						dist = sd + (td >> 1);
					else dist = td + (sd >> 1);

					if (dist < minlight)
					{
						// we only need to do the stuff in here once for each surf we check, so we'll
						// set cleartonolight to false in here to make sure it doesn't get done a
						// second time
						if (cleartonolight)
						{
							// clear to no light - only the first time we get to here!!!
							for (j = 0; j < surf->smax * surf->tmax; j++)
								blocklightcolours[0][j] = 
								blocklightcolours[1][j] = 
								blocklightcolours[2][j] = 0;

							// it's been cleared so don't do it again
							cleartonolight = false;

							// don't clear it in R_BuildLightmap either!!!
							blocklightinit = true;

							// we're only absolutely sure we've got a hit if we get here.
							// confirming a hit any earlier may result in lightmaps being
							// unnecessarily uploaded
							dlhit = true;
						}

						blocklightcolours[0][add + s] += (radius - dist) * dlr;
						blocklightcolours[1][add + s] += (radius - dist) * dlg;
						blocklightcolours[2][add + s] += (radius - dist) * dlb;
					}
				}
			}
		}

		if (dlhit)
		{
			surf->cached_dlight = true;
			goto dynamic;
		}
	}

	// dynamic previous frame
	if (surf->cached_dlight)
	{
		// no cached dlight next frame unless we explicitly specify otherwise
		surf->cached_dlight = false;
		goto dynamic;
	}

	// check for modified lightstyles on the surf (only on surfs that *have* lightstyles)
	if (!surf->staticmap)
	{
		for (map = 0; map < MAXLIGHTMAPS && surf->styles[map] != 255; map++)
		{
			if (d_lightstylevalue[surf->styles[map]] != surf->cached_light[map])
			{
				goto dynamic;
			}
		}
	}

	// reset the surf parameters
	// we need this even if the lightmap is not uploaded so that dlight stats are updated
	// properly (remember - surf->dlightframe == r_framecount only indicates a possible hit)
	surf->dlightframe = -1;
	surf->dlightbits = 0;
	surf->stained = false;

	// not modified
	return;

dynamic:;
	// reset the surf parameters
	// because we could have gotten here from any of the gotos above, we have to 
	// cover these here so they don't interfere with the next light update
	surf->dlightframe = -1;
	surf->dlightbits = 0;
	surf->stained = false;

	// flag the lightmap as having been modified in both the surf and the lightmapinfo
	lightmapinfo[surf->lightmaptexturenum].modified = true;
	surf->lightmapmodified = true;

	// calculate how much of the lightmap has changed so we only upload 
	// as much as we need to.  this has never worked properly.
	theRect = &lightmapinfo[surf->lightmaptexturenum].rectchange;

	if (surf->light_s < theRect->l)
	{
		if (theRect->w) theRect->w += theRect->l - surf->light_s;

		theRect->l = surf->light_s;
	}

	if (surf->light_t < theRect->b)
	{
		if (theRect->h) theRect->h += theRect->b - surf->light_t;

		theRect->b = surf->light_t;
	}

	if ((theRect->l + theRect->w) < (surf->light_s + surf->smax))
		theRect->w = (surf->light_s + surf->smax) - theRect->l;

	if ((theRect->h + theRect->b) < (surf->light_t + surf->tmax))
		theRect->h = (surf->light_t - theRect->b) + surf->tmax;

	R_BuildLightMap (surf, surf->lmbase, BLOCK_WIDTH * LIGHTMAP_BYTES, surf->samples, false);

	// clear it to no light in R_BuildLightMap next time unless we specify otherwise again
	blocklightinit = false;
}


void R_SetSurfaceVertexLight (msurface_t *surf, float *verts)
{
	int s;
	int t;
	byte *base;

	// get a pointer to the texel for this vertex
	s = verts[5] * BLOCK_WIDTH;
	t = verts[6] * BLOCK_WIDTH;

	while (s > BLOCK_WIDTH) s -= BLOCK_WIDTH;
	while (t > BLOCK_WIDTH) t -= BLOCK_WIDTH;

	while (s < 0) s += BLOCK_WIDTH;
	while (t < 0) t += BLOCK_WIDTH;

	base = lightmapinfo[surf->lightmaptexturenum].mapdata + (t * BLOCK_WIDTH + s) * LIGHTMAP_BYTES;

	// boost the colours
	verts[9] = ((float) *(base)) / 128.0f;
	verts[10] = ((float) *(base + 1)) / 128.0f;
	verts[11] = ((float) *(base + 2)) / 128.0f;

	// alpha - always 1
	verts[12] = 1;
}


/*
==========================================================================================

				STAINMAPS

Hacked in using idles Q2 code as a base.

==========================================================================================
*/

// go here
void R_StainNode (stain_t *st, mnode_t *node)
{
	msurface_t *surf;
	float dist;
	int c;
	qboolean stained;
	qboolean stainedsurf;
	int sidebit;

	if (node->contents < 0) return;

	// distance stuff
	dist = DotProduct (st->origin, node->plane->normal) - node->plane->dist;

	// too far away
	if (dist > st->size)
	{
		R_StainNode (st, node->children[0]);
		return;
	}

	// too far away
	if (dist < -st->size)
	{
		R_StainNode (st, node->children[1]);
		return;
	}

	if (dist >= 0)
		sidebit = 0;
	else
		sidebit = SURF_PLANEBACK;

	// node is in range so stain it
	stainedsurf = false;

	for (c = node->numsurfaces, surf = cl.worldmodel->surfaces + node->firstsurface; c ; c--, surf++)
	{
		mtexinfo_t *tex;
		vec3_t impact, local;
		byte *bl;
		int i, sd, td, s, t, smax, tmax;
		float fdist, frad, fminlight, fsacc, ftacc;
		long col;

		// initially unstained cos it may not even be hit!!!
		stained = false;

		if (surf->flags & (SURF_DRAWTURB | SURF_DRAWSKY)) continue;
		if (!surf->samples) continue;
		if ((surf->flags & SURF_PLANEBACK) != sidebit) continue;

		smax = surf->smax;
		tmax = surf->tmax;

		tex = surf->texinfo;

		frad = st->size;

		// removed axial optimisations cos plane->type is relative to the world orientation, not the
		// point orientation
		fdist = DotProduct (st->origin, surf->plane->normal) - surf->plane->dist;

		// too far away
		if (fdist > st->size || fdist < -st->size) continue;

		if ((surf->flags & SURF_PLANEBACK)) fdist *= -1;

		frad -= fabs (fdist);

		if (frad < 0) continue;

		// surf is in range
		fminlight = frad;

		impact[0] = st->origin[0] - surf->plane->normal[0] * fdist;
		impact[1] = st->origin[1] - surf->plane->normal[1] * fdist;
		impact[2] = st->origin[2] - surf->plane->normal[2] * fdist;

		local[0] = DotProduct (impact, tex->vecs[0]) + tex->vecs[0][3] - surf->texturemins[0];
		local[1] = DotProduct (impact, tex->vecs[1]) + tex->vecs[1][3] - surf->texturemins[1];

		// use stainsamples here...
		bl = surf->stainsamples;

		for (t = 0, ftacc = 0; t < tmax; t++, ftacc += 16)
		{
			td = local[1] - ftacc;

			if (td < 0) td = -td;

			for (s = 0, fsacc = 0; s < smax; s++, fsacc += 16, bl += 3)
			{
				sd = local[0] - fsacc;

				if (sd < 0) sd = -sd;

				fdist = (sd > td) ? sd + (td >> 1) : td + (sd >> 1);

				if (fdist < fminlight)
				{
					for (i = 0; i < 3; i++)
					{
						col = bl[i] * st->color[i];

						// prevent from going to black with multiple writes
						if (col < st->mincolour[i]) col = st->mincolour[i];
						else if (col < 0) col = 0;
						else if (col > 255) col = 255;

						bl[i] = (byte)col;
					}

					// only mark the surf as stained if it actually *has* been...
					stained = true;
				}
			}
		}

		if (stained)
		{
			surf->everstained = surf->stained = true;
			stainedsurf = true;
		}

		// fixme - if the *entire* stain is inside the surfs boundary we don't need to
		// check any more - at all actually.  how expensive is this to calculate?
	}

	R_StainNode (st, node->children[0]);
	R_StainNode (st, node->children[1]);
}


void R_ApplyStains (mnode_t *node)
{
	stain_t *st;
	mnode_t *newnode;
	int i;

	if (!num_newstains) return;

	for (i = 0, st = newstains, newnode = node; i < num_newstains; i++, st++, newnode = node)
	{
		R_StainNode (st, newnode);
	}

	// debugging
	// most of the 600 odd stains per frame in old MHQuakes were from static gibs
	// i've fixed that, so a max of 32 stains per frame is perfectly adequate
	//Con_Printf ("%i stains added\n", num_newstains);
}


void Light_Init (void)
{
	int i;

	for (i = 0; i < MAX_LIGHTMAPS; i++)
	{
		//lightmapinfo[i].mapdata = NULL;
		lightmapinfo[i].polys = NULL;
		//stainmapinfo[i].mapdata = NULL;
		stainmapinfo[i].polys = NULL;
	}
}

