/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// r_main.c

#include "quakedef.h"
#include "glext.h"
#include "gl_decals.h"

extern cvar_t chase_active;

extern qboolean automapOn;

entity_t	r_worldentity;

vec3_t		modelorg, r_entorigin;
entity_t	*currententity;

int			r_visframecount;	// bumped when going to a new PVS
unsigned int			r_framecount = 1;		// used for dlight push checking

mplane_t	frustum[4];

int			c_brush_polys, c_alias_polys;

qboolean	envmap;				// true during envmap command capture 

int			currenttexture[4] = {-1, -1, -1, -1};
int			currenttmu;

int			cnttextures[4] = {-1, -1, -1, -1};     // cached

int			playertextures;		// up to 16 color translated skins

// view origin
vec3_t	vup;
vec3_t	vpn;
vec3_t	vright;
vec3_t	r_origin;

//
// screen size info
//
refdef_t	r_refdef;

mleaf_t		*r_viewleaf, *r_oldviewleaf;

int		d_lightstylevalue[256];	// 8.8 fraction of base light value


void R_MarkLeaves (void);

cvar_t	r_norefresh = {"r_norefresh","0"};
cvar_t	r_drawentities = {"r_drawentities","1"};
cvar_t	r_drawviewmodel = {"r_drawviewmodel","1"};
cvar_t	r_speeds = {"r_speeds","0", true};
cvar_t	r_lightmap = {"r_lightmap","0"};
cvar_t	r_shadows = {"r_shadows","1", true};
cvar_t	r_wateralpha = {"r_wateralpha","1"};
cvar_t	r_dynamic = {"r_dynamic","1"};
cvar_t	r_novis = {"r_novis","0"};

cvar_t	gl_finish = {"gl_finish","0"};
cvar_t	gl_clear = {"gl_clear","0"};
cvar_t	gl_cull = {"gl_cull","1"};
cvar_t	gl_smoothmodels = {"gl_smoothmodels","1"};
cvar_t	gl_affinemodels = {"gl_affinemodels","0"};
cvar_t	gl_polyblend = {"gl_polyblend","1"};
cvar_t	gl_flashblend = {"gl_flashblend","0"};
cvar_t	gl_playermip = {"gl_playermip","0"};
cvar_t	gl_nocolors = {"gl_nocolors","0"};
cvar_t	gl_keeptjunctions = {"gl_keeptjunctions","1"};
cvar_t	gl_reporttjunctions = {"gl_reporttjunctions","0"};
cvar_t	gl_doubleeyes = {"gl_doubleeys", "1"};

extern	cvar_t	gl_ztrick;

float frametime;

float slowpulse = 0;
int slowpulsedir = 1;
float slowcycle = 0;

int numAliasShadows;
int numAliasModels;
int numBrushModels;
int numSpriteModels;

// changed this to support using it as an auxillary buffer for maps (e.g., when we don't want to overwrite the
// original verts but need to modify them before drawing)
GLfloat VArray[MAX_MAP_VERTS * VERTEXSIZE * 2];


// sin and cos tables from 0 to 1 in 0.0625 increments to speed up glow rendering
float glowcos[17] = 
{
	1.000000,
	0.923879,
	0.707106,
	0.382682,
	-0.000002,
	-0.382686,
	-0.707109,
	-0.923881,
	-1.000000,
	-0.923878,
	-0.707103,
	-0.382678,
	0.000006,
	0.382689,
	0.707112,
	0.923882,
	1.000000,
};


float glowsin[17] = 
{
	0.000000,
	0.382684,
	0.707107,
	0.923880,
	1.000000,
	0.923879,
	0.707105,
	0.382680,
	-0.000004,
	-0.382687,
	-0.707110,
	-0.923882,
	-1.000000,
	-0.923877,
	-0.707102,
	-0.382677,
	0.000008,
};


// glow generation code
#define MAX_GLOWS 2048
int num_glows = 0;

typedef struct glows_s
{
	float red;
	float green;
	float blue;
	float radius;
	vec3_t origin;
} glows_t;


glows_t glow_effects[MAX_GLOWS];


void AddLightBlend (float r, float g, float b, float a2)
{
	float	a;

	v_blend[3] = a = v_blend[3] + a2 * (1 - v_blend[3]);

	a2 = a2 / a;

	v_blend[0] = v_blend[1] * (1 - a2) + r * a2;
	v_blend[1] = v_blend[1] * (1 - a2) + g * a2;
	v_blend[2] = v_blend[2] * (1 - a2) + b * a2;
}


void R_RenderGlowEffects (void)
{
	int i, j, k;
	vec3_t v;
	float rad;

	if (!num_glows) return;

	glDepthMask (GL_FALSE);
	glDisable (GL_TEXTURE_2D);
	glEnable (GL_BLEND);
	glBlendFunc (GL_ONE, GL_ONE);

	for (k = 0; k < num_glows; k++)
	{
		rad = glow_effects[k].radius;

		VectorSubtract (glow_effects[k].origin, r_origin, v);

		// see is the view inside the glow
		if (Length (v) < rad)
		{
			float max = 0;
			float rgb[3];
			float mody;

			// find the max and scale as appropriate
			if (glow_effects[k].red > max) max = glow_effects[k].red;
			if (glow_effects[k].green > max) max = glow_effects[k].green;
			if (glow_effects[k].blue > max) max = glow_effects[k].blue;

			// prevent division by 0
			if (max)
			{
				mody = 1.0 / max;

				rgb[0] = glow_effects[k].red * mody;
				rgb[1] = glow_effects[k].green * mody;
				rgb[2] = glow_effects[k].blue * mody;

				AddLightBlend (rgb[0], rgb[1], rgb[2], glow_effects[k].radius * 0.0003);
			}

			continue;
		}

		glBegin (GL_TRIANGLE_FAN);

		glColor3f (glow_effects[k].red, glow_effects[k].green, glow_effects[k].blue);

		for (i = 0; i < 3; i++)
			v[i] = glow_effects[k].origin[i] - vpn[i] * rad;

		glVertex3fv (v);

		glColor3f (0, 0, 0);

		for (i = 16; i >= 0; i--)
		{
			v[0] = glow_effects[k].origin[0] + vright[0] * glowcos[i] * rad + vup[0] * glowsin[i] * rad;
			v[1] = glow_effects[k].origin[1] + vright[1] * glowcos[i] * rad + vup[1] * glowsin[i] * rad;
			v[2] = glow_effects[k].origin[2] + vright[2] * glowcos[i] * rad + vup[2] * glowsin[i] * rad;

			glVertex3fv (v);
		}

		glEnd ();
	}

	glColor3f (1, 1, 1);
	glDisable (GL_BLEND);
	glEnable (GL_TEXTURE_2D);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDepthMask (GL_TRUE);

	num_glows = 0;
}


void R_AddGlowEffect (float red, float green, float blue, float radius, vec3_t origin)
{
	if (num_glows >= MAX_GLOWS)
	{
		Con_DPrintf ("Too many glows\n");
		return;
	}

	glow_effects[num_glows].red = red;
	glow_effects[num_glows].green = green;
	glow_effects[num_glows].blue = blue;

	glow_effects[num_glows].radius = radius;

	glow_effects[num_glows].origin[0] = origin[0];
	glow_effects[num_glows].origin[1] = origin[1];
	glow_effects[num_glows].origin[2] = origin[2];

	num_glows++;
}


float water_fog[4] = {0.0, 0.0, 0.0, 0.2};
float world_fog[4] = {0.0, 0.0, 0.0, 0.2};
float sky_fog[4] = {0.0, 0.0, 0.0, 0.2};
float generic_fog[4] = {0.1, 0.1, 0.1, 0.2};


int fogSave;

void R_SetFog (int fogType)
{
	float colours[4] = {0.1, 0.1, 0.1, 0.2};
	float start = 100.0;
	float end = 4096.0;
	float density = 0.2;
	static int oldFog = FOG_NONE;

	int i;

	switch (fogType)
	{
	case FOG_WORLD:
		// settings for world fog (empty space)
		// this can either be a user configurable option or a default based on the world type
		if (!gl_fogenable.value)
		{
			// the user don't want no steenkin' world fog...
			R_SetFog (FOG_NONE);
			return;
		}

		if (gl_foguser.value)
		{
			// user specified fog variables
			colours[0] = gl_fogr.value;
			colours[1] = gl_fogg.value;
			colours[2] = gl_fogb.value;
			colours[3] = 0.2;

			start = gl_fogstart.value;
			end = gl_fogend.value;

			density = gl_fogdensity.value;
		}
		else
		{
			// defaults based on the world type
			colours[0] = world_fog[0];
			colours[1] = world_fog[1];
			colours[2] = world_fog[2];
			colours[3] = world_fog[3];

			start = 10;
			end = 4072;
		}

		break;

	case FOG_SKY:
		// settings for skybrush fogging - the user cannot specify these, they
		// are derived from the colour of the sky texture
		for (i = 0; i < 4; i++)
			colours[i] = sky_fog[i];

		start = 10;
		end = 400;
		density = 0.6;

		break;

	case FOG_WATER:
		// settings for underwater fogging - the user cannot specify these, they
		// are derived from the colour of the water texture(s)
		colours[0] = water_fog[0];
		colours[1] = water_fog[1];
		colours[2] = water_fog[2];
		colours[3] = 0.9;

		start = 0;
		end = 700;

		density = 0.2;

		break;

	case FOG_WATERSURF:
		// water surfaces when viewed from underwater.  fogging on a translucent surf
		// looks crap so reduce the distance
		for (i = 0; i < 4; i++)
			colours[i] = water_fog[i];

		start = 10;
		end = 50;
		density = 0.2;

		break;

	case FOG_REVERT:
		// revert to the previous fog version
		// this check is probly not necessary but doesn't hurt and may
		// save me many tears later...
		if (oldFog != FOG_REVERT) R_SetFog (oldFog);
		return;

	case FOG_NONE:
	default:
		// turn off fogging and get out
		glDisable (GL_FOG);
		return;
	}

	// store the previous fog type if we need to revert
	oldFog = fogType;

	// turn on fogging
	glFogi (GL_FOG_MODE, GL_LINEAR);
	glFogfv (GL_FOG_COLOR, colours);
	glFogf (GL_FOG_START, start);
	glFogf (GL_FOG_END, end);
	glFogf (GL_FOG_DENSITY, density);
	glEnable (GL_FOG);
}


int R_GetFog (void)
{
	// returns the current fog setting
	return fogSave;
}


/*
=================
R_CullBox

Returns true if the box is completely outside the frustom
=================
*/
qboolean R_CullBox (vec3_t mins, vec3_t maxs)
{
	int		i;

	for (i = 0; i < 4; i++)
	{
		if (BOX_ON_PLANE_SIDE (mins, maxs, &frustum[i]) == 2)
		{
			return true;
		}
	}

	return false;
}

qboolean R_CullBox2 (vec3_t mins, vec3_t maxs)
{
	int		i;

	for (i=0 ; i<4 ; i++)
		if (BoxOnPlaneSide (mins, maxs, &frustum[i]) == 2)
			return true;
	return false;
}


void R_RotateForEntity (entity_t *e, qboolean blendLerp)
{
	float timepassed;
	float blend;
	vec3_t d;
	int i;

	if (!blendLerp)
	{
	    glTranslatef (e->origin[0],  e->origin[1],  e->origin[2]);

	    glRotatef (e->angles[1],  0, 0, 1);

		// store the matrix out for shadow drawing (saves us having to do all those 
		// transforms again).  save it out *before* doing the scaling or the rest of 
		// the rotation, cos that fucks with our shadow positioning
		glGetFloatv (GL_MODELVIEW_MATRIX, e->MViewMatrix);

	    glRotatef (-e->angles[0],  0, 1, 0);
	    glRotatef (e->angles[2],  1, 0, 0);

		return;
	}

	timepassed = realtime - e->translateStartTime; 

	if (e->translateStartTime == 0 || timepassed > 1)
	{
		e->translateStartTime = realtime;
		VectorCopy (e->origin, e->origin1);
		VectorCopy (e->origin, e->origin2);
	}

	if (!VectorCompare (e->origin, e->origin2))
	{
		e->translateStartTime = realtime;
		VectorCopy (e->origin2, e->origin1);
		VectorCopy (e->origin,  e->origin2);
		blend = 0;
	}
	else
	{
		blend = timepassed / 0.1;

		if (cl.paused || blend > 1) blend = 1;
	}

	VectorSubtract (e->origin2, e->origin1, d);

	glTranslatef (e->origin1[0] + (blend * d[0]),
				  e->origin1[1] + (blend * d[1]),
				  e->origin1[2] + (blend * d[2]));

	// orientation interpolation (Euler angles, yuck!)
	timepassed = realtime - e->rotateStartTime; 

	if (e->rotateStartTime == 0 || timepassed > 1)
	{
		e->rotateStartTime = realtime;
		VectorCopy (e->angles, e->angles1);
		VectorCopy (e->angles, e->angles2);
	}

	if (!VectorCompare (e->angles, e->angles2))
	{
		e->rotateStartTime = realtime;
		VectorCopy (e->angles2, e->angles1);
		VectorCopy (e->angles,  e->angles2);
		blend = 0;
	}
	else
	{
		blend = timepassed / 0.1;
 
		if (cl.paused || blend > 1) blend = 1;
	}

	VectorSubtract (e->angles2, e->angles1, d);

	// always interpolate along the shortest path
	for (i = 0; i < 3; i++) 
	{
		if (d[i] > 180)
			d[i] -= 360;
		else if (d[i] < -180)
			d[i] += 360;
	}

	glRotatef ( e->angles1[1] + ( blend * d[1]),  0, 0, 1);

	// store the matrix out for shadow drawing (saves us having to do all those 
	// transforms again).  save it out *before* doing the scaling or the rest of 
	// the rotation, cos that fucks with our shadow positioning
	glGetFloatv (GL_MODELVIEW_MATRIX, e->MViewMatrix);

	glRotatef (-e->angles1[0] + (-blend * d[0]),  0, 1, 0);
	glRotatef ( e->angles1[2] + ( blend * d[2]),  1, 0, 0);
}

/*
=============================================================

  SPRITE MODELS

=============================================================
*/

/*
================
R_GetSpriteFrame
================
*/
mspriteframe_t *R_GetSpriteFrame (entity_t *currententity)
{
	msprite_t		*psprite;
	mspritegroup_t	*pspritegroup;
	mspriteframe_t	*pspriteframe;
	int				i, numframes, frame;
	float			*pintervals, fullinterval, targettime, time;

	psprite = currententity->model->cache.data;
	frame = currententity->frame;

	if ((frame >= psprite->numframes) || (frame < 0))
	{
		Con_Printf ("R_DrawSprite: no such frame %d\n", frame);
		frame = 0;
	}

	if (psprite->frames[frame].type == SPR_SINGLE)
	{
		pspriteframe = psprite->frames[frame].frameptr;
	}
	else
	{
		pspritegroup = (mspritegroup_t *)psprite->frames[frame].frameptr;
		pintervals = pspritegroup->intervals;
		numframes = pspritegroup->numframes;
		fullinterval = pintervals[numframes-1];

		time = cl.time + currententity->syncbase;

		// when loading in Mod_LoadSpriteGroup, we guaranteed all interval values
		// are positive, so we don't have to worry about division by 0
		targettime = time - ((int)(time / fullinterval)) * fullinterval;

		for (i=0 ; i<(numframes-1) ; i++)
		{
			if (pintervals[i] > targettime)
				break;
		}

		pspriteframe = pspritegroup->frames[i];
	}

	return pspriteframe;
}


/*
=================
R_DrawSpriteModel

=================
*/
void R_ZerstorerRain (vec3_t org);
void R_BubbleParticle (vec3_t org);

void R_DrawSpriteModel (entity_t *e)
{
	vec3_t	point;
	mspriteframe_t	*frame;
	float		*up, *right;
	vec3_t		v_forward, v_right, v_up;
	msprite_t		*psprite;

	psprite = currententity->model->cache.data;

	// standard quake sprites are handled differently.  other types use the normal sprite code
	if (psprite->drawType == SPR_EXPLODE)
	{
		return;
	}
	else if (psprite->drawType == SPR_LIGHT)
	{
		// add a glow for this
		R_AddGlowEffect (0.89, 0.59, 0.31, 15, e->origin);
		return;
	}
	else if (psprite->drawType == SPR_BUBBLE)
	{
		// draw as a particle
		// R_BubbleParticle (e->origin);
		R_AddGlowEffect (0.5, 0.75, 1.0, 3, e->origin);
		return;
	}
	else if (psprite->drawType == SPR_RAIN)
	{
		// zerstorer rain
		// draw as a particle
		R_ZerstorerRain (currententity->origin);
		return;
	}
	else if (psprite->drawType != SPR_DRAW)
	{
		// handles sprites that have not been given a drawing routine yet
		return;
	}

	// don't even bother drawing any other sprites (does this do the same as above?)
	return;

	// don't even bother culling, because it's just a single
	// polygon without a surface cache
	frame = R_GetSpriteFrame (e);

	if (psprite->type == SPR_ORIENTED)
	{	// bullet marks on walls
		AngleVectors (currententity->angles, v_forward, v_right, v_up);
		up = v_up;
		right = v_right;
	}
	else
	{	// normal sprite
		up = vup;
		right = vright;
	}

	glColor3f (1,1,1);

    glBindTexture (GL_TEXTURE_2D, frame->gl_texture->texnum);

	glEnable (GL_ALPHA_TEST);
	glBegin (GL_QUADS);

	glTexCoord2f (0, 1);
	VectorMA (e->origin, frame->down, up, point);
	VectorMA (point, frame->left, right, point);
	glVertex3fv (point);

	glTexCoord2f (0, 0);
	VectorMA (e->origin, frame->up, up, point);
	VectorMA (point, frame->left, right, point);
	glVertex3fv (point);

	glTexCoord2f (1, 0);
	VectorMA (e->origin, frame->up, up, point);
	VectorMA (point, frame->right, right, point);
	glVertex3fv (point);

	glTexCoord2f (1, 1);
	VectorMA (e->origin, frame->down, up, point);
	VectorMA (point, frame->right, right, point);
	glVertex3fv (point);
	
	glEnd ();

	glDisable (GL_ALPHA_TEST);
}

/*
=============================================================

  ALIAS MODELS

=============================================================
*/


#define NUMVERTEXNORMALS	162

float	r_avertexnormals[NUMVERTEXNORMALS][3] =
{
#include "anorms.h"
};

vec3_t	shadelight;


// precalculated dot products for quantized angles
#define SHADEDOT_QUANT 16

float	r_avertexnormal_dots[SHADEDOT_QUANT][256] =
#include "anorm_dots.h"
;

float	*shadedots = r_avertexnormal_dots[0];
float	*shadedots2 = r_avertexnormal_dots[0];
float	lightLerpOffset;

int	lastposenum;
int	lastposenum0;

int aliasverts;

void GL_DrawAliasFrameLIGHTLERP (struct aliashdr_s *paliashdr, int pose1, int pose2, float blend)
{
	int		*order;
	int		count;
	int		VCount;

	float	*aliasVerts;
	float	l, l1, l2;
	float	diff;

	GLenum	TRI_TYPE;

	drawvertx_t	*verts1;
	drawvertx_t	*verts2;

	vec3_t	d;

	lastposenum0 = pose1;
	lastposenum = pose2;

	verts1 = (drawvertx_t *)((byte *)paliashdr + paliashdr->posedata);
	verts2 = verts1;

	verts1 += pose1 * paliashdr->poseverts;
	verts2 += pose2 * paliashdr->poseverts;

	order = (int *)((byte *)paliashdr + paliashdr->commands);

	aliasverts = 0;

	while (1)
	{
		currententity->shadowverts[aliasverts++] = count = *order++;

		if (!count)
			break;

		if (count < 0)
		{
			count = -count;
			TRI_TYPE = GL_TRIANGLE_FAN;
		}
		else TRI_TYPE = GL_TRIANGLE_STRIP;

		aliasVerts = VArray;
		VCount = 0;

		do
		{
			// vertex co-ords
			VectorSubtract (verts2->v, verts1->v, d);

			currententity->shadowverts[aliasverts++] = aliasVerts[0] = verts1->v[0] + (blend * d[0]);
			currententity->shadowverts[aliasverts++] = aliasVerts[1] = verts1->v[1] + (blend * d[1]);
			currententity->shadowverts[aliasverts++] = aliasVerts[2] = verts1->v[2] + (blend * d[2]);

			// texture co-ords
			aliasVerts[3] = ((float *)order)[0];
			aliasVerts[4] = ((float *)order)[1];

			// lighting
			d[0] = shadedots[verts2->lightnormalindex] - shadedots[verts1->lightnormalindex];
			d[1] = shadedots2[verts2->lightnormalindex] - shadedots2[verts1->lightnormalindex];

			l1 = (shadedots[verts1->lightnormalindex] + (blend * d[0]));
			l2 = (shadedots2[verts1->lightnormalindex] + (blend * d[1]));

			if (l1 != l2)
			{
				if (l1 > l2)
				{
					diff = l1 - l2;
					diff *= lightLerpOffset;
					l = l1 - diff;
				}
				else
				{
					diff = l2 - l1;
					diff *= lightLerpOffset;
					l = l1 + diff;
				}
			}
			else l = l1;

			l *= l;

			aliasVerts[5] = l * shadelight[0];
			aliasVerts[6] = l * shadelight[1];
			aliasVerts[7] = l * shadelight[2];
			aliasVerts[8] = paliashdr->translevel;

			order += 2;
			verts1++;
			verts2++;

			aliasVerts += 9;
			VCount++;
		} while (--count);

		glDrawArrays (TRI_TYPE, 0, VCount);
	}
}


void GL_DrawAliasFrameLIGHTNOLERP (aliashdr_t *paliashdr, int posenum)
{
	drawvertx_t	*verts;
	int		*order;
	int		count;
	float	*aliasVerts;
	GLenum	TRI_TYPE;
	int		VCount;
	float	l, l1, l2, diff;

	lastposenum = posenum;

	verts = (drawvertx_t *)((byte *)paliashdr + paliashdr->posedata);
	verts += posenum * paliashdr->poseverts;
	order = (int *)((byte *)paliashdr + paliashdr->commands);

	aliasverts = 0;

	while (1)
	{
		currententity->shadowverts[aliasverts++] = count = *order++;

		if (!count)
			break;

		if (count < 0)
		{
			count = -count;
			TRI_TYPE = GL_TRIANGLE_FAN;
		}
		else TRI_TYPE = GL_TRIANGLE_STRIP;

		aliasVerts = VArray;
		VCount = 0;

		do
		{
			currententity->shadowverts[aliasverts++] = aliasVerts[0] = verts->v[0];
			currententity->shadowverts[aliasverts++] = aliasVerts[1] = verts->v[1];
			currententity->shadowverts[aliasverts++] = aliasVerts[2] = verts->v[2];

			aliasVerts[3] = ((float *)order)[0];
			aliasVerts[4] = ((float *)order)[1];

			l1 = shadedots[verts->lightnormalindex];
			l2 = shadedots2[verts->lightnormalindex];

			if (l1 != l2)
			{
				if (l1 > l2)
				{
					diff = l1 - l2;
					diff *= lightLerpOffset;
					l = l1 - diff;
				}
				else
				{
					diff = l2 - l1;
					diff *= lightLerpOffset;
					l = l1 + diff;
				}
			}
			else l = l1;

			l *= l;

			aliasVerts[5] = l * shadelight[0];
			aliasVerts[6] = l * shadelight[1];
			aliasVerts[7] = l * shadelight[2];
			aliasVerts[8] = paliashdr->translevel;

			order += 2;
			verts++;

			aliasVerts += 9;
			VCount++;
		} while (--count);

		glDrawArrays (TRI_TYPE, 0, VCount);
	}
}


// don't store shadow info here
void GL_DrawAliasFrameLERPFLAT (struct aliashdr_s *paliashdr, int pose1, int pose2, float blend, float Light)
{
	int		*order;
	int		count;
	int		VCount;

	float	*aliasVerts;

	GLenum	TRI_TYPE;

	drawvertx_t	*verts1;
	drawvertx_t	*verts2;

	vec3_t	d;

	lastposenum0 = pose1;
	lastposenum = pose2;

	verts1 = (drawvertx_t *)((byte *)paliashdr + paliashdr->posedata);
	verts2 = verts1;

	verts1 += pose1 * paliashdr->poseverts;
	verts2 += pose2 * paliashdr->poseverts;

	order = (int *)((byte *)paliashdr + paliashdr->commands);

	while (1)
	{
		count = *order++;

		if (!count)
			break;

		if (count < 0)
		{
			count = -count;
			TRI_TYPE = GL_TRIANGLE_FAN;
		}
		else TRI_TYPE = GL_TRIANGLE_STRIP;

		aliasVerts = VArray;
		VCount = 0;

		do
		{
			// vertex co-ords
			VectorSubtract (verts2->v, verts1->v, d);

			aliasVerts[0] = verts1->v[0] + (blend * d[0]);
			aliasVerts[1] = verts1->v[1] + (blend * d[1]);
			aliasVerts[2] = verts1->v[2] + (blend * d[2]);

			// texture co-ords
			aliasVerts[3] = ((float *)order)[0];
			aliasVerts[4] = ((float *)order)[1];

			// lighting
			aliasVerts[5] = Light;
			aliasVerts[6] = Light;
			aliasVerts[7] = Light;
			aliasVerts[8] = paliashdr->translevel;

			order += 2;
			verts1++;
			verts2++;

			aliasVerts += 9;
			VCount++;
		} while (--count);

		glDrawArrays (TRI_TYPE, 0, VCount);
	}
}


// don't store shadow info here
void GL_DrawAliasFrameNOLERPFLAT (aliashdr_t *paliashdr, int posenum, float Light)
{
	drawvertx_t	*verts;
	int		*order;
	int		count;
	float	*aliasVerts;
	GLenum	TRI_TYPE;
	int		VCount;

	lastposenum = posenum;

	verts = (drawvertx_t *)((byte *)paliashdr + paliashdr->posedata);
	verts += posenum * paliashdr->poseverts;
	order = (int *)((byte *)paliashdr + paliashdr->commands);

	while (1)
	{
		count = *order++;

		if (!count)
			break;

		if (count < 0)
		{
			count = -count;
			TRI_TYPE = GL_TRIANGLE_FAN;
		}
		else TRI_TYPE = GL_TRIANGLE_STRIP;

		aliasVerts = VArray;
		VCount = 0;

		do
		{
			aliasVerts[0] = verts->v[0];
			aliasVerts[1] = verts->v[1];
			aliasVerts[2] = verts->v[2];

			aliasVerts[3] = ((float *)order)[0];
			aliasVerts[4] = ((float *)order)[1];

			aliasVerts[5] = Light;
			aliasVerts[6] = Light;
			aliasVerts[7] = Light;
			aliasVerts[8] = paliashdr->translevel;

			order += 2;
			verts++;

			aliasVerts += 9;
			VCount++;
		} while (--count);

		glDrawArrays (TRI_TYPE, 0, VCount);
	}
}


void R_SetupAliasFrameClassic (int frame, aliashdr_t *paliashdr, float Light)
{
	int pose;
	int numposes;

	float interval;

	if ((frame >= paliashdr->numframes) || (frame < 0)) frame = 0;

	pose = paliashdr->frames[frame].firstpose;
	numposes = paliashdr->frames[frame].numposes;

	if (numposes > 1)
	{
		interval = paliashdr->frames[frame].interval;
		pose += (int)(realtime / interval) % numposes;
	}

	// never interpolate these cos we're using temporary entities
	GL_DrawAliasFrameNOLERPFLAT (paliashdr, pose, Light);
}


/*
=================
R_SetupAliasFrame

contains full interpolation code - it doesn't make any difference having it, cos
even if we don't interpolate, nothing adverse is done here...
=================
*/
void R_SetupAliasFrame (int frame, struct aliashdr_s *paliashdr, image_t *fbr, entity_t *e, qboolean viewalias)
{
	int				pose, numposes;
	float			interval;
	float			blend;
	qboolean		interpolate;
	qboolean		useFBR = true;

	if ((frame >= paliashdr->numframes) || (frame < 0))
	{
		Con_DPrintf ("R_AliasSetupFrame: no such frame %d\n", frame);
		frame = 0;
	}

	pose = paliashdr->frames[frame].firstpose;
	numposes = paliashdr->frames[frame].numposes;

	if (numposes > 1)
	{
		interval = paliashdr->frames[frame].interval;
		pose += (int)(realtime / interval) % numposes;

		// not really needed for non-interpolated mdls, but does no harm
		e->frameInterval = interval;
	}
	else e->frameInterval = 0.1;

	if (e->pose2 != pose)
	{
		e->frameStartTime = realtime;
		e->pose1 = e->pose2;
		e->pose2 = pose;
		blend = 0;
	}
	else blend = (realtime - e->frameStartTime) / e->frameInterval;

	// don't let blend pass 1
	if (cl.paused || blend > 1.0) blend = 1.0;

	interpolate = paliashdr->interpolate;

	// only one pose
	if (paliashdr->numposes == 1) interpolate = false;

	// nailgun spinning down bug
	if (viewalias && frame == 0) interpolate = false;

	// don;t interpolate the first few frames (fix long standing Q1 interpolation bug where
	// model frames haven't yet settled down)
	if (r_framecount < 10) interpolate = false;

	// no fullbright muzzle-flashes on frame 0
	if (viewalias && frame == 0) useFBR = false;

	// the drawing function to use depends on the model type and various MH flags set
	// in the header when it's loaded.  there are 4 basic variations - interpolated,
	// not interpolated, dynamically lit or flatly lit.  this helps speed by sending
	// drawing through the correct routine with the minimum of calculations necessary
	// to support what is required
	if (interpolate)
		GL_DrawAliasFrameLIGHTLERP (paliashdr, e->pose1, e->pose2, blend);
	else GL_DrawAliasFrameLIGHTNOLERP (paliashdr, pose);

	// this is a KILLER
	if (fbr && useFBR)
	{
		// we can get another coupla frames by sorting FBs out
		// shut down any fog
		R_SetFog (FOG_NONE);

		// no interpolation on muzzleflashes
		if (viewalias) interpolate = false;

		glBindTexture (GL_TEXTURE_2D, fbr->texnum);

		// one of these doesn't properly support translucency...
		// no point in using the stored verts cos we may not interpolate this even if we do
		// interpolate the original mdl
		if (interpolate)
			GL_DrawAliasFrameLERPFLAT (paliashdr, e->pose1, e->pose2, blend, 1.0);
		else GL_DrawAliasFrameNOLERPFLAT (paliashdr, pose, 1.0);
	}
}


void R_LightPoint (vec3_t p, vec3_t colour);

extern vec3_t lightspot;
extern mplane_t *lightplane;

void R_LightAliasModel (entity_t *e)
{
	int			i, j;
	float		an;
	float		max, mody;
	float		ang_ceil, ang_floor;

	// always do this cos we need lightspot for shadows
	R_LightPoint (e->origin, shadelight);

	// save out lightspot for shadows
	e->lightspot[0] = lightspot[0];
	e->lightspot[1] = lightspot[1];
	e->lightspot[2] = lightspot[2];
	e->lightplane = lightplane;

	// rotating objects (i.e. pick-ups) will always have some constant light in order to
	// make them noticeable...
	if (e->model->flags & EF_ROTATE)
	{
		shadelight[0] = 192;
		shadelight[1] = 192;
		shadelight[2] = 192;
	}

	// always give the gun some light
	if (e == &cl.viewent)
	{
		if (shadelight[0] < 16) shadelight[0] = 16;
		if (shadelight[1] < 16) shadelight[1] = 16;
		if (shadelight[2] < 16) shadelight[2] = 16;
	}

	// never allow players to go totally black
	if ((e - cl_entities) >= 1 && (e - cl_entities) <= cl.maxclients)
	{
		if (shadelight[0] < 8) shadelight[0] = 8;
		if (shadelight[1] < 8) shadelight[1] = 8;
		if (shadelight[2] < 8) shadelight[2] = 8;
	}

	// clamp lighting so it doesn't overbright as much
	if (shadelight[0] > 192) shadelight[0] = 192;
	if (shadelight[1] > 192) shadelight[1] = 192;
	if (shadelight[2] > 192) shadelight[2] = 192;

	// light interpolation
	lightLerpOffset = (e->angles[1] + e->angles[0]) * (SHADEDOT_QUANT / 360.0);

	ang_ceil = ceil (lightLerpOffset);
	ang_floor = floor (lightLerpOffset);

	lightLerpOffset = ang_ceil - lightLerpOffset;

	shadedots = r_avertexnormal_dots[(int) ang_ceil & (SHADEDOT_QUANT - 1)];
	shadedots2 = r_avertexnormal_dots[(int) ang_floor & (SHADEDOT_QUANT - 1)];

	// set to a 0 to 1 scale
	shadelight[0] = shadelight[0] / 224.0;
	shadelight[1] = shadelight[1] / 224.0;
	shadelight[2] = shadelight[2] / 224.0;

	for (i = 0; i < 3; i++)
	{
		if (e->lastShadeLight[i])
			shadelight[i] = (shadelight[i] + e->lastShadeLight[i]) / 2;

		e->lastShadeLight[i] = shadelight[i];
	}
}


/*
=================
R_DrawAliasModel

=================
*/
void R_DrawAliasModelSBar (entity_t *e, float Light, int skin, float trans, float scale)
{
	struct aliashdr_s *paliashdr = (struct aliashdr_s *) Mod_Extradata (e->model);
	float oldTrans;

    glPushMatrix ();

	// apply modelview transformations
	glTranslatef (e->origin[0], -e->origin[1], e->origin[2]);
	glRotatef (e->angles[1],  0, 0, 1);
	glRotatef (-e->angles[0],  0, 1, 0);
	glRotatef (e->angles[2],  1, 0, 0);

	// some models need scaling
	if (scale) glScalef (scale, scale, scale);

	glBindTexture (GL_TEXTURE_2D, paliashdr->gl_texture[e->skinnum + skin][0]->texnum);
	R_SetupAliasFrameClassic (e->frame, paliashdr, Light);

	glPopMatrix ();

	c_alias_polys++;
}


void R_DrawBrushModelSbar (entity_t *e, float Light)
{
	model_t *clmodel = e->model;
	int i;
	int j;
	msurface_t *surf;
	float *v;
	glpoly_t *p;

	float boxverts[3];

	float modHeight = e->model->maxs[2] - e->model->mins[2];
	float modWidth = e->model->maxs[1] - e->model->mins[1];
	float modDepth = e->model->maxs[0] - e->model->mins[0];

	glPushMatrix ();

	// apply modelview transformations
	glTranslatef (e->origin[0], -e->origin[1], e->origin[2]);

	glRotatef (e->angles[1],  0, 0, 1);
	glRotatef (-e->angles[0],  0, 1, 0);
	glRotatef (e->angles[2],  1, 0, 0);

	// and draw it
	surf = &clmodel->surfaces[clmodel->firstmodelsurface];

	glColor3f (Light, Light, Light);

	// the assumption is made that we have 6 surfs each with 4 verts
	for (i = 0; i < e->model->numsurfaces; i++, surf++)
	{
		glBindTexture (GL_TEXTURE_2D, surf->texinfo->texture->gl_texture->texnum);

		p = surf->polys;
		v = p->verts;

		glBegin (GL_QUADS);

		for (j = 0; j < surf->polys->numverts; j++, v += VERTEXSIZE)
		{
			// scale the verts so that it's sized correctly for the sbar
			boxverts[0] = ((v[0] * 24) / modDepth) - 12;
			boxverts[1] = ((v[1] * 24) / modWidth) - 12;
			boxverts[2] = ((v[2] * 24) / modHeight) - 12;

			glTexCoord2fv (&v[3]);
			glVertex3fv (boxverts);
		}

		glEnd ();
	}

	glPopMatrix ();
}


extern qboolean waterthisframe;
void R_SmokeEffect (vec3_t org);
qboolean shadowsthisframe;

void R_DrawAliasModel (entity_t *e, qboolean viewalias)
{
	int					i;
	model_t				*clmodel;
	vec3_t				mins, maxs;
	struct aliashdr_s	*paliashdr;
	int					anim;
	image_t				*fbr;

	vec3_t				lightorigin;
	float				radius;
	vec3_t				v;
	int					k;

	mleaf_t				*aliasleaf;

	clmodel = e->model;

	VectorAdd (e->origin, clmodel->mins, mins);
	VectorAdd (e->origin, clmodel->maxs, maxs);

	// we always want to draw the viewmodel
	if (R_CullBox (mins, maxs) && !viewalias)
		return;

	// we don't have fog just right yet
	if (waterthisframe)
	{
		// find which leaf the model is in
		aliasleaf = Mod_PointInLeaf (e->origin, cl.worldmodel);
	}
	else
	{
		// the leaf the alias model is in always has the same properties as r_viewleaf
		aliasleaf = r_viewleaf;
	}

	c_alias_polys++;

	VectorCopy (e->origin, r_entorigin);
	VectorSubtract (r_origin, r_entorigin, modelorg);

	// get lighting information
	R_LightAliasModel (e);

	// locate the proper data
	paliashdr = (struct aliashdr_s *) Mod_Extradata (e->model);

	if (paliashdr->shadow) shadowsthisframe = true;

    glPushMatrix ();

	// set the appropriate fog - some leafs don't have caustics set - in general,
	// these are behind teleports and suchlike, so there is no visual harm done.
	// don't fog if it's lava!!! (you'll die anyway, and besides, it wrecks stuff like in e1m7)
	if ((aliasleaf->contents == CONTENTS_WATER ||
		aliasleaf->contents == CONTENTS_SLIME) && aliasleaf->caustic)
	{
		water_fog[0] = aliasleaf->caustic->uwater_fog[0];
		water_fog[1] = aliasleaf->caustic->uwater_fog[1];
		water_fog[2] = aliasleaf->caustic->uwater_fog[2];
		water_fog[3] = aliasleaf->caustic->uwater_fog[3];

		// put standard openGL fog on the model
		R_SetFog (FOG_WATER);
	}

	// don't interpolate the viewmodel position/rotation
	R_RotateForEntity (e, !viewalias);

	anim = (int)(realtime*10) & 3;

	glBindTexture (GL_TEXTURE_2D, paliashdr->gl_texture[e->skinnum][anim]->texnum);

	// we can't dynamically colormap textures, so they are cached
	// seperately for the players.  Heads are just uncolored.
	fbr = paliashdr->fullbright[e->skinnum][anim];

	if (e->colormap != vid.colormap && !gl_nocolors.value)
	{
		i = e - cl_entities;

		if (i >= 1 && i<=cl.maxclients)
		{
			glBindTexture (GL_TEXTURE_2D, playertextures - 1 + i);
			fbr = NULL;
		}
	}

	if (viewalias)
	{
		if (cl.items & IT_INVISIBILITY)
		{
			paliashdr->trans = true;
			paliashdr->translevel = 0.25;
		}
		else
		{
			paliashdr->trans = false;
			paliashdr->translevel = 1.0;
		}
	}

	R_SetupAliasFrame (e->frame, paliashdr, fbr, e, viewalias);

	// shut down any fog
	R_SetFog (FOG_NONE);

	glPopMatrix ();

	if (paliashdr->glow)
	{
		// set up for drawing later
		VectorCopy (currententity->origin, lightorigin);
		radius = paliashdr->radius;

		if (paliashdr->torch)
		{
			lightorigin[2] += 8.0;
			radius += (float) (rand () & 3);

			// hmmmm - looks bad...
			//R_SmokeEffect (lightorigin);
		}
		else if (paliashdr->bolt)
		{
			radius += (float) (rand () & 7);
		}
		else if (paliashdr->powerup)
		{
			lightorigin[2] += 20.0;
			radius += (slowcycle * 40);
		}
		else if (paliashdr->missile)
		{
			// move the glow behind the rocket - fixme - do a fast sin/cos lookup table for 
			// this would warpsin do? - actually, what's the point since there's really only
			// ever gonna be a handful of these per frame...?
			lightorigin[0] += cos (e->angles[1] * DIV180MULTPI) * (-20.0f);
			lightorigin[1] += sin (e->angles[1] * DIV180MULTPI) * (-20.0f);
			lightorigin[2] += sin (e->angles[0] * DIV180MULTPI) * (-20.0f);
		}
		else if (paliashdr->shambler)
		{
			// shambler magic hack cos the origin of the s_light mdl used for this is
			// in the shamblers groin - which, though interesting, is just not right...
			lightorigin[2] += 75.0;
			radius = 60.0;
		}

		R_AddGlowEffect (paliashdr->glowColours[0],
						 paliashdr->glowColours[1],
						 paliashdr->glowColours[2],
						 radius,
						 lightorigin);
	}
}

//==================================================================================


/*
=================
R_DrawAliasModelShadow

here's how it SHOULD be done.  heh heh heh.

we stored out the verts for the current frame when drawing the model, so we can just use
them again here without having to run through frame setup and blend calculation.  instant
interpolated shadows with half the CPU hit!!!  woo-oooooo-oooooooo-oooo!!!

everything is batched up and drawn in pass of it;s own so as to be able to do it with only
one set of state changes.  the modelview matrix for each alias model is saved out to the
entity structure so we can just load it straight in rather than sending it through
r_rotateforentity (which can be expensive if there's loads of sins and cos's).
=================
*/
void R_DrawAliasModelShadow (entity_t *ent)
{
	int count;
	int VCount;
	float	*aliasVerts;
	GLenum	TRI_TYPE;
	float originhack;
	float height;
	float s1, c1;
	struct aliashdr_s *paliashdr = (struct aliashdr_s *) Mod_Extradata (ent->model);
	vec3_t shadenull;

	// no shadows on this ent
	if (!paliashdr->shadow) return;

	// no lightplane so don't even bother
	if (!ent->lightplane) return;
// 0.017453292519943295769236907684886
	// load the models modelview matrix directly
	// see how this is cleaner than running through the transforms
	glLoadMatrixf (ent->MViewMatrix);

	aliasverts = 0;

	// use the DIST_EPSILON from world.c (moved into quakedef.h)
	height = DIST_EPSILON - (ent->origin[2] - ent->lightspot[2]);

	//s1 = sin (ent->angles[1] / 180 * M_PI);
	//c1 = cos (ent->angles[1] / 180 * M_PI);
	s1 = sin (ent->angles[1] * DIV180MULTPI);
	c1 = cos (ent->angles[1] * DIV180MULTPI);

	while (1)
	{
		count = (int) ent->shadowverts[aliasverts++];

		if (!count) break;

		if (count < 0)
		{
			count = -count;
			TRI_TYPE = GL_TRIANGLE_FAN;
		}
		else TRI_TYPE = GL_TRIANGLE_STRIP;

		aliasVerts = VArray;
		VCount = 0;

		do
		{
			// alias verts are precaled at load time and stored as floats
			aliasVerts[0] = ent->shadowverts[aliasverts++];
			aliasVerts[1] = ent->shadowverts[aliasverts++];
			aliasVerts[2] = ent->shadowverts[aliasverts++];

			// collapse the verts on a vertical plane
			// i'm not angling the verts cos i want the shadow directly under the model, but i do the
			// aliasVerts[2] thing anyway cos (a) it keeps the value of aliasverts (lcase v) consistent,
			// and (b) it's easier for someone to mod my engine if they want this way.
			aliasVerts[2] = height;

			// do the sloped surface thing.  this isn't perfect - e.g. where a sloped surface meets a regular
			// one, it gets fucked.  may have to project each point in each vert downwards
			aliasVerts[2] += ((aliasVerts[1] * (s1 * ent->lightplane->normal[0])) -
							 (aliasVerts[0] * (c1 * ent->lightplane->normal[0])) -
							 (aliasVerts[0] * (s1 * ent->lightplane->normal[1])) -
							 (aliasVerts[1] * (c1 * ent->lightplane->normal[1]))) +  
							 ((1.0 - ent->lightplane->normal[2]) * 20);

			aliasVerts += 3;
			VCount++;
		} while (--count);

		glDrawArrays (TRI_TYPE, 0, VCount);
	}
}


qboolean haveStencil = false;

void R_DrawAliasShadows (void)
{
	int i;

	glEnableClientState (GL_VERTEX_ARRAY);
	glVertexPointer (3, GL_FLOAT, (3 * sizeof (GLfloat)), &VArray[0]);
	glDisable (GL_TEXTURE_2D);
	glColor4f (0, 0, 0, 0.5);

	glEnable (GL_BLEND);
	glDepthMask (GL_FALSE);

	if (haveStencil)
	{
		glEnable (GL_STENCIL_TEST);
		glStencilFunc (GL_EQUAL, 1, 2);
		glStencilOp (GL_KEEP, GL_KEEP, GL_INCR);
	}

	for (i = 0; i < cl_numvisedicts; i++)
	{
		currententity = cl_visedicts[i];

		// don;t bother shadowing the view entity
		if (currententity == &cl_entities[cl.viewentity]) continue;

		// draw the shadow
		if (currententity->model->type == mod_alias) R_DrawAliasModelShadow (currententity);
	}

	// done - restore state
	glDisable (GL_BLEND);
	glDepthMask (GL_TRUE);

	if (haveStencil) glDisable (GL_STENCIL_TEST);

	glDisableClientState (GL_VERTEX_ARRAY);
	glEnable (GL_TEXTURE_2D);
	glColor4f (1, 1, 1, 1);

	// go back to the world matrix
	glLoadMatrixf (cl_entities[0].MViewMatrix);
}

/*
=============
R_DrawEntitiesOnList
=============
*/
void R_DrawEntitiesOnList (void)
{
	int		i;

	if (!r_drawentities.value)
		return;

	// draw seperately, because of alpha blending
	for (i = 0; i < cl_numvisedicts; i++)
	{
		currententity = cl_visedicts[i];

		// we're only drawing brush models to begin with - but
		// we will count the number of other models to see if
		// we actually need to draw them.
		switch (currententity->model->type)
		{
		case mod_brush:
			numBrushModels++;
			break;

		case mod_alias:
			numAliasModels++;
			break;

		case mod_sprite:
			numSpriteModels++;
			break;

		default:
			break;
		}
	}

	if (numBrushModels)
	{
		for (i = 0; i < cl_numvisedicts; i++)
		{
			currententity = cl_visedicts[i];

			switch (currententity->model->type)
			{
			case mod_brush:
				R_DrawBrushModel (currententity);
				break;

			default:
				break;
			}
		}
	}

	glColor3f (1,1,1);

	// alias models
	if (numAliasModels)
	{
		// moved these out of R_DrawAliasModel so they're not done once for
		// each model that has to be drawn - slight speedup...
		glEnableClientState (GL_VERTEX_ARRAY);
		glVertexPointer (3, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[0]);

		glEnableClientState (GL_TEXTURE_COORD_ARRAY);
		glClientActiveTexture (GL_TEXTURE0);
		glTexCoordPointer (2, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[3]);

		// because ClientActiveTexture only affects the TexCoordPointer and we're using
		// ColorPointer rather than lightmaps on these babes, we can't multitexture them.
		glEnableClientState (GL_COLOR_ARRAY);
		glColorPointer (4, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[5]);

		glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);	// for translucency
		glEnable (GL_BLEND);	// for fullbrights

		// no shadows until we find one
		shadowsthisframe = false;

		for (i = 0; i < cl_numvisedicts; i++)
		{
			currententity = cl_visedicts[i];

			if (currententity == &cl_entities[cl.viewentity]) 
				currententity->angles[0] *= 0.3;

			switch (currententity->model->type)
			{
			case mod_alias:
				// an alias model's origin is relative to itself
				R_DrawAliasModel (currententity, false);
				break;

			default:
				break;
			}
		}

		glDisable (GL_BLEND);
		glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

		glDisableClientState (GL_VERTEX_ARRAY);
		glDisableClientState (GL_TEXTURE_COORD_ARRAY);
		glDisableClientState (GL_COLOR_ARRAY);

		if (r_shadows.value && shadowsthisframe) R_DrawAliasShadows ();
	}

	if (numSpriteModels)
	{
		for (i = 0; i < cl_numvisedicts; i++)
		{
			currententity = cl_visedicts[i];

			switch (currententity->model->type)
			{
			case mod_sprite:
				R_DrawSpriteModel (currententity);
				break;

			default:
				break;
			}
		}
	}
}

/*
=============
R_DrawViewModel
=============
*/
// hack - zerstorer cutscenes
qboolean sbardraw;

void R_DrawViewModel (void)
{
	if (!r_drawviewmodel.value) return;
	if (chase_active.value) return;
	if (envmap) return;
	if (!r_drawentities.value) return;
	if (cl.stats[STAT_HEALTH] <= 0) return;

	currententity = &cl.viewent;

	if (!currententity->model) return;

	// all of the lighting stuff that was in here was recalculated anyway in
	// R_DrawAliasModel so i took it out...
	// amazing no-one noticed THAT before...

	// need to do this again cos it's drawn seperately
	glEnableClientState (GL_VERTEX_ARRAY);
	glVertexPointer (3, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[0]);

	glEnableClientState (GL_TEXTURE_COORD_ARRAY);
	glClientActiveTexture (GL_TEXTURE0);
	glTexCoordPointer (2, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[3]);

	glEnableClientState (GL_COLOR_ARRAY);
	glColorPointer (4, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[5]);

	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);	// for translucency
	glEnable (GL_BLEND);	// for fullbrights

	// hack the depth range to prevent view model from poking into walls
	// MH - and other alias models (torches, etc) on walls!!! (was 0.3)
	glDepthRange (gldepthmin, gldepthmin + 0.1 * (gldepthmax - gldepthmin));
	R_DrawAliasModel (currententity, true);
	glDepthRange (gldepthmin, gldepthmax);

	// need to do this again cos it's drawn seperately
	glDisable (GL_BLEND);

	glDisableClientState (GL_VERTEX_ARRAY);
	glDisableClientState (GL_TEXTURE_COORD_ARRAY);
	glDisableClientState (GL_COLOR_ARRAY);

	// hack - zerstorer cutscenes
	// only draw the HUD if we have the viewmodel
	// (not good with chase_active 1)
	sbardraw = true;
}


/*
============
R_PolyBlend
============
*/
void R_PolyBlend (void)
{
	if (!gl_polyblend.value) return;
	if (!v_blend[3]) return;

	glDisable (GL_ALPHA_TEST);
	glEnable (GL_BLEND);
	glDisable (GL_DEPTH_TEST);
	glDisable (GL_TEXTURE_2D);

    glLoadIdentity ();

    glRotatef (-90,  1, 0, 0);	    // put Z going up
    glRotatef (90,  0, 0, 1);	    // put Z going up

	glColor4fv (v_blend);

	glBegin (GL_QUADS);

	glVertex3f (10, 100, 100);
	glVertex3f (10, -100, 100);
	glVertex3f (10, -100, -100);
	glVertex3f (10, 100, -100);
	glEnd ();

	// restore the colour or items will flash when picked up!!!
	glColor4f (1, 1, 1, 1);

	glDisable (GL_BLEND);
	glEnable (GL_TEXTURE_2D);
	glEnable (GL_ALPHA_TEST);
}


int SignbitsForPlane (mplane_t *out)
{
	int	bits, j;

	// for fast box on planeside test

	bits = 0;
	for (j=0 ; j<3 ; j++)
	{
		if (out->normal[j] < 0)
			bits |= 1<<j;
	}
	return bits;
}


void R_SetFrustum (void)
{
	int		i;

	// rotate VPN right by FOV_X/2 degrees
	RotatePointAroundVector( frustum[0].normal, vup, vpn, -(90-r_refdef.fov_x / 2 ) );

	// rotate VPN left by FOV_X/2 degrees
	RotatePointAroundVector( frustum[1].normal, vup, vpn, 90-r_refdef.fov_x / 2 );

	// rotate VPN up by FOV_X/2 degrees
	RotatePointAroundVector( frustum[2].normal, vright, vpn, 90-r_refdef.fov_y / 2 );

	// rotate VPN down by FOV_X/2 degrees
	RotatePointAroundVector( frustum[3].normal, vright, vpn, -( 90 - r_refdef.fov_y / 2 ) );

	for (i=0 ; i<4 ; i++)
	{
		frustum[i].type = PLANE_ANYZ;
		frustum[i].dist = DotProduct (r_origin, frustum[i].normal);
		frustum[i].signbits = SignbitsForPlane (&frustum[i]);
	}
}



/*
===============
R_SetupFrame
===============
*/
void R_SetupFrame (void)
{
	// set up the light animations for this frame
	R_AnimateLight ();

	r_framecount++;

	// build the transformation matrix for the given view angles
	VectorCopy (r_refdef.vieworg, r_origin);

	AngleVectors (r_refdef.viewangles, vpn, vright, vup);

	// current viewleaf
	r_oldviewleaf = r_viewleaf;
	r_viewleaf = Mod_PointInLeaf (r_origin, cl.worldmodel);

	V_SetContentsColor (r_viewleaf->contents);
	V_CalcBlend ();

	c_brush_polys = 0;
	c_alias_polys = 0;
}


/*
=============
R_SetupGL
=============
*/
extern qboolean inMapshot;

void R_SetupGL (void)
{
	extern	int glwidth, glheight;
	float aspect;

	// keep the correct screen aspect at all times
	aspect = (float) glwidth / (float) glheight;

	// temporarily set the viewport to the size of our mapshot textures
	if (inMapshot) glwidth = glheight = 256;

	glViewport (glx, gly, glwidth, glheight);

	glMatrixMode(GL_PROJECTION);
    glLoadIdentity ();

	// set the far clipping plane to r_farclip.value - this *could* in theory be dynamically set per
	// frame to what's actually required, but it seems more bother than it's worth, and figuring out
	// what's required for the setting may hit the ol' speed
	gluPerspective (r_refdef.fov_y, aspect, 4, r_farclip.value);

	glCullFace(GL_FRONT);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity ();

	// put Z going up and Y going deep
    glRotatef (-90,  1, 0, 0);
    glRotatef (90,  0, 0, 1);

	// set the view according to where the player actually is in the map!!!
    glRotatef (-r_refdef.viewangles[2],  1, 0, 0);
    glRotatef (-r_refdef.viewangles[0],  0, 1, 0);
    glRotatef (-r_refdef.viewangles[1],  0, 0, 1);

    glTranslatef (-r_refdef.vieworg[0], -r_refdef.vieworg[1], -r_refdef.vieworg[2]);

	glGetFloatv (GL_MODELVIEW_MATRIX, cl_entities[0].MViewMatrix);

	// set drawing parms
	if (gl_cull.value)
		glEnable (GL_CULL_FACE);
	else
		glDisable (GL_CULL_FACE);

	glDisable(GL_BLEND);
	glDisable(GL_ALPHA_TEST);
	glEnable(GL_DEPTH_TEST);
	glEnable (GL_TEXTURE_2D);
}


/*
================
R_RenderScene

r_refdef must be set before the first call
================
*/
void R_RenderScene (void)
{
	// initialise our counters
	numAliasShadows = 0;
	numBrushModels = 0;
	numAliasModels = 0;
	numSpriteModels = 0;

	// draw order comes from the old MHQuake 6
	// with a few modifications
	R_SetupFrame ();
	R_SetFrustum ();
	R_SetupGL ();
	R_MarkLeaves ();	// done here so we know if we're in water
	R_DrawWorld ();		// adds static entities to the list

	S_ExtraUpdate ();	// don't let sound get messed up if going slow

	glDisable (GL_ALPHA_TEST);
	R_DrawEntitiesOnList ();

	R_DrawWaterSurfaces ();

	// none of the remainder should be fogged
	R_DrawParticles ();
	R_DrawDecals ();
	R_RenderGlowEffects ();

	// viewmodel is last of all
	if (!inMapshot) R_DrawViewModel ();
}


/*
=============
R_Clear
=============
*/
void R_Clear (void)
{
	if (gl_ztrick.value)
	{
		static int trickframe;

		if (gl_clear.value)
			glClear (GL_COLOR_BUFFER_BIT);

		trickframe++;

		if (trickframe & 1)
		{
			gldepthmin = 0;
			gldepthmax = 0.49999;
			glDepthFunc (GL_LEQUAL);
		}
		else
		{
			gldepthmin = 1;
			gldepthmax = 0.5;
			glDepthFunc (GL_GEQUAL);
		}
	}
	else
	{
		if (gl_clear.value)
			glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		else
			glClear (GL_DEPTH_BUFFER_BIT);

		gldepthmin = 0;
		gldepthmax = 1;
		glDepthFunc (GL_LEQUAL);
	}

	if (haveStencil && r_shadows.value && shadowsthisframe)
	{
		glClearStencil (1);
		glClear (GL_STENCIL_BUFFER_BIT);
	}

	glDepthRange (gldepthmin, gldepthmax);
}


/*
================
R_RenderView

r_refdef must be set before the first call
================
*/
cvar_t r_skyspeed = {"r_skyspeed", "10", true};

int maxAliasVerts = 0;
extern float rotateBack;
extern float rotateFore;

extern int addSurfs;

void GL_DrawAutomap (void);
void MDLDump (void);

void R_RenderView (void)
{
	if (!r_worldentity.model || !cl.worldmodel) return;

	if (r_framecount == 1)
	{
		// i like this...
		SCR_CenterPrint (cl.levelname);
	}

	// we want this every frame whether we draw or not
	frametime = cl.time - cl.oldtime;

	// one pulse cycle (down to up and back again) every second
	slowpulse = slowpulse + (frametime * slowpulsedir);

	if (slowpulse > 0.5)
	{
		slowpulse = 0.5;
		slowpulsedir = -1;
	}

	if (slowpulse < 0.0)
	{
		slowpulse = 0.0;
		slowpulsedir = 1;
	}

	slowcycle = slowpulse;

	// sky rotation
	// bound rotation speed 0 to 100
	if (r_skyspeed.value < 0) r_skyspeed.value = 0;
	if (r_skyspeed.value > 100) r_skyspeed.value = 100;

	// always rotate even if we're paused!!!
	rotateBack = anglemod (realtime * (2.5 * r_skyspeed.value));
	rotateFore = anglemod (realtime * (4.0 * r_skyspeed.value));

	c_brush_polys = 0;
	c_alias_polys = 0;

	// dont allow the automap in the first few frames
	if (automapOn && r_framecount > 5)
	{
		GL_DrawAutomap ();
		return;
	}

	if (gl_finish.value) glFinish ();

	R_Clear ();

	// hack - zerstorer cutscenes
	sbardraw = false;

	// ---------------------------------------------------------------
	// stainmaps
	num_newstains = r_numstains;
	newstains = r_stains;
	r_numstains = 0;
	// ---------------------------------------------------------------

	R_RenderScene ();

	R_PolyBlend ();

	// more test code
	//MDLDump ();
}


