/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// r_misc.c

#include "quakedef.h"
#include "glext.h"
#include "gl_decals.h"


/*
==================
R_InitTextures
==================
*/
void GL_Upload8 (byte *data, int width, int height, qboolean mipmap, qboolean alpha);

texture_t *r_notexture_mip;
texture_t *r_white_texture;


void R_InitTextures (void)
{
	int x, y, m;
	byte *dest;

	// create a simple checkerboard texture for the default
	r_notexture_mip = Hunk_AllocName (sizeof (texture_t) + 16 * 16 + 8 * 8 + 4 * 4 + 2 * 2, "notexture");

	r_notexture_mip->width = r_notexture_mip->height = 16;
	r_notexture_mip->offsets[0] = sizeof (texture_t);
	r_notexture_mip->offsets[1] = r_notexture_mip->offsets[0] + 16 * 16;
	r_notexture_mip->offsets[2] = r_notexture_mip->offsets[1] + 8 * 8;
	r_notexture_mip->offsets[3] = r_notexture_mip->offsets[2] + 4 * 4;

	for (m = 0; m < 4; m++)
	{
		dest = (byte *) r_notexture_mip + r_notexture_mip->offsets[m];

		for (y = 0; y < (16 >> m); y++)
		{
			for (x = 0; x < (16 >> m); x++)
			{
				if ((y < (8 >> m)) ^ (x < (8 >> m)))
					*dest++ = 0;
				else
					*dest++ = 0xff;
			}
		}
	}

	// create a simple checkerboard texture for the default
	r_white_texture = Hunk_AllocName (sizeof (texture_t) + 16 * 16 + 8 * 8 + 4 * 4 + 2 * 2, "|white|");

	r_white_texture->width = r_white_texture->height = 16;
	r_white_texture->offsets[0] = sizeof (texture_t);
	r_white_texture->offsets[1] = r_white_texture->offsets[0] + 16 * 16;
	r_white_texture->offsets[2] = r_white_texture->offsets[1] + 8 * 8;
	r_white_texture->offsets[3] = r_white_texture->offsets[2] + 4 * 4;

	for (m = 0; m < 4; m++)
	{
		dest = (byte *) r_white_texture + r_white_texture->offsets[m];

		for (y = 0; y < (16 >> m); y++)
		{
			for (x = 0; x < (16 >> m); x++)
			{
				*dest++ = 254;
			}
		}
	}
}


void R_InitParticleTexture (void)
{
}

/*
===============
R_Envmap_f

Grab six views for environment mapping tests
===============
*/
void R_Envmap_f (void)
{
}


void R_MapData_f (void)
{
	int i;
	int leafsurfs;

	if (!cl.worldmodel) return;

	Con_Printf ("Map: %i\n", cl.worldmodel->name);
	Con_Printf ("- Textures: %i\n", cl.worldmodel->numtextures);
	Con_Printf ("- Surfaces: %i\n", cl.worldmodel->numsurfaces);
	Con_Printf ("- Leafs: %i\n", cl.worldmodel->numleafs);
	Con_Printf ("- Nodes: %i\n", cl.worldmodel->numnodes);
	Con_Printf ("- MarkSurfaces: %i\n", cl.worldmodel->nummarksurfaces);

	leafsurfs = 0;

	for (i = 0; i < cl.worldmodel->numleafs + 1; i++)
	{
		leafsurfs += cl.worldmodel->leafs[i].nummarksurfaces;
	}

	Con_Printf ("- LeafSurfaces: %i\n", leafsurfs);
}


void R_DumpIndividualSurface (msurface_t *surf, FILE *f)
{
}


void R_DumpSolidSurf (msurface_t *surf, FILE *f)
{
}


void R_SurfDump_f (void)
{
	int i;
	FILE *f;
	msurface_t *surf;

	if (!cl.worldmodel) return;

	f = fopen ("surfdump.txt", "w");

	for (i = 0, surf = cl.worldmodel->surfaces; i < cl.worldmodel->numsurfaces; i++, surf++)
		R_DumpIndividualSurface (surf, f);

	fclose (f);
}


void R_LMDump_f (void)
{
	int i;
	int j;
	int scount;

	FILE *f;
	msurface_t *surf;

	if (!cl.worldmodel) return;

	f = fopen ("lmdump.txt", "w");

	for (j = 0; ; j++)
	{
		scount = 0;

		for (i = 0, surf = cl.worldmodel->surfaces; i < cl.worldmodel->numsurfaces; i++, surf++)
			if (surf->lightmaptexturenum == j) scount++;

		if (!scount) break;

		fprintf (f, "LIGHTMAP %02i Used by %i Surfs\n", j, scount);
	}

	fprintf (f, "\n%i Lightmaps Used\n", j);

	fclose (f);
}


void R_LeafDump_f (void)
{
	int i;
	int j;
	FILE *f;

	if (!cl.worldmodel) return;

	f = fopen ("leafdump.txt", "w");

	for (i = 0; i < cl.worldmodel->numleafs + 1; i++)
	{
		switch (cl.worldmodel->leafs[i].contents)
		{
		case CONTENTS_SOLID:
			fprintf (f, "Leaf %4i Has CONTENTS_SOLID\n", cl.worldmodel->leafs[i].leafnum);
			break;

		case CONTENTS_EMPTY:
			fprintf (f, "Leaf %4i Has CONTENTS_EMPTY\n", cl.worldmodel->leafs[i].leafnum);
			break;

		case CONTENTS_WATER:
			fprintf (f, "Leaf %4i Has CONTENTS_WATER\n", cl.worldmodel->leafs[i].leafnum);
			break;

		case CONTENTS_LAVA:
			fprintf (f, "Leaf %4i Has CONTENTS_LAVA\n", cl.worldmodel->leafs[i].leafnum);
			break;

		case CONTENTS_SLIME:
			fprintf (f, "Leaf %4i Has CONTENTS_SLIME\n", cl.worldmodel->leafs[i].leafnum);
			break;

		case CONTENTS_SKY:
			fprintf (f, "Leaf %4i Has CONTENTS_SKY\n", cl.worldmodel->leafs[i].leafnum);
			break;

		case CONTENTS_ORIGIN:
			fprintf (f, "Leaf %4i Has CONTENTS_ORIGIN\n", cl.worldmodel->leafs[i].leafnum);
			break;

		case CONTENTS_CLIP:
			fprintf (f, "Leaf %4i Has CONTENTS_CLIP\n", cl.worldmodel->leafs[i].leafnum);
			break;

		default:
			fprintf (f, "Leaf %4i Has UNKNOWN CONTENTS!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n", cl.worldmodel->leafs[i].leafnum);
			break;
		}
	}

	fclose (f);
}


void R_VolDump_f (void)
{
	int i;
	FILE *f;
	mleaf_t *leaf;

	if (!cl.worldmodel) return;

	f = fopen ("voldump.txt", "w");

	for (i = 0; i < cl.worldmodel->numVolumes; i++)
	{
		fprintf (f, "VOLUME %i  Contents %i", i, cl.worldmodel->volumes[i].contents);

		leaf = cl.worldmodel->volumes[i].chain;

		if (cl.worldmodel->volumes[i].visframe != -2)
			fprintf (f, "      Drawing volume");

		fprintf (f, "\n\n");
	}

	fclose (f);
}


// exports visdata to a .VIS file
void R_Visexport_f (void)
{
}


/*
===============
R_Init
===============
*/
cvar_t gl_fogenable = {"gl_fogenable", "1", true};
cvar_t gl_fogstart = {"gl_fogstart", "100", true};
cvar_t gl_fogend = {"gl_fogend", "4096", true};
cvar_t gl_fogdensity = {"gl_fogdensity", "0.2", true};
cvar_t gl_fogr = {"gl_fogr", "0.1", true};
cvar_t gl_fogg = {"gl_fogg", "0.1", true};
cvar_t gl_fogb = {"gl_fogb", "0.1", true};
cvar_t gl_lockpvs = {"gl_lockpvs", "0", false};

cvar_t gl_foguser = {"gl_foguser", "0", true};

cvar_t gl_fogvol = {"gl_fogvol", "0", true};

extern cvar_t r_skyspeed;

int mapshottexture;
int currentmapshottexture[21];


void R_Mapshots_f (void)
{
	int i;

	for (i = 0; i < 21; i++)
	{
		Con_Printf ("Mapshot %02i   Texture Number: %i\n", i, currentmapshottexture[i]);
	}
}


void R_GL_Extensions_f (void)
{
	FILE *f;
	char *exts = (char *) glGetString (GL_EXTENSIONS);
	int i;

	for (i = 0; ; i++)
	{
		if (!exts[i]) break;

		if (exts[i] == ' ') exts[i] = '\n';
	}

	f = fopen ("glextensions.txt", "w");

	fprintf (f, "%s\n", exts);

	Con_Printf ("%s\n", exts);

	fclose (f);
}


void GL_InitLightmapTextures (void);
void GL_RSurfInit (void);
void R_Automap_f (void);
void R_AutomapScaleUp_f (void);
void R_AutomapScaleDown_f (void);

// true if we have a skybox
qboolean sbOK;

void R_Loadsky_f (void);
void R_UnLoadsky_f (void)
{
	sbOK = false;
}

cvar_t r_farclip = {"r_farclip", "4096"};


void R_Init (void)
{
	int i;
	extern byte *hunk_base;
	extern cvar_t gl_finish;

	Cmd_AddCommand ("timerefresh", R_TimeRefresh_f);
	Cmd_AddCommand ("loadsky", R_Loadsky_f);
	Cmd_AddCommand ("unloadsky", R_UnLoadsky_f);
	Cmd_AddCommand ("envmap", R_Envmap_f);
	Cmd_AddCommand ("pointfile", R_ReadPointFile_f);
	Cmd_AddCommand ("mapdata", R_MapData_f);
	Cmd_AddCommand ("surfdump", R_SurfDump_f);
	Cmd_AddCommand ("leafdump", R_LeafDump_f);
	Cmd_AddCommand ("voldump", R_VolDump_f);
	Cmd_AddCommand ("mapshots", R_Mapshots_f);
	Cmd_AddCommand ("lmdump", R_LMDump_f);

	Cmd_AddCommand ("visexport", R_Visexport_f);
	Cmd_AddCommand ("gl_extensions", R_GL_Extensions_f);

	Cmd_AddCommand ("automap", R_Automap_f);
	Cmd_AddCommand ("mscaleup", R_AutomapScaleUp_f);
	Cmd_AddCommand ("mscaledown", R_AutomapScaleDown_f);

	Cvar_RegisterVariable (&gl_blend_src);
	Cvar_RegisterVariable (&gl_blend_dst);
	Cvar_RegisterVariable (&r_farclip);

	Cvar_RegisterVariable (&gl_lockpvs);
	Cvar_RegisterVariable (&r_norefresh);
	Cvar_RegisterVariable (&r_lightmap);
	Cvar_RegisterVariable (&r_drawentities);
	Cvar_RegisterVariable (&r_drawviewmodel);
	Cvar_RegisterVariable (&r_shadows);
	Cvar_RegisterVariable (&r_wateralpha);
	Cvar_RegisterVariable (&r_dynamic);
	Cvar_RegisterVariable (&r_novis);
	Cvar_RegisterVariable (&r_speeds);

	Cvar_RegisterVariable (&gl_finish);

	Cvar_RegisterVariable (&gl_fogenable);

	Cvar_RegisterVariable (&gl_fogstart);
	Cvar_RegisterVariable (&gl_fogend);
	Cvar_RegisterVariable (&gl_fogdensity);
	Cvar_RegisterVariable (&gl_fogr);
	Cvar_RegisterVariable (&gl_fogg);
	Cvar_RegisterVariable (&gl_fogb);

	Cvar_RegisterVariable (&gl_foguser);

	Cvar_RegisterVariable (&gl_clear);

	Cvar_RegisterVariable (&gl_cull);
	Cvar_RegisterVariable (&gl_smoothmodels);
	Cvar_RegisterVariable (&gl_affinemodels);
	Cvar_RegisterVariable (&gl_polyblend);
	Cvar_RegisterVariable (&gl_playermip);
	Cvar_RegisterVariable (&gl_nocolors);

	Cvar_RegisterVariable (&gl_keeptjunctions);
	Cvar_RegisterVariable (&gl_reporttjunctions);

	Cvar_RegisterVariable (&gl_doubleeyes);

	R_InitParticles ();
	R_InitDecals ();
	R_InitParticleTexture ();
	GL_InitLightmapTextures ();

	playertextures = texture_extension_number;
	texture_extension_number += 16;

	mapshottexture = texture_extension_number;
	texture_extension_number += 21;

	for (i = 0; i < 21; i++)
		currentmapshottexture[i] = -1;

	Cvar_RegisterVariable (&r_skyspeed);
	GL_RSurfInit ();
}

/*
===============
R_TranslatePlayerSkin

Translates a skin texture by the per-player color lookup
===============
*/
void R_TranslatePlayerSkin (int playernum)
{
	int		top, bottom;
	byte	translate[256];
	unsigned	translate32[256];
	int		i, j, s;
	model_t	*model;
	struct aliashdr_s *paliashdr;
	byte	*original;
	unsigned	pixels[512*256], *out;
	unsigned	scaled_width, scaled_height;
	int			inwidth, inheight;
	byte		*inrow;
	unsigned	frac, fracstep;
	extern	byte		**player_8bit_texels_tbl;
	byte	translated[320*200];

	top = cl.scores[playernum].colors & 0xf0;
	bottom = (cl.scores[playernum].colors &15)<<4;

	for (i=0 ; i<256 ; i++)
		translate[i] = i;

	for (i=0 ; i<16 ; i++)
	{
		if (top < 128)	// the artists made some backwards ranges.  sigh.
			translate[TOP_RANGE+i] = top+i;
		else
			translate[TOP_RANGE+i] = top+15-i;
				
		if (bottom < 128)
			translate[BOTTOM_RANGE+i] = bottom+i;
		else
			translate[BOTTOM_RANGE+i] = bottom+15-i;
	}

	//
	// locate the original skin pixels
	//
	currententity = &cl_entities[1+playernum];
	model = currententity->model;
	if (!model)
		return;		// player doesn't have a model yet
	if (model->type != mod_alias)
		return; // only translate skins on alias models

	paliashdr = (struct aliashdr_s *)Mod_Extradata (model);
	s = paliashdr->skinwidth * paliashdr->skinheight;

	if (currententity->skinnum < 0 || currententity->skinnum >= paliashdr->numskins)
	{
		Con_Printf("(%d): Invalid player skin #%d\n", playernum, currententity->skinnum);
		original = (byte *)paliashdr + paliashdr->texels[0];
	}
	else
		original = (byte *)paliashdr + paliashdr->texels[currententity->skinnum];

	if (s & 3)
		Sys_Error ("R_TranslateSkin: s&3");

	inwidth = paliashdr->skinwidth;
	inheight = paliashdr->skinheight;

	// because this happens during gameplay, do it fast
	// instead of sending it through gl_upload 8
    glBindTexture (GL_TEXTURE_2D, playertextures + playernum);

	for (i=0 ; i<s ; i+=4)
	{
		translated[i] = translate[original[i]];
		translated[i+1] = translate[original[i+1]];
		translated[i+2] = translate[original[i+2]];
		translated[i+3] = translate[original[i+3]];
	}

	// don't mipmap these, because it takes too long
	GL_Upload8 (translated, paliashdr->skinwidth, paliashdr->skinheight, true, false);
}


extern qboolean WorldModelLoaded;


extern int worldtype;
extern qboolean SbarPreload;

void MDLDump (void)
{
	int x;
	model_t *mod;
	FILE *f;

	f = fopen ("mdldump.txt", "w");

	for (x = 1; x < MAX_MODELS; x++)
	{
		mod = cl.model_precache[x];

		if (!mod) break;

		if (mod->name[0] != '*') continue;

		fprintf (f, "%s [%f] [%f] [%f]\n", mod->name, mod->origin[0], mod->origin[1], mod->origin[2]);
	}

	fclose (f);
}


/*
====================
R_IdentifyUnderwater

reliable determination of an * model's underwater status
it's a dirty hack but - hey! - it works.
====================
*/
void R_IdentifyUnderwater (void)
{
	int x;
	model_t *mod;
	msurface_t *psurf;
	vec3_t org;
	float *v;
	glpoly_t *p;
	int numVerts;
	int i, j;

	for (x = 1; x < MAX_MODELS; x++)
	{
		mod = cl.model_precache[x];

		if (!mod) break;
		mod->modelleaf = NULL;

		if (mod->name[0] != '*') continue;

		psurf = &mod->surfaces[mod->firstmodelsurface];
		numVerts = org[0] = org[1] = org[2] = 0;

		// get the midpoint of the bmodel based on the verts in each surf
		for (i = 0; i < mod->nummodelsurfaces; i++, psurf++)
		{
			// some bmodels have these
			if (psurf->flags & (SURF_DRAWSKY | SURF_DRAWTURB)) continue;

			p = psurf->polys;
			v = p->verts;

			for (j = 0; j < p->numverts; j++, v += VERTEXSIZE)
			{
				org[0] += v[0];
				org[1] += v[1];
				org[2] += v[2];

				numVerts++;
			}
		}

		if (!numVerts) continue;

		org[0] /= (float) numVerts;
		org[1] /= (float) numVerts;
		org[2] /= (float) numVerts;

		mod->modelleaf = Mod_PointInLeaf (org, cl.worldmodel);
		mod->origin[0] = org[0];
		mod->origin[1] = org[1];
		mod->origin[2] = org[2];

		// now that we have the info we need, mark the surfs in the model
		psurf = &mod->surfaces[mod->firstmodelsurface];

		for (i = 0; i < mod->nummodelsurfaces; i++, psurf++)
		{
			psurf->caustic = mod->modelleaf->caustic;

			if (mod->modelleaf->contents == CONTENTS_WATER ||
				mod->modelleaf->contents == CONTENTS_SLIME ||
				mod->modelleaf->contents == CONTENTS_LAVA)
			{
				psurf->flags |= SURF_UNDERWATER;
			}
		}
	}
}


// bout 23K... not bad...
// there's never gonna be more than 128 teleports in a map...
#define MAX_MH_PORTALS 128

MHTeleports_t PortalData[MAX_MH_PORTALS];


void Mod_SaveTeleportData (char *entdata)
{
	int i;
	byte targName[1024];
	byte modelName[1024];
	byte key[1024];
	byte value[1024];
	int count;

	memset (targName, 0, sizeof (targName));
	memset (modelName, 0, sizeof (modelName));

	for (i = 0;;)
	{
		memset (key, 0, sizeof (key));
		memset (value, 0, sizeof (value));

		// read to the character after the first "
		for (;; i++)
			if (entdata[i - 1] == '\"') break;

		// read in the key
		for (count = 0;; i++, count++)
		{
			key[count] = entdata[i];
			if (entdata[i + 1] == '\"') break;
		}

		// skip to after the quote
		i += 3;

		// skip to the start of the value
		for (;; i++)
			if (entdata[i - 1] == '\"') break;

		// read in the value
		for (count = 0;; i++, count++)
		{
			value[count] = entdata[i];
			if (entdata[i + 1] == '\"') break;
		}

		// and advance
		i += 3;

		// process the data
		if (!strcmp (key, "target")) strcpy (targName, value);
		if (!strcmp (key, "model")) strcpy (modelName, value);

		// see have we got it all
		if (modelName[0] != '\0' && targName[0] != '\0')
		{
			// save it and get out
			for (count = 0; count < MAX_MH_PORTALS; count++)
			{
				if (!PortalData[count].active)
				{
					// save it
					PortalData[count].active = true;
					strcpy (PortalData[count].targ, targName);
					strcpy (PortalData[count].model, modelName);
					break;
				}
			}

			return;
		}

		// emergency termination condition
		if (entdata[i + 5] == '\0') break;
	}
}


void Mod_SaveDestinationData (char *entdata)
{
	int i;
	byte targName[1024];
	byte angle[1024];
	byte origin[1024];
	byte key[1024];
	byte value[1024];
	int count;
	int		j;
	char	string[128];
	char	*v, *w;

	memset (targName, 0, sizeof (targName));
	memset (angle, 0, sizeof (targName));
	memset (origin, 0, sizeof (targName));

	for (i = 0;;)
	{
		memset (key, 0, sizeof (key));
		memset (value, 0, sizeof (value));

		// read to the character after the first "
		for (;; i++)
			if (entdata[i - 1] == '\"') break;

		// read in the key
		for (count = 0;; i++, count++)
		{
			key[count] = entdata[i];
			if (entdata[i + 1] == '\"') break;
		}

		// skip to after the quote
		i += 3;

		// skip to the start of the value
		for (;; i++)
			if (entdata[i - 1] == '\"') break;

		// read in the value
		for (count = 0;; i++, count++)
		{
			value[count] = entdata[i];
			if (entdata[i + 1] == '\"') break;
		}

		// and advance
		i += 3;

		// process the data
		if (!strcmp (key, "targetname")) strcpy (targName, value);
		if (!strcmp (key, "angle")) strcpy (angle, value);
		if (!strcmp (key, "origin")) strcpy (origin, value);

		// see have we got it all - angle is not obligatory, it defaults to 0 if not present
		if (origin[0] != '\0' && targName[0] != '\0')
		{
			// save it and get out
			for (count = 0; count < MAX_MH_PORTALS; count++)
			{
				if (!PortalData[count].active)
					break;

				if (!strcmp (targName, PortalData[count].targ))
				{
					// save the angle and origin (convert to floats first
					PortalData[count].destangles[0] = 0;
					PortalData[count].destangles[1] = 0;
					PortalData[count].destangles[2] = 0;

					// angle is easy
					if (angle[0] != '\0') PortalData[count].destangles[1] = atof (angle);

					// origin...
					strcpy (string, origin);
					v = string;
					w = string;

					for (j = 0; j < 3; j++)
					{
						while (*v && *v != ' ')
							v++;

						*v = 0;
						PortalData[count].destorigin[j] = atof (w);
						w = v = v + 1;
					}

					break;
				}
			}

			return;
		}

		// emergency termination condition
		if (entdata[i + 5] == '\0') break;
	}
}


void Mod_LinkToStarMods (void)
{
	int x;
	model_t *mod;
	int i, j;

	for (x = 1; x < MAX_MODELS; x++)
	{
		mod = cl.model_precache[x];

		if (!mod) break;
		if (mod->name[0] != '*') continue;

		for (i = 0; i < MAX_MH_PORTALS; i++)
		{
			if (!PortalData[i].active) break;

			if (!strcmp (mod->name, PortalData[i].model))
				mod->portaldata = &PortalData[i];
		}
	}
}


// could be done at load time but the data is only valid for the client so it's done here
void Mod_LinkTeleports (model_t *mod)
{
	int i;
	byte buffer[4096];	// unsafe???
	int count;
	byte *buf;

	if (!mod->entLumpSize) return;

	memset (PortalData, 0, sizeof (PortalData));

	// PASS 1 - identify teleports
	// no-one ever said it was gonna be pretty...
	for (i = 0;;)
	{
		// read to an opening brace
		for (;; i++)
			if (mod->entities[i] == '{') break;

		// init the buffer
		memset (buffer, 0, sizeof (buffer));

		// load into the buffer until we hit a closing brace
		for (count = 0; ; i++, count++)
		{
			buffer[count] = mod->entities[i];
			if (buffer[count] == '}') break;
		}

		// see have we got a trigger teleport in the buffer
		buf = buffer;

		for (;;)
		{
			if (!buf[0]) break;

			if (!strncmp (buf, "trigger_teleport", 16))
			{
				// save out the data - we need the target and the model
				Mod_SaveTeleportData (buffer);

				break;
			}

			buf++;
		}

		// emergency termination condition - if i is getting near to the max size, bomb out
		// 40 is the minimum number of bytes required for a meaningful entry
		if (i > (mod->entLumpSize - 39)) break;
	}

	// PASS 2 - identify teleport targets
	for (i = 0;;)
	{
		// read to an opening brace
		for (;; i++)
			if (mod->entities[i] == '{') break;

		// init the buffer
		memset (buffer, 0, sizeof (buffer));

		// load into the buffer until we hit a closing brace
		for (count = 0; ; i++, count++)
		{
			buffer[count] = mod->entities[i];
			if (buffer[count] == '}') break;
		}

		// see have we got a trigger teleport in the buffer
		buf = buffer;

		for (;;)
		{
			if (!buf[0]) break;

			if (!strncmp (buf, "info_teleport_destination", 25))
			{
				// save out the data - we need the target, the angle and the origin
				Mod_SaveDestinationData (buffer);

				break;
			}

			buf++;
		}

		// emergency termination condition - if i is getting near to the max size, bomb out
		// 40 is the minimum number of bytes required for a meaningful entry
		if (i > (mod->entLumpSize - 39)) break;
	}

	Mod_LinkToStarMods ();
}


/*
===============
R_NewMap
===============
*/
void GL_BuildAllLightmaps (void);
void GL_TexManMapLoadFinish (void);
extern int uploads;
extern int addSurfs;

// automap support
void LoadModelMinMaxXYZ (void);

// standard sky
void R_DrawSphere (void);

// skybox loading
qboolean R_LoadSkyBox (char *sbNameBase);

void R_NewMap (void)
{
	int i;
	int j;
	dlight_t *dl = cl_dlights;
	qboolean oldSBOK = sbOK;
	GLfloat *vbData;

	memset (cl_dlights, 0, sizeof (cl_dlights));

	// set all dlights to die
	for (i = 0; i < MAX_DLIGHTS; i++, dl++)
	{
		dl->die = -1;
	}

	// make sure lightmaps aren't rebuilt the first time they're seen!!!
	for (i = 0; i < 256; i++)
		d_lightstylevalue[i] = 264;

	uploads = 0;
	addSurfs = 0;

	memset (&r_worldentity, 0, sizeof(r_worldentity));
	r_worldentity.model = cl.worldmodel;

	// clear out efrags in case the level hasn't been reloaded
	// FIXME: is this one short?
	// MH - it is...
	for (i = 0; i < cl.worldmodel->numleafs + 1; i++)
		cl.worldmodel->leafs[i].efrags = NULL;

	r_viewleaf = NULL;
	R_ClearParticles ();
	R_ClearDecals ();

	// this has to be set *before* building the lightmaps or the dlightframe for the
	// surfs will be set incorrectly, resulting in rogue dlights...!
	r_framecount = 1;

	// build lightstyles
	R_AnimateLight ();
	GL_BuildAllLightmaps ();

	// identify underwater * models
	R_IdentifyUnderwater ();

	// put all chains in a known initial state
	for (i = 0; i < cl.worldmodel->numtextures; i++)
	{
		if (!cl.worldmodel->textures[i])
			continue;

		cl.worldmodel->textures[i]->fbChain = NULL;
		cl.worldmodel->textures[i]->lqChain = NULL;

		// in theory we *could* use a seperate texture chain for each volume,
		// but volume merging and some astutely placed Con_Printfs indicates
		// that no more than 4 drawing volumes (i.e. volumes after merging) are
		// used in ID1.  I've upped it to 16 to be utterly safe.
		for (j = 0; j < 16; j++)
		{
			cl.worldmodel->textures[i]->texturechain[j] = NULL;
		}

		// and the vertex light chain
		cl.worldmodel->textures[i]->vertlightchain = NULL;
	}

	// set the world fog based on worldtype
	// also, load a generic type skybox if we don't have a specific one for the map
	worldtype = -1;

	// fix for wrong worldtype in dm2
	if (!strcmp (cl.levelname, "Claustrophobopolis"))
	{
		worldtype = 1;
	}
	else
	{
		// hmmm - what happens if we send a NULL in here...
		// FIXED - modded Mod_FindWorldSpawnEntry so it never returns NULL
		worldtype = atoi (Mod_FindWorldSpawnEntry (cl.worldmodel, "worldtype"));
	}

	switch (worldtype)
	{
	case 1:		// runic red
		world_fog[0] = 0.6;
		world_fog[1] = 0.1;
		world_fog[2] = 0.0;

		break;

	case 2:		// base grey
		world_fog[0] = 0.4;
		world_fog[1] = 0.4;
		world_fog[2] = 0.4;

		break;

	case 0:		// medieval mud
	default:	// y' never know
		world_fog[0] = 0.5;
		world_fog[1] = 0.4;
		world_fog[2] = 0.3;

		break;
	}

	// do some texture management housekeeping
	GL_TexManMapLoadFinish ();
	LoadModelMinMaxXYZ ();

	// load up the sky display list if necessary
	R_DrawSphere ();

	WorldModelLoaded = false;
	SbarPreload = false;

	Con_DPrintf ("\n\n%i verts loaded\n", worldverts);
	worldverts = 0;

	// get me a sky box
	sbOK = R_LoadSkyBox (Mod_FindWorldSpawnEntry (cl.worldmodel, "sky"));

	if (!sbOK)
	{
		sbOK = R_LoadSkyBox (Mod_FindWorldSpawnEntry (cl.worldmodel, "_sky"));

		if (!sbOK)
		{
			sbOK = R_LoadSkyBox (Mod_FindWorldSpawnEntry (cl.worldmodel, "skybox"));

			if (!sbOK)
			{
				sbOK = R_LoadSkyBox (Mod_FindWorldSpawnEntry (cl.worldmodel, "_skybox"));
			}
		}
	}

	// if they've loaded a valid skybox on the previous map they may want to keep it here, so
	// let's be nice and non-destructive...
	if (!sbOK) sbOK = oldSBOK;

	// no vertex buffer support
	if (!OpenGL_1_5) return;

	/*
	// set up the world vertex buffer
	if (r_worldvb != -1)
	{
		// if we already have a buffer object, flush it down first
		glDeleteBuffers (1, &r_worldvb);
	}

	// generate a new vertex buffer object for the world
	glGenBuffers (1, &r_worldvb);

	// bind the buffer object so we can use it
	glBindBuffer (GL_ARRAY_BUFFER, r_worldvb);

	// allocate a new array to hold the vertex buffer data
	// this is a 32 bit vertex (4 bits redundant) containing only geometry and texcoords
	vbData = (GLfloat *) malloc (sizeof (GLfloat) * (MAX_MAP_VERTS * 3 * 2));

	// copy only the verts we need from surfarray to vbData
	for (i = 0; i < MAX_MAP_VERTS * 2; i++)
	{
		// geometry
		vbData[i * 3 + 0] = surfarray[i * VERTEXSIZE + 0];
		vbData[i * 3 + 1] = surfarray[i * VERTEXSIZE + 1];
		vbData[i * 3 + 2] = surfarray[i * VERTEXSIZE + 2];
	}

	// load in the data for the world.  because we're going to be using the vertex buffer with static data
	// we will create a static buffer for this.  my surfarray is practically perfect for this job cos it
	// is all one big contiguous array.  about 8 megs of video RAM...
	glBufferData (GL_ARRAY_BUFFER, sizeof (GLfloat) * (MAX_MAP_VERTS * 3 * 2), vbData, GL_STATIC_DRAW);

	// unbind the buffer object so it's not used until we ask for it.  this is done by calling glBindBuffer
	// with a buffer number of 0
	glBindBuffer (GL_ARRAY_BUFFER, 0);

	// we don't need the data after it's been transferred to video RAM so kill it
	free (vbData);
	*/
}


/*
====================
R_TimeRefresh_f

For program optimization
====================
*/
void R_TimeRefresh_f (void)
{
	int			i;
	float		start, stop, time;
	int			startangle;
	vrect_t		vr;

	glDrawBuffer  (GL_FRONT);
	glFinish ();

	start = Sys_FloatTime ();

	for (i=0 ; i<128 ; i++)
	{
		r_refdef.viewangles[1] = i/128.0*360.0;
		R_RenderView ();
	}

	glFinish ();
	stop = Sys_FloatTime ();
	time = stop-start;
	Con_Printf ("%f seconds (%f fps)\n", time, 128/time);

	glDrawBuffer  (GL_BACK);
	GL_EndRendering ();
}

