/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


===========================================================================

MH GONZOID PARTICLE OVERKILL

===========================================================================
*/

#include "quakedef.h"
#include "glext.h"


typedef enum
{
	pt_static,
	pt_grav,
	pt_slowgrav,
	pt_fire,
	pt_explode,
	pt_blob,
	pt_blob2,
	pt_smoke,
	pt_smoketrail,
	pt_blood,
	pt_gunsmoke,
	pt_kill,
} ptype_t;


typedef enum
{
	// P_POINT is default and will be 0 if not set
	P_POINT = 0,

	P_BLOOD1 = 1, 
	P_BLOOD2 = 2, 
	P_BLOOD3 = 3, 
	P_BLOOD4 = 4, 
	P_BLOOD5 = 5, 
	P_BLOOD6 = 6, 
	P_BLOOD7 = 7, 
	P_BLOOD8 = 8, 
	P_BLOOD9 = 9, 
	P_BLOOD10 = 10, 
	P_BLOOD11 = 11, 
	P_BLOOD12 = 12, 
	P_BLOOD13 = 13, 
	P_BLOOD14 = 14, 
	P_BLOOD15 = 15, 
	P_BLOOD16 = 16, 

	P_BLAST1 = 17,
	P_BLAST2 = 18,
	P_BLAST3 = 19,
	P_BLAST4 = 20,

	P_SMOKE = 21,

	P_SPARK = 22,

	P_FLARE1 = 23,
	P_FLARE2 = 24,

	P_WALLHIT1 = 25,
	P_WALLHIT2 = 26,
	P_WALLHIT3 = 27,
	P_WALLHIT4 = 28,

	P_TRACER = 29,

	// P_FIXME is a developer notification
	P_FIXME
} pdraw_t;

#define NUM_PARTICLE_TYPES 32

float texcoord_s[NUM_PARTICLE_TYPES][2];
float texcoord_t[NUM_PARTICLE_TYPES][2];

float pScales[NUM_PARTICLE_TYPES];
int pChains[NUM_PARTICLE_TYPES];

float ParticleR[256];
float ParticleG[256];
float ParticleB[256];

#define CHAIN_POINTS	0
#define CHAIN_SPRITE	1

// alternate blend
#define CHAIN_SPRITE2	2

// gunshot effects
#define CHAIN_GUNSHOT	3

#define NUM_PCHAINS		4

void R_InitParticleStuff (void)
{
	int i;

	// particle colours
	for (i = 0; i < 256; i++)
	{
		ParticleR[i] = (float) ((byte *) &d_8to24table[i])[0] * ONEOVER255;
		ParticleG[i] = (float) ((byte *) &d_8to24table[i])[1] * ONEOVER255;
		ParticleB[i] = (float) ((byte *) &d_8to24table[i])[2] * ONEOVER255;
	}

	// blood
	texcoord_s[P_BLOOD1][0] = 0.0;
	texcoord_s[P_BLOOD1][1] = 0.25;
	texcoord_t[P_BLOOD1][0] = 0.0;
	texcoord_t[P_BLOOD1][1] = 0.25;

	texcoord_s[P_BLOOD2][0] = 0.25;
	texcoord_s[P_BLOOD2][1] = 0.0;
	texcoord_t[P_BLOOD2][0] = 0.0;
	texcoord_t[P_BLOOD2][1] = 0.25;

	texcoord_s[P_BLOOD3][0] = 0.0;
	texcoord_s[P_BLOOD3][1] = 0.25;
	texcoord_t[P_BLOOD3][0] = 0.25;
	texcoord_t[P_BLOOD3][1] = 0.0;

	texcoord_s[P_BLOOD4][0] = 0.25;
	texcoord_s[P_BLOOD4][1] = 0.0;
	texcoord_t[P_BLOOD4][0] = 0.25;
	texcoord_t[P_BLOOD4][1] = 0.0;

	texcoord_s[P_BLOOD5][0] = 0.25;
	texcoord_s[P_BLOOD5][1] = 0.5;
	texcoord_t[P_BLOOD5][0] = 0.0;
	texcoord_t[P_BLOOD5][1] = 0.25;

	texcoord_s[P_BLOOD6][0] = 0.5;
	texcoord_s[P_BLOOD6][1] = 0.25;
	texcoord_t[P_BLOOD6][0] = 0.0;
	texcoord_t[P_BLOOD6][1] = 0.25;

	texcoord_s[P_BLOOD7][0] = 0.25;
	texcoord_s[P_BLOOD7][1] = 0.5;
	texcoord_t[P_BLOOD7][0] = 0.25;
	texcoord_t[P_BLOOD7][1] = 0.0;

	texcoord_s[P_BLOOD8][0] = 0.5;
	texcoord_s[P_BLOOD8][1] = 0.25;
	texcoord_t[P_BLOOD8][0] = 0.25;
	texcoord_t[P_BLOOD8][1] = 0.0;

	texcoord_s[P_BLOOD9][0] = 0.5;
	texcoord_s[P_BLOOD9][1] = 0.75;
	texcoord_t[P_BLOOD9][0] = 0.0;
	texcoord_t[P_BLOOD9][1] = 0.25;

	texcoord_s[P_BLOOD10][0] = 0.75;
	texcoord_s[P_BLOOD10][1] = 0.5;
	texcoord_t[P_BLOOD10][0] = 0.0;
	texcoord_t[P_BLOOD10][1] = 0.25;

	texcoord_s[P_BLOOD11][0] = 0.5;
	texcoord_s[P_BLOOD11][1] = 0.75;
	texcoord_t[P_BLOOD11][0] = 0.25;
	texcoord_t[P_BLOOD11][1] = 0.0;

	texcoord_s[P_BLOOD12][0] = 0.75;
	texcoord_s[P_BLOOD12][1] = 0.5;
	texcoord_t[P_BLOOD12][0] = 0.25;
	texcoord_t[P_BLOOD12][1] = 0.0;

	texcoord_s[P_BLOOD13][0] = 0.75;
	texcoord_s[P_BLOOD13][1] = 1.0;
	texcoord_t[P_BLOOD13][0] = 0.0;
	texcoord_t[P_BLOOD13][1] = 0.25;

	texcoord_s[P_BLOOD14][0] = 1.0;
	texcoord_s[P_BLOOD14][1] = 0.75;
	texcoord_t[P_BLOOD14][0] = 0.0;
	texcoord_t[P_BLOOD14][1] = 0.25;

	texcoord_s[P_BLOOD15][0] = 0.75;
	texcoord_s[P_BLOOD15][1] = 1.0;
	texcoord_t[P_BLOOD15][0] = 0.25;
	texcoord_t[P_BLOOD15][1] = 0.0;

	texcoord_s[P_BLOOD16][0] = 1.0;
	texcoord_s[P_BLOOD16][1] = 0.75;
	texcoord_t[P_BLOOD16][0] = 0.25;
	texcoord_t[P_BLOOD16][1] = 0.0;

	// blast
	texcoord_s[P_BLAST1][0] = 0.0;
	texcoord_s[P_BLAST1][1] = 0.5;
	texcoord_t[P_BLAST1][0] = 0.25;
	texcoord_t[P_BLAST1][1] = 0.75;

	texcoord_s[P_BLAST2][0] = 0.5;
	texcoord_s[P_BLAST2][1] = 0.0;
	texcoord_t[P_BLAST2][0] = 0.25;
	texcoord_t[P_BLAST2][1] = 0.75;

	texcoord_s[P_BLAST3][0] = 0.0;
	texcoord_s[P_BLAST3][1] = 0.5;
	texcoord_t[P_BLAST3][0] = 0.75;
	texcoord_t[P_BLAST3][1] = 0.25;

	texcoord_s[P_BLAST4][0] = 0.5;
	texcoord_s[P_BLAST4][1] = 0.0;
	texcoord_t[P_BLAST4][0] = 0.75;
	texcoord_t[P_BLAST4][1] = 0.25;

	// smoke
	texcoord_s[P_SMOKE][0] = 0.0;
	texcoord_s[P_SMOKE][1] = 0.25;
	texcoord_t[P_SMOKE][0] = 0.75;
	texcoord_t[P_SMOKE][1] = 1.0;

	// spark
	texcoord_s[P_SPARK][0] = 0.5;
	texcoord_s[P_SPARK][1] = 0.75;
	texcoord_t[P_SPARK][0] = 0.5;
	texcoord_t[P_SPARK][1] = 0.75;

	// wall-hit
	texcoord_s[P_WALLHIT1][0] = 0.75;
	texcoord_s[P_WALLHIT1][1] = 1.0;
	texcoord_t[P_WALLHIT1][0] = 0.25;
	texcoord_t[P_WALLHIT1][1] = 0.5;

	texcoord_s[P_WALLHIT2][0] = 1.0;
	texcoord_s[P_WALLHIT2][1] = 0.75;
	texcoord_t[P_WALLHIT2][0] = 0.25;
	texcoord_t[P_WALLHIT2][1] = 0.5;

	texcoord_s[P_WALLHIT3][0] = 0.75;
	texcoord_s[P_WALLHIT3][1] = 1.0;
	texcoord_t[P_WALLHIT3][0] = 0.5;
	texcoord_t[P_WALLHIT3][1] = 0.25;

	texcoord_s[P_WALLHIT4][0] = 1.0;
	texcoord_s[P_WALLHIT4][1] = 0.75;
	texcoord_t[P_WALLHIT4][0] = 0.5;
	texcoord_t[P_WALLHIT4][1] = 0.25;

	// tracer
	texcoord_s[P_TRACER][0] = 0.75;
	texcoord_s[P_TRACER][1] = 1.0;
	texcoord_t[P_TRACER][0] = 0.75;
	texcoord_t[P_TRACER][1] = 1.0;

	// scales
	pScales[P_BLOOD1] = 0.001;
	pScales[P_BLOOD2] = 0.001;
	pScales[P_BLOOD3] = 0.001;
	pScales[P_BLOOD4] = 0.001;
	pScales[P_BLOOD5] = 0.001;
	pScales[P_BLOOD6] = 0.001;
	pScales[P_BLOOD7] = 0.001;
	pScales[P_BLOOD8] = 0.001;
	pScales[P_BLOOD9] = 0.001;
	pScales[P_BLOOD10] = 0.001;
	pScales[P_BLOOD11] = 0.001;
	pScales[P_BLOOD12] = 0.001;
	pScales[P_BLOOD13] = 0.001;
	pScales[P_BLOOD14] = 0.001;
	pScales[P_BLOOD15] = 0.001;
	pScales[P_BLOOD16] = 0.001;

	pScales[P_BLAST1] = 0.002;
	pScales[P_BLAST2] = 0.002;
	pScales[P_BLAST3] = 0.002;
	pScales[P_BLAST4] = 0.002;

	pScales[P_SMOKE] = 0;

	pScales[P_SPARK] = 0.001;

	pScales[P_WALLHIT1] = 0;
	pScales[P_WALLHIT2] = 0;
	pScales[P_WALLHIT3] = 0;
	pScales[P_WALLHIT4] = 0;

	pScales[P_TRACER] = 0.001;

	// chains
	pChains[P_BLOOD1] = CHAIN_SPRITE2;
	pChains[P_BLOOD2] = CHAIN_SPRITE2;
	pChains[P_BLOOD3] = CHAIN_SPRITE2;
	pChains[P_BLOOD4] = CHAIN_SPRITE2;
	pChains[P_BLOOD5] = CHAIN_SPRITE2;
	pChains[P_BLOOD6] = CHAIN_SPRITE2;
	pChains[P_BLOOD7] = CHAIN_SPRITE2;
	pChains[P_BLOOD8] = CHAIN_SPRITE2;
	pChains[P_BLOOD9] = CHAIN_SPRITE2;
	pChains[P_BLOOD10] = CHAIN_SPRITE2;
	pChains[P_BLOOD11] = CHAIN_SPRITE2;
	pChains[P_BLOOD12] = CHAIN_SPRITE2;
	pChains[P_BLOOD13] = CHAIN_SPRITE2;
	pChains[P_BLOOD14] = CHAIN_SPRITE2;
	pChains[P_BLOOD15] = CHAIN_SPRITE2;
	pChains[P_BLOOD16] = CHAIN_SPRITE2;

	pChains[P_BLAST1] = CHAIN_SPRITE;
	pChains[P_BLAST2] = CHAIN_SPRITE;
	pChains[P_BLAST3] = CHAIN_SPRITE;
	pChains[P_BLAST4] = CHAIN_SPRITE;

	pChains[P_SMOKE] = CHAIN_SPRITE;

	pChains[P_SPARK] = CHAIN_SPRITE;

	pChains[P_WALLHIT1] = CHAIN_SPRITE; //CHAIN_GUNSHOT;
	pChains[P_WALLHIT2] = CHAIN_SPRITE; //CHAIN_GUNSHOT;
	pChains[P_WALLHIT3] = CHAIN_SPRITE; //CHAIN_GUNSHOT;
	pChains[P_WALLHIT4] = CHAIN_SPRITE; //CHAIN_GUNSHOT;

	pChains[P_TRACER] = CHAIN_SPRITE;
}


void R_AddGlowEffect (float red, float green, float blue, float radius, vec3_t origin);

typedef struct particle_s
{
	// driver-usable fields
	vec3_t		org;
	int			colorindex;
	float		color[3];

	// drivers never touch the following fields
	struct particle_s	*next;
	vec3_t		vel;
	float		ramp;
	float		die;

	ptype_t		type;
	pdraw_t		draw;

	float		radius;

	struct particle_s *pchain[NUM_PCHAINS];

	float scale;

	GLclampf alpha;

	int *pramp;

	vec3_t up;
	vec3_t right;
	vec3_t angles;
} particle_t;



extern	cvar_t	sv_gravity;

// default max # of particles at one time
#define MAX_PARTICLES			2048

// no fewer than this no matter what's on the command line
#define ABSOLUTE_MIN_PARTICLES	512


#define NUM_EXPLODE_PARTICLES 32

int ramp1[8] = {111, 109, 107, 105, 103, 101, 99, 97};
int ramp2[8] = {111, 111, 109, 108, 107, 106, 104, 103};
int ramp3[8] = {109, 6, 107, 5, 105, 4};
int ramp4[8] = {8, 7, 6, 5, 4, 3};

particle_t	*particles, *active_particles, *free_particles;

int			r_numparticles;

struct particle_s *pchain[NUM_PCHAINS];

extern int particlefont;

/*
===============
R_InitParticles
===============
*/
void R_InitParticles (void)
{
	int	i;

	i = COM_CheckParm ("-particles");

	if (i)
	{
		r_numparticles = (int) (Q_atoi (com_argv[i + 1]));

		if (r_numparticles < ABSOLUTE_MIN_PARTICLES) r_numparticles = ABSOLUTE_MIN_PARTICLES;
	}
	else
	{
		r_numparticles = MAX_PARTICLES;
	}

	particles = (particle_t *) Hunk_AllocName (r_numparticles * sizeof (particle_t), "particles");

	R_InitParticleStuff ();
}


/*
===============
R_EntityParticles
===============
*/

#define NUMVERTEXNORMALS	162

extern	float	r_avertexnormals[NUMVERTEXNORMALS][3];
vec3_t	avelocities[NUMVERTEXNORMALS];
float	beamlength = 16;
vec3_t	avelocity = {23, 7, 3};
float	partstep = 0.01;
float	timescale = 0.01;

void R_EntityParticles (entity_t *ent)
{
	int			count;
	int			i;
	particle_t	*p;
	float		angle;
	float		sr, sp, sy, cr, cp, cy;
	vec3_t		forward;
	float		dist;
	
	dist = 64;
	count = 50;

	if (!avelocities[0][0])
		for (i = 0; i < NUMVERTEXNORMALS * 3; i++)
			avelocities[0][i] = (rand () & 255) * 0.01;

	for (i = 0; i < NUMVERTEXNORMALS; i++)
	{
		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		angle = cl.time * avelocities[i][0];

		sy = sin (angle);
		cy = cos (angle);

		angle = cl.time * avelocities[i][1];

		sp = sin (angle);
		cp = cos (angle);

		angle = cl.time * avelocities[i][2];

		sr = sin (angle);
		cr = cos (angle);

		forward[0] = cp * cy;
		forward[1] = cp * sy;
		forward[2] = -sp;

		p->die = cl.time + 0.01;

		p->color[0] = ParticleR[0x6];
		p->color[1] = ParticleG[0x6];
		p->color[2] = ParticleB[0x6];

		p->type = pt_explode;

		p->org[0] = ent->origin[0] + r_avertexnormals[i][0] * dist + forward[0] * beamlength;			
		p->org[1] = ent->origin[1] + r_avertexnormals[i][1] * dist + forward[1] * beamlength;			
		p->org[2] = ent->origin[2] + r_avertexnormals[i][2] * dist + forward[2] * beamlength;			

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_FIXME;
	}
}


/*
===============
R_ClearParticles
===============
*/
void R_ClearParticles (void)
{
	int	i;

	free_particles = &particles[0];
	active_particles = NULL;

	for (i = 0; i < r_numparticles; i++)
		particles[i].next = &particles[i + 1];

	particles[r_numparticles - 1].next = NULL;
}


void R_ReadPointFile_f (void)
{
	FILE	*f;
	vec3_t	org;
	int		r;
	int		c;
	particle_t	*p;
	char	name[MAX_OSPATH];

	sprintf (name,"maps/%s.pts", sv.name);

	COM_FOpenFile (name, &f);

	if (!f)
	{
		Con_Printf ("couldn't open %s\n", name);
		return;
	}
	
	Con_Printf ("Reading %s...\n", name);
	c = 0;

	while (1)
	{
		r = fscanf (f,"%f %f %f\n", &org[0], &org[1], &org[2]);

		if (r != 3) break;

		c++;

		if (!free_particles)
		{
			Con_Printf ("Not enough free particles\n");
			break;
		}

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = 99999;

		p->color[0] = ParticleR[(-c) & 15];
		p->color[1] = ParticleG[(-c) & 15];
		p->color[2] = ParticleB[(-c) & 15];

		p->type = pt_static;

		VectorCopy (vec3_origin, p->vel);
		VectorCopy (org, p->org);

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		// this should be the only one left using standard points
		p->draw = P_POINT;
	}

	fclose (f);
	Con_Printf ("%i points read\n", c);
}


/*
===============
R_ParticleExplosion

===============
*/
void R_ParticleExplosion (vec3_t org)
{
	int i;
	int j;

	particle_t *p;

	for (i = 0; i < NUM_EXPLODE_PARTICLES; i++)
	{
		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 15;

		p->radius = 64 + (rand () % 96) - 48;
		p->ramp = 1;

		p->org[0] = org[0] + ((rand () % 32) - 16);
		p->org[1] = org[1] + ((rand () % 32) - 16);
		p->org[2] = org[2] + ((rand () % 32) - 16);

		// explosion velocities don't get modified
		p->vel[0] = 0;
		p->vel[1] = 0;
		p->vel[2] = 0;

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_BLAST1 + rand () % 4;
		p->type = pt_explode;
		p->alpha = 1.0;
		p->pramp = ramp1;

		p->color[0] = ParticleR[p->pramp[(int) p->ramp]];
		p->color[1] = ParticleG[p->pramp[(int) p->ramp]];
		p->color[2] = ParticleB[p->pramp[(int) p->ramp]];
	}
}


/*
===============
R_ParticleExplosion2

keep with points for the colormapping
===============
*/
void R_ParticleExplosion2 (vec3_t org, int colorStart, int colorLength)
{
	int			i, j;
	particle_t	*p;
	int			colorMod = 0;

	for (i = 0; i < 512; i++)
	{
		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 0.3;

		p->color[0] = ParticleR[colorStart + (colorMod % colorLength)];
		p->color[1] = ParticleG[colorStart + (colorMod % colorLength)];
		p->color[2] = ParticleB[colorStart + (colorMod % colorLength)];

		colorMod++;

		p->type = pt_blob;

		p->org[0] = org[0] + ((rand () % 32) - 16);
		p->org[1] = org[1] + ((rand () % 32) - 16);
		p->org[2] = org[2] + ((rand () % 32) - 16);

		p->vel[0] = (rand () % 512) - 256;
		p->vel[1] = (rand () % 512) - 256;
		p->vel[2] = (rand () % 512) - 256;

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_FIXME;
	}
}

/*
===============
R_BlobExplosion

===============
*/
void R_BlobExplosion (vec3_t org)
{
	int i;
	int j;
	particle_t	*p;

	for (i = 0; i < 1024; i++)
	{
		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 1 + (rand () & 8) * 0.05;

		if (i & 1)
		{
			p->type = pt_blob;

			p->colorindex = (int) (66 + rand () % 6);
			p->color[0] = ParticleR[p->colorindex];
			p->color[1] = ParticleG[p->colorindex];
			p->color[2] = ParticleB[p->colorindex];
		}
		else
		{
			p->type = pt_blob2;

			p->colorindex = (int) (150 + rand () % 6);
			p->color[0] = ParticleR[p->colorindex];
			p->color[1] = ParticleG[p->colorindex];
			p->color[2] = ParticleB[p->colorindex];
		}

		p->org[0] = org[0] + ((rand () % 32) - 16);
		p->org[1] = org[1] + ((rand () % 32) - 16);
		p->org[2] = org[2] + ((rand () % 32) - 16);

		p->vel[0] = (rand () % 512) - 256;
		p->vel[1] = (rand () % 512) - 256;
		p->vel[2] = (rand () % 512) - 256;

		p->draw = P_FIXME;

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);
	}
}


void R_SmokeEffect (vec3_t org)
{
	particle_t *p;

	// add some smoke
	if (!free_particles) return;

	p = free_particles;
	free_particles = p->next;
	p->next = active_particles;
	active_particles = p;

	// set the die ludicrously high so that the smoke will always go through the full colour ramp
	p->die = cl.time + 666;

	p->type = pt_gunsmoke;
	p->ramp = 0;
	p->pramp = ramp4;

	p->color[0] = ParticleR[p->pramp[(int) p->ramp]];
	p->color[1] = ParticleG[p->pramp[(int) p->ramp]];
	p->color[2] = ParticleB[p->pramp[(int) p->ramp]];

	p->org[0] = org[0];
	p->org[1] = org[1];
	p->org[2] = org[2];

	p->vel[0] = 0;
	p->vel[1] = 0;
	p->vel[2] = 0;

	VectorCopy (vup, p->up);
	VectorCopy (vright, p->right);

	p->draw = P_SMOKE;
	p->radius = 4;
	p->alpha = 1.0;
}


void R_SmokeEffect2 (vec3_t org)
{
	particle_t *p;

	// add some smoke
	if (!free_particles) return;

	p = free_particles;
	free_particles = p->next;
	p->next = active_particles;
	active_particles = p;

	// set the die so low that the particle only lives 1 frame
	p->die = cl.time + 666;

	p->type = pt_smoke;
	p->ramp = 0;
	p->pramp = ramp1;

	p->color[0] = ParticleR[p->pramp[(int) p->ramp]];
	p->color[1] = ParticleG[p->pramp[(int) p->ramp]];
	p->color[2] = ParticleB[p->pramp[(int) p->ramp]];

	p->org[0] = org[0] + (rand () % 64) - 32;
	p->org[1] = org[1] + (rand () % 64) - 32;
	p->org[2] = org[2] + (rand () % 64);

	p->vel[0] = 0;
	p->vel[1] = 0;
	p->vel[2] = 50;

	VectorCopy (vup, p->up);
	VectorCopy (vright, p->right);

	p->draw = P_SMOKE;
	p->radius = 64 + (rand () % 64) - 32;
	p->alpha = 0.2;
}


void R_RunGunshotEffect2 (vec3_t org, vec3_t angles, int count, int colour)
{
	int i;
	int j;
	particle_t *p;

	for (i = 0; i < count; i++)
	{
		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 0.1 * (rand () % 5);

		p->colorindex = colour;
		p->color[0] = ParticleR[p->colorindex];
		p->color[1] = ParticleG[p->colorindex];
		p->color[2] = ParticleB[p->colorindex];

		p->type = pt_static;

		p->org[0] = org[0];
		p->org[1] = org[1];
		p->org[2] = org[2];

		p->vel[0] = 0;
		p->vel[1] = 0;
		p->vel[2] = 0;

		p->radius = 10 + (rand () % 6);

		p->angles[0] = angles[0];
		p->angles[1] = angles[1];

		p->draw = P_WALLHIT1 + (rand () % 4);
	}
}


void R_RunGunshotEffect (vec3_t org, int count, int colour)
{
	int i;
	int j;
	particle_t *p;

	for (i = 0; i < count; i++)
	{
		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 0.1 * (rand () % 5);

		p->colorindex = colour;
		p->color[0] = ParticleR[p->colorindex];
		p->color[1] = ParticleG[p->colorindex];
		p->color[2] = ParticleB[p->colorindex];

		p->type = pt_static;

		p->org[0] = org[0] + ((rand() & 3) - 2);
		p->org[1] = org[1] + ((rand() & 3) - 2);
		p->org[2] = org[2] + ((rand() & 3) - 2);

		p->vel[0] = 0;
		p->vel[1] = 0;
		p->vel[2] = 0;

		p->radius = 10;

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_WALLHIT1 + (rand () % 4);
		p->alpha = 3.0;
	}
}


/*
===============
R_RunParticleEffect

Only particles coming from the server go through here.
===============
*/
void R_RunParticleEffect (vec3_t org, vec3_t dir, int color, int count)
{
	int			i, j;
	particle_t	*p;

	//Con_Printf ("%i Particles with color %i\n", count, color);

	for (i = 0; i < count; i++)
	{
		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 0.1 * (rand () % 5);

		p->colorindex = (int) ((color & ~7) + (rand() & 7));
		p->color[0] = ParticleR[p->colorindex];
		p->color[1] = ParticleG[p->colorindex];
		p->color[2] = ParticleB[p->colorindex];

		p->type = pt_slowgrav;

		p->org[0] = org[0] + ((rand () & 15) - 8);
		p->org[1] = org[1] + ((rand () & 15) - 8);
		p->org[2] = org[2] + ((rand () & 15) - 8);

		p->vel[0] = dir[0] * 15;
		p->vel[1] = dir[1] * 15;
		p->vel[2] = dir[2] * 15;

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_FIXME;
	}
}


void R_BloodParticleSplash (vec3_t org, vec3_t dir, int color, int count)
{
	int			i, j;
	particle_t	*p;

	for (i = 0; i < count; i++)
	{
		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 0.1 * (rand () % 5);

		p->colorindex = (int) ((color & ~7) + (rand() & 7));

		p->color[0] = ParticleR[p->colorindex];
		p->color[1] = ParticleG[p->colorindex];
		p->color[2] = ParticleB[p->colorindex];

		p->type = pt_blood;

		p->org[0] = org[0] + ((rand () & 15) - 8);
		p->org[1] = org[1] + ((rand () & 15) - 8);
		p->org[2] = org[2] + ((rand () & 15) - 8);

		p->vel[0] = dir[0] * 15;
		p->vel[1] = dir[1] * 15;
		p->vel[2] = dir[2] * 15;

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_BLOOD1 + (rand () % 16);
		p->radius = 5 + rand () % 5;
		p->alpha = 1.0;
	}
}


void R_HipnoticParticleField (vec3_t org, vec3_t dir, int color, int count)
{
	int			i, j;
	particle_t	*p;

	for (i = 0; i < count * 5; i++)
	{
		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 0.3 * (rand () % 5);

		p->colorindex = (int) ((color & ~7) + (rand() & 7));
		p->color[0] = ParticleR[p->colorindex];
		p->color[1] = ParticleG[p->colorindex];
		p->color[2] = ParticleB[p->colorindex];

		p->type = pt_slowgrav;

		p->org[0] = org[0] + ((rand () & 15) - 8);
		p->org[1] = org[1] + ((rand () & 15) - 8);
		p->org[2] = org[2] + ((rand () & 15) - 8);

		p->vel[0] = dir[0] * 15;
		p->vel[1] = dir[1] * 15;
		p->vel[2] = dir[2] * 15;

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_SPARK;
		p->radius = 1;
		p->alpha = 1.0;
	}
}


/*
===============
R_ParseParticleEffect

Parse an effect out of the server message
===============
*/
void R_ParseParticleEffect (void)
{
	vec3_t org, dir;
	int i, count, color;

	for (i = 0; i < 3; i++)
		org[i] = MSG_ReadCoord ();

	for (i = 0; i < 3; i++)
		dir[i] = MSG_ReadChar () * (1.0 / 16);

	count = MSG_ReadByte ();
	color = MSG_ReadByte ();

	if (hipnotic && count <= 4)
	{
		// particle field
		R_HipnoticParticleField (org, dir, color, count);
	}
	else if (count == 255)
	{
		// always an explosion
		R_ParticleExplosion (org);
	}
	else if (color == 73 || color == 225)
	{
		// blood splashes
		R_BloodParticleSplash (org, dir, color, count);
	}
	else
	{
		// generic particle effect type thing (should be very few of these...)
		R_RunParticleEffect (org, dir, color, count);
	}
}
	
/*
===============
R_LavaSplash

===============
*/
void R_LavaSplash (vec3_t org)
{
	int			i, j, k;
	particle_t	*p;
	float		vel;
	vec3_t		dir;

	for (i =- 16; i < 16; i++)
	{
		for (j =- 16; j < 16; j++)
		{
			for (k = 0; k < 1; k++)
			{
				if (!free_particles) return;

				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;

				p->die = cl.time + 2 + (rand () & 31) * 0.02;

				p->colorindex = (int) (224 + (rand () & 7));
				p->color[0] = ParticleR[p->colorindex];
				p->color[1] = ParticleG[p->colorindex];
				p->color[2] = ParticleB[p->colorindex];

				p->type = pt_slowgrav;

				dir[0] = j * 8 + (rand () & 7);
				dir[1] = i * 8 + (rand () & 7);
				dir[2] = 256;

				p->org[0] = org[0] + dir[0];
				p->org[1] = org[1] + dir[1];
				p->org[2] = org[2] + (rand () & 63);

				VectorNormalize (dir);						
				vel = 50 + (rand () & 63);
				VectorScale (dir, vel, p->vel);

				VectorCopy (vup, p->up);
				VectorCopy (vright, p->right);

				p->draw = P_FIXME;
			}
		}
	}
}

/*
===============
R_TeleportSplash

===============
*/
void R_TeleportSplash (vec3_t org)
{
	int			i, j, k;
	particle_t	*p;
	float		vel;
	vec3_t		dir;

	for (i =- 16; i < 16; i += 4)
	{
		for (j =- 16; j < 16; j += 4)
		{
			for (k =- 24; k < 32; k += 4)
			{
				if (!free_particles) return;

				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;

				p->die = cl.time + 0.2 + (rand () & 7) * 0.02;

				p->colorindex = (int) (7 + (rand () & 7));
				p->color[0] = ParticleR[p->colorindex];
				p->color[1] = ParticleG[p->colorindex];
				p->color[2] = ParticleB[p->colorindex];

				p->type = pt_static;

				dir[0] = j * 8;
				dir[1] = i * 8;
				dir[2] = k * 8;

				p->org[0] = org[0] + i + (rand () & 3);
				p->org[1] = org[1] + j + (rand () & 3);
				p->org[2] = org[2] + k + (rand () & 3);

				VectorNormalize (dir);						
				vel = 50 + (rand () & 63);
				VectorScale (dir, vel, p->vel);

				VectorCopy (vup, p->up);
				VectorCopy (vright, p->right);

				p->draw = P_SPARK;
				p->radius = 1.5;
				p->alpha = 1.0;
			}
		}
	}
}



void R_ZerstorerRain (vec3_t org)
{
	int i;
	particle_t	*p;

	// don't draw if paused or in the console, or otherwise
	if (key_dest != key_game) return;

	for (i = 0; i < 16; i++)
	{
		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 1;

		p->type = pt_static;

		p->colorindex = (int) (39 + (rand () % 5));
		p->color[0] = ParticleR[p->colorindex];
		p->color[1] = ParticleG[p->colorindex];
		p->color[2] = ParticleB[p->colorindex];

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_FIXME;

		p->org[0] = org[0] + (rand () % 256) - 128;
		p->org[1] = org[1] + (rand () % 256) - 128;
		p->org[2] = org[2] + (rand () % 256) - 128;

		p->vel[0] = 0;
		p->vel[0] = 0;
		p->vel[2] = -128;
	}
}


void R_BubbleParticle (vec3_t org)
{
	int i;
	particle_t	*p;

	// don't draw if paused or in the console, or otherwise
	if (key_dest != key_game) return;

	if (!free_particles) return;
	p = free_particles;
	free_particles = p->next;
	p->next = active_particles;
	active_particles = p;

	// only last one frame
	p->die = cl.time + 0.001;

	p->type = pt_static;

	p->color[0] = ParticleR[246];
	p->color[1] = ParticleG[246];
	p->color[2] = ParticleB[246];

	VectorCopy (vup, p->up);
	VectorCopy (vright, p->right);

	p->draw = P_FIXME;

	p->org[0] = org[0];
	p->org[1] = org[1];
	p->org[2] = org[2];

	p->vel[0] = 0;
	p->vel[1] = 0;
	p->vel[2] = 0;
}



void P_VoreTrail (vec3_t start, vec3_t end, int type)
{
	vec3_t		vec;
	float		len;
	int			j;
	particle_t	*p;
	int			dec;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);

	if (type < 128)
		dec = 2;
	else
	{
		dec = 1;
		type -= 128;
	}

	while (len > 0)
	{
		len -= dec;

		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		VectorCopy (vec3_origin, p->vel);
		p->die = cl.time + 2;

		p->colorindex = (int) (152 + (rand () & 3));
		p->color[0] = ParticleR[p->colorindex];
		p->color[1] = ParticleG[p->colorindex];
		p->color[2] = ParticleB[p->colorindex];

		p->type = pt_static;
		p->die = cl.time + 0.3;

		p->org[0] = start[0] + ((rand () % 15) - 8);
		p->org[1] = start[1] + ((rand () % 15) - 8);
		p->org[2] = start[2] + ((rand () % 15) - 8);

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_TRACER;
		p->radius = 3;
		p->alpha = 1.0;

		VectorAdd (start, vec, start);
	}
}


void P_Tracer5Trail (vec3_t start, vec3_t end, int type)
{
	vec3_t		vec;
	float		len;
	int			j;
	particle_t	*p;
	int			dec;
	static int	tracercount;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);

	if (type < 128)
		dec = 2;
	else
	{
		dec = 1;
		type -= 128;
	}

	while (len > 0)
	{
		len -= dec;

		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		VectorCopy (vec3_origin, p->vel);

		p->die = cl.time + 0.5;
		p->type = pt_static;

		p->color[0] = ParticleR[230 + ((tracercount & 4) << 1)];
		p->color[1] = ParticleG[230 + ((tracercount & 4) << 1)];
		p->color[2] = ParticleB[230 + ((tracercount & 4) << 1)];

		tracercount++;

		VectorCopy (start, p->org);

		if (tracercount & 1)
		{
			p->vel[0] = 30 * vec[1];
			p->vel[1] = 30 *- vec[0];
		}
		else
		{
			p->vel[0] = 30 *- vec[1];
			p->vel[1] = 30 * vec[0];
		}

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_TRACER;
		p->radius = 3;
		p->alpha = 1.0;

		VectorAdd (start, vec, start);
	}
}


void P_Blood4Trail (vec3_t start, vec3_t end, int type)
{
	vec3_t		vec;
	float		len;
	int			j;
	particle_t	*p;
	int			dec;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);

	if (type < 128)
		dec = 2;
	else
	{
		dec = 1;
		type -= 128;
	}

	while (len > 0)
	{
		len -= dec;

		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		VectorCopy (vec3_origin, p->vel);
		p->die = cl.time + 2;

		p->type = pt_blood;

		p->colorindex = (int) (67 + (rand () & 3));

		p->color[0] = ParticleR[p->colorindex];
		p->color[1] = ParticleG[p->colorindex];
		p->color[2] = ParticleB[p->colorindex];

		p->org[0] = start[0] + ((rand () % 6) - 3);
		p->org[1] = start[1] + ((rand () % 6) - 3);
		p->org[2] = start[2] + ((rand () % 6) - 3);

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_BLOOD1 + (rand () % 16);
		p->radius = 5 + rand () % 5;
		p->alpha = 1.0;

		// was 3
		len -= 1;

		VectorAdd (start, vec, start);
	}
}


void P_Tracer3Trail (vec3_t start, vec3_t end, int type)
{
	vec3_t		vec;
	float		len;
	int			j;
	particle_t	*p;
	int			dec;
	static int	tracercount;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);

	if (type < 128)
		dec = 2;
	else
	{
		dec = 1;
		type -= 128;
	}

	while (len > 0)
	{
		len -= dec;

		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		VectorCopy (vec3_origin, p->vel);

		p->die = cl.time + 0.5;
		p->type = pt_static;

		p->color[0] = ParticleR[52 + ((tracercount & 4) << 1)];
		p->color[1] = ParticleG[52 + ((tracercount & 4) << 1)];
		p->color[2] = ParticleB[52 + ((tracercount & 4) << 1)];

		tracercount++;

		VectorCopy (start, p->org);

		if (tracercount & 1)
		{
			p->vel[0] = 30 * vec[1];
			p->vel[1] = 30 *- vec[0];
		}
		else
		{
			p->vel[0] = 30 *- vec[1];
			p->vel[1] = 30 * vec[0];
		}

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_TRACER;
		p->radius = 3;
		p->alpha = 1.0;

		VectorAdd (start, vec, start);
	}
}


void P_Blood2Trail (vec3_t start, vec3_t end, int type)
{
	vec3_t		vec;
	float		len;
	int			j;
	particle_t	*p;
	int			dec;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);

	if (type < 128)
		dec = 2;
	else
	{
		dec = 1;
		type -= 128;
	}

	while (len > 0)
	{
		len -= dec;

		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		VectorCopy (vec3_origin, p->vel);
		p->die = cl.time + 2;

		p->type = pt_blood;

		p->colorindex = (int) (67 + (rand () & 3));

		p->color[0] = ParticleR[p->colorindex];
		p->color[1] = ParticleG[p->colorindex];
		p->color[2] = ParticleB[p->colorindex];

		p->org[0] = start[0] + ((rand () % 6) - 3);
		p->org[1] = start[1] + ((rand () % 6) - 3);
		p->org[2] = start[2] + ((rand () % 6) - 3);

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_BLOOD1 + (rand () % 16);
		p->radius = 5 + rand () % 5;
		p->alpha = 1.0;

		VectorAdd (start, vec, start);
	}
}

extern cvar_t gl_sparky;

void P_SmokeTrail (vec3_t start, vec3_t end, int type)
{
	vec3_t		vec;
	float		len;
	int			j;
	particle_t	*p;
	int			dec;

	int oldlen;
	vec3_t oldstart;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);

	if (type < 128)
		dec = 2;
	else
	{
		dec = 1;
		type -= 128;
	}

	oldlen = len;

	oldstart[0] = start[0];
	oldstart[1] = start[1];
	oldstart[2] = start[2];

	while (len > 0)
	{
		len -= dec;

		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		VectorCopy (vec3_origin, p->vel);
		p->die = cl.time + 0.75;

		p->ramp = (rand () & 3) + 2;
		p->pramp = ramp3;

		p->color[0] = ParticleR[p->pramp[(int) p->ramp]];
		p->color[1] = ParticleG[p->pramp[(int) p->ramp]];
		p->color[2] = ParticleB[p->pramp[(int) p->ramp]];

		p->type = pt_smoketrail;

		p->org[0] = start[0] + ((rand () % 6) - 3);
		p->org[1] = start[1] + ((rand () % 6) - 3);
		p->org[2] = start[2] + ((rand () % 6) - 3);

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_SMOKE;
		p->radius = 2;
		p->alpha = 1.0;

		VectorAdd (start, vec, start);
	}

	if (!gl_sparky.value) return;

	// add a spark cloud around the smoke trails
	while (oldlen > 0)
	{
		oldlen -= 5;

		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 2.0;

		p->ramp = (rand () & 3) + 1;
		p->pramp = ramp3;

		p->color[0] = ParticleR[p->pramp[(int) p->ramp]];
		p->color[1] = ParticleG[p->pramp[(int) p->ramp]];
		p->color[2] = ParticleB[p->pramp[(int) p->ramp]];

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->type = pt_fire;
		p->draw = P_SPARK;
		p->radius = 1;
		p->alpha = 1.0;

		p->org[0] = oldstart[0] + ((rand () % 24) - 12);
		p->org[1] = oldstart[1] + ((rand () % 24) - 12);
		p->org[2] = oldstart[2] + ((rand () % 24) - 12);

		p->vel[0] = 0;
		p->vel[1] = 0;
		p->vel[2] = 0;

		VectorAdd (oldstart, vec, oldstart);
	}
}


void P_RocketTrail (vec3_t start, vec3_t end, int type)
{
	vec3_t		vec;
	float		len;
	int			j;
	particle_t	*p;
	int			dec;

	int oldlen;
	vec3_t oldstart;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);

	if (type < 128)
		dec = 2;
	else
	{
		dec = 1;
		type -= 128;
	}

	oldlen = len;

	oldstart[0] = start[0];
	oldstart[1] = start[1];
	oldstart[2] = start[2];

	while (len > 0)
	{
		len -= dec;

		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		VectorCopy (vec3_origin, p->vel);
		p->die = cl.time + 0.75;

		p->ramp = (rand () & 3);
		p->pramp = ramp3;

		p->color[0] = ParticleR[p->pramp[(int) p->ramp]];
		p->color[1] = ParticleG[p->pramp[(int) p->ramp]];
		p->color[2] = ParticleB[p->pramp[(int) p->ramp]];

		p->type = pt_smoketrail;

		p->org[0] = start[0] + ((rand () % 6) - 3);
		p->org[1] = start[1] + ((rand () % 6) - 3);
		p->org[2] = start[2] + ((rand () % 6) - 3);

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_SMOKE;
		p->radius = 2;
		p->alpha = 1.0;

		VectorAdd (start, vec, start);
	}

	if (!gl_sparky.value) return;

	// add a spark cloud around the smoke trails
	while (oldlen > 0)
	{
		oldlen -= 5;

		if (!free_particles) return;

		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->die = cl.time + 2.0;

		p->ramp = (rand () & 3) + 1;
		p->pramp = ramp3;

		p->color[0] = ParticleR[p->pramp[(int) p->ramp]];
		p->color[1] = ParticleG[p->pramp[(int) p->ramp]];
		p->color[2] = ParticleB[p->pramp[(int) p->ramp]];

		p->type = pt_fire;

		VectorCopy (vup, p->up);
		VectorCopy (vright, p->right);

		p->draw = P_SPARK;
		p->radius = 1;
		p->alpha = 1.0;

		p->org[0] = oldstart[0] + ((rand () % 24) - 12);
		p->org[1] = oldstart[1] + ((rand () % 24) - 12);
		p->org[2] = oldstart[2] + ((rand () % 24) - 12);

		p->vel[0] = 0;
		p->vel[1] = 0;
		p->vel[2] = 0;

		VectorAdd (oldstart, vec, oldstart);
	}
}


void R_RocketTrail (vec3_t start, vec3_t end, int type)
{
	switch (type)
	{
	case 0:
	case 128:
		P_RocketTrail (start, end, type);
		break;

	case 1:
	case 129:
		P_SmokeTrail (start, end, type);
		break;

	case 2:
	case 130:
		P_Blood2Trail (start, end, type);
		break;

	case 3:
	case 131:
		// wizard
		P_Tracer3Trail (start, end, type);
		break;

	case 4:
	case 132:
		P_Blood4Trail (start, end, type);
		break;

	case 5:
	case 133:
		// knight
		P_Tracer5Trail (start, end, type);
		break;

	case 6:
	case 134:
		P_VoreTrail (start, end, type);
		break;

	default:
		break;
	}
}


void R_FlareSprite (float r, float g, float b, float radius, vec3_t origin, int type)
{
	particle_t	*p;

	if (!free_particles) return;

	p = free_particles;
	free_particles = p->next;
	p->next = active_particles;
	active_particles = p;

	// these are automatically destroyed when the particle is updated
	p->die = cl.time + 2.0;

	p->color[0] = r;
	p->color[1] = g;
	p->color[2] = b;

	p->type = pt_kill;

	if (type == 0)
		p->draw = P_FLARE1;
	else
		p->draw = P_FLARE2;

	VectorCopy (vup, p->up);
	VectorCopy (vright, p->right);

	p->radius = radius;
	p->alpha = 1.0;

	p->org[0] = origin[0];
	p->org[1] = origin[1];
	p->org[2] = origin[2];

	p->vel[0] = 0;
	p->vel[1] = 0;
	p->vel[2] = 0;
}


/*
===============
R_FreeParticles

update the active particle list by clearing dead particles from the linked list
===============
*/
void R_FreeParticles (void)
{
	particle_t *p, *kill;

	// clear the chains
	pchain[CHAIN_POINTS] = NULL;
	pchain[CHAIN_SPRITE] = NULL;
	pchain[CHAIN_SPRITE2] = NULL;
	pchain[CHAIN_GUNSHOT] = NULL;

	// jump to the first active particle in the list
	while (1)
	{
		kill = active_particles;

		if (kill && kill->die < cl.time)
		{
			active_particles = kill->next;
			kill->next = free_particles;
			free_particles = kill;

			continue;
		}

		break;
	}

	for (p = active_particles; p; p = p->next)
	{
		// jump to the next active particle in the list
		while (1)
		{
			kill = p->next;

			if (kill && kill->die < cl.time)
			{
				p->next = kill->next;
				kill->next = free_particles;
				free_particles = kill;

				continue;
			}

			break;
		}

		// sort the particles for drawing
		switch (p->draw)
		{
		case P_POINT:
		case P_FIXME:
			p->pchain[CHAIN_POINTS] = pchain[CHAIN_POINTS];
			pchain[CHAIN_POINTS] = p;
			break;

		default:
			// calculate the scale
			p->scale = (p->org[0] - r_origin[0]) * vpn[0] + (p->org[1] - r_origin[1]) * vpn[1] + (p->org[2] - r_origin[2]) * vpn[2];

			// don't draw particles that are too near the viewpoint
			if (p->scale < 10.0)
			{
				// don't kill them cos they might move away later.
				continue;
			}

			// build the chain
			p->pchain[pChains[p->draw]] = pchain[pChains[p->draw]];
			pchain[pChains[p->draw]] = p;

			// get the final scale
			p->scale = 1.0 + p->scale * pScales[p->type];

			break;
		}
	}
}


/*
===============
R_UpdateParticles

update particle velocity and origin
===============
*/
void R_UpdateParticles (void)
{
	int i;

	float grav = frametime * sv_gravity.value * 0.05;
	float grav2 = frametime * sv_gravity.value * 0.00001;
	float time1 = frametime * 5;
	float time2 = frametime * 12.5;
	float time3 = frametime * 15;
	float time4 = frametime * 7.5;
	float dvel = 4 * frametime;
	float alphatime = frametime / 4;

	particle_t *p;

	for (p = active_particles; p; p = p->next)
	{
		// modify the origin by the particle velocity
		if (p->vel[0]) p->org[0] += p->vel[0] * frametime;
		if (p->vel[1]) p->org[1] += p->vel[1] * frametime;
		if (p->vel[2]) p->org[2] += p->vel[2] * frametime;

		switch (p->type)
		{
		case pt_static:
			break;

		case pt_smoketrail:
			// brownian motion
			p->org[0] += (float) ((rand () & 2) - 1) / 2;
			p->org[1] += (float) ((rand () & 2) - 1) / 2;
			p->org[2] += (float) ((rand () & 2) - 1) / 2;
			goto SKIP_GUNSMOKE;

		case pt_gunsmoke:
			// same as time2 * 3
			p->radius += time2;

		case pt_smoke:
SKIP_GUNSMOKE:;
			p->radius += time2 * 2;
			p->alpha -= frametime;
			if (p->alpha < 0) p->die = -1;

		case pt_fire:
			p->ramp += time1;

			if (p->ramp >= 6)
				p->die = -1;
			else
			{
				p->color[0] = ParticleR[p->pramp[(int) p->ramp]];
				p->color[1] = ParticleG[p->pramp[(int) p->ramp]];
				p->color[2] = ParticleB[p->pramp[(int) p->ramp]];
			}

			p->vel[2] += grav;

			break;

		case pt_explode:
			// the explosion effect should last the same time as the explosion dlight
			p->ramp += time2;

			if (p->ramp >= 8)
				p->die = -1;
			else
			{
				p->color[0] = ParticleR[p->pramp[(int) p->ramp]];
				p->color[1] = ParticleG[p->pramp[(int) p->ramp]];
				p->color[2] = ParticleB[p->pramp[(int) p->ramp]];
			}

			break;

		case pt_blob:
			p->vel[0] += p->vel[0] * dvel;
			p->vel[1] += p->vel[1] * dvel;
			p->vel[2] += p->vel[2] * dvel;

			p->vel[2] -= grav;

			break;

		case pt_blob2:
			p->vel[0] += p->vel[0] * dvel;
			p->vel[1] += p->vel[1] * dvel;

			p->vel[2] -= grav;

			break;

		case pt_blood:
			// ooooh yeeahhhh
			p->radius += frametime;

		case pt_grav:
		case pt_slowgrav:
			p->vel[2] -= grav;

			break;

		case pt_kill:
			// these are sprites that get redrawn every frame
			p->die = -1;
			break;
		}
	}
}


void R_DrawParticleSpriteChain (int spriteChain, GLenum GL_BLENDFUNC_SRC_FUNC, GLenum GL_BLENDFUNC_DST_FUNC)
{
	particle_t *p;
	vec3_t 	up, right;
	vec3_t pUp, pRight, pOrg;
	GLfloat *partVerts;
	int numparts;

	partVerts = VArray;
	numparts = 0;

	glBlendFunc (GL_BLENDFUNC_SRC_FUNC, GL_BLENDFUNC_DST_FUNC);

	for (p = pchain[spriteChain]; p; p = p->pchain[spriteChain])
	{
		// calculate the vertex offset factors
		VectorScale (vup, p->radius, up);
		VectorScale (vright, p->radius, right);

		// get the drawing origins - top to bottom
		pUp[0] = up[0] * p->scale;
		pUp[1] = up[1] * p->scale;
		pUp[2] = up[2] * p->scale;

		// get the drawing origins - left to right
		pRight[0] = right[0] * p->scale;
		pRight[1] = right[1] * p->scale;
		pRight[2] = right[2] * p->scale;

		// these particles will be off-center, so correct that
		pOrg[0] = p->org[0] - (pRight[0] / 2) - (pUp[0] / 2);
		pOrg[1] = p->org[1] - (pRight[1] / 2) - (pUp[1] / 2);
		pOrg[2] = p->org[2] - (pRight[2] / 2) - (pUp[2] / 2);

		// draw the particle - colours
		partVerts[5] = partVerts[14] = partVerts[23] = partVerts[32] = p->color[0];
		partVerts[6] = partVerts[15] = partVerts[24] = partVerts[33] = p->color[1];
		partVerts[7] = partVerts[16] = partVerts[25] = partVerts[34] = p->color[2];
		partVerts[8] = partVerts[17] = partVerts[26] = partVerts[35] = p->alpha;

		// draw the particle
		partVerts[0] = pOrg[0];
		partVerts[1] = pOrg[1];
		partVerts[2] = pOrg[2];

		partVerts[3] = texcoord_s[p->draw][0];
		partVerts[4] = texcoord_t[p->draw][1];

		partVerts[27] = pOrg[0] + pRight[0];
		partVerts[28] = pOrg[1] + pRight[1];
		partVerts[29] = pOrg[2] + pRight[2];

		partVerts[30] = texcoord_s[p->draw][1];
		partVerts[31] = texcoord_t[p->draw][1];

		partVerts[18] = pOrg[0] + pUp[0] + pRight[0];
		partVerts[19] = pOrg[1] + pUp[1] + pRight[1];
		partVerts[20] = pOrg[2] + pUp[2] + pRight[2];

		partVerts[21] = texcoord_s[p->draw][1];
		partVerts[22] = texcoord_t[p->draw][0];

		partVerts[9] = pOrg[0] + pUp[0];
		partVerts[10] = pOrg[1] + pUp[1];
		partVerts[11] = pOrg[2] + pUp[2];

		partVerts[12] = texcoord_s[p->draw][0];
		partVerts[13] = texcoord_t[p->draw][0];

		partVerts += 36;
		numparts += 4;
	}

	glDrawArrays (GL_QUADS, 0, numparts);
}


#define MODEL_GUNSHOT	0

// for R_RunGunshotEffect2
// much of this code can be reused for decals!!!
void R_DrawParticleModelChain (int modelChain, int modelnum)
{
	particle_t *p;
	GLfloat *partVerts;
	int i;
	float size1, size2, size3;

	partVerts = VArray;

	glBlendFunc (GL_SRC_ALPHA, GL_DST_ALPHA);

	for (p = pchain[modelChain]; p; p = p->pchain[modelChain])
	{
		// see the theorem of pythagoras for an explanation of where these numbers came from!!!
		size1 = p->radius * 0.2 * p->scale;
		size2 = p->radius * 0.4828427 * p->scale;
		size3 = p->radius * 0.8485281 * p->scale;

		glPushMatrix ();

		glTranslatef (p->org[0],  p->org[1],  p->org[2]);
		glRotatef (p->angles[1],  0, 0, 1);
		glRotatef (p->angles[0],  0, 1, 0);		// was -p->angles[0] - the old "stupid quake bug" thing here i reckon
		//glRotatef (p->angles[2],  1, 0, 0);	// not used

		// ------------------------------------------------------------------------------------------------
		// vertex pointer
		partVerts[0] = 0;
		partVerts[1] = 0;
		partVerts[2] = 0;

		// tex-coord pointer
		partVerts[3] = 0.875;
		partVerts[4] = 0.375;

		// color pointer
		partVerts[5] = p->color[0];
		partVerts[6] = p->color[1];
		partVerts[7] = p->color[2];
		partVerts[8] = 1;

		// ------------------------------------------------------------------------------------------------
		// vertex pointer
		partVerts[9] = size1;
		partVerts[10] = size2;
		partVerts[11] = size3;

		// tex-coord pointer
		partVerts[12] = 0.75;
		partVerts[13] = 0.25;

		// color pointer
		partVerts[14] = p->color[0];
		partVerts[15] = p->color[1];
		partVerts[16] = p->color[2];
		partVerts[17] = 0;

		// ------------------------------------------------------------------------------------------------
		// vertex pointer
		partVerts[18] = size2;
		partVerts[19] = size1;
		partVerts[20] = size3;

		// tex-coord pointer
		partVerts[21] = 0.875;
		partVerts[22] = 0.25;

		// color pointer
		partVerts[23] = p->color[0];
		partVerts[24] = p->color[1];
		partVerts[25] = p->color[2];
		partVerts[26] = 0;

		// ------------------------------------------------------------------------------------------------
		// vertex pointer
		partVerts[27] = size2;
		partVerts[28] = -size1;
		partVerts[29] = size3;

		// tex-coord pointer
		partVerts[30] = 1;
		partVerts[31] = 0.25;

		// color pointer
		partVerts[32] = p->color[0];
		partVerts[33] = p->color[1];
		partVerts[34] = p->color[2];
		partVerts[35] = 0;

		// ------------------------------------------------------------------------------------------------
		// vertex pointer
		partVerts[36] = size1;
		partVerts[37] = -size2;
		partVerts[38] = size3;

		// tex-coord pointer
		partVerts[39] = 1;
		partVerts[40] = 0.375;

		// color pointer
		partVerts[41] = p->color[0];
		partVerts[42] = p->color[1];
		partVerts[43] = p->color[2];
		partVerts[44] = 0;

		// ------------------------------------------------------------------------------------------------
		// vertex pointer
		partVerts[45] = -size1;
		partVerts[46] = -size2;
		partVerts[47] = size3;

		// tex-coord pointer
		partVerts[48] = 1;
		partVerts[49] = 0.5;

		// color pointer
		partVerts[50] = p->color[0];
		partVerts[51] = p->color[1];
		partVerts[52] = p->color[2];
		partVerts[53] = 0;


		// ------------------------------------------------------------------------------------------------
		// vertex pointer
		partVerts[54] = -size2;
		partVerts[55] = -size1;
		partVerts[56] = size3;

		// tex-coord pointer
		partVerts[57] = 0.875;
		partVerts[58] = 0.5;

		// color pointer
		partVerts[59] = p->color[0];
		partVerts[60] = p->color[1];
		partVerts[61] = p->color[2];
		partVerts[62] = 0;

		// ------------------------------------------------------------------------------------------------
		// vertex pointer
		partVerts[63] = -size2;
		partVerts[64] = size1;
		partVerts[65] = size3;

		// tex-coord pointer
		partVerts[66] = 0.75;
		partVerts[67] = 0.5;

		// color pointer
		partVerts[68] = p->color[0];
		partVerts[69] = p->color[1];
		partVerts[70] = p->color[2];
		partVerts[71] = 0;

		// ------------------------------------------------------------------------------------------------
		// vertex pointer
		partVerts[72] = -size1;
		partVerts[73] = size2;
		partVerts[74] = size3;

		// tex-coord pointer
		partVerts[75] = 0.75;
		partVerts[76] = 0.375;

		// color pointer
		partVerts[77] = p->color[0];
		partVerts[78] = p->color[1];
		partVerts[79] = p->color[2];
		partVerts[80] = 0;

		// ------------------------------------------------------------------------------------------------
		// vertex pointer
		partVerts[81] = size1;
		partVerts[82] = size2;
		partVerts[83] = size3;

		// tex-coord pointer
		partVerts[84] = 0.75;
		partVerts[85] = 0.25;

		// color pointer
		partVerts[86] = p->color[0];
		partVerts[87] = p->color[1];
		partVerts[88] = p->color[2];
		partVerts[89] = 0;

		// 10 verts
		glDrawArrays (GL_TRIANGLE_FAN, 0, 10);

		glPopMatrix ();
	}

	// rampant paranoia
	glColor3f (1, 1, 1);
}


/*
===============
R_DrawParticles
===============
*/
void R_DrawParticles (void)
{
	particle_t *p;

	R_FreeParticles ();

	if (!active_particles) return;

	glDepthMask (GL_FALSE);
	glEnable (GL_BLEND);

	if (pchain[CHAIN_POINTS])
	{
		// point parameters are set in GL_Init ()
		glEnable (GL_POINT_SMOOTH);
		glDisable (GL_TEXTURE_2D);

		glBegin (GL_POINTS);

		for (p = pchain[CHAIN_POINTS]; p; p = p->pchain[CHAIN_POINTS])
		{
			// draw the particle
			glColor3f (p->color[0], p->color[1], p->color[2]);
			glVertex3fv (p->org);
		}

		glEnd ();

		glColor4f (1.0f, 1.0f, 1.0f, 1.0f);
		glEnable (GL_TEXTURE_2D);
		glDisable (GL_POINT_SMOOTH);
	}

	if (pchain[CHAIN_SPRITE] || pchain[CHAIN_SPRITE2] || pchain[CHAIN_GUNSHOT])
	{
		glDisable (GL_CULL_FACE);

		glActiveTexture (GL_TEXTURE0);
		glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
		glBindTexture (GL_TEXTURE_2D, particlefont);

		// draw these using vertex arrays cos it's mucho faster amigo
		glPushClientAttrib (GL_CLIENT_VERTEX_ARRAY_BIT);

		glEnableClientState (GL_VERTEX_ARRAY);
		glClientActiveTexture (GL_TEXTURE0);
		glEnableClientState (GL_TEXTURE_COORD_ARRAY);
		glEnableClientState (GL_COLOR_ARRAY);

		glVertexPointer (3, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[0]);
		glTexCoordPointer (2, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[3]);
		glColorPointer (4, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[5]);

		// draw 2 before 1
		if (pchain[CHAIN_SPRITE2]) R_DrawParticleSpriteChain (CHAIN_SPRITE2, GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		if (pchain[CHAIN_SPRITE]) R_DrawParticleSpriteChain (CHAIN_SPRITE, GL_SRC_ALPHA, GL_DST_ALPHA);
		if (pchain[CHAIN_GUNSHOT]) R_DrawParticleModelChain (CHAIN_GUNSHOT, MODEL_GUNSHOT);

		glDisableClientState (GL_VERTEX_ARRAY);
		glDisableClientState (GL_TEXTURE_COORD_ARRAY);
		glDisableClientState (GL_COLOR_ARRAY);

		glPopClientAttrib ();

		glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

		// restore the blend func
		glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		glEnable (GL_CULL_FACE);
	}

	glDisable (GL_BLEND);
	glDepthMask (GL_TRUE);

	R_UpdateParticles ();
}

