/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// r_surf.c: surface-related refresh code

#include "quakedef.h"
#include "glext.h"


qboolean skythisframe;
qboolean waterthisframe;


extern int lightmap_textures;
extern int stainmap_textures;

int r_detailtexture = -1;
int r_worldvb = -1;
msurface_t *r_detailsurfaces;

int		addSurfs;

// we need a "fake" volume for brush models
volume_t bmvol[2];

qboolean fbChain = false;

void R_PrebuildLightmaps (void);

void R_PushDlights (mnode_t *node, qboolean bmnode);
void R_PrebuildSurfaceLightmap (msurface_t *surf);


// increase it to provide some headroom for warp surface subdivision
float surfarray[MAX_MAP_VERTS * VERTEXSIZE * 2];

extern cvar_t gl_detailtexture;

/*
===============
Sin approximations.  These work in a similar fashion to the old gl_warpsin.h but are
calculated at engine startup rather than stored in a fixed array.  This lets me keep
a higher precision copy of the table in memory without having to use a great big
include file containing loadsa data, plus I can also tweak the precision.  These are
faster than the library sin function and are significantly faster than JoeQuake's
SINTABLE_APPROX function (which was actually *slower* than the library sin function
when I had it in here...)

SINTABLE_SIZE is currently 1024, which gives a good degree of accuracy without the need
for a massive table lookup.  You can increase it by changing the define in glquake.h,
but be aware that beyond a certain level rounding errors start to come in and the
visual result gets incredibly choppy.  Decreasing it to 256 gives comparable results to
the old gl_warpsin.h method.
===============
*/
float sintable[SINTABLE_SIZE];

int STBitMask;

float STMult;


extern int caustictexture;


/*
===============
R_TextureAnimation

Returns the proper texture for a given time and base texture
===============
*/
static texture_t *R_TextureAnimation (texture_t *base)
{
	int		relative;
	int		count;

	// save out the base texture in case we have a bad cycle
	texture_t *bad = base;

	if (currententity->frame)
	{
		if (base->alternate_anims)
			base = base->alternate_anims;
	}

	// no animations
	if (!base->anim_total) return base;

	relative = (int) (realtime * 10) % base->anim_total;

	count = 0;

	while (base->anim_min > relative || base->anim_max <= relative)
	{
		base = base->anim_next;

		// bad animation cycle so just return the base texture
		if (!base) return bad;
		if (++count > 100) return bad;
	}

	return base;
}


/*
========================================================================================================

						UTILITY/SUPPORT FUNCTIONS

========================================================================================================
*/
__inline void R_ScaleTextureMatrix (float scale)
{
	glMatrixMode (GL_TEXTURE);
	glLoadIdentity ();
	if (scale) glScalef (scale, scale, 1);
	glMatrixMode (GL_MODELVIEW);
}


/*
===================
R_DrawFullbrightChain

Draws the chain of fullbrights sorted by texture
This should be the very last thing drawn as we don't want fullbrights to be
overdrawn by anything else.  They are drawn as a seperate pass cos of the different BlendFunc
===================
*/
static void R_DrawFullbrightChain (model_t *mod)
{
	int i;
	int j;
	float *v;
	msurface_t *surf;
	texture_t *t;

	// MH - my previous alpha test method didn't work with the quake hi-res luma textures
	// so i changed it.  bleagh.
	glDepthMask (GL_FALSE);
	glEnable (GL_BLEND);
	glBlendFunc (GL_ONE, GL_ONE);

	for (i = 0; i < mod->numtextures; i++)
	{
		if (!mod->textures[i]) continue;
		if (!mod->textures[i]->fbChain) continue;

		t = R_TextureAnimation (mod->textures[i]);

		surf = mod->textures[i]->fbChain;

		glBindTexture (GL_TEXTURE_2D, t->fullbright->texnum);

		for (; surf; surf = surf->fbChain)
			glDrawArrays (GL_TRIANGLE_FAN, surf->polys->saindex, surf->polys->numverts);

		mod->textures[i]->fbChain = NULL;
	}

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDisable (GL_BLEND);
	glDepthMask (GL_TRUE);

	fbChain = false;
}


/*
===============
R_DrawCausticChain

===============
*/
#define CAUSTIC_SCALE 0.003

static void R_DrawCausticChain (msurface_t *lqChain)
{
	int i;
	float *v;
	msurface_t *surf;
	float nv[2];
	float fogdist;
	float fogalpha;

	// enable blending for caustic textures
	glEnable (GL_BLEND);
	glBlendFunc (GL_ZERO, GL_SRC_COLOR);
	glDepthMask (GL_FALSE);
	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	// bind the caustic texture (same texture is used for all surfs)
	glBindTexture (GL_TEXTURE_2D, caustictexture);

	// scale the texture matrix to make the caustic texture bigger
	R_ScaleTextureMatrix (CAUSTIC_SCALE);

	// draw it
	for (surf = lqChain; surf; surf = surf->lqChain)
	{
		for (i = 0, v = surf->polys->verts; i < surf->polys->numverts; i++, v += VERTEXSIZE)
		{
			v[13] = v[surf->warpindex[0]] + sintable[(int) ((v[surf->warpindex[1]] * 0.025 + realtime) * STMult) & STBitMask];
			v[14] = v[surf->warpindex[1]] + sintable[(int) ((v[surf->warpindex[0]] * 0.025 + realtime) * STMult) & STBitMask];
		}

		glDrawArrays (GL_TRIANGLE_FAN, surf->polys->saindex, surf->polys->numverts);
	}

	// restore the texture matrix
	R_ScaleTextureMatrix (0);

	// shut down blending
	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	glDisable (GL_BLEND);
	glDepthMask (GL_TRUE);
}


/*
================
R_DrawFogSurfaces

OpenGL fog is not very robust when combined with multiple blended passes, so here's a hack for ya!  Fog
is still distance based, but the distance calcs are done in software while the actual fogging is done
in hardware using glColor4f.  Texturing is disabled for this pass.

This effectively emulates a hardware vertex shader :)
================
*/
static void R_DrawFogSurfaces (model_t *mod)
{
	int i;
	float *v;
	msurface_t *surf;
	glpoly_t *p;
	float fogdist;
	float fogalpha;

	// enable blending for caustic textures
	glEnable (GL_BLEND);
	glDepthMask (GL_FALSE);

	// "fake" underwater fog - do it in software cos hardware fog messes up when blending is done
	glDisable (GL_TEXTURE_2D);
	glShadeModel (GL_SMOOTH);

	for (surf = mod->fogSurfs; surf; surf = surf->fogSurfs)
	{
		if (surf->flags & SURF_DRAWTURB)
		{
			for (p = surf->polys; p; p = p->next)
			{
				for (i = 0, v = p->verts; i < p->numverts; i++, v += VERTEXSIZE)
				{
					fogdist = (v[0] - r_origin[0]) * vpn[0] + (v[1] - r_origin[1]) * vpn[1] + (v[2] - r_origin[2]) * vpn[2];

					if (fogdist < surf->fogBase)
						fogdist = surf->fogBase;
					else if (fogdist > surf->fogRange)
						fogdist = surf->fogRange;

					fogdist -= surf->fogBase;

					fogalpha = fogdist / surf->fogRange * surf->fogAlphaMax;

					v[12] = fogalpha;
				}

				glDrawArrays (GL_TRIANGLE_FAN, p->saindex, p->numverts);
			}
		}
		else
		{
			for (i = 0, v = surf->polys->verts; i < surf->polys->numverts; i++, v += VERTEXSIZE)
			{
				fogdist = (v[0] - r_origin[0]) * vpn[0] + (v[1] - r_origin[1]) * vpn[1] + (v[2] - r_origin[2]) * vpn[2];

				if (fogdist < surf->fogBase)
					fogdist = surf->fogBase;
				else if (fogdist > surf->fogRange)
					fogdist = surf->fogRange;

				fogdist -= surf->fogBase;

				fogalpha = fogdist / surf->fogRange * surf->fogAlphaMax;

				// have to set this dynamically cos brush models can potentially move from
				// one water volume to another
				v[9] = surf->fogColour[0];
				v[10] = surf->fogColour[1];
				v[11] = surf->fogColour[2];
				v[12] = fogalpha;
			}

			glDrawArrays (GL_TRIANGLE_FAN, surf->polys->saindex, surf->polys->numverts);
		}
	}

	glEnable (GL_TEXTURE_2D);

	glDisable (GL_BLEND);
	glDepthMask (GL_TRUE);

	// paranoia!!!
	glColor4f (1, 1, 1, 1);
}


void R_SetSurfaceVertexLight (msurface_t *surf, float *verts);

static void R_DrawVertLightSurfaces (model_t *mod)
{
	int i;
	int j;
	float *v;
	msurface_t *surf;
	glpoly_t *p;
	texture_t *t;

	// modulate with the surface primary colour (set by the color pointer)
	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	glShadeModel (GL_SMOOTH);

	for (i = 0; i < mod->numtextures; i++)
	{
		if (!mod->textures[i]) continue;
		if (!mod->textures[i]->vertlightchain) continue;

		t = R_TextureAnimation (mod->textures[i]);

		surf = mod->textures[i]->vertlightchain;

		glBindTexture (GL_TEXTURE_2D, t->gl_texture->texnum);

		for (; surf; surf = surf->vertlightchain)
		{
			for (j = 0, v = surf->polys->verts; j < surf->polys->numverts; j++, v += VERTEXSIZE)
			{
				// need to build the colours by hand here
				R_SetSurfaceVertexLight (surf, v);
			}

			glDrawArrays (GL_TRIANGLE_FAN, surf->polys->saindex, surf->polys->numverts);
		}

		mod->textures[i]->vertlightchain = NULL;
	}

	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

	// paranoia!!!
	glColor4f (1, 1, 1, 1);
}


static void R_DrawDetailSurfaces (void)
{
	msurface_t *surf = r_detailsurfaces;
	int		i;
	float	*v;
	glpoly_t *p;

	// nothing to draw!
	if (!surf) return;

	// bind the detail texture
	glTexCoordPointer (2, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[7]);
	glBindTexture (GL_TEXTURE_2D, r_detailtexture);

	// set the correct blending mode - modulate, combined with everything else that's going on,
	// gives absolutely fucking excellent results here, matey..
	glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glBlendFunc (GL_DST_COLOR, GL_SRC_COLOR);
	glEnable (GL_BLEND);

	// here we get clever and do our texture scaling in the texture matrix.  this saves
	// us from more runtime calculations in software
	glMatrixMode (GL_TEXTURE);
	glLoadIdentity ();

	// we only need to scale the X and Y co-ords, corresponding to texture s and t.
	glScalef (gl_detailtexture.value, gl_detailtexture.value, 1);

	// we can safely leave the texture matrix as the active mode while we're doing the draw
	for (; surf; surf = surf->detailchain)
	{
		p = surf->polys;

		// using GL_TRIANGLE_FAN is theoretically more efficient than GL_POLYGON as
		// the GL implementation can know something about what we are going to draw
		// in advance, and set things up accordingly.  the visual result is identical.
		glDrawArrays (GL_TRIANGLE_FAN, surf->polys->saindex, surf->polys->numverts);
	}

	// restore original texture matrix
	glLoadIdentity ();
	glMatrixMode (GL_MODELVIEW);

	// restore original blend
	glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDisable (GL_BLEND);
}


void R_BlendStainmaps (void);
qboolean needStains;
msurface_t *lqChain;


/*
================
R_SurfacePass

================
*/
__inline void R_PickupCausticFog (model_t *mod, msurface_t *surf)
{
	texture_t *caustic = mod->volumes[surf->volume].caustic;

	// pick up caustics
	surf->lqChain = lqChain;
	lqChain = surf;

	// pick up fog
	surf->fogSurfs = mod->fogSurfs;
	mod->fogSurfs = surf;

	surf->fogAlphaMax = 0.9;
	surf->fogBase = 0;
	surf->fogRange = 700;

	surf->fogColour[0] = caustic->uwater_fog[0];
	surf->fogColour[1] = caustic->uwater_fog[1];
	surf->fogColour[2] = caustic->uwater_fog[2];
}


__inline void R_PickupStainmap (msurface_t *surf)
{
	surf->polys->smChain = stainmapinfo[surf->lightmaptexturenum].polys;
	stainmapinfo[surf->lightmaptexturenum].polys = surf->polys;

	needStains = true;
}

// global for vertex lighting
qboolean vlight;

static void R_SurfacePass (model_t *mod, int volume, texture_t *caustic)
{
	int i;
	texture_t *t;
	msurface_t *surf;
	int oldLM = -1;

	for (i = 0; i < mod->numtextures; i++)
	{
		// this only happens in e2m3
		if (!mod->textures[i]) continue;

		// not on the draw list
		if (!(surf = mod->textures[i]->texturechain[volume])) continue;

		// get the current texture animation
		t = R_TextureAnimation (mod->textures[i]);

		// bind the current texture
		glActiveTexture (GL_TEXTURE0);
		glBindTexture (GL_TEXTURE_2D, t->gl_texture->texnum);

		// activate TMU1 for lightmaps
		glActiveTexture (GL_TEXTURE1);

		// sort all of the surfs
		for (; surf; surf = surf->drawChain)
		{
			// fullbrights are always picked up later becase they have to be drawn last in order
			// to shine through any fog we might have
			if (t->fullbright)
			{
				surf->fbChain = surf->texinfo->texture->fbChain;
				surf->texinfo->texture->fbChain = surf;

				fbChain = true;
			}

			// pick up stainmaps, caustics and fog
			if (surf->everstained) R_PickupStainmap (surf);
			if (caustic) R_PickupCausticFog (mod, surf);

			// pick up the detail chain
			surf->detailchain = r_detailsurfaces;
			r_detailsurfaces = surf;

			// vertex light?
			if (gl_vertexlight.value)
			{
				// accumulate the vertex lighting chain
				surf->vertlightchain = mod->textures[i]->vertlightchain;
				mod->textures[i]->vertlightchain = surf;

				// notify the render that we will need to draw these chains
				vlight = true;
				surf->lightmapmodified = false;

				// don't send it through the regular render path
				continue;
			}

			// bind the lightmap texture if necessary
			if (surf->lightmaptexturenum != oldLM)
			{
				glBindTexture (GL_TEXTURE_2D, lightmap_textures + surf->lightmaptexturenum);
				oldLM = surf->lightmaptexturenum;
			}

			// draw the surf
			glDrawArrays (GL_TRIANGLE_FAN, surf->polys->saindex, surf->polys->numverts);
		}

		// explicitly NULL the texture chain
		mod->textures[i]->texturechain[volume] = NULL;
	}
}


static void R_DrawVolumeChains (model_t *mod)
{
	int i;

	msurface_t *surf;

	// apply all current stains before doing the lightmaps
	// we don't stain BSP brush models (aside from the world model)
	if (mod == cl.worldmodel)
		R_ApplyStains (mod->nodes);
	else if (mod != cl.worldmodel && mod->name[0] == '*')
		R_ApplyStains (mod->nodes + mod->hulls[0].firstclipnode);

	// prebuild all lightmaps for this model
	R_PrebuildLightmaps ();

	// initialize accumulated surfs - these are needed for any extra surfs we have to pick up
	mod->fogSurfs = NULL;
	needStains = false;
	lqChain = NULL;
	r_detailsurfaces = NULL;
	vlight = false;

	// batch up vertex arrays
	// they don't tell you that enable client state is really needed per tmu
	// (sometimes it works if you don't set it, sometimes it doesn't...)
	/*
	if (OpenGL_1_5)
	{*/
		/*
		=========================================================================================================
		VERTEX BUFFERS

		This functionality is *very* Direct 3D, isn't it?  Anyway, I wrote it, got it working, only to find
		that I dropped 100 FPS.  So I likewise decided to drop Vertex Buffers.  Ho hum.  And there I was
		expecting to get a few extra frames from even just putting it on the static world only.  Funny
		enough, I got much the same outcome from Display Lists, which similar outlandish claims were made
		of.  I've even seen zero difference in a demo program I got my mitts on.

		Maybe these things only work properly in demo programs, but are totally unsuited to real world apps.
		Free bottle of hair restorer with every TRIANGLE_FAN, I reckon.  The snake-oil merchants are in town.

		Anyway, I've left all of the supporting code in but commented out, so if you're interested in having a 
		look at this for yourself, feel free.  Just search for OpenGL_1_5 in gl_rsurf, gl_rmisc and gl_vidnt
		=========================================================================================================
		*/
	/*
		// bind the buffer object so we can use it
		glBindBuffer (GL_ARRAY_BUFFER, r_worldvb);

		// use offsets into the vertex buffer rather than pointer indirection for our pointers here
		glEnableClientState (GL_VERTEX_ARRAY);
		glVertexPointer (3, GL_FLOAT, 0, 0);

		glBindBuffer (GL_ARRAY_BUFFER, 0);

		glClientActiveTexture (GL_TEXTURE0);
		glEnableClientState (GL_TEXTURE_COORD_ARRAY);
		//glTexCoordPointer (2, GL_FLOAT, (8 * sizeof (GLfloat)), (GLfloat *) (3 * sizeof (GLfloat)));
		glTexCoordPointer (2, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[3]);

		glClientActiveTexture (GL_TEXTURE1);
		glEnableClientState (GL_TEXTURE_COORD_ARRAY);
		//glTexCoordPointer (2, GL_FLOAT, (8 * sizeof (GLfloat)), (GLfloat *) (5 * sizeof (GLfloat)));
		glTexCoordPointer (2, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[5]);
	}
	else
	{*/
		glEnableClientState (GL_VERTEX_ARRAY);
		glVertexPointer (3, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[0]);

		glClientActiveTexture (GL_TEXTURE0);
		glEnableClientState (GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer (2, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[3]);

		glClientActiveTexture (GL_TEXTURE1);
		glEnableClientState (GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer (2, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[5]);
	/*}*/

	glActiveTexture (GL_TEXTURE1);
	glEnable (GL_TEXTURE_2D);
	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	// sort out the surfaces
	for (i = 0; i < 16; i++)
	{
		// brush models only have one volume set!!!
		if (i >= mod->numVolumes) break;

		// cos all the volumes are moved to the front of the list, once we hit one with
		// a visframe of -2 we can be sure that the rest will have no surfs
		if (mod->volumes[i].visframe == -2) break;

		// the volume has no drawlist for this frame
		if (mod->volumes[i].visframe != r_framecount) continue;

		// run the initial surface pass.  if vertex lighting is enabled this will just accumulate surfs but
		// not actually draw anything.  if lightmaps are enabled, it will draw.  we'll have some unnecessary
		// state changes with vertex lighting here, but it's probably not such a big deal.  i might tidy it
		// up later if i feel like it
		R_SurfacePass (mod, i, mod->volumes[i].caustic);
	}

	// bring down TMU 1 (it's always active coming out of here...)
	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	glDisable (GL_TEXTURE_2D);
	glClientActiveTexture (GL_TEXTURE1);
	glDisableClientState (GL_TEXTURE_COORD_ARRAY);
	glActiveTexture (GL_TEXTURE0);
	glClientActiveTexture (GL_TEXTURE0);
	glEnableClientState (GL_TEXTURE_COORD_ARRAY);

	/*
	if (OpenGL_1_5)
	{
		// revert our vertex and texcoord pointers to standard vertex arrays
		// first, unbind the buffer object so it's not used until we ask for it.  this is done by 
		// calling glBindBuffer with a buffer number of 0
		glBindBuffer (GL_ARRAY_BUFFER, 0);

		// and now set the pointers
		glVertexPointer (3, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[0]);
		glTexCoordPointer (2, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[3]);
	}
	*/

	// detail
	// I *had* done a 3TMU version, but it was a bit slower than this.
	// odd... anyway, I could never get the blend looking the same.
	if (gl_detailtexture.value) R_DrawDetailSurfaces ();

	// run the pick up passes if necessary
	if (needStains || lqChain || mod->fogSurfs || fbChain || vlight)
	{
		qboolean DrawFog = false;

		// vertex light chains
		if (vlight)
		{
			// set up a color pointer for these as well
			glEnableClientState (GL_COLOR_ARRAY);
			glColorPointer (4, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[9]);

			// run through the vertex light surface drawer
			R_DrawVertLightSurfaces (mod);

			// take down the color pointer
			glDisableClientState (GL_COLOR_ARRAY);
		}

		// stainmaps
		if (needStains)
		{
			glTexCoordPointer (2, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[5]);
			R_BlendStainmaps ();
		}

		// water caustics
		if (lqChain)
		{
			glTexCoordPointer (2, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[13]);
			R_DrawCausticChain (lqChain);
		}

		// vertex fog
		if (mod->fogSurfs)
		{
			DrawFog = true;
			glDisableClientState (GL_TEXTURE_COORD_ARRAY);
			glEnableClientState (GL_COLOR_ARRAY);
			glColorPointer (4, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[9]);
			R_DrawFogSurfaces (mod);
			glDisableClientState (GL_COLOR_ARRAY);
		}

		// fullbrights
		if (fbChain)
		{
			if (DrawFog) glEnableClientState (GL_TEXTURE_COORD_ARRAY);
			glTexCoordPointer (2, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[3]);
			R_DrawFullbrightChain (mod);
		}
	}

	// kill vertex arrays
	glDisableClientState (GL_TEXTURE_COORD_ARRAY);
	glDisableClientState (GL_VERTEX_ARRAY);
}


/*
================
R_DrawWaterSurfaces

Each water texture is given a different scaling factor and blend mode
================
*/
void R_DrawWaterChain (msurface_t *surf, float alpha);


void R_DrawWaterSurfaces (void)
{
	int i;
	msurface_t *surf;
	texture_t *t;
	float scale;
	float alpha;

	// only go through here if we have to
	if (!waterthisframe) return;

	// go back to the world matrix
	glLoadMatrixf (cl_entities[0].MViewMatrix);

	cl.worldmodel->fogSurfs = NULL;

	glEnable (GL_BLEND);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDepthMask (GL_FALSE);

	glEnableClientState (GL_VERTEX_ARRAY);
	glVertexPointer (3, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[0]);

	glActiveTexture (GL_TEXTURE0);
	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	glEnableClientState (GL_TEXTURE_COORD_ARRAY);
	glClientActiveTexture (GL_TEXTURE0);
	glTexCoordPointer (2, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[13]);

	// hack - use 5 for the index of colour, cos we'll also want to use 9 for fog on the same verts
	glEnableClientState (GL_COLOR_ARRAY);
	glColorPointer (4, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[5]);

	for (i = 0; i < cl.worldmodel->numtextures; i++)
	{
		t = cl.worldmodel->textures[i];
		if (!t) continue;
		if (t->name[0] != '*') continue;
		if (!t->lqChain) continue;

		surf = t->lqChain;

		if (surf->flags & SURF_DRAWSLIME)
		{
			// slime
			scale = 128.0;
			alpha = 0.8;
		}
		else if (surf->flags & SURF_DRAWLAVA)
		{
			// lava
			scale = 96.0;
			alpha = 0.7;
		}
		else if (surf->flags & SURF_DRAWTELE)
		{
			// teleport
			scale = 64.0;
			alpha = 1.0;
		}
		else
		{
			// the wet stuff
			scale = 192.0;
			alpha = 0.6;
		}

		// do water calculations in hardware - I can't do a global setting of this cos the
		// scale can be different for each liquid type
		glActiveTexture (GL_TEXTURE0);
		glMatrixMode (GL_TEXTURE);
		glLoadIdentity ();
		glScalef (1.0 / scale, 1.0 / scale, 1.0);
		glMatrixMode (GL_MODELVIEW);

		glBindTexture (GL_TEXTURE_2D, surf->texinfo->texture->gl_texture->texnum);

		R_DrawWaterChain (surf, alpha);

		t->lqChain = NULL;
	}

	glColor4f (1, 1, 1, 1);

	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDisable (GL_BLEND);
	glDepthMask (GL_TRUE);

	glDisableClientState (GL_TEXTURE_COORD_ARRAY);

	glActiveTexture (GL_TEXTURE0);
	glMatrixMode (GL_TEXTURE);
	glLoadIdentity ();
	glMatrixMode (GL_MODELVIEW);

	if (cl.worldmodel->fogSurfs)
	{
		glColorPointer (4, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[9]);
		R_DrawFogSurfaces (cl.worldmodel);
	}

	glDisableClientState (GL_COLOR_ARRAY);
	glDisableClientState (GL_VERTEX_ARRAY);
}


/*
=================
R_DrawBrushModel
=================
*/
static void R_DrawInlineBModel (entity_t *ent, model_t *clmodel)
{
	int i, k;
	mplane_t *pplane;
	float dotp;
	msurface_t *surf;
	mleaf_t *bLeaf;
	vec3_t neworg;

	// calculate dynamic lighting for bmodel
	if (clmodel->firstmodelsurface != 0 && clmodel->name[0] == '*')
	{
		R_PushDlights (clmodel->nodes + clmodel->hulls[0].firstclipnode, true);
	}

	// here we get or set the leaf the bmodel is in
	if (clmodel->name[0] == '*' && waterthisframe && clmodel->modelleaf)
	{
		// see has the bmodel moved
		if (ent->origin[0] != 0 && ent->origin[1] != 0 && ent->origin[2] != 0)
		{
			// move it up a bit
			neworg[0] = clmodel->origin[0] + ent->origin[0];
			neworg[1] = clmodel->origin[1] + ent->origin[1];
			neworg[2] = clmodel->origin[2] + ent->origin[2] + 8;

			bLeaf = Mod_PointInLeaf (neworg, cl.worldmodel);
		}
		else bLeaf = clmodel->modelleaf;
	}
	else
	{
		// just set to the viewleaf
		bLeaf = r_viewleaf;
	}

	// set up the "fake" volume
	bmvol[0].visframe = r_framecount;
	bmvol[1].visframe = -2;
	bmvol[0].volumenum = 0;

	// ammo and health should never be marked underwater
	if (clmodel->name[0] != '*')
		bmvol[0].caustic = NULL;
	else bmvol[0].caustic = bLeaf->caustic;

	// allocate it to the model
	clmodel->volumes = bmvol;
	clmodel->numVolumes = 2;

	// now get it again cos we need it again
	surf = &clmodel->surfaces[clmodel->firstmodelsurface];

	for (i = 0; i < clmodel->nummodelsurfaces; i++, surf++)
	{
		if (surf->flags & (SURF_DRAWSKY | SURF_DRAWTURB)) continue;

		switch (surf->plane->type)
		{
		case PLANE_X:
			dotp = modelorg[0] - surf->plane->dist;
			break;
		case PLANE_Y:
			dotp = modelorg[1] - surf->plane->dist;
			break;
		case PLANE_Z:
			dotp = modelorg[2] - surf->plane->dist;
			break;
		default:
			dotp = DotProduct (modelorg, surf->plane->normal) - surf->plane->dist;
			break;
		}

		if (((surf->flags & SURF_PLANEBACK) && (dotp < -BACKFACE_EPSILON)) || (!(surf->flags & SURF_PLANEBACK) && (dotp > BACKFACE_EPSILON)))
		{
			// if it's a BSP bmodel don't bother updating light
			if (clmodel->name[0] == '*') R_PrebuildSurfaceLightmap (surf);

			// build the surface drawing chain
			surf->drawChain = surf->texinfo->texture->texturechain[0];
			surf->texinfo->texture->texturechain[0] = surf;

			addSurfs++;
		}
	}

	// and draw, blondie you son of a thousand fathers, all bastards like you!!!
	R_DrawVolumeChains (clmodel);
}


void R_RotateForEntity (entity_t *e, qboolean blendLerp);


void R_DrawBrushModel (entity_t *e)
{
	vec3_t mins, maxs;
	int i, j;
	qboolean rotated;
	model_t	*clmodel;
	float originhack[3];

	clmodel = e->model;

	if (clmodel->nummodelsurfaces == 0) return;

	currententity = e;

	if (e->angles[0] || e->angles[1] || e->angles[2])
	{
		rotated = true;

		for (i = 0; i < 3; i++)
		{
			mins[i] = e->origin[i] - clmodel->radius;
			maxs[i] = e->origin[i] + clmodel->radius;
		}
	}
	else
	{
		rotated = false;

		VectorAdd (e->origin, clmodel->mins, mins);
		VectorAdd (e->origin, clmodel->maxs, maxs);
	}

	if (R_CullBox (mins, maxs)) return;

	glColor3f (1, 1, 1);

	VectorSubtract (r_refdef.vieworg, e->origin, modelorg);

	if (rotated)
	{
		vec3_t	temp;
		vec3_t	forward, right, up;

		VectorCopy (modelorg, temp);
		AngleVectors (e->angles, forward, right, up);
		modelorg[0] = DotProduct (temp, forward);
		modelorg[1] = -DotProduct (temp, right);
		modelorg[2] = DotProduct (temp, up);
	}

	glPushMatrix ();

	// seems i fixed this somewhere else...  do do dooo do do do doooo (looks away)
	//e->angles[0] = -e->angles[0];	// stupid quake bug
	//e->angles[2] = -e->angles[2];	// stupid quake bug

	// hack to stop z-fighting in places like the plat to the quad in e1m1
	originhack[0] = e->origin[0];
	originhack[1] = e->origin[1];
	originhack[2] = e->origin[2];

	e->origin[0] -= DIST_EPSILON;
	e->origin[1] -= DIST_EPSILON;
	e->origin[2] -= DIST_EPSILON;

	R_RotateForEntity (e, true);

	e->origin[0] = originhack[0];
	e->origin[1] = originhack[1];
	e->origin[2] = originhack[2];

	// seems i fixed this somewhere else...  do do dooo do do do doooo (looks away)
	//e->angles[0] = -e->angles[0];	// stupid quake bug
	//e->angles[2] = -e->angles[2];	// stupid quake bug

	R_DrawInlineBModel (e, clmodel);

	glPopMatrix ();
}


/*
=============================================================

	WORLD MODEL

=============================================================
*/

/*
================
R_RecursiveWorldNode
================
*/
int	addLeafs;

static void R_RecursiveWorldNode (mnode_t *node)
{
	int i;
	int c;
	int side;
	mplane_t *plane;
	msurface_t *surf;
	msurface_t **mark;
	mleaf_t *pleaf;
	double dist;
	int sidebit;

	if (node->contents == CONTENTS_SOLID) return;
	if (node->visframe != r_visframecount) return;
	if (R_CullBox (node->minmaxs, node->minmaxs + 3)) return;

	// if a leaf node, mark the surfaces as visible
	if (node->contents < 0)
	{
		pleaf = (mleaf_t *) node;

		mark = pleaf->firstmarksurface;
		c = pleaf->nummarksurfaces;

		if (c)
		{
			do
			{
				// mark the surfaces in this leaf for drawing
				(*mark)->visframe = r_framecount;
				mark++;
			} while (--c);
		}

		// deal with model fragments in this leaf
		// (only used for static entities - torches and suchlike)
		if (pleaf->efrags) R_StoreEfrags (&pleaf->efrags);

		return;
	}

	// node is just a decision point, so go down the apropriate sides
	// find which side of the node we are on
	plane = node->plane;

	switch (plane->type)
	{
	case PLANE_X:
		dist = modelorg[0] - plane->dist;
		break;
	case PLANE_Y:
		dist = modelorg[1] - plane->dist;
		break;
	case PLANE_Z:
		dist = modelorg[2] - plane->dist;
		break;
	default:
		dist = DotProduct (modelorg, plane->normal) - plane->dist;
		break;
	}

	if (dist >= 0)
	{
		// front side
		side = 0;
		sidebit = 0;
	}
	else
	{
		// back side
		side = 1;
		sidebit = SURF_PLANEBACK;
	}

	// recurse down the children, front side first
	R_RecursiveWorldNode (node->children[side]);

	if (node->numsurfaces)
	{
		for (c = node->numsurfaces, surf = cl.worldmodel->surfaces + node->firstsurface; c; c--, surf++)
		{
			// not marked as visible
			if (surf->visframe != r_framecount) continue;

			// node backfaces
			if ((surf->flags & SURF_PLANEBACK) != sidebit) continue;

			// check for surf backfacing
			switch (surf->plane->type)
			{
			case PLANE_X:
				dist = modelorg[0] - surf->plane->dist;
				break;
			case PLANE_Y:
				dist = modelorg[1] - surf->plane->dist;
				break;
			case PLANE_Z:
				dist = modelorg[2] - surf->plane->dist;
				break;
			default:
				dist = DotProduct (modelorg, surf->plane->normal) - surf->plane->dist;
				break;
			}

			// surf backfaces
			if (!(((surf->flags & SURF_PLANEBACK) && (dist < -BACKFACE_EPSILON)) || 
				(!(surf->flags & SURF_PLANEBACK) && (dist > BACKFACE_EPSILON)))) continue;

			if (surf->flags & SURF_DRAWSKY)
			{
				skythisframe = true;
			}
			else if (surf->flags & SURF_DRAWTURB)
			{
				// store whether water is visible this frame to reduce Mod_PointInLeaf checking
				// for brush models, as well as to determine whether or not we need to draw it!!!
				waterthisframe = true;

				surf->lqChain = surf->texinfo->texture->lqChain;
				surf->texinfo->texture->lqChain = surf;

				addSurfs++;

				// mark any surfs we've seen for automapping
				surf->inClientPVS = true;
			}
			else
			{
				// now that we have the lighting info we can update it here.
				R_PrebuildSurfaceLightmap (surf);

				// build a seperate texture chain for each volume
				surf->drawChain = surf->texinfo->texture->texturechain[surf->volume];
				surf->texinfo->texture->texturechain[surf->volume] = surf;

				// mark the volume as visible
				cl.worldmodel->volumes[surf->volume].visframe = r_framecount;

				addSurfs++;

				// mark any surfs we've seen for automapping
				surf->inClientPVS = true;
			}
		}
	}

	// recurse down the back side
	R_RecursiveWorldNode (node->children[!side]);
}


/*
=============
R_DrawWorld
=============
*/
float lighttime = 0;
void R_PickSkyDrawer (void);


void R_DrawWorld (void)
{
	entity_t	ent;

	memset (&ent, 0, sizeof (ent));
	ent.model = cl.worldmodel;

	// modelorg here is the player's origin in the world
	VectorCopy (r_refdef.vieworg, modelorg);

	currententity = &ent;

	// bound cvars to reasonable/sensible values
	Cvar_Bound (&gl_detailtexture, 0, 999);
	glColor3f (1, 1, 1);

	// only update lighting 10 times per second.  next to no visible difference.
	// this has to be done before anything is either drawn or marked otherwise lighting on
	// the world will go out of sync with lighting on bmodels
	if (lighttime >= LIGHT_UPLOAD) lighttime = 0;
	lighttime += frametime;

	// default values
	skythisframe = false;
	waterthisframe = false;

	// mark dynamic lights for the world
	R_PushDlights (cl.worldmodel->nodes, false);

	// marks surfaces and updates lighting
	R_RecursiveWorldNode (cl.worldmodel->nodes);

	// and draw, Texan!!!
	// (The best jokes work on more than one level.  Here I'm not only punning on the word
	// "draw" by putting it in a completely irrelevant Western context, but also bringing in
	// the fact that ID are based in Texas.  I'll get me coat...)
	R_DrawVolumeChains (cl.worldmodel);

	// we make sure here that we only draw sky when required
	if (skythisframe) R_PickSkyDrawer ();
}


/*
===============
AddVisibleLeaf

Adds the specified leaf to the PVS
This is tolerant of adding the same leaf twice...
===============
*/
static void AddVisibleLeaf (mleaf_t *VLeaf)
{
	byte	*vis;
	mnode_t	*node;
	int		i;
	byte	solid[4096];

	// already been added this frame
	if (VLeaf->waterVisFrame == r_visframecount) return;

	if (r_novis.value)
	{
		vis = solid;

		// set it to 255 so the bits always match
		memset (solid, 0xff, (cl.worldmodel->numleafs + 7) >> 3);
	}
	else vis = Mod_LeafPVS (VLeaf, cl.worldmodel);

	addLeafs = 0;

	for (i = 0; i < cl.worldmodel->numleafs; i++)
	{
		if (vis[i >> 3] & (1 << (i & 7)))
		{
			addLeafs++;

			// mark the visible nodes
			node = (mnode_t *) &cl.worldmodel->leafs[i + 1];

			do
			{
				// already added
				if (node->visframe == r_visframecount) break;

				node->visframe = r_visframecount;
				node = node->parent;
			}
			while (node);
		}
	}

	// mark it as having been checked
	VLeaf->waterVisFrame = r_visframecount;
}


/*
===============
R_MarkLeaves
===============
*/
extern cvar_t gl_lockpvs;

void R_MarkLeaves (void)
{
	//addSurfs = 0;

	// still in the same leaf
	if (r_oldviewleaf == r_viewleaf) return;

	if (gl_lockpvs.value) return;

	// this only increments when the viewleaf changes
	r_visframecount++;

	// add the viewleaf (we can add PVS for any leaf we want with this function)
	AddVisibleLeaf (r_viewleaf);

	// is this necessary any more???
	if (r_oldviewleaf)
	{
		if (r_oldviewleaf->contents != r_viewleaf->contents)
		{
			// fix water transitions.  If we have translucent water it's probably not
			// much of a bother as the vis for both leafs is likely to be very similar.
			// The only time this could cause bother is if we teleport from above water
			// to underwater or vice versa, although the hit is probably not that bad.
			AddVisibleLeaf (r_oldviewleaf);
		}
	}

	r_oldviewleaf = r_viewleaf;
}



int bound (int lower, int num, int upper)
{
	if (num < lower) return lower;
	if (num > upper) return upper;
	return num;
}


int worldverts = 0;


void R_PolyMidPoint (glpoly_t *p)
{
	int i;
	vec3_t mid = {0, 0, 0};
	float *v;

	if (!p->numverts) return;

	for (i = 0, v = p->verts; i < p->numverts; i++, v += VERTEXSIZE)
	{
		mid[0] += v[0];
		mid[1] += v[1];
		mid[2] += v[2];
	}

	p->midpoint[0] = mid[0] / p->numverts;
	p->midpoint[1] = mid[1] / p->numverts;
	p->midpoint[2] = mid[2] / p->numverts;
}

void GL_BuildPolygonFromSurface (model_t *mod, msurface_t *surf)
{
	int			i, lindex, lnumverts;
	medge_t		*pedges, *r_pedge;
	float		*vec;
	float		s, t;
	glpoly_t	*poly;
	float		*v;
	vec3_t		midpoint;

	// reconstruct the polygon
	pedges = mod->edges;
	lnumverts = surf->numedges;

	// draw texture
	poly = Hunk_Alloc (sizeof (glpoly_t));
	poly->next = surf->polys;
	poly->flags = surf->flags;
	surf->polys = poly;
	poly->numverts = lnumverts;

	poly->saindex = worldverts;
	poly->verts = &surfarray[worldverts * VERTEXSIZE];

	midpoint[0] = midpoint[1] = midpoint[2] = 0;

	for (i = 0, v = poly->verts; i < lnumverts; i++, v += VERTEXSIZE)
	{
		lindex = mod->surfedges[surf->firstedge + i];

		if (lindex > 0)
		{
			r_pedge = &pedges[lindex];
			vec = mod->vertexes[r_pedge->v[0]].position;
		}
		else
		{
			r_pedge = &pedges[-lindex];
			vec = mod->vertexes[r_pedge->v[1]].position;
		}

		// poly world verts
		v[0] = vec[0];
		v[1] = vec[1];
		v[2] = vec[2];

		// midpoint
		midpoint[0] += v[0];
		midpoint[1] += v[1];
		midpoint[2] += v[2];

		// texture s and t
		v[3] = (DotProduct (vec, surf->texinfo->vecs[0]) + surf->texinfo->vecs[0][3]) / surf->texinfo->texture->width;
		v[4] = (DotProduct (vec, surf->texinfo->vecs[1]) + surf->texinfo->vecs[1][3]) / surf->texinfo->texture->height;

		// make a 128 x 128 scale version
		v[7] = (DotProduct (vec, surf->texinfo->vecs[0]) + surf->texinfo->vecs[0][3]) / 128;
		v[8] = (DotProduct (vec, surf->texinfo->vecs[1]) + surf->texinfo->vecs[1][3]) / 128;

		worldverts++;

		if (worldverts > MAX_MAP_VERTS * 2) Sys_Error ("Worldverts > %i", MAX_MAP_VERTS * 2);
	}

	// surface midpoint
	surf->midpoint[0] = midpoint[0] / surf->polys->numverts;
	surf->midpoint[1] = midpoint[1] / surf->polys->numverts;
	surf->midpoint[2] = midpoint[2] / surf->polys->numverts;
}


void R_InitSkyChain (void);

void Light_Init (void);

void fractalnoise(byte *noise, int size, int startgrid)
{
	int x, y, g, g2, amplitude, min, max, size1 = size - 1, sizepower, gridpower;
	int *noisebuf;
#define n(x,y) noisebuf[((y)&size1)*size+((x)&size1)]

	for (sizepower = 0;(1 << sizepower) < size;sizepower++);
	if (size != (1 << sizepower))
		Sys_Error("fractalnoise: size must be power of 2\n");

	for (gridpower = 0;(1 << gridpower) < startgrid;gridpower++);
	if (startgrid != (1 << gridpower))
		Sys_Error("fractalnoise: grid must be power of 2\n");

	startgrid = bound (0, startgrid, size);

	amplitude = 0xFFFF; // this gets halved before use
	noisebuf = malloc(size*size*sizeof(int));
	memset(noisebuf, 0, size*size*sizeof(int));

	for (g2 = startgrid;g2;g2 >>= 1)
	{
		// brownian motion (at every smaller level there is random behavior)
		amplitude >>= 1;
		for (y = 0;y < size;y += g2)
			for (x = 0;x < size;x += g2)
				n(x,y) += (rand()&amplitude);

		g = g2 >> 1;
		if (g)
		{
			// subdivide, diamond-square algorithm (really this has little to do with squares)
			// diamond
			for (y = 0;y < size;y += g2)
				for (x = 0;x < size;x += g2)
					n(x+g,y+g) = (n(x,y) + n(x+g2,y) + n(x,y+g2) + n(x+g2,y+g2)) >> 2;
			// square
			for (y = 0;y < size;y += g2)
				for (x = 0;x < size;x += g2)
				{
					n(x+g,y) = (n(x,y) + n(x+g2,y) + n(x+g,y-g) + n(x+g,y+g)) >> 2;
					n(x,y+g) = (n(x,y) + n(x,y+g2) + n(x-g,y+g) + n(x+g,y+g)) >> 2;
				}
		}
	}
	// find range of noise values
	min = max = 0;
	for (y = 0;y < size;y++)
		for (x = 0;x < size;x++)
		{
			if (n(x,y) < min) min = n(x,y);
			if (n(x,y) > max) max = n(x,y);
		}
	max -= min;
	max++;
	// normalize noise and copy to output
	for (y = 0;y < size;y++)
		for (x = 0;x < size;x++)
			*noise++ = (byte) (((n(x,y) - min) * 256) / max);
	free(noisebuf);
#undef n
}


void R_BuildDetailTexture (void)
{
	int x, y, light;
	float vc[3], vx[3], vy[3], vn[3], lightdir[3];

// increase this if you want
#define DETAILRESOLUTION 256

	byte data[DETAILRESOLUTION][DETAILRESOLUTION][4], noise[DETAILRESOLUTION][DETAILRESOLUTION];

	lightdir[0] = 0.5;
	lightdir[1] = 1;
	lightdir[2] = -0.25;
	VectorNormalize(lightdir);

	fractalnoise(&noise[0][0], DETAILRESOLUTION, DETAILRESOLUTION >> 4);

	for (y = 0;y < DETAILRESOLUTION;y++)
	{
		for (x = 0;x < DETAILRESOLUTION;x++)
		{
			vc[0] = x;
			vc[1] = y;
			vc[2] = noise[y][x] * (1.0f / 32.0f);
			vx[0] = x + 1;
			vx[1] = y;
			vx[2] = noise[y][(x + 1) % DETAILRESOLUTION] * (1.0f / 32.0f);
			vy[0] = x;
			vy[1] = y + 1;
			vy[2] = noise[(y + 1) % DETAILRESOLUTION][x] * (1.0f / 32.0f);
			VectorSubtract(vx, vc, vx);
			VectorSubtract(vy, vc, vy);
			CrossProduct(vx, vy, vn);
			VectorNormalize(vn);
			light = 128 - DotProduct(vn, lightdir) * 128;
			light = bound(0, light, 255);
			data[y][x][0] = data[y][x][1] = data[y][x][2] = light;
			data[y][x][3] = 255;
		}
	}

	// first, bind the texture.
	glBindTexture (GL_TEXTURE_2D, r_detailtexture);

	// set the hint for mipmap generation
	glHint (GL_GENERATE_MIPMAP_HINT, GL_NICEST);

	// now switch mipmap generation on.
	// this has been part of the core OpenGL specification since version 1.4, so there's 
	// no real need to check for extension support and other tiresomely silly blather,
	// unless you have a really weird OpenGL implementation.  it does, however, even
	// work on ATI cards, so you should have no trouble with it.
	glTexParameteri (GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);

	// upload the correct texture data without any lightscaling interference
	glTexImage2D (GL_TEXTURE_2D, 
				  0, 
				  GL_RGBA, 
				  DETAILRESOLUTION, 
				  DETAILRESOLUTION, 
				  0, 
				  GL_RGBA, 
				  GL_UNSIGNED_BYTE, 
				  (byte *) data);

	// set some quick and ugly filtering modes.  detail texturing works by scrunching a
	// large image into a small space, so there's no need for sexy filtering (change this,
	// turn off the blend in R_DrawDetailSurfaces, and compare if you don't believe me).
	// this also helps to take some of the speed hit from detail texturing away.
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
}


void GL_RSurfInit (void)
{
	int i;
	float j;
	float inc = (2.0 * M_PI) / (float) SINTABLE_SIZE;

	// shut down all texture units except TMU 0 (just make sure...)
	for (i = 1; i < gl_textureunits; i++)
	{
		glActiveTexture (GL_TEXTURE0 + i);
		glDisable (GL_TEXTURE_2D);
	}

	// TMU 0 should always be up
	glActiveTexture (GL_TEXTURE0);
	glEnable (GL_TEXTURE_2D);

	// set up the sin table for warps
	for (i = 0, j = 0; i < SINTABLE_SIZE; i++, j += inc)
	{
		sintable[i] = sin (j) * 8.0;
	}

	STBitMask = SINTABLE_SIZE - 1;

	STMult = (float) SINTABLE_SIZE / (2.0 * M_PI);

	// set up the sky display list
	R_InitSkyChain ();

	Light_Init ();

	// set up the detail texture
	if (r_detailtexture == -1)
		r_detailtexture = texture_extension_number++;

	R_BuildDetailTexture ();
}

