/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/


/*
=============================================================================================

									SHADERS

		MHQuake stores shaders as resources within the engine itself.  I do appreciate
		that this goes against one of the main points about shaders, but it makes
		things easier here.  They're not really meant to be modified anyway...

		Because shaders are required to be NULL terminated, MHQuake hacks around this
		by adding ~END to the end of the shader lump... this is then parsed out of the
		lump and replaced with a NULL termiantion before the shader is sent through
		the normal compilation/linking routines...

		These shader management routines are primarily my own work.  I don;t know if it's
		a good or bad way to manage shaders, cos to be honest there isn't much in the
		way of good sample code based on practical real-world examples out there.

=============================================================================================
*/


#include "quakedef.h"
#include "glext.h"
#include "resource.h"
#include "gl_shaders.h"


// make sure to initialize this struct!!!
shaders_t GL_Shaders[TOTAL_SHADERS] =
{
//   handle   resourceID-------------    PO OK     number and name (just for ease of reference)
	{{0, 0}, {/*IDB_SHADER1, IDB_SHADER2*/ 0, 0}, 0, false, SHADER_SURFACEREFRESH, "Normal Surface Shader"},
};


extern HINSTANCE global_hInstance;

char shaderBuffer[10240];

qboolean LoadShaderFromResource (int shaderResourceID)
{
	char *shaderString;
	HANDLE hResInfo, hRes;
	int i;

	// Find the SHADER resource.
	hResInfo = FindResource (global_hInstance, MAKEINTRESOURCE (shaderResourceID), "SHADER");

	if (hResInfo == NULL)
		return false;

	// Load the SHADER resource.
	hRes = LoadResource (global_hInstance, hResInfo);

	if (hRes == NULL)
		return false;

	// Lock the SHADER resource and use it.
	shaderString = LockResource (hRes);

	if (shaderString != NULL)
	{
		// copy to shaderbuffer and NULL terminate
		for (i = 0; ; i++)
		{
			shaderBuffer[i] = shaderString[i];

			// a shader is meaningless unless it has void main () {}
			if (i > 10)
			{
				if (shaderBuffer[i - 3] == '~' && shaderBuffer[i - 2] == 'E' &&
					shaderBuffer[i - 1] == 'N' && shaderBuffer[i] == 'D')
				{
					shaderBuffer[i - 3] = '\0';
					break;
				}
			}
			else if (i == 10239)
			{
				// shader too long
				UnlockResource (hRes);
				FreeResource (hRes);

				return false;
			}
		}

		// finish up
		UnlockResource (hRes);
	}

	// Free the SHADER resource and return success or failure. 
	FreeResource (hRes);

	return true;
}


qboolean GL_LoadShader (int shaderNum, GLenum GL_ShaderType)
{
	GLint compileCheck;
	int QGL_ShaderType;
	char *shaderSource[1];

	if (shaderNum >= TOTAL_SHADERS) return false;

	if (GL_ShaderType == GL_VERTEX_SHADER)
		QGL_ShaderType = QGL_VERTEX_SHADER;
	else QGL_ShaderType = QGL_FRAGMENT_SHADER;

	// load the shader
	if (!LoadShaderFromResource (GL_Shaders[shaderNum].ResourceID[QGL_ShaderType])) return false;

	// create a shader object for it
	GL_Shaders[shaderNum].Shader[QGL_ShaderType] = glCreateShaderObject (GL_ShaderType);

	// set the pointer for the source string
	shaderSource[0] = shaderBuffer;
	glShaderSource (GL_Shaders[shaderNum].Shader[QGL_ShaderType], 1, shaderSource, NULL);

	// compile shader and check for any errors
	glCompileShader (GL_Shaders[shaderNum].Shader[QGL_ShaderType]);
	glGetObjectParameteriv (GL_Shaders[shaderNum].Shader[QGL_ShaderType], GL_OBJECT_COMPILE_STATUS, &compileCheck);

	if (!compileCheck) return false;

	return true;
}


void GL_MakeProgramObject (int shaderNum)
{
	int linkValidateOK;

	// load the shaders
	if (!GL_LoadShader (shaderNum, GL_VERTEX_SHADER)) return;
	if (!GL_LoadShader (shaderNum, GL_FRAGMENT_SHADER)) return;

	// create a program object
	GL_Shaders[shaderNum].programObject = glCreateProgramObject ();

	// attach the shader objects to it
	glAttachObject (GL_Shaders[shaderNum].programObject, GL_Shaders[shaderNum].Shader[QGL_VERTEX_SHADER]);
	glAttachObject (GL_Shaders[shaderNum].programObject, GL_Shaders[shaderNum].Shader[QGL_FRAGMENT_SHADER]);

	// link the program and check for success
	glLinkProgram (GL_Shaders[shaderNum].programObject);
	glGetObjectParameteriv (GL_Shaders[shaderNum].programObject, GL_OBJECT_LINK_STATUS, &linkValidateOK);

	if (!linkValidateOK) return;

	// and validate (this is not strictly necessary but can catch any runtime bugs)
	glValidateProgram (GL_Shaders[shaderNum].programObject);
	glGetObjectParameteriv (GL_Shaders[shaderNum].programObject, GL_OBJECT_VALIDATE_STATUS, &linkValidateOK);

	if (!linkValidateOK) return;

	// we have us a viable shader!!!
	GL_Shaders[shaderNum].ShaderOK = true;

	// we don't need the shader objects any more so we can safely destroy them
	glDeleteObject (GL_Shaders[shaderNum].Shader[QGL_VERTEX_SHADER]);
	glDeleteObject (GL_Shaders[shaderNum].Shader[QGL_FRAGMENT_SHADER]);

	Con_Printf ("Shader \"%s\" successfully loaded\n", GL_Shaders[shaderNum].shaderName);
}


void GL_DeleteShaders (void)
{
	int i;

	for (i = 0; i < TOTAL_SHADERS; i++)
	{
		// delete the program object
		glDeleteObject (GL_Shaders[i].programObject);
	}
}
