/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// texture management system
// this would probably make a nice C++ class, but I'm a dinosaur, so what the hey?


#include "quakedef.h"
#include "glext.h"
#include "jpeglib.h"
#include "png.h"
#include "resource.h"


image_t *gltextures = NULL;
image_t *gltexturesListEnd = NULL;

int caustictexture = -1;
int hazetexture = -1;

/*
==========================================================================================

					TEXTURE MANAGEMENT

==========================================================================================
*/


image_t *GL_FindImageTexnum (int texnum)
{
	image_t *mht = gltextures;

	for (; ;)
	{
		if (!mht) break;

		if (mht->texnum == texnum)
			return mht;

		mht = mht->next;
	}

	// not found
	return NULL;
}


image_t *GL_FindImageName (char *identifier)
{
	image_t *mht = gltextures;

	for (; ;)
	{
		if (!mht) break;

		if (!strcmp (identifier, mht->identifier))
			return mht;

		mht = mht->next;
	}

	// not found
	return NULL;
}


image_t *GL_AllocImage (int texnum, char *identifier)
{
	image_t *mht;

	if (texnum != -1)
	{
		mht = GL_FindImageTexnum (texnum);
		if (mht) return mht;
	}
	else if (identifier)
	{
		mht = GL_FindImageName (identifier);
		if (mht) return mht;
	}

	// set up a new one
	mht = (image_t *) malloc (sizeof (image_t));

	// add to the linked list
	mht->next = gltextures;
	gltextures = mht;

	// set the previous pointers
	if (mht->next) mht->next->prev = mht;
	mht->prev = NULL;

	return gltextures;
}


image_t *GL_CreateImage (char *identifier, int texnum, int width, int height, byte *data, qboolean mipmap, qboolean alpha)
{
	image_t *mht;

	// set up a new one
	mht = (image_t *) malloc (sizeof (image_t));

	// add to the linked list
	mht->next = gltextures;
	gltextures = mht;

	// set the previous pointers
	mht->prev = NULL;
	if (mht->next) mht->next->prev = mht;

	// fill it in
	strcpy (mht->identifier, identifier);
	mht->texnum = texnum;
	mht->width = width;
	mht->height = height;
	mht->dyndata = false;
	mht->data = data;
	mht->mipmap = mipmap;
	mht->alpha = alpha;
	mht->bpp = 1;

	return gltextures;
}


void WriteTGA (char *name, int width, int height, int bpp, byte *data);


void GL_TileTexture (unsigned int *in, unsigned int *out, int inWidth, int inHeight, int tileX, int tileY)
{
	int outPos = 0;
	int i, j, k, l;
	int Row;

	for (l = 0; l < tileY; l++)
	{
		for (k = 0, Row = 0; k < inHeight; k++, Row += inHeight)
		{
			for (j = 0; j < tileX; j++)
			{
				for (i = 0; i < inWidth; i++)
				{
					out[outPos++] = in[i + Row];
				}
			}
		}
	}
}


/*
================
GL_FindTexture
================
*/
int GL_FindTexture (char *identifier)
{
	image_t *mht = gltextures;

	for (; ;)
	{
		if (!mht) break;

		if (!strcmp (identifier, mht->identifier))
			return mht->texnum;

		mht = mht->next;
	}

	return -1;
}


void GL_TexDump_f (void)
{
	int i = 0;
	image_t *mht = gltextures;

	Con_Printf ("Dumping texes...\n");

	for (; ;)
	{
		if (!mht) break;

		i++;
		Con_Printf ("Loaded %s\n", mht->identifier);

		mht = mht->next;
	}
}


#define FILTER_SIZE 5
#define BLUR_FILTER 0
#define LIGHT_BLUR	1
#define EDGE_FILTER 2
#define EMBOSS_FILTER_LIGHT 3

float FilterMatrix[][FILTER_SIZE][FILTER_SIZE] = 
{
	// regular blur
	{
		{0, 0, 0, 0, 0},
		{0, 1, 1, 1, 0},
		{0, 1, 1, 1, 0},
		{0, 1, 1, 1, 0},
		{0, 0, 0, 0, 0},
	},
	// light blur
	{
		{0, 0, 0, 0, 0},
		{0, 1, 1, 1, 0},
		{0, 1, 4, 1, 0},
		{0, 1, 1, 1, 0},
		{0, 0, 0, 0, 0},
	},
	// find edges
	{
		{0,  0,  0,  0, 0},
		{0, -1, -1, -1, 0},
		{0, -1,  8, -1, 0},
		{0, -1, -1, -1, 0},
		{0,  0,  0,  0, 0},
	},
	// light emboss
	{
		{0,  0,  0, 0, 0},
		{0, -1, -1, 0, 0},
		{0, -1,  0, 1, 0},
		{0,  0,  1, 1, 0},
		{0,  0,  0, 0, 0},
	},
};


/*
==================
R_FilterTexture

Applies a 5 x 5 filtering matrix to the texture, then runs it through a simulated OpenGL texture environment
blend with the original data to derive a new texture.  Freaky, funky, and *f--king* *fantastic*.  You can do
reasonable enough "fake bumpmapping" with this baby...

Filtering algorithm from http://www.student.kuleuven.ac.be/~m0216922/CG/filtering.html
All credit due
==================
*/
void R_FilterTexture (int filterindex, unsigned int *data, int width, int height, float factor, float bias, qboolean greyscale, GLenum GLBlendOperator)
{
	int i;
	int x;
	int y;
	int filterX;
	int filterY;
	unsigned int *temp;

	// allocate a temp buffer
	temp = malloc (width * height * 4);

	for (x = 0; x < width; x++)
	{
		for (y = 0; y < height; y++)
		{
			float rgbFloat[3] = {0, 0, 0};

			// write to temp - first, write data in (to get the alpha channel quickly and 
			// easily, which will be left well alone by this particular operation...!)
			temp[y * width + x] = data[y * width + x];

			// don't mess with alpha
			if (((byte *) &data[y * width + x])[3] == 0) continue;

			for (filterX = 0; filterX < FILTER_SIZE; filterX++)
			{
				for (filterY = 0; filterY < FILTER_SIZE; filterY++)
				{
					int imageX = (x - (FILTER_SIZE / 2) + filterX + width) % width;
					int imageY = (y - (FILTER_SIZE / 2) + filterY + height) % height;

					// casting's a unary operation anyway, so the othermost set of brackets in the left part
					// of the rvalue should not be necessary... but i'm paranoid when it comes to C...
					rgbFloat[0] += ((float) ((byte *) &data[imageY * width + imageX])[0]) * FilterMatrix[filterindex][filterX][filterY];
					rgbFloat[1] += ((float) ((byte *) &data[imageY * width + imageX])[1]) * FilterMatrix[filterindex][filterX][filterY];
					rgbFloat[2] += ((float) ((byte *) &data[imageY * width + imageX])[2]) * FilterMatrix[filterindex][filterX][filterY];
				}
			}

			// multiply by factor, add bias, and clamp
			for (i = 0; i < 3; i++)
			{
				rgbFloat[i] *= factor;
				rgbFloat[i] += bias;

				if (rgbFloat[i] < 0) rgbFloat[i] = 0;
				if (rgbFloat[i] > 255) rgbFloat[i] = 255;
			}

			if (greyscale)
			{
				// NTSC greyscale conversion standard
				float avg = (rgbFloat[0] * 30 + rgbFloat[1] * 59 + rgbFloat[2] * 11) / 100;

				// divide by 255 so GL operations work as expected
				rgbFloat[0] = avg * ONEOVER255;
				rgbFloat[1] = avg * ONEOVER255;
				rgbFloat[2] = avg * ONEOVER255;
			}

			// now write in each element, applying the blend operator.  blend
			// operators are based on standard OpenGL TexEnv modes, and the
			// formulae are derived from the OpenGL specs (http://www.opengl.org).
			for (i = 0; i < 3; i++)
			{
				// divide by 255 so GL operations work as expected
				float TempTarget;
				float SrcData = ((float) ((byte *) &data[y * width + x])[i]) * ONEOVER255;

				switch (GLBlendOperator)
				{
				case GL_ADD:
					TempTarget = rgbFloat[i] + SrcData;
					break;

				case GL_BLEND:
					// default is FUNC_ADD here
					// CsS + CdD works out as Src * Dst * 2
					TempTarget = rgbFloat[i] * SrcData * 2.0;
					break;

				case GL_DECAL:
					// same as GL_REPLACE unless there's alpha, which we ignore for this
				case GL_REPLACE:
					TempTarget = rgbFloat[i];
					break;

				case GL_ADD_SIGNED:
					TempTarget = (rgbFloat[i] + SrcData) - 0.5;
					break;

				case GL_MODULATE:
					// same as default
				default:
					TempTarget = rgbFloat[i] * SrcData;
					break;
				}

				// multiply back by 255 to get the proper byte scale
				TempTarget *= 255.0;

				// bound the temp target now, cos the operation may have thrown it out
				if (TempTarget < 0) TempTarget = 0;
				if (TempTarget > 255) TempTarget = 255;

				// and copy it in
				((byte *) &temp[y * width + x])[i] = (byte) TempTarget;
			}
		}
	}

	// copy temp back to data
	for (i = 0; i < (width * height); i++)
	{
		data[i] = temp[i];
	}

	// release the temp buffer
	free (temp);
}


/*
===============
BuildIntensityTable

Got this from Quake II and if you don't like it it's just too bad
===============
*/
// we have a gamma table in Quake 1 already
static byte intensitytable[256];
static byte intensitytable2d[256];

void BuildIntensityTable (void)
{
	int i, j;

	for (i = 0; i < 256; i++)
	{
		// intensity table for world texes
		j = i * 2;
		if (j > 255) j = 255;

		intensitytable[i] = j;

		// intensity table for 2d view texes
		j = (i * 5) / 4;
		if (j > 255) j = 255;

		intensitytable2d[i] = j;
	}
}


/*
===============
GL_LightScaleTexture

Got this from Quake II and if you don't like it it's just too bad
===============
*/
void GL_LightScaleTexture (unsigned *texin, int inwidth, int inheight)
{
	int	i, c;
	byte *p;
	int avg;

	p = (byte *) texin;

	c = inwidth * inheight;

	for (i = 0; i < c; i++, p += 4)
	{
		p[0] = gammatable[intensitytable[p[0]]];
		p[1] = gammatable[intensitytable[p[1]]];
		p[2] = gammatable[intensitytable[p[2]]];

		continue;

		// experimental - average them towards grey to decrease over-colouring (looks worse...)
		avg = (p[0] + p[1] + p[2]) / 3;

		p[0] = (avg + p[0] * 2) / 3;
		p[1] = (avg + p[1] * 2) / 3;
		p[2] = (avg + p[2] * 2) / 3;
	}
}


/*
===============
GL_LightScale2DTexture

Similar to above but used for non-mipped texes, gives a less extreme scaling
===============
*/
void GL_LightScale2DTexture (unsigned *texin, int inwidth, int inheight)
{
	int	i, c;
	byte *p;

	p = (byte *) texin;

	c = inwidth * inheight;

	for (i = 0; i < c; i++, p += 4)
	{
		p[0] = gammatable[intensitytable2d[p[0]]];
		p[1] = gammatable[intensitytable2d[p[1]]];
		p[2] = gammatable[intensitytable2d[p[2]]];
	}
}


/*
===============
GL_ResampleTexture

Got this from Quake II, based on the assumption that the Quake II version of it *must*
necessarily be better (dunno if it actually is, can't really see much difference)
===============
*/
void GL_ResampleTexture (unsigned *texIn, int inwidth, int inheight, unsigned *texOut, int outwidth, int outheight)
{
	int i, j;
	unsigned *inrow, *inrow2;
	unsigned theFrac, fracstep;
	unsigned *p1;
	unsigned *p2;
	byte *pix1, *pix2, *pix3, *pix4;

	if (inwidth == outwidth && inheight == outheight)
	{
		// do a straight copy
		for (i = 0; i < inwidth * inheight; i++)
			texOut[i] = texIn[i];

		return;
	}

	p1 = (unsigned *) malloc (sizeof (unsigned) * outwidth);
	p2 = (unsigned *) malloc (sizeof (unsigned) * outwidth);

	fracstep = inwidth * 0x10000 / outwidth;
	theFrac = fracstep >> 2;

	for (i = 0; i < outwidth; i++)
	{
		p1[i] = 4 * (theFrac >> 16);
		theFrac += fracstep;
	}

	theFrac = 3 * (fracstep >> 2);

	for (i = 0; i < outwidth; i++)
	{
		p2[i] = 4 * (theFrac >> 16);
		theFrac += fracstep;
	}

	for (i = 0; i < outheight; i++, texOut += outwidth)
	{
		inrow = texIn + inwidth * (int) ((i + 0.25) * inheight / outheight);
		inrow2 = texIn + inwidth * (int) ((i + 0.75) * inheight / outheight);

		// this isn't actually *used* any more
		theFrac = fracstep >> 1;

		for (j = 0; j < outwidth; j++)
		{
			pix1 = (byte *) inrow + p1[j];
			pix2 = (byte *) inrow + p2[j];
			pix3 = (byte *) inrow2 + p1[j];
			pix4 = (byte *) inrow2 + p2[j];

			((byte *) (texOut + j))[0] = (pix1[0] + pix2[0] + pix3[0] + pix4[0]) >> 2;
			((byte *) (texOut + j))[1] = (pix1[1] + pix2[1] + pix3[1] + pix4[1]) >> 2;
			((byte *) (texOut + j))[2] = (pix1[2] + pix2[2] + pix3[2] + pix4[2]) >> 2;
			((byte *) (texOut + j))[3] = (pix1[3] + pix2[3] + pix3[3] + pix4[3]) >> 2;
		}
	}

	free (p1);
	free (p2);
}


// maximum texture size
extern int max_size;


/*
===============
ColorDiff

Get the difference between 2 colours.  This is accomplished by first converting to
greyscale, then comparing and subtracting.  There are better ways of doing this, but
greyscaling gives the most appropriate visual results for Quake textures.  The
alpha component is generally ignored.
===============
*/
unsigned int ColorDiff (unsigned int p1, unsigned int p2)
{
	byte *rgba1 = (byte *) &p1;
	byte *rgba2 = (byte *) &p2;

	int avg1;
	int avg2;

	// NTSC greyscale conversion standard
	avg1 = ((int) rgba1[0] * 30 + (int) rgba1[1] * 59 + (int) rgba1[2] * 11) / 300;
	avg2 = ((int) rgba2[0] * 30 + (int) rgba2[1] * 59 + (int) rgba2[2] * 11) / 300;

	if (avg1 > avg2)
		return avg1 - avg2;
	else return avg2 - avg1;
}


/*
===============
ColorAvg

Get the average of 2 colours, including alpha component
===============
*/
unsigned int ColorAvg (unsigned int p1, unsigned int p2)
{
	byte *rgba1 = (byte *) &p1;
	byte *rgba2 = (byte *) &p2;
	byte *rgbaavg;
	unsigned int avg;

	rgbaavg = (byte *) &avg;

	rgbaavg[0] = (byte) (((int) rgba1[0] + (int) rgba2[0]) / 2);
	rgbaavg[1] = (byte) (((int) rgba1[1] + (int) rgba2[1]) / 2);
	rgbaavg[2] = (byte) (((int) rgba1[2] + (int) rgba2[2]) / 2);
	rgbaavg[3] = (byte) (((int) rgba1[3] + (int) rgba2[3]) / 2);

	return avg;
}


/*
===============
ChooseLerp

Choose an interpolation function based on the lowest difference (i.e. which colour is
nearest to the original pixel): "colormin" is a confusing name, really...
===============
*/
unsigned int ChooseLerp (unsigned int me, unsigned int row, unsigned int col, unsigned int diag)
{
	unsigned int d1;
	unsigned int d2;
	unsigned int d3;
	unsigned int d4;
	unsigned int colormin;

	d1 = ColorDiff (me, row);
	d2 = ColorDiff (me, col);
	d3 = ColorDiff (me, diag);
	d4 = ColorDiff (row, col);

	colormin = d1;

	if (colormin > d2) colormin = d2;
	if (colormin > d3) colormin = d3;
	if (colormin > d4) colormin = d4;

	if (colormin == d1)
		return ColorAvg (me, row);
	else if (colormin == d2)
		return ColorAvg (me, col);
	else if (colormin == d3)
		return ColorAvg (me, diag);
	else return ColorAvg (me, ColorAvg (row, col));
}


/*
=================================
GL_InterpolatePixels

directional average scaling algorithm from http://www.compuphase.com/graphic/scale2.htm

this is edge-based, so it's very good at bringing out detail in complex textures when viewed
close up.  gives sharper results than bilinear or bicubic, without the "spotty" appearance of
my own algorithm (removed).  it's tolerant of multiple passes too.

the paper at the url above discusses use of bit interleaving to determine colour differences.
i've abandoned that as it gives undesirable results with quake textures (likely the fault of
quake textures rather than anything in the paper).  instead i'm using option 2: converting to
greyscale.  the speed implications mentioned in this paper don't really apply to quake as the 
calculations don't occur at run time and the bottleneck is elsewhere (OpenGL texture uploading).
=================================
*/
void GL_InterpolatePixels (unsigned int *scaled, int w, int h, qboolean alpha)
{
	int hcount;
	int wcount;

	int x;

	unsigned *temp = (unsigned *) malloc (sizeof (unsigned) * w * h * 4);

	// row up, col left, row down, col right, same row, same col
	int pos[6];

	int avg[4];	// rgba

	for (hcount = 0; hcount < h; hcount++)
	{
		// determine the positions of the 8 surrounding pixels - part one: rows
		// row up
		pos[0] = hcount - 1;
		if (pos[0] < 0) pos[0] += h;

		// row down
		pos[2] = hcount + 1;
		if (pos[2] >= h) pos[2] -= h;

		// same row
		pos[4] = hcount;

		for (wcount = 0; wcount < w; wcount++)
		{
			// determine the positions of the 8 surrounding pixels - part two: columns
			// col left
			pos[1] = wcount - 1;
			if (pos[1] < 0) pos[1] += w;

			// col right
			pos[3] = wcount + 1;
			if (pos[3] >= w) pos[3] -= w;

			// same col
			pos[5] = wcount;

			// each pixel in scaled maps to 4 pixels in temp
			temp[hcount * 2 * w * 2 + wcount * 2] = ChooseLerp (scaled[pos[4] * w + pos[5]], scaled[pos[0] * w + pos[5]], scaled[pos[4] * w + pos[1]], scaled[pos[0] * w + pos[1]]);
			temp[hcount * 2 * w * 2 + wcount * 2 + 1] = ChooseLerp (scaled[pos[4] * w + pos[5]], scaled[pos[0] * w + pos[5]], scaled[pos[4] * w + pos[3]], scaled[pos[0] * w + pos[3]]);
			temp[(hcount * 2 + 1) * w * 2 + wcount * 2] = ChooseLerp (scaled[pos[4] * w + pos[5]], scaled[pos[2] * w + pos[5]], scaled[pos[4] * w + pos[1]], scaled[pos[2] * w + pos[1]]);
			temp[(hcount * 2 + 1) * w * 2 + wcount * 2 + 1] = ChooseLerp (scaled[pos[4] * w + pos[5]], scaled[pos[2] * w + pos[5]], scaled[pos[4] * w + pos[3]], scaled[pos[2] * w + pos[3]]);
		}
	}

	// copy temp back to scaled
	for (hcount = 0, x = 0; hcount < h * 2; hcount++)
	{
		for (wcount = 0; wcount < w * 2; wcount++, x++)
		{
			scaled[x] = temp[hcount * (w * 2) + wcount];
		}
	}

	free (temp);
}


/*
===============
GL_Upload32

We make sure in advance that all 2D images coming in here are padded with alpha to make them
a power of 2 and their s and t coords modified accordingly (only if they aren't already a
power of 2) - 2D images are less tolerant of resampling, etc, than the 3D stuff is.

Anything that is not a power of 2 is resampled *up* to the next power of 2, then capped at
either 1024 * 1024 or the max texture size, whichever is lower.  LINEAR_MIPMAP_LINEAR filtering
is always used so as to avoid abrubt transitions between miplevels in the world.  A GL_RGBA 
internal format is preferred for everything - using the full 32 bits should always be faster 
than having to unpack texture data before use.  Even the lowliest budget "price of a pack of 
fags" card has 32MB these days so memory is not a consideration.

Mipmapped textures are lightscaled in advance of uploading to increase the contrast range
and thereby give the illusion that our crappy 8-bit sources are something a little better!!!
===============
*/
qboolean lerptex = false;
qboolean lightscale = true;
qboolean emboss = false;

void GL_Upload32 (unsigned *data, int width, int height, qboolean mipmap, qboolean alpha)
{
	int w;
	int h;
	unsigned int *scaled;

	// get the power of 2 equal to or above the texture size
	for (w = 1; w < width; w *= 2);
	for (h = 1; h < height; h *= 2);

	// cap at max texture size
	if (w > max_size) w = max_size;
	if (h > max_size) h = max_size;

	// need to increase the size of this buffer cos we could be interpolating it upwards
	scaled = (unsigned *) malloc (sizeof (unsigned) * w * h * 4);

	// either scale it to the new size or do a straight copy
	GL_ResampleTexture (data, width, height, scaled, w, h);

	if (lerptex)
	{
		GL_InterpolatePixels (scaled, w, h, alpha);

		w *= 2;
		h *= 2;

		lerptex = false;
	}

	if (mipmap)
	{
		// At this stage we get to make an "emboss map".  This is *NOT* PROPER BUMPMAPPING.  Proper Bumpmapping
		// relies on stuff like normals and light vectors and other doo-dahs.  It *CAN* be done using a similar
		// texture to the one I'm about to generate, but you do need light positions for it.  This is an evil,
		// quick, and cheap hack which (hopefully) gives a "don't look too close and it's OK" effect.  Like the
		// emboss filter on your favourite paint app.  So... let's go for broke...
		if (emboss)
		{
			R_FilterTexture (EMBOSS_FILTER_LIGHT, scaled, w, h, 1, 208, true, GL_MODULATE);
			emboss = false;
		}

		if (lightscale) GL_LightScaleTexture (scaled, w, h);

		// anisotropic filtering
		if (gl_anisotropicfilter.value > 1.0)
			glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY, gl_anisotropicfilter.value);

		// hardware mipmapping (OpenGL 1.4 or higher an absolute must)
		glTexParameteri (GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);

		glTexImage2D (GL_TEXTURE_2D, 
					  0, 
					  GL_RGBA, 
					  w, 
					  h, 
					  0, 
					  GL_RGBA, 
					  GL_UNSIGNED_BYTE, 
					  scaled);

		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	}
	else
	{
		if (lightscale) GL_LightScale2DTexture (scaled, w, h);

		// don;t wanna mipmap these babies
		glTexParameteri (GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_FALSE);

		glTexImage2D (GL_TEXTURE_2D, 
					  0, 
					  GL_RGBA, 
					  w, 
					  h, 
					  0, 
					  GL_RGBA, 
					  GL_UNSIGNED_BYTE, 
					  scaled);

		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	}

	free (scaled);
}


/*
====================
GL_Upload32Simple

Simplified version of regular GL_Upload32 for resource bitmap textures (which don't get the fancy stuff applied)
====================
*/
void GL_Upload32Simple (unsigned *data, int width, int height, qboolean mipmap, qboolean alpha)
{
	int w;
	int h;
	unsigned int *scaled;

	// get the power of 2 equal to or above the texture size
	for (w = 1; w < width; w *= 2);
	for (h = 1; h < height; h *= 2);

	// cap at max texture size
	if (w > max_size) w = max_size;
	if (h > max_size) h = max_size;

	// need to increase the size of this buffer cos we could be interpolating it upwards
	scaled = (unsigned *) malloc (sizeof (unsigned) * w * h * 4);

	// either scale it to the new size or do a straight copy
	GL_ResampleTexture (data, width, height, scaled, w, h);

	if (mipmap)
	{
		// anisotropic filtering
		if (gl_anisotropicfilter.value > 1.0)
			glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY, gl_anisotropicfilter.value);

		// hardware mipmapping (OpenGL 1.4 or higher an absolute must)
		glTexParameteri (GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);

		glTexImage2D (GL_TEXTURE_2D, 
					  0, 
					  GL_RGBA, 
					  w, 
					  h, 
					  0, 
					  GL_RGBA, 
					  GL_UNSIGNED_BYTE, 
					  scaled);

		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	}
	else
	{
		// don;t wanna mipmap these babies
		glTexParameteri (GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_FALSE);

		glTexImage2D (GL_TEXTURE_2D, 
					  0, 
					  GL_RGBA, 
					  w, 
					  h, 
					  0, 
					  GL_RGBA, 
					  GL_UNSIGNED_BYTE, 
					  scaled);

		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	}

	free (scaled);
}


qboolean LoadPNG (FILE *fin, qboolean mipmap)
{
	byte *tex_rgba;
	byte		header[8], **rowpointers;
	png_structp	png;
	png_infop	pnginfo;
	int		y, width, height, bitdepth, colortype;
	int		interlace, compression, filter, bytesperpixel;
	unsigned long	rowbytes;
	qboolean alpha = false;

	fread (header, 1, 8, fin);

	if (png_sig_cmp(header, 0, 8))
		goto badPNG;

	if (!(png = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL)))
		goto badPNG;

	if (!(pnginfo = png_create_info_struct(png)))
	{
		png_destroy_read_struct (&png, &pnginfo, NULL);
		goto badPNG;
	}

	if (setjmp(png->jmpbuf))
	{
		png_destroy_read_struct (&png, &pnginfo, NULL);
		goto badPNG;
	}

	png_init_io (png, fin);
	png_set_sig_bytes (png, 8);
	png_read_info (png, pnginfo);
	png_get_IHDR (png, pnginfo, (png_uint_32 *)&width, (png_uint_32 *)&height, &bitdepth, &colortype, &interlace, &compression, &filter);

	if (colortype == PNG_COLOR_TYPE_PALETTE)
	{
		png_set_palette_to_rgb (png);
		png_set_filler (png, 255, PNG_FILLER_AFTER);			
	}

	if (colortype == PNG_COLOR_TYPE_GRAY && bitdepth < 8)
		png_set_gray_1_2_4_to_8 (png);
	
	if (png_get_valid(png, pnginfo, PNG_INFO_tRNS))
	{
		png_set_tRNS_to_alpha (png);
		alpha = true;
	}

	if (colortype == PNG_COLOR_TYPE_GRAY || colortype == PNG_COLOR_TYPE_GRAY_ALPHA)
	{
		png_set_gray_to_rgb (png);
		png_set_filler (png, 255, PNG_FILLER_AFTER);			
	}

	if (colortype != PNG_COLOR_TYPE_RGBA)
		png_set_filler (png, 255, PNG_FILLER_AFTER);
	
	if (bitdepth < 8)
		png_set_expand (png);
	else if (bitdepth == 16)
		png_set_strip_16 (png);

	png_read_update_info (png, pnginfo);
	rowbytes = png_get_rowbytes (png, pnginfo);
	bytesperpixel = png_get_channels (png, pnginfo);
	bitdepth = png_get_bit_depth (png, pnginfo);

	if (bitdepth != 8 || bytesperpixel != 4)
	{	
		png_destroy_read_struct (&png, &pnginfo, NULL);
		goto badPNG;
	}

	tex_rgba = malloc (height * rowbytes);
	rowpointers = malloc (height * 4);

	for (y = 0; y < height; y++)
		rowpointers[y] = tex_rgba + y * rowbytes;

	png_read_image (png, rowpointers);
	png_read_end (png, NULL);

	png_destroy_read_struct (&png, &pnginfo, NULL);
	free (rowpointers);

	// don't interpolate or lightscale these
	lerptex = false;
	lightscale = false;

	GL_Upload32 ((unsigned int *) tex_rgba, width, height, mipmap, alpha);

	free (tex_rgba);
	fclose (fin);

	return true;

badPNG:;
	fclose (fin);
	return false;
}


typedef struct TGAHeader_s
{
	byte id_length;
	byte colormap_type;
	byte image_type;

	unsigned short colormap_index;
	unsigned short colormap_length;

	byte colormap_size;

	unsigned short x_origin;
	unsigned short y_origin;
	unsigned short width;
	unsigned short height;

	byte pixel_size;
	byte attributes;
} TGAHeader_t;


TGAHeader_t targa_header;


int fgetLittleShort (FILE *f)
{
	byte b1 = fgetc (f);
	byte b2 = fgetc (f);

	return (short) (b1 + b2 * 256);
}


int fgetLittleLong (FILE *f)
{
	byte b1 = fgetc (f);
	byte b2 = fgetc (f);
	byte b3 = fgetc (f);
	byte b4 = fgetc (f);

	return b1 + (b2 << 8) + (b3 << 16) + (b4 << 24);
}


qboolean LoadTGA (FILE *fin, qboolean mipmap)
{
	byte *tex_rgba;
	int columns;
	int rows;
	int numPixels;
	byte *pixbuf;
	int row, column;
	qboolean alpha;
	qboolean flipper;

	targa_header.id_length = fgetc (fin);
	targa_header.colormap_type = fgetc (fin);
	targa_header.image_type = fgetc (fin);

	targa_header.colormap_index = fgetLittleShort (fin);
	targa_header.colormap_length = fgetLittleShort (fin);
	targa_header.colormap_size = fgetc (fin);
	targa_header.x_origin = fgetLittleShort (fin);
	targa_header.y_origin = fgetLittleShort (fin);
	targa_header.width = fgetLittleShort (fin);
	targa_header.height = fgetLittleShort (fin);
	targa_header.pixel_size = fgetc (fin);
	targa_header.attributes = fgetc (fin);

	// no-one you see, is smarter than he.  thanks to shadow for the heads up on this.
	flipper = !((targa_header.attributes & 0x20) >> 5);

	if (targa_header.image_type != 2 && targa_header.image_type != 10)
		goto badTGA;

	if (targa_header.colormap_type != 0 || (targa_header.pixel_size != 32 && targa_header.pixel_size != 24))
		goto badTGA;

	// BPP other than 24 or 32 should not happen with TGAs!!!
	// (unless the file is corrupt...)
	if (targa_header.pixel_size == 24)
		alpha = false;
	else if (targa_header.pixel_size == 24)
		alpha = false;
	else goto badTGA;

	columns = targa_header.width;
	rows = targa_header.height;
	numPixels = columns * rows;

	tex_rgba = malloc (numPixels * 4);

	// skip TARGA image comment
	if (targa_header.id_length != 0) fseek (fin, targa_header.id_length, SEEK_CUR);
	
	if (targa_header.image_type == 2)
	{
		// Uncompressed, RGB images
		for (row = rows - 1; row >= 0; row--)
		{
			if (flipper)
				pixbuf = tex_rgba + row * columns * 4;
			else pixbuf = tex_rgba + ((row - 1) - row) * columns * 4;

			for (column = 0; column < columns; column++)
			{
				byte blue = getc (fin);
				byte green = getc (fin);
				byte red = getc (fin);
				byte alphabyte;

				*pixbuf++ = red;
				*pixbuf++ = green;
				*pixbuf++ = blue;

				switch (targa_header.pixel_size)
				{
				case 24:
					*pixbuf++ = 255;
					break;

				case 32:
					alphabyte = getc (fin);
					*pixbuf++ = alphabyte;
					break;
				}
			}
		}
	}
	else if (targa_header.image_type == 10)
	{
		// Runlength encoded RGB images
		byte red;
		byte green;
		byte blue;
		byte alphabyte;
		byte packetHeader;
		byte packetSize;
		byte j;

		for (row = rows - 1; row >= 0; row--)
		{
			if (flipper)
				pixbuf = tex_rgba + row * columns * 4;
			else pixbuf = tex_rgba + ((row - 1) - row) * columns * 4;

			for (column = 0; column < columns;)
			{
				packetHeader = getc (fin);
				packetSize = 1 + (packetHeader & 0x7f);

				if (packetHeader & 0x80)
				{
					// run-length packet
					blue = getc (fin);
					green = getc (fin);
					red = getc (fin);

					switch (targa_header.pixel_size)
					{
					case 24:
						alphabyte = 255;
						break;

					case 32:
						alphabyte = getc (fin);
						break;
					}
	
					for (j = 0; j < packetSize; j++)
					{
						*pixbuf++ = red;
						*pixbuf++ = green;
						*pixbuf++ = blue;
						*pixbuf++ = alphabyte;

						column++;

						if (column == columns)
						{
							// run spans across rows
							column = 0;

							if (row > 0)
								row--;
							else goto breakOut;

							if (flipper)
								pixbuf = tex_rgba + row * columns * 4;
							else pixbuf = tex_rgba + ((row - 1) - row) * columns * 4;
						}
					}
				}
				else
				{
					// non run-length packet
					for (j = 0; j < packetSize; j++)
					{
						blue = getc (fin);
						green = getc (fin);
						red = getc (fin);

						*pixbuf++ = red;
						*pixbuf++ = green;
						*pixbuf++ = blue;

						switch (targa_header.pixel_size)
						{
						case 24:
							*pixbuf++ = 255;
							break;

						case 32:
							alphabyte = getc (fin);
							*pixbuf++ = alphabyte;
							break;
						}

						column++;

						if (column == columns)
						{
							// pixel packet run spans across rows
							column = 0;

							if (row > 0)
								row--;
							else goto breakOut;

							if (flipper)
								pixbuf = tex_rgba + row * columns * 4;
							else pixbuf = tex_rgba + ((row - 1) - row) * columns * 4;
						}
					}
				}
			}
			breakOut:;
		}
	}

	// don't interpolate or lightscale these
	lerptex = false;
	lightscale = false;

	// only 32 bit TGA files can have alpha
	GL_Upload32 ((unsigned int *) tex_rgba, targa_header.width, targa_header.height, mipmap, alpha);

	free (tex_rgba);
	fclose (fin);

	return true;

badTGA:;
	fclose (fin);
	return false;
}


qboolean LoadBMP (FILE *f, qboolean mipmap)
{
	byte *tex_rgba;
	int i, j;
	int run;
	int runlength;
	byte bmIn;
	BITMAPINFO bmInfo;
	BITMAPFILEHEADER bmHeader;
	RGBQUAD bmPalette[256];
	byte *bmData;

	// read the file header
	fread (&bmHeader, 1, sizeof (bmHeader), f);

	// validate the type
	if (bmHeader.bfType != 0x4D42) goto badBMP;

	// read the BITMAPINFO
	fread (&bmInfo, 1, sizeof (bmInfo), f);

	if (bmInfo.bmiHeader.biCompression != BI_RGB)
	{
		if (!(bmInfo.bmiHeader.biCompression == BI_RLE8 && bmInfo.bmiHeader.biBitCount == 8))
			goto badBMP;
	}

	// too big
	if (bmInfo.bmiHeader.biWidth > 1024) goto badBMP;
	if (bmInfo.bmiHeader.biHeight > 1024 || bmInfo.bmiHeader.biHeight < -1024) goto badBMP;

	// support 4, 8 and 24 BPP modes
	if (bmInfo.bmiHeader.biBitCount != 4 && 
		bmInfo.bmiHeader.biBitCount != 8 && 
		bmInfo.bmiHeader.biBitCount != 24) goto badBMP;

	tex_rgba = malloc (bmInfo.bmiHeader.biWidth * abs (bmInfo.bmiHeader.biHeight) * 4);
	bmData = tex_rgba;

	if (bmInfo.bmiHeader.biBitCount != 24)
	{
		// read the palette entries
		// the first one is in the BITMAPINFO struct
		bmPalette[0].rgbBlue = bmInfo.bmiColors[0].rgbBlue;
		bmPalette[0].rgbGreen = bmInfo.bmiColors[0].rgbGreen;
		bmPalette[0].rgbRed = bmInfo.bmiColors[0].rgbRed;
		bmPalette[0].rgbReserved = bmInfo.bmiColors[0].rgbReserved;

		// remaining entries follow the struct
		for (i = 1; i < bmInfo.bmiHeader.biClrUsed; i++)
		{
			fread (&bmPalette[i], 1, sizeof (RGBQUAD), f);
		}
	}

	// load the data
	for (i = 0; i < bmInfo.bmiHeader.biWidth * abs (bmInfo.bmiHeader.biHeight); i++)
	{
		if (bmInfo.bmiHeader.biBitCount == 4)
		{
			bmIn = fgetc (f);

			bmData[0] = bmPalette[bmIn >> 4].rgbRed;
			bmData[1] = bmPalette[bmIn >> 4].rgbGreen;
			bmData[2] = bmPalette[bmIn >> 4].rgbBlue;
			bmData[3] = 255;

			bmData[4] = bmPalette[bmIn % 16].rgbRed;
			bmData[5] = bmPalette[bmIn % 16].rgbGreen;
			bmData[6] = bmPalette[bmIn % 16].rgbBlue;
			bmData[7] = 255;

			bmData += 8;

			// increment i again cos we're reading two entires at a time
			i++;

			continue;
		}
		else if (bmInfo.bmiHeader.biBitCount == 8)
		{
			if (bmInfo.bmiHeader.biCompression == BI_RLE8)
			{
				bmIn = fgetc (f);

				if (bmIn == 0)
				{
					bmIn = fgetc (f);
					i++;

					switch (bmIn)
					{
					case 0:
						// End of Line - just do nothing - trust me, it works.
						break;
					case 1:
						// End of Bitmap
						goto bmDone;
						break;
					case 2:
						// Delta
						// (tests indicate this isn't actually used... most likely cos the bitmap format
						// doesn't specify what the skipped pixels are to be filled with!!!)
						bmIn = fgetc (f);
						i++;

						bmData += bmIn * 4;

						bmIn = fgetc (f);
						i++;

						bmData += bmIn * bmInfo.bmiHeader.biWidth * 4;

						break;
					default:
						// absolute mode
						runlength = bmIn;

						for (run = 0; run < runlength; run++)
						{
							bmIn = fgetc (f);
							i++;

							bmData[0] = bmPalette[bmIn].rgbRed;
							bmData[1] = bmPalette[bmIn].rgbGreen;
							bmData[2] = bmPalette[bmIn].rgbBlue;
							bmData[3] = 255;

							bmData += 4;
						}

						if (runlength % 2)
						{
							// align on word boundaries
							bmIn = fgetc (f);
							i++;
						}

						break;
					}
				}
				else
				{
					// encoded mode
					runlength = bmIn;

					bmIn = fgetc (f);
					i++;

					for (run = 0; run < runlength; run++)
					{
						bmData[0] = bmPalette[bmIn].rgbRed;
						bmData[1] = bmPalette[bmIn].rgbGreen;
						bmData[2] = bmPalette[bmIn].rgbBlue;
						bmData[3] = 255;

						bmData += 4;
					}
				}

				continue;
			}

			bmIn = fgetc (f);

			bmData[0] = bmPalette[bmIn].rgbRed;
			bmData[1] = bmPalette[bmIn].rgbGreen;
			bmData[2] = bmPalette[bmIn].rgbBlue;
			bmData[3] = 255;

			bmData += 4;

			continue;
		}
		else	// 24 BPP - GRB format!!!  WHYYYY???
		{
			bmData[1] = fgetc (f);
			bmData[0] = fgetc (f);
			bmData[2] = fgetc (f);
			bmData[3] = 255;

			bmData += 4;
		}
	}

bmDone:;

	if (bmInfo.bmiHeader.biHeight > 0)
	{
		// bottom-up bitmap so we must invert it (sigh)
		for (i = 0; i < abs (bmInfo.bmiHeader.biHeight) / 2; i++)
		{
			for (j = 0; j < bmInfo.bmiHeader.biWidth; j++)
			{
				unsigned int srcpos = (i * bmInfo.bmiHeader.biWidth) + j;
				unsigned int dstpos = (((abs (bmInfo.bmiHeader.biHeight) - 1) - i) * bmInfo.bmiHeader.biWidth) + j;
				unsigned int tmp;

				tmp = ((unsigned int *) tex_rgba)[dstpos];
				((unsigned int *) tex_rgba)[dstpos] = ((unsigned int *) tex_rgba)[srcpos];
				((unsigned int *) tex_rgba)[srcpos] = tmp;
			}
		}
	}

	// don't interpolate or lightscale these
	lerptex = false;
	lightscale = false;

	// BMP files never have alpha
	GL_Upload32 ((unsigned int *) tex_rgba, bmInfo.bmiHeader.biWidth, abs (bmInfo.bmiHeader.biHeight), mipmap, false);

	free (tex_rgba);
	fclose (f);

	return true;

badBMP:;
	fclose (f);
	return false;
}


/*
===================
LoadBMPResource

Loads a bitmap from a resource contained in the MHQuake executable.

BMPFormat: GL_ALPHA indicates that the bitmap is a 255 colour greyscale image representing an alpha channel
	that will later be coloured by glColor* ().  GL_RGB indicates that the bitmap is a colour bitmap.  An
	alpha channel bitmap that contains colours will have the colours averaged and written into the alpha
	channel.  Anything else is invalid (changed to GL_RGB for graceful fallback).
BMPResourceID: The ID number of the resource (normally IDB_BITMAPx) found in MHQuake.rc and resource.h - if
	the bitmap is not found MHQuake will just load it as all-black.
glTextureNum: GL texture_extension_number for the bitmap.
mipmap: should the bitmap be mipmapped?
===================
*/
void WriteToAlpha (byte *RGBABytes)
{
	// the alpha channel is the average of the colour channels
	RGBABytes[3] = (byte) (((int) RGBABytes[0] + (int) RGBABytes[1] + (int) RGBABytes[2]) / 3);

	// set the colour channels to 255
	RGBABytes[0] = 255;
	RGBABytes[1] = 255;
	RGBABytes[2] = 255;
}


// only bitmaps can be loaded from resources, cos versions of Windows prior to XP don't support other
// image formats in resources
qboolean LoadBMPResource (GLenum BMPFormat, int BMPResourceID, int glTextureNum, qboolean mipmap, qboolean alpha)
{
	byte *tex_rgba;
	int i, j;
	int run;
	int runlength;
	byte *bmIn;
	BITMAPINFO *bmInfo;
	BITMAPFILEHEADER bmHeader;
	RGBQUAD *PaletteIn;
	RGBQUAD bmPalette[256];
	HINSTANCE hInstance;
	HANDLE hBitmap;
	byte *bmData;
	byte *bmRaw;
	int bmPos;

	if (BMPFormat != GL_ALPHA && BMPFormat != GL_RGBA)
		BMPFormat = GL_RGB;

	// get a handle to the MHQuake executable
	hInstance = GetModuleHandle (NULL);

	// load the resource
	hBitmap = LoadResource (hInstance, FindResource (hInstance, MAKEINTRESOURCE (BMPResourceID), RT_BITMAP));

	// lock it
	bmRaw = (byte *) LockResource (hBitmap);

	// read the header
	bmInfo = (BITMAPINFO *) bmRaw;

	// skip past the header (for 24 BPP RGB images the pixel data immediately follows, otherwise a palette
	// is next)
	bmRaw += sizeof (BITMAPINFO);

	// check for a supported compression type
	if (bmInfo->bmiHeader.biCompression != BI_RGB)
	{
		if (!(bmInfo->bmiHeader.biCompression == BI_RLE8 && bmInfo->bmiHeader.biBitCount == 8))
			return false;
	}

	// too big
	if (bmInfo->bmiHeader.biWidth > 1024) return false;
	if (bmInfo->bmiHeader.biHeight > 1024 || bmInfo->bmiHeader.biHeight < -1024) return false;

	// support 4, 8 and 24 BPP modes
	if (bmInfo->bmiHeader.biBitCount != 4 && 
		bmInfo->bmiHeader.biBitCount != 8 && 
		bmInfo->bmiHeader.biBitCount != 24) return false;

	// allocate enough memory for the texture data
	tex_rgba = malloc (bmInfo->bmiHeader.biWidth * abs (bmInfo->bmiHeader.biHeight) * 4);

	// clear to all black
	memset (tex_rgba, 0, sizeof (tex_rgba));

	// save another pointer to the data for dynamic checking
	bmData = tex_rgba;

	if (bmInfo->bmiHeader.biBitCount != 24)
	{
		// read the palette entries
		// the first one is in the BITMAPINFO struct
		bmPalette[0].rgbBlue = bmInfo->bmiColors[0].rgbBlue;
		bmPalette[0].rgbGreen = bmInfo->bmiColors[0].rgbGreen;
		bmPalette[0].rgbRed = bmInfo->bmiColors[0].rgbRed;
		bmPalette[0].rgbReserved = bmInfo->bmiColors[0].rgbReserved;

		// remaining entries follow the struct
		for (i = 1; i < bmInfo->bmiHeader.biClrUsed; i++)
		{
			PaletteIn = (RGBQUAD *) bmRaw;

			bmPalette[i].rgbBlue = PaletteIn->rgbBlue;
			bmPalette[i].rgbGreen = PaletteIn->rgbGreen;
			bmPalette[i].rgbRed = PaletteIn->rgbRed;
			bmPalette[i].rgbReserved = PaletteIn->rgbReserved;

			bmRaw += sizeof (RGBQUAD);
		}
	}

	// load the data
	for (i = 0; i < bmInfo->bmiHeader.biWidth * abs (bmInfo->bmiHeader.biHeight); i++)
	{
		if (bmInfo->bmiHeader.biBitCount == 4)
		{
			bmIn = bmRaw++;

			bmData[0] = bmPalette[*bmIn >> 4].rgbRed;
			bmData[1] = bmPalette[*bmIn >> 4].rgbGreen;
			bmData[2] = bmPalette[*bmIn >> 4].rgbBlue;
			bmData[3] = 255;

			if (BMPFormat == GL_ALPHA) WriteToAlpha (bmData);
			if (!(*bmIn >> 4) && alpha) bmData[0] = bmData[1] = bmData[2] = bmData[3] = 0;

			bmData[4] = bmPalette[*bmIn % 16].rgbRed;
			bmData[5] = bmPalette[*bmIn % 16].rgbGreen;
			bmData[6] = bmPalette[*bmIn % 16].rgbBlue;
			bmData[7] = 255;

			if (BMPFormat == GL_ALPHA) WriteToAlpha (&bmData[4]);
			if (!(*bmIn % 16) && alpha) bmData[4] = bmData[5] = bmData[6] = bmData[7] = 0;

			bmData += 8;

			// increment i again cos we're reading two entires at a time
			i++;

			continue;
		}
		else if (bmInfo->bmiHeader.biBitCount == 8)
		{
			if (bmInfo->bmiHeader.biCompression == BI_RLE8)
			{
				bmIn = bmRaw++;

				if (*bmIn == 0)
				{
					bmIn = bmRaw++;
					i++;

					switch (*bmIn)
					{
					case 0:
						// End of Line - just do nothing - trust me, it works.
						break;
					case 1:
						// End of Bitmap
						goto bmDone;
						break;
					case 2:
						// Delta
						// (tests indicate this isn't actually used... most likely cos the bitmap format
						// doesn't specify what the skipped pixels are to be filled with!!!)
						bmIn = bmRaw++;
						i++;

						bmData += *bmIn * 4;

						bmIn = bmRaw++;
						i++;

						bmData += *bmIn * bmInfo->bmiHeader.biWidth * 4;

						break;
					default:
						// absolute mode
						runlength = *bmIn;

						for (run = 0; run < runlength; run++)
						{
							bmIn = bmRaw++;
							i++;

							bmData[0] = bmPalette[*bmIn].rgbRed;
							bmData[1] = bmPalette[*bmIn].rgbGreen;
							bmData[2] = bmPalette[*bmIn].rgbBlue;
							bmData[3] = 255;

							if (BMPFormat == GL_ALPHA) WriteToAlpha (bmData);
							if (!(*bmIn) && alpha) bmData[0] = bmData[1] = bmData[2] = bmData[3] = 0;

							bmData += 4;
						}

						if (runlength % 2)
						{
							// align on word boundaries
							bmIn = bmRaw++;
							i++;
						}

						break;
					}
				}
				else
				{
					// encoded mode
					runlength = *bmIn;

					bmIn = bmRaw++;
					i++;

					for (run = 0; run < runlength; run++)
					{
						bmData[0] = bmPalette[*bmIn].rgbRed;
						bmData[1] = bmPalette[*bmIn].rgbGreen;
						bmData[2] = bmPalette[*bmIn].rgbBlue;
						bmData[3] = 255;

						if (BMPFormat == GL_ALPHA) WriteToAlpha (bmData);
						if (!(*bmIn) && alpha) bmData[0] = bmData[1] = bmData[2] = bmData[3] = 0;

						bmData += 4;
					}
				}

				continue;
			}

			bmIn = bmRaw++;

			bmData[0] = bmPalette[*bmIn].rgbRed;
			bmData[1] = bmPalette[*bmIn].rgbGreen;
			bmData[2] = bmPalette[*bmIn].rgbBlue;
			bmData[3] = 255;

			if (BMPFormat == GL_ALPHA) WriteToAlpha (bmData);
			if (!(*bmIn) && alpha) bmData[0] = bmData[1] = bmData[2] = bmData[3] = 0;

			bmData += 4;

			continue;
		}
		else	// 24 BPP - GRB format!!!  WHYYYY???
		{
			bmData[1] = *bmRaw++;
			bmData[0] = *bmRaw++;
			bmData[2] = *bmRaw++;
			bmData[3] = 255;

			if (BMPFormat == GL_ALPHA) WriteToAlpha (bmData);

			bmData += 4;
		}
	}

bmDone:;

	if (bmInfo->bmiHeader.biHeight > 0)
	{
		// bottom-up bitmap so we must invert it (sigh)
		for (i = 0; i < abs (bmInfo->bmiHeader.biHeight) / 2; i++)
		{
			for (j = 0; j < bmInfo->bmiHeader.biWidth; j++)
			{
				unsigned int srcpos = (i * bmInfo->bmiHeader.biWidth) + j;
				unsigned int dstpos = (((abs (bmInfo->bmiHeader.biHeight) - 1) - i) * bmInfo->bmiHeader.biWidth) + j;
				unsigned int tmp;

				tmp = ((unsigned int *) tex_rgba)[dstpos];
				((unsigned int *) tex_rgba)[dstpos] = ((unsigned int *) tex_rgba)[srcpos];
				((unsigned int *) tex_rgba)[srcpos] = tmp;
			}
		}
	}

	// don't interpolate or lightscale these
	lerptex = false;
	lightscale = false;

	glBindTexture (GL_TEXTURE_2D, glTextureNum);

	// BMP files never have alpha.
	// don't send this through GL_Upload32 as it's evil incarnate on these.
	// instead, we have a simplified version of it for this kinda thing
	GL_Upload32Simple ((unsigned int *) tex_rgba, bmInfo->bmiHeader.biWidth, abs (bmInfo->bmiHeader.biHeight), mipmap, alpha);

	return true;
}


qboolean LoadJPG (FILE *f, qboolean mipmap)
{
	byte *tex_rgba;
	struct jpeg_decompress_struct cinfo;
	JDIMENSION num_scanlines;
	JSAMPARRAY in;
	struct jpeg_error_mgr jerr;
	int numPixels;
	int row_stride;
	byte *out;
	int count;
	int i;
	int r, g, b;

	// set up the decompression.
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_decompress (&cinfo);

	// inititalize the source
	jpeg_stdio_src (&cinfo, f);

	// initialize decompression
	(void) jpeg_read_header (&cinfo, TRUE);
	(void) jpeg_start_decompress (&cinfo);

	numPixels = cinfo.image_width * cinfo.image_height;

	// initialize the input buffer - we'll use the in-built memory management routines in the
	// JPEG library because it will automatically free the used memory for us when we destroy
	// the decompression structure.  cool.
	row_stride = cinfo.output_width * cinfo.output_components;
	in = (*cinfo.mem->alloc_sarray) ((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);

	// bit of error checking
	if (cinfo.output_components != 3)
		goto error;

	if ((numPixels * 4) != ((row_stride * cinfo.output_height) + numPixels))
		goto error;

	// read the jpeg
	count = 0;

	// initialize the return data
	tex_rgba = malloc ((numPixels * 4));

	while (cinfo.output_scanline < cinfo.output_height) 
	{
		num_scanlines = jpeg_read_scanlines(&cinfo, in, 1);
		out = in[0];

		for (i = 0; i < row_stride;)
		{
			r = tex_rgba[count++] = out[i++];
			g = tex_rgba[count++] = out[i++];
			b = tex_rgba[count++] = out[i++];
			tex_rgba[count++] = 255;
		}
	}

	// finish decompression and destroy the jpeg
	(void) jpeg_finish_decompress (&cinfo);
	jpeg_destroy_decompress (&cinfo);

	// don't interpolate or lightscale these
	lerptex = false;
	lightscale = false;

	// JPG files never have alpha
	GL_Upload32 ((unsigned int *) tex_rgba, cinfo.image_width, cinfo.image_height, mipmap, false);

	free (tex_rgba);
	fclose (f);

	return true;

error:
	// this should rarely (if ever) happen, but just in case...
	jpeg_destroy_decompress (&cinfo);

	fclose (f);

	return false;
}


typedef struct pcx_s
{
	char manufacturer;
	char version;
	char encoding;
	char bits_per_pixel;

	unsigned short xmin;
	unsigned short ymin;
	unsigned short xmax;
	unsigned short ymax;

	unsigned short hres;
	unsigned short vres;

	byte palette[48];

	char reserved;
	char color_planes;

	unsigned short bytes_per_line;
	unsigned short palette_type;

	char filler[58];

	unsigned int data;			// unbounded
} pcx_t;


qboolean LoadPCX (FILE *f, qboolean mipmap)
{
	byte *tex_rgba;
	pcx_t *pcx, pcxbuf;
	byte palette[768];
	byte *pix;
	int x, y;
	int dataByte, runLength;
	int count;

	// parse the PCX file
	fread (&pcxbuf, 1, sizeof (pcxbuf), f);

	pcx = &pcxbuf;

	if (pcx->manufacturer != 0x0a || pcx->version != 5 || pcx->encoding != 1 || pcx->bits_per_pixel != 8 || pcx->xmax >= 1024 || pcx->ymax >= 1024)
	{
		fclose (f);
		return false;
	}

	// seek to palette
	fseek (f, -768, SEEK_END);
	fread (palette, 1, 768, f);

	fseek (f, sizeof (pcxbuf) - 4, SEEK_SET);

	count = (pcx->xmax + 1) * (pcx->ymax + 1);
	tex_rgba = malloc (count * 4);

	for (y = 0; y <= pcx->ymax; y++)
	{
		pix = tex_rgba + 4 * y * (pcx->xmax + 1);

		for (x = 0; x <= pcx->xmax;)
		{
			dataByte = fgetc (f);

			if ((dataByte & 0xC0) == 0xC0)
			{
				runLength = dataByte & 0x3F;
				dataByte = fgetc (f);
			}
			else runLength = 1;

			while(runLength-- > 0)
			{
				pix[0] = palette[dataByte * 3];
				pix[1] = palette[dataByte * 3 + 1];
				pix[2] = palette[dataByte * 3 + 2];
				pix[3] = 255;
				pix += 4;
				x++;
			}
		}
	}

	// don't interpolate or lightscale these
	lerptex = false;
	lightscale = false;

	// PCX files never have alpha
	GL_Upload32 ((unsigned int *) tex_rgba, (pcx->xmax + 1), (pcx->ymax + 1), mipmap, false);

	free (tex_rgba);
	fclose (f);

	return true;
}


qboolean GL_UploadExternalTexture (image_t *mht)
{
	int i;
	FILE *f;
	char texname[1024];

	sprintf (texname, "%s", mht->identifier);

	// replace invalid file name characters with _
	for (i = 0; i < strlen (texname); i++)
	{
		if (texname[i] == '*')
		{
			texname[i] = '_';
		}
	}

loadPNG:;
	// PNG is preferred over everything else cos of it's sheer flexibility.  compression, alpha, any
	// bit depth you care to mention, whoo-hoo!
	COM_FOpenFile (va ("textures/%s.png", texname), &f);

	if (!f) goto loadTGA;

	if (LoadPNG (f, mht->mipmap)) return true;

loadTGA:;
	// TGA is the second preferred type cos it supports much of what PNG does, although without comparable
	// compression and without the ability to use lower bit depths if appropriate
	COM_FOpenFile (va ("textures/%s.tga", texname), &f);

	if (!f) goto loadBMP;

	if (LoadTGA (f, mht->mipmap)) return true;

loadBMP:;
	// BMP is the third preferred type cos of it's multiple bit depths
	COM_FOpenFile (va ("textures/%s.bmp", texname), &f);

	if (!f) goto loadJPG;

	if (LoadBMP (f, mht->mipmap)) return true;

loadJPG:;
	// JPG is the fourth preferred type - lossy compression, yuck!
	COM_FOpenFile (va ("textures/%s.jpg", texname), &f);

	if (!f) goto loadPCX;

	if (LoadJPG (f, mht->mipmap)) return true;

loadPCX:;
	// PCX is the fourth - 8bpp only, but at least the compression is lossless (but v crude)
	COM_FOpenFile (va ("textures/%s.pcx", texname), &f);

	if (!f) return false;

	if (LoadPCX (f, mht->mipmap)) return true;

	// couldn't load anything
	return false;
}


/*
===============
GL_Upload8
===============
*/
void GL_Upload8 (byte *data, int width, int height, qboolean mipmap, qboolean alpha)
{
	unsigned *trans = (unsigned *) malloc (width * height * sizeof (unsigned));
	int i;
	int s;
	int p;

	s = width * height;
	alpha = false;

	for (i = 0; i < s; i++)
	{
		p = data[i];

		if (p == 255)
		{
			alpha = true;
			trans[i] = 0;
		}
		else trans[i] = d_8to24table[p];
	}

	GL_Upload32 (trans, width, height, mipmap, alpha);

	free (trans);
}


void GL_Upload32MHT (image_t *mht)
{
	lerptex = false;
	lightscale = false;
	GL_Upload32 ((unsigned *) mht->data, mht->width, mht->height, mht->mipmap, mht->alpha);

	mht->bpp = 4;
}


void GL_Upload8MHT (image_t *mht)
{
	GL_Upload8 (mht->data, mht->width, mht->height, mht->mipmap, mht->alpha);

	mht->bpp = 1;
}


/*
==================
GL_LoadTexture

switched the CRC to an MD5 cos it's miles more reliable for match identification.
MD5 stuff comes from RFC 1321.  I'm supposed to mention RSA stuff here, so... "RSA stuff"

seriously, all credit and yowsa to the guys for making this available
==================
*/
image_t *GL_LoadTexture (char *identifier, int width, int height, byte *data, qboolean mipmap, qboolean alpha)
{
	image_t *mht = gltextures;
	byte md5digest[17];		// 1 extra for null termination

	// make an md5 of the texture data
	MD5_Checksum (data, (width * height), md5digest);

	// NULL terminate it
	md5digest[16] = 0;

	for (; ;)
	{
		// texture does not exist
		if (!mht) break;

		// check the md5 and return the texture number if it matches
		if (MD5_Compare (md5digest, mht->md5digest))
		{
			Con_DPrintf ("Matched Texture %s\n", identifier);
			return mht;
		}

		// advance to the next texture
		mht = mht->next;
	}

	// no match so allocate a new one
	mht = GL_CreateImage (identifier[0] ? identifier : va ("TexMD5_%s", md5digest), 
						  texture_extension_number++, 
						  width, 
						  height, 
						  NULL, 
						  mipmap, 
						  alpha);

	// store the md5
	MD5_Copy (md5digest, mht->md5digest);

	// store out the texture data for ease of reloading
	mht->data = (byte *) malloc (width * height);
	memcpy (mht->data, data, (width * height));
	mht->dyndata = true;

	// bind the texture
	glBindTexture (GL_TEXTURE_2D, mht->texnum);

	// attempt to load an external texture file - if that fails, load the internal data
	if (!GL_UploadExternalTexture (mht))
		GL_Upload8MHT (mht);

	// make sure that the next one is lightscaled unless explicitly told not to
	lightscale = true;

	// return the stored texture
	return mht;
}


/*
================
GL_LoadPicTexture
================
*/
image_t *GL_LoadPicTexture (qpic_t *pic, char *picname)
{
	return GL_LoadTexture (picname, pic->width, pic->height, pic->data, false, true);
}


/*
================
FindFullbrightTexture

Determine if a given texture has fullbright components
================
*/
qboolean FindFullbrightTexture (byte *pixels, int num_pix)
{
    int i;

    for (i = 0; i < num_pix; i++)
        if (pixels[i] > 223 && pixels[i] != 255)
            return true;

    return false;
}


/*
================
ConvertPixels

Converts a textures non-fullbright components to a specified alpha colour
================
*/
void ConvertPixels (byte *pixels, int num_pixels, int alphaColour)
{
    int i;

    for (i = 0; i < num_pixels; i++)
        if (pixels[i] < 224)
            pixels[i] = alphaColour;
}



/*
================
RemoveFullBrights

Removes all fullbright components from a texture
================
*/
void RemoveFullBrights (byte *pixels, int num_pixels, int alphaColour)
{
    int i;

    for (i = 0; i < num_pixels; i++)
        if (pixels[i] > 223 && pixels[i] != 255)
            pixels[i] = alphaColour;
}



/*
================
AverageColour

Determine the average colour (expressed in RGB rather than palletted) of all colours in
a texture
================
*/
float avg[3];

void AverageColour (byte *pixels, int num_pixels)
{
    int i;
	unsigned rgba;
	int count;

	avg[0] = avg[1] = avg[2] = 0;

    for (i = 0, count = 0; i < num_pixels; i++)
	{
		// ignore alpha
		if (pixels[i] == 255) continue;

		rgba = d_8to24table[pixels[i]];
		count++;

		avg[0] += ((byte *) &rgba)[0];
		avg[1] += ((byte *) &rgba)[1];
		avg[2] += ((byte *) &rgba)[2];
	}

	if (!count) return;

	avg[0] /= count;
	avg[1] /= count;
	avg[2] /= count;
}


/*
================
GL_LoadMapTexture

Given a pre-filled in texture_t this will do the GL stuff on it
================
*/
void GL_LoadMapTexture (texture_t *tx)
{
	char fbr_mask_name[64];
	byte *temp;
	int pixels = tx->width * tx->height;

	if (tx->name[0] != '*')
	{
		// i should probably put these into a texturestate_t struct...
		lerptex = true;
		lightscale = true;
		emboss = true;
	}
	else
	{
		lerptex = false;
		lightscale = false;
	}

	// emboss will be set to false after this so any fullbright won't be embossed.
	// which is exactly what we want!!!
	tx->gl_texture = GL_LoadTexture (tx->name,
									 tx->width,
									 tx->height,
									 (byte *) (tx+1), 
									 true,
									 false);

	// explicitly set to false here cos the texture may not go through gl_upload32 at all,
	// meaning that it will still be true when it comes out of GL_LoadTexture!!!
	emboss = false;

	// check for fullbright pixels in the texture - but do not check liquid textures,
	// since they will be rendered fullbright anyway...
	if ((tx->name[0] != '*') && (FindFullbrightTexture ((byte *) (tx + 1), pixels)))
	{
		// first off, let's be nice and non-destructive to the original pixels
		temp = (byte *) malloc (tx->width * tx->height);

		memcpy ((byte *) temp, (byte *) (tx + 1), pixels);

		// convert any non-fullbright pixel to fully transparent
		ConvertPixels (temp, pixels, 255);

		// derive a new name for the texture to prevent mismatches
		sprintf (fbr_mask_name, "%s_luma", tx->name);

		// and now load it - we need alpha for the non-fullbright components
		lerptex = true;

		tx->fullbright = GL_LoadTexture (fbr_mask_name,
										 tx->width, 
										 tx->height, 
										 temp, 
										 true, 
										 true);

		free (temp);
	}
	else tx->fullbright = 0;

	// if it's a liquid texture we find an underwater fog colour for it
	if (tx->name[0] == '*')
	{
		AverageColour ((byte *) (tx + 1), pixels);

		tx->uwater_fog[0] = avg[0] * ONEOVER256;
		tx->uwater_fog[1] = avg[1] * ONEOVER256;
		tx->uwater_fog[2] = avg[2] * ONEOVER256;
		tx->uwater_fog[3] = 0.6;
	}
}


/****************************************/



void LoadResourceTextures (void)
{
	int i;

	int xHairBMPs[] = {IDB_BITMAP5, IDB_BITMAP6, IDB_BITMAP7, IDB_BITMAP8, IDB_BITMAP9, IDB_BITMAP10, IDB_BITMAP11, IDB_BITMAP12};

	if (crosshairfont == -1)
	{
		crosshairfont = texture_extension_number;
		texture_extension_number += 8;
	}

	for (i = 0; i < 8; i++)
		LoadBMPResource (GL_ALPHA, xHairBMPs[i], crosshairfont + i, false, false);

	if (logoTex == -1)
	{
		logoTex = texture_extension_number;
		texture_extension_number++;
	}

	LoadBMPResource (GL_RGB, IDB_BITMAP1, logoTex, false, true);

	if (particlefont == -1)
	{
		particlefont = texture_extension_number;
		texture_extension_number++;
	}

	LoadBMPResource (GL_ALPHA, IDB_BITMAP3, particlefont, false, false);

	if (caustictexture == -1)
	{
		caustictexture = texture_extension_number;
		texture_extension_number++;
	}

	LoadBMPResource (GL_RGB, IDB_BITMAP4, caustictexture, true, false);

	if (hazetexture == -1)
	{
		hazetexture = texture_extension_number;
		texture_extension_number++;
	}

	LoadBMPResource (GL_RGB, IDB_BITMAP2, hazetexture, true, false);
}


// after changing resolution, any texture which is current will be lost, so reload them all
void GL_ReloadLightmap (int lightmapnum);
void GL_ReloadStainmap (int stainmapnum);
void Scrap_Upload (void);
void R_ClearMenuCachePics (void);
void Sbar_Preload3DModels (void);
void Mod_ReloadAliasModel (model_t *mod);
extern model_t *conQuad;
extern qboolean SbarPreload;
extern qboolean WorldModelLoaded;

void GL_ReloadTextures (void)
{
	int i;
	image_t *mht = gltextures;

	for (; ;)
	{
		if (!mht) break;

		glBindTexture (GL_TEXTURE_2D, mht->texnum);

		if (mht->bpp == 1)
		{
			GL_Upload8MHT (mht);
		}
		else
		{
			//GL_Upload32MHT (mht);
		}

		mht = mht->next;
	}

	return;

	// do the console stuff first
	GL_LoadCharset ();

	// the MH logo - VERY important...
	LoadResourceTextures ();

	// console quad
	Mod_ReloadAliasModel (conQuad);

	// menu pics
	R_ClearMenuCachePics ();

	// scrap stuff
	Scrap_Upload ();

	// HUD models
	WorldModelLoaded = true;
	Sbar_Preload3DModels ();
	SbarPreload = false;

	// particles
	R_InitParticleTexture ();

	// map textures - if we have a map loaded, that is...!!!
	if (r_worldentity.model && cl.worldmodel)
	{
		int x;
		model_t *mod;

		for (x = 1; x < MAX_MODELS; x++)
		{
			mod = cl.model_precache[x];

			if (!mod) continue;

			switch (mod->type)
			{
			case mod_brush:
				// these will be loaded with the world
				if (mod->name[0] == '*') break;

				for (i = 0; i < mod->numtextures; i++)
				{
					if (mod->textures[i])
					{
						if (!Q_strncmp (mod->textures[i]->name,"sky",3))
						{
							R_InitSky (mod->textures[i]);
						}
						else
						{
							GL_LoadMapTexture (mod->textures[i]);
						}
					}
				}

				break;

			case mod_alias:
				Mod_ReloadAliasModel (mod);
				break;

			case mod_sprite:
				// MHQuake doesn't use sprites...!
				break;

			default:
				break;
			}
		}

		// lightmaps - the data is kept in main memory so we just reload from that
		for (i = 0; i < 64; i++)
		{
			GL_ReloadLightmap (i);
			GL_ReloadStainmap (i);
		}
	}

	// this will be fraudulently set to "true" after reloading the sbar brush models
	WorldModelLoaded = false;
}


/*
===============
GL_TexmanInit

Initialize the texture management system
===============
*/
void GL_TexmanInit (void)
{
	// set up the intensity table for texture lightscaling
	BuildIntensityTable ();

	// init the textures list
	gltextures = NULL;
}


void GL_TexmanShutDown (void)
{
	int i;
	image_t *curr = gltextures;

	// go forwards through the list
	for (;;)
	{
		// if next is NULL we're at the end of the list
		if (!curr->next) break;

		// free the data while we're here...
		if (curr->next->dyndata) free (curr->next->data);

		// and release the texture from the gl context
		glDeleteTextures (1, &curr->next->texnum);

		// set current texture to the next
		curr = curr->next;
	}

	// now go backwards through the list
	for (i = 0; ; i++)
	{
		// if previous is NULL we're at the start of the list
		if (!curr->prev) break;

		// set current texture to the previous
		curr = curr->prev;

		// free the next (which was the current at the start of the loop) unless it's NULL
		if (curr->next) free (curr->next);
	}

	// free the list beginning
	free (gltextures);

	// MessageBox (NULL, va ("Freeing %i Textures", i), "texture shutdown", 0);
}


/*
===============
GL_TexManMapLoadFinish

Does some housekeeping after loading the map
===============
*/
void GL_TexManMapLoadFinish (void)
{
}
