/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// gl_vidnt.c -- NT GL vid component

#include "quakedef.h"
#include "glext.h"
#include "winquake.h"
#include "resource.h"
#include <commctrl.h>


#define NUM_RESOLUTIONS 8

typedef struct VID_AvailableModes_s
{
	int ResWidth;
	int ResHeight;
	qboolean Res32BPP;
	qboolean ResWindowed;
	int modenum;
} VID_AvailableModes_t;


// 2 BPPs and 2 Windowed/Fullscreen for each resolution
VID_AvailableModes_t VID_AvailableModes[NUM_RESOLUTIONS * 2 * 2];


typedef struct VID_Resolutions_s
{
	int width;
	int height;
} VID_Resolutions_t;

VID_Resolutions_t VID_Resolutions[NUM_RESOLUTIONS] =
{
	{640, 480},
	{800, 600},
	{960, 720},
	{1024, 768},
	{1152, 864},
	{1280, 1024},
	{1600, 1200},
	{2048, 1536}
};


int NumAvailableModes;

void VID_EnumerateResolutions (void)
{
	int i, j, k;

	int maxWidth;
	int maxHeight;
	qboolean isWindowed;
	qboolean is32bpp;

	HWND hDesktopWnd;
	HDC hDesktopDC;

	// get the current desktop height and width
	hDesktopWnd = GetDesktopWindow ();
	hDesktopDC = GetDC (hDesktopWnd);

	maxWidth = GetDeviceCaps (hDesktopDC, HORZRES);		// width
	maxHeight = GetDeviceCaps (hDesktopDC, VERTRES);		// height

	// release the desktop device context
	ReleaseDC (hDesktopWnd, hDesktopDC);

	NumAvailableModes = 0;

	isWindowed = false;
	is32bpp = false;

	for (i = 0; i < 2; i++)
	{
		for (j = 0; j < 2; j++)
		{
			for (k = 0; k < NUM_RESOLUTIONS; k++)
			{
				// video mode too big
				if (isWindowed && (VID_Resolutions[k].width >= maxWidth || VID_Resolutions[k].height >= maxHeight)) continue;
				else if (!isWindowed && (VID_Resolutions[k].width > maxWidth || VID_Resolutions[k].height > maxHeight)) continue;

				VID_AvailableModes[NumAvailableModes].ResWidth = VID_Resolutions[k].width;
				VID_AvailableModes[NumAvailableModes].ResHeight = VID_Resolutions[k].height;

				VID_AvailableModes[NumAvailableModes].Res32BPP = is32bpp;
				VID_AvailableModes[NumAvailableModes].ResWindowed = isWindowed;
				VID_AvailableModes[NumAvailableModes].modenum = NumAvailableModes;

				NumAvailableModes++;
			}

			is32bpp = !is32bpp;
		}

		isWindowed = !isWindowed;
	}

	Con_Printf ("\nAvailable Video Modes\n");
	Con_Printf ("Width Height BPP Windowed\n");

	for (i = 0; i < NumAvailableModes; i++)
	{
		Con_Printf ("%5i %6i %3i %s\n", 
			VID_AvailableModes[i].ResWidth,
			VID_AvailableModes[i].ResHeight,
			VID_AvailableModes[i].Res32BPP ? 32 : 16,
			VID_AvailableModes[i].ResWindowed ? "Yes" : "No");
	}

	Con_Printf ("\n");
}



#define MAX_MODE_LIST	50
#define VID_ROW_SIZE	3
#define WARP_WIDTH		320
#define WARP_HEIGHT		200
#define MAXWIDTH		10000
#define MAXHEIGHT		10000
#define BASEWIDTH		320
#define BASEHEIGHT		200

#define MODE_WINDOWED			0
#define NO_MODE					(MODE_WINDOWED - 1)
#define MODE_FULLSCREEN_DEFAULT	(MODE_WINDOWED + 1)

typedef struct {
	modestate_t	type;
	int			width;
	int			height;
	int			modenum;
	int			dib;
	int			fullscreen;
	int			bpp;
	int			halfscreen;
	char		modedesc[17];
} vmode_t;

typedef struct {
	int			width;
	int			height;
} lmode_t;

lmode_t	lowresmodes[] = {
	{320, 200},
	{320, 240},
	{400, 300},
	{512, 384},
};

const char *gl_vendor;
const char *gl_renderer;
const char *gl_version;
const char *gl_extensions;
float GL_MaxAnisotropy;

qboolean		DDActive;
qboolean		scr_skipupdate;

static vmode_t	modelist[MAX_MODE_LIST];
static int		nummodes;
static vmode_t	*pcurrentmode;
static vmode_t	badmode;

static DEVMODE	gdevmode;
static qboolean	vid_initialized = false;
static qboolean	windowed, leavecurrentmode;
static qboolean vid_canalttab = false;
static qboolean vid_wassuspended = false;
static int		windowed_mouse;
extern qboolean	mouseactive;  // from in_win.c
static HICON	hIcon;

int			DIBWidth, DIBHeight;
RECT		WindowRect;
DWORD		WindowStyle, ExWindowStyle;

HWND	mainwindow, dibwindow;

int			vid_modenum = NO_MODE;
int			vid_realmode;
int			vid_default = MODE_WINDOWED;
static int	windowed_default;
unsigned char	vid_curpal[256*3];

static float vid_gamma = 1.0;

HGLRC	baseRC;
HDC		maindc;

cvar_t	gl_ztrick = {"gl_ztrick","1"};
cvar_t	gl_anisotropicfilter = {"gl_anisotropicfilter", "1", true};
cvar_t	gl_detailtexture = {"gl_detailtexture", "16", true};
cvar_t	gl_sparky = {"gl_sparky", "0", true};
cvar_t	gl_vertexlight = {"gl_vertexlight", "0", true};

HWND WINAPI InitializeWindow (HINSTANCE hInstance, int nCmdShow);


viddef_t	vid;				// global video state

unsigned	d_8to24table[256];

float		gldepthmin, gldepthmax;

modestate_t	modestate = MS_UNINIT;

void VID_MenuDraw (void);
void VID_MenuKey (int key);

LONG WINAPI MainWndProc (HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam);
void AppActivate(BOOL fActive, BOOL minimize);
char *VID_GetModeDescription (int mode);
void ClearAllStates (void);
void VID_UpdateWindowStatus (void);
void GL_Init (void);


int VIDMode_OldMode;

//====================================

cvar_t		vid_mode = {"vid_mode","0", false};
// Note that 0 is MODE_WINDOWED
cvar_t		_vid_default_mode = {"_vid_default_mode","0", true};
// Note that 3 is MODE_FULLSCREEN_DEFAULT
cvar_t		_vid_default_mode_win = {"_vid_default_mode_win","3", true};
cvar_t		vid_wait = {"vid_wait","0"};
cvar_t		vid_nopageflip = {"vid_nopageflip","0", true};
cvar_t		_vid_wait_override = {"_vid_wait_override", "0", true};
cvar_t		vid_config_x = {"vid_config_x","800", true};
cvar_t		vid_config_y = {"vid_config_y","600", true};
cvar_t		vid_stretch_by_2 = {"vid_stretch_by_2","1", true};
cvar_t		_windowed_mouse = {"_windowed_mouse","1", true};

int			window_center_x, window_center_y, window_x, window_y, window_width, window_height;
RECT		window_rect;


void CenterWindow(HWND hWndCenter, int width, int height, BOOL lefttopjustify)
{
    RECT    rect;
    int     CenterX, CenterY;

	CenterX = (GetSystemMetrics(SM_CXSCREEN) - width) / 2;
	CenterY = (GetSystemMetrics(SM_CYSCREEN) - height) / 2;
	if (CenterX > CenterY*2)
		CenterX >>= 1;	// dual screens
	CenterX = (CenterX < 0) ? 0: CenterX;
	CenterY = (CenterY < 0) ? 0: CenterY;
	SetWindowPos (hWndCenter, NULL, CenterX, CenterY, 0, 0,
			SWP_NOSIZE | SWP_NOZORDER | SWP_SHOWWINDOW | SWP_DRAWFRAME);
}

qboolean VID_SetWindowedMode (int modenum)
{
	HDC				hdc;
	int				lastmodestate, width, height;
	RECT			rect;

	lastmodestate = modestate;

	WindowRect.top = WindowRect.left = 0;

	WindowRect.right = modelist[modenum].width;
	WindowRect.bottom = modelist[modenum].height;

	DIBWidth = modelist[modenum].width;
	DIBHeight = modelist[modenum].height;

	WindowStyle = WS_OVERLAPPED | WS_BORDER | WS_CAPTION | WS_SYSMENU |
				  WS_MINIMIZEBOX;
	ExWindowStyle = 0;

	rect = WindowRect;
	AdjustWindowRectEx(&rect, WindowStyle, FALSE, 0);

	width = rect.right - rect.left;
	height = rect.bottom - rect.top;

	// Create the DIB window
	dibwindow = CreateWindowEx (
		 ExWindowStyle,
		 "WinQuake",
		 "MHQuake",
		 WindowStyle,
		 rect.left, rect.top,
		 width,
		 height,
		 NULL,
		 NULL,
		 global_hInstance,
		 NULL);

	if (!dibwindow)
		Sys_Error ("Couldn't create DIB window");

	// Center and show the DIB window
	CenterWindow(dibwindow, WindowRect.right - WindowRect.left,
				 WindowRect.bottom - WindowRect.top, false);

	ShowWindow (dibwindow, SW_SHOWDEFAULT);
	UpdateWindow (dibwindow);

	modestate = MS_WINDOWED;

// because we have set the background brush for the window to NULL
// (to avoid flickering when re-sizing the window on the desktop),
// we clear the window to black when created, otherwise it will be
// empty while Quake starts up.
	hdc = GetDC(dibwindow);
	PatBlt(hdc,0,0,WindowRect.right,WindowRect.bottom,BLACKNESS);
	ReleaseDC(dibwindow, hdc);

	if (vid.conheight > modelist[modenum].height)
		vid.conheight = modelist[modenum].height;
	if (vid.conwidth > modelist[modenum].width)
		vid.conwidth = modelist[modenum].width;
	vid.width = vid.conwidth;
	vid.height = vid.conheight;

	vid.numpages = 2;

	mainwindow = dibwindow;

	SendMessage (mainwindow, WM_SETICON, (WPARAM)TRUE, (LPARAM)hIcon);
	SendMessage (mainwindow, WM_SETICON, (WPARAM)FALSE, (LPARAM)hIcon);

	return true;
}


qboolean VID_SetFullDIBMode (int modenum)
{
	HDC				hdc;
	int				lastmodestate, width, height;
	RECT			rect;

	if (!leavecurrentmode)
	{
		gdevmode.dmFields = DM_BITSPERPEL | DM_PELSWIDTH | DM_PELSHEIGHT;
		gdevmode.dmBitsPerPel = modelist[modenum].bpp;
		gdevmode.dmPelsWidth = modelist[modenum].width <<
							   modelist[modenum].halfscreen;
		gdevmode.dmPelsHeight = modelist[modenum].height;
		gdevmode.dmSize = sizeof (gdevmode);

		if (ChangeDisplaySettings (&gdevmode, CDS_FULLSCREEN) != DISP_CHANGE_SUCCESSFUL)
			Sys_Error ("Couldn't set fullscreen DIB mode");
	}

	lastmodestate = modestate;
	modestate = MS_FULLDIB;

	WindowRect.top = WindowRect.left = 0;

	WindowRect.right = modelist[modenum].width;
	WindowRect.bottom = modelist[modenum].height;

	DIBWidth = modelist[modenum].width;
	DIBHeight = modelist[modenum].height;

	WindowStyle = WS_POPUP;
	ExWindowStyle = 0;

	rect = WindowRect;
	AdjustWindowRectEx(&rect, WindowStyle, FALSE, 0);

	width = rect.right - rect.left;
	height = rect.bottom - rect.top;

	// Create the DIB window
	dibwindow = CreateWindowEx (
		 ExWindowStyle,
		 "WinQuake",
		 "MHQuake",
		 WindowStyle,
		 rect.left, rect.top,
		 width,
		 height,
		 NULL,
		 NULL,
		 global_hInstance,
		 NULL);

	if (!dibwindow)
		Sys_Error ("Couldn't create DIB window");

	ShowWindow (dibwindow, SW_SHOWDEFAULT);
	UpdateWindow (dibwindow);

	// Because we have set the background brush for the window to NULL
	// (to avoid flickering when re-sizing the window on the desktop), we
	// clear the window to black when created, otherwise it will be
	// empty while Quake starts up.
	hdc = GetDC(dibwindow);
	PatBlt(hdc,0,0,WindowRect.right,WindowRect.bottom,BLACKNESS);
	ReleaseDC(dibwindow, hdc);

	if (vid.conheight > modelist[modenum].height)
		vid.conheight = modelist[modenum].height;
	if (vid.conwidth > modelist[modenum].width)
		vid.conwidth = modelist[modenum].width;
	vid.width = vid.conwidth;
	vid.height = vid.conheight;

	vid.numpages = 2;

// needed because we're not getting WM_MOVE messages fullscreen on NT
	window_x = 0;
	window_y = 0;

	mainwindow = dibwindow;

	SendMessage (mainwindow, WM_SETICON, (WPARAM)TRUE, (LPARAM)hIcon);
	SendMessage (mainwindow, WM_SETICON, (WPARAM)FALSE, (LPARAM)hIcon);

	return true;
}


int VID_SetMode (int modenum, unsigned char *palette)
{
	int				original_mode, temp;
	qboolean		stat;
    MSG				msg;
	HDC				hdc;

	if ((windowed && (modenum != 0)) ||
		(!windowed && (modenum < 1)) ||
		(!windowed && (modenum >= nummodes)))
	{
		Sys_Error ("Bad video mode\n");
	}

	// so Con_Printfs don't mess us up by forcing vid and snd updates
	temp = scr_disabled_for_loading;
	scr_disabled_for_loading = true;

	CDAudio_Pause ();

	if (vid_modenum == NO_MODE)
		original_mode = windowed_default;
	else
		original_mode = vid_modenum;

	// Set either the fullscreen or windowed mode
	if (modelist[modenum].type == MS_WINDOWED)
	{
		if (_windowed_mouse.value && key_dest == key_game)
		{
			stat = VID_SetWindowedMode(modenum);
			IN_ActivateMouse ();
			IN_HideMouse ();
		}
		else
		{
			IN_DeactivateMouse ();
			IN_ShowMouse ();
			stat = VID_SetWindowedMode(modenum);
		}
	}
	else if (modelist[modenum].type == MS_FULLDIB)
	{
		stat = VID_SetFullDIBMode(modenum);
		IN_ActivateMouse ();
		IN_HideMouse ();
	}
	else
	{
		Sys_Error ("VID_SetMode: Bad mode type in modelist");
	}

	window_width = DIBWidth;
	window_height = DIBHeight;
	VID_UpdateWindowStatus ();

	CDAudio_Resume ();
	scr_disabled_for_loading = temp;

	if (!stat)
	{
		Sys_Error ("Couldn't set video mode");
	}

	// now we try to make sure we get the focus on the mode switch, because
	// sometimes in some systems we don't.  We grab the foreground, then
	// finish setting up, pump all our messages, and sleep for a little while
	// to let messages finish bouncing around the system, then we put
	// ourselves at the top of the z order, then grab the foreground again,
	// Who knows if it helps, but it probably doesn't hurt
	SetForegroundWindow (mainwindow);
	VID_SetPalette (palette);
	vid_modenum = modenum;
	Cvar_SetValueDirect (&vid_mode, (float) vid_modenum);

	while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
	{
      	TranslateMessage (&msg);
      	DispatchMessage (&msg);
	}

	Sleep (100);

	SetWindowPos (mainwindow, HWND_TOP, 0, 0, 0, 0,
				  SWP_DRAWFRAME | SWP_NOMOVE | SWP_NOSIZE | SWP_SHOWWINDOW |
				  SWP_NOCOPYBITS);

	SetForegroundWindow (mainwindow);

	// fix the leftover Alt from any Alt-Tab or the like that switched us away
	ClearAllStates ();

	if (!msg_suppress_1)
		Con_SafePrintf ("\nVideo mode %s initialized.\n", VID_GetModeDescription (vid_modenum));

	VID_SetPalette (palette);

	vid.recalc_refdef = 1;

	VIDMode_OldMode = modenum;

	return true;
}


/*
================
VID_UpdateWindowStatus
================
*/
void VID_UpdateWindowStatus (void)
{

	window_rect.left = window_x;
	window_rect.top = window_y;
	window_rect.right = window_x + window_width;
	window_rect.bottom = window_y + window_height;
	window_center_x = (window_rect.left + window_rect.right) / 2;
	window_center_y = (window_rect.top + window_rect.bottom) / 2;

	IN_UpdateClipCursor ();
}


//====================================

int max_size;

// numbers below 100 are reserved for internal textures...
int		texture_extension_number = 100;

int gl_textureunits;

qboolean shitedriver;
qboolean GimmeShader = false;


/*
================
MHGLGetProcAddress

Implements a work-around for earlier OpenGL versions, by checking for extensions that
support the required functions.  I'm VERY DISPLEASED at having to do this (you don't
cut down the Mona Lisa to make it fit in a smaller frame, do you?), but ATI's most
recent OpenGL implementation is only at version 1.3 and even at that it doesn't support
some 1.2 stuff.  Thank God for Doom 3 - it may make them get their act together.
================
*/
static PROC MHGLGetProcAddress (char *funcname)
{
	static PROC thePROC;

	// see is the function supported by the core implementation
	if (thePROC = wglGetProcAddress (funcname)) return thePROC;

	// see is an ARB or EXT version supported
	if (thePROC = wglGetProcAddress (va ("%sARB", funcname))) return thePROC;
	if (thePROC = wglGetProcAddress (va ("%sEXT", funcname))) return thePROC;

	// see does the vendor have an extension for it (NVIDIA and ATI preferred, not every possible vendor
	// is here, SGI is preferred before the more minor vendors, you can expand this list if you have a
	// really really weird obscure card)
	if (thePROC = wglGetProcAddress (va ("%sNV", funcname))) return thePROC;
	if (thePROC = wglGetProcAddress (va ("%sATI", funcname))) return thePROC;
	if (thePROC = wglGetProcAddress (va ("%sSGI", funcname))) return thePROC;
	if (thePROC = wglGetProcAddress (va ("%sSGIS", funcname))) return thePROC;
	if (thePROC = wglGetProcAddress (va ("%sSGIX", funcname))) return thePROC;
	if (thePROC = wglGetProcAddress (va ("%sINTEL", funcname))) return thePROC;
	if (thePROC = wglGetProcAddress (va ("%s3DFX", funcname))) return thePROC;

	shitedriver = true;

	// stop compiler warnings
	return NULL;
}


/*
==================
GL EXTENSION MALARKEY

As a general rule, I don't like the idea of proprietary extensions.  A special case of this rule is that
anything which allows me to rub ATI's nose in it must necessarily be a good thing :D
==================
*/

qboolean GL_CheckExtension (char *ExtensionName)
{
	if (strstr (gl_extensions, ExtensionName))
	{
		Con_Printf ("%sExtension Found\n", ExtensionName);
		return true;
	}

	return false;
}

qboolean GL_EXT_texture_filter_anisotropic = false;
qboolean OpenGL_1_5 = false;

/*
================
GetGLCore

Get all the core OpenGL stuff.  It is assumed that the players implementation supports
these - otherwise the app will Sys_Error.  Get a better card or driver.

Of course, MHQuake doesn't actually USE all of these - but they're there anyway in case I
ever decide to in the future.

This goes up to OpenGL 1.4, which should be more than sufficient, and allows a bit of
a breather for those who have slightly older drivers.  No proprietary extensions are used.

Unused functions are commented out.  Doesn't say they won't ever be used though, plus a lot of
the defines *are*.  So ATI owners beware...
================
*/
void GetGLCore (void)
{
	shitedriver = false;
	OpenGL_1_5 = false;

	// OpenGL 1.2
	//glBlendColor = (void *) MHGLGetProcAddress ("glBlendColor");
	//glBlendEquation = (void *) MHGLGetProcAddress ("glBlendEquation");
	//glDrawRangeElements = (void *) MHGLGetProcAddress ("glDrawRangeElements");
	//glColorTable = (void *) MHGLGetProcAddress ("glColorTable");
	//glColorTableParameterfv = (void *) MHGLGetProcAddress ("glColorTableParameterfv");
	//glColorTableParameteriv = (void *) MHGLGetProcAddress ("glColorTableParameteriv");
	//glCopyColorTable = (void *) MHGLGetProcAddress ("glCopyColorTable");
	//glGetColorTable = (void *) MHGLGetProcAddress ("glGetColorTable");
	//glGetColorTableParameterfv = (void *) MHGLGetProcAddress ("glGetColorTableParameterfv");
	//glGetColorTableParameteriv = (void *) MHGLGetProcAddress ("glGetColorTableParameteriv");
	//glColorSubTable = (void *) MHGLGetProcAddress ("glColorSubTable");
	//glCopyColorSubTable = (void *) MHGLGetProcAddress ("glCopyColorSubTable");
	//glConvolutionFilter1D = (void *) MHGLGetProcAddress ("glConvolutionFilter1D");
	//glConvolutionFilter2D = (void *) MHGLGetProcAddress ("glConvolutionFilter2D");
	//glConvolutionParameterf = (void *) MHGLGetProcAddress ("glConvolutionParameterf");
	//glConvolutionParameterfv = (void *) MHGLGetProcAddress ("glConvolutionParameterfv");
	//glConvolutionParameteri = (void *) MHGLGetProcAddress ("glConvolutionParameteri");
	//glConvolutionParameteriv = (void *) MHGLGetProcAddress ("glConvolutionParameteriv");
	//glCopyConvolutionFilter1D = (void *) MHGLGetProcAddress ("glCopyConvolutionFilter1D");
	//glCopyConvolutionFilter2D = (void *) MHGLGetProcAddress ("glCopyConvolutionFilter2D");
	//glGetConvolutionFilter = (void *) MHGLGetProcAddress ("glGetConvolutionFilter");
	//glGetConvolutionParameterfv = (void *) MHGLGetProcAddress ("glGetConvolutionParameterfv");
	//glGetConvolutionParameteriv = (void *) MHGLGetProcAddress ("glGetConvolutionParameteriv");
	//glGetSeparableFilter = (void *) MHGLGetProcAddress ("glGetSeparableFilter");
	//glSeparableFilter2D = (void *) MHGLGetProcAddress ("glSeparableFilter2D");
	//glGetHistogram = (void *) MHGLGetProcAddress ("glGetHistogram");
	//glGetHistogramParameterfv = (void *) MHGLGetProcAddress ("glGetHistogramParameterfv");
	//glGetHistogramParameteriv = (void *) MHGLGetProcAddress ("glGetHistogramParameteriv");
	//glGetMinmax = (void *) MHGLGetProcAddress ("glGetMinmax");
	//glGetMinmaxParameterfv = (void *) MHGLGetProcAddress ("glGetMinmaxParameterfv");
	//glGetMinmaxParameteriv = (void *) MHGLGetProcAddress ("glGetMinmaxParameteriv");
	//glHistogram = (void *) MHGLGetProcAddress ("glHistogram");
	//glMinmax = (void *) MHGLGetProcAddress ("glMinmax");
	//glResetHistogram = (void *) MHGLGetProcAddress ("glResetHistogram");
	//glResetMinmax = (void *) MHGLGetProcAddress ("glResetMinmax");
	glTexImage3D = (void *) MHGLGetProcAddress ("glTexImage3D");
	//glTexSubImage3D = (void *) MHGLGetProcAddress ("glTexSubImage3D");
	//glCopyTexSubImage3D = (void *) MHGLGetProcAddress ("glCopyTexSubImage3D");

	// OpenGL 1.3
	glActiveTexture = (void *) MHGLGetProcAddress ("glActiveTexture");
	glClientActiveTexture = (void *) MHGLGetProcAddress ("glClientActiveTexture");
	//glMultiTexCoord1d = (void *) MHGLGetProcAddress ("glMultiTexCoord1d");
	//glMultiTexCoord1dv = (void *) MHGLGetProcAddress ("glMultiTexCoord1dv");
	//glMultiTexCoord1f = (void *) MHGLGetProcAddress ("glMultiTexCoord1f");
	//glMultiTexCoord1fv = (void *) MHGLGetProcAddress ("glMultiTexCoord1fv");
	//glMultiTexCoord1i = (void *) MHGLGetProcAddress ("glMultiTexCoord1i");
	//glMultiTexCoord1iv = (void *) MHGLGetProcAddress ("glMultiTexCoord1iv");
	//glMultiTexCoord1s = (void *) MHGLGetProcAddress ("glMultiTexCoord1s");
	//glMultiTexCoord1sv = (void *) MHGLGetProcAddress ("glMultiTexCoord1sv");
	//glMultiTexCoord2d = (void *) MHGLGetProcAddress ("glMultiTexCoord2d");
	//glMultiTexCoord2dv = (void *) MHGLGetProcAddress ("glMultiTexCoord2dv");
	//glMultiTexCoord2f = (void *) MHGLGetProcAddress ("glMultiTexCoord2f");
	//glMultiTexCoord2fv = (void *) MHGLGetProcAddress ("glMultiTexCoord2fv");
	//glMultiTexCoord2i = (void *) MHGLGetProcAddress ("glMultiTexCoord2i");
	//glMultiTexCoord2iv = (void *) MHGLGetProcAddress ("glMultiTexCoord2iv");
	//glMultiTexCoord2s = (void *) MHGLGetProcAddress ("glMultiTexCoord2s");
	//glMultiTexCoord2sv = (void *) MHGLGetProcAddress ("glMultiTexCoord2sv");
	//glMultiTexCoord3d = (void *) MHGLGetProcAddress ("glMultiTexCoord3d");
	//glMultiTexCoord3dv = (void *) MHGLGetProcAddress ("glMultiTexCoord3dv");
	//glMultiTexCoord3f = (void *) MHGLGetProcAddress ("glMultiTexCoord3f");
	//glMultiTexCoord3fv = (void *) MHGLGetProcAddress ("glMultiTexCoord3fv");
	//glMultiTexCoord3i = (void *) MHGLGetProcAddress ("glMultiTexCoord3i");
	//glMultiTexCoord3iv = (void *) MHGLGetProcAddress ("glMultiTexCoord3iv");
	//glMultiTexCoord3s = (void *) MHGLGetProcAddress ("glMultiTexCoord3s");
	//glMultiTexCoord3sv = (void *) MHGLGetProcAddress ("glMultiTexCoord3sv");
	//glMultiTexCoord4d = (void *) MHGLGetProcAddress ("glMultiTexCoord4d");
	//glMultiTexCoord4dv = (void *) MHGLGetProcAddress ("glMultiTexCoord4dv");
	//glMultiTexCoord4f = (void *) MHGLGetProcAddress ("glMultiTexCoord4f");
	//glMultiTexCoord4fv = (void *) MHGLGetProcAddress ("glMultiTexCoord4fv");
	//glMultiTexCoord4i = (void *) MHGLGetProcAddress ("glMultiTexCoord4i");
	//glMultiTexCoord4iv = (void *) MHGLGetProcAddress ("glMultiTexCoord4iv");
	//glMultiTexCoord4s = (void *) MHGLGetProcAddress ("glMultiTexCoord4s");
	//glMultiTexCoord4sv = (void *) MHGLGetProcAddress ("glMultiTexCoord4sv");
	//glLoadTransposeMatrixf = (void *) MHGLGetProcAddress ("glLoadTransposeMatrixf");
	//glLoadTransposeMatrixd = (void *) MHGLGetProcAddress ("glLoadTransposeMatrixd");
	//glMultTransposeMatrixf = (void *) MHGLGetProcAddress ("glMultTransposeMatrixf");
	//glMultTransposeMatrixd = (void *) MHGLGetProcAddress ("glMultTransposeMatrixd");
	//glSampleCoverage = (void *) MHGLGetProcAddress ("glSampleCoverage");
	//glCompressedTexImage3D = (void *) MHGLGetProcAddress ("glCompressedTexImage3D");
	//glCompressedTexImage2D = (void *) MHGLGetProcAddress ("glCompressedTexImage2D");
	//glCompressedTexImage1D = (void *) MHGLGetProcAddress ("glCompressedTexImage1D");
	//glCompressedTexSubImage3D = (void *) MHGLGetProcAddress ("glCompressedTexSubImage3D");
	//glCompressedTexSubImage2D = (void *) MHGLGetProcAddress ("glCompressedTexSubImage2D");
	//glCompressedTexSubImage1D = (void *) MHGLGetProcAddress ("glCompressedTexSubImage1D");
	//glGetCompressedTexImage = (void *) MHGLGetProcAddress ("glGetCompressedTexImage");

	// OpenGL 1,4
	//glBlendFuncSeparate = (void *) MHGLGetProcAddress ("glBlendFuncSeparate");
	//glFogCoordf = (void *) MHGLGetProcAddress ("glFogCoordf");
	//glFogCoordfv = (void *) MHGLGetProcAddress ("glFogCoordfv");
	//glFogCoordd = (void *) MHGLGetProcAddress ("glFogCoordd");
	//glFogCoorddv = (void *) MHGLGetProcAddress ("glFogCoorddv");
	//glFogCoordPointer = (void *) MHGLGetProcAddress ("glFogCoordPointer");
	//glMultiDrawArrays = (void *) MHGLGetProcAddress ("glMultiDrawArrays");
	//glMultiDrawElements = (void *) MHGLGetProcAddress ("glMultiDrawElements");
	glPointParameterf = (void *) MHGLGetProcAddress ("glPointParameterf");
	glPointParameterfv = (void *) MHGLGetProcAddress ("glPointParameterfv");
	//glPointParameteri = (void *) MHGLGetProcAddress ("glPointParameteri");
	//glPointParameteriv = (void *) MHGLGetProcAddress ("glPointParameteriv");
	//glSecondaryColor3b = (void *) MHGLGetProcAddress ("glSecondaryColor3b");
	//glSecondaryColor3bv = (void *) MHGLGetProcAddress ("glSecondaryColor3bv");
	//glSecondaryColor3d = (void *) MHGLGetProcAddress ("glSecondaryColor3d");
	//glSecondaryColor3dv = (void *) MHGLGetProcAddress ("glSecondaryColor3dv");
	//glSecondaryColor3f = (void *) MHGLGetProcAddress ("glSecondaryColor3f");
	//glSecondaryColor3fv = (void *) MHGLGetProcAddress ("glSecondaryColor3fv");
	//glSecondaryColor3i = (void *) MHGLGetProcAddress ("glSecondaryColor3i");
	//glSecondaryColor3iv = (void *) MHGLGetProcAddress ("glSecondaryColor3iv");
	//glSecondaryColor3s = (void *) MHGLGetProcAddress ("glSecondaryColor3s");
	//glSecondaryColor3sv = (void *) MHGLGetProcAddress ("glSecondaryColor3sv");
	//glSecondaryColor3ub = (void *) MHGLGetProcAddress ("glSecondaryColor3ub");
	//glSecondaryColor3ubv = (void *) MHGLGetProcAddress ("glSecondaryColor3ubv");
	//glSecondaryColor3ui = (void *) MHGLGetProcAddress ("glSecondaryColor3ui");
	//glSecondaryColor3uiv = (void *) MHGLGetProcAddress ("glSecondaryColor3uiv");
	//glSecondaryColor3us = (void *) MHGLGetProcAddress ("glSecondaryColor3us");
	//glSecondaryColor3usv = (void *) MHGLGetProcAddress ("glSecondaryColor3usv");
	//glSecondaryColorPointer = (void *) MHGLGetProcAddress ("glSecondaryColorPointer");
	//glWindowPos2d = (void *) MHGLGetProcAddress ("glWindowPos2d");
	//glWindowPos2dv = (void *) MHGLGetProcAddress ("glWindowPos2dv");
	//glWindowPos2f = (void *) MHGLGetProcAddress ("glWindowPos2f");
	//glWindowPos2fv = (void *) MHGLGetProcAddress ("glWindowPos2fv");
	//glWindowPos2i = (void *) MHGLGetProcAddress ("glWindowPos2i");
	//glWindowPos2iv = (void *) MHGLGetProcAddress ("glWindowPos2iv");
	//glWindowPos2s = (void *) MHGLGetProcAddress ("glWindowPos2s");
	//glWindowPos2sv = (void *) MHGLGetProcAddress ("glWindowPos2sv");
	//glWindowPos3d = (void *) MHGLGetProcAddress ("glWindowPos3d");
	//glWindowPos3dv = (void *) MHGLGetProcAddress ("glWindowPos3dv");
	//glWindowPos3f = (void *) MHGLGetProcAddress ("glWindowPos3f");
	//glWindowPos3fv = (void *) MHGLGetProcAddress ("glWindowPos3fv");
	//glWindowPos3i = (void *) MHGLGetProcAddress ("glWindowPos3i");
	//glWindowPos3iv = (void *) MHGLGetProcAddress ("glWindowPos3iv");
	//glWindowPos3s = (void *) MHGLGetProcAddress ("glWindowPos3s");
	//glWindowPos3sv = (void *) MHGLGetProcAddress ("glWindowPos3sv");

	if (shitedriver)
	{
		Con_Printf ("\nHave you an ATI Card?  Poor thing.\n");
	}
	else
	{
		Con_Printf ("\nRequired OpenGL Support OK\n");
	}

	shitedriver = false;

	// look for OpenGL 1.5 vertex buffer support only
	//glGenQueries = (void *) MHGLGetProcAddress ("glGenQueries");
	//glDeleteQueries = (void *) MHGLGetProcAddress ("glDeleteQueries");
	//glIsQuery = (void *) MHGLGetProcAddress ("glIsQuery");
	//glBeginQuery = (void *) MHGLGetProcAddress ("glBeginQuery");
	//glEndQuery = (void *) MHGLGetProcAddress ("glEndQuery");
	//glGetQueryiv = (void *) MHGLGetProcAddress ("glGetQueryiv");
	//glGetQueryObjectiv = (void *) MHGLGetProcAddress ("glGetQueryObjectiv");
	//glGetQueryObjectuiv = (void *) MHGLGetProcAddress ("glGetQueryObjectuiv");
	glBindBuffer = (void *) MHGLGetProcAddress ("glBindBuffer");
	glDeleteBuffers = (void *) MHGLGetProcAddress ("glDeleteBuffers");
	glGenBuffers = (void *) MHGLGetProcAddress ("glGenBuffers");
	glIsBuffer = (void *) MHGLGetProcAddress ("glIsBuffer");
	glBufferData = (void *) MHGLGetProcAddress ("glBufferData");
	glBufferSubData = (void *) MHGLGetProcAddress ("glBufferSubData");
	glGetBufferSubData = (void *) MHGLGetProcAddress ("glGetBufferSubData");
	glMapBuffer = (void *) MHGLGetProcAddress ("glMapBuffer");
	glUnmapBuffer = (void *) MHGLGetProcAddress ("glUnmapBuffer");
	glGetBufferParameteriv = (void *) MHGLGetProcAddress ("glGetBufferParameteriv");
	glGetBufferPointerv = (void *) MHGLGetProcAddress ("glGetBufferPointerv");

	if (shitedriver)
	{
		//Con_Printf ("Couldn't find OpenGL Vertex Buffer Support\n   Reverting to standard Vertex Arrays\n");
	}
	else
	{
		//Con_Printf ("OpenGL Vertex Buffer Support OK\n");
		OpenGL_1_5 = true;
	}

	// hmmmm.  vertex buffers aren't really such a big deal at all...
	// in fact, MHQuake drops 100 FPS when using them.  Naaaaasssstyyyy hobbitses.
	//OpenGL_1_5 = false;

	// OpenGL Shading Language
	shitedriver = false;

	glDeleteObject = (void *) MHGLGetProcAddress ("glDeleteObject");
	glDeleteObject = (void *) MHGLGetProcAddress ("glDeleteObject");
	glDetachObject = (void *) MHGLGetProcAddress ("glDetachObject");
	glCreateShaderObject = (void *) MHGLGetProcAddress ("glCreateShaderObject");
	glShaderSource = (void *) MHGLGetProcAddress ("glShaderSource");
	glCompileShader = (void *) MHGLGetProcAddress ("glCompileShader");
	glCreateProgramObject = (void *) MHGLGetProcAddress ("glCreateProgramObject");
	glAttachObject = (void *) MHGLGetProcAddress ("glAttachObject");
	glLinkProgram = (void *) MHGLGetProcAddress ("glLinkProgram");
	glUseProgramObject = (void *) MHGLGetProcAddress ("glUseProgramObject");
	glValidateProgram = (void *) MHGLGetProcAddress ("glValidateProgram");
	glUniform1f = (void *) MHGLGetProcAddress ("glUniform1f");
	glUniform2f = (void *) MHGLGetProcAddress ("glUniform2f");
	glUniform3f = (void *) MHGLGetProcAddress ("glUniform3f");
	glUniform4f = (void *) MHGLGetProcAddress ("glUniform4f");
	glUniform1i = (void *) MHGLGetProcAddress ("glUniform1i");
	glUniform2i = (void *) MHGLGetProcAddress ("glUniform2i");
	glUniform3i = (void *) MHGLGetProcAddress ("glUniform3i");
	glUniform4i = (void *) MHGLGetProcAddress ("glUniform4i");
	glUniform1fv = (void *) MHGLGetProcAddress ("glUniform1fv");
	glUniform2fv = (void *) MHGLGetProcAddress ("glUniform2fv");
	glUniform3fv = (void *) MHGLGetProcAddress ("glUniform3fv");
	glUniform4fv = (void *) MHGLGetProcAddress ("glUniform4fv");
	glUniform1iv = (void *) MHGLGetProcAddress ("glUniform1iv");
	glUniform2iv = (void *) MHGLGetProcAddress ("glUniform2iv");
	glUniform3iv = (void *) MHGLGetProcAddress ("glUniform3iv");
	glUniform4iv = (void *) MHGLGetProcAddress ("glUniform4iv");
	glUniformMatrix2fv = (void *) MHGLGetProcAddress ("glUniformMatrix2fv");
	glUniformMatrix3fv = (void *) MHGLGetProcAddress ("glUniformMatrix3fv");
	glUniformMatrix4fv = (void *) MHGLGetProcAddress ("glUniformMatrix4fv");
	glGetObjectParameterfv = (void *) MHGLGetProcAddress ("glGetObjectParameterfv");
	glGetObjectParameteriv = (void *) MHGLGetProcAddress ("glGetObjectParameteriv");
	glGetInfoLog = (void *) MHGLGetProcAddress ("glGetInfoLog");
	glGetAttachedObjects = (void *) MHGLGetProcAddress ("glGetAttachedObjects");
	glGetUniformLocation = (void *) MHGLGetProcAddress ("glGetUniformLocation");
	glGetActiveUniform = (void *) MHGLGetProcAddress ("glGetActiveUniform");
	glGetUniformfv = (void *) MHGLGetProcAddress ("glGetUniformfv");
	glGetUniformiv = (void *) MHGLGetProcAddress ("glGetUniformiv");
	glGetShaderSource = (void *) MHGLGetProcAddress ("glGetShaderSource");

	if (shitedriver)
	{
		//Con_Printf ("OpenGL Shading Language NOT Found\n");

		// oooh i see the storm is threatening...
		//GimmeShader = false;
	}
	else
	{
		//Con_Printf ("Found OpenGL Shading Language\n");
		//GimmeShader = true;
	}

	// get the maximum texture size
	glGetIntegerv (GL_MAX_TEXTURE_SIZE, &max_size);
	Con_Printf ("\nMaximum Texture Size: %i x %i\n", max_size, max_size);

	// get the number of TMUs available
	glGetIntegerv (GL_MAX_TEXTURE_UNITS, &gl_textureunits);
	Con_Printf("Found %i TMUs support.\n\n", gl_textureunits);

	if (gl_textureunits < 2) Sys_Error ("OOOPS!!!\nYour card is crap!!!");

	// check for supported extensions
	//GL_EXT_texture_filter_anisotropic = GL_CheckExtension ("GL_EXT_texture_filter_anisotropic ");
	GL_MaxAnisotropy = 2;
}


/*
===============
GL_Init
===============
*/
void GL_Init (void)
{
	float attenuations[3] = {0.01, 0.0, 0.01};

	gl_vendor = glGetString (GL_VENDOR);
	Con_Printf ("\nGL_VENDOR:   %s\n", gl_vendor);

	gl_renderer = glGetString (GL_RENDERER);
	Con_Printf ("GL_RENDERER: %s\n", gl_renderer);

	gl_version = glGetString (GL_VERSION);
	Con_Printf ("GL_VERSION:  %s\n", gl_version);

	gl_extensions = glGetString (GL_EXTENSIONS);
	Con_DPrintf ("\nGL_EXTENSIONS: %s\n", gl_extensions);

	if ((gl_version[0] - '0') <= 1 && (gl_version[2] - '0') <= 4)
		Con_Printf ("\nWARNING\nGL_VERSION %s detected - you may experience reduced functionality\n", gl_version);

	GetGLCore ();

	// default states
	glClearColor (0.0, 0.0, 0.0, 0.0);

	glCullFace(GL_FRONT);
	glEnable(GL_TEXTURE_2D);

	//glEnable(GL_ALPHA_TEST);
	glAlphaFunc(GL_GREATER, 0.666);

	glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);
	glShadeModel (GL_SMOOTH);

	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

	// set up the hints for drawing
	glHint (GL_POINT_SMOOTH_HINT, GL_NICEST);
	glHint (GL_LINE_SMOOTH_HINT, GL_NICEST);
	glHint (GL_POLYGON_SMOOTH_HINT, GL_NICEST);
	glHint (GL_FOG_HINT, GL_NICEST);
	glHint (GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
	glHint (GL_GENERATE_MIPMAP_HINT, GL_NICEST);

	// explicitly enable dithering
	glEnable (GL_DITHER);

	// use fragment depth fog for higher quality
	glFogi (GL_FOG_COORDINATE_SOURCE, GL_FRAGMENT_DEPTH);

	// point parameters
	// ATI doesn't have glPointParameteri...
	glPointParameterf (GL_POINT_SIZE_MIN, 2);
	glPointParameterf (GL_POINT_SIZE_MAX, 40);
	glPointParameterfv (GL_POINT_DISTANCE_ATTENUATION, attenuations);
	glPointSize (40);
}


/*
=================
GL_BeginRendering

=================
*/
void GL_BeginRendering (int *x, int *y, int *width, int *height)
{
	*x = *y = 0;

	*width = WindowRect.right - WindowRect.left;
	*height = WindowRect.bottom - WindowRect.top;
}


void GL_EndRendering (void)
{
	if (!scr_skipupdate || block_drawing)
		SwapBuffers(maindc);

	// handle the mouse state when windowed if that's changed
	if (modestate == MS_WINDOWED)
	{
		if (!_windowed_mouse.value)
		{
			if (windowed_mouse)
			{
				IN_DeactivateMouse ();
				IN_ShowMouse ();
				windowed_mouse = false;
			}
		}
		else
		{
			windowed_mouse = true;

			if (key_dest == key_game && !mouseactive && ActiveApp)
			{
				IN_ActivateMouse ();
				IN_HideMouse ();
			}
			else if (mouseactive && key_dest != key_game)
			{
				IN_DeactivateMouse ();
				IN_ShowMouse ();
			}
		}
	}
}


void VID_SetPalette (unsigned char *palette)
{
	byte	*pal;
	unsigned r,g,b;
	unsigned v;
	unsigned short i;
	unsigned	*table;

	// 8 8 8 encoding
	pal = palette;
	table = d_8to24table;

	for (i = 0; i < 256; i++)
	{
		r = pal[0];
		g = pal[1];
		b = pal[2];

		pal += 3;
		
		v = (255 << 24) + (r << 0) + (g << 8) + (b << 16);

		*table++ = v;
	}

	d_8to24table[255] &= 0xffffff;	// 255 is transparent
}


BOOL	gammaworks;


void VID_SetDefaultMode (void)
{
	IN_DeactivateMouse ();
}


void GL_TexmanShutDown (void);

void	VID_Shutdown (void)
{
   	HGLRC hRC;
   	HDC	  hDC;

	if (vid_initialized)
	{
		vid_canalttab = false;
		vid_initialized = false;

		GL_TexmanShutDown ();

		hRC = wglGetCurrentContext ();
    	hDC = wglGetCurrentDC ();

    	wglMakeCurrent(NULL, NULL);

    	if (hRC)
    	    wglDeleteContext(hRC);

		if (hDC && dibwindow)
			ReleaseDC(dibwindow, hDC);

		if (modestate == MS_FULLDIB)
			ChangeDisplaySettings (NULL, 0);

		if (maindc && dibwindow)
			ReleaseDC (dibwindow, maindc);

		AppActivate(false, false);
	}
}


//==========================================================================

extern qboolean haveStencil;

BOOL bSetupPixelFormat(HDC hDC)
{
	static PIXELFORMATDESCRIPTOR pfd =
	{
		sizeof(PIXELFORMATDESCRIPTOR),	// size of this pfd
		1,						// version number
		PFD_DRAW_TO_WINDOW 		// support window
		|  PFD_SUPPORT_OPENGL 	// support OpenGL
		|  PFD_DOUBLEBUFFER ,	// double buffered
		PFD_TYPE_RGBA,			// RGBA type
		32,						// 32-bit color depth, my good man
		0, 0, 0, 0, 0, 0,		// color bits ignored
		0,						// no alpha buffer
		0,						// shift bit ignored
		0,						// no accumulation buffer
		0, 0, 0, 0, 			// accum bits ignored
		24,						// 24-bit z-buffer, my good man
		8,						// one of your finest 8-bit stencil buffers please, my good man
		0,						// no auxiliary buffer
		PFD_MAIN_PLANE,			// main layer
		0,						// reserved
		0, 0, 0					// layer masks ignored
	};

    int pixelformat;

    if ( (pixelformat = ChoosePixelFormat(hDC, &pfd)) == 0 )
    {
        MessageBox(NULL, "ChoosePixelFormat failed", "Error", MB_OK);
        return FALSE;
    }

    if (SetPixelFormat(hDC, pixelformat, &pfd) == FALSE)
    {
        MessageBox(NULL, "SetPixelFormat failed", "Error", MB_OK);
        return FALSE;
    }

	Con_Printf ("\n%i Bit Colour Buffer\n", (int) pfd.cColorBits);
	Con_Printf ("%i Bit Z Buffer\n", (int) pfd.cDepthBits);
	Con_Printf ("%i Bit Stencil Buffer\n", (int) pfd.cStencilBits);

	if (pfd.cStencilBits) haveStencil = true;

    return TRUE;
}



byte        scantokey[128] = 
					{ 
//  0           1       2       3       4       5       6       7 
//  8           9       A       B       C       D       E       F 
	0  ,    27,     '1',    '2',    '3',    '4',    '5',    '6', 
	'7',    '8',    '9',    '0',    '-',    '=',    K_BACKSPACE, 9, // 0 
	'q',    'w',    'e',    'r',    't',    'y',    'u',    'i', 
	'o',    'p',    '[',    ']',    13 ,    K_CTRL,'a',  's',      // 1 
	'd',    'f',    'g',    'h',    'j',    'k',    'l',    ';', 
	'\'' ,    '`',    K_SHIFT,'\\',  'z',    'x',    'c',    'v',      // 2 
	'b',    'n',    'm',    ',',    '.',    '/',    K_SHIFT,'*', 
	K_ALT,' ',   0  ,    K_F1, K_F2, K_F3, K_F4, K_F5,   // 3 
	K_F6, K_F7, K_F8, K_F9, K_F10, K_PAUSE  ,    0  , K_HOME, 
	K_UPARROW,K_PGUP,'-',K_LEFTARROW,'5',K_RIGHTARROW,'+',K_END, //4 
	K_DOWNARROW,K_PGDN,K_INS,K_DEL,0,0,             0,              K_F11, 
	K_F12,0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0,        // 5 
	0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0, 
	0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0,        // 6 
	0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0, 
	0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0         // 7 
					}; 

byte        shiftscantokey[128] = 
					{ 
//  0           1       2       3       4       5       6       7 
//  8           9       A       B       C       D       E       F 
	0  ,    27,     '!',    '@',    '#',    '$',    '%',    '^', 
	'&',    '*',    '(',    ')',    '_',    '+',    K_BACKSPACE, 9, // 0 
	'Q',    'W',    'E',    'R',    'T',    'Y',    'U',    'I', 
	'O',    'P',    '{',    '}',    13 ,    K_CTRL,'A',  'S',      // 1 
	'D',    'F',    'G',    'H',    'J',    'K',    'L',    ':', 
	'"' ,    '~',    K_SHIFT,'|',  'Z',    'X',    'C',    'V',      // 2 
	'B',    'N',    'M',    '<',    '>',    '?',    K_SHIFT,'*', 
	K_ALT,' ',   0  ,    K_F1, K_F2, K_F3, K_F4, K_F5,   // 3 
	K_F6, K_F7, K_F8, K_F9, K_F10, K_PAUSE  ,    0  , K_HOME, 
	K_UPARROW,K_PGUP,'_',K_LEFTARROW,'%',K_RIGHTARROW,'+',K_END, //4 
	K_DOWNARROW,K_PGDN,K_INS,K_DEL,0,0,             0,              K_F11, 
	K_F12,0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0,        // 5 
	0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0, 
	0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0,        // 6 
	0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0, 
	0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0  ,    0         // 7 
					}; 


/*
=======
MapKey

Map from windows to quake keynums
=======
*/
int MapKey (int key)
{
	key = (key>>16)&255;
	if (key > 127)
		return 0;
	if (scantokey[key] == 0)
		Con_DPrintf("key 0x%02x has no translation\n", key);
	return scantokey[key];
}

/*
===================================================================

MAIN WINDOW

===================================================================
*/

/*
================
ClearAllStates
================
*/
void ClearAllStates (void)
{
	int		i;
	
// send an up event for each key, to make sure the server clears them all
	for (i=0 ; i<256 ; i++)
	{
		Key_Event (i, false);
	}

	Key_ClearStates ();
	IN_ClearStates ();
}

void AppActivate(BOOL fActive, BOOL minimize)
/****************************************************************************
*
* Function:     AppActivate
* Parameters:   fActive - True if app is activating
*
* Description:  If the application is activating, then swap the system
*               into SYSPAL_NOSTATIC mode so that our palettes will display
*               correctly.
*
****************************************************************************/
{
	MSG msg;
    HDC			hdc;
    int			i, t;
	static BOOL	sound_active;

	ActiveApp = fActive;
	Minimized = minimize;

// enable/disable sound on focus gain/loss
	if (!ActiveApp && sound_active)
	{
		S_BlockSound ();
		sound_active = false;
	}
	else if (ActiveApp && !sound_active)
	{
		S_UnblockSound ();
		sound_active = true;
	}

	if (fActive)
	{
		if (modestate == MS_FULLDIB)
		{
			IN_ActivateMouse ();
			IN_HideMouse ();
			if (vid_canalttab && vid_wassuspended) {
				vid_wassuspended = false;
				ChangeDisplaySettings (&gdevmode, CDS_FULLSCREEN);
				ShowWindow(mainwindow, SW_SHOWNORMAL);
			}
		}
		else if ((modestate == MS_WINDOWED) && _windowed_mouse.value && key_dest == key_game)
		{
			IN_ActivateMouse ();
			IN_HideMouse ();
		}
	}

	if (!fActive)
	{
		if (modestate == MS_FULLDIB)
		{
			IN_DeactivateMouse ();
			IN_ShowMouse ();
			if (vid_canalttab) { 
				ChangeDisplaySettings (NULL, 0);
				vid_wassuspended = true;
			}
		}
		else if ((modestate == MS_WINDOWED) && _windowed_mouse.value)
		{
			IN_DeactivateMouse ();
			IN_ShowMouse ();
		}
	}
}


/* main window procedure */
#define WM_GRAPHNOTIFY  WM_USER + 13

LONG WINAPI MainWndProc (
    HWND    hWnd,
    UINT    uMsg,
    WPARAM  wParam,
    LPARAM  lParam)
{
    LONG    lRet = 1;
	int		fwKeys, xPos, yPos, fActive, fMinimized, temp;
	extern unsigned int uiWheelMessage;

	if ( uMsg == uiWheelMessage )
		uMsg = WM_MOUSEWHEEL;

    switch (uMsg)
    {
		case WM_KILLFOCUS:
			if (modestate == MS_FULLDIB)
				ShowWindow(mainwindow, SW_SHOWMINNOACTIVE);
			break;

		case WM_CREATE:
			break;

		case WM_MOVE:
			window_x = (int) LOWORD(lParam);
			window_y = (int) HIWORD(lParam);
			VID_UpdateWindowStatus ();
			break;

		case WM_KEYDOWN:
		case WM_SYSKEYDOWN:
			Key_Event (MapKey(lParam), true);
			break;
			
		case WM_KEYUP:
		case WM_SYSKEYUP:
			Key_Event (MapKey(lParam), false);
			break;

		case WM_SYSCHAR:
		// keep Alt-Space from happening
			break;

	// this is complicated because Win32 seems to pack multiple mouse events into
	// one update sometimes, so we always check all states and look for events
		case WM_LBUTTONDOWN:
		case WM_LBUTTONUP:
		case WM_RBUTTONDOWN:
		case WM_RBUTTONUP:
		case WM_MBUTTONDOWN:
		case WM_MBUTTONUP:
		case WM_MOUSEMOVE:
			temp = 0;

			if (wParam & MK_LBUTTON)
				temp |= 1;

			if (wParam & MK_RBUTTON)
				temp |= 2;

			if (wParam & MK_MBUTTON)
				temp |= 4;

			IN_MouseEvent (temp);

			break;

		// JACK: This is the mouse wheel with the Intellimouse
		// Its delta is either positive or neg, and we generate the proper
		// Event.
		case WM_MOUSEWHEEL: 
			if ((short) HIWORD(wParam) > 0)
			{
				Key_Event(K_MWHEELUP, true);
				Key_Event(K_MWHEELUP, false);
			}
			else
			{
				Key_Event(K_MWHEELDOWN, true);
				Key_Event(K_MWHEELDOWN, false);
			}

			break;

    	case WM_SIZE:
            break;

   	    case WM_CLOSE:
			if (MessageBox (mainwindow, "Are you sure you want to quit?", "Confirm Exit",
						MB_YESNO | MB_SETFOREGROUND | MB_ICONQUESTION) == IDYES)
			{
				Sys_Quit ();
			}

	        break;

		case WM_ACTIVATE:
			fActive = LOWORD(wParam);
			fMinimized = (BOOL) HIWORD(wParam);
			AppActivate(!(fActive == WA_INACTIVE), fMinimized);

		// fix the leftover Alt from any Alt-Tab or the like that switched us away
			ClearAllStates ();

			break;

   	    case WM_DESTROY:
        {
			if (dibwindow)
				DestroyWindow (dibwindow);

            PostQuitMessage (0);
        }
        break;

		case WM_GRAPHNOTIFY:
            lRet = CDAudio_MessageHandler (hWnd, uMsg, wParam, lParam);
			break;

    	default:
            /* pass all unhandled messages to DefWindowProc */
            lRet = DefWindowProc (hWnd, uMsg, wParam, lParam);
        break;
    }

    /* return 1 if handled message, 0 if not */
    return lRet;
}


/*
=================
VID_NumModes
=================
*/
int VID_NumModes (void)
{
	return nummodes;
}

	
/*
=================
VID_GetModePtr
=================
*/
vmode_t *VID_GetModePtr (int modenum)
{

	if ((modenum >= 0) && (modenum < nummodes))
		return &modelist[modenum];
	else
		return &badmode;
}


/*
=================
VID_GetModeDescription
=================
*/
char *VID_GetModeDescription (int mode)
{
	char		*pinfo;
	vmode_t		*pv;
	static char	temp[100];

	if ((mode < 0) || (mode >= nummodes))
		return NULL;

	if (!leavecurrentmode)
	{
		pv = VID_GetModePtr (mode);

		sprintf (temp, "%d x %d x %d",
				 pv->width, pv->height, pv->bpp);

		pinfo = temp;
	}
	else
	{
		sprintf (temp, "Desktop resolution (%d x %d)",
				 modelist[MODE_FULLSCREEN_DEFAULT].width,
				 modelist[MODE_FULLSCREEN_DEFAULT].height);

		pinfo = temp;
	}

	return pinfo;
}


// KJB: Added this to return the mode driver name in description for console

char *VID_GetExtModeDescription (int mode)
{
	static char	pinfo[40];
	vmode_t		*pv;

	if ((mode < 0) || (mode >= nummodes))
		return NULL;

	pv = VID_GetModePtr (mode);
	if (modelist[mode].type == MS_FULLDIB)
	{
		if (!leavecurrentmode)
		{
			sprintf(pinfo,"%s fullscreen", pv->modedesc);
		}
		else
		{
			sprintf (pinfo, "Desktop resolution (%dx%d)",
					 modelist[MODE_FULLSCREEN_DEFAULT].width,
					 modelist[MODE_FULLSCREEN_DEFAULT].height);
		}
	}
	else
	{
		if (modestate == MS_WINDOWED)
			sprintf(pinfo, "%s windowed", pv->modedesc);
		else
			sprintf(pinfo, "windowed");
	}

	return pinfo;
}


/*
=================
VID_DescribeCurrentMode_f
=================
*/
void VID_DescribeCurrentMode_f (void)
{
	Con_Printf ("%s\n", VID_GetExtModeDescription (vid_modenum));
}


/*
=================
VID_NumModes_f
=================
*/
void VID_NumModes_f (void)
{

	if (nummodes == 1)
		Con_Printf ("%d video mode is available\n", nummodes);
	else
		Con_Printf ("%d video modes are available\n", nummodes);
}


/*
=================
VID_DescribeMode_f
=================
*/
void VID_DescribeMode_f (void)
{
	int		t, modenum;
	
	modenum = Q_atoi (Cmd_Argv(1));

	t = leavecurrentmode;
	leavecurrentmode = 0;

	Con_Printf ("%s\n", VID_GetExtModeDescription (modenum));

	leavecurrentmode = t;
}


/*
=================
VID_DescribeModes_f
=================
*/
void VID_DescribeModes_f (void)
{
	int			i, lnummodes, t;
	char		*pinfo;
	vmode_t		*pv;

	lnummodes = VID_NumModes ();

	t = leavecurrentmode;
	leavecurrentmode = 0;

	for (i=1 ; i<lnummodes ; i++)
	{
		pv = VID_GetModePtr (i);
		pinfo = VID_GetExtModeDescription (i);
		Con_Printf ("%2d: %s\n", i, pinfo);
	}

	leavecurrentmode = t;
}


void VID_InitDIB (HINSTANCE hInstance)
{
	WNDCLASS		wc;
	HDC				hdc;
	int				i;

	/* Register the frame class */
    wc.style         = 0;
    wc.lpfnWndProc   = (WNDPROC)MainWndProc;
    wc.cbClsExtra    = 0;
    wc.cbWndExtra    = 0;
    wc.hInstance     = hInstance;
    wc.hIcon         = 0;
    wc.hCursor       = LoadCursor (NULL,IDC_ARROW);
	wc.hbrBackground = NULL;
    wc.lpszMenuName  = 0;
    wc.lpszClassName = "WinQuake";

    if (!RegisterClass (&wc) )
		Sys_Error ("Couldn't register window class");

	modelist[0].type = MS_WINDOWED;

	if (COM_CheckParm("-width"))
		modelist[0].width = Q_atoi(com_argv[COM_CheckParm("-width")+1]);
	else
		modelist[0].width = 640;

	if (modelist[0].width < 640)
		modelist[0].width = 640;

	if (COM_CheckParm("-height"))
		modelist[0].height= Q_atoi(com_argv[COM_CheckParm("-height")+1]);
	else
		modelist[0].height = modelist[0].width * 240/320;

	if (modelist[0].height < 480)
		modelist[0].height = 480;

	sprintf (modelist[0].modedesc, "%d x %d",
			 modelist[0].width, modelist[0].height);

	modelist[0].modenum = MODE_WINDOWED;
	modelist[0].dib = 1;
	modelist[0].fullscreen = 0;
	modelist[0].halfscreen = 0;
	modelist[0].bpp = 0;

	nummodes = 1;
}


/*
=================
VID_InitFullDIB
=================
*/
typedef struct _mydevicemode { 
  BCHAR  dmDeviceName[CCHDEVICENAME]; 
  WORD   dmSpecVersion; 
  WORD   dmDriverVersion; 
  WORD   dmSize; 
  WORD   dmDriverExtra; 
  DWORD  dmFields; 
  union {
    struct {
      short dmOrientation;
      short dmPaperSize;
      short dmPaperLength;
      short dmPaperWidth;
      short dmScale; 
      short dmCopies; 
      short dmDefaultSource; 
      short dmPrintQuality; 
    };
    POINTL dmPosition;
    DWORD  dmDisplayOrientation;
    DWORD  dmDisplayFixedOutput;
  };

  short  dmColor; 
  short  dmDuplex; 
  short  dmYResolution; 
  short  dmTTOption; 
  short  dmCollate; 
  BYTE  dmFormName[CCHFORMNAME]; 
  WORD  dmLogPixels; 
#if 0
  DWORD  dmBitsPerPel; 
  DWORD  dmPelsWidth; 
  DWORD  dmPelsHeight; 
  union {
    DWORD  dmDisplayFlags; 
    DWORD  dmNup;
  };
  DWORD  dmDisplayFrequency; 
#if(WINVER >= 0x0400) 
  DWORD  dmICMMethod;
  DWORD  dmICMIntent;
  DWORD  dmMediaType;
  DWORD  dmDitherType;
  DWORD  dmReserved1;
  DWORD  dmReserved2;
#if (WINVER >= 0x0500) || (_WIN32_WINNT >= 0x0400)
  DWORD  dmPanningWidth;
  DWORD  dmPanningHeight;
#endif
#endif /* WINVER >= 0x0400 */
#endif
} myDEVMODE;

void VID_InitFullDIB (HINSTANCE hInstance)
{
	DEVMODE	devmode;
	int		i, modenum, cmodes, originalnummodes, existingmode, numlowresmodes;
	int		j, bpp, done;
	BOOL	stat;

	int maxWidth;
	int maxHeight;

	HWND hDesktopWnd;
	HDC hDesktopDC;

	// get the current desktop height and width
	hDesktopWnd = GetDesktopWindow ();
	hDesktopDC = GetDC (hDesktopWnd);

	maxWidth = GetDeviceCaps (hDesktopDC, HORZRES);		// width
	maxHeight = GetDeviceCaps (hDesktopDC, VERTRES);		// height

	// release the desktop device context
	ReleaseDC (hDesktopWnd, hDesktopDC);

	// enumerate >8 bpp modes
	originalnummodes = nummodes;
	modenum = 0;

	//MessageBox (NULL, va ("%i - %i", sizeof (myDEVMODE), sizeof (DWORD)), "", 0);
	do
	{
		stat = EnumDisplaySettings (NULL, modenum, &devmode);

		// don't report modes below 640 * 480 or above the desktop height and width
		if ((devmode.dmBitsPerPel == 16 || devmode.dmBitsPerPel == 32) && (devmode.dmPelsWidth <= maxWidth) &&
			(devmode.dmPelsWidth >= 640) && (devmode.dmPelsHeight >= 480) && (devmode.dmPelsHeight <= maxHeight) 
			&& (nummodes < MAX_MODE_LIST))
		{
			devmode.dmFields = DM_BITSPERPEL | DM_PELSWIDTH | DM_PELSHEIGHT;
			//MessageBox (NULL, va ("%i", devmode.dmFields), "", 0);

			if (ChangeDisplaySettings (&devmode, CDS_TEST | CDS_FULLSCREEN) == DISP_CHANGE_SUCCESSFUL)
			{
				modelist[nummodes].type = MS_FULLDIB;
				modelist[nummodes].width = devmode.dmPelsWidth;
				modelist[nummodes].height = devmode.dmPelsHeight;
				modelist[nummodes].modenum = 0;
				modelist[nummodes].halfscreen = 0;
				modelist[nummodes].dib = 1;
				modelist[nummodes].fullscreen = 1;
				modelist[nummodes].bpp = devmode.dmBitsPerPel;
				sprintf (modelist[nummodes].modedesc, "%d x %d x %d", devmode.dmPelsWidth, devmode.dmPelsHeight, devmode.dmBitsPerPel);

				// if the width is more than twice the height, reduce it by half because this
				// is probably a dual-screen monitor
				if (!COM_CheckParm("-noadjustaspect"))
				{
					if (modelist[nummodes].width > (modelist[nummodes].height << 1))
					{
						modelist[nummodes].width >>= 1;
						modelist[nummodes].halfscreen = 1;
						sprintf (modelist[nummodes].modedesc, "%d x %d x %d",
								 modelist[nummodes].width,
								 modelist[nummodes].height,
								 modelist[nummodes].bpp);
					}
				}

				for (i=originalnummodes, existingmode = 0 ; i<nummodes ; i++)
				{
					if ((modelist[nummodes].width == modelist[i].width)   &&
						(modelist[nummodes].height == modelist[i].height) &&
						(modelist[nummodes].bpp == modelist[i].bpp))
					{
						existingmode = 1;
						break;
					}
				}

				if (!existingmode)
				{
					nummodes++;
				}
			}
		}

		modenum++;
	} while (stat);

	// don't bother with the low res modes - this is 2005!!!
	if (nummodes == originalnummodes)
		Con_SafePrintf ("No fullscreen DIB modes found\n");
}


static void Check_Gamma (unsigned char *pal)
{
	float	f, inf;
	unsigned char palette[768];
	int i;

	if ((i = COM_CheckParm ("-gamma")) == 0)
	{
		// fix gamma
		vid_gamma = 1;
	}
	else vid_gamma = Q_atof (com_argv[i + 1]);

	for (i = 0; i < 768; i++)
	{
		f = pow ((pal[i] + 1) / 256.0, vid_gamma);

		inf = f * 255 + 0.5;

		if (inf < 0) inf = 0;
		if (inf > 255) inf = 255;

		palette[i] = inf;
	}

	memcpy (pal, palette, sizeof (palette));
}

/*
===================
VID_Init
===================
*/
void VID_InitMode (int initmode, unsigned char *palette)
{
	Check_Gamma (palette);
	VID_SetPalette (palette);

	VID_SetMode (initmode, palette);

    maindc = GetDC(mainwindow);
	bSetupPixelFormat(maindc);

    baseRC = wglCreateContext( maindc );

	if (!baseRC) Sys_Error ("Could not initialize GL (wglCreateContext failed).\n\nMake sure you in are 65535 color mode, and try running -window.");

    if (!wglMakeCurrent (maindc, baseRC)) Sys_Error ("wglMakeCurrent failed");

	vid_realmode = vid_modenum;
}


void VID_Init (unsigned char *palette)
{
	int		i, existingmode;
	int		basenummodes, width, height, bpp, findbpp, done;
	byte	*ptmp;
	HDC		hdc;
	DEVMODE	devmode;

	memset(&devmode, 0, sizeof(devmode));

	Cvar_RegisterVariable (&vid_mode);
	Cvar_RegisterVariable (&vid_wait);
	Cvar_RegisterVariable (&vid_nopageflip);
	Cvar_RegisterVariable (&_vid_wait_override);
	Cvar_RegisterVariable (&_vid_default_mode);
	Cvar_RegisterVariable (&_vid_default_mode_win);
	Cvar_RegisterVariable (&vid_config_x);
	Cvar_RegisterVariable (&vid_config_y);
	Cvar_RegisterVariable (&vid_stretch_by_2);
	Cvar_RegisterVariable (&_windowed_mouse);
	Cvar_RegisterVariable (&gl_ztrick);
	Cvar_RegisterVariable (&gl_anisotropicfilter);
	Cvar_RegisterVariable (&gl_detailtexture);
	Cvar_RegisterVariable (&gl_sparky);
	Cvar_RegisterVariable (&gl_vertexlight);

	Cmd_AddCommand ("vid_nummodes", VID_NumModes_f);
	Cmd_AddCommand ("vid_describecurrentmode", VID_DescribeCurrentMode_f);
	Cmd_AddCommand ("vid_describemode", VID_DescribeMode_f);
	Cmd_AddCommand ("vid_describemodes", VID_DescribeModes_f);

	hIcon = LoadIcon (global_hInstance, MAKEINTRESOURCE (IDI_ICON2));

	InitCommonControls();

	VID_InitDIB (global_hInstance);
	basenummodes = nummodes = 1;

	VID_InitFullDIB (global_hInstance);

	if (COM_CheckParm("-window"))
	{
		hdc = GetDC (NULL);

		if (GetDeviceCaps(hdc, RASTERCAPS) & RC_PALETTE)
		{
			Sys_Error ("Can't run in non-RGB mode");
		}

		ReleaseDC (NULL, hdc);

		windowed = true;

		vid_default = MODE_WINDOWED;
	}
	else
	{
		if (nummodes == 1)
			Sys_Error ("No RGB fullscreen modes available");

		windowed = false;

		if (COM_CheckParm("-mode"))
		{
			vid_default = Q_atoi(com_argv[COM_CheckParm("-mode")+1]);
		}
		else
		{
			if (COM_CheckParm("-current"))
			{
				modelist[MODE_FULLSCREEN_DEFAULT].width = GetSystemMetrics (SM_CXSCREEN);
				modelist[MODE_FULLSCREEN_DEFAULT].height = GetSystemMetrics (SM_CYSCREEN);
				vid_default = MODE_FULLSCREEN_DEFAULT;
				leavecurrentmode = 1;
			}
			else
			{
				if (COM_CheckParm("-width"))
				{
					width = Q_atoi(com_argv[COM_CheckParm("-width")+1]);
				}
				else
				{
					// set to screen resolution if not specified
					width = GetSystemMetrics (SM_CXSCREEN);
				}

				if (COM_CheckParm("-bpp"))
				{
					bpp = Q_atoi(com_argv[COM_CheckParm("-bpp")+1]);
					findbpp = 0;
				}
				else
				{
					// set to 32 bit if not specified (fixme - should be screen BPP)
					bpp = 32;
					findbpp = 1;
				}

				if (COM_CheckParm("-height"))
					height = Q_atoi(com_argv[COM_CheckParm("-height")+1]);

				// if they want to force it, add the specified mode to the list
				if (COM_CheckParm("-force") && (nummodes < MAX_MODE_LIST))
				{
					modelist[nummodes].type = MS_FULLDIB;
					modelist[nummodes].width = width;
					modelist[nummodes].height = height;
					modelist[nummodes].modenum = 0;
					modelist[nummodes].halfscreen = 0;
					modelist[nummodes].dib = 1;
					modelist[nummodes].fullscreen = 1;
					modelist[nummodes].bpp = bpp;
					sprintf (modelist[nummodes].modedesc, "%d x %d x %d",
							 devmode.dmPelsWidth, devmode.dmPelsHeight,
							 devmode.dmBitsPerPel);

					for (i=nummodes, existingmode = 0 ; i<nummodes ; i++)
					{
						if ((modelist[nummodes].width == modelist[i].width)   &&
							(modelist[nummodes].height == modelist[i].height) &&
							(modelist[nummodes].bpp == modelist[i].bpp))
						{
							existingmode = 1;
							break;
						}
					}

					if (!existingmode)
					{
						nummodes++;
					}
				}

				done = 0;

				do
				{
					if (COM_CheckParm("-height"))
					{
						height = Q_atoi(com_argv[COM_CheckParm("-height")+1]);

						for (i=1, vid_default=0 ; i<nummodes ; i++)
						{
							if ((modelist[i].width == width) &&
								(modelist[i].height == height) &&
								(modelist[i].bpp == bpp))
							{
								vid_default = i;
								done = 1;
								break;
							}
						}
					}
					else
					{
						for (i=1, vid_default=0 ; i<nummodes ; i++)
						{
							if ((modelist[i].width == width) && (modelist[i].bpp == bpp))
							{
								vid_default = i;
								done = 1;
								break;
							}
						}
					}

					if (!done)
					{
						if (findbpp)
						{
							/*
							switch (bpp)
							{
							case 15:
								bpp = 16;
								break;
							case 16:
								bpp = 32;
								break;
							case 32:
								bpp = 24;
								break;
							case 24:
								done = 1;
								break;
							}
							*/
							switch (bpp)
							{
							case 32:
								bpp = 16;
								break;
							case 16:
								done = 1;
								break;
							}
						}
						else
						{
							done = 1;
						}
					}
				} while (!done);

				if (!vid_default)
				{
					Sys_Error ("Specified video mode not available");
				}
			}
		}
	}

	vid_initialized = true;

	vid.conwidth = 640;
	vid.conheight = 480;

	vid.maxwarpwidth = WARP_WIDTH;
	vid.maxwarpheight = WARP_HEIGHT;
	vid.colormap = host_colormap;
	vid.fullbright = 256 - LittleLong (*((int *)vid.colormap + 2048));

	VID_InitMode (vid_default, palette);

	GL_Init ();

	vid_menudrawfn = VID_MenuDraw;
	vid_menukeyfn = VID_MenuKey;

	strcpy (badmode.modedesc, "Bad mode");
	vid_canalttab = true;

	//VID_EnumerateResolutions ();
}


//========================================================
// Video menu stuff
//========================================================

extern void M_Menu_Options_f (void);
extern void M_Print (int cx, int cy, char *str);
extern void M_PrintWhite (int cx, int cy, char *str);
extern void M_DrawCharacter (int cx, int line, int num);

void M_DrawQPlaque (void);
void M_DrawMenuTitle (char *title);
void M_Options_Option_Draw (char *text, int OptionType, float currValue);


int currVOptionPos;

int voptions_cursor;
#define VOPTIONS_ITEMS 9

int voptions_pos[] = {48, 96, 104, 112, 120, 128, 136, 144, 152};
int vmmode;
extern cvar_t crosshair;
extern cvar_t v_gunz;
void Draw_XHair (int basex, int basey, int size, float r, float g, float b);
void Draw_XHairWrapper (int basex, int basey, int size);

void VM_AdjustSliders (int dir)
{
	S_LocalSound ("misc/menu3.wav");

	switch (voptions_cursor)
	{
	case 0:
		// resolution change
		vmmode += dir;

		if (vmmode < 1) vmmode = 1;
		if (vmmode >= nummodes) vmmode = nummodes - 1;

		// prevent resolution changes in windowed mode cos it just doesn't work.
		if (modestate == MS_WINDOWED) vmmode = 0;
		break;

	case 1:
		// crosshair
		crosshair.value += dir;

		// the crosshair drawing routine will look after bounds checking this
		break;

	case 2:
		// crosshair colour
		crosshaircolour.value += dir;

		Cvar_SetValueDirect (&crosshaircolour, crosshaircolour.value);
		Cvar_Bound (&crosshaircolour, 0, 6);
		break;

	case 3:
		// anisotropic filter
		if (dir > 0) 
			gl_anisotropicfilter.value *= 2;
		else gl_anisotropicfilter.value /= 2;

		Cvar_SetValueDirect (&gl_anisotropicfilter, gl_anisotropicfilter.value);
		Cvar_Bound (&gl_anisotropicfilter, 0, GL_MaxAnisotropy);
		break;

	case 4:
		// detail texture scale - make it rise exponential like
		if (dir > 0 && gl_detailtexture.value < 1)
			gl_detailtexture.value = 1;
		else if (dir > 0)
			gl_detailtexture.value *= 2;
		else if (dir < 0 && gl_detailtexture.value <= 1)
			gl_detailtexture.value = 0;
		else gl_detailtexture.value /= 2;

		Cvar_SetValueDirect (&gl_detailtexture, gl_detailtexture.value);
		Cvar_Bound (&gl_detailtexture, 0, 999);

		break;

	case 5:
		// gun z offset
		v_gunz.value += dir * 0.25;

		Cvar_SetValueDirect (&v_gunz, v_gunz.value);
		Cvar_Bound (&v_gunz, -3, 3);
		break;

	case 6:
		Cvar_SetValueDirect (&r_shadows, !r_shadows.value);
		break;

	case 7:
		Cvar_SetValueDirect (&gl_sparky, !gl_sparky.value);
		break;

	case 8:
		Cvar_SetValueDirect (&gl_vertexlight, !gl_vertexlight.value);
		break;
	}
}


void M_PrintBinaryValue (float value, char *ZeroString, char *OneString, int line)
{
	if (value)
		M_Print (220, line, OneString);
	else M_Print (220, line, ZeroString);
}

/*
================
VID_MenuDraw
================
*/
void VID_MenuDraw (void)
{
	qpic_t	*pic;
	glpic_t *gl;

	vmode_t		*pv = VID_GetModePtr (vmmode);

	M_DrawQPlaque ();
	M_DrawMenuTitle ("gfx/vidmodes.lmp");

	currVOptionPos = 48;

	// current video mode
	M_PrintWhite (16, 48, "          CURRENT MODE");
	M_Print      (16, 56, "            Resolution");
	M_Print      (16, 64, "                   BPP");
	M_Print      (16, 72, "              Windowed");

	if (vmmode < 0 || vmmode >= nummodes)
	{
		// unknown video mode
		M_Print (220, 56, "*** Unknown");
		M_Print (220, 64, "*** Unknown");
		M_Print (220, 72, "*** Unknown");
	}
	else
	{
		M_Print (220, 48, va ("%i", vmmode));
		M_Print (220, 56, va ("%i x %i", pv->width, pv->height));
		M_Print (220, 64, va ("%i", pv->bpp));

		if (pv->type == MS_WINDOWED)
		{
			M_Print (220, 72, "Yes");
			M_PrintWhite (42, 212, "You cannot switch resolution in");
			M_PrintWhite (44, 220, "Windowed Mode. Yet. Life sucks.");
		}
		else
		{
			M_Print (220, 72, "No");
		}
	}

	M_PrintWhite (16, 88, "         VIDEO OPTIONS");
	M_Print      (16, 96, "             Crosshair");
	M_Print      (16, 104, "      Crosshair Colour");
	M_Print      (16, 112, "    Anisotropic Filter");
	M_Print      (16, 120, "       Detail Textures");
	M_Print      (16, 128, "          Gun Z Offset");
	M_Print      (16, 136, "               Shadows");
	M_Print      (16, 144, "          Trail Sparks");
	M_Print      (16, 152, "       Vertex Lighting");

	Draw_XHairWrapper (384, 104, 16);

	M_Print		(220, 112, va ("%0i x", (int) gl_anisotropicfilter.value));

	if (gl_detailtexture.value > 0)
		M_Print (220, 120, va ("%0i x Scale", (int) gl_detailtexture.value));
	else M_Print (220, 120, "Off");

	if (v_gunz.value < 0)
		M_Print (220, 128, va ("%0.2f Down", v_gunz.value * -1));
	else if (v_gunz.value > 0)
		M_Print (220, 128, va ("%0.2f Up", v_gunz.value));
	else M_Print (220, 128, "Off");

	M_PrintBinaryValue (r_shadows.value, "Off", "On", 136);
	M_PrintBinaryValue (gl_sparky.value, "Off", "On", 144);
	M_PrintBinaryValue (gl_vertexlight.value, "Off", "On", 152);

	M_DrawCharacter (200, voptions_pos[voptions_cursor], 12 + ((int)(realtime * 4) & 1));
}


/*
================
VID_MenuKey
================
*/
void VID_MenuKey (int key)
{
	switch (key)
	{
	case K_ESCAPE:
		S_LocalSound ("misc/menu1.wav");

		if (vmmode != vid_modenum)
		{
			Cvar_SetValueDirect (&vid_mode, vmmode);
		}

		M_Menu_Options_f ();
		break;

	case K_LEFTARROW:
		VM_AdjustSliders (-1);
		break;

	case K_RIGHTARROW:
	case K_ENTER:
		VM_AdjustSliders (1);
		break;

	case K_UPARROW:
		S_LocalSound ("misc/menu1.wav");

		voptions_cursor--;
		if (voptions_cursor < 0)
			voptions_cursor = VOPTIONS_ITEMS - 1;
		break;

	case K_DOWNARROW:
		S_LocalSound ("misc/menu1.wav");

		voptions_cursor++;

		if (voptions_cursor >= VOPTIONS_ITEMS)
			voptions_cursor = 0;
		break;

	default:
		break;
	}
}


// stuff to precheck each frame
typedef struct GL_Precheck_s
{
	int VidMode;
	float AnisotropicFilter;
} GL_Precheck_t;


GL_Precheck_t GL_Precheck;


qboolean VID_FramePrecheck (void)
{
	qboolean VidChanged = false;

	// check for resolution change
	if (VIDMode_OldMode != vid_mode.value)
	{
		// switch to the new vid mode
		VID_Shutdown ();
		VID_InitMode  (vid_mode.value, host_basepal);

		// default OpenGL state
		GL_Init ();

		VidChanged = true;
	}

	// check for gamma change

	return VidChanged;
}



