/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// gl_warp.c -- sky and water polygons

#include "quakedef.h"
#include "glext.h"

extern	model_t	*loadmodel;

int		solidskytexture = 0;
int		alphaskytexture = 0;
int		skyboxtextures = 0;


msurface_t	*warpface;

extern cvar_t gl_subdivide_size;


// these are incremented in R_RenderView cos we want them always updated even if
// the sky is not being currently drawn
float rotateBack = 0;
float rotateFore = 0;


// restore the original quake 1 subdivision
// SUBDIVIDE_SIZE has significant imapct on the extent of the breakup.  1 gives the best results but is slow,
// breakup starts to become noticeable at 128.  64 is very very very good.
#define	SUBDIVIDE_SIZE	64

void BoundPoly (int numverts, float *verts, vec3_t mins, vec3_t maxs)
{
	int		i, j;
	float	*v;

	mins[0] = mins[1] = mins[2] = 9999;
	maxs[0] = maxs[1] = maxs[2] = -9999;
	v = verts;
	for (i=0 ; i<numverts ; i++)
		for (j=0 ; j<3 ; j++, v++)
		{
			if (*v < mins[j])
				mins[j] = *v;
			if (*v > maxs[j])
				maxs[j] = *v;
		}
}

void SubdividePolygon (int numverts, float *verts)
{
	int		i, j, k;
	vec3_t	mins, maxs;
	float	m;
	float	*v;
	vec3_t	front[64], back[64];
	int		f, b;
	float	dist[64];
	float	frac;
	glpoly_t	*poly;
	float	s, t;

	if (numverts > 60)
		Sys_Error ("numverts = %i", numverts);

	BoundPoly (numverts, verts, mins, maxs);

	for (i = 0; i < 3; i++)
	{
		m = (mins[i] + maxs[i]) * 0.5;

		m = SUBDIVIDE_SIZE * floor (m / SUBDIVIDE_SIZE + 0.5);

		if (maxs[i] - m < 8) continue;
		if (m - mins[i] < 8) continue;

		// cut it
		v = verts + i;
		for (j = 0; j < numverts; j++, v += 3)
			dist[j] = *v - m;

		// wrap cases
		dist[j] = dist[0];
		v -= i;
		VectorCopy (verts, v);

		f = b = 0;
		v = verts;

		for (j = 0; j < numverts; j++, v += 3)
		{
			if (dist[j] >= 0)
			{
				VectorCopy (v, front[f]);
				f++;
			}

			if (dist[j] <= 0)
			{
				VectorCopy (v, back[b]);
				b++;
			}

			if (dist[j] == 0 || dist[j + 1] == 0) continue;
			if ((dist[j] > 0) != (dist[j + 1] > 0))
			{
				// clip point
				frac = dist[j] / (dist[j] - dist[j + 1]);

				for (k = 0; k < 3; k++)
					front[f][k] = back[b][k] = v[k] + frac * (v[3 + k] - v[k]);

				f++;
				b++;
			}
		}

		SubdividePolygon (f, front[0]);
		SubdividePolygon (b, back[0]);

		return;
	}

	poly = Hunk_Alloc (sizeof (glpoly_t));
	poly->next = warpface->polys;
	warpface->polys = poly;
	poly->numverts = numverts;

	poly->saindex = worldverts;
	poly->verts = &surfarray[worldverts * VERTEXSIZE];

	for (i = 0, v = poly->verts; i < numverts; i++, verts += 3, v += VERTEXSIZE)
	{
		v[0] = verts[0];
		v[1] = verts[1];
		v[2] = verts[2];

		// hack - use 5 for the index of colour, cos we'll also want to use 9 for fog on the same verts
		v[5] = 1;
		v[6] = 1;
		v[7] = 1;

		v[9] = warpface->fogColour[0];
		v[10] = warpface->fogColour[1];
		v[11] = warpface->fogColour[2];

		worldverts++;

		if (worldverts > MAX_MAP_VERTS * 2) Sys_Error ("Worldverts > %i", MAX_MAP_VERTS * 2);
	}

	warpface->numPolys ++;
}

/*
================
GL_SubdivideSurface

Breaks a polygon up along axial 64 unit
boundaries so that turbulent and sky warps
can be done reasonably.
================
*/
void R_PolyMidPoint (glpoly_t *p);

void GL_SubdivideSurface (msurface_t *surf)
{
	vec3_t		verts[64];
	int			numverts;
	int			i;
	int			lindex;
	float		*vec;
	glpoly_t	*p;
	//texture_t	*t;

	warpface = surf;

	// convert edges back to a normal polygon
	numverts = 0;

	// fill in some fog data here
	surf->fogAlphaMax = 0.9;
	surf->fogBase = 0;
	surf->fogRange = 700;

	surf->fogColour[0] = surf->texinfo->texture->uwater_fog[0];
	surf->fogColour[1] = surf->texinfo->texture->uwater_fog[1];
	surf->fogColour[2] = surf->texinfo->texture->uwater_fog[2];

	for (i = 0; i < surf->numedges; i++)
	{
		lindex = loadmodel->surfedges[surf->firstedge + i];

		if (lindex > 0)
			vec = loadmodel->vertexes[loadmodel->edges[lindex].v[0]].position;
		else
			vec = loadmodel->vertexes[loadmodel->edges[-lindex].v[1]].position;

		VectorCopy (vec, verts[numverts]);
		numverts++;
	}

	warpface->numPolys = 0;

	SubdividePolygon (numverts, verts[0]);

	for (p = surf->polys; p; p = p->next)
	{
		R_PolyMidPoint (p);
	}
}


/*
======================================================================================

									WATER WARP

======================================================================================
*/


/*
=============
R_DrawWaterChain

Does a water warp on the pre-fragmented glpoly_t chain
=============
*/
extern float *brushVerts;
extern float VCount;
void R_SmokeEffect (vec3_t org);
void R_SmokeEffect2 (vec3_t org);
extern int hazetexture;

__inline void HS_QuadVert (float *v, float *nv, float texs, float text, float height, float alpha)
{
	nv[0] = v[0] + DIST_EPSILON;
	nv[1] = v[1] + DIST_EPSILON;
	nv[2] = v[2] + height;
	nv[3] = texs;
	nv[4] = text + realtime;
	nv[5] = 1;
	nv[6] = 1;
	nv[7] = 1;
	nv[8] = alpha;
	nv += 9;
}

void R_HeatShimmer (msurface_t *heatsurf)
{
	int i;
	float *v;
	float *nv;
	glpoly_t *p;
	int numpolys;
	float TexMatrix[16];

	// turn off backface culling for this
	glDisable (GL_CULL_FACE);

	// change the vertex array pointers
	glVertexPointer (3, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[0]);
	glTexCoordPointer (2, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[3]);
	glColorPointer (4, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[5]);

	// bind the heathaze texture
	glBindTexture (GL_TEXTURE_2D, hazetexture);

	numpolys = 0;
	nv = VArray;

	// goto a clean texture matrix
	glMatrixMode (GL_TEXTURE);
	glPushMatrix ();
	glLoadIdentity ();

	for (p = heatsurf->polys; p; p = p->next)
	{
		for (i = 0, v = p->verts; i < p->numverts; i++, v += VERTEXSIZE)
		{
			numpolys += 4;

			if (i)
			{
				// finish the previous quad
				HS_QuadVert (v, nv, 1, 1, 100, 0);
				HS_QuadVert (v, nv, 1, 0, 0, 0.25);
			}

			// start a new quad
			HS_QuadVert (v, nv, 0, 0, 0, 0.25);
			HS_QuadVert (v, nv, 0, 1, 100, 0);
		}

		// finish the last quad by pointing it back around to the first
		HS_QuadVert (v, nv, 1, 1, 100, 0);
		HS_QuadVert (v, nv, 1, 0, 0, 0.25);
	}

	glDrawArrays (GL_QUADS, 0, numpolys);

	// restore the previous texture matrix
	glPopMatrix ();
	glMatrixMode (GL_MODELVIEW);

	// switch them back
	glVertexPointer (3, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[0]);
	glTexCoordPointer (2, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[13]);
	glColorPointer (4, GL_FLOAT, (VERTEXSIZE * sizeof (GLfloat)), &surfarray[5]);

	// put backface culling back on
	glEnable (GL_CULL_FACE);
}

void R_DrawWaterChain (msurface_t *surf, float alpha)
{
	int i;
	float *v;
	glpoly_t *p;

	for (; surf; surf = surf->lqChain)
	{
		// only fog the underwater side of the surf, unless it's a teleport, in which case don't fog it at all
		if (cl.worldmodel->volumes[surf->volume].caustic && !(surf->flags & SURF_DRAWTELE))
		{
			surf->fogSurfs = cl.worldmodel->fogSurfs;
			cl.worldmodel->fogSurfs = surf;
		}

		for (p = surf->polys; p; p = p->next)
		{
			for (i = 0, v = p->verts; i < p->numverts; i++, v += VERTEXSIZE)
			{
				// fix water warping - derive the texcoords from the verts!!!  mad, but it works!!!
				// warpindex is precalculated in gl_model.c - use the x2 sintable for 8 unit warps
				v[13] = v[surf->warpindex[0]] + sintable[(int) ((v[surf->warpindex[1]] * 0.025 + realtime) * STMult) & STBitMask];
				v[14] = v[surf->warpindex[1]] + sintable[(int) ((v[surf->warpindex[0]] * 0.025 + realtime) * STMult) & STBitMask];

				// hack - use 5 for the index of colour, cos we'll also want to use 9 for fog on the same verts
				v[8] = alpha;
			}

			glDrawArrays (GL_TRIANGLE_FAN, p->saindex, p->numverts);

			//if (surf->flags & SURF_DRAWLAVA) R_SmokeEffect2 (p->midpoint);
		}

		if (surf->flags & SURF_DRAWLAVA)
		{
			// do a heathaze effect
			//R_HeatShimmer (surf);

			// rebind the original lava texture (only if necessary)
			//if (surf->lqChain) glBindTexture (GL_TEXTURE_2D, surf->texinfo->texture->gl_texture->texnum);
		}
	}
}


/*
==============
R_DrawSphere

Draw a sphere!!!  The verts and texcoords are precalculated for extra efficiency.  The sphere is put into
a display list to reduce overhead even further.
==============
*/
float skytexes[440];
float skyverts[660];

int spherelist = -1;

void R_DrawSphere (void)
{
	if (spherelist == -1)
	{
		int i;
		int j;

		int vertspos = 0;
		int texespos = 0;

		// build the sphere display list
		spherelist = glGenLists (1);

		glNewList (spherelist, GL_COMPILE);

		for (i = 0; i < 10; i++)
		{
			glBegin (GL_TRIANGLE_STRIP);

			for (j = 0; j <= 10; j++)
			{
				glTexCoord2fv (&skytexes[texespos]);
				glVertex3fv (&skyverts[vertspos]);

				texespos += 2;
				vertspos += 3;

				glTexCoord2fv (&skytexes[texespos]);
				glVertex3fv (&skyverts[vertspos]);

				texespos += 2;
				vertspos += 3;
			}

			glEnd ();
		}

		glEndList ();
	}
}


/*
==============
R_InitSkyChain

Initialize the sky chain arrays
==============
*/
void R_InitSkyChain (void)
{
	float drho = 0.3141592653589;
	float dtheta = 0.6283185307180;

	float ds = 0.1;
	float dt = 0.1;

	float t = 1.0f;	
	float s = 0.0f;

	int i;
	int j;

	int vertspos = 0;
	int texespos = 0;

	for (i = 0; i < 10; i++)
	{
		float rho = (float) i * drho;
		float srho = (float) (sin (rho));
		float crho = (float) (cos (rho));
		float srhodrho = (float) (sin (rho + drho));
		float crhodrho = (float) (cos (rho + drho));

		s = 0.0f;

		for (j = 0; j <= 10; j++)
		{
			float theta = (j == 10) ? 0.0f : j * dtheta;
			float stheta = (float) (-sin( theta));
			float ctheta = (float) (cos (theta));

			skytexes[texespos++] = s * 2;
			skytexes[texespos++] = t * 2;

			skyverts[vertspos++] = stheta * srho * 4096.0;
			skyverts[vertspos++] = ctheta * srho * 4096.0;
			skyverts[vertspos++] = crho * 4096.0;

			skytexes[texespos++] = s * 2;
			skytexes[texespos++] = (t - dt) * 2;

			skyverts[vertspos++] = stheta * srhodrho * 4096.0;
			skyverts[vertspos++] = ctheta * srhodrho * 4096.0;
			skyverts[vertspos++] = crhodrho * 4096.0;

			s += ds;
		}

		t -= dt;
	}
}


void R_DrawSkyLayer (int TextureNum, float SkyScale, float SkyRotate)
{
	// go to a new matrix
	glPushMatrix ();

	// center it on the players position
	glTranslatef (r_origin[0], r_origin[1], r_origin[2]);

	// flatten the sphere
	glScalef (SkyScale, SkyScale, SkyScale / 2);

	// orient it so that the poles are unobtrusive
	glRotatef (-90, 1, 0, 0);

	// make it not always at right angles to the player
	glRotatef (-22, 0 ,1, 0);

	// rotate it around the poles
	glRotatef (SkyRotate, 0, 0, 1);

	// solid sky texture
	glBindTexture (GL_TEXTURE_2D, TextureNum);

	// draw the sphere
	glCallList (spherelist);

	// restore the previous matrix
	glPopMatrix ();
}


/*
=================
R_DrawSkyChain

Hot damn Jethro, this sho' is purty!!!
=================
*/
void R_DrawSkyChain (void)
{
	// sky scaling
	float fullscale = r_farclip.value / 4096;
	float halfscale = r_farclip.value / 8192;
	float reducedfull = (r_farclip.value / 4096) * 0.9;
	float reducedhalf = (r_farclip.value / 8192) * 0.9;

	// turn on fogging.  fog looks good on the skies - it gives them a more 
	// "airy" far-away look, and has the knock-on effect of preventing the 
	// old "texture distortion at the poles" problem.
	glFogi (GL_FOG_MODE, GL_LINEAR);
	glFogfv (GL_FOG_COLOR, sky_fog);
	glFogf (GL_FOG_START, 5);

	// must tweak the fog end too!!!
	glFogf (GL_FOG_END, r_farclip.value * 0.75);
	glEnable (GL_FOG);

	// sky texture scaling
	// previous releases made a tiled version of the sky texture.  here i just shrink it using the
	// texture matrix, for much the same effect
	glMatrixMode (GL_TEXTURE);
	glLoadIdentity ();
	glScalef (2, 1, 1);
	glMatrixMode (GL_MODELVIEW);

	// background
	// ==========
	// go to a new matrix
	glPushMatrix ();

	// center it on the players position
	glTranslatef (r_origin[0], r_origin[1], r_origin[2]);

	// flatten the sphere
	glScalef (fullscale, fullscale, halfscale);

	// orient it so that the poles are unobtrusive
	glRotatef (-90, 1, 0, 0);

	// make it not always at right angles to the player
	glRotatef (-22, 0 ,1, 0);

	// rotate it around the poles
	glRotatef (-rotateBack, 0, 0, 1);

	// solid sky texture
	glBindTexture (GL_TEXTURE_2D, solidskytexture);

	// draw the sphere
	glCallList (spherelist);

	// restore the previous matrix
	glPopMatrix ();

	// foreground
	// ==========
	// go to a new matrix
	glPushMatrix ();

	// center it on the players position
	glTranslatef (r_origin[0], r_origin[1], r_origin[2]);

	// flatten the sphere and shrink it a little - the reduced scale prevents artefacts appearing when
	// corners on the skysphere may potentially overlap
	glScalef (reducedfull, reducedfull, reducedhalf);

	// orient it so that the poles are unobtrusive
	glRotatef (-90, 1, 0, 0);

	// make it not always at right angles to the player
	glRotatef (-22, 0 ,1, 0);

	// rotate it around the poles
	glRotatef (-rotateFore, 0, 0, 1);

	// blend mode
	glEnable (GL_BLEND);

	// alpha sky texture
	glBindTexture (GL_TEXTURE_2D, alphaskytexture);

	// draw the sphere
	glCallList (spherelist);

	// back to normal mode
	glDisable (GL_BLEND);

	// restore the previous matrix
	glPopMatrix ();

	// turn off fog
	glDisable (GL_FOG);

	glMatrixMode (GL_TEXTURE);
	glLoadIdentity ();
	glMatrixMode (GL_MODELVIEW);

	/*
	test code
	glPushMatrix ();
	// center it on the players position
	glTranslatef (r_origin[0], r_origin[1], r_origin[2]);
	glTranslatef (0, 200, 0);
	glScalef (0.025, 0.025, 0.025);
	glCallList (spherelist);
	glPopMatrix ();
	*/
}


/*
=============
R_InitSky

A sky texture is 256*128, with the right side being a masked overlay
==============
*/
void R_InitSky (texture_t *mt)
{
	int			i, j, p;
	byte		*src;
	unsigned	transpix;
	int			r, g, b;
	unsigned	*rgba;
	unsigned	topalpha;
	int			div;

	unsigned	*trans;

	// initialize our pointers
	trans = malloc (128 * 128 * 4);

	// -----------------
	// solid sky texture
	// -----------------
	src = (byte *)mt + mt->offsets[0];

	// make an average value for the back to avoid
	// a fringe on the top level
	r = g = b = 0;

	for (i=0 ; i<128 ; i++)
	{
		for (j=0 ; j<128 ; j++)
		{
			p = src[i*256 + j + 128];
			rgba = &d_8to24table[p];
			trans[(i*128) + j] = *rgba;

			r += ((byte *)rgba)[0];
			g += ((byte *)rgba)[1];
			b += ((byte *)rgba)[2];
		}
	}

	((byte *)&transpix)[0] = r/(128*128);
	((byte *)&transpix)[1] = g/(128*128);
	((byte *)&transpix)[2] = b/(128*128);
	((byte *)&transpix)[3] = 0;

	if (!solidskytexture) solidskytexture = texture_extension_number++;

	glBindTexture (GL_TEXTURE_2D, solidskytexture);

	glTexImage2D (GL_TEXTURE_2D, 0, GL_RGBA, 128, 128, 0, GL_RGBA, GL_UNSIGNED_BYTE, trans);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	// -----------------
	// alpha sky texture
	// -----------------
	// get another average cos the bottom layer average can be too dark
	div = r = g = b = 0;

	for (i=0 ; i<128 ; i++)
	{
		for (j=0 ; j<128 ; j++)
		{
			p = src[i*256 + j];

			if (p == 0)
				topalpha = transpix;
			else
			{
				rgba = &d_8to24table[p];

				g += ((byte *)rgba)[1];

				r += ((byte *)rgba)[0];
				b += ((byte *)rgba)[2];

				div++;

				topalpha = d_8to24table[p];
			}

			((byte *)&topalpha)[3] = ((byte *)&topalpha)[3] / 2;

			trans[(i*128) + j] = topalpha;
		}
	}

	if (!alphaskytexture) alphaskytexture = texture_extension_number++;

	glBindTexture (GL_TEXTURE_2D, alphaskytexture);

	glTexImage2D (GL_TEXTURE_2D, 0, GL_RGBA, 128, 128, 0, GL_RGBA, GL_UNSIGNED_BYTE, trans);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	// get fog colours for the sky - we halve these
	sky_fog[0] = ((float)r / (float)div) / 512.0;
	sky_fog[1] = ((float)g / (float)div) / 512.0;
	sky_fog[2] = ((float)b / (float)div) / 512.0;
	sky_fog[3] = 0.1;

	// free the used memory
	free (trans);

	// recalc the sphere display list - delete the original one first otherwise
	// we'll compound the display list memory leak problem
	if (spherelist != -1) glDeleteLists (spherelist, 1);
	spherelist = -1;

	// save out 6 slots for the skybox textures
	if (!skyboxtextures)
	{
		skyboxtextures = texture_extension_number;
		texture_extension_number += 6;
	}
}



/*
================================================================================================================

												SKYBOXES

	I took most of this code from the old MHQuake 6 source.  It was an awful shock to go back there and look
	at some of the cringe-inducing embarrassing crap I had written back then.  Some of it was extremely good:
	the idea of storing QuadVecs in a static 3D array and then multiplying by the farclip was fantastic, and
	drawing the full skybox each frame rather than laboriously calculating what should and shouldn't be drawn
	was spot on.  The woefully botched attempt at doing rotating skyboxes was something I'd rather not
	revisit though.  The texture loader wasn't too good, and then the multitude of hacks just to get mirrors
	working... sheesh!!!  Couple that with an almost complete inability or unwillingness to put in the effort
	to try understand what ID's original code was doing, and you have disaster marching to your door pretty
	quickly.

	I may make this source available some day, just as a lesson on how messy things can get if you're
	prepared to take things as far as you can without really knowing what you're doing...

================================================================================================================
*/


/*
===================
R_LoadSkyBox

Load a skybox.  This gets messy cos skyboxes have been known to be in /gfx, /gfx/env OR /env, can be any texture 
format, can use either of the suffix formats, and there may or may not be underscores!!!  Holy multiple 
permutations Batman!!!
===================
*/
qboolean LoadTGA (FILE *f, qboolean mipmap);
qboolean LoadBMP (FILE *f, qboolean mipmap);
qboolean LoadJPG (FILE *f, qboolean mipmap);
qboolean LoadPCX (FILE *f, qboolean mipmap);

qboolean R_LoadSkyBox (char *sbNameBase)
{
	FILE *f;
	int i;
	int PermDir;
	int PermUS;
	int PermSuf;
	int PermExt;

	// full skybox suffixes
	char *sbSuf1[6] = {"rt", "bk", "lf", "ft", "up", "dn"};

	// used by half life.  why does that make me feel that i need a shower?
	char *sbSuf2[6] = {"r", "b", "l", "f", "u", "d"};

	// skybox location and name permutations
	// i've put the most likely first so we're more likely to get a hit earlier on
	char *sbPermSuf[2] = {"rt", "r"};
	char *sbPermUS[2] = {"_", ""};
	char *sbPermDir[3] = {"gfx/env", "env", "gfx"};
	char *sbPermExt[4] = {"tga", "bmp", "jpg", "pcx"};

	// store the name out so we can cut down on the old va madness
	char sbNameFound[MAX_OSPATH];

	// store a pointer to the texture loader
	qboolean (*TextureLoader) ();

	// store a pointer to the active suffix
	char **activeSuffix;

	// we could get a NULL passed here from R_NewMap
	//if (!sbNameBase) return false;
	// Mod_FindWorldSpawnEntry never returns NULL these days, so...
	if (sbNameBase[0] == '\0') return false;

	// try everything until *something* works...
	for (PermDir = 0; PermDir < 3; PermDir++)
	{
		for (PermSuf = 0; PermSuf < 2; PermSuf++)
		{
			for (PermUS = 0; PermUS < 2; PermUS++)
			{
				for (PermExt = 0; PermExt < 4; PermExt++)
				{
					// we only try the first skybox in the order this time - i.e. the right image
					COM_FOpenFile (va ("%s/%s%s%s.%s", sbPermDir[PermDir], sbNameBase, sbPermUS[PermUS], sbPermSuf[PermSuf], sbPermExt[PermExt]), &f);

					if (f)
					{
						// wooo!  got one!!!  find me a UFO and get the fuck outa here!!!
						fclose (f);
						goto Got_Me_A_Sky_Box;
					}
				}
			}
		}
	}

	Con_DPrintf ("No skybox found\n");

	return false;

Got_Me_A_Sky_Box:;
	// and now load it - this is also messy cos we have to decode the current loop values from above into
	// a valid name and texture loading function.  I've used more pointers here to keep it as clean as I can.
	// first of all, make the base name again from the original base name we passed into this function plus
	// the directory location and the underscore indicator.
	sprintf (sbNameFound, "%s/%s%s", sbPermDir[PermDir], sbNameBase, sbPermUS[PermUS]);

	// now set the correct texture loader
	switch (PermExt)
	{
	case 0:
		TextureLoader = LoadTGA;
		break;
	case 1:
		TextureLoader = LoadBMP;
		break;
	case 2:
		TextureLoader = LoadJPG;
		break;
	case 3:
		TextureLoader = LoadPCX;
		break;
	}

	// now set the suffix we will use
	switch (PermSuf)
	{
	case 0:
		activeSuffix = sbSuf1;
		break;
	case 1:
		activeSuffix = sbSuf2;
		break;
	}

	// FINALLY...!!!  We can start loading the textures.  Of course, not all textures may be present, and
	// the naming scheme may be followed even if they are, in which case we must continue treading carefully
	for (i = 0; i < 6; i++)
	{
		Con_DPrintf ("Loading... %s%s.%s\n", sbNameFound, activeSuffix[i], sbPermExt[PermExt]);

		COM_FOpenFile (va ("%s%s.%s", sbNameFound, activeSuffix[i], sbPermExt[PermExt]), &f);

		// woops!!!  incomplete skybox set
		if (!f)
		{
			Con_DPrintf ("Incomplete skybox set for: %s\n\n", sbNameFound);
			return false;
		}

		glBindTexture (GL_TEXTURE_2D, skyboxtextures + i);

		// we may even have a bad image file here!!!
		if (!(TextureLoader (f, false)))
		{
			Con_DPrintf ("Bad Image file for: %s%s.%s\n", sbNameFound, activeSuffix[i], sbPermExt[PermExt]);
			fclose (f);
			return false;
		}

		fclose (f);
	}

	// did I say "finally" above?  Well, FINALLY!!!!!!!!!!!!!!!!!!!!!!!!!!!!! now.
	// we have us a viable skybox.
	Con_DPrintf ("Found SkyBox in %s\n\n", sbNameFound);

	return true;
}


extern qboolean sbOK;

void R_Loadsky_f (void)
{
	if (Cmd_Argc () != 2)
	{
		Con_Printf ("Loadsky <skyboxname>: loads a skybox\n");

		// don;t be destructive to any previous skybox state
		return;
	}

	if ((Cmd_Argv (1))[0] == '0')
	{
		// unload skybox
		sbOK = false;
		return;
	}

	if (!R_LoadSkyBox (Cmd_Argv (1)))
	{
		Con_Printf ("Couldn't load %s\n", Cmd_Argv (1));

		// don;t be destructive to any previous skybox state
		return;
	}

	sbOK = true;
}


// fast vertex calculation
float QuadVecs[4][6][3] =
{
	{
		{1.0, 1.0, -1.0},
		{-1.0, 1.0, -1.0},
		{-1.0, -1.0, -1.0},
		{1.0, -1.0, -1.0},
		{1.0, 1.0, 1.0},
		{-1.0, 1.0, -1.0}
	},
	{
		{1.0, 1.0, 1.0},
		{-1.0, 1.0, 1.0},
		{-1.0, -1.0, 1.0},
		{1.0, -1.0, 1.0},
		{-1.0, 1.0, 1.0},
		{1.0, 1.0, -1.0}
	},
	{
		{1.0, -1.0, 1.0},
		{1.0, 1.0, 1.0},
		{-1.0, 1.0, 1.0},
		{-1.0, -1.0, 1.0},
		{-1.0, -1.0, 1.0},
		{1.0, -1.0, -1.0}
	},
	{
		{1.0, -1.0, -1.0},
		{1.0, 1.0, -1.0},
		{-1.0, 1.0, -1.0},
		{-1.0, -1.0, -1.0},
		{1.0, -1.0, 1.0},
		{-1.0, -1.0, -1.0}
	}
};


float sky_mult;

int skytexorder[6] = {0, 2, 1, 3, 4, 5};

void MakeSkyVecs (int quad, float s, float t, int axis)
{
	vec3_t v;

	v[0] = (QuadVecs[quad][axis][0] * sky_mult) + r_origin[0];
	v[1] = (QuadVecs[quad][axis][1] * sky_mult) + r_origin[1];
	v[2] = (QuadVecs[quad][axis][2] * sky_mult) + r_origin[2];

	glTexCoord2f (s, t);
	glVertex3fv (v);
}


void R_DrawSkyBox (void)
{
	int i;

	// cos we're effectively dealing with a kinda "spherical" far clipping type thing here, edges and corners
	// will get rounded off if we don;t do this...
	sky_mult = r_farclip.value / 1.75;

	for (i = 0; i < 6; i++)
	{
		glBindTexture (GL_TEXTURE_2D, skyboxtextures + skytexorder[i]);

		glBegin (GL_QUADS);

		MakeSkyVecs (0, 0.001953, 0.998047, skytexorder[i]);
		MakeSkyVecs (1, 0.001953, 0.001953, skytexorder[i]);
		MakeSkyVecs (2, 0.998047, 0.001953, skytexorder[i]);
		MakeSkyVecs (3, 0.998047, 0.998047, skytexorder[i]);

		glEnd ();
	}
}


void R_PickSkyDrawer (void)
{
	if (sbOK)
		R_DrawSkyBox ();
	else R_DrawSkyChain ();
}

