/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// sbar.c -- status bar code

#include "quakedef.h"
#include "glext.h"


int			sb_updates;		// if >= vid.numpages, no update needed

#define STAT_MINUS		10	// num frame for '-' stats digit
qpic_t		*sb_nums[2][11];
qpic_t		*sb_colon, *sb_slash;

qboolean	sb_showscores;

int			sb_lines;			// scan lines to draw

void Sbar_MiniDeathmatchOverlay (void);
void Sbar_DeathmatchOverlay (void);
void M_DrawPic (int x, int y, qpic_t *pic);
void M_DrawTextBoxFreeAlpha (int x, int y, int width, int lines, float alpha);


extern int addLeafs;
extern int addSurfs;

extern qboolean scr_drawloading;

/*
===============
Sbar_ShowScores

Tab key down
===============
*/
void Sbar_ShowScores (void)
{
	// toggle on/off rather than only on when down
	sb_showscores = !sb_showscores;

	sb_updates = 0;
}

/*
===============
Sbar_DontShowScores

Tab key up
===============
*/
void Sbar_DontShowScores (void)
{
}

/*
===============
Sbar_Changed
===============
*/
void Sbar_Changed (void)
{
	sb_updates = 0;	// update next frame
}



/*
===============
GL Quake crosshair

Drawn from the status bar so we can properly ignore it if we have the axe.
The crosshair is moved down about 20 pixels to better approximate where the
shot is going to hit.
===============
*/
extern cvar_t crosshair;

void Draw_XHair (int basex, int basey, int size, float r, float g, float b)
{
	// make it green
	glColor3f (r, g, b);

	glDepthMask (GL_FALSE);
	glEnable (GL_BLEND);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE);
	glDisable (GL_ALPHA_TEST);

	Cvar_SetValueDirect (&crosshair, crosshair.value);
	Cvar_Bound (&crosshair, 0, 7);

	glBindTexture (GL_TEXTURE_2D, crosshairfont + (int) crosshair.value);

	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	glBegin (GL_QUADS);

	glTexCoord2f (0, 0);
	glVertex2f (basex - size / 2, basey - size / 2);

	glTexCoord2f (0, 1);
	glVertex2f (basex - size / 2, basey + size / 2);

	glTexCoord2f (1, 1);
	glVertex2f (basex + size / 2, basey + size / 2);

	glTexCoord2f (1, 0);
	glVertex2f (basex + size / 2, basey - size / 2);

	glEnd ();

	glColor3f (1, 1, 1);

	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDisable (GL_BLEND);
	glDepthMask (GL_TRUE);
	glEnable (GL_ALPHA_TEST);
}


void Draw_XHairWrapper (int basex, int basey, int size)
{
	switch ((int) crosshaircolour.value)
	{
	case 1:
		Draw_XHair (basex, basey, size, 1, 0, 0);
		break;

	case 2:
		Draw_XHair (basex, basey, size, 1, 1, 0);
		break;

	case 3:
		Draw_XHair (basex, basey, size, 0, 1, 0);
		break;

	case 4:
		Draw_XHair (basex, basey, size, 0, 1, 1);
		break;

	case 5:
		Draw_XHair (basex, basey, size, 0, 0, 1);
		break;

	case 6:
		Draw_XHair (basex, basey, size, 1, 0, 1);
		break;

	case 0:
	default:
		crosshaircolour.value = 0;
		Draw_XHair (basex, basey, size, 1, 1, 1);
		break;
	}
}


/*
===============
Sbar_Init
===============
*/
void Sbar_Init (void)
{
	int		i;

	for (i=0 ; i<10 ; i++)
	{
		sb_nums[0][i] = Draw_PicFromWad (va("num_%i",i));
		sb_nums[1][i] = Draw_PicFromWad (va("anum_%i",i));
	}

	sb_nums[0][10] = Draw_PicFromWad ("num_minus");
	sb_nums[1][10] = Draw_PicFromWad ("anum_minus");

	sb_colon = Draw_PicFromWad ("num_colon");
	sb_slash = Draw_PicFromWad ("num_slash");

	Cmd_AddCommand ("+showscores", Sbar_ShowScores);
	Cmd_AddCommand ("-showscores", Sbar_DontShowScores);
}


/*
=============
Sbar_itoa
=============
*/
int Sbar_itoa (int num, char *buf)
{
	char	*str;
	int		pow10;
	int		dig;

	str = buf;

	if (num < 0)
	{
		*str++ = '-';
		num = -num;
	}

	for (pow10 = 10 ; num >= pow10 ; pow10 *= 10)
	;

	do
	{
		pow10 /= 10;
		dig = num/pow10;
		*str++ = '0'+dig;
		num -= dig*pow10;
	} while (pow10 != 1);

	*str = 0;

	return str-buf;
}


//=============================================================================

// drawing routines are relative to the status bar location

/*
=============
Sbar_DrawTransPic
=============
*/
void Sbar_DrawTransPic (int x, int y, qpic_t *pic)
{
	if (cl.gametype == GAME_DEATHMATCH)
		Draw_TransPic (x /*+ ((vid.width - 320)>>1)*/, y + (vid.height-SBAR_HEIGHT), pic);
	else
		Draw_TransPic (x + ((vid.width - 320)>>1), y + (vid.height-SBAR_HEIGHT), pic);
}

/*
================
Sbar_DrawCharacter

Draws one solid graphics character
================
*/
void Sbar_DrawCharacter (int x, int y, int num)
{
	if (cl.gametype == GAME_DEATHMATCH)
		Draw_Character ( x /*+ ((vid.width - 320)>>1) */ + 4 , y + vid.height-SBAR_HEIGHT, num);
	else
		Draw_Character ( x + ((vid.width - 320)>>1) + 4 , y + vid.height-SBAR_HEIGHT, num);
}

/*
================
Sbar_DrawString
================
*/
void Sbar_DrawString (int x, int y, char *str)
{
	if (cl.gametype == GAME_DEATHMATCH)
		Draw_String (x /*+ ((vid.width - 320)>>1)*/, y+ vid.height-SBAR_HEIGHT, str);
	else
		Draw_String (x + ((vid.width - 320)>>1), y+ vid.height-SBAR_HEIGHT, str);
}

/*
=============
Sbar_DrawNum
=============
*/
void Sbar_DrawNum (int x, int y, int num, int digits, int color)
{
	char			str[12];
	char			*ptr;
	int				l, frame;

	l = Sbar_itoa (num, str);
	ptr = str;
	if (l > digits)
		ptr += (l-digits);
	if (l < digits)
		x += (digits-l)*24;

	while (*ptr)
	{
		if (*ptr == '-')
			frame = STAT_MINUS;
		else
			frame = *ptr -'0';

		Sbar_DrawTransPic (x,y,sb_nums[color][frame]);
		x += 24;
		ptr++;
	}
}


void HUD_DrawNum (int x, int y, int num, int digits, int color)
{
	char			str[12];
	char			*ptr;
	int				l, frame;

	l = Sbar_itoa (num, str);

	ptr = str;

	if (l > digits) ptr += (l-digits);
	if (l < digits) x += (digits-l) * 24;

	if (num < -99)
		num = -99;
	if (num < 0)
		;
	else if (num < 10) x -= 24;
	else if (num < 100) x -= 12;

	while (*ptr)
	{
		if (*ptr == '-')
			frame = STAT_MINUS;
		else
			frame = *ptr -'0';

		Draw_Pic (x, y, sb_nums[color][frame]);

		x += 24;
		ptr++;
	}
}


//=============================================================================

int		fragsort[MAX_SCOREBOARD];

char	scoreboardtext[MAX_SCOREBOARD][20];
int		scoreboardtop[MAX_SCOREBOARD];
int		scoreboardbottom[MAX_SCOREBOARD];
int		scoreboardcount[MAX_SCOREBOARD];
int		scoreboardlines;

/*
===============
Sbar_SortFrags
===============
*/
void Sbar_SortFrags (void)
{
	int		i, j, k;

// sort by frags
	scoreboardlines = 0;
	for (i=0 ; i<cl.maxclients ; i++)
	{
		if (cl.scores[i].name[0])
		{
			fragsort[scoreboardlines] = i;
			scoreboardlines++;
		}
	}

	for (i=0 ; i<scoreboardlines ; i++)
		for (j=0 ; j<scoreboardlines-1-i ; j++)
			if (cl.scores[fragsort[j]].frags < cl.scores[fragsort[j+1]].frags)
			{
				k = fragsort[j];
				fragsort[j] = fragsort[j+1];
				fragsort[j+1] = k;
			}
}

int	Sbar_ColorForMap (int m)
{
	return m < 128 ? m + 8 : m + 8;
}

/*
===============
Sbar_UpdateScoreboard
===============
*/
void Sbar_UpdateScoreboard (void)
{
	int		i, k;
	int		top, bottom;
	scoreboard_t	*s;

	Sbar_SortFrags ();

// draw the text
	memset (scoreboardtext, 0, sizeof(scoreboardtext));

	for (i=0 ; i<scoreboardlines; i++)
	{
		k = fragsort[i];
		s = &cl.scores[k];
		sprintf (&scoreboardtext[i][1], "%3i %s", s->frags, s->name);

		top = s->colors & 0xf0;
		bottom = (s->colors & 15) <<4;
		scoreboardtop[i] = Sbar_ColorForMap (top);
		scoreboardbottom[i] = Sbar_ColorForMap (bottom);
	}
}



/*
===============
Sbar_SoloScoreboard
===============
*/
void Sbar_SoloScoreboard (void)
{
	char	str[80];
	int		minutes, seconds, tens, units;
	int		l;

#if 0
	sprintf (str,"Monsters:%3i /%3i", cl.stats[STAT_MONSTERS], cl.stats[STAT_TOTALMONSTERS]);
	Sbar_DrawString (8, 4, str);

	sprintf (str,"Secrets :%3i /%3i", cl.stats[STAT_SECRETS], cl.stats[STAT_TOTALSECRETS]);
	Sbar_DrawString (8, 12, str);

	// time
	minutes = cl.time / 60;
	seconds = cl.time - 60*minutes;
	tens = seconds / 10;
	units = seconds - 10*tens;
	sprintf (str,"Time :%3i:%i%i", minutes, tens, units);
	Sbar_DrawString (184, 4, str);

	// draw level name
	l = strlen (cl.levelname);
	Sbar_DrawString (232 - l*4, 12, cl.levelname);
#else
	// map name
	sprintf (str, "%s", cl.worldmodel->name);

	for (l = 0; ; l++)
	{
		str[l] = str[l + 5];

		if (str[l] == '.')
		{
			str[l] = '\0';
			break;
		}
	}

	Draw_String (304, 120, str);

	// level name
	l = strlen (cl.levelname);
	Draw_String (320 - l * 4, 130, cl.levelname);

	// time
	minutes = cl.time / 60;
	seconds = cl.time - 60 * minutes;
	sprintf (str,"Time:    %3i:%02i", minutes, seconds);
	Draw_String (260, 150, str);

	// secrets
	sprintf (str,"Secrets: %3i/%i", cl.stats[STAT_SECRETS], cl.stats[STAT_TOTALSECRETS]);
	Draw_String (260, 160, str);

	// monsters
	sprintf (str,"Kills:   %3i/%i", cl.stats[STAT_MONSTERS], cl.stats[STAT_TOTALMONSTERS]);
	Draw_String (260, 170, str);
#endif
}

/*
===============
Sbar_DrawScoreboard
===============
*/
void Sbar_DrawScoreboard (void)
{
	Sbar_SoloScoreboard ();

	if (cl.gametype == GAME_DEATHMATCH)
		Sbar_DeathmatchOverlay ();
#if 0
	int		i, j, c;
	int		x, y;
	int		l;
	int		top, bottom;
	scoreboard_t	*s;

	if (cl.gametype != GAME_DEATHMATCH)
	{
		Sbar_SoloScoreboard ();
		return;
	}

	Sbar_UpdateScoreboard ();

	l = scoreboardlines <= 6 ? scoreboardlines : 6;

	for (i=0 ; i<l ; i++)
	{
		x = 20*(i&1);
		y = i/2 * 8;

		s = &cl.scores[fragsort[i]];
		if (!s->name[0])
			continue;

	// draw background
		top = s->colors & 0xf0;
		bottom = (s->colors & 15)<<4;
		top = Sbar_ColorForMap (top);
		bottom = Sbar_ColorForMap (bottom);

		Draw_Fill ( x*8+10 + ((vid.width - 320)>>1), y + vid.height - SBAR_HEIGHT, 28, 4, top);
		Draw_Fill ( x*8+10 + ((vid.width - 320)>>1), y+4 + vid.height - SBAR_HEIGHT, 28, 4, bottom);

	// draw text
		for (j=0 ; j<20 ; j++)
		{
			c = scoreboardtext[i][j];
			if (c == 0 || c == ' ')
				continue;
			Sbar_DrawCharacter ( (x+j)*8, y, c);
		}
	}
#endif
}

//=============================================================================

/*
===============
Sbar_DrawInventory
===============
*/
void Sbar_DrawInventory (void)
{
}

//=============================================================================

/*
===============
Sbar_DrawFrags
===============
*/
void Sbar_DrawFrags (void)
{
	int				i, k, l;
	int				top, bottom;
	int				x, y, f;
	int				xofs;
	char			num[12];
	scoreboard_t	*s;

	Sbar_SortFrags ();

// draw the text
	l = scoreboardlines <= 4 ? scoreboardlines : 4;

	x = 23;
	if (cl.gametype == GAME_DEATHMATCH)
		xofs = 0;
	else
		xofs = (vid.width - 320)>>1;
	y = vid.height - SBAR_HEIGHT - 23;

	for (i=0 ; i<l ; i++)
	{
		k = fragsort[i];
		s = &cl.scores[k];
		if (!s->name[0])
			continue;

	// draw background
		top = s->colors & 0xf0;
		bottom = (s->colors & 15)<<4;
		top = Sbar_ColorForMap (top);
		bottom = Sbar_ColorForMap (bottom);

		Draw_Fill (xofs + x*8 + 10, y, 28, 4, top);
		Draw_Fill (xofs + x*8 + 10, y+4, 28, 3, bottom);

	// draw number
		f = s->frags;
		sprintf (num, "%3i",f);

		Sbar_DrawCharacter ( (x+1)*8 , -24, num[0]);
		Sbar_DrawCharacter ( (x+2)*8 , -24, num[1]);
		Sbar_DrawCharacter ( (x+3)*8 , -24, num[2]);

		if (k == cl.viewentity - 1)
		{
			Sbar_DrawCharacter (x*8+2, -24, 16);
			Sbar_DrawCharacter ( (x+4)*8-4, -24, 17);
		}
		x+=4;
	}
}

//=============================================================================


void Sbar_DrawArmor (void)
{
	qpic_t *armor_pic;
	int Origin[2];
	float Zoom;
	int ArmourValue;
	int colour;

	Origin[0] = 47;
	Origin[1] = 480 - 77;
	Zoom = 1.0;

	if (cl.items & IT_INVULNERABILITY)
	{
		ArmourValue = 666;
		colour = 1;
	}
	else if (!cl.stats[STAT_ARMOR])
		return;
	else
	{
		ArmourValue = cl.stats[STAT_ARMOR];
		colour = (cl.stats[STAT_ARMOR] <= 25);
	}

	HUD_DrawNum (Origin[0], 
				 Origin[1], 
				 ArmourValue, 
				 3, 
				 colour);
}


void Sbar_DrawAmmo (void)
{
	qboolean ammo_draw = false;

	// ammo icon
	if (rogue)
	{
		if (cl.items & (RIT_SHELLS | 
						RIT_NAILS | 
						RIT_ROCKETS | 
						RIT_CELLS | 
						RIT_LAVA_NAILS | 
						RIT_PLASMA_AMMO | 
						RIT_MULTI_ROCKETS))
			ammo_draw = true;
	}
	else
	{
		if (cl.items & (IT_SHELLS | IT_NAILS | IT_ROCKETS | IT_CELLS))
			ammo_draw = true;
	}

	// either we've got the axe, or some funky weapon we don't have a pic for, so just get out
	if (!ammo_draw) return;

	HUD_DrawNum (640 - 123, 480 - 33, cl.stats[STAT_AMMO], 3, cl.stats[STAT_AMMO] <= 10);

	// check for the stupid fucking hammer
	if (hipnotic && cl.stats[STAT_ACTIVEWEAPON] == 128)
	{
		// stupid fucking hammer
		return;
	}

	Draw_XHairWrapper (320, 260, 20);
}


void R_DrawAliasModelSBar (entity_t *e, float Light, int skin, float trans, float scale);

// no g_ for the standard shotgun so use the super shotgun instead - it sucks a little but
// what can ya do???
char *Weaponses[10] =
{
	// normal quake
	"progs/g_shot.mdl",
	"progs/g_shot.mdl",
	"progs/g_nail.mdl",
	"progs/g_nail2.mdl",
	"progs/g_rock.mdl",
	"progs/g_rock2.mdl",
	"progs/g_light.mdl",
	// hipnotic (note - if you have an earlier version of this source it will crash on
	// attempting to load weapon 10 - i forgot the comma after the gl_prox.mdl - d'oh!!!)
	"progs/g_prox.mdl",
	"progs/g_laserg.mdl",
	"progs/g_hammer.mdl",
	// no g_*.mdl files for rogue
};


int ID1YPos[] = {427, 402, 377, 352, 327, 302, 277};
int HIPYPos[] = {427, 402, 377, 352, 327, /* space for proxy gren */ 277, 252, 302 /* proxy gren */, 227, 202};


// globals for positioning of the widgets
// cut down on run-time calculations (...a little...) by only doing them once per frame
float bobjrotate;

model_t *InvWeaps[10] = {NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL};

//					  s   ss  n  sn  g   r   l
//					  0   1   2   3  4   5   6
int weaponrotate[] = {0, -45, 0, 45, 0, -45, 0};

//						s   ss  n  sn  g  r  l   p   lg  sfh 
//					    0   1   2   3  4  5  6   7    8   9
int hipnoticrotate[] = {0, -45, 0, 45, 0, 0, 45, -45, 0, -45};

void DrawInvWeap (int WeapNum)
{
	entity_t e;

	// set up a temp dummy entity to hold the weapon
	memset (&e, 0, sizeof (e));
	e.model = InvWeaps[WeapNum];

	// set origin and angles for any transforms we might want to do
	e.origin[1] = 38;

	if (hipnotic)
	{
		// hipnotic idiocy
		e.origin[2] = HIPYPos[WeapNum];

		// more hipnotic idiocy
		e.angles[1] = hipnoticrotate[WeapNum] + bobjrotate;

		if (WeapNum == 9)
		{
			// stupid fucking hammer
			// how many consecutive base levels can a sane person reasonably be
			// expected to take anyway???
			e.angles[2] = 180;
			e.origin[2] += 55;
		}
	}
	else
	{
		e.origin[2] = ID1YPos[WeapNum];
		e.angles[1] = weaponrotate[WeapNum] + bobjrotate;
	}

	// and draw it
	R_DrawAliasModelSBar (&e, 0.8, 0, 1.0, 0.0);
}


int ID1WeaponNums[] = {1, 2, 4, 8, 16, 32, 64};
int HIPWeaponNums[] = {65536, 8388608, 128};

void HUD_DrawWeapons (void)
{
	int i;

	// rogue has no g_ weapons, we just stick with the classics for that
	for (i = 0; i < 7; i++)
	{
		if (cl.items & ID1WeaponNums[i])
		{
			if (!InvWeaps[i]) continue;

			DrawInvWeap (i);
		}
	}

	// hipnotic extra weapons
	if (hipnotic)
	{
		for (i = 0; i < 3; i++)
		{
			if (cl.items & HIPWeaponNums[i])
			{
				if (!InvWeaps[i + 7]) continue;

				DrawInvWeap (i + 7);
			}
		}

		return;
	}

	// debug testing for hipnotic.  never did like those stupid weapons anyway.
	// stick with the classics i say.
	// Con_Printf ("Using %i\n", cl.stats[STAT_ACTIVEWEAPON]);
}


int ID1CountYPos[] = {15, 41, 67, 93, 119, 145, 171};
int ID1AmmoType[] = {STAT_SHELLS, STAT_SHELLS, STAT_NAILS, STAT_NAILS, STAT_ROCKETS, STAT_ROCKETS, STAT_CELLS};

int HIPCountYPos[] = {145, 223, 249};
int HIPAmmoType[] = {STAT_ROCKETS, STAT_CELLS, STAT_CELLS};

int RogueWeapons[] = {4096, 8192, 16384, 32768, 65536};

void HUD_DrawAmmoCount (void)
{
	int i;
	char num[6];
	int ypos;
	extern int char_texture;

	glBindTexture (GL_TEXTURE_2D, char_texture);

	glBegin (GL_QUADS);

	for (i = 0; i < 7; i++)
	{
		if (cl.items & ID1WeaponNums[i])
		{
			ypos = ID1CountYPos[i];

			if (hipnotic && i > 4) ypos += 26;

			sprintf (num, "%3i", cl.stats[ID1AmmoType[i]]);

			if (num[0] != ' ') Draw_SingleCharacter (52, ypos, 176 + num[0] - '0');
			if (num[1] != ' ') Draw_SingleCharacter (60, ypos, 176 + num[1] - '0');
			if (num[2] != ' ') Draw_SingleCharacter (68, ypos, 176 + num[2] - '0');
		}
	}

	if (hipnotic)
	{
		// at least it's clean enough this time...
		for (i = 0; i < 3; i++)
		{
			if (cl.items & HIPWeaponNums[i])
			{
				sprintf (num, "%3i", cl.stats[HIPAmmoType[i]]);

				if (num[0] != ' ') Draw_SingleCharacter (52, HIPCountYPos[i], 176 + num[0] - '0');
				if (num[1] != ' ') Draw_SingleCharacter (60, HIPCountYPos[i], 176 + num[1] - '0');
				if (num[2] != ' ') Draw_SingleCharacter (68, HIPCountYPos[i], 176 + num[2] - '0');
			}
		}
	}

	if (rogue && cl.stats[STAT_ACTIVEWEAPON] >= 4096)
	{
		// there's no STAT_ specifically for rogue ammo so we just replace the count with
		// the relevant rogue ammo count.  bleagh.  and I thought hipnotic was horrible...
		// moral of the story: DON'T FUCK WITH THE WEAPONS!!!
		for (i = 0; i < 5; i++)
		{
			if (cl.stats[STAT_ACTIVEWEAPON] == RogueWeapons[i])
			{
				sprintf (num, "%3i", cl.stats[STAT_AMMO]);

				if (num[0] != ' ') Draw_SingleCharacter (52, ID1CountYPos[i + 2], 176 + num[0] - '0');
				if (num[1] != ' ') Draw_SingleCharacter (60, ID1CountYPos[i + 2], 176 + num[1] - '0');
				if (num[2] != ' ') Draw_SingleCharacter (68, ID1CountYPos[i + 2], 176 + num[2] - '0');

				// some rogue ammo counts have to be drawn twice
				if (i == 4)
					continue;
				else if (i == 0 || i == 2)
					ypos = ID1CountYPos[i + 2] + 26;
				else ypos = ID1CountYPos[i + 2] - 26;

				if (num[0] != ' ') Draw_SingleCharacter (52, ypos, 176 + num[0] - '0');
				if (num[1] != ' ') Draw_SingleCharacter (60, ypos, 176 + num[1] - '0');
				if (num[2] != ' ') Draw_SingleCharacter (68, ypos, 176 + num[2] - '0');
			}
		}
	}

	glEnd ();
}


model_t *HUD_Arm = NULL;
model_t *HUD_Invl = NULL;

void HUD_DrawArmour (void)
{
	int skinnum = 0;
	model_t *Armour;
	int origin2;
	float scale;

	entity_t e;

	// HUD replacement for Sbar_DrawArmor
	if (cl.items & IT_INVULNERABILITY && !cl.stats[STAT_ARMOR])
	{
		if (!HUD_Invl) return;

		Armour = HUD_Invl;
		skinnum = 0;

		origin2 = 52;
		scale = 0.75;
	}
	else if (!cl.stats[STAT_ARMOR])
		return;
	else
	{
		if (!HUD_Arm) return;

		if (rogue)
		{
			if (cl.items & RIT_ARMOR3)
				skinnum = 2;
			else if (cl.items & RIT_ARMOR2)
				skinnum = 1;
			else if (cl.items & RIT_ARMOR1)
				skinnum = 0;
		}
		else
		{
			if (cl.items & IT_ARMOR3)
				skinnum = 2;
			else if (cl.items & IT_ARMOR2)
				skinnum = 1;
			else if (cl.items & IT_ARMOR1)
				skinnum = 0;
		}

		Armour = HUD_Arm;

		origin2 = 43;
		scale = 0.0;
	}

	// set up a temp dummy entity to hold the model
	memset (&e, 0, sizeof (e));
	e.model = Armour;

	e.origin[1] = 27;
	e.origin[2] = origin2;

	e.angles[1] = bobjrotate;

	// and draw it
	R_DrawAliasModelSBar (&e, 0.7, skinnum, 1.0, scale);
}


char *HUD_AmmoNames[] =
{
	// normal quake
	"maps/b_shell1.bsp",
	"maps/b_nail1.bsp",
	"maps/b_rock0.bsp",
	"maps/b_batt1.bsp",
	// rogue
	"maps/b_lnail1.bsp",
	"maps/b_plas0.bsp",	// this looks nice - just to show that i can do non-cubes
	"maps/b_mrock0.bsp",
	// hipnotic has no ammo box models
};


model_t *HUD_Ammo[7] = {NULL, NULL, NULL, NULL, NULL, NULL, NULL};


void R_DrawBrushModelSbar (entity_t *e, float Light);

int HUDVPSize;
int HUDAmmoOffset;


void HUD_DrawAmmo (void)
{
	entity_t e;
	int AmmoNum;

	// HUD Replacement for Sbar_DrawAmmo
	if (rogue)
	{
		if (cl.items & RIT_SHELLS)
			AmmoNum = 0;
		else if (cl.items & RIT_NAILS)
			AmmoNum = 1;
		else if (cl.items & RIT_ROCKETS)
			AmmoNum = 2;
		else if (cl.items & RIT_CELLS)
			AmmoNum = 3;
		else if (cl.items & RIT_LAVA_NAILS)
			AmmoNum = 4;
		else if (cl.items & RIT_PLASMA_AMMO)
			AmmoNum = 5;
		else if (cl.items & RIT_MULTI_ROCKETS)
			AmmoNum = 6;
		else return;
	}
	else
	{
		if (cl.items & IT_SHELLS)
			AmmoNum = 0;
		else if (cl.items & IT_NAILS)
			AmmoNum = 1;
		else if (cl.items & IT_ROCKETS)
			AmmoNum = 2;
		else if (cl.items & IT_CELLS)
			AmmoNum = 3;
		else return;
	}

	if (!HUD_Ammo[AmmoNum]) return;

	glViewport (HUDAmmoOffset, 0, HUDVPSize, HUDVPSize);

	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
	gluPerspective (r_refdef.fov_y, 1, 4, 150);

	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity ();

	glEnable (GL_DEPTH_TEST);
	glDepthRange (gldepthmin, gldepthmin + 0.1 * (gldepthmax - gldepthmin));

	// put Z going up and X going deep to match Quakes main render
	// Y remains at right to left, Z becomes top to bottom and X becomes near to far
	glRotatef (-90, 1, 0, 0);
	glRotatef (90, 0, 0, 1);

	// set up a temp dummy entity to hold the model
	memset (&e, 0, sizeof (e));
	e.model = HUD_Ammo[AmmoNum];

	// set origin and angles for any transforms we might want to do
	e.origin[0] = 45;		// always positive, higher numbers are deeper
	e.origin[1] = 0;		// positive = to the left, negative = to the right
	e.origin[2] = -19;		// negative = further down, positive = higher up

	e.angles[1] = anglemod (25 * realtime); //bobjrotate;

	glPushMatrix ();
	glRotatef (-17.5, 0, 1, 0);

	// and draw it
	R_DrawBrushModelSbar (&e, 0.7);

	glPopMatrix ();

	// restore the previous depth range
	glDepthRange (gldepthmin, gldepthmax);
	glDisable (GL_DEPTH_TEST);
}


model_t *HUD_FaceMDL = NULL;
model_t *HUD_FaceBSP = NULL;

float normalFrame = 12;		// normal animation, 12 to 16


void HUD_DrawFaceBSP (void)
{
	entity_t e;

	if (!HUD_FaceBSP) return;

	glViewport (0, 0, HUDVPSize, HUDVPSize);

	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
	gluPerspective (r_refdef.fov_y, 1, 4, 150);

	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity ();

	glEnable (GL_DEPTH_TEST);
	glDepthRange (gldepthmin, gldepthmin + 0.1 * (gldepthmax - gldepthmin));

	// put Z going up and X going deep to match Quakes main render
	// Y remains at right to left, Z becomes top to bottom and X becomes near to far
	glRotatef (-90, 1, 0, 0);
	glRotatef (90, 0, 0, 1);

	// set up a temp dummy entity to hold the model
	memset (&e, 0, sizeof (e));
	e.model = HUD_FaceBSP;

	// set origin and angles for any transforms we might want to do
	e.origin[0] = 45;		// always positive, higher numbers are deeper
	e.origin[1] = 0;		// positive = to the left, negative = to the right
	e.origin[2] = -19;		// negative = further down, positive = higher up

	e.angles[1] = anglemod (25 * realtime); //bobjrotate;

	glPushMatrix ();
	glRotatef (-17.5, 0, 1, 0);

	// and draw it
	R_DrawBrushModelSbar (&e, 0.7);

	glPopMatrix ();

	// restore the previous depth range
	glDepthRange (gldepthmin, gldepthmax);
	glDisable (GL_DEPTH_TEST);
}


extern int setup_top;
extern int setup_bottom;
extern qboolean skinChange;
void GL_Upload8 (byte *data, int width, int height, qboolean mipmap, qboolean alpha);

void HUD_TranslatePlayerMDL (void)
{
	// skin translation
	struct aliashdr_s *paliashdr;
	int s;
	byte translate[256];
	int i;
	byte *original;
	byte translated[320 * 200];

	int top = setup_top * 16;
	int bottom = setup_bottom * 16;

	paliashdr = (struct aliashdr_s *) Mod_Extradata (HUD_FaceMDL);
	s = paliashdr->skinwidth * paliashdr->skinheight;

	for (i = 0; i < 256; i++)
		translate[i] = i;

	for (i = 0; i < 16; i++)
	{
		if (top < 128)	// the artists made some backwards ranges.  sigh.
			translate[TOP_RANGE + i] = top + i;
		else
			translate[TOP_RANGE + i] = top + 15 - i;
				
		if (bottom < 128)
			translate[BOTTOM_RANGE + i] = bottom + i;
		else
			translate[BOTTOM_RANGE + i] = bottom + 15 - i;
	}

	// skinnum is always 0 here
	original = (byte *) paliashdr + paliashdr->texels[0];

	// always bind skin [0][0] cos we may not have any active clients!!!
	// (we are in a menu here after all...)
	glBindTexture (GL_TEXTURE_2D, paliashdr->gl_texture[0][0]->texnum);

	for (i = 0; i < s; i += 4)
	{
		translated[i] = translate[original[i]];
		translated[i + 1] = translate[original[i + 1]];
		translated[i + 2] = translate[original[i + 2]];
		translated[i + 3] = translate[original[i + 3]];
	}

	GL_Upload8 (translated, paliashdr->skinwidth, paliashdr->skinheight, true, false);

	skinChange = false;
}


void M_Menu_Draw_Player (void)
{
	entity_t e;

	int vpWidth;
	int vpHeight;
	int vpX;
	int vpY;

	float bobjrotate2 = anglemod (50 * Sys_FloatTime ());

	if (!HUD_FaceMDL)
	{
		// attempt to load the player model
		HUD_FaceMDL = Mod_ForName ("progs/player.mdl", false);

		// failed for whatever reason
		if (!HUD_FaceMDL) return;
	}

	// translate the skin (only if necessary)
	if (skinChange) HUD_TranslatePlayerMDL ();

	// open a viewport
	vpX = 338 * glwidth / 640;
	vpY = 276 * glheight / 480;

	vpWidth = 126 * glwidth / 640;
	vpHeight = 142 * glheight / 480;

	glViewport (vpX, vpY, vpWidth, vpHeight);

	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();

	gluPerspective (r_refdef.fov_y, (float) vpWidth / (float) vpHeight, 4, 150);

	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity ();

	// hack the depth range to prevent this being clipped by other 3d objects
	glEnable (GL_DEPTH_TEST);
	glDepthRange (gldepthmin, gldepthmin + 0.1 * (gldepthmax - gldepthmin));

	// put Z going up and X going deep to match Quakes main render
	// Y remains at right to left, Z becomes top to bottom and X becomes near to far
	glRotatef (-90, 1, 0, 0);
	glRotatef (90, 0, 0, 1);

	// set up a temp dummy entity to hold the model
	memset (&e, 0, sizeof (e));

	// set origin and angles for any transforms we might want to do
	e.origin[0] = 50;		// always positive, higher numbers are deeper
	e.origin[1] = 0;		// positive = to the left, negative = to the right
	e.origin[2] = 4;		// negative = further down, positive = higher up

	e.angles[1] = bobjrotate2;

	e.model = HUD_FaceMDL;
	e.frame = 12;

	// make sure this is up!!!
	glActiveTexture (GL_TEXTURE0);
	glEnable (GL_TEXTURE_2D);

	// enable vertex arrays
	glEnableClientState (GL_VERTEX_ARRAY);
	glVertexPointer (3, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[0]);

	glEnableClientState (GL_TEXTURE_COORD_ARRAY);
	glClientActiveTexture (GL_TEXTURE0);
	glTexCoordPointer (2, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[3]);

	glEnableClientState (GL_COLOR_ARRAY);
	glColorPointer (4, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[5]);

	// and draw it
	R_DrawAliasModelSBar (&e, 0.7, 0, 1.0, 0.0);

	// restore the previous depth range
	glDepthRange (gldepthmin, gldepthmax);

	glDisableClientState (GL_VERTEX_ARRAY);
	glDisableClientState (GL_TEXTURE_COORD_ARRAY);
	glDisableClientState (GL_COLOR_ARRAY);
}



model_t *HUD_Quad = NULL;
model_t *HUD_Invis = NULL;
model_t *HUD_Suit = NULL;

int YMod;

void HUD_DrawIndividualPowerup (model_t *HUD_Powerup, float scale)
{
	entity_t e;

	// HUD Replacement for Sbar_DrawFace
	if (!HUD_Powerup) return;

	// set up a temp dummy entity to hold the model
	memset (&e, 0, sizeof (e));
	e.model = HUD_Powerup;

	// set origin and angles for any transforms we might want to do
	e.origin[1] = 612;
	e.origin[2] = 435 - (YMod * 50);
	e.angles[1] = bobjrotate;

	// HACK
	// the pent, ring and suit need to be shifted about a bit more
	if (HUD_Powerup == HUD_Suit) e.origin[2] += 5;
	if (HUD_Powerup == HUD_Invis) e.origin[2] += 13;
	if (HUD_Powerup == HUD_Invl) e.origin[2] += 8;
	// END HACK

	// and draw it
	R_DrawAliasModelSBar (&e, 0.7, 0, 1.0, scale);

	YMod++;
}


void HUD_DrawPowerups (void)
{
	YMod = 0;

	// now draw them
	if (cl.items & IT_QUAD) HUD_DrawIndividualPowerup (HUD_Quad, 0.0);
	if (cl.items & IT_INVULNERABILITY) HUD_DrawIndividualPowerup (HUD_Invl, 0.8);
	if (cl.items & IT_INVISIBILITY) HUD_DrawIndividualPowerup (HUD_Invis, 2.5);
	if (cl.items & IT_SUIT) HUD_DrawIndividualPowerup (HUD_Suit, 0.75);

	// add r & h
}


int BitPos;

char *HUD_BitNames[] =
{
	"progs/end1.mdl",
	"progs/end2.mdl",
	"progs/end3.mdl",
	"progs/end4.mdl",
	"progs/b_s_key.mdl",
	"progs/b_g_key.mdl",
	"progs/m_s_key.mdl",
	"progs/m_g_key.mdl",
	"progs/w_s_key.mdl",
	"progs/w_g_key.mdl"
};

model_t *HUD_BitModels[10] = {NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL};

extern int worldtype;

void HUD_DrawIndividualBit (int BitNum)
{
	entity_t e;

	// switch the keys
	if (BitNum >= 4)
	{
		switch (worldtype)
		{
		case 1:
			// metal
			BitNum += 2;
			break;
		case 2:
			// base
			break;
		case 0:
		default:
			// medieval
			BitNum += 4;
			break;
		}
	}

	if (!HUD_BitModels[BitNum]) return;

	// set up a temp dummy entity to hold the model
	memset (&e, 0, sizeof (e));
	e.model = HUD_BitModels[BitNum];

	e.origin[1] = BitPos;
	e.origin[2] = 45;

	// give each bit a different rotation
	e.angles[1] = bobjrotate + (15 * (BitNum + 1));

	R_DrawAliasModelSBar (&e, 0.7, 0, 1.0, 0.0);

	BitPos -= 25;
}


void HUD_DrawBits (void)
{
	BitPos = 600;

	// drawing keys, sigils, etc
	// keys
	if (cl.items & IT_KEY1) HUD_DrawIndividualBit (4);
	if (cl.items & IT_KEY2) HUD_DrawIndividualBit (5);

	// don't draw sigils in rogue or hipnotic
	if (rogue || hipnotic) return;

	if (cl.items & IT_SIGIL1) HUD_DrawIndividualBit (0);
	if (cl.items & IT_SIGIL2) HUD_DrawIndividualBit (1);
	if (cl.items & IT_SIGIL3) HUD_DrawIndividualBit (2);
	if (cl.items & IT_SIGIL4) HUD_DrawIndividualBit (3);
}

void Sbar_Preload3DModels (void)
{
	// preload all the models to prevent disk access at run time
	// done every map load to also prevent Quake from discarding the models or textures
	int i;

	// weapons
	for (i = 0; i < 10; i++)
		InvWeaps[i] = Mod_ForName (Weaponses[i], false);

	// bits (keys/sigils)
	for (i = 0; i < 10; i++)
		HUD_BitModels[i] = Mod_ForName (HUD_BitNames[i], false);

	// ammo
	for (i = 0; i < 7; i++)
		HUD_Ammo[i] = Mod_ForName (HUD_AmmoNames[i], false);

	// powerups
	HUD_Invl = Mod_ForName ("progs/invulner.mdl", false);
	HUD_Quad = Mod_ForName ("progs/quaddama.mdl", false);
	HUD_Invis = Mod_ForName ("progs/invisibl.mdl", false);
	HUD_Suit = Mod_ForName ("progs/suit.mdl", false);

	// armour model
	HUD_Arm = Mod_ForName ("progs/armor.mdl", false);

	// face model
	HUD_FaceMDL = Mod_ForName ("progs/player.mdl", false);
	HUD_FaceBSP = Mod_ForName ("maps/b_bh100.bsp", false);

	// the player model needs to be translated the first time it's loaded
	HUD_TranslatePlayerMDL ();
}


/*
====================
SbarDraw3DModels

MH - draws 3D models for a lot of the little status bar widgets

FIX ME: if we don't have armour we don't need to go through this.

All 3D widgets aside from health and ammo are drawn here.  Health and ammo require a viewport
with a perspective projection so as to get them looking right, so they need to be done
before any of the other 2D stuff is drawn (otherwise we have an utter mess).
====================
*/
void SbarDraw3DModels (void)
{
	int i;

	bobjrotate = anglemod (100 * realtime);

	// enable vertex arrays
	glEnableClientState (GL_VERTEX_ARRAY);
	glVertexPointer (3, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[0]);

	glEnableClientState (GL_TEXTURE_COORD_ARRAY);
	glClientActiveTexture (GL_TEXTURE0);
	glTexCoordPointer (2, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[3]);

	glEnableClientState (GL_COLOR_ARRAY);
	glColorPointer (4, GL_FLOAT, (9 * sizeof (GLfloat)), &VArray[5]);

	// initialize our matrices
	glMatrixMode (GL_PROJECTION);
	glPushMatrix ();
	glLoadIdentity ();

	// flip the orthographic projection along both axes for the 3D stuff
	glOrtho  (0, 640, 0, 480, -400, 400);

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix ();
	glLoadIdentity ();

	// put Z going up and X going deep to match Quakes main render
	// Y remains at right to left, Z becomes top to bottom and X becomes near to far
	glRotatef (-90, 1, 0, 0);
	glRotatef (90, 0, 0, 1);

	// hack the depth range to prevent sbar models being clipped by world geometry
	glEnable (GL_DEPTH_TEST);
	glDepthRange (gldepthmin, gldepthmin + 0.1 * (gldepthmax - gldepthmin));

	// draw the models
	HUD_DrawArmour ();

	if (scr_viewsize.value < 110)
	{
		HUD_DrawWeapons ();
		HUD_DrawPowerups ();
		HUD_DrawBits ();
	}

	glMatrixMode (GL_PROJECTION);
	glPopMatrix ();

	glMatrixMode (GL_MODELVIEW);
	glPopMatrix ();

	// restore the previous depth range
	glDepthRange (gldepthmin, gldepthmax);
	glDisable (GL_DEPTH_TEST);

	glDisableClientState (GL_VERTEX_ARRAY);
	glDisableClientState (GL_TEXTURE_COORD_ARRAY);
	glDisableClientState (GL_COLOR_ARRAY);
}


#define HUD_VP_WIDTH 54


// hack - zerstorer cutscenes
extern qboolean sbardraw;

void SbarDraw3DBrushes (void)
{
	// console is full screen
	if (scr_con_current == vid.height) return;

	// don't draw during an intermission
	if (cl.intermission) return;

	if (cl.stats[STAT_HEALTH] <= 0) return;

	// hack - zerstorer cutscenes
	if (!sbardraw) return;

	if (scr_viewsize.value >= 120) return;

	HUDVPSize = (HUD_VP_WIDTH * glwidth) / 640;
	HUDAmmoOffset = ((640 - HUD_VP_WIDTH) * glwidth) / 640;

	// these are drawn in 2 seperate viewports so they must be done seperately
	// take your pick for the face...
	HUD_DrawFaceBSP ();
	HUD_DrawAmmo ();
}


float	fpscounter;
char	fpsss[32];
char	leafs[32];
char	surfs[32];
extern mleaf_t *r_viewleaf;

void Sbar_DrawFPS (void)
{
	static float second_count;
	static int fps;

	int endpos;

	// increment this whether or not we are displaying the FPS counter
	second_count = second_count + frametime;

	if (!r_speeds.value) return;

	endpos = 540;

	// only update 10 times per frame
	if (second_count >= 0.1)
	{
		fps = 1 / frametime;
		second_count = 0;
		sprintf (fpsss, "  FPS: %4i", fps);
		sprintf (surfs, "Surfs: %4i", addSurfs);
	}

	// always do addleafs since it updates seperately
	if (addLeafs) sprintf (leafs, "Leafs: %4i", addLeafs);

	Draw_String (endpos, 12, fpsss);
	Draw_String (endpos, 24, leafs);
	Draw_String (endpos, 36, surfs);

	// just in case...
	if (r_viewleaf)
	{
		Draw_String (endpos, 52, va (" Leaf: %4i", r_viewleaf->leafnum));
		Draw_String (endpos, 64, va ("Conts: %i", r_viewleaf->contents));
		Draw_String (endpos, 76, va ("  Vol: %3i", r_viewleaf->volume));
	}
}


int clolditems;

float quadcounter;
float pentcounter;
float ringcounter;
float suitcounter;

/*
===============
Sbar_Draw
===============
*/
void Sbar_Draw (void)
{
	int counterpos;

	if (scr_con_current == vid.height)
		return;		// console is full screen

	// don't draw during an intermission
	if (cl.intermission) return;

	// hack - zerstorer cutscenes
	if (!sbardraw) return;

	if (scr_viewsize.value >= 120) return;

	scr_copyeverything = 1;

	sb_updates++;

	if (sb_showscores || cl.stats[STAT_HEALTH] <= 0)
	{
		Sbar_DrawScoreboard ();
		sb_updates = 0;
	}

	if (cl.stats[STAT_HEALTH] > 0)
	{
		// decrement the counters
		quadcounter -= frametime;
		pentcounter -= frametime;
		ringcounter -= frametime;
		suitcounter -= frametime;

		// 3D Widgets
		SbarDraw3DModels ();

		// armor
		Sbar_DrawArmor ();

		// health
		HUD_DrawNum (47, 
					 447, 
					 cl.stats[STAT_HEALTH], 
					 3, 
					 cl.stats[STAT_HEALTH] <= 25);

		// ammo
		Sbar_DrawAmmo ();

		if (scr_viewsize.value < 110) HUD_DrawAmmoCount ();

		// Sbar_DrawFPS ();

		if (clolditems != cl.items)
		{
			// we've got something new - see what it is
			if ((cl.items & IT_QUAD) && !(clolditems & IT_QUAD) && quadcounter < 0)
				quadcounter = 30.99999999;

			// we've got something new - see what it is
			if ((cl.items & IT_INVULNERABILITY) && !(clolditems & IT_INVULNERABILITY) && pentcounter < 0)
				pentcounter = 30.99999999;

			// we've got something new - see what it is
			if ((cl.items & IT_INVISIBILITY) && !(clolditems & IT_INVISIBILITY) && ringcounter < 0)
				ringcounter = 30.99999999;

			// we've got something new - see what it is
			if ((cl.items & IT_SUIT) && !(clolditems & IT_SUIT) && suitcounter < 0)
				suitcounter = 30.99999999;

			// save out the items
			clolditems = cl.items;
		}

		if (scr_viewsize.value > 100) return;

		counterpos = 10;

		// draw the counters - these *MUST* be in the same order as the 3D widgets
		if ((cl.items & IT_QUAD) && quadcounter >= 1)
		{
			HUD_DrawNum (530,
						 counterpos, 
						 (int) quadcounter, 
						 3, 
						 quadcounter <= 4.0);

			counterpos += 50;
		}
		else quadcounter = 0.0;

		if ((cl.items & IT_INVULNERABILITY) && pentcounter >= 1)
		{
			HUD_DrawNum (530,
						 counterpos, 
						 (int) pentcounter, 
						 3, 
						 pentcounter <= 4.0);

			counterpos += 50;
		}
		else pentcounter = 0.0;

		if ((cl.items & IT_INVISIBILITY) && ringcounter >= 1)
		{
			HUD_DrawNum (530,
						 counterpos, 
						 (int) ringcounter, 
						 3, 
						 ringcounter <= 4.0);

			counterpos += 50;
		}
		else ringcounter = 0.0;

		if ((cl.items & IT_SUIT) && suitcounter >= 1)
		{
			HUD_DrawNum (530,
						 counterpos, 
						 (int) suitcounter, 
						 3, 
						 suitcounter <= 4.0);

			counterpos += 50;
		}
		else suitcounter = 0.0;
	}
}

//=============================================================================

/*
==================
Sbar_IntermissionNumber

==================
*/
void Sbar_IntermissionNumber (int x, int y, int num, int digits, int color)
{
	char			str[12];
	char			*ptr;
	int				l, frame;

	l = Sbar_itoa (num, str);
	ptr = str;
	if (l > digits)
		ptr += (l-digits);
	if (l < digits)
		x += (digits-l)*24;

	while (*ptr)
	{
		if (*ptr == '-')
			frame = STAT_MINUS;
		else
			frame = *ptr -'0';

		Draw_TransPic (x,y,sb_nums[color][frame]);
		x += 24;
		ptr++;
	}
}

/*
==================
Sbar_DeathmatchOverlay

==================
*/
void Sbar_DeathmatchOverlay (void)
{
	qpic_t			*pic;
	int				i, k, l;
	int				top, bottom;
	int				x, y, f;
	char			num[12];
	scoreboard_t	*s;

	scr_copyeverything = 1;
	scr_fullupdate = 0;

	pic = Draw_CachePic ("gfx/ranking.lmp");
	M_DrawPic ((320-pic->width)/2, 8, pic);

// scores
	Sbar_SortFrags ();

// draw the text
	l = scoreboardlines;

	x = 80 + ((vid.width - 320)>>1);
	y = 40;
	for (i=0 ; i<l ; i++)
	{
		k = fragsort[i];
		s = &cl.scores[k];
		if (!s->name[0])
			continue;

	// draw background
		top = s->colors & 0xf0;
		bottom = (s->colors & 15)<<4;
		top = Sbar_ColorForMap (top);
		bottom = Sbar_ColorForMap (bottom);

		Draw_Fill ( x, y, 40, 4, top);
		Draw_Fill ( x, y+4, 40, 4, bottom);

	// draw number
		f = s->frags;
		sprintf (num, "%3i",f);

		Draw_Character ( x+8 , y, num[0]);
		Draw_Character ( x+16 , y, num[1]);
		Draw_Character ( x+24 , y, num[2]);

		if (k == cl.viewentity - 1)
			Draw_Character ( x - 8, y, 12);

#if 0
{
	int				total;
	int				n, minutes, tens, units;

	// draw time
		total = cl.completed_time - s->entertime;
		minutes = (int)total/60;
		n = total - minutes*60;
		tens = n/10;
		units = n%10;

		sprintf (num, "%3i:%i%i", minutes, tens, units);

		Draw_String ( x+48 , y, num);
}
#endif

	// draw name
		Draw_String (x+64, y, s->name);

		y += 10;
	}
}

/*
==================
Sbar_DeathmatchOverlay

==================
*/
void Sbar_MiniDeathmatchOverlay (void)
{
	qpic_t			*pic;
	int				i, k, l;
	int				top, bottom;
	int				x, y, f;
	char			num[12];
	scoreboard_t	*s;
	int				numlines;

	if (vid.width < 512 || !sb_lines)
		return;

	scr_copyeverything = 1;
	scr_fullupdate = 0;

// scores
	Sbar_SortFrags ();

// draw the text
	l = scoreboardlines;
	y = vid.height - sb_lines;
	numlines = sb_lines/8;
	if (numlines < 3)
		return;

	//find us
	for (i = 0; i < scoreboardlines; i++)
		if (fragsort[i] == cl.viewentity - 1)
			break;

    if (i == scoreboardlines) // we're not there
            i = 0;
    else // figure out start
            i = i - numlines/2;

    if (i > scoreboardlines - numlines)
            i = scoreboardlines - numlines;
    if (i < 0)
            i = 0;

	x = 324;
	for (/* */; i < scoreboardlines && y < vid.height - 8 ; i++)
	{
		k = fragsort[i];
		s = &cl.scores[k];
		if (!s->name[0])
			continue;

	// draw background
		top = s->colors & 0xf0;
		bottom = (s->colors & 15)<<4;
		top = Sbar_ColorForMap (top);
		bottom = Sbar_ColorForMap (bottom);

		Draw_Fill ( x, y+1, 40, 3, top);
		Draw_Fill ( x, y+4, 40, 4, bottom);

	// draw number
		f = s->frags;
		sprintf (num, "%3i",f);

		Draw_Character ( x+8 , y, num[0]);
		Draw_Character ( x+16 , y, num[1]);
		Draw_Character ( x+24 , y, num[2]);

		if (k == cl.viewentity - 1) {
			Draw_Character ( x, y, 16);
			Draw_Character ( x + 32, y, 17);
		}

#if 0
{
	int				total;
	int				n, minutes, tens, units;

	// draw time
		total = cl.completed_time - s->entertime;
		minutes = (int)total/60;
		n = total - minutes*60;
		tens = n/10;
		units = n%10;

		sprintf (num, "%3i:%i%i", minutes, tens, units);

		Draw_String ( x+48 , y, num);
}
#endif

	// draw name
		Draw_String (x+48, y, s->name);

		y += 8;
	}
}

/*
==================
Sbar_IntermissionOverlay

==================
*/
void Sbar_IntermissionOverlay (void)
{
	qpic_t	*pic;
	int		dig;
	int		num;

	scr_copyeverything = 1;
	scr_fullupdate = 0;

	if (cl.gametype == GAME_DEATHMATCH)
	{
		Sbar_DeathmatchOverlay ();
		return;
	}

	// draws in the top left - not to everyone's taste i admit, but i like it.
	// stuff doesn't obscure the intermission view that way
	pic = Draw_CachePic ("gfx/complete.lmp");
	Draw_Pic (64, 24, pic);

	pic = Draw_CachePic ("gfx/inter.lmp");
	Draw_TransPic (0, 56, pic);

	// time
	dig = cl.completed_time/60;
	Sbar_IntermissionNumber (160, 64, dig, 3, 0);
	num = cl.completed_time - dig*60;
	Draw_TransPic (234,64,sb_colon);
	Draw_TransPic (246,64,sb_nums[0][num/10]);
	Draw_TransPic (266,64,sb_nums[0][num%10]);

	Sbar_IntermissionNumber (160, 104, cl.stats[STAT_SECRETS], 3, 0);
	Draw_TransPic (232,104,sb_slash);
	Sbar_IntermissionNumber (240, 104, cl.stats[STAT_TOTALSECRETS], 3, 0);

	Sbar_IntermissionNumber (160, 144, cl.stats[STAT_MONSTERS], 3, 0);
	Draw_TransPic (232,144,sb_slash);
	Sbar_IntermissionNumber (240, 144, cl.stats[STAT_TOTALMONSTERS], 3, 0);

}


/*
==================
Sbar_FinaleOverlay

==================
*/
void Sbar_FinaleOverlay (void)
{
	qpic_t	*pic;

	scr_copyeverything = 1;

	pic = Draw_CachePic ("gfx/finale.lmp");
	Draw_TransPic ( (vid.width-pic->width)/2, 16, pic);
}
