/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "quakedef.h"

#define MAX_PARTICLES			2048	// default max # of particles at one
										//  time
#define ABSOLUTE_MIN_PARTICLES	512		// no fewer than this no matter what's
										//  on the command line

int		ramp1[8] = {0x6f, 0x6d, 0x6b, 0x69, 0x67, 0x65, 0x63, 0x61};
int		ramp2[8] = {0x6f, 0x6e, 0x6d, 0x6c, 0x6b, 0x6a, 0x68, 0x66};
int		ramp3[8] = {0x6d, 0x6b, 6, 5, 4, 3};

int		particletexture;
int		rainparticletexture;
int		smoke1particletexture;
int		smoke2particletexture;
int		smoke3particletexture;
int		smoke4particletexture;
int		blood1particletexture;
int		blood2particletexture;
int		blood3particletexture;
int		blood4particletexture;
int		flareparticletexture;
int		bubbleparticletexture;

particle_t	*active_particles, *free_particles, *particles;

int			r_numparticles;

vec3_t		r_pright, r_pup, r_ppn;

float partrace(vec3_t p1, vec3_t p2)
{
	trace_t trace;

	memset (&trace, 0, sizeof(trace)); 
	SV_RecursiveHullCheck (cl.worldmodel->hulls, 0, 1, 1, p1, p2, &trace);

	return trace.fraction;
}

extern	byte	*targa_rgba;
extern	cvar_t	r_smokealpha;
cvar_t	r_smokecolor = {"r_smokecolor", "12", true};

void R_InitParticleTexture (void)
{
	byte	data[64][64][4];
	FILE	*f;

// ORIGINAL

	COM_FOpenFile ("tq/particles/particle.tga", &f);
	if (!f)
	{
		Sys_Error ("Couldn't load %s\n", "tq/particles/particle.tga");
		return;
	}
	LoadTGA (f);
	particletexture = texture_extension_number++;
	GL_Bind(particletexture);
	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, targa_rgba);
	free (targa_rgba);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

// SMOKE1

	COM_FOpenFile ("tq/particles/smoke1.tga", &f);
	if (!f)
	{
		Sys_Error ("Couldn't load %s\n", "tq/particles/smoke1.tga");
		return;
	}
	LoadTGA (f);
	smoke1particletexture = texture_extension_number++;
	GL_Bind(smoke1particletexture);
	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, targa_rgba);
	free (targa_rgba);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

// SMOKE2

	COM_FOpenFile ("tq/particles/smoke2.tga", &f);
	if (!f)
	{
		Sys_Error ("Couldn't load %s\n", "tq/particles/smoke2.tga");
		return;
	}
	LoadTGA (f);
	smoke2particletexture = texture_extension_number++;
	GL_Bind(smoke2particletexture);
	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, targa_rgba);
	free (targa_rgba);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

// SMOKE3

	COM_FOpenFile ("tq/particles/smoke3.tga", &f);
	if (!f)
	{
		Sys_Error ("Couldn't load %s\n", "tq/particles/smoke3.tga");
		return;
	}
	LoadTGA (f);
	smoke3particletexture = texture_extension_number++;
	GL_Bind(smoke3particletexture);
	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, targa_rgba);
	free (targa_rgba);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

// SMOKE4

	COM_FOpenFile ("tq/particles/smoke4.tga", &f);
	if (!f)
	{
		Sys_Error ("Couldn't load %s\n", "tq/particles/smoke4.tga");
		return;
	}
	LoadTGA (f);
	smoke4particletexture = texture_extension_number++;
	GL_Bind(smoke4particletexture);
	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, targa_rgba);
	free (targa_rgba);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

// BLOOD1

	COM_FOpenFile ("tq/particles/blood1.tga", &f);
	if (!f)
	{
		Sys_Error ("Couldn't load %s\n", "tq/particles/blood1.tga");
		return;
	}
	LoadTGA (f);
	blood1particletexture = texture_extension_number++;
	GL_Bind(blood1particletexture);
	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, targa_rgba);
	free (targa_rgba);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

// BLOOD2

	COM_FOpenFile ("tq/particles/blood2.tga", &f);
	if (!f)
	{
		Sys_Error ("Couldn't load %s\n", "tq/particles/blood2.tga");
		return;
	}
	LoadTGA (f);
	blood2particletexture = texture_extension_number++;
	GL_Bind(blood2particletexture);
	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, targa_rgba);
	free (targa_rgba);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

// BLOOD3

	COM_FOpenFile ("tq/particles/blood3.tga", &f);
	if (!f)
	{
		Sys_Error ("Couldn't load %s\n", "tq/particles/blood3.tga");
		return;
	}
	LoadTGA (f);
	blood3particletexture = texture_extension_number++;
	GL_Bind(blood3particletexture);
	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, targa_rgba);
	free (targa_rgba);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

// BLOOD4

	COM_FOpenFile ("tq/particles/blood4.tga", &f);
	if (!f)
	{
		Sys_Error ("Couldn't load %s\n", "tq/particles/blood4.tga");
		return;
	}
	LoadTGA (f);
	blood4particletexture = texture_extension_number++;
	GL_Bind(blood4particletexture);
	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, targa_rgba);
	free (targa_rgba);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

// FLARES

	COM_FOpenFile ("tq/particles/particle.tga", &f);
	if (!f)
	{
		Sys_Error ("Couldn't load %s\n", "tq/particles/particle.tga");
		return;
	}
	LoadTGA (f);
	flareparticletexture = texture_extension_number++;
	GL_Bind(flareparticletexture);
	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, targa_rgba);
	free (targa_rgba);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

// RAIN

	COM_FOpenFile ("tq/particles/smoke1.tga", &f);
	if (!f)
	{
		Sys_Error ("Couldn't load %s\n", "tq/particles/smoke1.tga");
		return;
	}
	LoadTGA (f);
	rainparticletexture = texture_extension_number++;
	GL_Bind(rainparticletexture);
	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 32, 32, 0, GL_RGBA, GL_UNSIGNED_BYTE, targa_rgba);
	free (targa_rgba);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

// BUBBLES

	COM_FOpenFile ("tq/particles/bubble.tga", &f);
	if (!f)
	{
		Sys_Error ("Couldn't load %s\n", "tq/particles/bubble.tga");
		return;
	}
	LoadTGA (f);
	bubbleparticletexture = texture_extension_number++;
	glBindTexture(GL_TEXTURE_2D, bubbleparticletexture);
	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, targa_rgba);
	free (targa_rgba);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
}

/*
===============
R_InitParticles
===============
*/
void R_InitParticles (void)
{
	int		i;

	i = COM_CheckParm ("-particles");

	if (i)
	{
		r_numparticles = (int)(atoi(com_argv[i+1]));
		if (r_numparticles < ABSOLUTE_MIN_PARTICLES)
			r_numparticles = ABSOLUTE_MIN_PARTICLES;
	}
	else
	{
		r_numparticles = MAX_PARTICLES;
	}

	particles = (particle_t *) Hunk_AllocName (r_numparticles * sizeof(particle_t), "particles");

	Cvar_RegisterVariable (&r_smokecolor);
	R_InitParticleTexture ();
}

/*
===============
R_EntityParticles
===============
*/

#define NUMVERTEXNORMALS	162
extern	float	r_avertexnormals[NUMVERTEXNORMALS][3];
vec3_t	avelocities[NUMVERTEXNORMALS];
float	beamlength = 16;
vec3_t	avelocity = {23, 7, 3};
float	partstep = 0.01;
float	timescale = 0.01;

void R_EntityParticles (entity_t *ent)
{
	int			count;
	int			i;
	particle_t	*p;
	float		angle;
	float		sr, sp, sy, cr, cp, cy;
	vec3_t		forward;
	float		dist;
	
	dist = 64;
	count = 50;

	if (!avelocities[0][0])
	{
		for (i=0 ; i<NUMVERTEXNORMALS*3 ; i++)
		{
			avelocities[0][i] = (rand()&255) * 0.01;
		}
	}


	for (i=0 ; i<NUMVERTEXNORMALS ; i++)
	{
		angle = cl.time * avelocities[i][0];
		sy = sin(angle);
		cy = cos(angle);
		angle = cl.time * avelocities[i][1];
		sp = sin(angle);
		cp = cos(angle);
		angle = cl.time * avelocities[i][2];
		sr = sin(angle);
		cr = cos(angle);
	
		forward[0] = cp*cy;
		forward[1] = cp*sy;
		forward[2] = -sp;

		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->texnum = flareparticletexture;
		p->scale = 2;
		p->alpha = 255;
		p->die = cl.time + 0.01;
		p->color = 0x6f;
		p->type = pt_explode;
		
		p->org[0] = ent->origin[0] + r_avertexnormals[i][0]*dist + forward[0]*beamlength;			
		p->org[1] = ent->origin[1] + r_avertexnormals[i][1]*dist + forward[1]*beamlength;			
		p->org[2] = ent->origin[2] + r_avertexnormals[i][2]*dist + forward[2]*beamlength;			
	}
}


/*
===============
R_ClearParticles
===============
*/
void R_ClearParticles (void)
{
	int		i;

	free_particles = &particles[0];
	active_particles = NULL;

	for (i=0 ;i<r_numparticles ; i++)
		particles[i].next = &particles[i+1];
	particles[r_numparticles-1].next = NULL;
}


void R_ReadPointFile_f (void)
{
	FILE	*f;
	vec3_t	org;
	int		r;
	int		c;
	particle_t	*p;
	char	name[MAX_OSPATH];

	sprintf (name,"maps/%s.pts", sv.name);

	COM_FOpenFile (name, &f);
	if (!f)
	{
		Con_Printf ("couldn't open %s\n", name);
		return;
	}
	
	Con_Printf ("Reading %s...\n", name);
	c = 0;
	for ( ;; )
	{
		r = fscanf (f,"%f %f %f\n", &org[0], &org[1], &org[2]);
		if (r != 3)
			break;
		c++;
		
		if (!free_particles)
		{
			Con_Printf ("Not enough free particles\n");
			break;
		}
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		
		p->texnum = particletexture;
		p->scale = 2;
		p->alpha = 255;
		p->die = 99;
		p->color = (-c)&15;
		p->type = pt_static;
		VectorCopy (vec3_origin, p->vel);
		VectorCopy (org, p->org);
	}

	fclose (f);
	Con_Printf ("%i points read\n", c);
}

/*
===============
R_ParseParticleEffect

Parse an effect out of the server message
===============
*/
void R_ParseParticleEffect (void)
{
	vec3_t		org, dir;
	int			i, count, msgcount, color;

	for (i=0 ; i<3 ; i++)
		org[i] = MSG_ReadCoord ();
	for (i=0 ; i<3 ; i++)
		dir[i] = MSG_ReadChar () * (1.0/16);
	msgcount = MSG_ReadByte ();
	color = MSG_ReadByte ();

if (msgcount == 255)
	count = 1024;
else
	count = msgcount;
	
	R_RunParticleEffect (org, dir, color, count);
}
	
/*
===============
R_ParticleExplosion

===============
*/
void R_ParticleExplosion (vec3_t org)
{
	int			i, j;
	particle_t	*p;

	if (!r_particles.value) 
		return;	

	for (i=0 ; i<256 ; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->texnum = flareparticletexture;
		p->scale = 4;
		p->alpha = 255;
		p->die = cl.time + 3;
		p->color = ramp1[0];
		p->ramp = rand()&3;
		if (i & 1)
			p->type = pt_explode;
		else
			p->type = pt_explode2;
		for (j=0 ; j<3 ; j++)
		{
			p->org[j] = org[j] + ((rand()%32)-16);
			p->vel[j] = (rand()%512)-256;
		}
	}
}

/*
===============
R_ParticleExplosion2

===============
*/
void R_ParticleExplosion2 (vec3_t org, int colorStart, int colorLength)
{
	int			i, j;
	particle_t	*p;
	int			colorMod = 0;

	if (!r_particles.value) 
		return;

	for (i=0; i<256; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->texnum = flareparticletexture;
		p->scale = 4;
		p->alpha = 255;
		p->die = cl.time + 0.3;
		p->color = colorStart + (colorMod % colorLength);
		colorMod++;

		p->type = pt_blob;
		for (j=0 ; j<3 ; j++)
		{
			p->org[j] = org[j] + ((rand()%32)-16);
			p->vel[j] = (rand()%512)-256;
		}
	}
}

/*
===============
R_BlobExplosion

===============
*/
void R_BlobExplosion (vec3_t org)
{
	int			i, j;
	particle_t	*p;

	if (!r_particles.value) 
		return;	

	for (i=0 ; i<1024 ; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->texnum = flareparticletexture;
		p->scale = 4;
		p->alpha = 255;
		p->die = cl.time + 1 + (rand()&8)*0.05;

		if (i & 1)
		{
			p->type = pt_blob;
			p->color = 66 + rand()%6;
			for (j=0 ; j<3 ; j++)
			{
				p->org[j] = org[j] + ((rand()%32)-16);
				p->vel[j] = (rand()%512)-256;
			}
		}
		else
		{
			p->type = pt_blob2;
			p->color = 255 + rand()%6;
			for (j=0 ; j<3 ; j++)
			{
				p->org[j] = org[j] + ((rand()%32)-16);
				p->vel[j] = (rand()%512)-256;
			}
		}
	}
}

/*
===============
R_RunParticleEffect

===============
*/

void R_RunParticleEffect (vec3_t org, vec3_t dir, int color, int count)
{
	int			i, j;
	particle_t	*p;

	if (!r_particles.value) 
		return;	

	for (i=0 ; i<count ; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		if (count == 256)
		{	// rocket explosion
			p->texnum = flareparticletexture;
			p->scale = 4;
			p->alpha = 255;
			p->die = cl.time + 5;
			p->color = ramp1[0];
			p->ramp = rand()&3;
			if (i & 1)
			{
				p->type = pt_explode;
				for (j=0 ; j<3 ; j++)
				{
					p->org[j] = org[j] + ((rand()%32)-16);
					p->vel[j] = (rand()%512)-256;
				}
			}
			else
			{
				p->type = pt_explode2;
				for (j=0 ; j<3 ; j++)
				{
					p->org[j] = org[j] + ((rand()%32)-16);
					p->vel[j] = (rand()%512)-256;
				}
			}
		}
		else
		{
			p->texnum = blood1particletexture + (rand() & 3);
			p->scale = 4;
			p->alpha = 255;
			p->die = cl.time + 5;
			p->color = (color&~7) + (rand()&7);
			p->type = pt_blood; //slowgrav;
			for (j=0 ; j<3 ; j++)
			{
				p->org[j] = org[j] + ((rand()&15)-8);
				p->vel[j] = dir[j]*15;
			}
		}
	}
}

/*
===============
R_SparkShower

===============
*/
void R_SparkShower (vec3_t org, vec3_t dir, int count, int type)
{
	int			i, j;
	particle_t	*p;

	if (!r_particles.value) 
		return;

	if (!free_particles)
		return;
	p = free_particles;
	free_particles = p->next;
	p->next = active_particles;
	active_particles = p;

	p->scale = randomrange(6,10);
	p->alpha = randomrange(127,255);
	if ((GetPointContents(p->org) == CONTENTS_EMPTY)||
		(GetPointContents(p->org) == CONTENTS_SOLID))
	{
		p->texnum = smoke1particletexture + (rand() & 3);
		p->type = pt_bulletpuff;
		p->vel[2] = randomrange(5,20);
	}
	else
	{
		p->texnum = bubbleparticletexture;
		p->type = pt_bubble;
	}
	p->die = cl.time + 99;
	VectorCopy(org, p->org);
	p->vel[0] = 0;
	p->vel[1] = 0;
	for (i=0 ; i<count ; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		p->texnum = flareparticletexture;
		p->scale = 4;
		p->alpha = 255;
		p->die = cl.time + 0.5;
		p->type = pt_dust;
		p->ramp = (rand()&3);
		p->color = ramp1[(int)p->ramp];
		for (j=0 ; j<3 ; j++)
		{
			p->org[j] = org[j] + ((rand()&7)-4);
			p->vel[j] = dir[j] + (rand()%192)-96;
		}
	}
}
void R_ParticleRain (vec3_t mins, vec3_t maxs, vec3_t dir, int count, int colorbase, int type)
{
	int			i;
	particle_t	*p;
	vec3_t		diff;
	vec3_t		org;
	vec3_t		vel;
	float		t, z;

	if (!r_particles.value) 
		return;

	if (maxs[0] <= mins[0]) {t = mins[0];mins[0] = maxs[0];maxs[0] = t;}
	if (maxs[1] <= mins[1]) {t = mins[1];mins[1] = maxs[1];maxs[1] = t;}
	if (maxs[2] <= mins[2]) {t = mins[2];mins[2] = maxs[2];maxs[2] = t;}
	if (dir[2] < 0) // falling
	{
		t = (maxs[2] - mins[2]) / -dir[2];
		z = maxs[2];
	}
	else // rising??
	{
		t = (maxs[2] - mins[2]) / dir[2];
		z = mins[2];
	}
	if (t < 0 || t > 2) // sanity check
		t = 2;
	t += cl.time;

	VectorSubtract(maxs, mins, diff);
	
	for (i=0 ; i<count ; i++)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;

		vel[0] = dir[0] + (rand()&31) - 16;
		vel[1] = dir[1] + (rand()&31) - 16;
		vel[2] = dir[2] + (rand()&63) - 32;
		org[0] = diff[0] * (float) (rand()&1023) * (0.0009765625) + mins[0];
		org[1] = diff[1] * (float) (rand()&1023) * (0.0009765625) + mins[1];
		org[2] = z;

		p->scale = 6;
		p->alpha = 255;
		p->die = t;
		if (type == 1)
		{
			p->texnum = particletexture;
			p->type = pt_snow;
		}
		else // 0
		{
			p->texnum = rainparticletexture;
			p->type = pt_static;
		}
		p->color = colorbase + (rand()&3);
		VectorCopy(org, p->org);
		VectorCopy(vel, p->vel);
	}
}
/*
===============
R_LavaSplash

===============
*/
void R_LavaSplash (vec3_t org)
{
	int			i, j, k;
	particle_t	*p;
	float		vel;
	vec3_t		dir;

	if (!r_particles.value) 
		return;

	for (i=-16 ; i<16 ; i+=2)
	{	
		for (j=-16 ; j<16 ; j+=2)
		{
			for (k=0 ; k<1 ; k++)
			{
				if (!free_particles)
					return;
				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;
		
				p->texnum = flareparticletexture;
				p->scale = 4;
				p->alpha = 255;
				p->die = cl.time + 2 + (rand()&31) * 0.02;
				p->color = 224 + (rand()&7);
				p->type = pt_slowgrav;
				
				dir[0] = j*8 + (rand()&7);
				dir[1] = i*8 + (rand()&7);
				dir[2] = 256;
	
				p->org[0] = org[0] + dir[0];
				p->org[1] = org[1] + dir[1];
				p->org[2] = org[2] + (rand()&63);
	
				f_vNorm(dir);						
				vel = 50 + (rand()&63);
				VectorScale (dir, vel, p->vel);
			}
		}
	}
}

/*
===============
R_TeleportSplash

===============
*/
void R_TeleportSplash (vec3_t org)
{
	int			i, j, k, a;
	particle_t	*p;

	if (!r_particles.value) 
		return;

	for (i=-24 ; i<24 ; i+=8)
	{	
		for (j=-24 ; j<24 ; j+=8)
		{
			for (k=-24 ; k<32 ; k+=8)
			{
				if (!free_particles)
					return;
				p = free_particles;
				free_particles = p->next;
				p->next = active_particles;
				active_particles = p;
		
				p->texnum = flareparticletexture;
				p->scale = 4;
				a = 1 + rand()&7;
				p->alpha = 255;
				p->die = cl.time + a * 0.125;
				p->color = 254; //8 + (rand()&7);
				p->type = pt_fadespark;
				
				p->org[0] = org[0] + i + (rand()&7);
				p->org[1] = org[1] + j + (rand()&7);
				p->org[2] = org[2] + k + (rand()&7);
	
				p->vel[0] = i*2 + (rand()%25) - 12;
				p->vel[1] = j*2 + (rand()%25) - 12;
				p->vel[2] = k*2 + (rand()%25) - 12 + 40;
			}
		}
	}
}

void R_RocketTrail (vec3_t start, vec3_t end, int type)
{
	vec3_t		vec;
	float		len;
	int			j, contents;
	particle_t	*p;
	int			dec;
	static int	tracercount;

	if (!r_particles.value) 
		return;
		
	contents = Mod_PointInLeaf(start, cl.worldmodel)->contents;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);
	if (type < 128)
	{
		dec = 3;
	}
	else
	{
		dec = 1;
		type -= 128;
	}

	while (len > 0)
	{
		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		
		p->vel[0] = p->vel[1] = p->vel[2] = 0;
		p->die = cl.time + 50;
		p->color = r_smokecolor.value;

		switch (type)
		{
			case 0:	// rocket trail
				p->vel[0] = randomrange(-10, 10);
				p->vel[1] = randomrange(-10, 10);
				p->vel[2] = randomrange(5, 35);
				p->alpha = randomrange(127,255);
				p->scale = randomrange(6, 10);
				VectorCopy(start, p->org);
				p->color = r_smokecolor.value;
				if ((GetPointContents(p->org) == CONTENTS_EMPTY)||
					(GetPointContents(p->org) == CONTENTS_SOLID))
				{
					p->texnum = smoke1particletexture + (rand() & 3);
					p->type = pt_fire;
					dec = 16;
				}
				else
				{
					p->texnum = bubbleparticletexture;
					p->type = pt_bubble;
					dec = 32; // rocket trail
				}
				break;

			case 1:	// grenade trail
			
				p->vel[0] = randomrange(-10, 10);
				p->vel[1] = randomrange(-10, 10);
				p->vel[2] = randomrange(5, 35);
				p->alpha = randomrange(127,255);
				p->scale = randomrange(6, 10);
				VectorCopy(start, p->org);
				p->color = r_smokecolor.value;
				if ((GetPointContents(p->org) == CONTENTS_EMPTY)||
					(GetPointContents(p->org) == CONTENTS_SOLID))
				{
				p->texnum = smoke1particletexture + (rand() & 3);
					p->type = pt_fire;
					dec = 16;
				}
				else
				{
					p->texnum = bubbleparticletexture;
					p->type = pt_bubble;
					dec = 32; // greande trail
				}
				break;
				
			case 2:	// blood
				dec = 16; // sparse trail
				p->texnum = blood1particletexture + (rand() & 3);
				p->scale = randomrange(6,10);
				p->alpha = 255;
				p->color = 67 + (rand()&3);
				p->type = pt_bloodcloud;
				p->die = cl.time + 2;
				for (j=0 ; j<3 ; j++)
				{
					p->vel[j] = (rand()&15)-8;
					p->org[j] = start[j] + ((rand()&3)-2);
				}
				break;

			case 3:
			case 5:	// tracer
				dec = 8;
				p->texnum = flareparticletexture;
				p->scale = 4;
				p->alpha = 255;
				p->die = cl.time + 0.2; //5;
				p->type = pt_static;
				if (type == 3)
				{
					p->color = 52 + ((tracercount&4)<<1);
				}
				else
				{
					p->color = 230 + ((tracercount&4)<<1);
				}

				tracercount++;

				VectorCopy (start, p->org);
				if (tracercount & 1)
				{
					p->vel[0] = 30*vec[1];
					p->vel[1] = 30*-vec[0];
				}
				else
				{
					p->vel[0] = 30*-vec[1];
					p->vel[1] = 30*vec[0];
				}
				break;

			case 4:	// slight blood
				dec = 16; // sparse trail
				p->texnum = blood1particletexture + (rand() & 3);
				p->scale = randomrange(6,10);
				p->alpha = 255;
				p->color = 67 + (rand()&3);
				p->type = pt_bloodcloud;
				p->die = cl.time + 2;
				for (j=0 ; j<3 ; j++)
				{
					p->vel[j] = (rand()&15)-8;
					p->org[j] = start[j] + ((rand()&3)-2);
				}
				break;

			case 6:	// voor trail
				dec = 16; // sparse trail
				p->texnum = flareparticletexture;
				p->scale = 4;
				p->alpha = 255;
				p->color = 9*16 + 8 + (rand()&3);
				p->type = pt_fadespark2;
				p->die = cl.time + 2;
				for (j=0 ; j<3 ; j++)
				{
					p->vel[j] = (rand()&15)-8;
					p->org[j] = start[j] + ((rand()&3)-2);
				}
				break;
			case 7:	// spike trail
				dec = 32; // spike trail
				p->alpha = randomrange(127,255);
				p->scale = randomrange(6, 10);
				p->vel[2] = 40;
				p->color = 244;
				p->type = pt_bubble;
				VectorCopy(start, p->org);
				p->texnum = bubbleparticletexture;
				if ((GetPointContents(p->org) == CONTENTS_EMPTY)||
					(GetPointContents(p->org) == CONTENTS_SOLID))
				{
					p->die = -1;
				}
		}
		
		VectorMA (start, dec, vec, start);
		len -= dec;
	}
}

void R_RocketTrail2 (vec3_t start, vec3_t end, int color)
{
	vec3_t		vec;
	float		len;
	particle_t	*p;
	static int	tracercount;

	if (!r_particles.value) 
		return;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);
	while (len > 0)
	{
		len -= 3;

		if (!free_particles)
			return;
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		
		VectorCopy (vec3_origin, p->vel);

		p->texnum = flareparticletexture;
		p->scale = 4;
		p->alpha = 256;
		p->color = color;
		p->type = pt_fire;
		p->die = cl.time + 1;
		VectorCopy(start, p->org);
//		for (j=0 ; j<3 ; j++)
//			p->org[j] = start[j] + ((rand()&15)-8);

		VectorAdd (start, vec, start);
	}
}


/*
===============
R_DrawParticles
===============
*/

extern	cvar_t	sv_gravity;

void R_DrawParticles (void)
{
	particle_t		*p, *kill;
	int				i, texnum;
	float			grav, grav1, time1, time2, time3, dvel, frametime, scale, scale2;
	byte			*color24, tmpAlpha;
	vec3_t			up, right, uprightangles, forward2, up2, right2;
	vec3_t			poldorg;	//dave

	if (!r_particles.value) 
		return;

	if (!active_particles)
		return;

	texnum = particletexture;
    glBindTexture(GL_TEXTURE_2D, texnum);
	glEnable (GL_BLEND);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glShadeModel(GL_FLAT);
	glDepthMask(false);
	glBegin (GL_TRIANGLES);

	VectorScale (vup, 1.5, up);
	VectorScale (vright, 1.5, right);

	uprightangles[0] = 0;
	uprightangles[1] = r_refdef.viewangles[1];
	uprightangles[2] = 0;
	AngleVectors (uprightangles, forward2, right2, up2);

	frametime = cl.time - cl.oldtime;
	time3 = frametime * 15;
	time2 = frametime * 10; // 15;
	time1 = frametime * 5;
	grav = (grav1 = frametime * sv_gravity.value) * 0.05;
	dvel = 4*frametime;

	for ( ;; ) 
	{
		kill = active_particles;
		if (kill && kill->die < cl.time)
		{
			active_particles = kill->next;
			kill->next = free_particles;
			free_particles = kill;
			continue;
		}
		break;
	}

	for (p=active_particles ; p ; p=p->next)
	{
		for ( ;; )
		{
			kill = p->next;
			if (kill && kill->die < cl.time)
			{
				p->next = kill->next;
				kill->next = free_particles;
				free_particles = kill;
				continue;
			}
			break;
		}

		scale = p->scale * -0.5;scale2 = p->scale * 0.5;
		if (p->texnum != texnum)
		{
			texnum = p->texnum;
			glEnd();
			glBindTexture(GL_TEXTURE_2D, texnum);
			glBegin(GL_TRIANGLES);
		}

		color24 = (byte *) &d_8to24table[(int)p->color];

		tmpAlpha = (byte) (p->alpha * r_smokealpha.value);

		if ((p->texnum != smoke1particletexture) &&
			(p->texnum != smoke2particletexture) &&
			(p->texnum != smoke3particletexture) &&
			(p->texnum != smoke4particletexture))
		{
			tmpAlpha	= (byte) p->alpha;
		}

		glColor4ub(color24[0], color24[1], color24[2], tmpAlpha);

		glTexCoord2f (0,1);
		glVertex3f (p->org[0] + up[0]*scale  + right[0]*scale , p->org[1] + up[1]*scale  + right[1]*scale , p->org[2] + up[2]*scale  + right[2]*scale );
		glTexCoord2f (0,0);
		glVertex3f (p->org[0] + up[0]*scale2 + right[0]*scale , p->org[1] + up[1]*scale2 + right[1]*scale , p->org[2] + up[2]*scale2 + right[2]*scale );
		glTexCoord2f (1,1);
		glVertex3f (p->org[0] + up[0]*scale  + right[0]*scale2, p->org[1] + up[1]*scale  + right[1]*scale2, p->org[2] + up[2]*scale  + right[2]*scale2);
		glTexCoord2f (0,0);
		glVertex3f (p->org[0] + up[0]*scale2 + right[0]*scale , p->org[1] + up[1]*scale2 + right[1]*scale , p->org[2] + up[2]*scale2 + right[2]*scale );
		glTexCoord2f (1,0);
		glVertex3f (p->org[0] + up[0]*scale2 + right[0]*scale2, p->org[1] + up[1]*scale2 + right[1]*scale2, p->org[2] + up[2]*scale2 + right[2]*scale2);
		glTexCoord2f (1,1);
		glVertex3f (p->org[0] + up[0]*scale  + right[0]*scale2, p->org[1] + up[1]*scale  + right[1]*scale2, p->org[2] + up[2]*scale  + right[2]*scale2);

		//dave kill particles that hit walls.
		VectorCopy(p->org, poldorg);
		//dave

		p->org[0] += p->vel[0]*frametime;
		p->org[1] += p->vel[1]*frametime;
		p->org[2] += p->vel[2]*frametime;

		//dave kill particles that hit walls
		//pmtrace_t trace; 
		if (partrace(poldorg, p->org))
		{
			p->die = 0;
		}
		//dave	
		
		switch (p->type)
		{
		case pt_static:
			break;

		case pt_fire:
			p->scale += frametime * 32;
			p->alpha -= frametime * 128;
			p->color = r_smokecolor.value;
			if (p->alpha < 1)
				p->die = -1;
			break;

		case pt_bubble:
			p->color = 244;
			p->vel[0] = randomrange(-15,15);
			p->vel[1] = randomrange(-15,15);
			p->vel[2] = randomrange(40,60);
			p->scale = randomrange(2,3);
			if(	(GetPointContents(p->org) == CONTENTS_EMPTY)||
				(GetPointContents(p->org) == CONTENTS_SOLID))
			{
				p->die = -1;
			}
			break;

		case pt_explode:
			p->scale += frametime * 16;
			p->alpha -= frametime * 256;
			p->ramp += time2;
			if (p->ramp >=12)
				p->die = -1;
			else
				p->color = ramp1[(int)p->ramp];
				p->vel[2] -= grav1;
			break;

		case pt_explode2:
			p->scale += frametime * 16;
			p->alpha -= frametime * 256;
			p->ramp += time3;
			if (p->ramp >=12)
				p->die = -1;
			else
				p->color = ramp2[(int)p->ramp];
				p->vel[2] -= grav1;
			break;

		case pt_blob:
			for (i=0 ; i<3 ; i++)
				p->vel[i] += p->vel[i]*dvel;
			p->vel[2] -= grav;
			break;

		case pt_blob2:
			for (i=0 ; i<2 ; i++)
				p->vel[i] -= p->vel[i]*dvel;
			p->vel[2] -= grav;
			break;

		case pt_grav:
			p->vel[2] -= grav1;
			break;

		case pt_slowgrav:
			p->vel[2] -= grav;
			break;

		case pt_dust:
			p->ramp += time1;
			p->scale -= frametime * 4;
			p->color = ramp3[(int)p->ramp];
			p->vel[2] -= grav1;
			break;

		case pt_blood:
			p->ramp += time1;
			p->alpha -= frametime * 128;
			p->color = 75;
			p->vel[2] -= grav1;
			break;
		
		case pt_snow:
			if (cl.time > p->time2)
			{
				p->time2 = cl.time + 0.4;
				p->vel[0] = (rand()&63)-32;
				p->vel[1] = (rand()&63)-32;
			}
			break;

		case pt_bulletpuff:
			p->scale += frametime * 32;
			p->alpha -= frametime * 128;
			p->color = r_smokecolor.value;
			if (p->alpha < 1)
				p->die = -1;
			break;

		case pt_bloodcloud:
			p->alpha -= frametime * 128;
			p->vel[2] -= grav;
			if (p->alpha < 1)
				p->die = -1;
			break;

		case pt_fadespark:
			p->alpha -= frametime * 256;
			p->vel[2] -= grav;
			if (p->alpha < 1)
				p->die = -1;
			break;

		case pt_fadespark2:
			p->alpha -= frametime * 512;
			p->vel[2] -= grav;
			if (p->alpha < 1)
				p->die = -1;
			break;
		}
	}

	glEnd ();
	glShadeModel(GL_SMOOTH);
	glDisable (GL_BLEND);
	glDepthMask(true);
}

int GetPointContents (vec3_t point)
{
	mleaf_t *leaf;

	leaf = Mod_PointInLeaf (point, cl.worldmodel);
	return (leaf->contents);
}
