/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "quakedef.h"

int		particletexture;
int		smoke1particletexture;
int		smoke2particletexture;
int		smoke3particletexture;
int		smoke4particletexture;
int		bloodparticletexture;
int		flareparticletexture;
int		bubbleparticletexture;
int		snowparticletexture;
int		rainparticletexture;

particle_t	*active_particles, *free_particles, *particles;

int			r_numparticles;

vec3_t		r_pright, r_pup, r_ppn;

void R_InitParticleTexture (void)
{
	particletexture			= loadtextureimage ("particles/particle.tga", 0, 0, true, true);
	smoke1particletexture	= loadtextureimage ("particles/smoke1.tga", 0, 0, true, true);
	smoke2particletexture	= loadtextureimage ("particles/smoke2.tga", 0, 0, true, true);
	smoke3particletexture	= loadtextureimage ("particles/smoke3.tga", 0, 0, true, true);
	smoke4particletexture	= loadtextureimage ("particles/smoke4.tga", 0, 0, true, true);
	bloodparticletexture	= loadtextureimage ("particles/blood.tga", 0, 0, true, true);
	flareparticletexture	= loadtextureimage ("particles/particle.tga", 0, 0, true, true);
	bubbleparticletexture	= loadtextureimage ("particles/bubble.tga", 0, 0, true, true);
	snowparticletexture		= loadtextureimage ("particles/snow.tga", 0, 0, true, true);
	rainparticletexture		= loadtextureimage ("particles/rain.tga", 0, 0, true, true);
}

/*
===============
R_InitParticles
===============
*/
void R_InitParticles (void)
{
	int		i = COM_CheckParm ("-particles");

	r_numparticles = 2048;

	if (i)
	{
		r_numparticles = (int)(atoi(com_argv[i+1]));
	}

	particles = (particle_t *) Hunk_AllocName (r_numparticles * sizeof(particle_t), "particles");

	R_InitParticleTexture ();
}

/*
===============
R_EntityParticles
===============
*/

extern float	r_avertexnormals[162][3];
vec3_t			avelocities[162];

void R_EntityParticles (entity_t *ent)
{
	int			count;
	int			i;
	particle_t	*p;
	float		angle;
	float		sr, sp, sy, cr, cp, cy;
	vec3_t		forward;
	float		dist;

	if (!r_particles.value)
		return;

	dist	= 64;
	count	= 50;

	if (!avelocities[0][0])
	{
		for (i=0 ; i<486 ; i++)
		{
			avelocities[0][i] = (rand()&255) * 0.01;
		}
	}


	for (i=0 ; i<162 ; i++)
	{
		angle				= cl.time * avelocities[i][0];
		sy					= sin(angle);
		cy					= cos(angle);
		angle				= cl.time * avelocities[i][1];
		sp					= sin(angle);
		cp					= cos(angle);
		angle				= cl.time * avelocities[i][2];
		sr					= sin(angle);
		cr					= cos(angle);
	
		forward[0]			= cp*cy;
		forward[1]			= cp*sy;
		forward[2]			= -sp;

		if (!free_particles)
			return;
		p					= free_particles;
		free_particles		= p->next;
		p->next				= active_particles;
		active_particles	= p;

		p->texnum			= flareparticletexture;
		p->scale			= (rand() & 3) +1;
		p->alpha			= 255;
		p->die				= cl.time + 0.1;
		p->color			= 251;
		p->type				= pt_explode;
		
		p->origin[0]		= ent->origin[0] + r_avertexnormals[i][0]*dist + forward[0]*16;			
		p->origin[1]		= ent->origin[1] + r_avertexnormals[i][1]*dist + forward[1]*16;			
		p->origin[2]		= ent->origin[2] + r_avertexnormals[i][2]*dist + forward[2]*16;			
	}
}


/*
===============
R_ClearParticles
===============
*/
void R_ClearParticles (void)
{
	int		i;

	free_particles		= &particles[0];
	active_particles	= NULL;

	for ( i=0 ; i<r_numparticles ; i++ )
		particles[i].next = &particles[i+1];

	particles[r_numparticles-1].next = NULL;
}


void R_ReadPointFile_f (void)
{
	FILE	*f;
	vec3_t	origin;
	int		r;
	int		c;
	particle_t	*p;
	char	name[MAX_OSPATH];

	if (!r_particles.value)
		return;

	sprintf (name,"maps/%s.pts", sv.name);

	COM_FOpenFile (name, &f);
	if (!f)
	{
		Con_Printf ("couldn't open %s\n", name);
		return;
	}
	
	Con_Printf ("Reading %s...\n", name);
	c = 0;
	for ( ;; )
	{
		r = fscanf (f,"%f %f %f\n", &origin[0], &origin[1], &origin[2]);
		if (r != 3)
			break;
		c++;
		
		if (!free_particles)
		{
			Con_Printf ("Not enough free particles\n");
			break;
		}
		p = free_particles;
		free_particles = p->next;
		p->next = active_particles;
		active_particles = p;
		
		p->texnum = particletexture;
		p->scale = (rand() & 3) +1;
		p->alpha = 255;
		p->die = 99;
		p->color = (-c)&15;
		p->type = pt_static;
		VectorCopy (vec3_origin, p->velocity);
		VectorCopy (origin, p->origin);
	}

	fclose (f);
	Con_Printf ("%i points read\n", c);
}

/*
===============
R_ParseParticleEffect

Parse an effect out of the server message
===============
*/
void R_ParseParticleEffect (void)
{
	vec3_t		origin, direction;
	int			count, color;

	if (!r_particles.value)
		return;

	origin[0]		= MSG_ReadCoord ();
	origin[1]		= MSG_ReadCoord ();
	origin[2]		= MSG_ReadCoord ();

	direction[0]	= MSG_ReadChar () * 0.0625;
	direction[1]	= MSG_ReadChar () * 0.0625;
	direction[2]	= MSG_ReadChar () * 0.0625;

	count			= MSG_ReadByte ();
	color			= MSG_ReadByte ();

	R_RunParticleEffect (origin, direction, color, count);
}
	
/*
===============
R_ParticleExplosion

===============
*/
void R_ParticleExplosion (vec3_t origin)
{
	int			i;
	particle_t	*p;

	if (!r_particles.value)
		return;

	for (i=0 ; i<256 ; i++)
	{
		if (!free_particles)
			return;
		p					= free_particles;
		free_particles		= p->next;
		p->next				= active_particles;
		active_particles	= p;

		p->scale			= (rand() & 3) +1;
		p->alpha			= 255;
		p->texnum			= flareparticletexture;
		p->die				= cl.time + 5;
		p->color			= 252;
		p->type				= pt_explode;
		p->change			= cl.time + 0.2;

		p->origin[0]		= origin[0] + ((rand() & 31) - 16);
		p->origin[1]		= origin[1] + ((rand() & 31) - 16);
		p->origin[2]		= origin[2] + ((rand() & 31) - 16);

		p->velocity[0]		= (rand() & 511) - 256;
		p->velocity[1]		= (rand() & 511) - 256;
		p->velocity[2]		= (rand() & 511) - 256;
	}
}

/*
===============
R_RunParticleEffect

===============
*/

void R_RunParticleEffect (vec3_t origin, vec3_t direction, int color, int count)
{
	int			i;
	particle_t	*p;

	if (!r_particles.value)
		return;

	if (count == 128)
	{
		for (i=0 ; i<count ; i++)
		{	// rocket explosion
			if (!free_particles)
				return;
			p					= free_particles;
			free_particles		= p->next;
			p->next				= active_particles;
			active_particles	= p;
						
			p->texnum			= flareparticletexture;
			p->scale			= (rand() & 3) +1;
			p->alpha			= 255;
			p->die				= cl.time + 5;
			p->color			= 252;
			p->type				= pt_explode;

			p->origin[0]		= origin[0] + ((rand() & 31) - 16);
			p->origin[1]		= origin[1] + ((rand() & 31) - 16);
			p->origin[2]		= origin[2] + ((rand() & 31) - 16);

			p->velocity[0]		= (rand() & 511) - 256;
			p->velocity[1]		= (rand() & 511) - 256;
			p->velocity[2]		= (rand() & 511) - 256;
		}
		return;
	}

	for (i=0 ; i<count ; i++)
	{
		if (!free_particles)
			return;
		p					= free_particles;
		free_particles		= p->next;
		p->next				= active_particles;
		active_particles	= p;

		p->texnum			= bloodparticletexture;
		p->scale			= (rand() & 3) +1;
		p->alpha			= 255;
		p->die				= cl.time + 5;
		p->color			= (color&~7) + (rand()&7);
		p->type				= pt_blood2;

		p->origin[0]		= origin[0] + ((rand() & 15) - 8);
		p->origin[1]		= origin[1] + ((rand() & 15) - 8);
		p->origin[2]		= origin[2] + ((rand() & 15) - 8);

		p->velocity[0]		= direction[0] * 15;
		p->velocity[1]		= direction[1] * 15;
		p->velocity[2]		= direction[2] * 15;
	}
}

/*
===============
R_SparkShower

===============
*/
void R_SparkShower (vec3_t origin, vec3_t direction, int count)
{
	int			i;
	particle_t	*p;

	if (!r_particles.value)
		return;

	if (!free_particles)
		return;
	p					= free_particles;
	free_particles		= p->next;
	p->next				= active_particles;
	active_particles	= p;

	p->scale			= (rand() & 3) +1;
	p->alpha			= 255;
	p->texnum			= smoke1particletexture + (rand() & 3);
	p->type				= pt_bulletpuff;

	p->velocity[0]		= (rand() & 15) - 8;
	p->velocity[1]		= (rand() & 15) - 8;
	p->velocity[2]		= (rand() & 31) + 5;

	p->color			= 12;
	p->die				= cl.time + 5;

	VectorCopy(origin, p->origin);

	for (i=0 ; i<count ; i++)
	{
		if (!free_particles)
			return;
		p					= free_particles;
		free_particles		= p->next;
		p->next				= active_particles;
		active_particles	= p;

		p->texnum			= flareparticletexture;
		p->scale			= (rand() & 3) +1;
		p->alpha			= 255;
		p->die				= cl.time + 5;
		p->type				= pt_explode;
		p->color			= 252;

		p->origin[0]		= origin[0] + ((rand() & 7) - 4);
		p->origin[1]		= origin[1] + ((rand() & 7) - 4);
		p->origin[2]		= origin[2] + ((rand() & 7) - 4);

		p->velocity[0]		= direction[0] + (rand() % 255) - 128;
		p->velocity[1]		= direction[1] + (rand() % 255) - 128;
		p->velocity[2]		= direction[2] + (rand() % 255) - 128;
	}
}

/*
===============
R_Snow
===============
*/
void R_Snow (vec3_t origin, vec3_t direction)
{
	particle_t	*p;
	int			contents;

	if (!r_particles.value)
		return;

	if (!free_particles)
		return;
	p					= free_particles;
	free_particles		= p->next;
	p->next				= active_particles;
	active_particles	= p;

	contents			= Mod_PointInLeaf(p->origin, cl.worldmodel)->contents;

	if ((contents		!= CONTENTS_EMPTY) && 
		(contents		!= CONTENTS_FOG))
	{
		p->die			= -1;
	}
	p->die				= cl.time + 10;

	p->scale			= (rand() & 3) +1;
	p->alpha			= 200;
	p->texnum			= snowparticletexture;
	p->type				= pt_snow;
	p->color			= 15;

	VectorCopy(origin, p->origin);

	p->velocity[0]		= 0;
	p->velocity[1]		= 0;
	p->velocity[2]		= -100;
}

/*
===============
R_Rain
===============
*/
void R_Rain (vec3_t origin, vec3_t direction)
{
	particle_t	*p;
	int			contents;

	if (!r_particles.value)
		return;

	if (!free_particles)
		return;
	p					= free_particles;
	free_particles		= p->next;
	p->next				= active_particles;
	active_particles	= p;

	contents			= Mod_PointInLeaf(p->origin, cl.worldmodel)->contents;

	if ((contents		!= CONTENTS_EMPTY) && 
		(contents		!= CONTENTS_FOG))
	{
		p->die			= -1;
	}
	p->die				= cl.time + 10;

	p->scale			= (rand() & 3) +1;
	p->alpha			= 200;

	p->texnum			= rainparticletexture;
	p->type				= pt_rain;
	p->color			= 15;

	VectorCopy(origin, p->origin);

	p->velocity[0]		= 0;
	p->velocity[1]		= 0;
	p->velocity[2]		= -400;
}

/*
===============
R_LavaSplash

===============
*/
void R_LavaSplash (vec3_t origin)
{
	int			i, j;
	particle_t	*p;
	float		velocity;
	vec3_t		direction;

	if (!r_particles.value)
		return;

	for (i=-16 ; i<16 ; i+=2)
	{	
		for (j=-16 ; j<16 ; j+=2)
		{
			if (!free_particles)
				return;
			p					= free_particles;
			free_particles		= p->next;
			p->next				= active_particles;
			active_particles	= p;
		
			p->texnum			= flareparticletexture;
			p->scale			= (rand() & 3) +1;
			p->alpha			= 255;
			p->die				= cl.time + 2;
			p->color			= 231;
			p->type				= pt_fadespark;
				
			direction[0]		= j*8 + (rand()&7);
			direction[1]		= i*8 + (rand()&7);
			direction[2]		= 256;
	
			p->origin[0]		= origin[0] + direction[0];
			p->origin[1]		= origin[1] + direction[1];
			p->origin[2]		= origin[2] + (rand() & 63);
	
			velocity			= 50 + (rand() & 63);
			VectorNormalize(direction);
			VectorScale (direction, velocity, p->velocity);
		}
	}
}

/*
===============
R_TeleportSplash

===============
*/
void R_TeleportSplash (vec3_t origin)
{
	int			i, j, k;
	particle_t	*p;
	float		velocoty;
	vec3_t		direction;

	if (!r_particles.value)
		return;

	for (i=-16 ; i<16 ; i+=4)
	{	
		for (j=-16 ; j<16 ; j+=4)
		{
			for (k=-24 ; k<32 ; k+=4)
			{
				if (!free_particles)
					return;
				p					= free_particles;
				free_particles		= p->next;
				p->next				= active_particles;
				active_particles	= p;
		
				p->texnum			= flareparticletexture;
				p->scale			= (rand() & 3) +1;
				p->alpha			= 255;
				p->die				= cl.time + 1;
				p->color			= 254;
				p->type				= pt_fadespark;
	
				p->origin[0]		= origin[0] + i + (rand() & 7);
				p->origin[1]		= origin[1] + j + (rand() & 7);
				p->origin[2]		= origin[2] + k + (rand() & 7);
	
				p->velocity[0]		= i*2 + (rand() & 31) - 16;
				p->velocity[1]		= j*2 + (rand() & 31) - 16;
				p->velocity[2]		= k*2 + (rand() & 31) + 24;
			}
		}
	}
}

void R_RocketTrail (vec3_t start, vec3_t end)
{
	int			i, contents;
	float		len;
	vec3_t		vec;
	particle_t	*p;

	if (!r_particles.value)
		return;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);

	VectorScale(vec, 48, vec);

	for( i = 0 ; i < len ; i += 48 )
	{
		if (!free_particles)
			return;
		p					= free_particles;
		free_particles		= p->next;
		p->next				= active_particles;
		active_particles	= p;

		VectorCopy(start, p->origin);

		contents			= Mod_PointInLeaf(p->origin, cl.worldmodel)->contents;

		if ((contents		== CONTENTS_EMPTY) ||
			(contents		== CONTENTS_SOLID) ||
			(contents		== CONTENTS_FOG))
		{
			p->die			= cl.time + 10;
			p->color		= 12;
			p->alpha		= 255;
			p->scale		= (rand() & 3) +1;
			p->texnum		= smoke1particletexture + (rand() & 3);
			p->type			= pt_fire;

			p->velocity[0]	= (rand() & 15) - 8;
			p->velocity[1]	= (rand() & 15) - 8;
			p->velocity[2]	= (rand() & 31) + 15;

			VectorAdd(start, vec, start);

			continue;
		}

		p->die			= cl.time + 10;
		p->color		= 244;
		p->alpha		= 255;
		p->scale		= (rand() & 3) +1;
		p->texnum		= bubbleparticletexture;
		p->type			= pt_bubble;

		p->velocity[0]	= 0;
		p->velocity[1]	= 0;
		p->velocity[2]	= 20;
	}
	VectorAdd(start, vec, start);
}

void R_BloodTrail (vec3_t start, vec3_t end)
{
	int			i;
	float		len;
	vec3_t		vec;
	particle_t	*p;

	if (!r_particles.value)
		return;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);

	VectorScale(vec, 16, vec);

	for( i = 0 ; i < len ; i += 16 )
	{
		if (!free_particles)
			return;
		p					= free_particles;
		free_particles		= p->next;
		p->next				= active_particles;
		active_particles	= p;

		p->die				= cl.time + 3;
		p->color			= 251;
		p->alpha			= 255;
		p->scale			= (rand() & 3) +1;
		p->texnum			= bloodparticletexture;
		p->type				= pt_blood;

		p->velocity[0]		= (rand() & 15) - 8;
		p->velocity[1]		= (rand() & 15) - 8;
		p->velocity[2]		= (rand() & 15) - 8;

		p->origin[0]		= start[0] + ((rand() & 3) - 2);
		p->origin[1]		= start[1] + ((rand() & 3) - 2);
		p->origin[2]		= start[2] + ((rand() & 3) - 2);

		VectorAdd(start, vec, start);
	}
}

void R_TracerTrail (vec3_t start, vec3_t end, byte color)
{
	int			i;
	float		len;
	vec3_t		vec;
	static int	tracercount;
	particle_t	*p;

	if (!r_particles.value)
		return;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);

	VectorScale(vec, 4, vec);

	for( i = 0 ; i < len ; i += 4 )
	{
		if (!free_particles)
			return;
		p					= free_particles;
		free_particles		= p->next;
		p->next				= active_particles;
		active_particles	= p;

		VectorCopy (start, p->origin);

		p->die				= cl.time + 0.3;
		p->color			= color;
		p->alpha			= 255;
		p->scale			= (rand() & 3) +1;
		p->texnum			= flareparticletexture;
		p->type				= pt_static;

		p->velocity[0]		= 30*-vec[1];
		p->velocity[1]		= 30*vec[0];
		p->velocity[2]		= 0;

		if (++tracercount & 1)
		{
			p->velocity[0]	= -(p->velocity[0]);
			p->velocity[1]	= -(p->velocity[1]);
		}
		VectorAdd(start, vec, start);
	}
}

void R_VoorTrail (vec3_t start, vec3_t end)
{
	int			i;
	float		len			= 0.0f;
	vec3_t		vec;
	particle_t	*p;

	if (!r_particles.value)
		return;

	VectorSubtract (end, start, vec);
	len = VectorNormalize (vec);

	VectorScale(vec, 4, vec);

	for( i = 0 ; i < len ; i += 4 )
	{
		if (!free_particles)
			return;
		p					= free_particles;
		free_particles		= p->next;
		p->next				= active_particles;
		active_particles	= p;

		p->die				= cl.time + 2;
		p->color			= 144;
		p->alpha			= 255;
		p->scale			= (rand() & 3) +1;
		p->texnum			= flareparticletexture;
		p->type				= pt_fadespark;

		p->velocity[0]		= (rand() & 15) - 8;
		p->velocity[1]		= (rand() & 15) - 8;
		p->velocity[2]		= (rand() & 15) - 8;

		p->origin[0]		= start[0] + ((rand() & 3) - 2);
		p->origin[1]		= start[1] + ((rand() & 3) - 2);
		p->origin[2]		= start[2] + ((rand() & 3) - 2);

		VectorAdd(start, vec, start);
	}
}

/*
===============
R_DrawParticles
===============
*/

extern	cvar_t	sv_gravity;

void R_DrawParticles (void)
{
	particle_t	*p, *kill;
	int			i;

	float		frametime	= cl.time - cl.oldtime;
	float		time		= frametime * 10;
	float		fastgrav	= frametime * sv_gravity.value * 0.5;
	float		slowgrav	= fastgrav * 0.1;
	byte		alpha;

	byte		*color24;
	vec3_t		up			= { vup[0] * 0.75f, vup[1] * 0.75f, vup[2] * 0.75f };
	vec3_t		right		= { vright[0] * 0.75f, vright[1] * 0.75f, vright[2] * 0.75f };

	int			contents;

	if (!r_particles.value)
		return;

	if (!active_particles)
		return;

	glEnable (GL_BLEND);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glShadeModel(GL_FLAT);
	glDepthMask(false);

	for ( ;; ) 
	{
		kill = active_particles;
		if (kill && kill->die < cl.time)
		{
			active_particles = kill->next;
			kill->next = free_particles;
			free_particles = kill;
			continue;
		}
		break;
	}

	for (p=active_particles ; p ; p=p->next)
	{
		for ( ;; )
		{
			kill = p->next;
			if (kill && kill->die < cl.time)
			{
				p->next = kill->next;
				kill->next = free_particles;
				free_particles = kill;
				continue;
			}
			break;
		}
    
		GL_Bind(p->texnum);
		glBegin (GL_QUADS);

		contents = Mod_PointInLeaf(p->origin, cl.worldmodel)->contents;

		color24 = (byte *) &d_8to24table[(int)p->color];

		alpha = p->alpha;

		glColor4ub(color24[0], color24[1], color24[2], alpha);

		glTexCoord2f (0,0);
		glVertex3f (p->origin[0] + (up[0] + right[0]) * p->scale , 
					p->origin[1] + (up[1] + right[1]) * p->scale , 
					p->origin[2] + (up[2] + right[2]) * p->scale );
		glTexCoord2f (1,0);
		glVertex3f (p->origin[0] + (- up[0] + right[0]) * p->scale , 
					p->origin[1] + (- up[1] + right[1]) * p->scale , 
					p->origin[2] + (- up[2] + right[2]) * p->scale );
		glTexCoord2f (1,1); 
		glVertex3f (p->origin[0] - (up[0] + right[0]) * p->scale, 
					p->origin[1] - (up[1] + right[1]) * p->scale, 
					p->origin[2] - (up[2] + right[2]) * p->scale);
		glTexCoord2f (0,1);
		glVertex3f (p->origin[0] + (up[0] - right[0]) * p->scale, 
					p->origin[1] + (up[1] - right[1]) * p->scale, 
					p->origin[2] + (up[2] - right[2]) * p->scale);

		p->origin[0] += p->velocity[0] * frametime;
		p->origin[1] += p->velocity[1] * frametime;
		p->origin[2] += p->velocity[2] * frametime;

		if (contents == CONTENTS_SOLID)
			p->die = -1;

		switch (p->type)
		{
		case pt_static:
			break;

		case pt_fire:
			p->scale += frametime * 24;
			p->alpha -= frametime * 256;
			if (p->alpha < 1)
				p->die = -1;
			break;

		case pt_bubble:

			if ((contents == CONTENTS_EMPTY) ||
				(contents == CONTENTS_FOG))
			{
				p->die = -1;
			}
			
			p->velocity[0] = (rand() & 15) - 8;
			p->velocity[1] = (rand() & 15) - 8;
			p->velocity[2] += (rand() & 7);

			if (p->velocity[2] < 0)
				p->velocity[2] += 15;
			break;

		case pt_explode:
			p->velocity[2] -= fastgrav;

			if ((contents != CONTENTS_EMPTY) &&
				(contents != CONTENTS_FOG))
			{
				p->texnum = bubbleparticletexture;
				p->color = 244;
				if (cl.time > p->change)
				{
					p->type = pt_bubble;
				}
			}
			else
			{
				if (p->color == 244)
					p->die = -1;
			}
			break;

		case pt_blood:
			p->alpha -= frametime * 128;
			p->velocity[2] -= slowgrav;
			if (p->alpha < 1)
				p->die = -1;
			break;

		case pt_blood2:
			p->alpha -= frametime * 128;
			p->velocity[2] -= fastgrav;
			if (p->alpha < 1)
				p->die = -1;
			break;

		case pt_snow:
			p->velocity[0] += (rand() & 15) - 8;
			p->velocity[1] += (rand() & 15) - 8;
			break;			

		case pt_rain:
			break;

		case pt_bulletpuff:
			if (((contents != CONTENTS_EMPTY) &&
				(contents != CONTENTS_FOG)) ||
				(p->alpha < 1))
				p->die = -1;
			p->scale += frametime * 24;
			p->alpha -= frametime * 256;
			break;

		case pt_fadespark:
			if (p->alpha < 1)
				p->die = -1;
			p->alpha -= frametime * 256;
			p->velocity[2] -= slowgrav;
			break;
		}
	}

	glEnd ();
	glShadeModel(GL_SMOOTH);
	glDisable (GL_BLEND);
	glDepthMask(true);
	glColor4f(1, 1, 1, 1);
}