/*
Copyright (C) 1996-2001 Id Software, Inc.
Copyright (C) 2002-2012 John Fitzgibbons and others

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 3
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// r_light.c

#include "quakedef.h"

int	r_dlightframecount;

extern cvar_t r_flatlightstyles; //johnfitz

/*
==================
R_AnimateLight
==================
*/
void R_AnimateLight (void)
{
	int			i,j,k;

//
// light animations
// 'm' is normal light, 'a' is no light, 'z' is double bright
#ifdef SUPPORTS_DEMO_REWIND_HARD_ANIMS_TIME // Baker change
	i = (int)(cl.ctime*10);
#else // Baker change +
	i = (int)(cl.time*10);
#endif // Baker change -

	for (j=0 ; j<MAX_LIGHTSTYLES ; j++)
	{
		if (!cl_lightstyle[j].length)
		{
			d_lightstylevalue[j] = 256;
			continue;
		}
		//johnfitz -- r_flatlightstyles
		if (r_flatlightstyles.value == 2)
			k = cl_lightstyle[j].peak - 'a';
		else if (r_flatlightstyles.value == 1)
			k = cl_lightstyle[j].average - 'a';
		else
		{
			k = i % cl_lightstyle[j].length;
			k = cl_lightstyle[j].map[k] - 'a';
		}
		d_lightstylevalue[j] = k*22;
		//johnfitz
	}
}

/*
=============================================================================

DYNAMIC LIGHTS BLEND RENDERING (gl_flashblend 1)

=============================================================================
*/

void AddLightBlend (float r, float g, float b, float a2)
{
	float	a;

	v_blend[3] = a = v_blend[3] + a2*(1-v_blend[3]);

	a2 = a2/a;

	v_blend[0] = v_blend[1]*(1-a2) + r*a2;
	v_blend[1] = v_blend[1]*(1-a2) + g*a2;
	v_blend[2] = v_blend[2]*(1-a2) + b*a2;
}

#ifdef SPEEDUP_PRECALC_FLASHBLEND_BUBBLE // Baker change
static float	bubble_sin[17], bubble_cos[17];

void R_Init_FlashBlend_Bubble (void)
{
	int	i;
	float	a;

	for (i = 16 ; i >= 0 ; i --)
	{
		a = i/16.0 * M_PI * 2;
		bubble_sin[i] = sin(a);
		bubble_cos[i] = cos(a);
	}
}
#endif // Baker change +

void R_RenderDlight (dlight_t *light)
{
	int		i, j;
	float	a;
	vec3_t	v;
	float	rad;

	rad = light->radius * 0.35;

	VectorSubtract (light->origin, r_origin, v);
	if (Length (v) < rad)
	{	// view is inside the dlight
		AddLightBlend (1, 0.5, 0, light->radius * 0.0003);
		return;
	}

	glBegin (GL_TRIANGLE_FAN);
	glColor3f (0.2f, 0.1f, 0.0f); // Baker change touch-up added f to constant (avoid MSVC Level 2+ compiler warning)
	for (i=0 ; i<3 ; i++)
		v[i] = light->origin[i] - vpn[i]*rad;
	glVertex3fv (v);
	glColor3f (0,0,0);

	for (i=16 ; i>=0 ; i--)
	{
#ifdef SPEEDUP_PRECALC_FLASHBLEND_BUBBLE // Baker change
		for (j=0 ; j<3 ; j++)
			v[j] = light->origin[j] + vright[j] * bubble_cos[i] * rad + vup[j] * bubble_sin[i] * rad;
#else // Baker change +
		a = i/16.0 * M_PI*2;
		for (j=0 ; j<3 ; j++)
			v[j] = light->origin[j] + vright[j]*cos(a)*rad
				+ vup[j]*sin(a)*rad;
#endif // Baker change -
		glVertex3fv (v);
	}
	glEnd ();
}

/*
=============
R_RenderDlights
=============
*/
void R_RenderDlights (void) // Flash blend dlights
{
	int		i;
	dlight_t	*l;

	if (!gl_flashblend.value)
		return;

	r_dlightframecount = r_framecount + 1;	// because the count hasn't
											//  advanced yet for this frame
	glDepthMask (0);
	glDisable (GL_TEXTURE_2D);
	glShadeModel (GL_SMOOTH);
	glEnable (GL_BLEND);
	glBlendFunc (GL_ONE, GL_ONE);

	l = cl_dlights;
	for (i=0 ; i<MAX_DLIGHTS ; i++, l++)
	{
		if (l->die < cl.time || !l->radius)
			continue;
		R_RenderDlight (l);
	}

	glColor3f (1,1,1);
	glDisable (GL_BLEND);
	glEnable (GL_TEXTURE_2D);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDepthMask (1);
}


/*
=============================================================================

DYNAMIC LIGHTS

=============================================================================
*/

/*
=============
R_MarkLights -- johnfitz -- rewritten to use LordHavoc's lighting speedup
=============
*/

#ifdef SUPPORTS_DYNAMIC_LIGHTS_CORRECTIONS // Baker change
void R_MarkLights (dlight_t *light, int num, mnode_t *node)
{
	mplane_t	*splitplane;
	msurface_t	*surf;
	vec3_t		impact;
	float		dist, l, maxdist;
	int			i, j, s, t;

	while (1)
	{
		if (node->contents < 0)
			return;
	
		splitplane = node->plane;
	
		if (splitplane->type < 3)
			dist = light->transformed[splitplane->type] - splitplane->dist;
		else
			dist = DotProduct (light->transformed, splitplane->normal) - splitplane->dist;
	
		if (dist > light->radius)
		{
			node = node->children[0];
				continue;
		}
	
		if (dist < -light->radius)
		{
			node = node->children[1];
				continue;
		}
		
		break;
	}

	maxdist = light->radius*light->radius;
// mark the polygons
	surf = cl.worldmodel->surfaces + node->firstsurface;
	for (i=0 ; i<node->numsurfaces ; i++, surf++)
	{
#ifdef SUPPORTS_DLIGHT_BLEED_THROUGH_FIX // Baker change
		int sidebit;
#endif // Baker change +
		if (surf->flags & SURF_DRAWTILED) // no lights on these
			continue;
#ifdef SUPPORTS_DLIGHT_BLEED_THROUGH_FIX // Baker change
		// Light bleed through fix.  Baker: Interesting, but incomplete ... see demo thrgib1 at 64:38 gl_flashblend 0
	    // The light doesn't bleed through.  But affects surfaces it shouldn't, none the less.
		// Still, this fixes some really annoying problems.
		dist = DotProduct (light->transformed, surf->plane->normal) - surf->plane->dist;		// JT030305 - fix light bleed through

		if (dist >= 0)
			sidebit = 0;
		else
			sidebit = SURF_PLANEBACK;

		if ( (surf->flags & SURF_PLANEBACK) != sidebit ) //Discoloda
			continue;
#endif // Baker change +

		for (j=0 ; j<3 ; j++)
			impact[j] = light->transformed[j] - surf->plane->normal[j]*dist;

		// clamp center of light to corner and check brightness
		l = DotProduct (impact, surf->texinfo->vecs[0]) + surf->texinfo->vecs[0][3] - surf->texturemins[0];

		s = l + 0.5;
		s = CLAMP (0, s, surf->extents[0]);
		s = l - s;
		l = DotProduct (impact, surf->texinfo->vecs[1]) + surf->texinfo->vecs[1][3] - surf->texturemins[1];
		t = l + 0.5;
		t = CLAMP (0, t, surf->extents[1]);
		t = l - t;

		// compare to minimum light
		if ((s*s+t*t+dist*dist) < maxdist)
		{
			if (surf->dlightframe != r_dlightframecount) // not dynamic until now
			{
				memset (surf->dlightbits, 0, sizeof (surf->dlightbits));
				surf->dlightframe = r_dlightframecount;
			}

			// mark the surf and ent for this dlight
			surf->dlightbits[num >> 5] |= 1 << (num & 31);
		}
	}
	if (node->children[0]->contents >= 0) R_MarkLights (light, num, node->children[0]);
	if (node->children[1]->contents >= 0) R_MarkLights (light, num, node->children[1]);
}
#else // Baker change +
void R_MarkLights (dlight_t *light, int bit, mnode_t *node)
{
	mplane_t	*splitplane;
	msurface_t	*surf;
	vec3_t		impact;
	float		dist, l, maxdist;
	int			i, j, s, t;

start:

	if (node->contents < 0)
		return;

	splitplane = node->plane;
	if (splitplane->type < 3)
		dist = light->origin[splitplane->type] - splitplane->dist;
	else
		dist = DotProduct (light->origin, splitplane->normal) - splitplane->dist;

	if (dist > light->radius)
	{
		node = node->children[0];
		goto start;
	}
	if (dist < -light->radius)
	{
		node = node->children[1];
		goto start;
	}

	maxdist = light->radius*light->radius;
// mark the polygons
	surf = cl.worldmodel->surfaces + node->firstsurface;
	for (i=0 ; i<node->numsurfaces ; i++, surf++)
	{
		for (j=0 ; j<3 ; j++)
			impact[j] = light->origin[j] - surf->plane->normal[j]*dist;
		// clamp center of light to corner and check brightness
		l = DotProduct (impact, surf->texinfo->vecs[0]) + surf->texinfo->vecs[0][3] - surf->texturemins[0];
		s = l+0.5;if (s < 0) s = 0;else if (s > surf->extents[0]) s = surf->extents[0];
		s = l - s;
		l = DotProduct (impact, surf->texinfo->vecs[1]) + surf->texinfo->vecs[1][3] - surf->texturemins[1];
		t = l+0.5;if (t < 0) t = 0;else if (t > surf->extents[1]) t = surf->extents[1];
		t = l - t;
		// compare to minimum light
		if ((s*s+t*t+dist*dist) < maxdist)
		{
			if (surf->dlightframe != r_dlightframecount) // not dynamic until now
			{
				surf->dlightbits = bit;
				surf->dlightframe = r_dlightframecount;
			}
			else // already dynamic
				surf->dlightbits |= bit;
		}
	}

	if (node->children[0]->contents >= 0)
		R_MarkLights (light, bit, node->children[0]);
	if (node->children[1]->contents >= 0)
		R_MarkLights (light, bit, node->children[1]);
}

#endif // Baker change -

/*
=============
R_PushDlights
=============
*/

#ifdef SUPPORTS_DYNAMIC_LIGHTS_CORRECTIONS // Baker change

void R_TransformDLight (glmatrix *m, float *transformed, float *origin)
{
	if (m)
	{
		// transformed
		transformed[0] = origin[0] * m->m16[0] + origin[1] * m->m16[4] + origin[2] * m->m16[8] + m->m16[12];
		transformed[1] = origin[0] * m->m16[1] + origin[1] * m->m16[5] + origin[2] * m->m16[9] + m->m16[13];
		transformed[2] = origin[0] * m->m16[2] + origin[1] * m->m16[6] + origin[2] * m->m16[10] + m->m16[14];
	}
	else
	{
		// untransformed
		transformed[0] = origin[0];
		transformed[1] = origin[1];
		transformed[2] = origin[2];
	}

}



void R_PushDlights (entity_t *ent)
{
	int		i;
	dlight_t	*l = cl_dlights;
	glmatrix 	*m = ent ? &ent->gl_matrix : NULL;
	mnode_t		*headnode;

	if (ent)
		headnode = ent->model->nodes + ent->model->hulls[0].firstclipnode;
	else headnode = cl.worldmodel->nodes;

	for (i=0 ; i<MAX_DLIGHTS ; i++, l++)
	{
		if (l->die < cl.time || (l->radius<=0) )
			continue;
		
	// move the light back to the same space as the model
	
		R_TransformDLight (m, l->transformed, l->origin);
		R_MarkLights ( l, i, headnode );
	}
}

void R_PushDlights_World (void)
{
	if (gl_flashblend.value) // Non-flashblend dlights
		return;
	else
	{
		extern int r_dlightframecount;
		r_dlightframecount = r_framecount + 1;	// because the count hasn't advanced yet for this frame
		R_PushDlights (NULL);
	}
}

#else // Baker change +
void R_PushDlights (void)
{
	int		i;
	dlight_t	*l;

	if (gl_flashblend.value) // Non-flash blend dlights
		return;

	r_dlightframecount = r_framecount + 1;	// because the count hasn't
											//  advanced yet for this frame
	l = cl_dlights;

	for (i=0 ; i<MAX_DLIGHTS ; i++, l++)
	{
		if (l->die < cl.time || !l->radius)
			continue;
		R_MarkLights ( l, 1<<i, cl.worldmodel->nodes );
	}
}

#endif // Baker change -

/*
=============================================================================

LIGHT SAMPLING - used exclusively for getting alias model lighting info
                 also used for determining the shadow location

=============================================================================
*/

#ifdef SUPPORTS_SHADOWS_ON_SUBMODELS // Baker change
#define SURFACEPOINT_GETLIGHT 1
#define SURFACEPOINT_GETSURFACE 2

//johnfitz -- lit support via lordhavoc

static void R_SurfacePoint_RecursiveSurfacePoint_SetLightColor (vec3_t myLightColor, msurface_t* surf, int ds, int dt)
{
	// LordHavoc: enhanced to interpolate lighting
	int		dsfrac = ds & 15;
	int		dtfrac = dt & 15;
#ifdef SPEEDUP_PRECALC_LM_SMAX_TMAX // Baker change (precalc surf->smax/tmax)
	int		line3 = surf->smax * 3; // LordHavoc: *3 for colored lighting
	byte	*lightmap = surf->samples + ( (dt >> 4) *  surf->smax + (ds >> 4) ) * 3;
#else // Baker change +
	int		smax = (surf->extents[0] >> 4) + 1;
	int		tmax = (surf->extents[1] >> 4) + 1;
	int		line3 = /*((surf->extents[0] >> 4) + 1)*/ smax * 3; // LordHavoc: *3 for colored lighting

	byte	*lightmap = surf->samples + ( (dt >> 4) *  smax + (ds >> 4) ) * 3;
#endif // Baker change -

	int		r00 = 0, g00 = 0, b00 = 0,
			r01 = 0, g01 = 0, b01 = 0,
			r10 = 0, g10 = 0, b10 = 0,
			r11 = 0, g11 = 0, b11 = 0;

	float	scale;
	int		maps;

	myLightColor[0] = myLightColor[1] = myLightColor[2] = 0;

	for (maps = 0;maps < MAXLIGHTMAPS && surf->styles[maps] != 255;maps++)
	{
		scale = (float) d_lightstylevalue[surf->styles[maps]] * 1.0 / 256.0;
		r00 += (float)lightmap[0] * scale;
		g00 += (float)lightmap[1] * scale;
		b00 += (float)lightmap[2] * scale;

		r01 += (float)lightmap[3] * scale;
		g01 += (float)lightmap[4] * scale;
		b01 += (float)lightmap[5] * scale;

		r10 += (float)lightmap[line3+0] * scale;
		g10 += (float)lightmap[line3+1] * scale;
		b10 += (float)lightmap[line3+2] * scale;

		r11 += (float)lightmap[line3+3] * scale;
		g11 += (float)lightmap[line3+4] * scale;
		b11 += (float)lightmap[line3+5] * scale;

#ifdef SPEEDUP_PRECALC_LM_SMAX_TMAX // Baker change (precalc surf->smax/tmax)
		lightmap += surf->smax * surf->tmax * 3;  // LordHavoc: *3 for colored lighting
#else // Baker change +
		lightmap += smax * tmax * 3;  // LordHavoc: *3 for colored lighting
#endif // Baker change -
	}

	myLightColor[0] += (float) ((int) ((((((((r11-r10) * dsfrac) >> 4) + r10)-((((r01-r00) * dsfrac) >> 4) + r00)) * dtfrac) >> 4) + ((((r01-r00) * dsfrac) >> 4) + r00)));
	myLightColor[1] += (float) ((int) ((((((((g11-g10) * dsfrac) >> 4) + g10)-((((g01-g00) * dsfrac) >> 4) + g00)) * dtfrac) >> 4) + ((((g01-g00) * dsfrac) >> 4) + g00)));
	myLightColor[2] += (float) ((int) ((((((((b11-b10) * dsfrac) >> 4) + b10)-((((b01-b00) * dsfrac) >> 4) + b00)) * dtfrac) >> 4) + ((((b01-b00) * dsfrac) >> 4) + b00)));
}

// Variables for most recent success


// Returns false if nothing was hit
#if 1//def TEMP_LIGHT_FIX
vec3_t		 r_lightspot;
vec3_t		 r_lightcolor; //johnfitz -- lit support via lordhavoc
#endif

vec3_t		 lightspot;
vec3_t		 lightcolor; //johnfitz -- lit support via lordhavoc


msurface_t	*lightpoint_surface;
mplane_t	*lightpoint_lightplane;
float		 lightpoint_distance;

/*
=============
R_SurfacePoint_RecursiveSurfacePoint
=============
*/
static qboolean R_SurfacePoint_RecursiveSurfacePoint (vec3_t color, mnode_t *node, vec3_t start, vec3_t end, int mode)
{
	float		front, back, frac;
	vec3_t		mid;

#if 1 // Baker: investigate in future why this can happen ...
	if (mode == SURFACEPOINT_GETSURFACE && !node)
		return false;
#endif

loc0:
	// didn't hit anything (CONTENTS_EMPTY or CONTENTS_WATER, etc.)
	if (node->contents < 0)
		return false;		// didn't hit anything

// calculate mid point
	if (node->plane->type < 3)
	{
		front = start[node->plane->type] - node->plane->dist;
		back = end[node->plane->type] - node->plane->dist;
	}
	else
	{
		front = DotProduct(start, node->plane->normal) - node->plane->dist;
		back = DotProduct(end, node->plane->normal) - node->plane->dist;
	}

	// LordHavoc: optimized recursion
	if ((back < 0) == (front < 0))
	{
		node = node->children[front < 0];
		goto loc0;
	}

	frac = front / (front-back);
	mid[0] = start[0] + (end[0] - start[0])*frac;
	mid[1] = start[1] + (end[1] - start[1])*frac;
	mid[2] = start[2] + (end[2] - start[2])*frac;

// go down front side
	if (R_SurfacePoint_RecursiveSurfacePoint (color, node->children[front < 0], start, mid, mode))
		return true; // hit something

// Didn't hit anything so ...

#ifdef TOUCHUP_R_LIGHTPOINT_FIX // Baker change
#ifdef TEMP_LIGHT_FIX
	VectorCopy (mid, r_lightspot);  // Apparently we need this if all the below fails
#else
	VectorCopy (mid, lightspot);  // Apparently we need this if all the below fails
#endif
#endif // Baker change +

// check for impact on this node
	do
	{
		msurface_t *surf = cl.worldmodel->surfaces + node->firstsurface;
		int i;

		for (i = 0;i < node->numsurfaces;i++, surf++)
		{
			int ds, dt;

			if (mode == SURFACEPOINT_GETLIGHT && surf->flags & SURF_DRAWTILED)
				continue; // no lightmaps

			ds = (int) ((float) DotProduct (mid, surf->texinfo->vecs[0]) + surf->texinfo->vecs[0][3]);
			dt = (int) ((float) DotProduct (mid, surf->texinfo->vecs[1]) + surf->texinfo->vecs[1][3]);

			if (ds < surf->texturemins[0] || dt < surf->texturemins[1])
				continue;  // out of range

			ds -= surf->texturemins[0];
			dt -= surf->texturemins[1];

			if (ds > surf->extents[0] || dt > surf->extents[1])
				continue; // out of range

#ifdef TEMP_LIGHT_FIX
			do // At this point we have a collision with this surface.  Set return variables
			{
				VectorCopy (mid, r_lightspot);
				lightpoint_surface = surf;
				lightpoint_lightplane = node->plane;

				if (surf->samples) // Get and set the color
					R_SurfacePoint_RecursiveSurfacePoint_SetLightColor (r_lightcolor, surf, ds, dt);
				return true; // success
			} while (0);
#else
			do // At this point we have a collision with this surface.  Set return variables
			{
				VectorCopy (mid, lightspot);
				lightpoint_surface = surf;
				lightpoint_lightplane = node->plane;

				if (surf->samples) // Get and set the color
					R_SurfacePoint_RecursiveSurfacePoint_SetLightColor (lightcolor, surf, ds, dt);

				return true; // success
			} while (0);
#endif
		}

	// go down back side
		return R_SurfacePoint_RecursiveSurfacePoint (color, node->children[front >= 0], mid, end, mode);
	} while (0);
}

qboolean R_SurfacePoint_Begin (int mode, vec3_t startpoint, vec3_t endpoint);

#ifdef SUPPORTS_TEXTURE_POINTER // Baker change
qboolean R_SurfacePoint (void)
{
	vec3_t startingpoint, endingpoint, forward, up, right;

	// r_refdef.vieworg/viewangles is the camera position
	VectorCopy (r_refdef.vieworg, startingpoint);

	// Obtain the forward vector
	AngleVectors (r_refdef.viewangles, forward, right, up);

	// Walk it forward by 4096 units
	VectorMA (startingpoint, 4096, forward, endingpoint);

	R_SurfacePoint_Begin (SURFACEPOINT_GETSURFACE, startingpoint, endingpoint);

#if 0
	// Undo old ..
	if (texturepointer_surface)
		texturepointer_surface->flags &= ~ SURF_SELECTED;	// Remove flag

	// Do new
	texturepointer_surface = best_surface;
	texturepointer_surface->flags |= SURF_SELECTED;

	// Save name
	texturepointer_name = texturepointer_surface->texinfo->texture->name;
#endif

	if (lightpoint_surface)
		return true;
	else
		return false; // No assurance anything will be hit (like noclip outside map)

}
#endif // Baker change +

#ifdef TEMP_LIGHT_FIX
int R_LightPoint2 (vec3_t startingpoint)
#else
int R_LightPoint (vec3_t startingpoint)
#endif
{
	vec3_t endingpoint;

	endingpoint[0] = startingpoint[0];
	endingpoint[1] = startingpoint[1];

#ifdef SUPPORTS_R_LIGHTPOINT_DEPTH_CVAR // Baker change
	if (r_lightpoint_depth.value == 0)
		endingpoint[2] = cl.worldmodel->mins[2];  // Maximum depth possible
	else
		endingpoint[2] = startingpoint[2] - r_lightpoint_depth.value;	//johnfitz increased to 8192, MH reversed due to speed. Baker: Now this is a cvar.

#if 0 // Baker .. under consideration
	endingpoint[2] = CLAMP(cl.worldmodel->maxs[2], endingpoint[2], cl.worldmodel->mins[2]); 
#endif // Baker ... under consideration

#else // Baker change +
	endingpoint[2] = startingpoint[2] - 2048;	//johnfitz increased to 8192, MH reversed due to speed.
#endif // Baker change -

	if (R_SurfacePoint_Begin (SURFACEPOINT_GETLIGHT, startingpoint, endingpoint) == false)
		return 255; // Didn't hit anything

#ifdef TEMP_LIGHT_FIX
	return ((r_lightcolor[0] + r_lightcolor[1] + r_lightcolor[2]) * (1.0f / 3.0f));
#else
	return ((lightcolor[0] + lightcolor[1] + lightcolor[2]) * (1.0f / 3.0f));
#endif
}

/*
=============
R_SurfacePoint_Begin
=============
*/


qboolean R_SurfacePoint_Begin (int mode, vec3_t startpoint, vec3_t endpoint)
{
	// Reset stuff
	lightpoint_surface = NULL;
	lightpoint_lightplane   = NULL;
	lightpoint_distance		= 0;
#ifdef TEMP_LIGHT_FIX
	r_lightcolor[0] = r_lightcolor[1] = r_lightcolor[2] = 255;
	r_lightspot[0]  = r_lightspot[1]  = r_lightspot[2]  = 0;

	if (mode == SURFACEPOINT_GETLIGHT && !cl.worldmodel->lightdata)
	{
		r_lightcolor[0] = r_lightcolor[1] = r_lightcolor[2] = 255;
		return false;
	}

#else
	lightcolor[0] = lightcolor[1] = lightcolor[2] = 255;
	lightspot[0]  = lightspot[1]  = lightspot[2]  = 0;

	if (mode == SURFACEPOINT_GETLIGHT && !cl.worldmodel->lightdata)
	{
		lightcolor[0] = lightcolor[1] = lightcolor[2] = 255;
		return false;
	}
#endif

	do	// Search function
	{
		// These variables store what we have for "best results" so far.
		msurface_t	*best_surface;
		mplane_t	*best_lightplane;
		vec3_t		 best_lightspot;
		vec3_t		 best_lightcolor;
		int			 best_distance_ent=-1;
		float		 best_distance;
		int			 i;
		qboolean	 best_found = false;

#ifdef TEMP_LIGHT_FIX
		if (R_SurfacePoint_RecursiveSurfacePoint (r_lightcolor, cl.worldmodel->nodes, startpoint, endpoint, mode))
		{
			best_found = true; // Now update bests ...

			best_surface = lightpoint_surface;
			best_distance   = DistanceBetween2Points (startpoint, r_lightspot); // lightpoint_distance = startpoint[2] - lightspot[2];
			best_lightplane	= lightpoint_lightplane;
			VectorCopy (r_lightspot,  best_lightspot);
			VectorCopy (r_lightcolor, best_lightcolor);
			best_distance_ent = 0;
			
		}

#else
		if (R_SurfacePoint_RecursiveSurfacePoint (lightcolor, cl.worldmodel->nodes, startpoint, endpoint, mode))
		{
			best_found = true; // Now update bests ...

			best_surface = lightpoint_surface;
			best_distance   = DistanceBetween2Points (startpoint, lightspot); // lightpoint_distance = startpoint[2] - lightspot[2];
			best_lightplane	= lightpoint_lightplane;
			VectorCopy (lightspot,  best_lightspot);
			VectorCopy (lightcolor, best_lightcolor);
			best_distance_ent = 0;
			
		}
#endif
#if 0 // Baker test
// Should I ever wish to disable or cvarize shadows on submodels, here is where ...
		if (mode == SURFACEPOINT_GETLIGHT)
			return best_found;

#endif // Baker test

		// Now check for hit with world submodels
		for (i = 0 ; i < cl_numvisedicts ; i ++)	// 0 is player.
		{
			entity_t	*pe = cl_visedicts[i]; // Baker: We check visible edicts, not cl_entities.  Why do shadows on non-visible ...
			vec3_t		adjusted_startpoint;
			vec3_t		adjusted_endpoint;
			vec3_t		adjusted_net;

			if (!pe->model)
				continue;   // no model for ent

			if (!(pe->model->surfaces == cl.worldmodel->surfaces))
				continue;	// model isnt part of world (i.e. no health boxes or what not ...)

			// Baker: We need to adjust the point locations for entity origin

			VectorSubtract (startpoint, pe->origin, adjusted_startpoint);
			VectorSubtract (endpoint,   pe->origin, adjusted_endpoint);
			VectorSubtract (startpoint, adjusted_startpoint, adjusted_net);

			// Make further adjustments if entity is rotated
			if (pe->angles[0] || pe->angles[1] || pe->angles[2])
			{
				vec3_t f, r, u, temp;
				AngleVectors(pe->angles, f, r, u);	// split entity angles to forward, right, up

				VectorCopy(adjusted_startpoint, temp);
				adjusted_startpoint[0] = DotProduct(temp, f);
				adjusted_startpoint[1] = -DotProduct(temp, r);
				adjusted_startpoint[2] = DotProduct(temp, u);

				VectorCopy(adjusted_endpoint, temp);
				adjusted_endpoint[0] = DotProduct(temp, f);
				adjusted_endpoint[1] = -DotProduct(temp, r);
				adjusted_endpoint[2] = DotProduct(temp, u);
			}

			// Reset stuff
			lightpoint_lightplane   = NULL;
			lightpoint_distance		= 0;
#ifdef TEMP_LIGHT_FIX
			r_lightcolor[0] = r_lightcolor[1] = r_lightcolor[2] = 0;
			r_lightspot[0]  = r_lightspot[1]  = r_lightspot[2]  = 0;  // This is kind of a silly "erased" lightspot; might be center of the map

			if (R_SurfacePoint_RecursiveSurfacePoint (r_lightcolor, pe->model->nodes+pe->model->hulls[0].firstclipnode /*pe->model->nodes*/, adjusted_startpoint, adjusted_endpoint, mode))
			{
				// Baker: We have to add the origin back into the results here!

				VectorAdd (r_lightspot, adjusted_net, r_lightspot);

				DistanceBetween2Points (startpoint, r_lightspot); // lightpoint_distance   = startpoint[2] - lightspot[2];

#else
			lightcolor[0] = lightcolor[1] = lightcolor[2] = 0;
			lightspot[0]  = lightspot[1]  = lightspot[2]  = 0;  // This is kind of a silly "erased" lightspot; might be center of the map

			if (R_SurfacePoint_RecursiveSurfacePoint (lightcolor, pe->model->nodes+pe->model->hulls[0].firstclipnode /*pe->model->nodes*/, adjusted_startpoint, adjusted_endpoint, mode))
			{
				// Baker: We have to add the origin back into the results here!

				VectorAdd (lightspot, adjusted_net, lightspot);

				DistanceBetween2Points (startpoint, lightspot); // lightpoint_distance   = startpoint[2] - lightspot[2];
#endif

				if (!best_found || lightpoint_distance < best_distance)
				{
					// New best
					best_found = true;  // Now update bests ...

					best_surface = lightpoint_surface;
					best_distance   = lightpoint_distance;
					best_lightplane	= lightpoint_lightplane;
					best_distance_ent = i;

#ifdef TEMP_LIGHT_FIX
					VectorCopy (r_lightspot,  best_lightspot);
					VectorCopy (r_lightcolor, best_lightcolor);
#else
					VectorCopy (lightspot,  best_lightspot);
					VectorCopy (lightcolor, best_lightcolor);
#endif
				}
			}

			// On to next entity
		}

		if (!best_found)	// Lightpoint results are 0 or null
			return false;

		// Copy over the best results as the lightpoint results

		lightpoint_distance = best_distance;
		lightpoint_lightplane = best_lightplane;
#ifdef TEMP_LIGHT_FIX
		VectorCopy (best_lightspot,  r_lightspot);
		VectorCopy (best_lightcolor, r_lightcolor);
#else
		VectorCopy (best_lightspot,  lightspot);
		VectorCopy (best_lightcolor, lightcolor);
#endif


	} while (0);


	return true;

}
#endif // Baker change + TEMP_LIGHT_FIX

#if 1
mplane_t		*lightplane;
vec3_t			lightspot;
vec3_t			lightcolor; //johnfitz -- lit support via lordhavoc

/*
=============
RecursiveLightPoint -- johnfitz -- replaced entire function for lit support via lordhavoc
=============
*/
int RecursiveLightPoint (vec3_t color, mnode_t *node, vec3_t start, vec3_t end)
{
	float		front, back, frac;
	vec3_t		mid;

loc0:
	if (node->contents < 0)
		return false;		// didn't hit anything

// calculate mid point
	if (node->plane->type < 3)
	{
		front = start[node->plane->type] - node->plane->dist;
		back = end[node->plane->type] - node->plane->dist;
	}
	else
	{
		front = DotProduct(start, node->plane->normal) - node->plane->dist;
		back = DotProduct(end, node->plane->normal) - node->plane->dist;
	}

	// LordHavoc: optimized recursion
	if ((back < 0) == (front < 0))
//		return RecursiveLightPoint (color, node->children[front < 0], start, end);
	{
		node = node->children[front < 0];
		goto loc0;
	}

	frac = front / (front-back);
	mid[0] = start[0] + (end[0] - start[0])*frac;
	mid[1] = start[1] + (end[1] - start[1])*frac;
	mid[2] = start[2] + (end[2] - start[2])*frac;

// go down front side
	if (RecursiveLightPoint (color, node->children[front < 0], start, mid))
		return true;	// hit something
	else
	{
		int i, ds, dt;
		msurface_t *surf;
	// check for impact on this node
		VectorCopy (mid, lightspot);
		lightplane = node->plane;

		surf = cl.worldmodel->surfaces + node->firstsurface;
		for (i = 0;i < node->numsurfaces;i++, surf++)
		{
			if (surf->flags & SURF_DRAWTILED)
				continue;	// no lightmaps

			ds = (int) ((float) DotProduct (mid, surf->texinfo->vecs[0]) + surf->texinfo->vecs[0][3]);
			dt = (int) ((float) DotProduct (mid, surf->texinfo->vecs[1]) + surf->texinfo->vecs[1][3]);

			if (ds < surf->texturemins[0] || dt < surf->texturemins[1])
				continue;

			ds -= surf->texturemins[0];
			dt -= surf->texturemins[1];

			if (ds > surf->extents[0] || dt > surf->extents[1])
				continue;

			if (surf->samples)
			{
				// LordHavoc: enhanced to interpolate lighting
				byte *lightmap;
				int maps, line3, dsfrac = ds & 15, dtfrac = dt & 15, r00 = 0, g00 = 0, b00 = 0, r01 = 0, g01 = 0, b01 = 0, r10 = 0, g10 = 0, b10 = 0, r11 = 0, g11 = 0, b11 = 0;
				float scale;
#ifdef SPEEDUP_PRECALC_LM_SMAX_TMAX // Baker change (precalc surf->smax/tmax)
				line3 = surf->smax *3;

				lightmap = surf->samples + ( (dt>>4) * (surf->smax + (ds>>4)) )* 3;  // LordHavoc: *3 for color
#else // Baker change +
				line3 = ((surf->extents[0]>>4)+1)*3;

				lightmap = surf->samples + ((dt>>4) * ((surf->extents[0]>>4)+1) + (ds>>4))*3; // LordHavoc: *3 for color

#endif // Baker change -
				for (maps = 0;maps < MAXLIGHTMAPS && surf->styles[maps] != 255;maps++)
				{
					scale = (float) d_lightstylevalue[surf->styles[maps]] * 1.0 / 256.0;
					r00 += (float) lightmap[      0] * scale;g00 += (float) lightmap[      1] * scale;b00 += (float) lightmap[2] * scale;
					r01 += (float) lightmap[      3] * scale;g01 += (float) lightmap[      4] * scale;b01 += (float) lightmap[5] * scale;
					r10 += (float) lightmap[line3+0] * scale;g10 += (float) lightmap[line3+1] * scale;b10 += (float) lightmap[line3+2] * scale;
					r11 += (float) lightmap[line3+3] * scale;g11 += (float) lightmap[line3+4] * scale;b11 += (float) lightmap[line3+5] * scale;
#ifdef SPEEDUP_PRECALC_LM_SMAX_TMAX // Baker change (precalc surf->smax/tmax)
					lightmap += surf->smax * surf->tmax * 3; // LordHavoc: *3 for colored lighting
#else // Baker change +
					lightmap += ((surf->extents[0]>>4)+1) * ((surf->extents[1]>>4)+1)*3; // LordHavoc: *3 for colored lighting
#endif // Baker change -
				}

				color[0] += (float) ((int) ((((((((r11-r10) * dsfrac) >> 4) + r10)-((((r01-r00) * dsfrac) >> 4) + r00)) * dtfrac) >> 4) + ((((r01-r00) * dsfrac) >> 4) + r00)));
				color[1] += (float) ((int) ((((((((g11-g10) * dsfrac) >> 4) + g10)-((((g01-g00) * dsfrac) >> 4) + g00)) * dtfrac) >> 4) + ((((g01-g00) * dsfrac) >> 4) + g00)));
				color[2] += (float) ((int) ((((((((b11-b10) * dsfrac) >> 4) + b10)-((((b01-b00) * dsfrac) >> 4) + b00)) * dtfrac) >> 4) + ((((b01-b00) * dsfrac) >> 4) + b00)));
			}
			return true; // success
		}

	// go down back side
		return RecursiveLightPoint (color, node->children[front >= 0], mid, end);
	}
}

/*
=============
R_LightPoint -- johnfitz -- replaced entire function for lit support via lordhavoc
=============
*/
int R_LightPoint (vec3_t p)
{
	vec3_t		end;

	if (!cl.worldmodel->lightdata)
	{
		lightcolor[0] = lightcolor[1] = lightcolor[2] = 255;
		return 255;
	}

	end[0] = p[0];
	end[1] = p[1];
	end[2] = p[2] - 8192; //johnfitz -- was 2048

	lightcolor[0] = lightcolor[1] = lightcolor[2] = 0;
	RecursiveLightPoint (lightcolor, cl.worldmodel->nodes, p, end);
	return ((lightcolor[0] + lightcolor[1] + lightcolor[2]) * (1.0f / 3.0f));
}
#endif // Baker change -
