/*
Copyright (C) 1996-2001 Id Software, Inc.
Copyright (C) 2002-2012 John Fitzgibbons and others

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 3
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// sys_win.c -- Win32 system interface code

#include "quakedef.h"
#include "winquake.h"
#include "resource.h"
#include "conproc.h"
#ifdef SUPPORTS_CLOCK_FIX // Baker change
#include <limits.h>
#endif // Baker change +
#include <errno.h>

#define MINIMUM_WIN_MEMORY		0x0880000
#ifdef SUPPORTS_HIGHER_MEM_DEFAULT // Baker change
#define MAXIMUM_WIN_MEMORY		0x4000000 //Baker: 64 MB.  Can it handle ARWOP?
#else // Baker change +
#define MAXIMUM_WIN_MEMORY		0x2000000 //johnfitz -- 32 mb, was 16 mb
#endif // Baker change -

#define CONSOLE_ERROR_TIMEOUT	60.0	// # of seconds to wait on Sys_Error running
										//  dedicated before exiting
#define PAUSE_SLEEP		50				// sleep time on pause or minimization
#define NOT_FOCUS_SLEEP	20				// sleep time when not focus

int			starttime;
qboolean	ActiveApp, Minimized;
qboolean	WinNT;

static double		pfreq;
static double		curtime = 0.0;
static double		lastcurtime = 0.0;
static int			lowshift;
qboolean			isDedicated;
static qboolean		sc_return_on_enter = false;
HANDLE				hinput, houtput;

static char			*tracking_tag = "Clams & Mooses";

static HANDLE	tevent;
static HANDLE	hFile;
static HANDLE	heventParent;
static HANDLE	heventChild;

void Sys_InitFloatTime (void);

volatile int					sys_checksum;


/*
================
Sys_PageIn
================
*/
void Sys_PageIn (void *ptr, int size)
{
	byte	*x;
	int		j, m, n;

// touch all the memory to make sure it's there. The 16-page skip is to
// keep Win 95 from thinking we're trying to page ourselves in (we are
// doing that, of course, but there's no reason we shouldn't)
	x = (byte *)ptr;

	for (n=0 ; n<4 ; n++)
	{
		for (m=0 ; m<(size - 16 * 0x1000) ; m += 4)
		{
			sys_checksum += *(int *)&x[m];
			sys_checksum += *(int *)&x[m + 16 * 0x1000];
		}
	}
}


/*
===============================================================================

FILE IO

===============================================================================
*/

#define	MAX_HANDLES		100 //johnfitz -- was 10
FILE	*sys_handles[MAX_HANDLES];

int		findhandle (void)
{
	int		i;

	for (i=1 ; i<MAX_HANDLES ; i++)
		if (!sys_handles[i])
			return i;
	Sys_Error ("out of handles");
	return -1;
}

/*
================
filelength
================
*/
int filelength (FILE *f)
{
	int		pos;
	int		end;
/*	int		t;

	t = VID_ForceUnlockedAndReturnState ();	Baker: WinQuake only */

	pos = ftell (f);
	fseek (f, 0, SEEK_END);
	end = ftell (f);
	fseek (f, pos, SEEK_SET);

/*	VID_ForceLockState (t);		Baker: WinQuake only */

	return end;
}

int Sys_FileOpenRead (char *path, int *hndl)
{
	FILE	*f;
	int		i, retval;
/*	int		t;  Baker: WinQuake only */

/*	t = VID_ForceUnlockedAndReturnState ();  Baker: WinQuake only */

	i = findhandle ();

	f = fopen(path, "rb");

	if (!f)
	{
		*hndl = -1;
		retval = -1;
	}
	else
	{
		sys_handles[i] = f;
		*hndl = i;
		retval = filelength(f);
	}

/*	VID_ForceLockState (t);		Baker: WinQuake only */

	return retval;
}

int Sys_FileOpenWrite (char *path)
{
	FILE	*f;
	int		i;
	int		t;

/*	t = VID_ForceUnlockedAndReturnState ();	Baker: WinQuake only */

	i = findhandle ();

	f = fopen(path, "wb");
	if (!f)
		Sys_Error ("Error opening %s: %s", path,strerror(errno));
	sys_handles[i] = f;

/*	VID_ForceLockState (t);		Baker: WinQuake only */

	return i;
}

void Sys_FileClose (int handle)
{
/*	int		t;

	t = VID_ForceUnlockedAndReturnState ();	Baker: WinQuake only */
	fclose (sys_handles[handle]);
	sys_handles[handle] = NULL;
/*	VID_ForceLockState (t);		Baker: WinQuake only */
}

void Sys_FileSeek (int handle, int position)
{
/*	int		t;

	t = VID_ForceUnlockedAndReturnState ();		Baker: WinQuake only */
	fseek (sys_handles[handle], position, SEEK_SET);
/*	VID_ForceLockState (t);		Baker: WinQuake only */
}

int Sys_FileRead (int handle, void *dest, int count)
{
	int		/* t, Baker: WinQuake only */ x;

/*	t = VID_ForceUnlockedAndReturnState ();		Baker: WinQuake only */
	x = fread (dest, 1, count, sys_handles[handle]);
/*	VID_ForceLockState (t);		Baker: WinQuake only */
	return x;
}

int Sys_FileWrite (int handle, void *data, int count)
{
	int		/* t, Baker: WinQuake only */ x;

/*	t = VID_ForceUnlockedAndReturnState ();	Baker: WinQuake only */
	x = fwrite (data, 1, count, sys_handles[handle]);
/*	VID_ForceLockState (t);	Baker: WinQuake only */
	return x;
}

int	Sys_FileTime (char *path)
{
	FILE	*f;
	int		/*t, Baker: WinQuake only */ retval;

/*	t = VID_ForceUnlockedAndReturnState ();  Baker: WinQuake only */

	f = fopen(path, "rb");

	if (f)
	{
		fclose(f);
		retval = 1;
	}
	else
	{
		retval = -1;
	}

/*	VID_ForceLockState (t);		Baker: WinQuake only  */
	return retval;
}

#include <direct.h> // Baker: Removes a warning
void Sys_mkdir (char *path)
{
	_mkdir (path);
}

#ifdef SUPPORTS_FOLDER_COMMAND // Baker change

#ifdef SUPPORTS_FOLDER_COMMAND_LASTFILE // Baker chage
qboolean Explorer_OpenFolder_HighlightFile (char *absolutefilename)
{
	char folder_to_open[MAX_OSPATH];
	char file_highlight[MAX_OSPATH];
	char command_line  [1024];
	int i;

	if (Sys_FileTime(absolutefilename) == -1)
	{
		Con_DPrintf ("File \"%s\" does not exist to show\n", absolutefilename);
		Con_Printf ("File does not exist to show\n");
		return false;
	}

	// Copy it
	strcpy (file_highlight, absolutefilename);

	// Windows format the slashes
	for (i = 0; file_highlight[i]; i++)
		if (file_highlight[i] == '/')
			file_highlight[i] = '\\';

	// Get the path
	strcpy (folder_to_open, file_highlight);
	COM_Reduce_To_Parent_Path (folder_to_open);

	sprintf (command_line, "/select,%s", file_highlight);

	// Zero is failure, non-zero is success
	Con_DPrintf ("Folder highlight: explorer.exe with \"%s\"\n", command_line);

	return (ShellExecute(0, "Open", "explorer.exe", command_line, NULL, SW_NORMAL) != 0);

}
#endif // Baker change +

qboolean Explorer_OpenFolder (char *fullpath)
{
	char folder_to_open[MAX_OSPATH];
	int i;

	// Copy it
	strcpy (folder_to_open, fullpath);

	// Windows format the slashes
	for (i = 0; folder_to_open[i]; i++)
		if (folder_to_open[i] == '/')
			folder_to_open[i] = '\\';

	return (ShellExecute(0, "Open", "explorer.exe", folder_to_open, NULL, SW_NORMAL) != 0);
}

void Sys_OpenFolder_f (void)
{
	if (isDedicated)
		return;

	if (modestate != MS_WINDOWED)
	{
		Con_Printf ("folder command only works in windowed mode\n");
		return;
	}

#ifdef SUPPORTS_FOLDER_COMMAND_LASTFILE // Baker change
	if (recent_file[0])
	{
		char tempname[1024];
		sprintf (tempname, "%s/%s", com_gamedir, recent_file);
		
		// See if this file exists ...
		if (Sys_FileTime(tempname) != -1)
		{
			// It does
			if (Explorer_OpenFolder_HighlightFile (tempname))
				Con_Printf ("Explorer opening folder and highlighting ...\n");
			else
				Con_Printf ("Opening folder failed\n");

			return;
		}

		// If the file didn't exist, we open the gamedir folder like normal
	}
#endif // Baker change +

	if (Explorer_OpenFolder (com_gamedir))
		Con_Printf  ("Explorer opening folder ...\n");
	else
		Con_Printf ("Opening folder failed\n");

	return;
}
#endif // Baker change +

#ifdef SUPPORTS_AVI_CAPTURE // Baker change
#include <vfw.h>

static void (CALLBACK *qAVIFileInit)(void);
static HRESULT (CALLBACK *qAVIFileOpen)(PAVIFILE *, LPCTSTR, UINT, LPCLSID);
static HRESULT (CALLBACK *qAVIFileCreateStream)(PAVIFILE, PAVISTREAM *, AVISTREAMINFO *);
static HRESULT (CALLBACK *qAVIMakeCompressedStream)(PAVISTREAM *, PAVISTREAM, AVICOMPRESSOPTIONS *, CLSID *);
static HRESULT (CALLBACK *qAVIStreamSetFormat)(PAVISTREAM, LONG, LPVOID, LONG);
static HRESULT (CALLBACK *qAVIStreamWrite)(PAVISTREAM, LONG, LONG, LPVOID, LONG, DWORD, LONG *, LONG *);
static ULONG (CALLBACK *qAVIStreamRelease)(PAVISTREAM);
static ULONG (CALLBACK *qAVIFileRelease)(PAVIFILE);
static void (CALLBACK *qAVIFileExit)(void);

static MMRESULT (ACMAPI *qacmDriverOpen)(LPHACMDRIVER, HACMDRIVERID, DWORD);
static MMRESULT (ACMAPI *qacmDriverDetails)(HACMDRIVERID, LPACMDRIVERDETAILS, DWORD);
static MMRESULT (ACMAPI *qacmDriverEnum)(ACMDRIVERENUMCB, DWORD, DWORD);
static MMRESULT (ACMAPI *qacmFormatTagDetails)(HACMDRIVER, LPACMFORMATTAGDETAILS, DWORD);
static MMRESULT (ACMAPI *qacmStreamOpen)(LPHACMSTREAM, HACMDRIVER, LPWAVEFORMATEX, LPWAVEFORMATEX, LPWAVEFILTER, DWORD, DWORD, DWORD);
static MMRESULT (ACMAPI *qacmStreamSize)(HACMSTREAM, DWORD, LPDWORD, DWORD);
static MMRESULT (ACMAPI *qacmStreamPrepareHeader)(HACMSTREAM, LPACMSTREAMHEADER, DWORD);
static MMRESULT (ACMAPI *qacmStreamUnprepareHeader)(HACMSTREAM, LPACMSTREAMHEADER, DWORD);
static MMRESULT (ACMAPI *qacmStreamConvert)(HACMSTREAM, LPACMSTREAMHEADER, DWORD);
static MMRESULT (ACMAPI *qacmStreamClose)(HACMSTREAM, DWORD);
static MMRESULT (ACMAPI *qacmDriverClose)(HACMDRIVER, DWORD);

static HINSTANCE avi_handle = NULL, acm_handle = NULL;

PAVIFILE	m_file;
PAVISTREAM	m_uncompressed_video_stream;
PAVISTREAM  m_compressed_video_stream;
PAVISTREAM  m_audio_stream;

unsigned long m_codec_fourcc;
int			m_video_frame_counter;
int			m_video_frame_size;

qboolean	m_audio_is_mp3;
int			m_audio_frame_counter;
WAVEFORMATEX m_wave_format;
MPEGLAYER3WAVEFORMAT m_mp3_format;
qboolean	mp3_encoder_exists;
HACMDRIVER	m_mp3_driver;
HACMSTREAM	m_mp3_stream;
ACMSTREAMHEADER	m_mp3_stream_header;

extern qboolean avi_loaded, acm_loaded;
extern	cvar_t	capture_codec, capture_fps, capture_mp3, capture_mp3_kbps;

#define AVI_GETFUNC(f) (qAVI##f = (void *)GetProcAddress(avi_handle, "AVI" #f))
#define ACM_GETFUNC(f) (qacm##f = (void *)GetProcAddress(acm_handle, "acm" #f))

void AVI_LoadLibrary (void)
{
	avi_loaded = false;

	if (!(avi_handle = LoadLibrary("avifil32.dll")))
	{
		Con_Warning ("Avi capturing module not found\n");
		goto fail;
	}

	AVI_GETFUNC(FileInit);
	AVI_GETFUNC(FileOpen);
	AVI_GETFUNC(FileCreateStream);
	AVI_GETFUNC(MakeCompressedStream);
	AVI_GETFUNC(StreamSetFormat);
	AVI_GETFUNC(StreamWrite);
	AVI_GETFUNC(StreamRelease);
	AVI_GETFUNC(FileRelease);
	AVI_GETFUNC(FileExit);

	avi_loaded = qAVIFileInit && qAVIFileOpen && qAVIFileCreateStream &&
			qAVIMakeCompressedStream && qAVIStreamSetFormat && qAVIStreamWrite &&
			qAVIStreamRelease && qAVIFileRelease && qAVIFileExit;

	if (!avi_loaded)
	{
		Con_SafePrintf ("Avi capturing module not initialized\n");
		goto fail;
	}

	Con_SafePrintf ("Avi capturing module initialized\n");
	return;

fail:
	if (avi_handle)
	{
		FreeLibrary (avi_handle);
		avi_handle = NULL;
	}
}

void ACM_LoadLibrary (void)
{
	acm_loaded = false;

	if (!(acm_handle = LoadLibrary("msacm32.dll")))
	{
		Con_Warning ("ACM module not found\n");
		goto fail;
	}

	ACM_GETFUNC(DriverOpen);
	ACM_GETFUNC(DriverEnum);
	ACM_GETFUNC(StreamOpen);
	ACM_GETFUNC(StreamSize);
	ACM_GETFUNC(StreamPrepareHeader);
	ACM_GETFUNC(StreamUnprepareHeader);
	ACM_GETFUNC(StreamConvert);
	ACM_GETFUNC(StreamClose);
	ACM_GETFUNC(DriverClose);
	qacmDriverDetails = (void *)GetProcAddress (acm_handle, "acmDriverDetailsA");
	qacmFormatTagDetails = (void *)GetProcAddress (acm_handle, "acmFormatTagDetailsA");

	acm_loaded = qacmDriverOpen && qacmDriverDetails && qacmDriverEnum &&
			qacmFormatTagDetails && qacmStreamOpen && qacmStreamSize &&
			qacmStreamPrepareHeader && qacmStreamUnprepareHeader &&
			qacmStreamConvert && qacmStreamClose && qacmDriverClose;

	if (!acm_loaded)
	{
		Con_SafePrintf ("ACM module not initialized\n");
		goto fail;
	}

	Con_SafePrintf ("ACM module initialized\n");
	return;

fail:
	if (acm_handle)
	{
		FreeLibrary (acm_handle);
		acm_handle = NULL;
	}
}

PAVISTREAM Capture_VideoStream (void)
{
	return m_codec_fourcc ? m_compressed_video_stream : m_uncompressed_video_stream;
}

BOOL CALLBACK acmDriverEnumCallback (HACMDRIVERID mp3_driver_id, DWORD dwInstance, DWORD fdwSupport)
{
	if (fdwSupport & ACMDRIVERDETAILS_SUPPORTF_CODEC)
	{
		int	i;
		ACMDRIVERDETAILS drvDetails;

		memset (&drvDetails, 0, sizeof(drvDetails));
		drvDetails.cbStruct = sizeof(drvDetails);
		qacmDriverDetails (mp3_driver_id, &drvDetails, 0);
		qacmDriverOpen (&m_mp3_driver, mp3_driver_id, 0);

		for (i = 0 ; i < drvDetails.cFormatTags ; i++)
		{
			ACMFORMATTAGDETAILS	fmtDetails;

			memset (&fmtDetails, 0, sizeof(fmtDetails));
			fmtDetails.cbStruct = sizeof(fmtDetails);
			fmtDetails.dwFormatTagIndex = i;
			qacmFormatTagDetails (m_mp3_driver, &fmtDetails, ACM_FORMATTAGDETAILSF_INDEX);
			if (fmtDetails.dwFormatTag == WAVE_FORMAT_MPEGLAYER3)
			{
				MMRESULT	mmr;

				Con_DPrintf ("MP3-capable ACM codec found: %s\n", drvDetails.szLongName);

				m_mp3_stream = NULL;
				if ((mmr = qacmStreamOpen(&m_mp3_stream, m_mp3_driver, &m_wave_format, &m_mp3_format.wfx, NULL, 0, 0, 0)))
				{
					switch (mmr)
					{
					case MMSYSERR_INVALPARAM:
						Con_DPrintf ("ERROR: Invalid parameters passed to acmStreamOpen\n");
						break;

					case ACMERR_NOTPOSSIBLE:
						Con_DPrintf ("ERROR: No ACM filter found capable of encoding MP3\n");
						break;

					default:
						Con_DPrintf ("ERROR: Couldn't open ACM encoding stream\n");
						break;
					}
					continue;
				}
				mp3_encoder_exists = true;

				return false;
			}
		}

		qacmDriverClose (m_mp3_driver, 0);
	}

	return true;
}

#ifdef SUPPORTS_CAPTUREVIDEO_AUTOCODEC // Baker change
int Capture_Open (char *filename, char *usercodec, qboolean silentish)
#else // Baker change +
qboolean Capture_Open (char *filename)
#endif // Baker change -
{
	HRESULT			hr;
	BITMAPINFOHEADER bitmap_info_header;
	AVISTREAMINFO	stream_header;
	char			*fourcc;

	m_video_frame_counter = m_audio_frame_counter = 0;
	m_file = NULL;
	m_codec_fourcc = 0;
	m_uncompressed_video_stream = m_compressed_video_stream = m_audio_stream = NULL;
	m_audio_is_mp3 = (qboolean)capture_mp3.value;

#ifdef SUPPORTS_CAPTUREVIDEO_AUTOCODEC // Baker change
	fourcc = usercodec;
	if (Q_strcasecmp(fourcc, "none") != 0)	// codec fourcc supplied
#else // Baker change +
	if (*(fourcc = capture_codec.string) != '0')	// codec fourcc supplied
#endif // Baker change -
		m_codec_fourcc = mmioFOURCC (fourcc[0], fourcc[1], fourcc[2], fourcc[3]);

	qAVIFileInit ();
	hr = qAVIFileOpen (&m_file, filename, OF_WRITE | OF_CREATE, NULL);
	if (FAILED(hr))
	{
#ifdef SUPPORTS_CAPTUREVIDEO_AUTOCODEC // Baker change
		if (!silentish)
			Con_Printf ("ERROR: Couldn't open AVI file for writing\n");
		return -1;
#else // Baker change +
		Con_Printf ("ERROR: Couldn't open AVI file\n");
		return false;
#endif // Baker change -

	}

	// initialize video data

	m_video_frame_size = glwidth * glheight * 3;

	memset (&bitmap_info_header, 0, sizeof(bitmap_info_header));
	bitmap_info_header.biSize = sizeof(BITMAPINFOHEADER);

	bitmap_info_header.biWidth = glwidth;
	bitmap_info_header.biHeight = glheight;

	bitmap_info_header.biPlanes = 1;
	bitmap_info_header.biBitCount = 24;
	bitmap_info_header.biCompression = BI_RGB;
	bitmap_info_header.biSizeImage = m_video_frame_size;

	memset (&stream_header, 0, sizeof(stream_header));
	stream_header.fccType = streamtypeVIDEO;
	stream_header.fccHandler = m_codec_fourcc;
	stream_header.dwScale = 1;
	stream_header.dwRate = (unsigned long)(0.5 + capture_fps.value);
	stream_header.dwSuggestedBufferSize = bitmap_info_header.biSizeImage;
	SetRect (&stream_header.rcFrame, 0, 0, bitmap_info_header.biWidth, bitmap_info_header.biHeight);

	hr = qAVIFileCreateStream (m_file, &m_uncompressed_video_stream, &stream_header);
	if (FAILED(hr))
	{
#ifdef SUPPORTS_CAPTUREVIDEO_AUTOCODEC // Baker change
		if (!silentish)
			Con_Printf ("ERROR: Couldn't create video stream\n");
		return -2;
#else // Baker change +
		Con_Printf ("ERROR: Couldn't create video stream\n");
		return false;
#endif // Baker change -
	}

	if (m_codec_fourcc)
	{
		AVICOMPRESSOPTIONS	opts;

		memset (&opts, 0, sizeof(opts));
		opts.fccType = stream_header.fccType;
		opts.fccHandler = m_codec_fourcc;

		// Make the stream according to compression
		hr = qAVIMakeCompressedStream (&m_compressed_video_stream, m_uncompressed_video_stream, &opts, NULL);
		if (FAILED(hr))
		{
#ifdef SUPPORTS_CAPTUREVIDEO_AUTOCODEC // Baker change
			if (!silentish)
				Con_Printf ("ERROR: Couldn't make compressed video stream\n");
			return -3;
#else // Baker change +
			Con_Printf ("ERROR: Couldn't make compressed video stream\n");
			return false;
#endif // Baker change -
		}
	}

	hr = qAVIStreamSetFormat (Capture_VideoStream(), 0, &bitmap_info_header, bitmap_info_header.biSize);
	if (FAILED(hr))
	{
		Con_Printf ("ERROR: Couldn't set video stream format\n");
		return false;
	}

	// initialize audio data
	memset (&m_wave_format, 0, sizeof(m_wave_format));
	m_wave_format.wFormatTag = WAVE_FORMAT_PCM;
	m_wave_format.nChannels = 2;		// always stereo in Quake sound engine
	m_wave_format.nSamplesPerSec = shm->speed;
	m_wave_format.wBitsPerSample = 16;	// always 16bit in Quake sound engine
	m_wave_format.nBlockAlign = m_wave_format.wBitsPerSample/8 * m_wave_format.nChannels;
	m_wave_format.nAvgBytesPerSec = m_wave_format.nSamplesPerSec * m_wave_format.nBlockAlign;

	memset (&stream_header, 0, sizeof(stream_header));
	stream_header.fccType = streamtypeAUDIO;
	stream_header.dwScale = m_wave_format.nBlockAlign;
	stream_header.dwRate = stream_header.dwScale * (unsigned long)m_wave_format.nSamplesPerSec;
	stream_header.dwSampleSize = m_wave_format.nBlockAlign;

	hr = qAVIFileCreateStream (m_file, &m_audio_stream, &stream_header);
	if (FAILED(hr))
	{
		Con_Printf ("ERROR: Couldn't create audio stream\n");
		return false;
	}

	if (m_audio_is_mp3)
	{
		memset (&m_mp3_format, 0, sizeof(m_mp3_format));
		m_mp3_format.wfx.wFormatTag = WAVE_FORMAT_MPEGLAYER3;
		m_mp3_format.wfx.nChannels = 2;
		m_mp3_format.wfx.nSamplesPerSec = shm->speed;
		m_mp3_format.wfx.wBitsPerSample = 0;
		m_mp3_format.wfx.nBlockAlign = 1;
		m_mp3_format.wfx.nAvgBytesPerSec = capture_mp3_kbps.value * 125;
		m_mp3_format.wfx.cbSize = MPEGLAYER3_WFX_EXTRA_BYTES;
		m_mp3_format.wID = MPEGLAYER3_ID_MPEG;
		m_mp3_format.fdwFlags = MPEGLAYER3_FLAG_PADDING_OFF;
		m_mp3_format.nBlockSize = m_mp3_format.wfx.nAvgBytesPerSec / capture_fps.value;
		m_mp3_format.nFramesPerBlock = 1;
		m_mp3_format.nCodecDelay = 1393;

		// try to find an MP3 codec
		m_mp3_driver = NULL;
		mp3_encoder_exists = false;
		qacmDriverEnum (acmDriverEnumCallback, 0, 0);
		if (!mp3_encoder_exists)
		{
			Con_Printf ("ERROR: Couldn't find any MP3 encoder\n");
			return false;
		}

		hr = qAVIStreamSetFormat (m_audio_stream, 0, &m_mp3_format, sizeof(MPEGLAYER3WAVEFORMAT));
		if (FAILED(hr))
		{
			Con_Printf ("ERROR: Couldn't set audio stream format\n");
			return false;
		}
	}
	else
	{
		hr = qAVIStreamSetFormat (m_audio_stream, 0, &m_wave_format, sizeof(WAVEFORMATEX));
		if (FAILED(hr))
		{
			Con_Printf ("ERROR: Couldn't set audio stream format\n");
			return false;
		}
	}

	return true;
}

void Capture_Close (void)
{
	if (m_uncompressed_video_stream)
		qAVIStreamRelease (m_uncompressed_video_stream);
	if (m_compressed_video_stream)
		qAVIStreamRelease (m_compressed_video_stream);
	if (m_audio_stream)
		qAVIStreamRelease (m_audio_stream);
	if (m_audio_is_mp3)
	{
		qacmStreamClose (m_mp3_stream, 0);
		qacmDriverClose (m_mp3_driver, 0);
	}
	if (m_file)
		qAVIFileRelease (m_file);

	qAVIFileExit ();
}

void Capture_WriteVideo (byte *pixel_buffer)
{
	HRESULT	hr;
#ifdef GLQUAKE
	int	size = glwidth * glheight * 3;
#else
	int	size = vid.width * vid.height * 3;
#endif

	// check frame size (TODO: other things too?) hasn't changed
	if (m_video_frame_size != size)
	{
		Con_Printf ("ERROR: Frame size changed\n");
		return;
	}

	if (!Capture_VideoStream())
	{
		Con_Printf ("ERROR: Video stream is NULL\n");
		return;
	}

	hr = qAVIStreamWrite (Capture_VideoStream(), m_video_frame_counter++, 1, pixel_buffer, m_video_frame_size, AVIIF_KEYFRAME, NULL, NULL);
	if (FAILED(hr))
	{
		Con_Printf ("ERROR: Couldn't write video stream\n");
		return;
	}
}

void Capture_WriteAudio (int samples, byte *sample_buffer)
{
	HRESULT		hr;
	unsigned long sample_bufsize;

	if (!m_audio_stream)
	{
		Con_Printf ("ERROR: Audio stream is NULL\n");
		return;
	}

	sample_bufsize = samples * m_wave_format.nBlockAlign;
	if (m_audio_is_mp3)
	{
		MMRESULT	mmr;
		byte		*mp3_buffer;
		unsigned long mp3_bufsize;

		if ((mmr = qacmStreamSize(m_mp3_stream, sample_bufsize, &mp3_bufsize, ACM_STREAMSIZEF_SOURCE)))
		{
			Con_Printf ("ERROR: Couldn't get mp3bufsize\n");
			return;
		}
		if (!mp3_bufsize)
		{
			Con_Printf ("ERROR: mp3bufsize is zero\n");
			return;
		}
		mp3_buffer = calloc (mp3_bufsize, 1);

		memset (&m_mp3_stream_header, 0, sizeof(m_mp3_stream_header));
		m_mp3_stream_header.cbStruct = sizeof(m_mp3_stream_header);
		m_mp3_stream_header.pbSrc = sample_buffer;
		m_mp3_stream_header.cbSrcLength = sample_bufsize;
		m_mp3_stream_header.pbDst = mp3_buffer;
		m_mp3_stream_header.cbDstLength = mp3_bufsize;

		if ((mmr = qacmStreamPrepareHeader(m_mp3_stream, &m_mp3_stream_header, 0)))
		{
			Con_Printf ("ERROR: Couldn't prepare header\n");
			free (mp3_buffer);
			return;
		}

		if ((mmr = qacmStreamConvert(m_mp3_stream, &m_mp3_stream_header, ACM_STREAMCONVERTF_BLOCKALIGN)))
		{
			Con_Printf ("ERROR: Couldn't convert audio stream\n");
			goto clean;
		}

		hr = qAVIStreamWrite (m_audio_stream, m_audio_frame_counter++, 1, mp3_buffer, m_mp3_stream_header.cbDstLengthUsed, AVIIF_KEYFRAME, NULL, NULL);

clean:
		if ((mmr = qacmStreamUnprepareHeader(m_mp3_stream, &m_mp3_stream_header, 0)))
		{
			Con_Printf ("ERROR: Couldn't unprepare header\n");
			free (mp3_buffer);
			return;
		}

		free (mp3_buffer);
	}
	else
	{
		hr = qAVIStreamWrite (m_audio_stream, m_audio_frame_counter++, 1, sample_buffer, sample_bufsize, AVIIF_KEYFRAME, NULL, NULL);
	}
	if (FAILED(hr))
	{
		Con_Printf ("ERROR: Couldn't write audio stream\n");
		return;
	}
}
#endif // Baker change +

/*
===============================================================================

SYSTEM IO

===============================================================================
*/

/*
================
Sys_MakeCodeWriteable
================
*/
void Sys_MakeCodeWriteable (unsigned long startaddr, unsigned long length)
{
	DWORD  flOldProtect;

	if (!VirtualProtect((LPVOID)startaddr, length, PAGE_READWRITE, &flOldProtect))
   		Sys_Error("Protection change failed\n");
}

/*
================
Sys_Init
================
*/
void Sys_Init (void)
{
#ifndef SUPPORTS_CLOCK_FIX // Baker change
	LARGE_INTEGER	PerformanceFreq;
	unsigned int	lowpart, highpart;
#endif // Baker change -
	OSVERSIONINFO	vinfo;
#ifndef SUPPORTS_CLOCK_FIX // Baker change
	if (!QueryPerformanceFrequency (&PerformanceFreq))
		Sys_Error ("No hardware timer available");

// get 32 out of the 64 time bits such that we have around
// 1 microsecond resolution
	lowpart = (unsigned int)PerformanceFreq.LowPart;
	highpart = (unsigned int)PerformanceFreq.HighPart;
	lowshift = 0;

	while (highpart || (lowpart > 2000000.0))
	{
		lowshift++;
		lowpart >>= 1;
		lowpart |= (highpart & 1) << 31;
		highpart >>= 1;
	}

	pfreq = 1.0 / (double)lowpart;
#endif // Baker change -

	Sys_InitFloatTime ();

	vinfo.dwOSVersionInfoSize = sizeof(vinfo);

	if (!GetVersionEx (&vinfo))
		Sys_Error ("Couldn't get OS info");

	if ((vinfo.dwMajorVersion < 4) ||
		(vinfo.dwPlatformId == VER_PLATFORM_WIN32s))
	{
		Sys_Error ("WinQuake requires at least Win95 or NT 4.0");
	}

	if (vinfo.dwPlatformId == VER_PLATFORM_WIN32_NT)
		WinNT = true;
	else
		WinNT = false;
}


void Sys_Error (char *error, ...)
{
	va_list		argptr;
	char		text[1024], text2[1024];
	char		*text3 = "Press Enter to exit\n";
	char		*text4 = "***********************************\n";
	char		*text5 = "\n";
	DWORD		dummy;
	double		starttime;
	static int	in_sys_error0 = 0;
	static int	in_sys_error1 = 0;
	static int	in_sys_error2 = 0;
	static int	in_sys_error3 = 0;

	if (!in_sys_error3)
	{
		in_sys_error3 = 1;
/*		VID_ForceUnlockedAndReturnState ();	Baker: WinQuake only */
	}

	va_start (argptr, error);
	vsprintf (text, error, argptr);
	va_end (argptr);

	if (isDedicated)
	{
		va_start (argptr, error);
		vsprintf (text, error, argptr);
		va_end (argptr);

		sprintf (text2, "ERROR: %s\n", text);
		WriteFile (houtput, text5, strlen (text5), &dummy, NULL);
		WriteFile (houtput, text4, strlen (text4), &dummy, NULL);
		WriteFile (houtput, text2, strlen (text2), &dummy, NULL);
		WriteFile (houtput, text3, strlen (text3), &dummy, NULL);
		WriteFile (houtput, text4, strlen (text4), &dummy, NULL);


		starttime = Sys_FloatTime ();
		sc_return_on_enter = true;	// so Enter will get us out of here

		while (!Sys_ConsoleInput () &&
				((Sys_FloatTime () - starttime) < CONSOLE_ERROR_TIMEOUT))
		{
		}
	}
	else
	{
	// switch to windowed so the message box is visible, unless we already
	// tried that and failed
		if (!in_sys_error0)
		{
			in_sys_error0 = 1;
			VID_SetDefaultMode ();
			MessageBox(NULL, text, "Quake Error",
					   MB_OK | MB_SETFOREGROUND | MB_ICONSTOP);
		}
		else
		{
			MessageBox(NULL, text, "Double Quake Error",
					   MB_OK | MB_SETFOREGROUND | MB_ICONSTOP);
		}
	}

	if (!in_sys_error1)
	{
		in_sys_error1 = 1;
		Host_Shutdown ();
	}

// shut down QHOST hooks if necessary
	if (!in_sys_error2)
	{
		in_sys_error2 = 1;
		DeinitConProc ();
	}

	exit (1);
}

void Sys_Printf (char *fmt, ...)
{
	va_list		argptr;
	char		text[1024];
	DWORD		dummy;

	if (isDedicated)
	{
		va_start (argptr,fmt);
		vsprintf (text, fmt, argptr);
		va_end (argptr);

		WriteFile(houtput, text, strlen (text), &dummy, NULL);
	}
}

void Sys_Quit (void)
{

/*	VID_ForceUnlockedAndReturnState (); Baker: WinQuake only */

	Host_Shutdown();

	if (tevent)
		CloseHandle (tevent);

	if (isDedicated)
		FreeConsole ();

// shut down QHOST hooks if necessary
	DeinitConProc ();

	exit (0);
}

#ifdef SUPPORTS_CLOCK_FIX // Baker change
static	double	pfreq;
static qboolean	hwtimer = false;

void Sys_InitFloatTime (void)
{
	__int64	freq;

	if (!COM_CheckParm("-nohwtimer") && QueryPerformanceFrequency ((LARGE_INTEGER *)&freq) && freq > 0)
	{
		// hardware timer available
		pfreq = (double)freq;
		hwtimer = true;
	}
	else
	{
		// make sure the timer is high precision, otherwise NT gets 18ms resolution
		timeBeginPeriod (1);
	}
}

double Sys_FloatTime (void)
{
	__int64		pcount;
	static	__int64	startcount;
	static	DWORD	starttime;
	static qboolean	first = true;
	DWORD	now;

	if (hwtimer)
	{
		QueryPerformanceCounter ((LARGE_INTEGER *)&pcount);
		if (first)
		{
			first = false;
			startcount = pcount;
			return 0.0;
		}
		// TODO: check for wrapping
		return (pcount - startcount) / pfreq;
	}

	now = timeGetTime ();

	if (first)
	{
		first = false;
		starttime = now;
		return 0.0;
	}

	if (now < starttime) // wrapped?
		return (now / 1000.0) + (LONG_MAX - starttime / 1000.0);

	if (now - starttime == 0)
		return 0.0;

	return (now - starttime) / 1000.0;
}
#else // Baker change +
/*
================
Sys_FloatTime
================
*/
double Sys_FloatTime (void)
{
	static int			sametimecount;
	static unsigned int	oldtime;
	static int			first = 1;
	LARGE_INTEGER		PerformanceCount;
	unsigned int		temp, t2;
	double				time;

	QueryPerformanceCounter (&PerformanceCount);

	temp = ((unsigned int)PerformanceCount.LowPart >> lowshift) |
		   ((unsigned int)PerformanceCount.HighPart << (32 - lowshift));

	if (first)
	{
		oldtime = temp;
		first = 0;
	}
	else
	{
	// check for turnover or backward time
		if ((temp <= oldtime) && ((oldtime - temp) < 0x10000000))
		{
			oldtime = temp;	// so we can't get stuck
		}
		else
		{
			t2 = temp - oldtime;

			time = (double)t2 * pfreq;
			oldtime = temp;

			curtime += time;

			if (curtime == lastcurtime)
			{
				sametimecount++;

				if (sametimecount > 100000)
				{
					curtime += 1.0;
					sametimecount = 0;
				}
			}
			else
			{
				sametimecount = 0;
			}

			lastcurtime = curtime;
		}
	}

    return curtime;
}


/*
================
Sys_InitFloatTime
================
*/
void Sys_InitFloatTime (void)
{
	int		j;

	Sys_FloatTime ();

	j = COM_CheckParm("-starttime");

	if (j)
	{
		curtime = (double) (Q_atof(com_argv[j+1]));
	}
	else
	{
		curtime = 0.0;
	}

	lastcurtime = curtime;
}
#endif // Baker change -

#ifdef SUPPORTS_CLIPBOARD // Baker change

// copies given text to clipboard
void Sys_CopyToClipboard(char *text)
{
	char *clipText;
	HGLOBAL hglbCopy;

	if (!OpenClipboard(NULL))
		return;

	if (!EmptyClipboard())
	{
		CloseClipboard();
		return;
	}

	if (!(hglbCopy = GlobalAlloc(GMEM_DDESHARE, strlen(text) + 1)))
	{
		CloseClipboard();
		return;
	}

	if (!(clipText = GlobalLock(hglbCopy)))
	{
		CloseClipboard();
		return;
	}

	strcpy((char *) clipText, text);
	GlobalUnlock(hglbCopy);
	SetClipboardData(CF_TEXT, hglbCopy);

	CloseClipboard();
}

void Sys_Image_BGRA_To_Clipboard (byte *bmbits, int width, int height, int size)
{

	HBITMAP hBitmap= CreateBitmap (width, height, 1, 32 /* bits per pixel is 32 */, bmbits);

	OpenClipboard (NULL);

	if (!EmptyClipboard())
	{
      CloseClipboard();
      return;
	}

	if ((SetClipboardData (CF_BITMAP, hBitmap)) == NULL)
	 Sys_Error ("SetClipboardData failed");

	CloseClipboard ();

}

#endif // Baker change +

char *Sys_ConsoleInput (void)
{
	static char	text[256];
	static int		len;
	INPUT_RECORD	recs[1024];
	int		count;
#ifdef GENERIC_TOUCH_UP // Baker change (the datatypes did not match the API (DWORD isn't an int)
	int		ch;
	DWORD	numread, numevents, dummy;
#else // Baker change +
	int		i, dummy;
	int		ch, numread, numevents;
#endif // Baker change -

	if (!isDedicated)
		return NULL;


	for ( ;; )
	{
		if (!GetNumberOfConsoleInputEvents (hinput, &numevents))
			Sys_Error ("Error getting # of console events");

		if (numevents <= 0)
			break;

		if (!ReadConsoleInput(hinput, recs, 1, &numread))
			Sys_Error ("Error reading console input");

		if (numread != 1)
			Sys_Error ("Couldn't read console input");

		if (recs[0].EventType == KEY_EVENT)
		{
			if (!recs[0].Event.KeyEvent.bKeyDown)
			{
				ch = recs[0].Event.KeyEvent.uChar.AsciiChar;

				switch (ch)
				{
					case '\r':
						WriteFile(houtput, "\r\n", 2, &dummy, NULL);

						if (len)
						{
							text[len] = 0;
							len = 0;
							return text;
						}
						else if (sc_return_on_enter)
						{
						// special case to allow exiting from the error handler on Enter
							text[0] = '\r';
							len = 0;
							return text;
						}

						break;

					case '\b':
						WriteFile(houtput, "\b \b", 3, &dummy, NULL);
						if (len)
						{
							len--;
						}
						break;

					default:
						if (ch >= ' ')
						{
							WriteFile(houtput, &ch, 1, &dummy, NULL);
							text[len] = ch;
							len = (len + 1) & 0xff;
						}

						break;

				}
			}
		}
	}

	return NULL;
}

void Sys_Sleep (void)
{
	Sleep (1);
}


void Sys_SendKeyEvents (void)
{
    MSG        msg;

	while (PeekMessage (&msg, NULL, 0, 0, PM_NOREMOVE))
	{
	// we always update if there are any event, even if we're paused
		scr_skipupdate = 0;

		if (!GetMessage (&msg, NULL, 0, 0))
			Sys_Quit ();

      	TranslateMessage (&msg);
      	DispatchMessage (&msg);
	}
}

#ifdef SUPPORTS_CHECK_RIGHT_DIR // Baker change
#include <sys/types.h>
#include <sys/stat.h>

qboolean Sys_FileExists (char *path)
{
	struct _stat buf;
	int	result = _stat(path, &buf);

	if( result != 0 )
      return false;

	return true;
}

char *Sys_ExecutableDirectory (void)
{
	static char executableAbsoluteName[1024];
	int i;

	i = GetModuleFileNameA (NULL, executableAbsoluteName, sizeof(executableAbsoluteName) - 1 );
	if (i == 0)
	{
		Sys_Error ("Couldn't determine executable directory");
		return NULL;
	}
	executableAbsoluteName[i] = 0;  //MSDN: Windows XP:  The string is truncated to nSize characters and is not null-terminated.

	COM_SlashesForward_Like_Unix	(executableAbsoluteName);
	COM_Reduce_To_Parent_Path		(executableAbsoluteName);

	return (char *)executableAbsoluteName;
}
#endif // Baker change +

#ifdef SUPPORTS_WINDOW_SET_TITLE // Baker change
void Sys_SetWindowCaption (char *newcaption)
{
	if (!mainwindow)
		return;

	if (!newcaption)
		SetWindowText (mainwindow, "FitzQuake");
	else
		SetWindowText (mainwindow, newcaption);
}
#endif // Baker change +



/*
==============================================================================

 WINDOWS CRAP

==============================================================================
*/


/*
==================
WinMain
==================
*/
void SleepUntilInput (int time)
{

	MsgWaitForMultipleObjects(1, &tevent, FALSE, time, QS_ALLINPUT);
}


/*
==================
WinMain
==================
*/
HINSTANCE	global_hInstance;
int			global_nCmdShow;
char		*argv[MAX_NUM_ARGVS];
static char	*empty_string = "";
HWND		hwnd_dialog;


int WINAPI WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
    MSG				msg;
	quakeparms_t	parms;
	double			time, oldtime, newtime;
	MEMORYSTATUS	lpBuffer;
	static	char	cwd[1024];
	int				t;
	RECT			rect;

    /* previous instances do not exist in Win32 */
    if (hPrevInstance)
        return 0;

	global_hInstance = hInstance;
	global_nCmdShow = nCmdShow;

	lpBuffer.dwLength = sizeof(MEMORYSTATUS);
	GlobalMemoryStatus (&lpBuffer);

	if (!GetCurrentDirectory (sizeof(cwd), cwd))
		Sys_Error ("Couldn't determine current directory");

	if (cwd[Q_strlen(cwd)-1] == '/')
		cwd[Q_strlen(cwd)-1] = 0;

#ifdef SUPPORTS_CHECK_RIGHT_DIR // Baker change
// Check for a pak0.pak in id1.  If not found, use the executable directory and check.
// If both aren't found, they will eventually hit the gfx.wad until they are a bizarre modder not using pakfiles.
// Or if the -basedir parameter is being used.  Basedir overrides this stuff later in initialization.

// What is this for?  If a user makes their own shortcut, they may not set the "Start in" folder that says
// what to use for the current working directory ... so it doesn't find Quake.  The user may not know why it
// isn't finding Quake.


	if (Sys_FileExists (va ("%s/id1/pak0.pak", cwd) ) == false)
	{
		// Check executable dir.  Silently fix if executable directory has an id1 folder with pak0.pak
		char* exe_dir = Sys_ExecutableDirectory ();

		if (Sys_FileExists (va ("%s/id1/pak0.pak", exe_dir) ) == true)
			strcpy (cwd, exe_dir); // Copy exe_dir to cwd
	}

#endif // Baker change +

	parms.basedir = cwd;
	parms.cachedir = NULL;

	parms.argc = 1;
	argv[0] = empty_string;

	while (*lpCmdLine && (parms.argc < MAX_NUM_ARGVS))
	{
		while (*lpCmdLine && ((*lpCmdLine <= 32) || (*lpCmdLine > 126)))
			lpCmdLine++;

		if (*lpCmdLine)
		{
			argv[parms.argc] = lpCmdLine;
			parms.argc++;

			while (*lpCmdLine && ((*lpCmdLine > 32) && (*lpCmdLine <= 126)))
				lpCmdLine++;

			if (*lpCmdLine)
			{
				*lpCmdLine = 0;
				lpCmdLine++;
			}

		}
	}

	parms.argv = argv;

	COM_InitArgv (parms.argc, parms.argv);

	parms.argc = com_argc;
	parms.argv = com_argv;

	isDedicated = (COM_CheckParm ("-dedicated") != 0);

#if 1 //johnfitz -- 0 to supress the 'starting quake' dialog
	if (!isDedicated)
	{
		hwnd_dialog = CreateDialog(hInstance, MAKEINTRESOURCE(IDD_DIALOG1), NULL, NULL);

		if (hwnd_dialog)
		{
#ifdef SUPPORTS_REALLY_CENTERING_WINDOW // Baker change
			RECT workarea;
			if (SystemParametersInfo (SPI_GETWORKAREA, 0, &workarea, 0) && GetWindowRect (hwnd_dialog, &rect) && AdjustWindowRectEx(&rect, 0, FALSE, 0))
			{
               // center it properly in the working area (don't assume that top and left are 0!!!)
				 SetWindowPos
				 (
					hwnd_dialog,
					NULL,
					workarea.left + ((workarea.right - workarea.left) - (rect.right - rect.left)) / 2,
					workarea.top + ((workarea.bottom - workarea.top) - (rect.bottom - rect.top)) / 2,
					0,
					0,
					SWP_NOZORDER | SWP_NOSIZE
				 );
			}
#else // Baker change +
			if (GetWindowRect (hwnd_dialog, &rect))
			{
				if (rect.left > (rect.top * 2))
				{
					SetWindowPos (hwnd_dialog, 0,
						(rect.left / 2) - ((rect.right - rect.left) / 2),
						rect.top, 0, 0,
						SWP_NOZORDER | SWP_NOSIZE);
				}
			}
#endif // Baker change -
			ShowWindow (hwnd_dialog, SW_SHOWDEFAULT);
			UpdateWindow (hwnd_dialog);
			SetForegroundWindow (hwnd_dialog);
		}
	}
#endif

// take the greater of all the available memory or half the total memory,
// but at least 8 Mb and no more than 16 Mb, unless they explicitly
// request otherwise
	parms.memsize = lpBuffer.dwAvailPhys;

	if (parms.memsize < MINIMUM_WIN_MEMORY)
		parms.memsize = MINIMUM_WIN_MEMORY;

	if (parms.memsize < (lpBuffer.dwTotalPhys >> 1))
		parms.memsize = lpBuffer.dwTotalPhys >> 1;

	if (parms.memsize > MAXIMUM_WIN_MEMORY)
		parms.memsize = MAXIMUM_WIN_MEMORY;

	if (COM_CheckParm ("-heapsize"))
	{
		t = COM_CheckParm("-heapsize") + 1;

		if (t < com_argc)
			parms.memsize = Q_atoi (com_argv[t]) * 1024;
	}

	parms.membase = malloc (parms.memsize);

	if (!parms.membase)
		Sys_Error ("Not enough memory free; check disk space\n");

	Sys_PageIn (parms.membase, parms.memsize);

	tevent = CreateEvent(NULL, FALSE, FALSE, NULL);

	if (!tevent)
		Sys_Error ("Couldn't create event");

	if (isDedicated)
	{
		if (!AllocConsole ())
		{
			Sys_Error ("Couldn't create dedicated server console");
		}

		hinput = GetStdHandle (STD_INPUT_HANDLE);
		houtput = GetStdHandle (STD_OUTPUT_HANDLE);

	// give QHOST a chance to hook into the console
		if ((t = COM_CheckParm ("-HFILE")) > 0)
		{
			if (t < com_argc)
				hFile = (HANDLE)Q_atoi (com_argv[t+1]);
		}

		if ((t = COM_CheckParm ("-HPARENT")) > 0)
		{
			if (t < com_argc)
				heventParent = (HANDLE)Q_atoi (com_argv[t+1]);
		}

		if ((t = COM_CheckParm ("-HCHILD")) > 0)
		{
			if (t < com_argc)
				heventChild = (HANDLE)Q_atoi (com_argv[t+1]);
		}

		InitConProc (hFile, heventParent, heventChild);
	}

	Sys_Init ();

// because sound is off until we become active
	S_BlockSound ();

	Sys_Printf ("Host_Init\n");
	Host_Init (&parms);

	oldtime = Sys_FloatTime ();

    /* main window message loop */
	while (1)
	{
		if (isDedicated)
		{
			newtime = Sys_FloatTime ();
			time = newtime - oldtime;

			while (time < sys_ticrate.value )
			{
				Sys_Sleep();
				newtime = Sys_FloatTime ();
				time = newtime - oldtime;
			}
		}
		else
		{
		// yield the CPU for a little while when paused, minimized, or not the focus
			if ((cl.paused && (!ActiveApp /* && !DDActiveBaker: WinQuake only */)) || Minimized /* || block_drawingBaker: WinQuake only */)
			{
				SleepUntilInput (PAUSE_SLEEP);
				scr_skipupdate = 1;		// no point in bothering to draw
			}
			else if (!ActiveApp /* && !DDActive Baker: WinQuake only */)
			{
				SleepUntilInput (NOT_FOCUS_SLEEP);
			}

			newtime = Sys_FloatTime ();
			time = newtime - oldtime;
		}

		Host_Frame (time);
		oldtime = newtime;
	}

    /* return success of application */
    return TRUE;
}

