/*
Copyright (C) 1996-2001 Id Software, Inc.
Copyright (C) 2002-2012 John Fitzgibbons and others

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 3
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "quakedef.h"

void CL_FinishTimeDemo (void);

#ifdef SUPPORTS_GHOSTING_VIA_DEMOS // Baker change
void CL_FinishGhostDemo (void);
#endif // Baker change +


#ifdef SUPPORTS_DEMO_REWIND // Baker change
typedef struct framepos_s
{
	long				baz;
	struct framepos_s	*next;
} framepos_t;

framepos_t	*dem_framepos = NULL;
qboolean	start_of_demo = false;
qboolean	bumper_on = false;
#endif // Baker change +

/*
==============================================================================

DEMO CODE

When a demo is playing back, all NET_SendMessages are skipped, and
NET_GetMessages are read from the demo file.

Whenever cl.time gets past the last received message, another message is
read from the demo file.
==============================================================================
*/

#ifdef SUPPORTS_DEMO_RECORD_ANY_TIME // Baker change
// From ProQuake: space to fill out the demo header for record at any time
byte	demo_head[3][MAX_MSGLEN];
int		demo_head_size[2];
#endif // Baker change +

/*
==============
CL_StopPlayback

Called when a demo file runs out, or the user starts a game
==============
*/
void CL_StopPlayback (void)
{
	if (!cls.demoplayback)
		return;

	fclose (cls.demofile);	
	cls.demoplayback = false;
	cls.demofile = NULL;
	cls.state = ca_disconnected;

#ifdef SUPPORTS_GHOSTING_VIA_DEMOS // Baker change
	if (cls.ghostdemo)
		CL_FinishGhostDemo ();
#endif // Baker change +
	
	if (cls.timedemo)
		CL_FinishTimeDemo ();

#ifdef SUPPORTS_AVI_CAPTURE
#ifdef SUPPORTS_DEMO_PROGRESS_ESTIMATION // Baker change
	if (cls.capturedemo)
		Sys_SetWindowCaption (NULL); // Restores it to default of "FitzQuake"
#endif // Baker change +

	Movie_StopPlayback ();
#endif

#ifdef SUPPORTS_TITLE_DEMOS // Baker change
		if (cls.titledemo)
		{
			vid.recalc_refdef = true; // also triggers sbar refresh
			cls.titledemo = 0;
		}
#endif // Baker change +
}

/*
====================
CL_WriteDemoMessage

Dumps the current net message, prefixed by the length and view angles
====================
*/
void CL_WriteDemoMessage (void)
{
	int		len;
	int		i;
	float	f;

	len = LittleLong (net_message.cursize);
	fwrite (&len, 4, 1, cls.demofile);
	for (i=0 ; i<3 ; i++)
	{
		f = LittleFloat (cl.viewangles[i]);
		fwrite (&f, 4, 1, cls.demofile);
	}
	fwrite (net_message.data, net_message.cursize, 1, cls.demofile);
	fflush (cls.demofile);
}

#ifdef SUPPORTS_DEMO_REWIND // Baker change
void PushFrameposEntry (long fbaz)
{
	framepos_t	*newf;

	newf = malloc (sizeof(framepos_t)); // Demo rewind
	newf->baz = fbaz;

	if (!dem_framepos)
	{
		newf->next = NULL;
		start_of_demo = false;
	}
	else
	{
		newf->next = dem_framepos;
	}
	dem_framepos = newf;
}

static void EraseTopEntry (void)
{
	framepos_t	*top;

	top = dem_framepos;
	dem_framepos = dem_framepos->next;
	free (top);
}
#endif // Baker change +

/*
====================
CL_GetMessage

Handles recording and playback of demos, on top of NET_ code
====================
*/
int CL_GetMessage (void)
{
	int		r, i;
	float	f;

#ifdef SUPPORTS_PAUSE_DEMO // Baker change
	if (cl.paused & 2)
		return 0;
#endif // Baker change +

	if	(cls.demoplayback)
	{
#ifdef SUPPORTS_DEMO_REWIND // Baker change
		if (start_of_demo && cls.demorewind)
			return 0;

		if (cls.signon < SIGNONS)	// clear stuffs if new demo
			while (dem_framepos)
				EraseTopEntry ();
#endif // Baker change +

	// decide if it is time to grab the next message
		if (cls.signon == SIGNONS)	// allways grab until fully connected
		{
			if (cls.timedemo)
			{
				if (host_framecount == cls.td_lastframe)
					return 0;		// allready read this frame's message
				cls.td_lastframe = host_framecount;
			// if this is the second frame, grab the real td_starttime
			// so the bogus time on the first frame doesn't count
				if (host_framecount == cls.td_startframe + 1)
					cls.td_starttime = realtime;
			}
#ifdef SUPPORTS_DEMO_REWIND // Baker change
			else if (!cls.demorewind && cl.ctime <= cl.mtime[0])
				return 0;		// don't need another message yet
			else if (cls.demorewind && cl.ctime >= cl.mtime[0])
				return 0;

			// joe: fill in the stack of frames' positions
			// enable on intermission or not...?
			// NOTE: it can't handle fixed intermission views!
			if (!cls.demorewind /*&& !cl.intermission*/)
				PushFrameposEntry (ftell(cls.demofile));
#else // Baker change +
			else if ( /* cl.time > 0 && */ cl.time <= cl.mtime[0])
			{
				return 0;		// don't need another message yet
			}
#endif // Baker change -

		}

	// get the next message
#ifdef SUPPORTS_DEMO_PROGRESS_ESTIMATION // Baker change
		cls.demo_offset_current = ftell(cls.demofile);
#endif // Baker change +
		fread (&net_message.cursize, 4, 1, cls.demofile);
		VectorCopy (cl.mviewangles[0], cl.mviewangles[1]);
		for (i=0 ; i<3 ; i++)
		{
			r = fread (&f, 4, 1, cls.demofile);
			cl.mviewangles[0][i] = LittleFloat (f);
		}

		net_message.cursize = LittleLong (net_message.cursize);
		if (net_message.cursize > MAX_MSGLEN)
			Sys_Error ("Demo message > MAX_MSGLEN");
		r = fread (net_message.data, net_message.cursize, 1, cls.demofile);
		if (r != 1)
		{
			CL_StopPlayback ();
			return 0;
		}

#ifdef SUPPORTS_DEMO_REWIND // Baker change
		// joe: get out framestack's top entry
		if (cls.demorewind /*&& !cl.intermission*/)
		{
			if (dem_framepos/* && dem_framepos->baz*/)	// Baker: in theory, if this occurs we ARE at the start of the demo with demo rewind on
			{
				fseek (cls.demofile, dem_framepos->baz, SEEK_SET);
				EraseTopEntry (); // Baker: we might be able to improve this better but not right now.
			}
			if (!dem_framepos)
				bumper_on = start_of_demo = true;
		}
#endif // Baker change +

		return 1;
	}

	while (1)
	{
		r = NET_GetMessage (cls.netcon);

		if (r != 1 && r != 2)
			return r;

	// discard nop keepalive message
		if (net_message.cursize == 1 && net_message.data[0] == svc_nop)
			Con_Printf ("<-- server to client keepalive\n");
		else
			break;
	}

	if (cls.demorecording)
		CL_WriteDemoMessage ();

#ifdef SUPPORTS_DEMO_RECORD_ANY_TIME // Baker change
	if (cls.signon < 2)
	{
		memcpy(demo_head[cls.signon], net_message.data, net_message.cursize);
		demo_head_size[cls.signon] = net_message.cursize;
	}
#endif // Baker change +

	return r;
}


/*
====================
CL_Stop_f

stop recording a demo
====================
*/
void CL_Stop_f (void)
{
	if (cmd_source != src_command)
		return;

	if (!cls.demorecording)
	{
		Con_Printf ("Not recording a demo.\n");
		return;
	}

// write a disconnect message to the demo file
	SZ_Clear (&net_message);
	MSG_WriteByte (&net_message, svc_disconnect);
	CL_WriteDemoMessage ();

// finish up
	fclose (cls.demofile);
#ifdef SUPPORTS_FOLDER_COMMAND_LASTFILE // Baker change
	strcpy (recent_file, cls.demoname); // Close demo instance #2 (record end)
#endif // Baker change +

	cls.demofile = NULL;
	cls.demorecording = false;
	Con_Printf ("Completed demo\n");
}

#ifdef SUPPORTS_CLEAR_DEMOQUEUE // Baker change
void CL_Clear_Demos_Queue (void)
{
	int i;
	for (i = 0;i < MAX_DEMOS; i ++)	// Clear demo loop queue
		cls.demos[i][0] = 0;
	cls.demonum = -1;				// Set next demo to none
}
#endif // Baker change +

/*
====================
CL_Record_f

record <demoname> <map> [cd track]
====================
*/
void CL_Record_f (void)
{
	int		c;
	char	name[MAX_OSPATH];
	int		track;

	if (cmd_source != src_command)
		return;

#ifdef TOUCHUP_CANNOT_RECORD_DURING_PLAYDEMO // Baker change
	if (cls.demoplayback)
	{
		Con_Printf ("Can't record during demo playback\n");
		return;
	}
#endif // Baker change +
	c = Cmd_Argc();
	if (c != 2 && c != 3 && c != 4)
	{
		Con_Printf ("record <demoname> [<map> [cd track]]\n");
		return;
	}

	if (strstr(Cmd_Argv(1), ".."))
	{
		Con_Printf ("Relative pathnames are not allowed.\n");
		return;
	}

#ifdef SUPPORTS_DEMO_RECORD_ANY_TIME // Baker change
	if (cls.demorecording)
		CL_Stop_f();

	if (c == 2 && cls.state == ca_connected && cls.signon < 2)
	{
		Con_Printf("Can't record - try again when connected\n");
		return;
	}
#else // Baker change +
	if (c == 2 && cls.state == ca_connected)
	{
		Con_Printf("Can not record - already connected to server\nClient demo recording must be started before connecting\n");
		return;
	}
#endif // Baker change -

// write the forced cd track number, or -1
	if (c == 4)
	{
		track = atoi(Cmd_Argv(3));
		Con_Printf ("Forcing CD track to %i\n", cls.forcetrack);
	}
	else
		track = -1;

	sprintf (name, "%s/%s", com_gamedir, Cmd_Argv(1));

#ifdef SUPPORTS_CLEAR_DEMOQUEUE // Baker change
	CL_Clear_Demos_Queue (); // timedemo is a very intentional action
#endif // Baker change +


//
// start the map up
//
	if (c > 2)
	{
		Cmd_ExecuteString ( va("map %s", Cmd_Argv(2)), src_command);
#ifdef GENERIC_TOUCH_UP // Baker change
		// Joe: if couldn't find the map, don't start recording
		if (cls.state != ca_connected)
			return;
#endif // Baker change +
	}

//
// open the demo file
//
	COM_DefaultExtension (name, ".dem");

	Con_Printf ("recording to %s.\n", name);
	cls.demofile = fopen (name, "wb");
	if (!cls.demofile)
	{
		Con_Printf ("ERROR: couldn't open.\n");
		return;
	}

#ifdef SUPPORTS_FOLDER_COMMAND_LASTFILE // Baker change
	// Officially recording ... copy the name for reference
	strcpy (cls.demoname, name);
#endif // Baker change +

	cls.forcetrack = track;
	fprintf (cls.demofile, "%i\n", cls.forcetrack);

	cls.demorecording = true;

#ifdef SUPPORTS_DEMO_RECORD_ANY_TIME // Baker change
	// From ProQuake: initialize the demo file if we're already connected
	if (c == 2 && cls.state == ca_connected)
	{
		byte *data = net_message.data;
		int cursize = net_message.cursize;
		int i;


#ifdef SUPPORTS_GAMEDIR_HUD_STYLE_CONNECTION // Baker change
// Print info to the demo file telling us the gamedir
		const char* gamedir_string = COM_SkipPath (com_gamedir);
		const char* hud_style_string = rogue ? "-rogue" : (quoth ? "-quoth" : (hipnotic ? "-hipnotic" : "") );		
		char		message[2048];
#if 0	// Baker: Let's not force a gamedir change in the demo.
		char		message2[2048];
		sprintf (message2, "//game %s %s\n", gamedir_string, hud_style_string); //johnfitz -- include fitzquake version
		// Baker: I can think of reasons why this is "wrong".
		// First, playing a demo shouldn't change a gamedir
		// Second, what if I am working on a mod and decide to change the preferred gamedir
		// Third: Shouldn't a demo in a gamedir just be a "dumb" data file.  Don't let it "think".
		// Fourth: This makes it impossible to rename a gamedir or use a different one than anyone else.
		MSG_WriteByte (&net_message, svc_stufftext);
		MSG_WriteString (&net_message, message2);
#endif		
		
		SZ_Clear (&net_message);
		MSG_WriteByte (&net_message, svc_print);
		sprintf (message, "%cgame %s %s\n", 2, gamedir_string, hud_style_string); //johnfitz -- include fitzquake version
		MSG_WriteString (&net_message, message);
		CL_WriteDemoMessage();
#endif // Baker change + SUPPORTS_GAMEDIR_HUD_STYLE_CONNECTION


		for (i = 0 ; i < 2 ; i++)
		{
			net_message.data = demo_head[i];
			net_message.cursize = demo_head_size[i];
			CL_WriteDemoMessage();
		}

		net_message.data = demo_head[2];
		SZ_Clear (&net_message);

		// current names, colors, and frag counts
		for (i=0 ; i < cl.maxclients ; i++)
		{
			MSG_WriteByte (&net_message, svc_updatename);
			MSG_WriteByte (&net_message, i);
			MSG_WriteString (&net_message, cl.scores[i].name);
			MSG_WriteByte (&net_message, svc_updatefrags);
			MSG_WriteByte (&net_message, i);
			MSG_WriteShort (&net_message, cl.scores[i].frags);
			MSG_WriteByte (&net_message, svc_updatecolors);
			MSG_WriteByte (&net_message, i);
			MSG_WriteByte (&net_message, cl.scores[i].colors);
		}

		// send all current light styles
		for (i = 0 ; i < MAX_LIGHTSTYLES ; i++)
		{
			MSG_WriteByte (&net_message, svc_lightstyle);
			MSG_WriteByte (&net_message, i);
			MSG_WriteString (&net_message, cl_lightstyle[i].map);
		}

#ifdef TOUCHUP_DEMO_ANY_TIME_MISSED_SOME_STATS // Baker change
// Baker note: What about the CD track or SVC fog ... future consideration.
// Baker: Revision 9 note.  This wasn't multiplayer friendly (re: coop)
		MSG_WriteByte (&net_message, svc_updatestat);
		MSG_WriteByte (&net_message, STAT_TOTALSECRETS);
		
#ifdef TOUCHUP_DEMO_RECORD_MULTIPLAYER_FIX // Baker change
		MSG_WriteLong (&net_message, cl.stats[STAT_TOTALSECRETS]);
#else // Baker change +
		MSG_WriteLong (&net_message, pr_global_struct->total_secrets);
#endif // Baker change -

		MSG_WriteByte (&net_message, svc_updatestat);
		MSG_WriteByte (&net_message, STAT_TOTALMONSTERS);
#ifdef TOUCHUP_DEMO_RECORD_MULTIPLAYER_FIX // Baker change
		MSG_WriteLong (&net_message, cl.stats[STAT_TOTALMONSTERS]);
#else // Baker change +
		MSG_WriteLong (&net_message, pr_global_struct->total_monsters);
#endif // Baker change -

		MSG_WriteByte (&net_message, svc_updatestat);
		MSG_WriteByte (&net_message, STAT_SECRETS);
#ifdef TOUCHUP_DEMO_RECORD_MULTIPLAYER_FIX // Baker change
		MSG_WriteLong (&net_message, cl.stats[STAT_SECRETS]);
#else // Baker change +
		MSG_WriteLong (&net_message, pr_global_struct->found_secrets);
#endif // Baker change -
		MSG_WriteByte (&net_message, svc_updatestat);
		MSG_WriteByte (&net_message, STAT_MONSTERS);
#ifdef TOUCHUP_DEMO_RECORD_MULTIPLAYER_FIX // Baker change
		MSG_WriteLong (&net_message, cl.stats[STAT_MONSTERS]);
#else // Baker change +
		MSG_WriteLong (&net_message, pr_global_struct->killed_monsters);
#endif // Baker change -

#endif // Baker change +

		// view entity
		MSG_WriteByte (&net_message, svc_setview);
		MSG_WriteShort (&net_message, cl.viewentity);

		// signon
		MSG_WriteByte (&net_message, svc_signonnum);
		MSG_WriteByte (&net_message, 3);

		CL_WriteDemoMessage();

		// restore net_message
		net_message.data = data;
		net_message.cursize = cursize;
	}
#endif // Baker change +
}



/*
====================
CL_PlayDemo_f

play [demoname]
====================
*/

#ifdef SUPPORTS_TITLE_DEMOS // Baker change
cvar_t cl_titledemos_list = {"cl_titledemos_list", ""};
#endif // Baker change +

#ifdef TOUCHUP_CLEAR_DEMOQUEUE_PLAYDEMO // Baker change
// Baker: So we know this is a real start demo
qboolean play_as_start_demo = false;
void CL_PlayDemo_NextStartDemo_f (void)
{
	play_as_start_demo = true;
	CL_PlayDemo_f (); // Inherits the cmd_argc and cmd_argv
	play_as_start_demo = false;
}
#endif // Baker change +

void CL_PlayDemo_f (void)
{
	char	name[256];
	int c;
	qboolean neg = false;

	if (cmd_source != src_command)
		return;

#ifdef TOUCHUP_CLEAR_DEMOQUEUE_PLAYDEMO // Baker change
	if (!play_as_start_demo)
		CL_Clear_Demos_Queue ();
#endif // Baker change +

	if (Cmd_Argc() != 2)
	{
#ifdef GENERIC_TOUCH_UP // Baker change just correction: play ---> playdemo
		Con_Printf ("playdemo <demoname> : plays a demo\n");
#else
		Con_Printf ("play <demoname> : plays a demo\n");
#endif // Baker change neutral
		return;
	}


//
// disconnect from server
//
	CL_Disconnect ();

//
#ifdef SUPPORTS_DEMO_REWIND // Baker change
	// Revert
	cls.demorewind = false;
	cls.demospeed = 0; // 0 = Don't use
	bumper_on = false;
#endif // Baker change +

// open the demo file
//
	strcpy (name, Cmd_Argv(1));
	COM_DefaultExtension (name, ".dem");

#ifdef SUPPORTS_GHOSTING_VIA_DEMOS // Baker change
	Con_LivePrintf ("Playing demo from %s.\n", name);
#else // Baker change +
	Con_Printf ("Playing demo from %s.\n", name);
#endif // Baker change -

	COM_FOpenFile (name, &cls.demofile);
	if (!cls.demofile)
	{
		Con_Printf ("ERROR: couldn't open.\n");
		cls.demonum = -1;		// stop demo loop
		return;
	}

#ifdef SUPPORTS_DEMO_PROGRESS_ESTIMATION // Baker change
	strcpy (cls.demoname, name);
	cls.demo_offset_start = ftell (cls.demofile);	// qfs_lastload.offset instead?
	cls.demo_file_length = com_filesize;
	cls.demo_hosttime_start	= cls.demo_hosttime_elapsed = 0; // Fill this in ... host_time;
	cls.demo_cltime_start = cls.demo_cltime_elapsed = 0; // Fill this in
#endif // Baker change +

#ifdef SUPPORTS_TITLE_DEMOS // Baker change
	// Title demos get no HUD, no crosshair, no con_notify, FOV 90, viewsize 120
	if (COM_ListMatch (cl_titledemos_list.string, cls.demoname))
	{
		vid.recalc_refdef = true;
		cls.titledemo = true;
	}
#endif // Baker change +

	cls.demoplayback = true;
	cls.state = ca_connected;
	cls.forcetrack = 0;

	while ((c = getc(cls.demofile)) != '\n')
		if (c == '-')
			neg = true;
		else 
			cls.forcetrack = cls.forcetrack * 10 + (c - '0');

	if (neg)
		cls.forcetrack = -cls.forcetrack;
// ZOID, fscanf is evil
//	fscanf (cls.demofile, "%i\n", &cls.forcetrack);
}

/*
====================
CL_FinishTimeDemo

====================
*/
void CL_FinishTimeDemo (void)
{
	int		frames;
	float	time;

	cls.timedemo = false;

// the first frame didn't count
	frames = (host_framecount - cls.td_startframe) - 1;
	time = realtime - cls.td_starttime;
	if (!time)
		time = 1;
	Con_Printf ("%i frames %5.1f seconds %5.1f fps\n", frames, time, frames/time);
}

/*
====================
CL_TimeDemo_f

timedemo [demoname]
====================
*/
void CL_TimeDemo_f (void)
{
	if (cmd_source != src_command)
		return;

	if (Cmd_Argc() != 2)
	{
		Con_Printf ("timedemo <demoname> : gets demo speeds\n");
		return;
	}

#ifdef SUPPORTS_TIMEDEMO_CLOSES_CONSOLE // Baker change
	// Baker: This is a performance benchmark.  No reason to have console up.
	if (key_dest != key_game)
		key_dest = key_game;
#endif // Baker change +

#ifdef SUPPORTS_CLEAR_DEMOQUEUE // Baker change
	CL_Clear_Demos_Queue (); // timedemo is a very intentional action
#endif // Baker change +

	CL_PlayDemo_f ();

#ifdef TOUCHUP_RMQ_QUAKESPASM_MILD_TWEAKS // Baker change
	// don't trigger timedemo mode if playdemo fails
	if (!cls.demofile) return;
#endif // Baker change +

// cls.td_starttime will be grabbed at the second frame of the demo, so
// all the loading time doesn't get counted

	cls.timedemo = true;
	cls.td_startframe = host_framecount;
	cls.td_lastframe = -1;		// get a new message this frame

}

#ifdef SUPPORTS_GHOSTING_VIA_DEMOS // Baker change

/*
====================
CL_FinishGhostDemo

====================
*/
void CL_FinishGhostDemo (void)
{
	int		frames;
	float	time;

	if (!cls.ghostfile)
		return; // Not ready yet!  

	cls.ghostdemo = cls.timedemo = false;

//	Close ghost demo file

	fclose (cls.ghostfile);
	cls.ghostfile = NULL;

	strcpy (recent_file, cls.ghostname);

// the first frame didn't count
	frames = (host_framecount - cls.td_startframe) - 1;
	time = realtime - cls.td_starttime;
	if (!time)
		time = 1;
	Con_Printf ("%i frames %5.1f seconds %5.1f fps\n", frames, time, frames/time);
#ifdef SUPPORTS_GHOSTING_AUTOMAP // Baker change
	if (cls.ghostfirstmap)
	{
		SCR_ShowLoading_f ();
		Cbuf_AddText (va("map %s", cls.ghostfirstmap) );
	}
#endif // Baker change +
}


/*
====================
CL_GhostDemo_f

ghostdemo [demoname]
====================
*/
void CL_GhostDemo_f (void)
{
	if (cmd_source != src_command)
		return;

	if (Cmd_Argc() != 2)
	{
		Con_Printf ("%s <demoname> : generates a ghostfile\n", Cmd_Argv(0));
		return;
	}

#if 0
	// Baker: This is a performance benchmark.  No reason to have console up.
	if (key_dest != key_game)
		key_dest = key_game;
#endif

	CL_Clear_Demos_Queue (); // ghostdemo is a very intentional action

	cls.ghostdemo = true;
	CL_PlayDemo_f ();
	cls.ghostdemo = false;

	// don't trigger ghostdemo mode if playdemo fails
	if (!cls.demofile) return;

//	cls.ghost_player_x = cls.ghost_player_y = cls.ghost_player_z =
//	cls.ghost_player_pitch = cls.ghost_player_yaw = cls.ghost_player_roll = 0 ;

//	cls.ghost_player_map[0] = 0;
// Erase ghost write history
	memset (&cls.ghost_write, 0, sizeof(cls.ghost_write) );

	sprintf (cls.ghostname, "%s/%s", com_gamedir, Cmd_Argv(1));
	COM_DefaultExtension (cls.ghostname, ".ghost");
//	Con_Printf ("recording to %s.\n", name);
	cls.ghostfile = fopen (cls.ghostname, "wb");
	if (!cls.ghostfile)
	{
		cls.ghostname[0] = 0;
		Con_Printf ("ERROR: couldn't open.\n");
		return;
	}


// cls.td_starttime will be grabbed at the second frame of the demo, so
// all the loading time doesn't get counted

	cls.ghostdemo = cls.timedemo = true;
#ifdef SUPPORTS_GHOSTING_AUTOMAP // Baker change
	// If creating a new ghost file erase the firstmap field
	cls.ghostfirstmap[0] = 0;
#endif // Baker change +

	cls.td_startframe = host_framecount;
	cls.td_lastframe = -1;		// get a new message this frame

#if 1
	Cmd_Wait_f (); // Stop execution of command buffer until we are done
#endif
}

// Returns true is the point in the cube
qboolean PointInCube(vec3_t point, vec3_t cube_mins, vec3_t cube_maxs)
{
	if (cube_mins[0] <= point[0] && point[0] <= cube_maxs[0])
		if (cube_mins[1] <= point[1] && point[1] <= cube_maxs[1])
			if (cube_mins[2] <= point[2] && point[2] <= cube_maxs[2])
				return true;

	return false;
}


// Baker: This is sort of a cl_main.c kind of function
qboolean CL_GhostVisible (vec3_t ghost_origin)
{
	vec3_t real_player = {cl_entities[cl.viewentity].origin[0], cl_entities[cl.viewentity].origin[1], cl_entities[cl.viewentity].origin[2] /*+ cl.viewheight*/};
	vec3_t ghost_center = {ghost_origin[0], ghost_origin[1], ghost_origin[2]};
	vec3_t ghost_head = {ghost_origin[0], ghost_origin[1], ghost_origin[2] + 32 - 0.1};
	vec3_t ghost_feet = {ghost_origin[0], ghost_origin[1], ghost_origin[2] - 24 + 0.1};
	vec3_t corner;
	//model_t *player_model = Mod_ForName("progs/player.mdl", false);
	entity_t *ent;
	int i, j;

	// Check the center against the static unchanging world
	if (TraceLine (real_player, ghost_center, corner) == false)
		if (TraceLine (real_player, ghost_head, corner) == false)
			if (TraceLine (real_player, ghost_feet, corner) == false)
				return false; // Can't see center, head or feet

	// We can see the center of the player
	// Yet if the player is inside ANY of the world submodels
	// Stop him
	// So check all 8 corners of the player against every
	// Visible submodel.  Keep in mind that this should be ok
	// Since a submodel that blocks the player from being seen
	// should be visible if the player is visible

	for (i = 0 ; i < cl_numvisedicts ; i++)
	{
		vec3_t mins, maxs;
		ent = cl_visedicts[i];

		if (!ent->model)
			continue;   // no model for ent

		if (ent->model->type != mod_brush)
			continue; // Only want brush models

		if (!(ent->model->surfaces == cl.worldmodel->surfaces))
			continue; // This is a health box or something ...

		// Do we need to adjust?  I don't think so
		// But if we do, we'll find out I guess
		if (ent->angles[0] || ent->angles[2]) //pitch or roll
		{
			VectorAdd (ent->origin, ent->model->rmins, mins);
			VectorAdd (ent->origin, ent->model->rmaxs, maxs);
		}
		else if (ent->angles[1]) //yaw
		{
			VectorAdd (ent->origin, ent->model->ymins, mins);
			VectorAdd (ent->origin, ent->model->ymaxs, maxs);
		}
		else //no rotation
		{
			VectorAdd (ent->origin, ent->model->mins, mins);
			VectorAdd (ent->origin, ent->model->maxs, maxs);
		}

//	VectorSet (hull->clip_mins, -16, -16, -24);
//	VectorSet (hull->clip_maxs,  16,  16,  32);
		
		for (j = 0; j < 8; j++)
		{
			corner[0] = ghost_origin[0] + ((j&1)? -16 + 0.1: +16 - 0.1);
			corner[1] = ghost_origin[1] + ((j&2)? -16 + 0.1: +16 - 0.1);
			corner[2] = ghost_origin[2] + ((j&4)? -24 + 0.1: +32 - 0.1);

			// This assumes a tiny brush isn't entirely inside the ghost player
			// I'm not going to worry about that.
			if (PointInCube(corner, mins, maxs) == true) 
			{
//				Con_Printf ("Player is in submodel\n");
				return false; // some part of the player is in the brush
			}
		}		
	}

	return true;
}


void CL_GhostReadRecord (mghost_t *read_data, int newrec, int *out_current_record)
{
	#define FIXEDPOINT_READ(var) fread(&read_int, 1, sizeof(read_int), cls.ghostfileread); var = read_int / 1000.0;
	byte rectype;
	int read_int;
	short frame_in;

	fseek (cls.ghostfileread, newrec * 32, SEEK_SET);
	fread (&rectype, 1, sizeof(rectype), cls.ghostfileread);
	fread (&frame_in, 1, sizeof(frame_in), cls.ghostfileread); // Make sure this is 2!
	read_data->frame = frame_in; // Yay
	FIXEDPOINT_READ (read_data->ghost_time);
	FIXEDPOINT_READ (read_data->origin[0]);
	FIXEDPOINT_READ (read_data->origin[1]);
	FIXEDPOINT_READ (read_data->origin[2]);
	FIXEDPOINT_READ (read_data->angles[0]);
	FIXEDPOINT_READ (read_data->angles[1]);
	FIXEDPOINT_READ (read_data->angles[2]);

	*out_current_record = newrec;
}

#define CL_TIME_FOR(x) (x - cls.ghost_read.virtual_base_time + cls.ghost_read.waiting_accumulator)
#define GHOST_TIME_FOR(x) (x + cls.ghost_read.virtual_base_time - cls.ghost_read.waiting_accumulator)
static qboolean sCL_Ghost_Init_Think (void)
{
	
	if (cls.ghost_record_map_current != 0)
		return false;
	
	// Erase
	memset (&cls.ghost_read, 0, sizeof(cls.ghost_read));

	// READ AND INITIALIZE
	CL_GhostReadRecord (&cls.ghost_read.past, cls.ghost_record_map_begin, &cls.ghost_record_map_current);
	CL_GhostReadRecord (&cls.ghost_read.future, cls.ghost_record_map_begin + 1, &cls.ghost_record_map_current);

	// Establish time base
	// Example ... ghost time starts at 29.5 and cl.time is 1
	// virtual_base_time = 28.5 = 29.5 - 1
	cls.ghost_read.virtual_base_time = cls.ghost_read.past.ghost_time - cl.time;		
		
	// Construct the present.  Important because
	// this might otherwise get rejected.

	cls.ghost_read.current.ghost_time = GHOST_TIME_FOR(cl.time);
//	cls.ghost_read.current.cl_time = cl.time; // Is this necessary?
	cls.ghost_read.current.frac_final = 0; // Use past as-is
	cls.ghost_read.current.framepast = cls.ghost_read.past.frame;
	cls.ghost_read.current.framefuture = cls.ghost_read.future.frame;
	VectorCopy (cls.ghost_read.past.origin, cls.ghost_read.current.origin);
	VectorCopy (cls.ghost_read.past.angles, cls.ghost_read.current.angles);
	cls.ghost_read.last_cl_time = cl.time;

	if (CL_GhostVisible(cls.ghost_read.current.origin) == false)
	{
		// Frozen from start scenario
		memcpy (&cls.ghost_read.current_failed, &cls.ghost_read.current, sizeof(cls.ghost_read.current) );
		cls.ghost_read.out_of_sight = true;
		cls.ghost_read.out_of_sight_cltime = cl.time;

		// Accumulate waiting time and do not progress until this can be cleared.
		// Which means that current_failed must be rendered.

		// The reason we accumulate the wait in the following frame
		// Is our theoretical current_failed ate this timeslice
		// It is the NEXT timeslice that isn't going to happen

	}
	
	// We should be read to go.

	return true;
}





// Determine the current location and interpolation of the ghost
// Possibly pause the ghost
void sCL_Ghost_Current_Think (void)
{
	// Determine if we are paused?
	// How can we do that?
	// Try to update things and if it doesn't work out
	// Then don't and add a time slice into the accumulator

	// Simulate theoretical current state
	// If this state breaks our rules, restore everything and
	// Add some accumated time

	ghostfme_t	simcurrent;
	mghost_t	simpast;
	mghost_t	simfuture;
	double		temp_ghost_time; // Needs to become REAL
	double		temp_time_sliceg;
	double		temp_time_slicec;
	qboolean	temp_end_of_time = false;
	float		temp_frac;
	vec3_t		angle_delta;
	int			sim_record_map_current, i;

	// Check and see if the ghost is "dead".  If so, get out
	if (cls.ghost_read.end_of_time)
		return; // No theoretical future for the ghost ... he's done

	// Check for blocked future (a frame that would be rendered out-of-sight
	// Try to unblock.  If we can't, we must accumulate time.
	// If we can render it, still accumulate the wait, but then
	// Unblock.
	if (cls.ghost_read.out_of_sight)
	{
		// Accumulate the wait no matter what!
		cls.ghost_read.waiting_accumulator += cl.time - cls.ghost_read.out_of_sight_cltime;
		cls.ghost_read.last_cl_time = cl.time;
		// Can we see it now?
		if (CL_GhostVisible(cls.ghost_read.current_failed.origin) == true)
		{
			// We can see it!   Copy to current and remove block.
			cls.ghost_read.out_of_sight = false;
			cls.ghost_read.out_of_sight_cltime = 0;
			// We have a current frame and put the time slice into
			// The wait accumulator.  So get out!
			memcpy (&cls.ghost_read.current, &cls.ghost_read.current_failed, sizeof(cls.ghost_read.current_failed) );

//			Let a potential end of time resolve itself next frame.

			return;
		}
		
		// And ... we still can't see it ... so get out.
		// Current view didn't change at all.
		cls.ghost_read.out_of_sight_cltime = cl.time;
		cls.ghost_read.out_of_sight = true; // For clarity
		return;
	}

	// CONSTRUCT POSSIBLE GHOST CURRENT SCENARIO
	// We discard this IF player can't see it
	// If player can see it, we keep it.

	memset (&simcurrent, 0, sizeof(simcurrent));
	memcpy (&simpast, &cls.ghost_read.past, sizeof(simpast));
	memcpy (&simfuture, &cls.ghost_read.future, sizeof(simfuture));
	sim_record_map_current = cls.ghost_record_map_current;

	temp_ghost_time = GHOST_TIME_FOR(cl.time);
	
//	temp_time_slicec = cl.time - cls.ghost_read.last_cl_time;
//	temp_time_sliceg = GHOST_TIME_FOR(cl.time) - cls.ghost_read.last_ghost_time;
//	cls.ghost_read.last_ghost_time = temp_ghost_time;


	cls.ghost_read.last_cl_time = cl.time;

	// We need to eat the time slice.  How do we do that?
	// Progress as if everything is normal.
	
	// If we our "future" is stale, keep cycling through new futures
	// until we find one or must declare "end of time"
	while (temp_ghost_time > simfuture.ghost_time)
	{
		if (sim_record_map_current >= cls.ghost_record_map_end)
		{
			temp_end_of_time = true; // Set frac to 1
			break; // Can't move into the future any further
		}
		
		// Read a record.  Move future to past.  Read future in from file.
		// Move into the future
		memcpy (&simpast, &simfuture, sizeof(simpast));
		CL_GhostReadRecord (&simfuture, sim_record_map_current + 1, &sim_record_map_current);

//		break;
		// Repeat loop to make sure this "future" is ahead of
		// our ghost_time
	}

	// We have our "past" and "future" set.  Calculate "present"
	// Calculate "true frac time"
	 
	if (temp_end_of_time)
		temp_frac = 1.0f;
	else
	{
		qboolean isteleport = true;

		// TODO: VectorLength would be less messy
		// Only reason didn't use VectorLength is that cl_main.c doesn't use to test for teleport
		// If all 3 deltas are under 100, it isn't a teleport
		if ( (simpast.origin[0] - simfuture.origin[0]) < 100)
			if ( (simpast.origin[1] - simfuture.origin[1]) < 100)
				if ( (simpast.origin[2] - simfuture.origin[2]) < 100)
					isteleport = false;

		if (isteleport)
			temp_frac = 1.0f;
		else
		{
			double range = simfuture.ghost_time - simpast.ghost_time;
			double amount_into_range = temp_ghost_time - simpast.ghost_time;
		
			if (range < 0)
				range = range;
			temp_frac = amount_into_range / range;
			if (temp_frac > 1)
				temp_frac = 1;
		}
	}
	
	// Apply the frac = the past plus some percent of the different into the future.

	VectorSubtract (simfuture.origin, simpast.origin, angle_delta);

	simcurrent.origin[0] = simpast.origin[0] + angle_delta[0] * simcurrent.frac_final;
	simcurrent.origin[1] = simpast.origin[1] + angle_delta[1] * simcurrent.frac_final;
	simcurrent.origin[2] = simpast.origin[2] + angle_delta[2] * simcurrent.frac_final;

	VectorSubtract (simfuture.angles, simpast.angles, angle_delta);

	// always interpolate along the shortest path
	for (i = 0 ; i < 3 ; i++)
	{
		if (angle_delta[i] > 180)
			angle_delta[i] -= 360;
		else if (angle_delta[i] < -180)
			angle_delta[i] += 360;
	}
	
	
	simcurrent.angles[0] = simpast.angles[0] + angle_delta[0] * simcurrent.frac_final;
	simcurrent.angles[1] = simpast.angles[1] + angle_delta[1] * simcurrent.frac_final;
	simcurrent.angles[2] = simpast.angles[2] + angle_delta[2] * simcurrent.frac_final;
	simcurrent.framepast = simpast.frame;
	simcurrent.framefuture=simfuture.frame;
	simcurrent.frac_final= temp_frac;
	
	// Can we see this new "present"?  If so, all is ok.
	// Determine if this is a real or failed future ...
	if (CL_GhostVisible(simcurrent.origin) == false)
	{
		// A discarded future.
		// Preserve it in current_failed
		memcpy (&cls.ghost_read.current_failed, &simcurrent, sizeof(simcurrent));
		cls.ghost_read.out_of_sight_cltime = cl.time;
		cls.ghost_read.out_of_sight = true; // For clarity
		
		// Nothing else that occurred here was real ....
		// We have a stalled ghost with a wannabee future
		// This wannabee future ate the time.
		// And maybe it can used next frame
		return;
	}

	// If we are here, everything that happened is real.

//	Con_Printf ("frac final is %f with elapsed %f\n", (float)temp_frac, (float)(temp_ghost_time - simpast.ghost_time));

	memcpy (&cls.ghost_read.current, &simcurrent, sizeof(simcurrent));
	memcpy (&cls.ghost_read.past, &simpast, sizeof(simpast));
	memcpy (&cls.ghost_read.future, &simfuture, sizeof(simfuture));
	cls.ghost_record_map_current = sim_record_map_current;
	cls.ghost_read.end_of_time = temp_end_of_time;
}

void CL_Ghost_Think (void)
{
	if (!cls.using_ghost)
		return;

	if (!cl.time)
		return; // Wait until we have a time

 	// Initialize/Reset if everything is new
	if (sCL_Ghost_Init_Think ())
		return; // We got initialized.  No more pre-rendering calcs needed

	// If paused, nothing should need to be updated
	// Because cl.time doesn't change when paused.
	// Maybe sv.frozen should be looked at too
	if (cl.paused)
		return;

	// These things happen in a normal frame with elapsed time
	sCL_Ghost_Current_Think (); // Determine state of current frame

}

#endif // Baker change +
