/*
Copyright (C) 1996-2001 Id Software, Inc.
Copyright (C) 2002-2012 John Fitzgibbons and others

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 3
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
#include "quakedef.h"

/*

key up events are sent even if in console mode

*/


#define		MAXCMDLINE	256
#ifdef SUPPORTS_HISTORY_FILE // Baker change
#define		CMDLINES	64

char		key_lines[CMDLINES][MAXCMDLINE];
#ifdef GENERIC_TOUCH_UP // Baker change  (history file is owned by the engine, not the gamedir)
#define		HISTORY_FILE_NAME	"quake_history.txt"
#else // Baker change +
#define		HISTORY_FILE_NAME	"id1/quake_history.txt"
#endif // Baker change -
#else // Baker change +
char	key_lines[32][MAXCMDLINE];
#endif // Baker change -
int		key_linepos;
int		shift_down=false;
int		key_lastpress;
int		key_insert;	//johnfitz -- insert key toggle (for editing)
double	key_blinktime; //johnfitz -- fudge cursor blinking to make it easier to spot in certain cases

int		edit_line=0;
int		history_line=0;

keydest_t	key_dest;

int		key_count;			// incremented every key event

char	*keybindings[256];
int		keyshift[256];		// key to map to if shift held down in console
int		key_repeats[256];	// if > 1, it is autorepeating
qboolean	consolekeys[256];	// if true, can't be rebound while in console
qboolean	menubound[256];	// if true, can't be rebound while in menu
qboolean	keydown[256];

qboolean	repeatkeys[256]; //johnfitz -- if true, autorepeat is enabled for this key

typedef struct
{
	char	*name;
	int		keynum;
} keyname_t;

keyname_t keynames[] =
{
	{"TAB", K_TAB},
	{"ENTER", K_ENTER},
	{"ESCAPE", K_ESCAPE},
	{"SPACE", K_SPACE},
	{"BACKSPACE", K_BACKSPACE},
	{"UPARROW", K_UPARROW},
	{"DOWNARROW", K_DOWNARROW},
	{"LEFTARROW", K_LEFTARROW},
	{"RIGHTARROW", K_RIGHTARROW},

	{"ALT", K_ALT},
	{"CTRL", K_CTRL},
	{"SHIFT", K_SHIFT},

	//johnfitz -- keypad
	{"KP_NUMLOCK",		KP_NUMLOCK},
	{"KP_SLASH",		KP_SLASH },
	{"KP_STAR",			KP_STAR },
	{"KP_MINUS",		KP_MINUS },
	{"KP_HOME",			KP_HOME },
	{"KP_UPARROW",		KP_UPARROW },
	{"KP_PGUP",			KP_PGUP },
	{"KP_PLUS",			KP_PLUS },
	{"KP_LEFTARROW",	KP_LEFTARROW },
	{"KP_5",			KP_5 },
	{"KP_RIGHTARROW",	KP_RIGHTARROW },
	{"KP_END",			KP_END },
	{"KP_DOWNARROW",	KP_DOWNARROW },
	{"KP_PGDN",			KP_PGDN },
	{"KP_ENTER",		KP_ENTER },
	{"KP_INS",			KP_INS },
	{"KP_DEL",			KP_DEL },
	//johnfitz

	{"F1", K_F1},
	{"F2", K_F2},
	{"F3", K_F3},
	{"F4", K_F4},
	{"F5", K_F5},
	{"F6", K_F6},
	{"F7", K_F7},
	{"F8", K_F8},
	{"F9", K_F9},
	{"F10", K_F10},
	{"F11", K_F11},
	{"F12", K_F12},

	{"INS", K_INS},
	{"DEL", K_DEL},
	{"PGDN", K_PGDN},
	{"PGUP", K_PGUP},
	{"HOME", K_HOME},
	{"END", K_END},

	{"MOUSE1", K_MOUSE1},
	{"MOUSE2", K_MOUSE2},
	{"MOUSE3", K_MOUSE3},
#ifdef SUPPORTS_5_MOUSE_BUTTONS // Baker change
	{"MOUSE4", K_MOUSE4},  // Baker: 5 button support
	{"MOUSE5", K_MOUSE5},
	{"MOUSE6", K_MOUSE6},
	{"MOUSE7", K_MOUSE7},
	{"MOUSE8", K_MOUSE8},
#endif // Baker change +

	{"JOY1", K_JOY1},
	{"JOY2", K_JOY2},
	{"JOY3", K_JOY3},
	{"JOY4", K_JOY4},

	{"AUX1", K_AUX1},
	{"AUX2", K_AUX2},
	{"AUX3", K_AUX3},
	{"AUX4", K_AUX4},
	{"AUX5", K_AUX5},
	{"AUX6", K_AUX6},
	{"AUX7", K_AUX7},
	{"AUX8", K_AUX8},
	{"AUX9", K_AUX9},
	{"AUX10", K_AUX10},
	{"AUX11", K_AUX11},
	{"AUX12", K_AUX12},
	{"AUX13", K_AUX13},
	{"AUX14", K_AUX14},
	{"AUX15", K_AUX15},
	{"AUX16", K_AUX16},
	{"AUX17", K_AUX17},
	{"AUX18", K_AUX18},
	{"AUX19", K_AUX19},
	{"AUX20", K_AUX20},
	{"AUX21", K_AUX21},
	{"AUX22", K_AUX22},
	{"AUX23", K_AUX23},
	{"AUX24", K_AUX24},
	{"AUX25", K_AUX25},
	{"AUX26", K_AUX26},
	{"AUX27", K_AUX27},
	{"AUX28", K_AUX28},
	{"AUX29", K_AUX29},
	{"AUX30", K_AUX30},
	{"AUX31", K_AUX31},
	{"AUX32", K_AUX32},

	{"PAUSE", K_PAUSE},

	{"MWHEELUP", K_MWHEELUP},
	{"MWHEELDOWN", K_MWHEELDOWN},

	{"SEMICOLON", ';'},	// because a raw semicolon seperates commands

	{NULL,0}
};

/*
==============================================================================

			LINE TYPING INTO THE CONSOLE

==============================================================================
*/

#ifdef SUPPORTS_CONSOLE_SIZING // Baker change
float	scr_con_size = 0.5;
void AdjustConsoleHeight (const int delta)
{
	int		height;


	if (!cl.worldmodel || cls.signon != SIGNONS)
		return;
	height = (scr_con_size * glheight + delta + 5) / 10;
	height *= 10;
	if (delta < 0 && height < 30)
		height = 30;
	if (delta > 0 && height > vid.height - 10)
		height = vid.height - 10;
	scr_con_size = (float)height / glheight;
}
#endif // Baker change +
/*
====================
Key_Console -- johnfitz -- heavy revision

Interactive line editing and console scrollback
====================
*/
void Key_Console (int key)
{
	extern	int con_vislines;
	extern	char key_tabpartial[MAXCMDLINE];
	char	*cmd;

	switch (key)
	{
	case K_ENTER:
	case KP_ENTER:
		key_tabpartial[0] = 0;
		Cbuf_AddText (key_lines[edit_line]+1);	// skip the prompt
		Cbuf_AddText ("\n");
		Con_Printf ("%s\n",key_lines[edit_line]);
#ifdef SUPPORTS_HISTORY_FILE // Baker change
		edit_line = (edit_line + 1) & (CMDLINES-1);
#else // Baker change +
		edit_line = (edit_line + 1) & 31;
#endif // Baker change -
		history_line = edit_line;
		key_lines[edit_line][0] = ']';
		key_lines[edit_line][1] = 0; //johnfitz -- otherwise old history items show up in the new edit line
		key_linepos = 1;
		if (cls.state == ca_disconnected)
			SCR_UpdateScreen (); // force an update, because the command may take some time
		return;

	case K_TAB:
		Con_TabComplete ();
		return;

	case K_BACKSPACE:

		key_tabpartial[0] = 0;
		if (key_linepos > 1)
		{
			strcpy(key_lines[edit_line] + key_linepos - 1, key_lines[edit_line] + key_linepos);
			key_linepos--;
		}
		return;

	case K_INS:
		key_insert ^= 1;
		return;

	case K_DEL:
		key_tabpartial[0] = 0;
		if (key_linepos < strlen(key_lines[edit_line]))
			strcpy(key_lines[edit_line] + key_linepos, key_lines[edit_line] + key_linepos + 1);
		return;

	case K_HOME:
		if (keydown[K_CTRL])
		{
			//skip initial empty lines
			int		i, x;
			char	*line;
			extern int con_current, con_linewidth;
			extern char *con_text;
			extern float scr_con_current;

			for (i = con_current - con_totallines + 1 ; i <= con_current ; i++)
			{
				line = con_text + (i%con_totallines)*con_linewidth;
				for (x=0 ; x<con_linewidth ; x++)
					if (line[x] != ' ')
						break;
				if (x != con_linewidth)
					break;
			}

			con_backscroll = CLAMP(0, con_current-i%con_totallines-2, con_totallines-(glheight>>3)-1);
		}
		else
			key_linepos = 1;
		return;

	case K_END:
		if (keydown[K_CTRL])
			con_backscroll = 0;
		else
			key_linepos = strlen(key_lines[edit_line]);
		return;

	case K_PGUP:
#ifdef SUPPORTS_MOUSEWHEEL_SCROLL_CONSOLE // Baker change
	case K_MWHEELUP:
#else // Baker change +
	//case K_MWHEELUP:
#endif // Baker change -
		con_backscroll += keydown[K_CTRL] ? ((con_vislines>>3) - 4) : 2;
		if (con_backscroll > con_totallines - (vid.height>>3) - 1)
			con_backscroll = con_totallines - (vid.height>>3) - 1;
		return;

	case K_PGDN:
#ifdef SUPPORTS_MOUSEWHEEL_SCROLL_CONSOLE // Baker change
	case K_MWHEELDOWN:
#else // Baker change +
	//case K_MWHEELDOWN:
#endif // Baker change -
		con_backscroll -= keydown[K_CTRL] ? ((con_vislines>>3) - 4) : 2;
		if (con_backscroll < 0)
			con_backscroll = 0;
		return;

	case K_LEFTARROW:
		if (key_linepos > 1)
		{
			key_linepos--;
			key_blinktime = realtime;
		}
		return;

	case K_RIGHTARROW:
		if (strlen(key_lines[edit_line]) == key_linepos)
		{
#ifdef SUPPORTS_HISTORY_FILE // Baker change
			if (strlen(key_lines[(edit_line + (CMDLINES-1)) & (CMDLINES-1)]) <= key_linepos)
#else // Baker change +
			if (strlen(key_lines[(edit_line + 31) & 31]) <= key_linepos)
#endif // Baker change -
				return; // no character to get

#ifdef SUPPORTS_HISTORY_FILE // Baker change
			key_lines[edit_line][key_linepos] = key_lines[(edit_line + (CMDLINES-1)) & (CMDLINES-1)][key_linepos];
#else // Baker change +
			key_lines[edit_line][key_linepos] = key_lines[(edit_line + 31) & 31][key_linepos];
#endif // Baker change -
			key_linepos++;
			key_lines[edit_line][key_linepos] = 0;
		}
		else
		{
			key_linepos++;
			key_blinktime = realtime;
		}
		return;

	case K_UPARROW:
#ifdef SUPPORTS_CONSOLE_SIZING // Baker change
		if (keydown[K_CTRL])
		{
			AdjustConsoleHeight (-10);
			return;
		}
#endif // Baker change +
		key_tabpartial[0] = 0;
		do
		{
#ifdef SUPPORTS_HISTORY_FILE // Baker change
			history_line = (history_line - 1) & (CMDLINES-1);
#else // Baker change +
			history_line = (history_line - 1) & 31;
#endif // Baker change -
		} while (history_line != edit_line
				&& !key_lines[history_line][1]);
		if (history_line == edit_line)
#ifdef SUPPORTS_HISTORY_FILE // Baker change
			history_line = (edit_line+1)&(CMDLINES-1);
#else // Baker change +
			history_line = (edit_line+1)&31;
#endif // Baker change -
		Q_strcpy(key_lines[edit_line], key_lines[history_line]);
		key_linepos = Q_strlen(key_lines[edit_line]);
		return;

	case K_DOWNARROW:
#ifdef SUPPORTS_CONSOLE_SIZING // Baker change
		if (keydown[K_CTRL])
		{
			AdjustConsoleHeight (10);
			return;
		}
#endif // Baker change +
		key_tabpartial[0] = 0;

		if (history_line == edit_line)
		{
			//clear editline
			key_lines[edit_line][1] = 0;
			key_linepos = 1;
			return;
		}

		do {
#ifdef SUPPORTS_HISTORY_FILE // Baker change
		history_line = (history_line + 1) & (CMDLINES-1);
#else // Baker change +
			history_line = (history_line + 1) & 31;
#endif // Baker change -
		} while (history_line != edit_line
			&& !key_lines[history_line][1]);
		if (history_line == edit_line)
		{
			key_lines[edit_line][0] = ']';
			key_linepos = 1;
		}
		else
		{
			Q_strcpy(key_lines[edit_line], key_lines[history_line]);
			key_linepos = Q_strlen(key_lines[edit_line]);
		}
		return;
	}

//johnfitz -- clipboard pasting, stolen from zquake
//TODO: move win32 specific code to sys_win.c
#ifdef _WIN32
	if ((key=='V' || key=='v') && keydown[K_CTRL])
	{
		HANDLE	th;
		char	*clipText;
		int		i;

		if (OpenClipboard(NULL)) {
			th = GetClipboardData(CF_TEXT);
			if (th) {
				clipText = GlobalLock(th);
				if (clipText) {
					for (i=0; clipText[i]; i++)
						if (clipText[i]=='\n' || clipText[i]=='\r' || clipText[i]=='\b')
							break;
					if (i + strlen(key_lines[edit_line]) > MAXCMDLINE-1)
						i = MAXCMDLINE-1 - strlen(key_lines[edit_line]);
					if (i > 0)
					{	// insert the string
						memmove (key_lines[edit_line] + key_linepos + i,
							key_lines[edit_line] + key_linepos, strlen(key_lines[edit_line]) - key_linepos + 1);
						memcpy (key_lines[edit_line] + key_linepos, clipText, i);
						key_linepos += i;
					}
				}
				GlobalUnlock(th);
			}
			CloseClipboard();
		}
		return;
	}
#endif

	if (key < 32 || key > 127)
		return;	// non printable

	//johnfitz -- stolen from darkplaces
	if (key_linepos < MAXCMDLINE-1)
	{
		int i;

		key_tabpartial[0] = 0; //johnfitz

		if (key_insert)	// check insert mode
		{
			// can't do strcpy to move string to right
			i = strlen(key_lines[edit_line]) - 1;

			if (i == 254)
				i--;

			for (; i >= key_linepos; i--)
				key_lines[edit_line][i + 1] = key_lines[edit_line][i];
		}

		// only null terminate if at the end
		i = key_lines[edit_line][key_linepos];
		key_lines[edit_line][key_linepos] = key;
		key_linepos++;

		if (!i)
			key_lines[edit_line][key_linepos] = 0;
	}
	//johnfitz
}

//============================================================================

#ifdef SUPPORTS_TALK_MACROS // Baker change
#define MAX_CHAT_SIZE 45
char chat_buffer[MAX_CHAT_SIZE];
#else // Baker change +
char chat_buffer[32];
#endif // Baker change -
qboolean team_message = false;

void Key_Message (int key)
{
	static int chat_bufferlen = 0;

	if (key == K_ENTER)
	{
		if (team_message)
			Cbuf_AddText ("say_team \"");
		else
			Cbuf_AddText ("say \"");
		Cbuf_AddText(chat_buffer);
		Cbuf_AddText("\"\n");

		key_dest = key_game;
		chat_bufferlen = 0;
		chat_buffer[0] = 0;
		return;
	}

	if (key == K_ESCAPE)
	{
		key_dest = key_game;
		chat_bufferlen = 0;
		chat_buffer[0] = 0;
		return;
	}

	if (key < 32 || key > 127)
		return;	// non printable

	if (key == K_BACKSPACE)
	{
		if (chat_bufferlen)
		{
			chat_bufferlen--;
			chat_buffer[chat_bufferlen] = 0;
		}
		return;
	}

#ifdef SUPPORTS_TALK_MACROS // Baker change
	if (chat_bufferlen == MAX_CHAT_SIZE - (team_message ? 6 : 1)) // 6 vs 1 = so same amount of text onscreen in "say" versus "say_team"
		return; // all full
#else // Baker change +
	if (chat_bufferlen == 31)
		return; // all full
#endif // Baker change -
	chat_buffer[chat_bufferlen++] = key;
	chat_buffer[chat_bufferlen] = 0;
}

//============================================================================


/*
===================
Key_StringToKeynum

Returns a key number to be used to index keybindings[] by looking at
the given string.  Single ascii characters return themselves, while
the K_* names are matched up.
===================
*/
int Key_StringToKeynum (char *str)
{
	keyname_t	*kn;

	if (!str || !str[0])
		return -1;
	if (!str[1])
		return str[0];

	for (kn=keynames ; kn->name ; kn++)
	{
		if (!Q_strcasecmp(str,kn->name))
			return kn->keynum;
	}
	return -1;
}

/*
===================
Key_KeynumToString

Returns a string (either a single ascii char, or a K_* name) for the
given keynum.
FIXME: handle quote special (general escape sequence?)
===================
*/
char *Key_KeynumToString (int keynum)
{
	keyname_t	*kn;
	static	char	tinystr[2];

	if (keynum == -1)
		return "<KEY NOT FOUND>";
	if (keynum > 32 && keynum < 127)
	{	// printable ascii
		tinystr[0] = keynum;
		tinystr[1] = 0;
		return tinystr;
	}

	for (kn=keynames ; kn->name ; kn++)
		if (keynum == kn->keynum)
			return kn->name;

	return "<UNKNOWN KEYNUM>";
}


/*
===================
Key_SetBinding
===================
*/
void Key_SetBinding (int keynum, char *binding)
{
	char	*new;
	int		l;

	if (keynum == -1)
		return;

// free old bindings
	if (keybindings[keynum])
	{
		Z_Free (keybindings[keynum]);
		keybindings[keynum] = NULL;
	}

// allocate memory for new binding
	l = Q_strlen (binding);
	new = Z_Malloc (l+1);
	Q_strcpy (new, binding);
	new[l] = 0;
	keybindings[keynum] = new;
}

/*
===================
Key_Unbind_f
===================
*/
void Key_Unbind_f (void)
{
	int		b;

	if (Cmd_Argc() != 2)
	{
		Con_Printf ("unbind <key> : remove commands from a key\n");
		return;
	}

	b = Key_StringToKeynum (Cmd_Argv(1));
	if (b==-1)
	{
		Con_Printf ("\"%s\" isn't a valid key\n", Cmd_Argv(1));
		return;
	}

	Key_SetBinding (b, "");
}

void Key_Unbindall_f (void)
{
	int		i;

	for (i=0 ; i<256 ; i++)
		if (keybindings[i])
			Key_SetBinding (i, "");
}

#ifdef SUPPORTS_WINDOWS_MENUS // Baker change 

#include <ctype.h> // isdigit

// NULL terminated list of notable common keys in the order I want
char *customized[][2] = {
	{"togglemenu", 		"menu"},
	{"toggleconsole", 	"goto console"},
	{"+showscores", 	"scoreboard"},
	{"", 				""},
	{"+attack", 		"attack"},
	{"impulse 10", 		"change weapon"},
	{"+jump", 			"jump / swim up"},
	{"", 				""},
	{"+left", 			"turn left"},
	{"+right", 			"turn right"},
	{"+forward", 		"move forward"},
	{"+back", 			"move backward"},
	{"+moveleft", 		"move left"},
	{"+moveright", 		"move right"},
	{"", 				""},
	{"+lookup", 		"look up"},
	{"+lookdown", 		"look down"},
	{"+moveup",			"swim up"},
	{"+movedown",		"swim down"},
	{NULL} };

// Is the string like "messagemode" in the above list?
int is_customize_controls_bind (const char* string)
{
	const int maxx = 18;
	int i;
	for (i = 0; customized[i][0]; i ++)
		if (strcasecmp(customized[i][0], string)==0)
			return 1;

	return 0;
}

ezSortList* Keybind_Menu_List_Alloc (unsigned int* out_count)
{
	ezSortList* out = NULL;
	int i;

	ezSortList* tail = NULL; // This process requires a head and tail, at least for speed
	for (i = 0; customized[i][0]; i++)
	{
		const char* string_string = "";
		const char* itemdata_string = "";
		if (customized[i][0][0]) // Not a spacer in the table
		{
			int		keys[2] = {0};
			M_FindKeysForCommand (customized[i][0], keys); // Find the right key

			string_string = customized[i][1];
			itemdata_string = va(keys[1] != -1 ? "%s or %s" : "%s", Key_KeynumToString (keys[0]), Key_KeynumToString (keys[1]) );
		}		
		tail = ezLinkEnd_Add_With_ItemData (&out, tail, string_string,  itemdata_string );
	}

	return out;
}

ezSortList* _Keybind_List_Alloc (unsigned int* out_count, int no_functions)
{
	ezSortList*		out = NULL;
	unsigned int count = 0, i;
	

	for (i=0 ; i<256 ; i++)
		if (keybindings[i])
			if (*keybindings[i])
			{
				const char* keyname = Key_KeynumToString(i);
				int is_functionkey =  (keyname[0] == 'F' && keyname[1] != 0) || ( isdigit(keyname[0]) && keyname[1]==0 );
				if (!no_functions == !!is_functionkey)
				{
					if (no_functions) // Don't do 
					{
						// See if the bind is in the table is +showscores toggleconsole or menu
						int is_customize_controls_menu = is_customize_controls_bind(keybindings[i]);
						if (no_functions == 2 && is_customize_controls_menu)
							continue;
						if (no_functions == 1 && !is_customize_controls_menu)
							continue;
					}
					ezSortList_Add_With_ItemData (&out, keyname, keybindings[i]);
					count++;
				}
			}

	if (out_count)
		*out_count = count;
	return out;
}

ezSortList* Keybind_Function_List_Alloc (unsigned int* out_count)
{
	return _Keybind_List_Alloc (out_count, 0);
}

ezSortList* Keybind_MenuAll_List_Alloc (unsigned int* out_count)
{
	return _Keybind_List_Alloc (out_count, 1);
}

ezSortList* Keybind_Other_List_Alloc (unsigned int* out_count)
{
	return _Keybind_List_Alloc (out_count, 2);
}

#endif // Baker change +
/*
============
Key_Bindlist_f -- johnfitz
============
*/
void Key_Bindlist_f (void)
{
	int		i, count;

	count = 0;
	for (i=0 ; i<256 ; i++)
		if (keybindings[i])
			if (*keybindings[i])
			{
				Con_SafePrintf ("   %s \"%s\"\n", Key_KeynumToString(i), keybindings[i]);
				count++;
			}
	Con_SafePrintf ("%i bindings\n", count);
}

/*
===================
Key_Bind_f
===================
*/
void Key_Bind_f (void)
{
	int			i, c, b;
	char		cmd[1024];

	c = Cmd_Argc();

	if (c != 2 && c != 3)
	{
		Con_Printf ("bind <key> [command] : attach a command to a key\n");
		return;
	}
	b = Key_StringToKeynum (Cmd_Argv(1));
	if (b==-1)
	{
		Con_Printf ("\"%s\" isn't a valid key\n", Cmd_Argv(1));
		return;
	}

	if (c == 2)
	{
		if (keybindings[b])
			Con_Printf ("\"%s\" = \"%s\"\n", Cmd_Argv(1), keybindings[b] );
		else
			Con_Printf ("\"%s\" is not bound\n", Cmd_Argv(1) );
		return;
	}

// copy the rest of the command line
	cmd[0] = 0;		// start out with a null string
	for (i=2 ; i< c ; i++)
	{
		if (i > 2)
			strcat (cmd, " ");
		strcat (cmd, Cmd_Argv(i));
	}

	Key_SetBinding (b, cmd);
}

/*
============
Key_WriteBindings

Writes lines containing "bind key value"
============
*/
void Key_WriteBindings (FILE *f)
{
	int		i;

	for (i=0 ; i<256 ; i++)
		if (keybindings[i])
			if (*keybindings[i])
				fprintf (f, "bind \"%s\" \"%s\"\n", Key_KeynumToString(i), keybindings[i]);
}



#ifdef SUPPORTS_HISTORY_FILE // Baker change
void History_Init (void)
{
	int i, c;
	FILE *hf;

	for (i = 0; i < CMDLINES; i++) {
		key_lines[i][0] = ']';
		key_lines[i][1] = 0;
	}
	key_linepos = 1;

	if ((hf = fopen(va("%s/%s", com_basedir, HISTORY_FILE_NAME), "rt")))
	{
		do
		{
			i = 1;
			do
			{
				c = fgetc(hf);
				key_lines[edit_line][i++] = c;
			} while (c != '\n' && c != EOF && i < MAXCMDLINE);
			key_lines[edit_line][i - 1] = 0;
			edit_line = (edit_line + 1) & (CMDLINES - 1);
		} while (c != EOF && edit_line < CMDLINES);
		fclose(hf);

		history_line = edit_line = (edit_line - 1) & (CMDLINES - 1);
		key_lines[edit_line][0] = ']';
		key_lines[edit_line][1] = 0;
	}
}

void History_Shutdown (void)
{
	int i;
	FILE *hf;
	char lastentry[1024] = {0};

	if ((hf = fopen(va("%s/%s", com_basedir, HISTORY_FILE_NAME), "wt")))
	{
		i = edit_line;
		do
		{
			i = (i + 1) & (CMDLINES - 1);
		} while (i != edit_line && !key_lines[i][1]);

		do
		{
			if (lastentry[0]==0 || Q_strcasecmp (lastentry, key_lines[i] + 1) != 0) // Baker: prevent saving the same line multiple times in a row
				if (Q_strncasecmp(key_lines[i]+1, "quit", 4) != 0) // Baker: why save quit to the history file
					fprintf(hf, "%s\n", key_lines[i] + 1);

			strcpy (lastentry, key_lines[i] + 1);
			i = (i + 1) & (CMDLINES - 1);

		} while (i != edit_line && key_lines[i][1]);
		fclose(hf);
	}
}
#endif // Baker change +

/*
===================
Key_Init
===================
*/
void Key_Init (void)
{
	int		i;

#ifdef SUPPORTS_HISTORY_FILE // Baker change
	History_Init ();

// Baker ... we are now reading history instead of doing this
//	for (i=0 ; i<32 ; i++)
//	{
//		key_lines[i][0] = ']';
//		key_lines[i][1] = 0;
//	}
//	key_linepos = 1;
#else // Baker change +
	for (i=0 ; i<32 ; i++)
	{
		key_lines[i][0] = ']';
		key_lines[i][1] = 0;
	}
	key_linepos = 1;
#endif // Baker change -

	key_blinktime = realtime; //johnfitz

//
// init ascii characters in console mode
//
	for (i=32 ; i<128 ; i++)
		consolekeys[i] = true;
	consolekeys[K_ENTER] = true;
	consolekeys[KP_ENTER] = true; //johnfitz
	consolekeys[K_TAB] = true;
	consolekeys[K_LEFTARROW] = true;
	consolekeys[K_RIGHTARROW] = true;
	consolekeys[K_UPARROW] = true;
	consolekeys[K_DOWNARROW] = true;
	consolekeys[K_BACKSPACE] = true;
	consolekeys[K_DEL] = true; //johnfitz
	consolekeys[K_INS] = true; //johnfitz
	consolekeys[K_HOME] = true; //johnfitz
	consolekeys[K_END] = true; //johnfitz
	consolekeys[K_PGUP] = true;
	consolekeys[K_PGDN] = true;
	consolekeys[K_SHIFT] = true;
	consolekeys[K_MWHEELUP] = true;
	consolekeys[K_MWHEELDOWN] = true;
	consolekeys['`'] = false;
	consolekeys['~'] = false;

	//johnfitz -- repeating keys
	for (i=0 ; i<256 ; i++)
		repeatkeys[i] = false;
	repeatkeys[K_BACKSPACE] = true;
	repeatkeys[K_DEL] = true;
	repeatkeys[K_PAUSE] = true;
	repeatkeys[K_PGUP] = true;
	repeatkeys[K_PGDN] = true;
	repeatkeys[K_LEFTARROW] = true;
	repeatkeys[K_RIGHTARROW] = true;
	//johnfitz

	for (i=0 ; i<256 ; i++)
		keyshift[i] = i;
	for (i='a' ; i<='z' ; i++)
		keyshift[i] = i - 'a' + 'A';
	keyshift['1'] = '!';
	keyshift['2'] = '@';
	keyshift['3'] = '#';
	keyshift['4'] = '$';
	keyshift['5'] = '%';
	keyshift['6'] = '^';
	keyshift['7'] = '&';
	keyshift['8'] = '*';
	keyshift['9'] = '(';
	keyshift['0'] = ')';
	keyshift['-'] = '_';
	keyshift['='] = '+';
	keyshift[','] = '<';
	keyshift['.'] = '>';
	keyshift['/'] = '?';
	keyshift[';'] = ':';
	keyshift['\''] = '"';
	keyshift['['] = '{';
	keyshift[']'] = '}';
	keyshift['`'] = '~';
	keyshift['\\'] = '|';

	menubound[K_ESCAPE] = true;
	for (i=0 ; i<12 ; i++)
		menubound[K_F1+i] = true;

//
// register our functions
//
	Cmd_AddCommand ("bindlist",Key_Bindlist_f); //johnfitz
	Cmd_AddCommand ("bind",Key_Bind_f);
	Cmd_AddCommand ("unbind",Key_Unbind_f);
	Cmd_AddCommand ("unbindall",Key_Unbindall_f);
}

/*
===================
Key_Event

Called by the system between frames for both key up and key down events
Should NOT be called during an interrupt!
===================
*/
#ifdef SUPPORTS_SPECIAL_KEY_COMBINES // Baker change
qboolean Keys_All_Up_But_Me (int key)
{
	qboolean anydown = false;
	int i;

	for (i = 0; i < 256; i ++)
	{
		if (key == i) // Don't check this key of course, it will be up soon.  But we can't do it here
			continue;
		else if (keydown[i])
			return false;
	}

	// Nothing down except the key being released
	return true;
}

#endif // Baker change +

void Key_Event (int key, qboolean down)
{
	char		*kb;
	char		cmd[1024];

#ifdef SUPPORTS_SPECIAL_KEY_COMBINES // Baker change

#define COMBO_NONE 		0
#define COMBO_ALT_ENTER 1
#define COMBO_CTRL_M 	2

	static int	in_key_combo = COMBO_NONE; // Combo hit.  Wait for all keys up.
	static int  enact_combo  = COMBO_NONE; // Activate combo
	
	if (in_key_combo && !down && enact_combo == 0)
		if (Keys_All_Up_But_Me (key))
		{
			// Combo was in effect and all keys are now released
			// Except current key which was just released.
			// Enact this combo.
			enact_combo = in_key_combo;

			Key_Event (key, down); // Process like normal ...

			// Now perform the combo action ...
			if (enact_combo == COMBO_ALT_ENTER) VID_Alt_Enter_f ();
			if (enact_combo == COMBO_CTRL_M)		Sound_Toggle_Mute_f ();

			enact_combo = in_key_combo = 0;  // Reset
			return; // Get out.  We are done.  
		}

#endif // Baker change +

	keydown[key] = down;

#ifdef SUPPORTS_SPECIAL_KEY_COMBINES // Baker change
	// Enter key released while 
	if (key == K_ENTER  && !down && keydown[K_ALT] && in_key_combo == COMBO_NONE /* && vid_altenter_toggle.integer*/)
		in_key_combo = COMBO_ALT_ENTER; // Once all keys and buttons are released, vid change occurs.

	if (key == 109 /* M */ && !down && keydown[K_CTRL] && in_key_combo == COMBO_NONE)
		in_key_combo = COMBO_CTRL_M;
#endif // Baker change +

	if (!down)
		key_repeats[key] = 0;

	key_lastpress = key;
	key_count++;
	if (key_count <= 0)
	{
		return;		// just catching keys for Con_NotifyBox
	}

// update auto-repeat status
	if (down)
	{
		key_repeats[key]++;
		if (key_repeats[key] > 1 && !repeatkeys[key]) //johnfitz -- use repeatkeys[]
			return;	// ignore most autorepeats
#ifdef SUPPORTS_5_MOUSE_BUTTONS // Baker change
		if (key >= K_MOUSE5 && key<= K_MOUSE8 && !keybindings[key])
#else // Baker change +
		if (key >= 200 && !keybindings[key])
#endif // Baker change -
			Con_Printf ("%s is unbound, hit F4 to set.\n", Key_KeynumToString (key) );
	}

	if (key == K_SHIFT)
		shift_down = down;

//
// handle escape specialy, so the user can never unbind it
//
	if (key == K_ESCAPE)
	{
		if (!down)
			return;
		switch (key_dest)
		{
		case key_message:
			Key_Message (key);
			break;
		case key_menu:
			M_Keydown (key);
			break;
		case key_game:
		case key_console:
			M_ToggleMenu_f ();
			break;
		default:
			Sys_Error ("Bad key_dest");
		}
		return;
	}

// johnfitz -- alt-enter to toggle fullscreen. FIXME -- this does NOT work
#if 0
	if (!down && (key == K_ENTER) && keydown[K_ALT])
	{
		extern cvar_t vid_fullscreen;
		if (vid_fullscreen.value)
			Cvar_Set ("vid_fullscreen", "0");
		else
			Cvar_Set ("vid_fullscreen", "1");
		VID_Restart ();
	}
#endif
// johnfitz

#ifdef SUPPORTS_DEMO_REWIND // Baker change
// PGUP and PGDN rewind and fast-forward demos
	if (cls.demoplayback && cls.demonum == -1 && !cls.timedemo && !cls.capturedemo)
		if (key == K_PGUP || key == K_PGDN)
			if (key_dest == key_game && down /* && cls.demospeed == 0 && cls.demorewind == false*/)
			{
				// During normal demoplayback, PGUP/PGDN will rewind and fast forward (if key_dest is game)
				if (key == K_PGUP)
				{
					cls.demospeed = 5;
					cls.demorewind =  false;
				}
				else if (key == K_PGDN)
				{
					cls.demospeed = 5;
					cls.demorewind = true;
				}
				return; // If something is bound to it, do not process it.
			}
			else //if (!down && (cls.demospeed != 0 || cls.demorewind != 0))
			{
				// During normal demoplayback, releasing PGUP/PGDN resets the speed
				// We need to check even if not key_game in case something silly happened (to be safe)
				cls.demospeed = 0;
				cls.demorewind = false;

				if (key_dest == key_game)
					return; // Otherwise carry on ...
			}
#endif // Baker change +

//
// key up events only generate commands if the game key binding is
// a button command (leading + sign).  These will occur even in console mode,
// to keep the character from continuing an action started before a console
// switch.  Button commands include the kenum as a parameter, so multiple
// downs can be matched with ups
//
	if (!down)
	{
		kb = keybindings[key];
		if (kb && kb[0] == '+')
		{
			sprintf (cmd, "-%s %i\n", kb+1, key);
			Cbuf_AddText (cmd);
		}
		if (keyshift[key] != key)
		{
			kb = keybindings[keyshift[key]];
			if (kb && kb[0] == '+')
			{
				sprintf (cmd, "-%s %i\n", kb+1, key);
				Cbuf_AddText (cmd);
			}
		}
		return;
	}

//
// during demo playback, most keys bring up the main menu
//

#ifdef SUPPORTS_PROPER_DEMO_KEY_BEHAVIOR // Baker change
// Baker:  Quake was intended to bring up the menu with keys during the intro.
// so the user knew what to do.  But if someone does "playdemo" that isn't the intro.
// So we want this behavior ONLY when startdemos are in action.  If startdemos are
// not in action, cls.demonum == -1

#ifdef GENERIC_TOUCH_UP  // Baker change (Needs to be >=0 as demonum 0 is first in loop
	if (cls.demonum >= 0) // We are in startdemos intro.  Bring up menu for keys.
#else // Baker change +
	if (cls.demonum > 0) // We are in startdemos intro.  Bring up menu for keys.
#endif // Baker change -

#endif // Baker change +
		if (cls.demoplayback && down && consolekeys[key] && key_dest == key_game)
		{
			M_ToggleMenu_f ();
			return;
		}


//
// if not a consolekey, send to the interpreter no matter what mode is
//
	if ( (key_dest == key_menu && menubound[key])
	|| (key_dest == key_console && !consolekeys[key])
	|| (key_dest == key_game && ( !con_forcedup || !consolekeys[key] ) ) )
	{
		kb = keybindings[key];
		if (kb)
		{
			if (kb[0] == '+')
			{	// button commands add keynum as a parm
				sprintf (cmd, "%s %i\n", kb, key);
				Cbuf_AddText (cmd);
			}
			else
			{
				Cbuf_AddText (kb);
				Cbuf_AddText ("\n");
			}
		}
		return;
	}

	if (!down)
		return;		// other systems only care about key down events

	if (shift_down)
	{
		key = keyshift[key];
	}

	switch (key_dest)
	{
	case key_message:
		Key_Message (key);
		break;
	case key_menu:
		M_Keydown (key);
		break;

	case key_game:
	case key_console:
		Key_Console (key);
		break;
	default:
		Sys_Error ("Bad key_dest");
	}
}


/*
===================
Key_ClearStates -- johnfitz -- replaced with new function from Baker
===================
*/
void Key_ClearStates (void)
{
   int      i;

   for (i=0 ; i<256 ; i++)
   {
      // If the key is down, trigger the up action if, say, +showscores or another +bind is activated
      if (keydown[i])
         Key_Event (i, false);

   }
}

