/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// snd_mem.c: sound caching

#include <stdlib.h>
#include <string.h>

#include "quakedef.h"
#include "filesystem.h"
#include "sound.h"

#include "fmod.h"

extern struct SoundCard *soundcard;

/*
================
ResampleSfx
================
*/
void ResampleSfx (sfx_t *sfx, int inrate, int inwidth, byte *data)
{
	int		outcount;
	int		srcsample;
	float	stepscale;
	int		i;
	int		sample, samplefrac, fracstep;
	sfxcache_t	*sc;

	sc = sfx->sfxcache;
	if (!sc)
		return;

	stepscale = (float)inrate / soundcard->speed;	// this is usually 0.5, 1, or 2

	outcount = sc->length / stepscale;
	sc->length = outcount;
	if (sc->loopstart != -1)
		sc->loopstart = sc->loopstart / stepscale;

	sc->speed = soundcard->speed;
	if (s_loadas8bit.value)
		sc->width = 1;
	else
		sc->width = inwidth;
	sc->stereo = 0;

// resample / decimate to the current source rate

	if (stepscale == 1 && inwidth == 1 && sc->width == 1)
	{
// fast special case
		for (i=0 ; i<outcount ; i++)
			((signed char *)sc->data)[i]
			= (int)( (unsigned char)(data[i]) - 128);
	}
	else
	{
// general case
		samplefrac = 0;
		fracstep = stepscale*256;
		for (i=0 ; i<outcount ; i++)
		{
			srcsample = samplefrac >> 8;
			samplefrac += fracstep;
			if (inwidth == 2)
				sample = LittleShort ( ((short *)data)[srcsample] );
			else
				sample = (int)( (unsigned char)(data[srcsample]) - 128) << 8;
			if (sc->width == 2)
				((short *)sc->data)[i] = sample;
			else
				((signed char *)sc->data)[i] = sample >> 8;
		}
	}
}

//=============================================================================

/*
==============
S_LoadSound
==============
*/
sfxcache_t *S_LoadSound(sfx_t *s)
{
	char	namebuffer[256];
	byte	*data;
	wavinfo_t	info;
	int		len;
	float	stepscale;
	sfxcache_t	*sc;

	if (!soundcard)
		return 0;

// see if still in memory
	if (s->sfxcache)
		return s->sfxcache;

// load it in
	snprintf(namebuffer, sizeof(namebuffer), "sound/%s", s->name);

	data = FS_LoadMallocFile(namebuffer);

	sc = 0;

	if (!data)
	{
		Com_Printf ("Couldn't load %s\n", namebuffer);
	}
	else
	{
		FMod_CheckModel(namebuffer, data, com_filesize);

		info = GetWavinfo (s->name, data, com_filesize);
		if (info.channels != 1)
		{
			Com_Printf ("%s is a stereo sample\n",s->name);
		}
		else
		{
			stepscale = (float)info.rate / soundcard->speed;
			len = info.samples / stepscale;

			len = len * info.width * info.channels;

			sc = malloc(len + sizeof(sfxcache_t));
			if (sc)
			{
				sc->length = info.samples;
				sc->loopstart = info.loopstart;
				sc->speed = info.rate;
				sc->width = info.width;
				sc->stereo = info.channels;

				s->sfxcache = sc;

				ResampleSfx (s, sc->speed, sc->width, data + info.dataofs);
			}
		}

		free(data);
	}

	return sc;
}



/*
===============================================================================

WAV loading

===============================================================================
*/

#warning FIX THIS! FIX THIS! FIX THIS! FIX THIS!

byte	*data_p;
byte 	*iff_end;
byte 	*last_chunk;
byte 	*iff_data;


short GetLittleShort(void)
{
	short val = 0;
	val = *data_p;
	val = val + (*(data_p+1)<<8);
	data_p += 2;
	return val;
}

int GetLittleLong(void)
{
	int val = 0;
	val = *data_p;
	val = val + (*(data_p+1)<<8);
	val = val + (*(data_p+2)<<16);
	val = val + (*(data_p+3)<<24);
	data_p += 4;
	return val;
}

void FindNextChunk(char *name)
{
	int iff_chunk_len;
	unsigned dataleft;

	while (1)
	{
		data_p=last_chunk;
		dataleft = iff_end - data_p;

		if (dataleft < 8)
		{	// didn't find the chunk
			data_p = NULL;
			return;
		}

		data_p += 4;
		iff_chunk_len = GetLittleLong();
		dataleft-= 8;
		if (iff_chunk_len < 0 || iff_chunk_len > dataleft)
		{
			data_p = NULL;
			return;
		}
		dataleft-= iff_chunk_len;
		data_p-= 8;
		last_chunk = data_p + 8 + iff_chunk_len;
		if ((iff_chunk_len&1) && dataleft)
			last_chunk++;
		if (!strncmp(data_p, name, 4))
			return;
	}
}

void FindChunk(char *name)
{
	last_chunk = iff_data;
	FindNextChunk (name);
}

/*
============
GetWavinfo
============
*/
wavinfo_t GetWavinfo (char *name, byte *wav, int wavlength)
{
	wavinfo_t	info;
	int     i;
	int     format;
	int		samples;

	memset (&info, 0, sizeof(info));

	if (!wav)
		return info;

	iff_data = wav;
	iff_end = wav + wavlength;

// find "RIFF" chunk
	FindChunk("RIFF");
	if (!(data_p && !strncmp(data_p+8, "WAVE", 4)))
	{
		Com_Printf ("Missing RIFF/WAVE chunks\n");
		return info;
	}

// get "fmt " chunk
	iff_data = data_p + 12;

	FindChunk("fmt ");
	if (!data_p)
	{
		Com_Printf ("Missing fmt chunk\n");
		return info;
	}

	data_p += 8;
	format = GetLittleShort();
	if (format != 1)
	{
		Com_Printf ("Microsoft PCM format only\n");
		return info;
	}

	info.channels = GetLittleShort();
	info.rate = GetLittleLong();
	data_p += 4+2;
	info.width = GetLittleShort() / 8;

// get cue chunk
	FindChunk("cue ");
	if (data_p)
	{
		data_p += 32;
		info.loopstart = GetLittleLong();

	// if the next chunk is a LIST chunk, look for a cue length marker
		FindNextChunk ("LIST");
		if (data_p)
		{
			if (!strncmp (data_p + 28, "mark", 4))
			{	// this is not a proper parse, but it works with cooledit...
				data_p += 24;
				i = GetLittleLong ();	// samples in loop
				info.samples = info.loopstart + i;
			}
		}
	}
	else
		info.loopstart = -1;

// find data chunk
	FindChunk("data");
	if (!data_p)
	{
		Com_Printf ("Missing data chunk\n");
		return info;
	}

	data_p += 4;
	samples = GetLittleLong () / info.width;

	if (info.samples)
	{
		if (samples < info.samples)
			Sys_Error ("Sound %s has a bad loop length", name);
	}
	else
		info.samples = samples;

	info.dataofs = data_p - wav;

	return info;
}

