/*
Copyright (C) 2002 Quake done Quick

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// CaptureHelpers.c -- AVI capturing

#include "quakedef.h"
#include "CaptureHelpers.h"
#include "ICapture.h"

extern	cvar_t		host_framerate;
extern	float		scr_con_current;
extern	qboolean	scr_drawloading;
extern	short		*snd_out;
extern	int		snd_linear_count, soundtime;

// Variables for buffering audio
short	capture_audio_samples[44100]; // big enough buffer for 1fps at 44100KHz
int	captured_audio_samples;

// Code in engine is written relatively independent of kind of capture.
// When wired to AVI as currently, capture_codec is a fourcc.
// "0" indicates no compression codec.
cvar_t	capture_codec	= {"capture_codec", "0", true};
cvar_t	capture_fps	= {"capture_fps", "30.0", true};


qboolean CaptureHelper_IsActive (void)
{
	// don't output whilst console is down or 'loading' is displayed
	if (scr_con_current > 0)
		return false;
	if (scr_drawloading)
		return false;

	// otherwise output if a file is open to write to
	return Capture_IsRecording ();
}


void CaptureHelper_Start_f (void)
{
	char	filename[MAX_FILELENGTH];
	char	path[256];
	FILE	*f;

	if (Cmd_Argc() != 2)
	{
		Con_Printf ("capture_start <filename>: Start capturing to named file.\n");
		return;
	}

	Cvar_SetValue ("host_framerate", 1.0 / capture_fps.value);

	Q_strcpy (filename, Cmd_Argv(1));
	COM_ForceExtension (filename, Capture_DOTEXTENSION); // currently we capture AVI

	// Q2K4-0020 change start
	// Q_strcpy (path, va("%s/capture/%s", com_basedir, filename));
	Q_strcpy (path, va("%s/capture/%s", com_gamedir, filename));
	// Q2K4-0020 change end

	if (!(f = fopen (path, "wb")))
	{
		COM_CreatePath (path);
		if (!(f = fopen (path, "wb")))
			Sys_Error ("Error opening %s: %s", path, strerror(errno));
	}
	// Q2K4-0020 change start
	// Capture_Open (va("%s/capture/%s", com_basedir, filename), 
	Capture_Open (va("%s/capture/%s", com_gamedir, filename), 
	// Q2K4-0020 change end

		(*capture_codec.string != '0') ? capture_codec.string : 0, capture_fps.value, shm->speed);
}

void CaptureHelper_Stop (void)
{
	Capture_Close ();
	Cvar_SetValue ("host_framerate", 0);
}

void CaptureHelper_Stop_f (void)
{
	if (!CaptureHelper_IsActive())
	{
		Con_Printf ("Not capturing\n");
		return;
	}

	CaptureHelper_Stop ();

	Con_Printf ("Stopped capturing\n");
}

void CaptureHelper_CaptureDemo_f (void)
{
	if (Cmd_Argc() != 2)
	{
		Con_Printf ("capturedemo <demoname> : capture <demoname>.dem then exit\n");
		return;
	}

	Con_Printf ("Capturing %s.dem\n", Cmd_Argv(1));

	CL_PlayDemo_f ();
	if (!cls.demoplayback)
		return;

	CaptureHelper_Start_f ();

	cls.capturedemo = true;
}


void CaptureHelper_Init (void)
{
	captured_audio_samples = 0;

	Cmd_AddCommand ("capture_start", CaptureHelper_Start_f);
	Cmd_AddCommand ("capture_stop", CaptureHelper_Stop_f);
	Cmd_AddCommand ("capturedemo", CaptureHelper_CaptureDemo_f);
	Cvar_RegisterVariable (&capture_codec);
	Cvar_RegisterVariable (&capture_fps);
}


void CaptureHelper_OnStopPlayback (void)
{
	if (!cls.capturedemo)
		return;

	cls.capturedemo = false;
	CaptureHelper_Stop ();
	CL_Disconnect ();
//	Host_ShutdownServer (false);
//	Sys_Quit ();
}


void CaptureHelper_OnUpdateScreen (void)
{
	int		i;
	byte	temp;
	byte	*buffer;
	int		size = glwidth * glheight * 3;

	if (!CaptureHelper_IsActive())
		return;

	buffer = Q_Malloc (size);
	glReadPixels (glx, gly, glwidth, glheight, GL_RGB, GL_UNSIGNED_BYTE, buffer);
	// Q2K4-0020 change start
	// ApplyGamma (buffer, size);
	// Q2K4-0020 change end

	for (i=0 ; i<size ; i+=3)
	{
		temp = buffer[i];
		buffer[i] = buffer[i+2];
		buffer[i+2] = temp;
	}

	if (!Capture_WriteVideo (glwidth, glheight, buffer))
		Con_Print ("Problem capturing video frame!\n");

	free (buffer);
}

void CaptureHelper_OnTransferStereo16 (void)
{
	if (!CaptureHelper_IsActive())
		return;

	// Copy last audio chunk written into our temporary buffer
	Q_memcpy (capture_audio_samples + 2 * captured_audio_samples, snd_out, snd_linear_count * shm->channels);
	captured_audio_samples += snd_linear_count / 2;

	if (captured_audio_samples >= (int)(1.0 + host_framerate.value * shm->speed))
	{
		// We have enough audio samples to match one frame of video
		if (!Capture_WriteAudio (captured_audio_samples, (unsigned char *)capture_audio_samples))
			Con_Print ("Problem capturing audio!\n");
		captured_audio_samples = 0;
	}
}

qboolean CaptureHelper_GetSoundtime (void)
{
	if (CaptureHelper_IsActive())
	{
		soundtime += (int)(1.0 + host_framerate.value * shm->speed);
		return true;
	}
	else
		return false;
}

