/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// draw.c -- this is the only file outside the refresh that touches the
// vid buffer

#include "quakedef.h"

// Q2K4-0042 start
#include "png.h"
// Q2K4-0042 end

// Q2K4-0081 start
#include "io.h"
// Q2K4-0081 end

// Q2K4-0016 start
extern byte* LoadTgaTexture(char *filename, int width, int height);
// Q2K4-0016 end

#define GL_COLOR_INDEX8_EXT     0x80E5

// Q2K4-0086 start
#define	GL_ADD_SIGNED			0x8574
#define FILTER_SIZE				5
#define BLUR_FILTER				0
#define LIGHT_BLUR				1
#define EDGE_FILTER				2
#define EMBOSS_FILTER			3

float FilterMatrix[][FILTER_SIZE][FILTER_SIZE] =
{
   // regular blur
   {
      {0, 0, 0, 0, 0},
      {0, 1, 1, 1, 0},
      {0, 1, 1, 1, 0},
      {0, 1, 1, 1, 0},
      {0, 0, 0, 0, 0},
   },
   // light blur
   {
      {0, 0, 0, 0, 0},
      {0, 1, 1, 1, 0},
      {0, 1, 4, 1, 0},
      {0, 1, 1, 1, 0},
      {0, 0, 0, 0, 0},
   },
   // find edges
   {
      {0,  0,  0,  0, 0},
      {0, -1, -1, -1, 0},
      {0, -1,  8, -1, 0},
      {0, -1, -1, -1, 0},
      {0,  0,  0,  0, 0},
   },
   // emboss
   {
      {-1, -1, -1, -1, 0},
      {-1, -1, -1,  0, 1},
      {-1, -1,  0,  1, 1},
      {-1,  0,  1,  1, 1},
      { 0,  1,  1,  1, 1},
   }
};

// Q2K4-0086 end


extern unsigned char d_15to8table[65536];

cvar_t		gl_nobind = {"gl_nobind", "0"};
cvar_t		gl_max_size = {"gl_max_size", "1024"};
cvar_t		gl_picmip = {"gl_picmip", "0"};

byte		*draw_chars;				// 8*8 graphic characters
qpic_t		*draw_disc;
qpic_t		*draw_backtile;

int			translate_texture;
int			char_texture;

// Q2K4-0078 start
int			char_texture2;
qboolean	force_alpha = false;	// HACK HACK HACK
// Q2K4-0078 end

typedef struct
{
	int		texnum;
	float	sl, tl, sh, th;
} glpic_t;

byte		conback_buffer[sizeof(qpic_t) + sizeof(glpic_t)];
qpic_t		*conback = (qpic_t *)&conback_buffer;

int		gl_lightmap_format = 4;
int		gl_solid_format = 3;
int		gl_alpha_format = 4;

int		gl_filter_min = GL_LINEAR_MIPMAP_NEAREST;
int		gl_filter_max = GL_LINEAR;


int		texels;

typedef struct
{
	int		texnum;
	char	identifier[64];
	int		width, height;
	qboolean	mipmap;
// Q2K4-0016 start
// Tomaz || TGA Begin
	int			bytesperpixel;
	int			lhcsum;
// Tomaz || TGA End
// Q2K4-0016 end
} gltexture_t;

// Q2K4-0062 start
// #define	MAX_GLTEXTURES	1024
#define	MAX_GLTEXTURES	2048
// Q2K4-0062 end
gltexture_t	gltextures[MAX_GLTEXTURES];
int			numgltextures;


// Q2K4-0021 start
static byte		intensitytable[256];
// Q2K4-0021 end


// Q2K4-0078 start
qboolean		hasConchar2 = false;

// Q2K4-0086 start
/*
==================
R_FilterTexture

Applies a 5 x 5 filtering matrix to the texture, then runs it through a simulated OpenGL texture environment
blend with the original data to derive a new texture.  Freaky, funky, and *f--king* *fantastic*.  You can do
reasonable enough "fake bumpmapping" with this baby...

Filtering algorithm from http://www.student.kuleuven.ac.be/~m0216922/CG/filtering.html
All credit due
==================
*/
void R_FilterTexture (int filterindex, unsigned int *data, int width, int height, float factor, float bias, qboolean greyscale, GLenum GLBlendOperator)
{
   int i;
   int x;
   int y;
   int filterX;
   int filterY;
   unsigned int *temp;

   // allocate a temp buffer
   temp = malloc (width * height * 4);
   for (x = 0; x < width; x++)
   {
      for (y = 0; y < height; y++)
      {
// Q2K4-0093 start
         // float rgbFloat[3] = {0, 0, 0};
         float rgbFloat[4] = {0, 0, 0, 0};
// Q2K4-0093 end
         for (filterX = 0; filterX < FILTER_SIZE; filterX++)
         {
            for (filterY = 0; filterY < FILTER_SIZE; filterY++)
            {
               int imageX = (x - (FILTER_SIZE / 2) + filterX + width) % width;
               int imageY = (y - (FILTER_SIZE / 2) + filterY + height) % height;

               // casting's a unary operation anyway, so the othermost set of brackets in the left part
               // of the rvalue should not be necessary... but i'm paranoid when it comes to C...
               rgbFloat[0] += ((float) ((byte *) &data[imageY * width + imageX])[0]) * FilterMatrix[filterindex][filterX][filterY];
               rgbFloat[1] += ((float) ((byte *) &data[imageY * width + imageX])[1]) * FilterMatrix[filterindex][filterX][filterY];
               rgbFloat[2] += ((float) ((byte *) &data[imageY * width + imageX])[2]) * FilterMatrix[filterindex][filterX][filterY];
            }
         }

         // multiply by factor, add bias, and clamp
         for (i = 0; i < 3; i++)
         {
            rgbFloat[i] *= factor;
            rgbFloat[i] += bias;

            if (rgbFloat[i] < 0) rgbFloat[i] = 0;
            if (rgbFloat[i] > 255) rgbFloat[i] = 255;
         }

         if (greyscale)
         {
            // NTSC greyscale conversion standard
            float avg = (rgbFloat[0] * 30 + rgbFloat[1] * 59 + rgbFloat[2] * 11) / 100;

            // divide by 255 so GL operations work as expected
            rgbFloat[0] = avg / 255.0;
            rgbFloat[1] = avg / 255.0;
            rgbFloat[2] = avg / 255.0;
            rgbFloat[3] = 255.0;
         }

         // write to temp - first, write data in (to get the alpha channel quickly and
         // easily, which will be left well alone by this particular operation...!)
         temp[y * width + x] = data[y * width + x];

         // now write in each element, applying the blend operator.  blend
         // operators are based on standard OpenGL TexEnv modes, and the
         // formulae are derived from the OpenGL specs (http://www.opengl.org).
         for (i = 0; i < 3; i++)
         {
            // divide by 255 so GL operations work as expected
            float TempTarget;
            float SrcData = ((float) ((byte *) &data[y * width + x])[i]) / 255.0;

            switch (GLBlendOperator)
            {
            case GL_ADD:
               TempTarget = rgbFloat[i] + SrcData;
               break;

            case GL_BLEND:
               // default is FUNC_ADD here
               // CsS + CdD works out as Src * Dst * 2
               TempTarget = rgbFloat[i] * SrcData * 2.0;
               break;

            case GL_DECAL:
// Q2K4-0093 start
               // same as GL_REPLACE unless there's alpha, which we ignore for this
				if (rgbFloat[3])
				{
					TempTarget = rgbFloat[i];
				}

				break;
// Q2K4-0093 end
            case GL_REPLACE:
               TempTarget = rgbFloat[i];
               break;

            case GL_ADD_SIGNED:
               TempTarget = (rgbFloat[i] + SrcData) - 0.5;
               break;

            case GL_MODULATE:
               // same as default
            default:
               TempTarget = rgbFloat[i] * SrcData;
               break;
            }

            // multiply back by 255 to get the proper byte scale
            TempTarget *= 255.0;

            // bound the temp target again now, cos the operation may have thrown it out
            if (TempTarget < 0) TempTarget = 0;
            if (TempTarget > 255) TempTarget = 255;

            // and copy it in
            ((byte *) &temp[y * width + x])[i] = (byte) TempTarget;
         }
      }
   }

   // copy temp back to data
   for (i = 0; i < (width * height); i++)
   {
      data[i] = temp[i];
   }

   // release the temp buffer
   free (temp);
}
// Q2K4-0086 end

void GL_Bind (int texnum)
{
	if (gl_nobind.value)
		texnum = char_texture;
	if (currenttexture == texnum)
		return;
	currenttexture = texnum;
#ifdef _WIN32
	bindTexFunc (GL_TEXTURE_2D, texnum);
#else
	glBindTexture(GL_TEXTURE_2D, texnum);
#endif
}


/*
=============================================================================

  scrap allocation

  Allocate all the little status bar obejcts into a single texture
  to crutch up stupid hardware / drivers

=============================================================================
*/

#define	MAX_SCRAPS		2
#define	BLOCK_WIDTH		256
#define	BLOCK_HEIGHT	256

int			scrap_allocated[MAX_SCRAPS][BLOCK_WIDTH];
byte		scrap_texels[MAX_SCRAPS][BLOCK_WIDTH*BLOCK_HEIGHT*4];
qboolean	scrap_dirty;
int			scrap_texnum;



// returns a texture number and the position inside it
int Scrap_AllocBlock (int w, int h, int *x, int *y)
{
	int		i, j;
	int		best, best2;
	int		bestx;
	int		texnum;

	for (texnum=0 ; texnum<MAX_SCRAPS ; texnum++)
	{
		best = BLOCK_HEIGHT;

		for (i=0 ; i<BLOCK_WIDTH-w ; i++)
		{
			best2 = 0;

			for (j=0 ; j<w ; j++)
			{
				if (scrap_allocated[texnum][i+j] >= best)
					break;
				if (scrap_allocated[texnum][i+j] > best2)
					best2 = scrap_allocated[texnum][i+j];
			}
			if (j == w)
			{	// this is a valid spot
				*x = i;
				*y = best = best2;
			}
		}

		if (best + h > BLOCK_HEIGHT)
			continue;

		for (i=0 ; i<w ; i++)
			scrap_allocated[texnum][*x + i] = best + h;

		return texnum;
	}

	Sys_Error ("Scrap_AllocBlock: full");
// Q2K4-0003 start
// SHUT UP, COMPILER!!!
	return(-1);
// Q2K4-0003 end
}

int	scrap_uploads;

void Scrap_Upload (void)
{
	int		texnum;

	scrap_uploads++;

	for (texnum=0 ; texnum<MAX_SCRAPS ; texnum++) {
		GL_Bind(scrap_texnum + texnum);
		GL_Upload8 (scrap_texels[texnum], BLOCK_WIDTH, BLOCK_HEIGHT, false, true);
	}
	scrap_dirty = false;
}

//=============================================================================
/* Support Routines */

typedef struct cachepic_s
{
	char		name[MAX_QPATH];
	qpic_t		pic;
	byte		padding[32];	// for appended glpic
} cachepic_t;

#define	MAX_CACHED_PICS		128

cachepic_t	menu_cachepics[MAX_CACHED_PICS];
int			menu_numcachepics;

byte		menuplyr_pixels[4096];

int		pic_texels;
int		pic_count;

qpic_t *Draw_PicFromWad (char *name)
{
	qpic_t	*p;
	glpic_t	*gl;

	p = W_GetLumpName (name);
	gl = (glpic_t *)p->data;

	// load little ones into the scrap
	if (p->width < 64 && p->height < 64)
	{
		int		x, y;
		int		i, j, k;
		int		texnum;

		texnum = Scrap_AllocBlock (p->width, p->height, &x, &y);
		scrap_dirty = true;
		k = 0;
		for (i=0 ; i<p->height ; i++)
			for (j=0 ; j<p->width ; j++, k++)
				scrap_texels[texnum][(y+i)*BLOCK_WIDTH + x + j] = p->data[k];
		texnum += scrap_texnum;
		gl->texnum = texnum;
		gl->sl = (x+0.01)/(float)BLOCK_WIDTH;
		gl->sh = (x+p->width-0.01)/(float)BLOCK_WIDTH;
		gl->tl = (y+0.01)/(float)BLOCK_WIDTH;
		gl->th = (y+p->height-0.01)/(float)BLOCK_WIDTH;

		pic_count++;
		pic_texels += p->width*p->height;
	}
	else
	{
		gl->texnum = GL_LoadPicTexture (p);
		gl->sl = 0;
		gl->sh = 1;
		gl->tl = 0;
		gl->th = 1;
	}
	return p;
}


/*
================
Draw_CachePic
================
*/
qpic_t	*Draw_CachePic (char *path)
{
	cachepic_t	*pic;
	int			i;
	qpic_t		*dat;
	glpic_t		*gl;


	for (pic=menu_cachepics, i=0 ; i<menu_numcachepics ; pic++, i++)
		if (!strcmp (path, pic->name))
		{

			return &pic->pic;
		}

	if (menu_numcachepics == MAX_CACHED_PICS)
	{

		Sys_Error ("menu_numcachepics == MAX_CACHED_PICS");
	}
	menu_numcachepics++;
	strcpy (pic->name, path);

//
// load the pic from disk
//

	dat = (qpic_t *)COM_LoadTempFile (path);


	if (!dat)
	{

		Sys_Error ("Draw_CachePic: failed to load %s", path);
	}

	SwapPic (dat);


	// HACK HACK HACK --- we need to keep the bytes for
	// the translatable player picture just for the menu
	// configuration dialog
	if (!strcmp (path, "gfx/menuplyr.lmp"))
		memcpy (menuplyr_pixels, dat->data, dat->width*dat->height);

	pic->pic.width = dat->width;
	pic->pic.height = dat->height;
	gl = (glpic_t *)pic->pic.data;

	gl->texnum = GL_LoadPicTexture (dat);

	gl->sl = 0;
	gl->sh = 1;
	gl->tl = 0;
	gl->th = 1;


	return &pic->pic;
}


void Draw_CharToConback (int num, byte *dest)
{
	int		row, col;
	byte	*source;
	int		drawline;
	int		x;

	row = num>>4;
	col = num&15;
	source = draw_chars + (row<<10) + (col<<3);

	drawline = 8;

	while (drawline--)
	{
		for (x=0 ; x<8 ; x++)
			if (source[x] != 255)
				dest[x] = 0x60 + source[x];
		source += 128;
		dest += 320;
	}

}

typedef struct
{
	char *name;
	int	minimize, maximize;
} glmode_t;

glmode_t modes[] = {
	{"GL_NEAREST", GL_NEAREST, GL_NEAREST},
	{"GL_LINEAR", GL_LINEAR, GL_LINEAR},
	{"GL_NEAREST_MIPMAP_NEAREST", GL_NEAREST_MIPMAP_NEAREST, GL_NEAREST},
	{"GL_LINEAR_MIPMAP_NEAREST", GL_LINEAR_MIPMAP_NEAREST, GL_LINEAR},
	{"GL_NEAREST_MIPMAP_LINEAR", GL_NEAREST_MIPMAP_LINEAR, GL_NEAREST},
	{"GL_LINEAR_MIPMAP_LINEAR", GL_LINEAR_MIPMAP_LINEAR, GL_LINEAR}
};

/*
===============
Draw_TextureMode_f
===============
*/
void Draw_TextureMode_f (void)
{
	int		i;
	gltexture_t	*glt;

	if (Cmd_Argc() == 1)
	{
		for (i=0 ; i< 6 ; i++)
			if (gl_filter_min == modes[i].minimize)
			{
				Con_Printf ("%s\n", modes[i].name);
				return;
			}
		Con_Printf ("current filter is unknown???\n");
		return;
	}

	for (i=0 ; i< 6 ; i++)
	{
		if (!Q_strcasecmp (modes[i].name, Cmd_Argv(1) ) )
			break;
	}
	if (i == 6)
	{
		Con_Printf ("bad filter name\n");
		return;
	}

	gl_filter_min = modes[i].minimize;
	gl_filter_max = modes[i].maximize;

	// change all the existing mipmap texture objects
	for (i=0, glt=gltextures ; i<numgltextures ; i++, glt++)
	{
		if (glt->mipmap)
		{
			GL_Bind (glt->texnum);
			glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, gl_filter_min);
			glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, gl_filter_max);
		}
	}
}

/*
===============
Draw_Init
===============
*/
void Draw_Init (void)
{
	int		i;
	qpic_t	*cb;
	byte	*dest, *src;
	int		x, y;
	char	ver[40];
	glpic_t	*gl;
	int		start;
	byte	*ncdata;
	int		f, fstep;

	Cvar_RegisterVariable (&gl_nobind);
	Cvar_RegisterVariable (&gl_max_size);
	Cvar_RegisterVariable (&gl_picmip);

// Q2K4-0021 start
	Cvar_RegisterVariable(&gl_intensity);
// Q2K4-0021 end

	// 3dfx can only handle 256 wide textures
	if (!Q_strncasecmp ((char *)gl_renderer, "3dfx",4) ||
		strstr((char *)gl_renderer, "Glide"))
		Cvar_Set ("gl_max_size", "256");

	Cmd_AddCommand ("gl_texturemode", &Draw_TextureMode_f);

	// load the console background and the charset
	// by hand, because we need to write the version
	// string into the background before turning
	// it into a texture
	draw_chars = W_GetLumpName ("conchars");
	for (i=0 ; i<256*64 ; i++)
		if (draw_chars[i] == 0)
			draw_chars[i] = 255;	// proper transparent color

	// now turn them into textures
// Q2K4-0016 start
//	char_texture = GL_LoadTexture ("charset", 128, 128, draw_chars, false, true);

	char_texture = loadtextureimage ("charset", 0, 0, false, false);
	if (char_texture == 0)// did not find a matching TGA...
		char_texture = GL_LoadTexture ("charset", 128, 128, draw_chars, false, true, 1);

// Q2K4-0016 end

// Q2K4-0078 start
	force_alpha = true;
	char_texture2 = loadtextureimage ("conchars2", 0, 0, false, false);
	force_alpha = false;
	hasConchar2 = !(char_texture2 == 0);
// Q2K4-0078 end

	start = Hunk_LowMark();

	cb = (qpic_t *)COM_LoadTempFile ("gfx/conback.lmp");
	if (!cb)
		Sys_Error ("Couldn't load gfx/conback.lmp");
	SwapPic (cb);

	// hack the version number directly into the pic
// Q2K4 start
/*
#if defined(__linux__)
	sprintf (ver, "(Linux %2.2f, gl %4.2f) %4.2f", (float)LINUX_VERSION, (float)GLQUAKE_VERSION, (float)VERSION);
#else
	sprintf (ver, "(gl %4.2f) %4.2f", (float)GLQUAKE_VERSION, (float)VERSION);
#endif
*/
	sprintf (ver, "%s [%d.%02d]", ENGINE_NAME, ENGINE_VERSION, ENGINE_RELEASE);
// Q2K4 end
	dest = cb->data + 320*186 + 320 - 11 - 8*strlen(ver);
	y = strlen(ver);
	for (x=0 ; x<y ; x++)
		Draw_CharToConback (ver[x], dest+(x<<3));

#if 0
	conback->width = vid.conwidth;
	conback->height = vid.conheight;

 	// scale console to vid size
 	dest = ncdata = Hunk_AllocName(vid.conwidth * vid.conheight, "conback");

 	for (y=0 ; y<vid.conheight ; y++, dest += vid.conwidth)
 	{
 		src = cb->data + cb->width * (y*cb->height/vid.conheight);
 		if (vid.conwidth == cb->width)
 			memcpy (dest, src, vid.conwidth);
 		else
 		{
 			f = 0;
 			fstep = cb->width*0x10000/vid.conwidth;
 			for (x=0 ; x<vid.conwidth ; x+=4)
 			{
 				dest[x] = src[f>>16];
 				f += fstep;
 				dest[x+1] = src[f>>16];
 				f += fstep;
 				dest[x+2] = src[f>>16];
 				f += fstep;
 				dest[x+3] = src[f>>16];
 				f += fstep;
 			}
 		}
 	}
#else
	conback->width = cb->width;
	conback->height = cb->height;
	ncdata = cb->data;
#endif

	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

	gl = (glpic_t *)conback->data;
// Q2K4-0016 start
//	gl->texnum = GL_LoadTexture ("conback", conback->width, conback->height, ncdata, false, false);

	gl->texnum = loadtextureimage ("conback", 0, 0, false, false);
	if (gl->texnum == 0)// did not find a matching TGA...
		gl->texnum = GL_LoadTexture ("conback", conback->width, conback->height, ncdata, false, false, 1);

// Q2K4-0016 end
	gl->sl = 0;
	gl->sh = 1;
	gl->tl = 0;
	gl->th = 1;
	conback->width = vid.width;
	conback->height = vid.height;

	// free loaded console
	Hunk_FreeToLowMark(start);

	// save a texture slot for translated picture
	translate_texture = texture_extension_number++;

	// save slots for scraps
	scrap_texnum = texture_extension_number;
	texture_extension_number += MAX_SCRAPS;

	//
	// get the other pics we need
	//
	draw_disc = Draw_PicFromWad ("disc");
	draw_backtile = Draw_PicFromWad ("backtile");

// Q2K4-0021 start
	for (i=0 ; i<256 ; i++)
	{
		f = gl_intensity.value * i;
		if (f > 255)
		{
			f = 255;
		}

		intensitytable[i] = f;
	}
// Q2K4-0021 end

}

// Q2K4-0078 start

/**
 * Draw_Char16
 * Pretty much the same as Draw_Character() but with a 16x16 font
 **/
 void Draw_Char16 (int x, int y, int num)
 {
	int				row, col;
	float			frow, fcol, size;

	if (!hasConchar2)
	{
		return;
	}

	if (num == 32)
		return;		// space

	num &= 255;

	if (y <= -8)
		return;			// totally off screen

	row = num>>4;
	col = num&15;

	frow = row*0.0625;
	fcol = col*0.0625;
	size = 0.0625;

	GL_Bind (char_texture2);

	glBegin (GL_QUADS);
	glTexCoord2f (fcol, frow);
	glVertex2f (x, y);
	glTexCoord2f (fcol + size, frow);
	glVertex2f (x+16, y);
	glTexCoord2f (fcol + size, frow + size);
	glVertex2f (x+16, y+16);
	glTexCoord2f (fcol, frow + size);
	glVertex2f (x, y+16);
	glEnd ();
 }

/**
 * You can choose
 * the charset (normal or alt),
 * background style (none, iron bar or rusted metal) and
 * text orientation (horizontal or vertical)
 **/
void Draw_Str16 (int x, int y, char *str, int bgstyle, qboolean altfont, qboolean vertical)
{
	int	c, bgcount = 0;

	if (!hasConchar2)
	{
		return;
	}

	while (*str)
	{
		c = (int) *str;
		str++;
		c = c & 127;
		if ((altfont) && (c != 32))
		{
			c += 128;
		}

		switch (bgstyle)
		{
			case	BG_IRONBAR:
				if (!bgcount)
				{
					if (vertical)
					{
						Draw_Char16(x, y, 4);
					}
					else
					{
						Draw_Char16(x, y, 0);
					}

					bgcount++;
				}
				else if (!*str)
				{
					if (vertical)
					{
						Draw_Char16(x, y, 6);
					}
					else
					{
						Draw_Char16(x, y, 2);
					}
				}
				else
				{
					if (vertical)
					{
						Draw_Char16(x, y, 5);
					}
					else
					{
						Draw_Char16(x, y, 1);
					}
				}
			break;
			case	BG_RUSTED:
				Draw_Char16(x, y, 12 + (bgcount & 3));
				bgcount++;
			break;
			case	BG_RUSTSHAD:
				if ((!vertical) || (!bgcount))
				{
					Draw_Char16(x, y, 8 + (bgcount & 3));
				}
				else
				{
					Draw_Char16(x, y, 12 + (bgcount & 3));
				}

				bgcount++;
			break;
			case	BG_STONEHI:
				if (!bgcount)
				{
					if (vertical)
					{
						Draw_Char16(x, y, 4);
					}
					else
					{
						Draw_Char16(x, y, 16);
					}

					bgcount++;
				}
				else if (!*str)
				{
					if (vertical)
					{
						Draw_Char16(x, y, 6);
					}
					else
					{
						Draw_Char16(x, y, 18);
					}
				}
				else
				{
					if (vertical)
					{
						Draw_Char16(x, y, 5);
					}
					else
					{
						Draw_Char16(x, y, 17);
					}
				}
			break;
		}

		Draw_Char16 (x, y, c);
		if (vertical)
		{
			y += 16;
		}
		else
		{
			x += 16;
		}
	}
}

/**
 * Same as Draw_Str16, but using 8 x 8 pixel font
 **/
void Draw_Str8 (int x, int y, char *str, qboolean altfont, qboolean vertical)
{
	int	c;
	while (*str)
	{
		c = (int) *str;
		str++;
		c = c & 127;
		if ((altfont) && (c != 32))
		{
			c += 128;
		}


		Draw_Character (x, y, c);
		if (vertical)
		{
			y += 8;
		}
		else
		{
			x += 8;
		}
	}
}

// Q2K4-0078 end
/*
================
Draw_Character

Draws one 8*8 graphics character with 0 being transparent.
It can be clipped to the top of the screen to allow the console to be
smoothly scrolled off.
================
*/
void Draw_Character (int x, int y, int num)
{
	byte			*dest;
	byte			*source;
	unsigned short	*pusdest;
	int				drawline;
	int				row, col;
	float			frow, fcol, size;

	if (num == 32)
		return;		// space

	num &= 255;

	if (y <= -8)
		return;			// totally off screen

	row = num>>4;
	col = num&15;

	frow = row*0.0625;
	fcol = col*0.0625;
	size = 0.0625;

	GL_Bind (char_texture);

	glBegin (GL_QUADS);
	glTexCoord2f (fcol, frow);
	glVertex2f (x, y);
	glTexCoord2f (fcol + size, frow);
	glVertex2f (x+8, y);
	glTexCoord2f (fcol + size, frow + size);
	glVertex2f (x+8, y+8);
	glTexCoord2f (fcol, frow + size);
	glVertex2f (x, y+8);
	glEnd ();
}

/*
================
Draw_String
================
*/
void Draw_String (int x, int y, char *str)
{
	while (*str)
	{
		Draw_Character (x, y, *str);
		str++;
		x += 8;
	}
}

/*
================
Draw_DebugChar

Draws a single character directly to the upper right corner of the screen.
This is for debugging lockups by drawing different chars in different parts
of the code.
================
*/
void Draw_DebugChar (char num)
{
}

/*
=============
Draw_AlphaPic
=============
*/
void Draw_AlphaPic (int x, int y, qpic_t *pic, float alpha)
{
	byte			*dest, *source;
	unsigned short	*pusdest;
	int				v, u;
	glpic_t			*gl;

	if (scrap_dirty)
		Scrap_Upload ();
	gl = (glpic_t *)pic->data;
	glDisable(GL_ALPHA_TEST);
	glEnable (GL_BLEND);
//	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
//	glCullFace(GL_FRONT);
	glColor4f (1,1,1,alpha);
	GL_Bind (gl->texnum);
	glBegin (GL_QUADS);
	glTexCoord2f (gl->sl, gl->tl);
	glVertex2f (x, y);
	glTexCoord2f (gl->sh, gl->tl);
	glVertex2f (x+pic->width, y);
	glTexCoord2f (gl->sh, gl->th);
	glVertex2f (x+pic->width, y+pic->height);
	glTexCoord2f (gl->sl, gl->th);
	glVertex2f (x, y+pic->height);
	glEnd ();
	glColor4f (1,1,1,1);
	glEnable(GL_ALPHA_TEST);
	glDisable (GL_BLEND);
}


/*
=============
Draw_Pic
=============
*/
void Draw_Pic (int x, int y, qpic_t *pic)
{
	byte			*dest, *source;
	unsigned short	*pusdest;
	int				v, u;
	glpic_t			*gl;

	if (scrap_dirty)
		Scrap_Upload ();
	gl = (glpic_t *)pic->data;
	glColor4f (1,1,1,1);
	GL_Bind (gl->texnum);
	glBegin (GL_QUADS);
	glTexCoord2f (gl->sl, gl->tl);
	glVertex2f (x, y);
	glTexCoord2f (gl->sh, gl->tl);
	glVertex2f (x+pic->width, y);
	glTexCoord2f (gl->sh, gl->th);
	glVertex2f (x+pic->width, y+pic->height);
	glTexCoord2f (gl->sl, gl->th);
	glVertex2f (x, y+pic->height);
	glEnd ();
}


/*
=============
Draw_TransPic
=============
*/
void Draw_TransPic (int x, int y, qpic_t *pic)
{
	byte	*dest, *source, tbyte;
	unsigned short	*pusdest;
	int				v, u;

	if (x < 0 || (unsigned)(x + pic->width) > vid.width || y < 0 ||
		 (unsigned)(y + pic->height) > vid.height)
	{
		Sys_Error ("Draw_TransPic: bad coordinates");
	}

	Draw_Pic (x, y, pic);
}


/*
=============
Draw_TransPicTranslate

Only used for the player color selection menu
=============
*/
void Draw_TransPicTranslate (int x, int y, qpic_t *pic, byte *translation)
{
	int				v, u, c;
	unsigned		trans[64*64], *dest;
	byte			*src;
	int				p;

	GL_Bind (translate_texture);

	c = pic->width * pic->height;

	dest = trans;
	for (v=0 ; v<64 ; v++, dest += 64)
	{
		src = &menuplyr_pixels[ ((v*pic->height)>>6) *pic->width];
		for (u=0 ; u<64 ; u++)
		{
			p = src[(u*pic->width)>>6];
			if (p == 255)
				dest[u] = p;
			else
				dest[u] =  d_8to24table[translation[p]];
		}
	}

	glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, 64, 64, 0, GL_RGBA, GL_UNSIGNED_BYTE, trans);

	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	glColor3f (1,1,1);
	glBegin (GL_QUADS);
	glTexCoord2f (0, 0);
	glVertex2f (x, y);
	glTexCoord2f (1, 0);
	glVertex2f (x+pic->width, y);
	glTexCoord2f (1, 1);
	glVertex2f (x+pic->width, y+pic->height);
	glTexCoord2f (0, 1);
	glVertex2f (x, y+pic->height);
	glEnd ();
}


/*
================
Draw_ConsoleBackground

================
*/
void Draw_ConsoleBackground (int lines)
{
	int y = (vid.height * 3) >> 2;

	if (lines > y)
		Draw_Pic(0, lines - vid.height, conback);
	else
		Draw_AlphaPic (0, lines - vid.height, conback, (float)(1.2 * lines)/y);
}


/*
=============
Draw_TileClear

This repeats a 64*64 tile graphic to fill the screen around a sized down
refresh window.
=============
*/
void Draw_TileClear (int x, int y, int w, int h)
{
	glColor3f (1,1,1);
	GL_Bind (*(int *)draw_backtile->data);
	glBegin (GL_QUADS);
	glTexCoord2f (x/64.0, y/64.0);
	glVertex2f (x, y);
	glTexCoord2f ( (x+w)/64.0, y/64.0);
	glVertex2f (x+w, y);
	glTexCoord2f ( (x+w)/64.0, (y+h)/64.0);
	glVertex2f (x+w, y+h);
	glTexCoord2f ( x/64.0, (y+h)/64.0 );
	glVertex2f (x, y+h);
	glEnd ();
}


/*
=============
Draw_Fill

Fills a box of pixels with a single color
=============
*/
void Draw_Fill (int x, int y, int w, int h, int c)
{
	glDisable (GL_TEXTURE_2D);
	glColor3f (host_basepal[c*3]/255.0,
		host_basepal[c*3+1]/255.0,
		host_basepal[c*3+2]/255.0);

	glBegin (GL_QUADS);

	glVertex2f (x,y);
	glVertex2f (x+w, y);
	glVertex2f (x+w, y+h);
	glVertex2f (x, y+h);

	glEnd ();
	glColor3f (1,1,1);
	glEnable (GL_TEXTURE_2D);
}
//=============================================================================

/*
================
Draw_FadeScreen

================
*/
void Draw_FadeScreen (void)
{
	glEnable (GL_BLEND);
	glDisable (GL_TEXTURE_2D);
// Q2K4 - 0011 start
//	glColor4f (0, 0, 0, 0.8);
	glColor4f (0, 0, 0, 0.4);
// Q2K4 - 0011 end
	glBegin (GL_QUADS);

	glVertex2f (0,0);
	glVertex2f (vid.width, 0);
	glVertex2f (vid.width, vid.height);
	glVertex2f (0, vid.height);

	glEnd ();
	glColor4f (1,1,1,1);
// Q2K4 - 0010 start
    // NATAS - BramBo - enable again
    glEnable(GL_ALPHA_TEST);
    // END
// Q2K4 - 0011 end
	glEnable (GL_TEXTURE_2D);
	glDisable (GL_BLEND);

	Sbar_Changed();
}

//=============================================================================

/*
================
Draw_BeginDisc

Draws the little blue disc in the corner of the screen.
Call before beginning any disc IO.
================
*/
void Draw_BeginDisc (void)
{
	if (!draw_disc)
		return;
	glDrawBuffer  (GL_FRONT);
	Draw_Pic (vid.width - 24, 0, draw_disc);
	glDrawBuffer  (GL_BACK);
}


/*
================
Draw_EndDisc

Erases the disc icon.
Call after completing any disc IO
================
*/
void Draw_EndDisc (void)
{
}

/*
================
GL_Set2D

Setup as if the screen was 320*200
================
*/
void GL_Set2D (void)
{
	glViewport (glx, gly, glwidth, glheight);

	glMatrixMode(GL_PROJECTION);
    glLoadIdentity ();
	glOrtho  (0, vid.width, vid.height, 0, -99999, 99999);

	glMatrixMode(GL_MODELVIEW);
    glLoadIdentity ();

	glDisable (GL_DEPTH_TEST);
	glDisable (GL_CULL_FACE);
	glDisable (GL_BLEND);
	glEnable (GL_ALPHA_TEST);
//	glDisable (GL_ALPHA_TEST);

	glColor4f (1,1,1,1);
}

//====================================================================

/*
================
GL_FindTexture
================
*/
int GL_FindTexture (char *identifier)
{
	int		i;
	gltexture_t	*glt;

	for (i=0, glt=gltextures ; i<numgltextures ; i++, glt++)
	{
		if (!strcmp (identifier, glt->identifier))
			return gltextures[i].texnum;
	}

	return -1;
}

/*
================
GL_ResampleTexture
================
*/
void GL_ResampleTexture (unsigned *in, int inwidth, int inheight, unsigned *out,  int outwidth, int outheight)
{
	int		i, j;
	unsigned	*inrow;
	unsigned	frac, fracstep;

	fracstep = inwidth*0x10000/outwidth;
	for (i=0 ; i<outheight ; i++, out += outwidth)
	{
		inrow = in + inwidth*(i*inheight/outheight);
		frac = fracstep >> 1;
		for (j=0 ; j<outwidth ; j+=4)
		{
			out[j] = inrow[frac>>16];
			frac += fracstep;
			out[j+1] = inrow[frac>>16];
			frac += fracstep;
			out[j+2] = inrow[frac>>16];
			frac += fracstep;
			out[j+3] = inrow[frac>>16];
			frac += fracstep;
		}
	}
}

/*
================
GL_Resample8BitTexture -- JACK
================
*/
void GL_Resample8BitTexture (unsigned char *in, int inwidth, int inheight, unsigned char *out,  int outwidth, int outheight)
{
	int		i, j;
	unsigned	char *inrow;
	unsigned	frac, fracstep;

	fracstep = inwidth*0x10000/outwidth;
	for (i=0 ; i<outheight ; i++, out += outwidth)
	{
		inrow = in + inwidth*(i*inheight/outheight);
		frac = fracstep >> 1;
		for (j=0 ; j<outwidth ; j+=4)
		{
			out[j] = inrow[frac>>16];
			frac += fracstep;
			out[j+1] = inrow[frac>>16];
			frac += fracstep;
			out[j+2] = inrow[frac>>16];
			frac += fracstep;
			out[j+3] = inrow[frac>>16];
			frac += fracstep;
		}
	}
}


/*
================
GL_MipMap

Operates in place, quartering the size of the texture
================
*/
void GL_MipMap (byte *in, int width, int height)
{
	int		i, j;
	byte	*out;

	width <<=2;
	height >>= 1;
	out = in;
	for (i=0 ; i<height ; i++, in+=width)
	{
		for (j=0 ; j<width ; j+=8, out+=4, in+=8)
		{
			out[0] = (in[0] + in[4] + in[width+0] + in[width+4])>>2;
			out[1] = (in[1] + in[5] + in[width+1] + in[width+5])>>2;
			out[2] = (in[2] + in[6] + in[width+2] + in[width+6])>>2;
			out[3] = (in[3] + in[7] + in[width+3] + in[width+7])>>2;
		}
	}
}

/*
================
GL_MipMap8Bit

Mipping for 8 bit textures
================
*/
void GL_MipMap8Bit (byte *in, int width, int height)
{
	int		i, j;
	unsigned short     r,g,b;
	byte	*out, *at1, *at2, *at3, *at4;

//	width <<=2;
	height >>= 1;
	out = in;
	for (i=0 ; i<height ; i++, in+=width)
	{
		for (j=0 ; j<width ; j+=2, out+=1, in+=2)
		{
			at1 = (byte *) (d_8to24table + in[0]);
			at2 = (byte *) (d_8to24table + in[1]);
			at3 = (byte *) (d_8to24table + in[width+0]);
			at4 = (byte *) (d_8to24table + in[width+1]);

 			r = (at1[0]+at2[0]+at3[0]+at4[0]); r>>=5;
 			g = (at1[1]+at2[1]+at3[1]+at4[1]); g>>=5;
 			b = (at1[2]+at2[2]+at3[2]+at4[2]); b>>=5;

			out[0] = d_15to8table[(r<<0) + (g<<5) + (b<<10)];
		}
	}
}

// Q2K4-0021 start
/*
================
GL_LightScaleTexture

Scale up the pixel values in a texture to increase the
lighting range
================
*/
void GL_LightScaleTexture (unsigned *in, int inwidth, int inheight, qboolean only_gamma )
{
	if ( only_gamma )
	{
		int		i, c;
		byte	*p;

		p = (byte *)in;

		c = inwidth*inheight;
		for (i=0 ; i<c ; i++, p+=4)
		{
			p[0] = gammatable[p[0]];
			p[1] = gammatable[p[1]];
			p[2] = gammatable[p[2]];
		}
	}
	else
	{
		int		i, c;
		byte	*p;

		p = (byte *)in;

		c = inwidth*inheight;
		for (i=0 ; i<c ; i++, p+=4)
		{
			p[0] = gammatable[intensitytable[p[0]]];
			p[1] = gammatable[intensitytable[p[1]]];
			p[2] = gammatable[intensitytable[p[2]]];
		}
	}
}
// Q2K4-0021 end

// Q2K4-0095 start

/*
===============
ColorDiff

Get the difference between 2 colours.  This is accomplished by first converting to
greyscale, then comparing and subtracting.  There are better ways of doing this, but
greyscaling gives the most appropriate visual results for Quake textures.  The
alpha component is generally ignored.
===============
*/
unsigned int ColorDiff (unsigned int p1, unsigned int p2)
{
	byte *rgba1 = (byte *) &p1;
	byte *rgba2 = (byte *) &p2;

	int avg1;
	int avg2;

	// NTSC greyscale conversion standard
	avg1 = ((int) rgba1[0] * 30 + (int) rgba1[1] * 59 + (int) rgba1[2] * 11) / 300;
	avg2 = ((int) rgba2[0] * 30 + (int) rgba2[1] * 59 + (int) rgba2[2] * 11) / 300;

	if (avg1 > avg2)
		return avg1 - avg2;
	else return avg2 - avg1;
}

/*
===============
ColorAvg

Get the average of 2 colours, including alpha component
===============
*/
unsigned int ColorAvg (unsigned int p1, unsigned int p2)
{
	byte *rgba1 = (byte *) &p1;
	byte *rgba2 = (byte *) &p2;
	byte *rgbaavg;
	unsigned int avg;

	rgbaavg = (byte *) &avg;

	rgbaavg[0] = (byte) (((int) rgba1[0] + (int) rgba2[0]) / 2);
	rgbaavg[1] = (byte) (((int) rgba1[1] + (int) rgba2[1]) / 2);
	rgbaavg[2] = (byte) (((int) rgba1[2] + (int) rgba2[2]) / 2);
	rgbaavg[3] = (byte) (((int) rgba1[3] + (int) rgba2[3]) / 2);

	return avg;
}

/*
===============
ChooseLerp

Choose an interpolation function based on the lowest difference (i.e. which colour is
nearest to the original pixel): "colormin" is a confusing name, really...
===============
*/
unsigned int ChooseLerp (unsigned int me, unsigned int row, unsigned int col, unsigned int diag)
{
	unsigned int d1;
	unsigned int d2;
	unsigned int d3;
	unsigned int d4;
	unsigned int colormin;

	d1 = ColorDiff (me, row);
	d2 = ColorDiff (me, col);
	d3 = ColorDiff (me, diag);
	d4 = ColorDiff (row, col);

	colormin = d1;

	if (colormin > d2) colormin = d2;
	if (colormin > d3) colormin = d3;
	if (colormin > d4) colormin = d4;

	if (colormin == d1)
		return ColorAvg (me, row);
	else if (colormin == d2)
		return ColorAvg (me, col);
	else if (colormin == d3)
		return ColorAvg (me, diag);
	else return ColorAvg (me, ColorAvg (row, col));
}


/*
=================================
GL_InterpolatePixels

directional average scaling algorithm from http://www.compuphase.com/graphic/scale2.htm

this is edge-based, so it's very good at bringing out detail in complex textures when viewed
close up.  gives sharper results than bilinear or bicubic, without the "spotty" appearance of
my own algorithm (removed).  it's tolerant of multiple passes too.

the paper at the url above discusses use of bit interleaving to determine colour differences.
i've abandoned that as it gives undesirable results with quake textures (likely the fault of
quake textures rather than anything in the paper).  instead i'm using option 2: converting to
greyscale.  the speed implications mentioned in this paper don't really apply to quake as the 
calculations don't occur at run time and the bottleneck is elsewhere (OpenGL texture uploading).
=================================
*/
void GL_InterpolatePixels (unsigned int *scaled, int w, int h, qboolean alpha)
{
	int hcount;
	int wcount;

	int x;

	unsigned *temp = (unsigned *) malloc (sizeof (unsigned) * w * h * 4);

	// row up, col left, row down, col right, same row, same col
	int pos[6];

	int avg[4];	// rgba

	for (hcount = 0; hcount < h; hcount++)
	{
		// determine the positions of the 8 surrounding pixels - part one: rows
		// row up
		pos[0] = hcount - 1;
		if (pos[0] < 0) pos[0] += h;

		// row down
		pos[2] = hcount + 1;
		if (pos[2] >= h) pos[2] -= h;

		// same row
		pos[4] = hcount;

		for (wcount = 0; wcount < w; wcount++)
		{
			// determine the positions of the 8 surrounding pixels - part two: columns
			// col left
			pos[1] = wcount - 1;
			if (pos[1] < 0) pos[1] += w;

			// col right
			pos[3] = wcount + 1;
			if (pos[3] >= w) pos[3] -= w;

			// same col
			pos[5] = wcount;

			// each pixel in scaled maps to 4 pixels in temp
			temp[hcount * 2 * w * 2 + wcount * 2] = ChooseLerp (scaled[pos[4] * w + pos[5]], scaled[pos[0] * w + pos[5]], scaled[pos[4] * w + pos[1]], scaled[pos[0] * w + pos[1]]);
			temp[hcount * 2 * w * 2 + wcount * 2 + 1] = ChooseLerp (scaled[pos[4] * w + pos[5]], scaled[pos[0] * w + pos[5]], scaled[pos[4] * w + pos[3]], scaled[pos[0] * w + pos[3]]);
			temp[(hcount * 2 + 1) * w * 2 + wcount * 2] = ChooseLerp (scaled[pos[4] * w + pos[5]], scaled[pos[2] * w + pos[5]], scaled[pos[4] * w + pos[1]], scaled[pos[2] * w + pos[1]]);
			temp[(hcount * 2 + 1) * w * 2 + wcount * 2 + 1] = ChooseLerp (scaled[pos[4] * w + pos[5]], scaled[pos[2] * w + pos[5]], scaled[pos[4] * w + pos[3]], scaled[pos[2] * w + pos[3]]);
		}
	}

	// copy temp back to scaled
	for (hcount = 0, x = 0; hcount < h * 2; hcount++)
	{
		for (wcount = 0; wcount < w * 2; wcount++, x++)
		{
			scaled[x] = temp[hcount * (w * 2) + wcount];
		}
	}

	free (temp);
}

qboolean dbltexsize = false;

void	checkDblTexSize (char *filename)
{
	dbltexsize = false;

	if (hasValue (script, "filters", "dblsize", filename) != -1)
	{
		dbltexsize = true;
	}
}

// Q2K4-0095 end

/*
===============
GL_Upload32
===============
*/
void GL_Upload32 (unsigned *data, int width, int height, qboolean mipmap, qboolean alpha)
{
// Q2K4-0021 start
	int				samples;
	unsigned 		*scaled;
	int				scaled_width, scaled_height;
	int				upload_width, upload_height;
	int				gl_tex_solid_format = 3;
	int				gl_tex_alpha_format = 4;
// Q2K4-0021 end
	int				i, c;
	byte			*scan;
	int				comp;
// Q2K4-0086 start
	int				filter = -1;
// Q2K4-0086 end
	int				max_size;

	c = width*height;
	samples = gl_solid_format;
	if (alpha)
	{
		// Q2K4-0078 start
		// HACK HACK HACK
		if (force_alpha)
		{
			scan = ((byte *)data);
			for (i=0; i<c ; i++, scan += 4)
			{
				if (!scan[0] && !scan[1] && !scan[2])
				{
					scan[3] = 0;
				}
			}
		}

		scan = ((byte *)data) + 3;
		// Q2K4-0078 end
		for (i=0 ; i<c ; i++, scan += 4)
		{
			if ( *scan != 255 )
			{
				samples = gl_alpha_format;
				break;
			}
		}
	}

	// Q2K4-0078 start
	// HACK HACK HACK
	if (force_alpha)
	{
		comp = gl_tex_alpha_format;
		force_alpha = false;
	}
	else
	{
		comp = (samples == gl_solid_format) ? gl_tex_solid_format : gl_tex_alpha_format;
	}
	// Q2K4-0078 end

	{
		// Q2K4-0033 start
		int powers_of_two[11] = {8,16,32,64,128,256,512,1024,2048,4096,4096};
		// Q2K4-0033 end
		int i;
		// Q2K4-0021 change start
		glGetIntegerv(GL_MAX_TEXTURE_SIZE,&max_size);
		// Q2K4-0021 change end
		scaled_width = scaled_height = 0;

		// Q2K4-0033 start
		for (i=0; i<10; i++)
		// Q2K4-0033 end
		{
			// Q2K4 FIXME: with a texture of 4096x4096 this will crash
			// because will try to access powers_of_two[10] (== NULL)
			// change line above to:
			// int powers_of_two[11] = {8,16,32,64,128,256,512,1024,2048,4096,4096};
			// UPDATE: fix applied
			if (width >= powers_of_two[i] && width < powers_of_two[i+1])
			{
				if (width > ((powers_of_two[i] + powers_of_two[i+1])/2))
					scaled_width = powers_of_two[i+1];
				else
					scaled_width = powers_of_two[i];
			}
			else if (width == powers_of_two[i+1])
			{
				scaled_width = powers_of_two[i+1];
			}
			if (scaled_width && scaled_height)
				break;

			if (height >= powers_of_two[i] && height < powers_of_two[i+1])
			{
				if (height > ((powers_of_two[i] + powers_of_two[i+1])/2))
					scaled_height = powers_of_two[i+1];
				else
					scaled_height = powers_of_two[i];
			}
			else if (height == powers_of_two[i+1])
			{
				scaled_height = powers_of_two[i+1];
			}

			if (scaled_width && scaled_height)
				break;
		}

		// Q2K4-0064 start
		if (width < powers_of_two[0])
		{
			scaled_width = powers_of_two[0];
		}

		if (height < powers_of_two[0])
		{
			scaled_height = powers_of_two[0];
		}
		// Q2K4-0064 end

		if (scaled_width > max_size)
			scaled_width = max_size;

		if (scaled_height > max_size)
			scaled_height = max_size;
	}

	if (scaled_width != width || scaled_height != height)
	{
		// Q2K4-0095 start
		if (dbltexsize)
		{
			scaled=malloc((scaled_width * scaled_height) * 4 * sizeof(unsigned));
		}
		else
		{
			scaled=malloc((scaled_width * scaled_height) * sizeof(unsigned));
		}
		// Q2K4-0095 end

		GL_ResampleTexture(data,width,height,scaled,scaled_width,scaled_height);
	}
	else
	{
		scaled_width=width;
		scaled_height=height;
		scaled=data;
	}

// Q2K4-0095 start
	if ((dbltexsize) && (scaled_width <= max_size / 2) && (scaled_height <= max_size / 2))
	{
//		logActive = true;
//		Sys_DebugLog ("q2k4.log", "Original size: %dx%d", scaled_width, scaled_height);
		GL_InterpolatePixels (scaled, scaled_width, scaled_height, alpha);
		scaled_width *= 2;
		scaled_height *= 2;
//		Sys_DebugLog ("q2k4.log", "Resized to: %dx%d", scaled_width, scaled_height);
//		logActive = false;
	}
// Q2K4-0095 end

// Q2K4-0086 start
	if (mipmap)
	{
		if (hasValue (script, "filters", "edge", gltextures[numgltextures - 1].identifier) != -1)
		{
			filter = EDGE_FILTER;
		}
		else if (hasValue (script, "filters", "emboss", gltextures[numgltextures - 1].identifier) != -1)
		{
			filter = EMBOSS_FILTER;
		}
		else if (hasValue (script, "filters", "blur", gltextures[numgltextures - 1].identifier) != -1)
		{
			filter = BLUR_FILTER;
		}
		else if (hasValue (script, "filters", "light", gltextures[numgltextures - 1].identifier) != -1)
		{
			filter = LIGHT_BLUR;
		}

		if (filter != -1)
		{
			// Q2K4-0095 start
			// R_FilterTexture (filter, data, width, height, 1, 128, true, GL_MODULATE);
			R_FilterTexture (filter, data, scaled_width, scaled_height, 1, 128, true, GL_MODULATE);
			// Q2K4-0095 end
		}
	}
// Q2K4-0086 end


	if (mipmap)
	{
		GL_LightScaleTexture (scaled, scaled_width, scaled_height, !mipmap);
		gluBuild2DMipmaps (GL_TEXTURE_2D, samples, scaled_width, scaled_height, GL_RGBA, GL_UNSIGNED_BYTE, scaled);
	}
	else
	{
		GL_LightScaleTexture (scaled, scaled_width, scaled_height, !mipmap );
// Q2K4-0021 change start
		glTexImage2D (GL_TEXTURE_2D, 0, comp, scaled_width, scaled_height, 0, GL_RGBA, GL_UNSIGNED_BYTE, scaled);
	}

	if (scaled_width != width || scaled_height != height)
		free(scaled);

	upload_width=scaled_width; upload_height = scaled_height;
	if (mipmap)
	{
// Q2K4-0021 change start
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, gl_filter_min);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, gl_filter_max);

// Q2K4-0021 change end
	}
	else
	{
// Q2K4-0021 change start
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, gl_filter_max);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, gl_filter_max);
// Q2K4-0021 change end
	}
// Q2K4-0021 end
}

void GL_Upload8_EXT (byte *data, int width, int height,  qboolean mipmap, qboolean alpha)
{
	int			i, s;
	qboolean	noalpha;
	int			p;
	static unsigned j;
	int			samples;
    static	unsigned char scaled[1024*512];	// [512*256];
	int			scaled_width, scaled_height;

	s = width*height;
	// if there are no transparent pixels, make it a 3 component
	// texture even if it was specified as otherwise
	if (alpha)
	{
		noalpha = true;
		for (i=0 ; i<s ; i++)
		{
			if (data[i] == 255)
				noalpha = false;
		}

		if (alpha && noalpha)
			alpha = false;
	}
	for (scaled_width = 1 ; scaled_width < width ; scaled_width<<=1)
		;
	for (scaled_height = 1 ; scaled_height < height ; scaled_height<<=1)
		;

	scaled_width >>= (int)gl_picmip.value;
	scaled_height >>= (int)gl_picmip.value;

	if (scaled_width > gl_max_size.value)
		scaled_width = gl_max_size.value;
	if (scaled_height > gl_max_size.value)
		scaled_height = gl_max_size.value;

	if (scaled_width * scaled_height > sizeof(scaled))
		Sys_Error ("GL_LoadTexture: too big");

	samples = 1; // alpha ? gl_alpha_format : gl_solid_format;

	texels += scaled_width * scaled_height;

	if (scaled_width == width && scaled_height == height)
	{
		if (!mipmap)
		{
			glTexImage2D (GL_TEXTURE_2D, 0, GL_COLOR_INDEX8_EXT, scaled_width, scaled_height, 0, GL_COLOR_INDEX , GL_UNSIGNED_BYTE, data);
			goto done;
		}
		memcpy (scaled, data, width*height);
	}
	else
		GL_Resample8BitTexture (data, width, height, scaled, scaled_width, scaled_height);

	glTexImage2D (GL_TEXTURE_2D, 0, GL_COLOR_INDEX8_EXT, scaled_width, scaled_height, 0, GL_COLOR_INDEX, GL_UNSIGNED_BYTE, scaled);
	if (mipmap)
	{
		int		miplevel;

		miplevel = 0;
		while (scaled_width > 1 || scaled_height > 1)
		{
			GL_MipMap8Bit ((byte *)scaled, scaled_width, scaled_height);
			scaled_width >>= 1;
			scaled_height >>= 1;
			if (scaled_width < 1)
				scaled_width = 1;
			if (scaled_height < 1)
				scaled_height = 1;
			miplevel++;
			glTexImage2D (GL_TEXTURE_2D, miplevel, GL_COLOR_INDEX8_EXT, scaled_width, scaled_height, 0, GL_COLOR_INDEX, GL_UNSIGNED_BYTE, scaled);
		}
	}
done: ;


	if (mipmap)
	{
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, gl_filter_min);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, gl_filter_max);
	}
	else
	{
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, gl_filter_max);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, gl_filter_max);
	}
}

/*
===============
GL_Upload8
===============
*/
void GL_Upload8 (byte *data, int width, int height,  qboolean mipmap, qboolean alpha)
{
static	unsigned	trans[640*480];		// FIXME, temporary
	int			i, s;
	qboolean	noalpha;
	int			p;

	s = width*height;
	// if there are no transparent pixels, make it a 3 component
	// texture even if it was specified as otherwise


	if (alpha)
	{
		noalpha = true;
		for (i=0 ; i<s ; i++)
		{
			p = data[i];
			if (p == 255)
				noalpha = false;
			trans[i] = d_8to24table[p];
		}

		if (alpha && noalpha)
			alpha = false;
	}
	else
	{
		if (s&3)
		{
			Sys_Error ("GL_Upload8: s&3");
		}

		for (i=0 ; i<s ; i+=4)
		{
			trans[i] = d_8to24table[data[i]];
			trans[i+1] = d_8to24table[data[i+1]];
			trans[i+2] = d_8to24table[data[i+2]];
			trans[i+3] = d_8to24table[data[i+3]];
		}

	}

 	if (VID_Is8bit() && !alpha && (data!=scrap_texels[0])) {
 		GL_Upload8_EXT (data, width, height, mipmap, alpha);
 		return;
	}


	GL_Upload32 (trans, width, height, mipmap, alpha);
}

/*
================
GL_LoadTexture
================
*/
// Q2K4-0016 start
/*
int GL_LoadTexture (char *identifier, int width, int height, byte *data, qboolean mipmap, qboolean alpha)
{
	qboolean	noalpha;
	int			i, p, s;
	gltexture_t	*glt;

	// see if the texture is allready present
	if (identifier[0])
	{
		for (i=0, glt=gltextures ; i<numgltextures ; i++, glt++)
		{
			if (!strcmp (identifier, glt->identifier))
			{
				if (width != glt->width || height != glt->height)
					Sys_Error ("GL_LoadTexture: cache mismatch");
				return gltextures[i].texnum;
			}
		}
	}
	else {
		glt = &gltextures[numgltextures];
		numgltextures++;
	}

	strcpy (glt->identifier, identifier);
	glt->texnum = texture_extension_number;
	glt->width = width;
	glt->height = height;
	glt->mipmap = mipmap;

	GL_Bind(texture_extension_number );

	GL_Upload8 (data, width, height, mipmap, alpha);

	texture_extension_number++;

	return texture_extension_number-1;
}
*/


// Tomaz || TGA Begin
int lhcsumtable[256];
int GL_LoadTexture (char *identifier, int width, int height, byte *data, qboolean mipmap, qboolean alpha, int bytesperpixel)
{
	qboolean	noalpha;
	int			i, p, s, lhcsum;
	gltexture_t	*glt;


	// occurances. well this isn't exactly a checksum, it's better than that but
	// not following any standards.
	lhcsum = 0;
	s = width*height*bytesperpixel;
	for (i = 0;i < 256;i++) lhcsumtable[i] = i + 1;
	for (i = 0;i < s;i++) lhcsum += (lhcsumtable[data[i] & 255]++);

	// see if the texture is allready present
	if (identifier[0])
	{
		for (i=0, glt=gltextures ; i < numgltextures ; i++, glt++)
		{
			if (!strcmp (identifier, glt->identifier))
			{
				if (lhcsum != glt->lhcsum || width != glt->width || height != glt->height)
				{

					Con_DPrintf("GL_LoadTexture: cache mismatch\n");
					goto GL_LoadTexture_setup;
				}

				return glt->texnum;
			}
		}
	}

// Q2K4-0023 start
	// whoever at id or threewave must've been half asleep...
	glt = &gltextures[numgltextures];
	numgltextures++;
// Q2K4-0023 end

	strcpy (glt->identifier, identifier);
	glt->texnum = texture_extension_number;
	texture_extension_number++;
	GL_LoadTexture_setup:

	glt->lhcsum = lhcsum;
	glt->width = width;
	glt->height = height;
	glt->mipmap = mipmap;
	glt->bytesperpixel = bytesperpixel;
	if (!isDedicated)
	{

		GL_Bind(glt->texnum);
		if (bytesperpixel == 1)
		{
			// Q2K4 TEST START
			GL_Upload8 (data, width, height, mipmap, alpha);
			// Q2K4 TEXT END
		}
		else if (bytesperpixel == 4)
		{
			// Q2K4-0078 start
			// alpha channel in ext textures should not be ignored
			// GL_Upload32 ((void *)data, width, height, mipmap, true);
			GL_Upload32 ((void *)data, width, height, mipmap, alpha);

		}
		else
		{
			Sys_Error("GL_LoadTexture: unknown bytesperpixel\n");
		}

		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	}


	return glt->texnum;
}
// Tomaz || TGA End


// Q2K4-0016 end
/*
================
GL_LoadPicTexture
================
*/
int GL_LoadPicTexture (qpic_t *pic)
{
// Q2K4-0016 start
//	return GL_LoadTexture ("", pic->width, pic->height, pic->data, false, true);
	return GL_LoadTexture ("", pic->width, pic->height, pic->data, false, true, 1);
// Q2K4-0016 end
}

/****************************************/
// Q2K4-0059 start
// static GLenum oldtarget = TEXTURE0_SGIS;
static GLenum oldtarget = TEXTURE0_ARB;
// Q2K4-0059 end
void GL_SelectTexture (GLenum target)
{
	if (!gl_mtexable)
		return;
	qglSelectTextureSGIS(target);
	if (target == oldtarget)
		return;
	// cnttextures[oldtarget-TEXTURE0_SGIS] = currenttexture;
	// Q2K4-0059 start
	cnttextures[oldtarget-TEXTURE0_SGIS_ARB] = currenttexture;
	// currenttexture = cnttextures[target-TEXTURE0_SGIS];
	currenttexture = cnttextures[target-TEXTURE0_SGIS_ARB];
	// Q2K4-0059 end
	oldtarget = target;
}

// Q2K4-0016 start
// Tomaz || TGA Begin
int		image_width;
int		image_height;

/*
=============
TARGA LOADING
=============
*/

typedef struct _TargaHeader
{
	unsigned char 	id_length, colormap_type, image_type;
	unsigned short	colormap_index, colormap_length;
	unsigned char	colormap_size;
	unsigned short	x_origin, y_origin, width, height;
	unsigned char	pixel_size, attributes;
} TargaHeader;


TargaHeader		targa_header;

/*
extern int fgetLittleShort (FILE *f);
extern int fgetLittleLong (FILE *f);
*/

int fgetLittleShort (FILE *f)
{
	byte	b1, b2;

	b1 = fgetc(f);
	b2 = fgetc(f);

	return (short)(b1 + b2*256);
}
/*
========
LoadTGA
========
*/
byte* LoadTGATexture (FILE *fin, int matchwidth, int matchheight)
{
	int	columns, rows, numPixels;
	byte	*pixbuf;
	int	row, column;
	byte	*targa_rgba;

	targa_header.id_length = fgetc(fin);
	targa_header.colormap_type = fgetc(fin);
	targa_header.image_type = fgetc(fin);

	targa_header.colormap_index = fgetLittleShort(fin);
	targa_header.colormap_length = fgetLittleShort(fin);
	targa_header.colormap_size = fgetc(fin);
	targa_header.x_origin = fgetLittleShort(fin);
	targa_header.y_origin = fgetLittleShort(fin);
	targa_header.width = fgetLittleShort(fin);
	targa_header.height = fgetLittleShort(fin);
	if (matchwidth && targa_header.width != matchwidth)
	{
		Con_Printf("LoadTGATexture: bad TGA width\n");
		fclose(fin);
		return NULL;
	}

	if (matchheight && targa_header.height != matchheight)
	{
		Con_Printf("LoadTGATexture: bad TGA height\n");
		fclose(fin);
		return NULL;
	}

	targa_header.pixel_size = fgetc(fin);
	targa_header.attributes = fgetc(fin);

	if (targa_header.image_type!=2 && targa_header.image_type!=10)
	{
		fclose(fin);
		Host_Error ("LoadTGAtexture: Only type 2 and 10 targa RGB images supported\n");
	}

	if (targa_header.colormap_type !=0 || (targa_header.pixel_size!=32 && targa_header.pixel_size!=24))
	{
		fclose(fin);
		Host_Error ("LoadTGAtexture: Only 32 or 24 bit images supported (no colormaps)\n");
	}

	columns = targa_header.width;
	rows = targa_header.height;
	numPixels = columns * rows;
	targa_rgba = malloc (numPixels*4);
	if (targa_header.id_length != 0)
		fseek(fin, targa_header.id_length, SEEK_CUR);  // skip TARGA image comment

// ---

	if (targa_header.image_type==2) {  // Uncompressed, RGB images
		for(row=rows-1; row>=0; row--) {
			pixbuf = targa_rgba + row*columns*4;
			for(column=0; column<columns; column++) {
				unsigned char red,green,blue,alphabyte;
				switch (targa_header.pixel_size) {
					case 24:

							blue = getc(fin);
							green = getc(fin);
							red = getc(fin);
							*pixbuf++ = red;
							*pixbuf++ = green;
							*pixbuf++ = blue;
							*pixbuf++ = 255;
							break;
					case 32:
							blue = getc(fin);
							green = getc(fin);
							red = getc(fin);
							alphabyte = getc(fin);
							*pixbuf++ = red;
							*pixbuf++ = green;
							*pixbuf++ = blue;
							*pixbuf++ = alphabyte;
							break;
				}
			}
		}
	}
	else if (targa_header.image_type==10) {   // Runlength encoded RGB images
		unsigned char red,green,blue,alphabyte,packetHeader,packetSize,j;
		for(row=rows-1; row>=0; row--) {
			pixbuf = targa_rgba + row*columns*4;
			for(column=0; column<columns; ) {
				packetHeader=getc(fin);
				packetSize = 1 + (packetHeader & 0x7f);
				if (packetHeader & 0x80) {        // run-length packet
					switch (targa_header.pixel_size) {
						case 24:
								blue = getc(fin);
								green = getc(fin);
								red = getc(fin);
								alphabyte = 255;
								break;
						case 32:
								blue = getc(fin);
								green = getc(fin);
								red = getc(fin);
								alphabyte = getc(fin);
								break;
					}

					for(j=0;j<packetSize;j++) {
						*pixbuf++=red;
						*pixbuf++=green;
						*pixbuf++=blue;
						*pixbuf++=alphabyte;
						column++;
						if (column==columns) { // run spans across rows
							column=0;
							if (row>0)
								row--;
							else
								goto breakOut;
							pixbuf = targa_rgba + row*columns*4;
						}
					}
				}
				else {                            // non run-length packet
					for(j=0;j<packetSize;j++) {
						switch (targa_header.pixel_size) {
							case 24:
									blue = getc(fin);
									green = getc(fin);
									red = getc(fin);
									*pixbuf++ = red;
									*pixbuf++ = green;
									*pixbuf++ = blue;
									*pixbuf++ = 255;
									break;
							case 32:
									blue = getc(fin);
									green = getc(fin);
									red = getc(fin);
									alphabyte = getc(fin);
									*pixbuf++ = red;
									*pixbuf++ = green;
									*pixbuf++ = blue;
									*pixbuf++ = alphabyte;
									break;
						}
						column++;
						if (column==columns) { // pixel packet run spans across rows
							column=0;
							if (row>0)
								row--;
							else
								goto breakOut;
							pixbuf = targa_rgba + row*columns*4;
						}
					}
				}
			}
			breakOut:;
		}
	}


// ---

	fclose(fin);
	image_width = columns;
	image_height = rows;
	return targa_rgba;
}

//Q2K4-0041 start
/**
 * JPEG LOADING
 */

void jpgload_error_exit (j_common_ptr cinfo)
{
	q2k4_error_ptr myerr = (q2k4_error_ptr) cinfo->err;
	Con_Printf("loadJPGTexture(): libjpeg returned \"%s\".", cinfo->err->jpeg_message_table[cinfo->err->msg_code]);
	longjmp(myerr->setjmp_buffer, 1);
}

byte *loadJPGTexture (FILE *fin, int matchwidth, int matchheight)
{
	struct		jpeg_decompress_struct cinfo;
	struct		q2k4_error_mgr jerr;
	JSAMPARRAY	buffer;
	int			row_stride, bufsize, x, s, d;
	byte		*tex = NULL;

	cinfo.err = jpeg_std_error(&jerr.pub);
	jerr.pub.error_exit = jpgload_error_exit;
	if (setjmp(jerr.setjmp_buffer))
	{
		// Something went wrong, abort
		Con_Printf("loadJPGTexture(): jpeglib ABORT\n");
		jpeg_destroy_decompress(&cinfo);
		fclose(fin);
		return(NULL);
	}

	jpeg_create_decompress(&cinfo);
	jpeg_stdio_src(&cinfo, fin);
	jpeg_read_header(&cinfo, TRUE);
	jpeg_start_decompress(&cinfo);
	// only RGB JPEGs are supported (no grayscale support)
	if (cinfo.output_components != 3)
	{
		Con_Printf("loadJPGTexture(): JPEG image has %d components, should be 3\n", cinfo.output_components);
		jpeg_destroy_decompress(&cinfo);
		fclose(fin);
		return(NULL);
	}


	if ((matchwidth) && (cinfo.output_width != matchwidth))
	{
		Con_Printf("loadJPGTexture(): bad JPG width\n");
		jpeg_destroy_decompress(&cinfo);
		fclose(fin);
		return(NULL);
	}


	if ((matchheight) && (matchheight != cinfo.output_height))
	{
		Con_Printf("loadJPGTexture(): bad JPG height\n");
		jpeg_destroy_decompress(&cinfo);
		fclose(fin);
		return(NULL);
	}

	row_stride = cinfo.output_width * cinfo.output_components;
	// we need a RGBA image
	bufsize = cinfo.output_width * cinfo.output_height * 4;
	if ((tex = malloc(bufsize)) == NULL)
	{
		Con_Printf("loadJPGTexture(): failed to alloc image buffer\n");
		jpeg_destroy_decompress(&cinfo);
		fclose(fin);
		return(NULL);
	}

	memset(tex, 0x0, bufsize);
	buffer = (*cinfo.mem->alloc_sarray)((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);
	while (cinfo.output_scanline < cinfo.output_height)
	{
		jpeg_read_scanlines(&cinfo, buffer, 1);
		s = 0;
		d = (cinfo.output_scanline - 1) * (cinfo.output_width * 4);
		for (x = 0; x < cinfo.output_width; x++)
		{
			tex[d++] = buffer[0][s++];
			tex[d++] = buffer[0][s++];
			tex[d++] = buffer[0][s++];
			tex[d++] = (byte)255;
		}
	}

	jpeg_finish_decompress(&cinfo);
	jpeg_destroy_decompress(&cinfo);
	fclose(fin);
	image_width = cinfo.output_width;
	image_height = cinfo.output_height;
	return(tex);
}
// Q2K4-0041 end

// Q2K4-0042 start
/*
=========================================================

  PNG Loading

=========================================================
*/

/*
=============
LoadPNG
=============
*/
byte *loadPNGTexture(FILE *fin, int matchwidth, int matchheight)
{
	byte		header[8], **rowpointers, *data;
	png_structp	png;
	png_infop	pnginfo;
	int		y, width, height, bitdepth, colortype;
	int		interlace, compression, filter, bytesperpixel;
	unsigned long	rowbytes;


	fread (header, 1, 8, fin);
	if (png_sig_cmp(header, 0, 8))
	{
		fclose (fin);
		return NULL;
	}

	if (!(png = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL)))
	{
		fclose (fin);
		return NULL;
	}

	if (!(pnginfo = png_create_info_struct(png)))
	{
		png_destroy_read_struct (&png, &pnginfo, NULL);
		fclose (fin);
		return NULL;
	}

	if (setjmp(png->jmpbuf))
	{
		png_destroy_read_struct (&png, &pnginfo, NULL);
		fclose (fin);
		return NULL;
	}

	png_init_io (png, fin);
	png_set_sig_bytes (png, 8);
	png_read_info (png, pnginfo);
	png_get_IHDR (png, pnginfo, (png_uint_32 *)&width, (png_uint_32 *)&height, &bitdepth, &colortype, &interlace, &compression, &filter);

	if ((matchwidth && width != matchwidth) || (matchheight && height != matchheight))
	{
		png_destroy_read_struct (&png, &pnginfo, NULL);
		fclose (fin);
		return NULL;
	}

	if (colortype == PNG_COLOR_TYPE_PALETTE)
	{
		png_set_palette_to_rgb (png);
		png_set_filler (png, 255, PNG_FILLER_AFTER);
	}

	if (colortype == PNG_COLOR_TYPE_GRAY && bitdepth < 8)
	{
		png_set_gray_1_2_4_to_8 (png);
	}

	if (png_get_valid(png, pnginfo, PNG_INFO_tRNS))
	{
		png_set_tRNS_to_alpha (png);
	}

	if (bitdepth == 8 && colortype == PNG_COLOR_TYPE_RGB)
	{
		png_set_filler (png, 255, PNG_FILLER_AFTER);
	}

	if (colortype == PNG_COLOR_TYPE_GRAY || colortype == PNG_COLOR_TYPE_GRAY_ALPHA)
	{
		png_set_gray_to_rgb (png);
		png_set_filler (png, 255, PNG_FILLER_AFTER);
	}

	if (bitdepth < 8)
		png_set_expand (png);
	else if (bitdepth == 16)
		png_set_strip_16 (png);

	png_read_update_info (png, pnginfo);
	rowbytes = png_get_rowbytes (png, pnginfo);
	bytesperpixel = png_get_channels (png, pnginfo);
	bitdepth = png_get_bit_depth (png, pnginfo);

	if (bitdepth != 8 || bytesperpixel != 4)
	{
		fclose (fin);
		png_destroy_read_struct (&png, &pnginfo, NULL);
		return NULL;
	}

	data = Q_Malloc (height * rowbytes);
	rowpointers = Q_Malloc (height * 4);

	for (y = 0; y < height; y++)
		rowpointers[y] = data + y * rowbytes;

	png_read_image (png, rowpointers);
	png_read_end (png, NULL);

	png_destroy_read_struct (&png, &pnginfo, NULL);
	free (rowpointers);
	fclose (fin);
	image_width = width;
	image_height = height;
	return data;
}
// Q2K4-0042 end


byte* loadimagepixels (char* filename, qboolean complain, int matchwidth, int matchheight)
{
	FILE	*f;
	char	temp[128], basename[128], name[128];
	byte	*image_rgba, *c;

// Q2K4-0081 start
	long	checkFile;
	struct	_finddata_t n_file;
// Q2K4-0081 end

	COM_StripExtension(filename, temp); // strip the extension to allow TGA, JPG or PNG
// QRPG FIX START
// Q2K4-0096 start
	// sprintf(basename, "%s/%s", r_theme.string, temp);
	if (strlen(r_theme.string))
	{
		sprintf(basename, "%s/%s", r_theme.string, temp);
	}
	else
	{
		// allows compatibility with TomazQuake and other engines
		sprintf (basename, "textures/%s", temp);
	}
// Q2K4-0096 end
// QRPG FIX END

// Q2K4-0081 start
// checks if exists any matching file into the theme dir
	sprintf (temp, "%s\\%s.*", com_gamedir, basename);
	checkFile = _findfirst(temp, &n_file);
	if (checkFile == -1)
	{
// not found into the theme dir, looking for into the game basedir
		COM_StripExtension(filename, basename);
		sprintf (temp, "%s\\%s.*", com_gamedir, basename);
		checkFile = _findfirst(temp, &n_file);
		if (checkFile == -1)
		{
			// don't bother looking further, file does not exists
			// CHECK: or should try to load from id1 ?
			return(NULL);
		}
	}

// Q2K4-0081 end

	c = basename;
	while (*c)
	{
		if (*c == '*')
			*c = '+';
		c++;
	}

// Q2K4-0042 start
	sprintf (name, "%s.png", basename);
	COM_FOpenFile (name, &f);
	if (f)
	{
		return loadPNGTexture(f, matchwidth, matchheight);
	}
// Q2K4-0042 end
// Q2K4-0041 start
	sprintf (name, "%s.jpg", basename);
	COM_FOpenFile (name, &f);
	if (f)
	{
		return loadJPGTexture (f, matchwidth, matchheight);
	}

	sprintf (name, "%s.jpeg", basename);
	COM_FOpenFile (name, &f);
	if (f)
	{
		return loadJPGTexture (f, matchwidth, matchheight);
	}
// Q2K4-0041 end
	sprintf (name, "%s.tga", basename);
	COM_FOpenFile (name, &f);
	if (f)
	{
		return LoadTGATexture (f, matchwidth, matchheight);
	}

	if (complain)
		Con_Printf ("Couldn't load texture \"%s\"\n", filename);
	return NULL;
}

int loadtextureimage (char* filename, int matchwidth, int matchheight, qboolean complain, qboolean mipmap)
{
	int texnum;
	byte *data;

	// Q2K4-0095 start
	checkDblTexSize (filename);
	// Q2K4-0095 end
	if (!(data = loadimagepixels (filename, complain, matchwidth, matchheight)))
		return 0;
	texnum = GL_LoadTexture (filename, image_width, image_height, data, mipmap, true, 4);
	free(data);
	return texnum;
}
// Tomaz || TGA End
// Q2K4-0016 end