/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// gl_warp.c -- sky and water polygons

#include "quakedef.h"

extern	model_t	*loadmodel;

// Q2K4-0019 start
// extern	byte* LoadTGATexture (FILE *fin, int matchwidth, int matchheight);
// Q2K4-0019 end
// Q2K4-0041 start
// extern	byte *loadJPGTexture (FILE *fin, int matchwidth, int matchheight);
// Q2K4-0041 end
// Q2K4-0042 start
// extern	byte *loadPNGTexture(FILE *fin, int matchwidth, int matchheight);
// Q2K4-0042 end
int		skytexturenum;

int		solidskytexture;
int		alphaskytexture;
float	speedscale;		// for top sky and bottom sky

msurface_t	*warpface;

extern cvar_t gl_subdivide_size;

// Q2K4-0082 start
float sky_fog[4]; 
int spherelist = -1; 
// Q2K4-0082 end

void BoundPoly (int numverts, float *verts, vec3_t mins, vec3_t maxs)
{
	int		i, j;
	float	*v;

	mins[0] = mins[1] = mins[2] = 9999;
	maxs[0] = maxs[1] = maxs[2] = -9999;
	v = verts;
	for (i=0 ; i<numverts ; i++)
		for (j=0 ; j<3 ; j++, v++)
		{
			if (*v < mins[j])
				mins[j] = *v;
			if (*v > maxs[j])
				maxs[j] = *v;
		}
}

void SubdividePolygon (int numverts, float *verts)
{
	int		i, j, k;
	vec3_t	mins, maxs;
	float	m;
	float	*v;
	vec3_t	front[64], back[64];
	int		f, b;
	float	dist[64];
	float	frac;
	glpoly_t	*poly;
	float	s, t;

	if (numverts > 60)
		Sys_Error ("numverts = %i", numverts);

	BoundPoly (numverts, verts, mins, maxs);

	for (i=0 ; i<3 ; i++)
	{
		m = (mins[i] + maxs[i]) * 0.5;
		m = gl_subdivide_size.value * floor (m/gl_subdivide_size.value + 0.5);
		if (maxs[i] - m < 8)
			continue;
		if (m - mins[i] < 8)
			continue;

		// cut it
		v = verts + i;
		for (j=0 ; j<numverts ; j++, v+= 3)
			dist[j] = *v - m;

		// wrap cases
		dist[j] = dist[0];
		v-=i;
		VectorCopy (verts, v);

		f = b = 0;
		v = verts;
		for (j=0 ; j<numverts ; j++, v+= 3)
		{
			if (dist[j] >= 0)
			{
				VectorCopy (v, front[f]);
				f++;
			}
			if (dist[j] <= 0)
			{
				VectorCopy (v, back[b]);
				b++;
			}
			if (dist[j] == 0 || dist[j+1] == 0)
				continue;
			if ( (dist[j] > 0) != (dist[j+1] > 0) )
			{
				// clip point
				frac = dist[j] / (dist[j] - dist[j+1]);
				for (k=0 ; k<3 ; k++)
					front[f][k] = back[b][k] = v[k] + frac*(v[3+k] - v[k]);
				f++;
				b++;
			}
		}

		SubdividePolygon (f, front[0]);
		SubdividePolygon (b, back[0]);
		return;
	}

	poly = Hunk_Alloc (sizeof(glpoly_t) + (numverts-4) * VERTEXSIZE*sizeof(float));
	poly->next = warpface->polys;
	warpface->polys = poly;
	poly->numverts = numverts;
	for (i=0 ; i<numverts ; i++, verts+= 3)
	{
		VectorCopy (verts, poly->verts[i]);
		s = DotProduct (verts, warpface->texinfo->vecs[0]);
		t = DotProduct (verts, warpface->texinfo->vecs[1]);
		poly->verts[i][3] = s;
		poly->verts[i][4] = t;
	}
}

/*
================
GL_SubdivideSurface

Breaks a polygon up along axial 64 unit
boundaries so that turbulent and sky warps
can be done reasonably.
================
*/
void GL_SubdivideSurface (msurface_t *fa)
{
	vec3_t		verts[64];
	int			numverts;
	int			i;
	int			lindex;
	float		*vec;
	texture_t	*t;

	warpface = fa;

	//
	// convert edges back to a normal polygon
	//
	numverts = 0;
	for (i=0 ; i<fa->numedges ; i++)
	{
		lindex = loadmodel->surfedges[fa->firstedge + i];

		if (lindex > 0)
			vec = loadmodel->vertexes[loadmodel->edges[lindex].v[0]].position;
		else
			vec = loadmodel->vertexes[loadmodel->edges[-lindex].v[1]].position;
		VectorCopy (vec, verts[numverts]);
		numverts++;
	}

	SubdividePolygon (numverts, verts[0]);
}

//=========================================================



// speed up sin calculations - Ed
float	turbsin[] =
{
	#include "gl_warp_sin.h"
};
#define TURBSCALE (256.0 / (2 * M_PI))

/*
=============
EmitWaterPolys

Does a water warp on the pre-fragmented glpoly_t chain
=============
*/
void EmitWaterPolys (msurface_t *fa)
{
	glpoly_t	*p;
	float		*v;
	int			i;
	float		s, t, os, ot;

// Q2K4-0027 start
	vec3_t		nv; 
// Q2K4-0027 end

	for (p=fa->polys ; p ; p=p->next)
	{
		glBegin (GL_POLYGON);
		for (i=0,v=p->verts[0] ; i<p->numverts ; i++, v+=VERTEXSIZE)
		{
			os = v[3];
			ot = v[4];

			s = os + turbsin[(int)((ot*0.125+realtime) * TURBSCALE) & 255];
			s *= (1.0/64);

			t = ot + turbsin[(int)((os*0.125+realtime) * TURBSCALE) & 255];
			t *= (1.0/64);

			glTexCoord2f (s, t);
// Q2K4-0027 start
			if((r_waterwarp.value == 0) && (r_waterripple.value)) 
			{
				nv[0] = v[0];
				nv[1] = v[1];
				//qbism//jf 2000-03-01 adjusted ripple speed/amplitude
				nv[2] = v[2] + r_waterripple.value*sin(v[0]*0.05+realtime*3)*sin(v[2]*0.05+realtime*3);
				glVertex3fv (nv);
			}
			else 
			{
				glVertex3fv (v);
			}
// Q2K4-0027 end
		}
		glEnd ();
	}
}




/*
=============
EmitSkyPolys
=============
*/
void EmitSkyPolys (msurface_t *fa)
{
	glpoly_t	*p;
	float		*v;
	int			i;
	float	s, t;
	vec3_t	dir;
	float	length;

	for (p=fa->polys ; p ; p=p->next)
	{
		glBegin (GL_POLYGON);
		for (i=0,v=p->verts[0] ; i<p->numverts ; i++, v+=VERTEXSIZE)
		{
			VectorSubtract (v, r_origin, dir);
			dir[2] *= 3;	// flatten the sphere

			length = dir[0]*dir[0] + dir[1]*dir[1] + dir[2]*dir[2];
			length = sqrt (length);
			length = 6*63/length;

			dir[0] *= length;
			dir[1] *= length;

			s = (speedscale + dir[0]) * (1.0/128);
			t = (speedscale + dir[1]) * (1.0/128);

			glTexCoord2f (s, t);
			glVertex3fv (v);
		}
		glEnd ();
	}
}

/*
===============
EmitBothSkyLayers

Does a sky warp on the pre-fragmented glpoly_t chain
This will be called for brushmodels, the world
will have them chained together.
===============
*/
void EmitBothSkyLayers (msurface_t *fa)
{
	int			i;
	int			lindex;
	float		*vec;

	GL_DisableMultitexture();

	GL_Bind (solidskytexture);
// Q2K4-0046 start
	// speedscale = realtime*8;
	if ((skyspeed0.value >=0) && (skyspeed0.value <= 32))
	{
		speedscale = realtime * (int)skyspeed0.value;
	}
	else
	{
		speedscale = realtime*8;
	}
// Q2K4-0046 end
// 
	speedscale -= (int)speedscale & ~127 ;

	EmitSkyPolys (fa);

	glEnable (GL_BLEND);
	// Q2K4-0045 start
	glEnable (GL_ALPHA_TEST);
	// Q2K4-0045 end
	GL_Bind (alphaskytexture);
	// Q2K4-0046 start
	// speedscale = realtime*16;
	if ((skyspeed1.value >=0) && (skyspeed1.value <= 32))
	{
		speedscale = realtime * (int)skyspeed1.value;
	}
	else
	{
		speedscale = realtime*16;
	}
	// Q2K4-0046 end

	speedscale -= (int)speedscale & ~127 ;

	EmitSkyPolys (fa);
	// Q2K4-0045 start
	glDisable (GL_ALPHA_TEST);
	// Q2K4-0045 end
	glDisable (GL_BLEND);
}


/*
=================================================================

  Quake 2 environment sky

=================================================================
*/

// #ifdef QUAKE2


#define	SKY_TEX		2000
/*
==================
R_LoadSkys
==================
*/

char	*suf[6] = {"rt", "bk", "lf", "ft", "up", "dn"};
byte	*sky_rgba;
extern	int		image_width;
extern	int		image_height;
void R_LoadSkys (void)
{
	int		i, found;
	FILE	*f;
	char	name[128];
	char	basename[128];
// Q2K4-0043 start
	float	oldskybox = skybox.value;

// by default, turns off skybox rendering (to avoid crashes in case of texture loading failure)
	Cvar_Set("skybox", "0");

// Q2K4-0043 end
	for (i=0 ; i<6 ; i++)
	{
		GL_Bind (SKY_TEX + i);
		found = false;

// Q2K4-0067 start
		sprintf(basename, "env/%s%s", skyname.string, suf[i]);
		sky_rgba = (byte *)loadimagepixels(basename, false, 0, 0);
// Q2K4-0067 end
/*
		sprintf (basename, "%s/env/%s%s", r_theme.string, skyname.string, suf[i]);

		sprintf (name, "%s.png", basename);
		COM_FOpenFile (name, &f);
		if (f)
		{
			sky_rgba = loadPNGTexture (f, 0, 0);
			if (sky_rgba != NULL)
			{
				found = true;
			}
		}

		if (!found)
		{
			sprintf (name, "%s.jpg", basename);
			COM_FOpenFile (name, &f);
			if (f)
			{
				sky_rgba = loadJPGTexture (f, 0, 0);
				if (sky_rgba != NULL)
				{
					found = true;
				}
			}
		}

		if (!found)
		{
			sprintf (name, "%s.jpeg", basename);
			COM_FOpenFile (name, &f);
			if (f)
			{
				sky_rgba = loadJPGTexture (f, 0, 0);
				if (sky_rgba != NULL)
				{
					found = true;
				}
			}
		}

		if (!found)
		{
			sprintf (name, "%s.tga", basename);
			COM_FOpenFile (name, &f);
			if (f)
			{
				sky_rgba = LoadTGATexture(f, 0, 0);
				if (sky_rgba != NULL)
				{
					found = true;
				}
			}
		}
*/
		if (sky_rgba == NULL)
		{
			return;
		}

		glTexImage2D (GL_TEXTURE_2D, 0, gl_solid_format, image_width, image_width, 0, GL_RGBA, GL_UNSIGNED_BYTE, sky_rgba);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		free (sky_rgba);
	}

	Cvar_SetValue("skybox", oldskybox);
}


vec3_t	skyclip[6] = {
	{1,1,0},
	{1,-1,0},
	{0,-1,1},
	{0,1,1},
	{1,0,1},
	{-1,0,1} 
};
int	c_sky;

// 1 = s, 2 = t, 3 = 2048
int	st_to_vec[6][3] =
{
	{3,-1,2},
	{-3,1,2},

	{1,3,2},
	{-1,-3,2},

	{-2,-1,3},		// 0 degrees yaw, look straight up
	{2,-1,-3}		// look straight down

//	{-1,2,3},
//	{1,2,-3}
};

// s = [0]/[2], t = [1]/[2]
int	vec_to_st[6][3] =
{
	{-2,3,1},
	{2,3,-1},

	{1,3,2},
	{-1,3,-2},

	{-2,-1,3},
	{-2,1,-3}

//	{-1,2,3},
//	{1,2,-3}
};

float	skymins[2][6], skymaxs[2][6];

void DrawSkyPolygon (int nump, vec3_t vecs)
{
	int		i,j;
	vec3_t	v, av;
	float	s, t, dv;
	int		axis;
	float	*vp;

	c_sky++;
#if 0
glBegin (GL_POLYGON);
for (i=0 ; i<nump ; i++, vecs+=3)
{
	VectorAdd(vecs, r_origin, v);
	glVertex3fv (v);
}
glEnd();
return;
#endif
	// decide which face it maps to
	VectorCopy (vec3_origin, v);
	for (i=0, vp=vecs ; i<nump ; i++, vp+=3)
	{
		VectorAdd (vp, v, v);
	}
	av[0] = fabs(v[0]);
	av[1] = fabs(v[1]);
	av[2] = fabs(v[2]);
	if (av[0] > av[1] && av[0] > av[2])
	{
		if (v[0] < 0)
			axis = 1;
		else
			axis = 0;
	}
	else if (av[1] > av[2] && av[1] > av[0])
	{
		if (v[1] < 0)
			axis = 3;
		else
			axis = 2;
	}
	else
	{
		if (v[2] < 0)
			axis = 5;
		else
			axis = 4;
	}

	// project new texture coords
	for (i=0 ; i<nump ; i++, vecs+=3)
	{
		j = vec_to_st[axis][2];
		if (j > 0)
			dv = vecs[j - 1];
		else
			dv = -vecs[-j - 1];

		j = vec_to_st[axis][0];
		if (j < 0)
			s = -vecs[-j -1] / dv;
		else
			s = vecs[j-1] / dv;
		j = vec_to_st[axis][1];
		if (j < 0)
			t = -vecs[-j -1] / dv;
		else
			t = vecs[j-1] / dv;

		if (s < skymins[0][axis])
			skymins[0][axis] = s;
		if (t < skymins[1][axis])
			skymins[1][axis] = t;
		if (s > skymaxs[0][axis])
			skymaxs[0][axis] = s;
		if (t > skymaxs[1][axis])
			skymaxs[1][axis] = t;
	}
}

#define	MAX_CLIP_VERTS	64
void ClipSkyPolygon (int nump, vec3_t vecs, int stage)
{
	float	*norm;
	float	*v;
	qboolean	front, back;
	float	d, e;
	float	dists[MAX_CLIP_VERTS];
	int		sides[MAX_CLIP_VERTS];
	vec3_t	newv[2][MAX_CLIP_VERTS];
	int		newc[2];
	int		i, j;

	if (nump > MAX_CLIP_VERTS-2)
		Sys_Error ("ClipSkyPolygon: MAX_CLIP_VERTS");
	if (stage == 6)
	{	// fully clipped, so draw it
		DrawSkyPolygon (nump, vecs);
		return;
	}

	front = back = false;
	norm = skyclip[stage];
	for (i=0, v = vecs ; i<nump ; i++, v+=3)
	{
		d = DotProduct (v, norm);
		if (d > ON_EPSILON)
		{
			front = true;
			sides[i] = SIDE_FRONT;
		}
		else if (d < ON_EPSILON)
		{
			back = true;
			sides[i] = SIDE_BACK;
		}
		else
			sides[i] = SIDE_ON;
		dists[i] = d;
	}

	if (!front || !back)
	{	// not clipped
		ClipSkyPolygon (nump, vecs, stage+1);
		return;
	}

	// clip it
	sides[i] = sides[0];
	dists[i] = dists[0];
	VectorCopy (vecs, (vecs+(i*3)) );
	newc[0] = newc[1] = 0;

	for (i=0, v = vecs ; i<nump ; i++, v+=3)
	{
		switch (sides[i])
		{
		case SIDE_FRONT:
			VectorCopy (v, newv[0][newc[0]]);
			newc[0]++;
			break;
		case SIDE_BACK:
			VectorCopy (v, newv[1][newc[1]]);
			newc[1]++;
			break;
		case SIDE_ON:
			VectorCopy (v, newv[0][newc[0]]);
			newc[0]++;
			VectorCopy (v, newv[1][newc[1]]);
			newc[1]++;
			break;
		}

		if (sides[i] == SIDE_ON || sides[i+1] == SIDE_ON || sides[i+1] == sides[i])
			continue;

		d = dists[i] / (dists[i] - dists[i+1]);
		for (j=0 ; j<3 ; j++)
		{
			e = v[j] + d*(v[j+3] - v[j]);
			newv[0][newc[0]][j] = e;
			newv[1][newc[1]][j] = e;
		}
		newc[0]++;
		newc[1]++;
	}

	// continue
	ClipSkyPolygon (newc[0], newv[0][0], stage+1);
	ClipSkyPolygon (newc[1], newv[1][0], stage+1);
}

// Q2K4-0082 start
void R_DrawSkySphere (msurface_t *s)
{
   float fullscale = NEWGLRENDERLIMIT / 4096; 
   float halfscale = NEWGLRENDERLIMIT / 8192; 
   float reducedfull = (NEWGLRENDERLIMIT / 4096) * 0.9; 
   float reducedhalf = (NEWGLRENDERLIMIT / 8192) * 0.9; 

   // rotation 
   float rotateBack = anglemod (realtime * skyspeed0.value); 
   float rotateFore = anglemod (realtime * skyspeed1.value); 

   // turn on fogging.  fog looks good on the skies - it gives them a more 
   // "airy" far-away look, and has the knock-on effect of preventing the 
   // old "texture distortion at the poles" problem. 
   glFogi (GL_FOG_MODE, GL_LINEAR); 
   glFogfv (GL_FOG_COLOR, sky_fog); 
   glFogf (GL_FOG_START, 5); 

   // must tweak the fog end too!!! 
   glFogf (GL_FOG_END, NEWGLRENDERLIMIT * 0.75); 
   glEnable (GL_FOG); 

   // sky texture scaling 
   // previous releases made a tiled version of the sky texture.  here i just shrink it using the 
   // texture matrix, for much the same effect 
   glMatrixMode (GL_TEXTURE); 
   glLoadIdentity (); 
   glScalef (2, 1, 1); 
   glMatrixMode (GL_MODELVIEW); 

   // background 
   // ========== 
   // go to a new matrix 
   glPushMatrix (); 

   // center it on the players position 
   glTranslatef (r_origin[0], r_origin[1], r_origin[2]); 

   // flatten the sphere 
   glScalef (fullscale, fullscale, halfscale); 

   // orient it so that the poles are unobtrusive 
   glRotatef (-90, 1, 0, 0); 

   // make it not always at right angles to the player 
   glRotatef (-22, 0 ,1, 0); 

   // rotate it around the poles 
   glRotatef (-rotateBack, 0, 0, 1); 

   // solid sky texture 
   glBindTexture (GL_TEXTURE_2D, solidskytexture); 

   // draw the sphere 
   glCallList (spherelist); 

   // restore the previous matrix 
   glPopMatrix (); 

   // foreground 
   // ========== 
   // go to a new matrix 
   glPushMatrix (); 

   // center it on the players position 
   glTranslatef (r_origin[0], r_origin[1], r_origin[2]); 

   // flatten the sphere and shrink it a little - the reduced scale prevents artefacts appearing when 
   // corners on the skysphere may potentially overlap 
   glScalef (reducedfull, reducedfull, reducedhalf); 

   // orient it so that the poles are unobtrusive 
   glRotatef (-90, 1, 0, 0); 

   // make it not always at right angles to the player 
   glRotatef (-22, 0 ,1, 0); 

   // rotate it around the poles 
   glRotatef (-rotateFore, 0, 0, 1); 

   // blend mode 
   glEnable (GL_BLEND); 

   // alpha sky texture 
   glBindTexture (GL_TEXTURE_2D, alphaskytexture); 

   // draw the sphere 
   glCallList (spherelist); 

   // back to normal mode 
   glDisable (GL_BLEND); 

   // restore the previous matrix 
   glPopMatrix (); 

   // turn off fog 
   glDisable (GL_FOG); 

   glMatrixMode (GL_TEXTURE); 
   glLoadIdentity (); 
   glMatrixMode (GL_MODELVIEW); 
}
// Q2K4-0082 end

/*
=================
R_DrawSkyChain
=================
*/
void R_DrawSkyChain (msurface_t *s)
{
// Q2K4-0019 start
msurface_t	*fa;
int		i;
vec3_t	verts[MAX_CLIP_VERTS];
glpoly_t	*p;

if (skybox.value) // if the skybox value is one, draw the skybox

	{
	c_sky = 0;
	GL_Bind(solidskytexture);

	// calculate vertex values for sky box

	for (fa=s ; fa ; fa=fa->texturechain)
	{
		for (p=fa->polys ; p ; p=p->next)
		{
			for (i=0 ; i < p->numverts ; i++)
			{
				VectorSubtract (p->verts[i], r_origin, verts[i]);
			}
			ClipSkyPolygon (p->numverts, verts[0], 0);
		}
	}
	
	}
else // otherwise, draw the normal quake sky
	{
// Q2K4-0082 start
		if (spherelist != -1)
		{
			R_DrawSkySphere (s);
		}
		else
		{
			GL_DisableMultitexture();

			// used when gl_texsort is on
			GL_Bind(solidskytexture);
		// Q2K4-0046 start
			// speedscale = realtime*8;
			if ((skyspeed0.value >=0) && (skyspeed0.value <= 32))
			{
				speedscale = realtime * (int)skyspeed0.value;
			}
			else
			{
				speedscale = realtime*8;
			}
		// Q2K4-0046 end

			speedscale -= (int)speedscale & ~127 ;

			for (fa=s ; fa ; fa=fa->texturechain)
				EmitSkyPolys (fa);

			glEnable (GL_BLEND);
			GL_Bind (alphaskytexture);
			// Q2K4-0046 start
			// speedscale = realtime*16;
			if ((skyspeed1.value >=0) && (skyspeed1.value <= 32))
			{
				speedscale = realtime * (int)skyspeed1.value;
			}
			else
			{
				speedscale = realtime*16;
			}
			// Q2K4-0046 end

			speedscale -= (int)speedscale & ~127 ;

			for (fa=s ; fa ; fa=fa->texturechain)
				EmitSkyPolys (fa);

			glDisable (GL_BLEND);
		}
// Q2K4-0082 end
	}
// Q2K4-0019 end
}


/*
==============
R_ClearSkyBox
==============
*/
void R_ClearSkyBox (void)
{
	int		i;

	for (i=0 ; i<6 ; i++)
	{
// Q2K4-0062 start
//		skymins[0][i] = skymins[1][i] = 9999;
//		skymaxs[0][i] = skymaxs[1][i] = -9999;
		skymins[0][i] = skymins[1][i] = NEWGLRENDERLIMIT + 256;
		skymaxs[0][i] = skymaxs[1][i] = -(NEWGLRENDERLIMIT + 256);
// Q2K4-0062 end
	}
}


void MakeSkyVec (float s, float t, int axis)
{
	vec3_t		v, b;
	int			j, k;

	// Q2K4-0062 start
	/*
	b[0] = s*2048;
	b[1] = t*2048;
	b[2] = 2048;
	*/
	b[0] = s*(NEWGLRENDERLIMIT / 2);
	b[1] = t*(NEWGLRENDERLIMIT / 2);
	b[2] = (NEWGLRENDERLIMIT / 2);
	// Q2K4-0062 end

	for (j=0 ; j<3 ; j++)
	{
		k = st_to_vec[axis][j];
		if (k < 0)
			v[j] = -b[-k - 1];
		else
			v[j] = b[k - 1];
		v[j] += r_origin[j];
	}

	// avoid bilerp seam
	s = (s+1)*0.5;
	t = (t+1)*0.5;

	if (s < 1.0/512)
		s = 1.0/512;
	else if (s > 511.0/512)
		s = 511.0/512;
	if (t < 1.0/512)
		t = 1.0/512;
	else if (t > 511.0/512)
		t = 511.0/512;

	t = 1.0 - t;
	glTexCoord2f (s, t);
	glVertex3fv (v);
}

int	skytexorder[6] = {0,2,1,3,4,5};
void R_DrawSkyBox (void)
{
	int		i, j, k;
	vec3_t	v;
	float	s, t;

// disable drawing fog into sky when skybox is active 
	if (gl_fogenable.value) {
		glDisable(GL_FOG); 
	}
// end

#if 0
glEnable (GL_BLEND);
glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
glColor4f (1,1,1,0.5);
glDisable (GL_DEPTH_TEST);
#endif
	for (i=0 ; i<6 ; i++)
	{
		if (skymins[0][i] >= skymaxs[0][i]
		|| skymins[1][i] >= skymaxs[1][i])
			continue;

		GL_Bind (SKY_TEX+skytexorder[i]);
#if 0
skymins[0][i] = -1;
skymins[1][i] = -1;
skymaxs[0][i] = 1;
skymaxs[1][i] = 1;
#endif
		glBegin (GL_QUADS);
		MakeSkyVec (skymins[0][i], skymins[1][i], i);
		MakeSkyVec (skymins[0][i], skymaxs[1][i], i);
		MakeSkyVec (skymaxs[0][i], skymaxs[1][i], i);
		MakeSkyVec (skymaxs[0][i], skymins[1][i], i);
		glEnd ();
	}
#if 0
glDisable (GL_BLEND);
glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
glColor4f (1,1,1,0.5);
glEnable (GL_DEPTH_TEST);
#endif

	glPopMatrix();

//Re-enable fog drawing again 
	if (gl_fogenable.value) {
		glEnable(GL_FOG); 
	}
// END

}

// Q2K4-0019 end
//===============================================================

// Q2K4-0082 start
/* 
============== 
R_DrawSphere 

Draw a sphere!!!  The verts and texcoords are precalculated for extra efficiency.  The sphere is put into 
a display list to reduce overhead even further. 
============== 
*/ 
float skytexes[440]; 
float skyverts[660]; 


void R_DrawSphere (void) 
{ 
   if (spherelist == -1) 
   { 
      int i; 
      int j; 

      int vertspos = 0; 
      int texespos = 0; 

      // build the sphere display list 
      spherelist = glGenLists (1); 

      glNewList (spherelist, GL_COMPILE); 

      for (i = 0; i < 10; i++) 
      { 
         glBegin (GL_TRIANGLE_STRIP); 

         for (j = 0; j <= 10; j++) 
         { 
            glTexCoord2fv (&skytexes[texespos]); 
            glVertex3fv (&skyverts[vertspos]); 

            texespos += 2; 
            vertspos += 3; 

            glTexCoord2fv (&skytexes[texespos]); 
            glVertex3fv (&skyverts[vertspos]); 

            texespos += 2; 
            vertspos += 3; 
         } 

         glEnd (); 
      } 

      glEndList (); 
   } 
} 


/* 
============== 
R_InitSkyChain 

Initialize the sky chain arrays 
============== 
*/ 
void R_InitSkyChain (void) 
{ 
   float drho = 0.3141592653589; 
   float dtheta = 0.6283185307180; 

   float ds = 0.1; 
   float dt = 0.1; 

   float t = 1.0f; 
   float s = 0.0f; 

   int i; 
   int j; 

   int vertspos = 0; 
   int texespos = 0; 

   for (i = 0; i < 10; i++) 
   { 
      float rho = (float) i * drho; 
      float srho = (float) (sin (rho)); 
      float crho = (float) (cos (rho)); 
      float srhodrho = (float) (sin (rho + drho)); 
      float crhodrho = (float) (cos (rho + drho)); 

      s = 0.0f; 

      for (j = 0; j <= 10; j++) 
      { 
         float theta = (j == 10) ? 0.0f : j * dtheta; 
         float stheta = (float) (-sin( theta)); 
         float ctheta = (float) (cos (theta)); 

         skytexes[texespos++] = s * 2; 
         skytexes[texespos++] = t * 2; 

         skyverts[vertspos++] = stheta * srho * 4096.0; 
         skyverts[vertspos++] = ctheta * srho * 4096.0; 
         skyverts[vertspos++] = crho * 4096.0; 

         skytexes[texespos++] = s * 2; 
         skytexes[texespos++] = (t - dt) * 2; 

         skyverts[vertspos++] = stheta * srhodrho * 4096.0; 
         skyverts[vertspos++] = ctheta * srhodrho * 4096.0; 
         skyverts[vertspos++] = crhodrho * 4096.0; 

         s += ds; 
      } 

      t -= dt; 
   } 
} 
// Q2K4-0082 end

/*
=============
R_InitSky

A sky texture is 256*128, with the right side being a masked overlay
==============
*/
// Q2K4-0045 start

void R_OldInitSky (texture_t *mt)
{
   int         i, j, p; 
   byte      *src; 
   unsigned   transpix; 
   int         r, g, b; 
   unsigned   *rgba; 
   unsigned   topalpha; 
   int         div; 

   unsigned   *trans; 
   trans = malloc (128 * 128 * 4); 
   // ----------------- 
   // solid sky texture 
   // ----------------- 
   src = (byte *)mt + mt->offsets[0]; 

   // make an average value for the back to avoid 
   // a fringe on the top level 
   r = g = b = 0; 

   for (i=0 ; i<128 ; i++) 
   { 
      for (j=0 ; j<128 ; j++) 
      { 
         p = src[i*256 + j + 128]; 
         rgba = &d_8to24table[p]; 
         trans[(i*128) + j] = *rgba; 

         r += ((byte *)rgba)[0]; 
         g += ((byte *)rgba)[1]; 
         b += ((byte *)rgba)[2]; 
      } 
   } 

   ((byte *)&transpix)[0] = r/(128*128); 
   ((byte *)&transpix)[1] = g/(128*128); 
   ((byte *)&transpix)[2] = b/(128*128); 
   ((byte *)&transpix)[3] = 0; 

   if (!solidskytexture) solidskytexture = texture_extension_number++; 

   glBindTexture (GL_TEXTURE_2D, solidskytexture); 

   glTexImage2D (GL_TEXTURE_2D, 0, GL_RGBA, 128, 128, 0, GL_RGBA, GL_UNSIGNED_BYTE, trans); 
   glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR); 
   glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR); 

   // ----------------- 
   // alpha sky texture 
   // ----------------- 
   // get another average cos the bottom layer average can be too dark 
   div = r = g = b = 0; 

   for (i=0 ; i<128 ; i++) 
   { 
      for (j=0 ; j<128 ; j++) 
      { 
         p = src[i*256 + j]; 

         if (p == 0) 
            topalpha = transpix; 
         else 
         { 
            rgba = &d_8to24table[p]; 

            g += ((byte *)rgba)[1]; 

            r += ((byte *)rgba)[0]; 
            b += ((byte *)rgba)[2]; 

            div++; 

            topalpha = d_8to24table[p]; 
         } 

         ((byte *)&topalpha)[3] = ((byte *)&topalpha)[3] / 2; 

         trans[(i*128) + j] = topalpha; 
      } 
   } 

   if (!alphaskytexture) alphaskytexture = texture_extension_number++; 

   glBindTexture (GL_TEXTURE_2D, alphaskytexture); 

   glTexImage2D (GL_TEXTURE_2D, 0, GL_RGBA, 128, 128, 0, GL_RGBA, GL_UNSIGNED_BYTE, trans); 
   glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR); 
   glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR); 

   // get fog colours for the sky - we halve these 
   sky_fog[0] = ((float)r / (float)div) / 512.0; 
   sky_fog[1] = ((float)g / (float)div) / 512.0; 
   sky_fog[2] = ((float)b / (float)div) / 512.0; 
   // sky_fog[3] = 0.1; 
   sky_fog[3] = 0.075; 

   // free the used memory 
   free (trans); 

   // recalc the sphere display list - delete the original one first!!! 
   if (spherelist != -1) glDeleteLists (spherelist, 1); 
   spherelist = -1; 
   R_InitSkyChain (); 
   R_DrawSphere (); 
// Q2K4-0082 end
}

/**
 * Allows to load any sky texture size,
 * as long the proportion is the same (2:1)
 * assumes the pixel at left top is the transparent color
 */
void R_InitSky (texture_t *mt)
{
	byte		*data, *solid, *alpha, *src, *dst, *dst2;
	int			x, y;
	FILE		*fout;
	char		buf[128];

	sprintf(buf, "maps/%s", mt->name);
	if (!(data = (char *)loadimagepixels (buf, false, 0, 0)))
	{
// Q2K4-0068 start
	// force to reload the texture
		solidskytexture = 0;
		alphaskytexture = 0;
// Q2K4-0068 end
		R_OldInitSky(mt);
		return;
	}
	else
	{
		if ((image_width / image_height) != 2)
		{
			Sys_Error("R_InitSky: \"%s\" has bad proportions (%dx%d)", buf, image_width, image_height);
		}

	// Q2K4-0068 start
		solidskytexture = 0;
		alphaskytexture = 0;
	// Q2K4-0068 end
		// break the image in two halves (solid and transparent layer)
		// TODO: could actually be TWO separated images instead (+0sky and +1sky) ?
		solid = malloc((image_width / 2) * image_height * 4);
		alpha = malloc((image_width / 2) * image_height * 4);
		src   = data;

		for (y = 0; y < image_height; y++)
		{
			dst = &alpha[(image_width / 2) *4 * y];
			dst2 = &solid[(image_width / 2) * 4 * y];
			for (x = 0; x < image_width; x++)
			{
				if (x < image_width / 2)
				{
					if ((src[0] == data[0]) && (src[1] == data[1]) && (src[2] == data[2]))
					{
						dst[0] = 0;
						dst[1] = 0;
						dst[2] = 0;
						dst[3] = 0;
					}
					else
					{
						dst[0] = src[0];
						dst[1] = src[1];
						dst[2] = src[2];
						dst[3] = 255;
					}

					dst   += 4;
				}
				else
				{
					dst2[0] = src[0];
					dst2[1] = src[1];
					dst2[2] = src[2];
					dst2[3] = 255;
					dst2   += 4;
				}

				src	+= 4;
			}
		}

		if (!alphaskytexture)
		{
			alphaskytexture = texture_extension_number++;
			GL_Bind(alphaskytexture);
			glTexImage2D (GL_TEXTURE_2D, 0, gl_alpha_format, image_width / 2, image_height, 0, GL_RGBA, GL_UNSIGNED_BYTE, alpha);
			glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
			glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		}

		if (!solidskytexture)
		{
			solidskytexture = texture_extension_number++;
			GL_Bind (solidskytexture );
			glTexImage2D (GL_TEXTURE_2D, 0, gl_solid_format, image_width / 2, image_height, 0, GL_RGBA, GL_UNSIGNED_BYTE, solid);
			glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
			glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		}
	}
}

// Q2K4-0045 end
