/*
 * Copyright (C) 1996-1997 Id Software, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */
// vid_glx.c -- Linux GLX driver

#include <termios.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/vt.h>
#include <stdarg.h>
#include <stdio.h>
#include <signal.h>

#include <dlfcn.h>

#include "quakedef.h"

#include <GL/glx.h>
#include <X11/cursorfont.h>

#ifdef HAVE_DGA
#  include <X11/extensions/xf86dga.h>
static qboolean dgamouse = false, dgakeyb = false;
#endif

#ifdef HAVE_VMODE
#  include <X11/extensions/xf86vmode.h>
static XF86VidModeModeInfo **vidmodes;
static int    num_vidmodes;

unsigned short *currentgammaramp = NULL;
static unsigned short systemgammaramp[3][256];

qboolean      vid_gammaworks = false;
qboolean      customgamma = false;

static qboolean vidmode_ext = false;
static qboolean vidmode_active = false;
#endif
qboolean      vid_hwgamma_enabled = false;

#define	WARP_WIDTH	320
#define	WARP_HEIGHT	200

//static Display *dpy = NULL;
static Window win;
static GLXContext ctx = NULL;

static float  old_windowed_mouse = 0, mouse_x, mouse_y, old_mouse_x, old_mouse_y;

#define KEY_MASK (KeyPressMask | KeyReleaseMask)
#define MOUSE_MASK (ButtonPressMask | ButtonReleaseMask | PointerMotionMask)
#define X_MASK (KEY_MASK | MOUSE_MASK | VisibilityChangeMask)

qboolean      fullsbardraw = false;
qboolean      fullscreen = false;

qboolean      gl_have_stencil = false;

static int    scr_width, scr_height, scrnum;

cvar_t        vid_mode = { "vid_mode", "0" };
qboolean      OnChange_windowed_mouse (cvar_t * var, char *value);
cvar_t        _windowed_mouse = { "_windowed_mouse", "1", 0, OnChange_windowed_mouse };
cvar_t        m_filter = { "m_filter", "0", CVAR_ARCHIVE };
cvar_t        cl_keypad = { "cl_keypad", "1" };
cvar_t        vid_hwgammacontrol = { "vid_hwgammacontrol", "1" };

// lxndr: still required ?
void D_BeginDirectRect (int x, int y, byte * pbitmap, int width, int height) {
}
void D_EndDirectRect (int x, int y, int width, int height) {
}

const char   *glx_extensions;
qboolean CheckGLXExtension (Display * dpy, int scr, const char *extension) {
    char         *where, *terminator;
    const char   *start;

    if (!glx_extensions && !(glx_extensions = glXQueryExtensionsString (dpy, scr)))
        return false;

    if (!extension || *extension == 0 || strchr (extension, ' '))
        return false;

    for (start = glx_extensions; (where = strstr (start, extension)); start = terminator) {
        terminator = where + strlen (extension);
        if ((where == start || *(where - 1) == ' ') && (*terminator == 0 || *terminator == ' '))
            return true;
    }

    return false;
}

void         *qglGetProcAddress (const char *proc) {
    return glXGetProcAddressARB ((byte *) proc);
}

int           (*glxGetVideoSync) (unsigned int *count);
int           (*glxWaitVideoSync) (int div, int rem, unsigned int *count);
qboolean      gl_vsync_ext = false;

void CheckVSyncExtensions (Display * dpy, int scr) {
    if (CheckGLXExtension (dpy, scr, "GLX_SGI_video_sync")) {
        glxGetVideoSync = qglGetProcAddress ("glXGetVideoSyncSGI");
        glxWaitVideoSync = qglGetProcAddress ("glXWaitVideoSyncSGI");
        if (glxGetVideoSync && glxWaitVideoSync) {
            Con_Printf ("  GLX_SGI_video_sync\n");
            gl_vsync_ext = true;
            Cvar_Register (&vid_vsync);
        }
    }
}

static void install_grabs (void) {
    XGrabPointer (dpy, win, True, 0, GrabModeAsync, GrabModeAsync, win, None, CurrentTime);

#ifdef HAVE_DGA
    int           DGAflags = 0;
    if (!COM_CheckParm ("-nomdga"))
        DGAflags |= XF86DGADirectMouse;
    if (!COM_CheckParm ("-nokdga"))
        DGAflags |= XF86DGADirectKeyb;

    if (!COM_CheckParm ("-nodga") && DGAflags) {
        XF86DGADirectVideo (dpy, DefaultScreen (dpy), DGAflags);
        if (DGAflags & XF86DGADirectMouse)
            dgamouse = true;
        if (DGAflags & XF86DGADirectKeyb)
            dgakeyb = true;
    }
    if (!dgamouse)                     // lxndr: XWarpPointer only if dga mouse is disable
#endif

        XWarpPointer (dpy, None, win, 0, 0, 0, 0, vid.width / 2, vid.height / 2);
    XGrabKeyboard (dpy, win, False, GrabModeAsync, GrabModeAsync, CurrentTime);
}

static void uninstall_grabs (void) {
#ifdef HAVE_DGA
    if (dgamouse || dgakeyb) {
        XF86DGADirectVideo (dpy, DefaultScreen (dpy), 0);
        dgamouse = dgakeyb = false;
    }
#endif

    XUngrabPointer (dpy, CurrentTime);
    XUngrabKeyboard (dpy, CurrentTime);
}

qboolean OnChange_windowed_mouse (cvar_t * var, char *value) {
#ifdef HAVE_VMODE
    if (vidmode_active && !Q_atof (value)) {
        Con_Printf ("Cannot turn %s off when using -fullscreen mode\n", var->name);
        return true;
    }
#endif

    return false;
}

static void GetEvent (void) {
    XEvent        event;

    if (!dpy)
        return;

    XNextEvent (dpy, &event);

    switch (event.type) {
        case KeyPress:
        case KeyRelease:
            Key_Event (XLateKey (&event.xkey), event.type == KeyPress);
            break;

        case MotionNotify:
            if (_windowed_mouse.value) {
#ifdef HAVE_DGA
                if (dgamouse) {
                    mouse_x += event.xmotion.x_root;
                    mouse_y += event.xmotion.y_root;
                } else
#endif
                {
                    mouse_x = ((int) event.xmotion.x - (int) (vid.width / 2));
                    mouse_y = ((int) event.xmotion.y - (int) (vid.height / 2));

                    // move the mouse to the window center again
                    XSelectInput (dpy, win, X_MASK & ~PointerMotionMask);
                    XWarpPointer (dpy, None, win, 0, 0, 0, 0, (vid.width / 2), (vid.height / 2));
                    XSelectInput (dpy, win, X_MASK);
                }
            }
            break;

        case ButtonPress:
        case ButtonRelease:
            switch (event.xbutton.button) {
                case 1:
                    Key_Event (K_MOUSE1, event.type == ButtonPress);
                    break;

                case 2:
                    Key_Event (K_MOUSE3, event.type == ButtonPress);
                    break;

                case 3:
                    Key_Event (K_MOUSE2, event.type == ButtonPress);
                    break;

                case 4:
                    Key_Event (K_MWHEELUP, event.type == ButtonPress);
                    break;

                case 5:
                    Key_Event (K_MWHEELDOWN, event.type == ButtonPress);
                    break;
            }
            break;
    }

    if (old_windowed_mouse != _windowed_mouse.value) {
        old_windowed_mouse = _windowed_mouse.value;

        if (!_windowed_mouse.value)
            uninstall_grabs ();
        else
            install_grabs ();
    }
}

void signal_handler (int sig) {
    printf ("Received signal %d, exiting...\n", sig);
    Sys_Quit ();
    exit (0);
}

void InitSig (void) {
    signal (SIGHUP, signal_handler);
    signal (SIGINT, signal_handler);
    signal (SIGQUIT, signal_handler);
    signal (SIGILL, signal_handler);
    signal (SIGTRAP, signal_handler);
    signal (SIGIOT, signal_handler);
    signal (SIGBUS, signal_handler);
    signal (SIGFPE, signal_handler);
    signal (SIGSEGV, signal_handler);
    signal (SIGTERM, signal_handler);
}

void VID_ShiftPalette (unsigned char *p) {
}

#ifdef HAVE_VMODE
void InitHWGamma (void) {
    int           xf86vm_gammaramp_size;

    if (COM_CheckParm ("-nohwgamma"))
        return;

#  ifdef __DEBUG__
    Con_DPrintf ("InitHWGamma\n");
#  endif
    XF86VidModeGetGammaRampSize (dpy, scrnum, &xf86vm_gammaramp_size);

    vid_gammaworks = (xf86vm_gammaramp_size == 256);

    if (vid_gammaworks)
        XF86VidModeGetGammaRamp (dpy, scrnum, xf86vm_gammaramp_size, systemgammaramp[0], systemgammaramp[1],
                                 systemgammaramp[2]);
}

void RestoreHWGamma (void) {
#  ifdef __DEBUG__
    Con_DPrintf ("RestoreHWGamma\n");
#  endif
    if (vid_gammaworks && customgamma) {
        customgamma = false;
        XF86VidModeSetGammaRamp (dpy, scrnum, 256, systemgammaramp[0], systemgammaramp[1], systemgammaramp[2]);
    }
}

Colormap cmap;
#ifndef __DEVEL__
void VID_SetDeviceGammaRamp (unsigned short *ramps) {
#  ifdef __DEBUG__
    Con_DPrintf ("VID_SetDeviceGammaRamp\n");
#  endif
    if (vid_gammaworks) {
        currentgammaramp = ramps;
        if (vid_hwgamma_enabled) {
            XF86VidModeSetGammaRamp (dpy, scrnum, 256, ramps, ramps + 256, ramps + 512);
            customgamma = true;
        }
    }
}
#else
void VisualInfo (void);
XVisualInfo  *visinfo;
void VID_SetDeviceGammaRamp (unsigned short *ramps) {
    int i;
    XColor colors[256];
    XSetWindowAttributes attr;
    static qboolean alloc = false;

    if (visinfo->class != DirectColor)
        Sys_Error ("no directcolor");

    if (!alloc) {
        cmap = XCreateColormap (dpy, win, visinfo->visual, AllocAll);
        attr.colormap = cmap;
        XChangeWindowAttributes( dpy, win, CWColormap, &attr );
        alloc = true;
    }

    int f = 1;//65536;
    for (i = 0; i < 256; i++) {
        colors[i].pixel = i;
        colors[i].flags = DoRed | DoGreen | DoBlue;
        colors[i].red   = ramps[i] * f;
        colors[i].green = ramps[i + 256] * f;
        colors[i].blue  = ramps[i + 512] * f;
    }

    XLockDisplay (dpy);
    XStoreColors (dpy, cmap, colors, 256);
//    XInstallColormap (dpy, cmap);
    XSync(dpy, false);
    XFlush(dpy);
    XUnlockDisplay (dpy);
    XSetWindowColormap (dpy, win, cmap);
}
#endif
#else
void VID_SetDeviceGammaRamp (unsigned short *ramps) {
}
#endif

/*
 * GL_BeginRendering
 */
void GL_BeginRendering (int *x, int *y, int *width, int *height) {
    *x = *y = 0;
    *width = scr_width;
    *height = scr_height;
}

/*
 * GL_EndRendering
 */
void GL_EndRendering (void) {
    unsigned int  sync;

#ifdef HAVE_VMODE
    static qboolean old_hwgamma_enabled;

    vid_hwgamma_enabled = vid_hwgammacontrol.value && vid_gammaworks;
    vid_hwgamma_enabled = vid_hwgamma_enabled && (fullscreen || vid_hwgammacontrol.value == 2);
    if (vid_hwgamma_enabled != old_hwgamma_enabled) {
        old_hwgamma_enabled = vid_hwgamma_enabled;
        if (vid_hwgamma_enabled && currentgammaramp)
            VID_SetDeviceGammaRamp (currentgammaramp);
        else
            RestoreHWGamma ();
    }
#endif

    glFlush ();

    // lxndr: not really functional ?
    if (vid_vsync.value) {
        if (gl_vsync_ext) {            // lxndr: fixes framerate to a low value
            glxGetVideoSync (&sync);
            glxWaitVideoSync (2, (sync + 1) % 2, &sync);
        } else if (gl_swapcontrol_ext) {        // lxndr: fixes framerate to a lower value
            qglSwapInterval (1);
        }
    }

    glXSwapBuffers (dpy, win);

    if (fullsbardraw)
        Sbar_Changed ();
}

void VID_Shutdown (void) {
    if (!ctx)
        return;

    uninstall_grabs ();

#ifdef HAVE_VMODE
    RestoreHWGamma ();
#endif

    if (dpy) {
#ifdef HAVE_MISC
        VID_Manage3rdButton (dpy, false);
#endif
        VID_RestoreScreenSaver (dpy);

        glXDestroyContext (dpy, ctx);
        if (win)
            XDestroyWindow (dpy, win);
#ifdef HAVE_VMODE
        if (vidmode_active)
            XF86VidModeSwitchToMode (dpy, scrnum, vidmodes[0]);
        vidmode_active = false;
#endif
        XCloseDisplay (dpy);
    }
}

static Cursor CreateNullCursor (Display * display, Window root) {
    Pixmap        cursormask;
    XGCValues     xgc;
    GC            gc;
    XColor        dummycolour;
    Cursor        cursor;

    cursormask = XCreatePixmap (display, root, 1, 1, 1);
    xgc.function = GXclear;
    gc = XCreateGC (display, cursormask, GCFunction, &xgc);
    XFillRectangle (display, cursormask, gc, 0, 0, 1, 1);
    dummycolour.pixel = 0;
    dummycolour.red = 0;
    dummycolour.flags = 04;
    cursor = XCreatePixmapCursor (display, cursormask, cursormask, &dummycolour, &dummycolour, 0, 0);
    XFreePixmap (display, cursormask);
    XFreeGC (display, gc);

    return cursor;
}

void VID_Init (unsigned char *palette) {
    int           attrib[] = {
        GLX_RGBA,
        GLX_RED_SIZE, 1,
        GLX_GREEN_SIZE, 1,
        GLX_BLUE_SIZE, 1,
        GLX_DOUBLEBUFFER,
        GLX_DEPTH_SIZE, 1,
#ifdef __DEVEL__
        GLX_X_VISUAL_TYPE, GLX_DIRECT_COLOR, // lxndr: for gamma correction in windowed mode
#endif
#ifdef __DEVEL__0
        GLX_SAMPLE_BUFFERS_ARB, 1, GLX_SAMPLES_ARB, 16, // lxndr:
#endif
        None
    };
    int           i, width = 640, height = 480;
    XSetWindowAttributes attr;
    unsigned long mask;
#ifndef __DEVEL__
    XVisualInfo  *visinfo;
#endif
    Window        root;
    int           MajorVersion, MinorVersion, actualWidth, actualHeight;

    Cvar_Register (&vid_mode);
    Cvar_Register (&_windowed_mouse);
    Cvar_Register (&m_filter);
    Cvar_Register (&cl_keypad);
    Cvar_Register (&vid_hwgammacontrol);

    vid.maxwarpwidth = WARP_WIDTH;
    vid.maxwarpheight = WARP_HEIGHT;
    vid.colormap = host_colormap;
    vid.fullbright = 256 - LittleLong (*((int *) vid.colormap + 2048));

    if (!(dpy = XOpenDisplay (NULL)))
        Sys_Error ("Error: couldn't open the X display");

    if (!(visinfo = glXChooseVisual (dpy, scrnum, attrib)))
        Sys_Error ("Error: couldn't get an RGB, Double-buffered, Depth visual");

    scrnum = DefaultScreen (dpy);
    root = RootWindow (dpy, scrnum);

    if (COM_CheckParm ("-fullscreen"))
        fullscreen = true;
    if (COM_CheckParm ("-width") || COM_CheckParm ("-height")) {
        if ((i = COM_CheckParm ("-width")) && i + 1 < com_argc)
            width = Q_atoi (com_argv[i + 1]);

        if ((i = COM_CheckParm ("-height")) && i + 1 < com_argc)
            height = Q_atoi (com_argv[i + 1]);
    } else {
        if (!COM_CheckParm ("-current"))
            fullscreen = true;
        width = DisplayWidth (dpy, scrnum);
        height = DisplayHeight (dpy, scrnum);
    }

    vid.width = scr_width = width;
    vid.height = scr_height = height;
    //    vid.aspect = ((float) vid.height / (float) vid.width) * (320.0 / 240.0); // lxndr: not used in gl
    vid.numpages = 2;

    if ((i = COM_CheckParm ("-conwidth")) && i + 1 < com_argc)
        vid.conwidth = Q_atoi (com_argv[i + 1]);
    else
        vid.conwidth = width;
    vid.conwidth &= 0xfff8;            // make it a multiple of eight
    if ((i = COM_CheckParm ("-conheight")) && i + 1 < com_argc)
        vid.conheight = Q_atoi (com_argv[i + 1]);
    else
        vid.conheight = vid.conwidth * height / width;

#ifdef HAVE_VMODE
    // Get video mode list
    MajorVersion = MinorVersion = 0;
    if (!XF86VidModeQueryVersion (dpy, &MajorVersion, &MinorVersion)) {
        vidmode_ext = false;
    } else {
        Con_Printf ("Using XFree86-VidModeExtension Version %d.%d\n", MajorVersion, MinorVersion);
        vidmode_ext = true;
    }

    if (vidmode_ext) {
        int           best_fit, best_dist, dist, x, y;

        XF86VidModeGetAllModeLines (dpy, scrnum, &num_vidmodes, &vidmodes);

        // Are we going fullscreen? If so, let's change video mode
        if (fullscreen) {
            best_dist = 9999999;
            best_fit = -1;

            for (i = 0; i < num_vidmodes; i++) {
                if (width > vidmodes[i]->hdisplay || height > vidmodes[i]->vdisplay)
                    continue;

                x = width - vidmodes[i]->hdisplay;
                y = height - vidmodes[i]->vdisplay;
                dist = (x * x) + (y * y);
                if (dist < best_dist) {
                    best_dist = dist;
                    best_fit = i;
                }
            }

            if (best_fit != -1) {
                actualWidth = vidmodes[best_fit]->hdisplay;
                actualHeight = vidmodes[best_fit]->vdisplay;

                // change to the mode
                XF86VidModeSwitchToMode (dpy, scrnum, vidmodes[best_fit]);
                vidmode_active = true;

                // Move the viewport to top left
                XF86VidModeSetViewPort (dpy, scrnum, 0, 0);
            } else {
                fullscreen = false;
            }
        }
    }
#endif

    // window attributes
    attr.background_pixel = 0;
    attr.border_pixel = 0;
    attr.colormap = XCreateColormap (dpy, root, visinfo->visual, AllocNone);
    attr.event_mask = X_MASK;
    mask = CWBackPixel | CWBorderPixel | CWColormap | CWEventMask;

    // fullscreen
#ifdef HAVE_VMODE
    if (vidmode_active) {
        mask = CWBackPixel | CWColormap | CWEventMask | CWSaveUnder | CWBackingStore | CWOverrideRedirect;
        attr.override_redirect = True;
        attr.backing_store = NotUseful;
        attr.save_under = False;
    }
#endif

    win = XCreateWindow (dpy, root, 0, 0, width, height, 0, visinfo->depth, InputOutput, visinfo->visual, mask, &attr);
    XDefineCursor (dpy, win, CreateNullCursor (dpy, win));
    XMapWindow (dpy, win);

#ifdef HAVE_VMODE
    if (vidmode_active) {
        XRaiseWindow (dpy, win);
        XWarpPointer (dpy, None, win, 0, 0, 0, 0, 0, 0);
        XFlush (dpy);
        // Move the viewport to top left
        XF86VidModeSetViewPort (dpy, scrnum, 0, 0);
    }
#endif

    VID_DisableScreenSaver (dpy);
#ifdef HAVE_MISC
    VID_Manage3rdButton (dpy, true);
#endif

    XFlush (dpy);

    ctx = glXCreateContext (dpy, visinfo, NULL, True);

    glXMakeCurrent (dpy, win, ctx);

    InitSig ();                        // trap evil signals

    GL_Init ();
    CheckVSyncExtensions (dpy, scrnum);

    Check_Gamma (palette);
    VID_SetPalette (palette);

#ifdef HAVE_VMODE
    InitHWGamma ();
    Con_Printf ("Video mode %dx%d initialized.\n", width, height);
#endif

    if (fullscreen)
        vid_windowedmouse = false;

    vid.recalc_refdef = 1;             // force a surface cache flush

    if (COM_CheckParm ("-fullsbar"))
        fullsbardraw = true;
}

void Sys_SendKeyEvents (void) {
    if (dpy) {
        while (XPending (dpy))
            GetEvent ();
    }
}

void Force_CenterView_f (void) {
    cl.viewangles[PITCH] = 0;
}

void IN_Init (void) {
    Cmd_AddCommand ("force_centerview", Force_CenterView_f);
}

void IN_Shutdown (void) {
}

void IN_Commands (void) {
}

void IN_MouseMove (usercmd_t * cmd) {
    float         tx, ty;

    tx = mouse_x;
    ty = mouse_y;

    if (m_filter.value) {
        mouse_x = (tx + old_mouse_x) * 0.5;
        mouse_y = (ty + old_mouse_y) * 0.5;
    }

    old_mouse_x = tx;
    old_mouse_y = ty;

    mouse_x *= sensitivity.value;
    mouse_y *= sensitivity.value;

    // add mouse X/Y movement to cmd
    if ((in_strafe.state & 1) || (lookstrafe.value && mlook_active))
        cmd->sidemove += m_side.value * mouse_x;
    else
        cl.viewangles[YAW] -= m_yaw.value * mouse_x;

    if (mlook_active)
        V_StopPitchDrift ();

    if (mlook_active && !(in_strafe.state & 1)) {
        cl.viewangles[PITCH] += m_pitch.value * mouse_y;
        cl.viewangles[PITCH] = bound (-70, cl.viewangles[PITCH], 80);
    } else {
        if ((in_strafe.state & 1) && noclip_anglehack)
            cmd->upmove -= m_forward.value * mouse_y;
        else
            cmd->forwardmove -= m_forward.value * mouse_y;
    }
    mouse_x = mouse_y = 0.0;
}

void IN_Move (usercmd_t * cmd) {
    IN_MouseMove (cmd);
}

#ifdef __DEVEL__
// lxndr: from bzflag
void VisualInfo (void) {
    int           n_fbc;
    GLXFBConfig  *fbc;
    int           value, ret, i;

    fbc = glXGetFBConfigs (dpy, DefaultScreen (dpy), &n_fbc);

    if (fbc) {
        if (1) {
            /* print table header */
            Sys_Printf (" +-----+-------------------------+-----------------+----------+-------------+-------+------+\n");
            Sys_Printf (" |     |        visual           |      color      | ax dp st |    accum    |   ms  |  cav |\n");
            Sys_Printf (" |  id | tp xr cl fm db st lv xp |  sz  r  g  b  a | bf th cl | r  g  b  a  | ns  b |  eat |\n");
            Sys_Printf (" +-----+-------------------------+-----------------+----------+-------------+-------+------+\n");
            /* loop through all the fbcs */
            for (i = 0; i < n_fbc; i++) {
                /* print out the information for this fbc */
                /* visual id */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_FBCONFIG_ID, &value);
                if (ret != Success) {
                    Sys_Printf ("|  ?  |");
                } else {
                    Sys_Printf (" |% 4d | ", value);
                }
                /* visual type */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_DRAWABLE_TYPE, &value);
                if (ret != Success) {
                    Sys_Printf (" ? ");
                } else {
                    if (value & GLX_WINDOW_BIT) {
                        if (value & GLX_PBUFFER_BIT) {
                            Sys_Printf ("wp ");
                        } else {
                            Sys_Printf ("wn ");
                        }
                    } else {
                        if (value & GLX_PBUFFER_BIT) {
                            Sys_Printf ("pb ");
                        } else if (value & GLX_PIXMAP_BIT) {
                            Sys_Printf ("pm ");
                        } else {
                            Sys_Printf (" ? ");
                        }
                    }
                }
                /* x renderable */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_X_RENDERABLE, &value);
                if (ret != Success) {
                    Sys_Printf (" ? ");
                } else {
                    Sys_Printf (value ? " y " : " n ");
                }
                /* class */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_X_VISUAL_TYPE, &value);
                if (ret != Success) {
                    Sys_Printf (" ? ");
                } else {
                    if (GLX_TRUE_COLOR == value)
                        Sys_Printf ("tc ");
                    else if (GLX_DIRECT_COLOR == value)
                        Sys_Printf ("dc ");
                    else if (GLX_PSEUDO_COLOR == value)
                        Sys_Printf ("pc ");
                    else if (GLX_STATIC_COLOR == value)
                        Sys_Printf ("sc ");
                    else if (GLX_GRAY_SCALE == value)
                        Sys_Printf ("gs ");
                    else if (GLX_STATIC_GRAY == value)
                        Sys_Printf ("sg ");
                    else if (GLX_X_VISUAL_TYPE == value)
                        Sys_Printf (" . ");
                    else
                        Sys_Printf (" ? ");
                }
                /* format */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_RENDER_TYPE, &value);
                Sys_Printf (" ? ");
/*                if (ret != Success) {
                    Sys_Printf (" ? ");
                } else {
                    if (GLXEW_NV_float_buffer) {
                        int           ret2, value2;
                        ret2 = glXGetFBConfigAttrib (dpy, fbc[i], GLX_FLOAT_COMPONENTS_NV, &value2);
                        if (Success == ret2 && GL_TRUE == value2) {
                            Sys_Printf (" f ");
                        } else if (value & GLX_RGBA_BIT)
                            Sys_Printf (" i ");
                        else if (value & GLX_COLOR_INDEX_BIT)
                            Sys_Printf (" c ");
                        else
                            Sys_Printf (" ? ");
                    } else {
                        if (value & GLX_RGBA_FLOAT_ATI_BIT)
                            Sys_Printf (" f ");
                        else if (value & GLX_RGBA_BIT)
                            Sys_Printf (" i ");
                        else if (value & GLX_COLOR_INDEX_BIT)
                            Sys_Printf (" c ");
                        else
                            Sys_Printf (" ? ");
                    }
                }*/
                /* double buffer */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_DOUBLEBUFFER, &value);
                Sys_Printf (" %c ", Success != ret ? '?' : (value ? 'y' : '.'));
                /* stereo */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_STEREO, &value);
                Sys_Printf (" %c ", Success != ret ? '?' : (value ? 'y' : '.'));
                /* level */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_LEVEL, &value);
                if (Success != ret) {
                    Sys_Printf (" ? ");
                } else {
                    Sys_Printf ("%2d ", value);
                }
                /* transparency */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_TRANSPARENT_TYPE, &value);
                if (Success != ret) {
                    Sys_Printf (" ? | ");
                } else {
                    if (GLX_TRANSPARENT_RGB == value)
                        Sys_Printf (" r | ");
                    else if (GLX_TRANSPARENT_INDEX == value)
                        Sys_Printf (" i | ");
                    else if (GLX_NONE == value)
                        Sys_Printf (" . | ");
                    else
                        Sys_Printf (" ? | ");
                }
                /* color size */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_BUFFER_SIZE, &value);
                if (Success != ret) {
                    Sys_Printf ("  ? ");
                } else {
                    if (value)
                        Sys_Printf ("%3d ", value);
                    else
                        Sys_Printf ("  . ");
                }
                /* red size */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_RED_SIZE, &value);
                if (Success != ret) {
                    Sys_Printf (" ? ");
                } else {
                    if (value)
                        Sys_Printf ("%2d ", value);
                    else
                        Sys_Printf (" . ");
                }
                /* green size */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_GREEN_SIZE, &value);
                if (Success != ret) {
                    Sys_Printf (" ? ");
                } else {
                    if (value)
                        Sys_Printf ("%2d ", value);
                    else
                        Sys_Printf (" . ");
                }
                /* blue size */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_BLUE_SIZE, &value);
                if (Success != ret) {
                    Sys_Printf (" ? ");
                } else {
                    if (value)
                        Sys_Printf ("%2d ", value);
                    else
                        Sys_Printf (" . ");
                }
                /* alpha size */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_ALPHA_SIZE, &value);
                if (Success != ret) {
                    Sys_Printf (" ? | ");
                } else {
                    if (value)
                        Sys_Printf ("%2d | ", value);
                    else
                        Sys_Printf (" . | ");
                }
                /* aux buffers */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_AUX_BUFFERS, &value);
                if (Success != ret) {
                    Sys_Printf (" ? ");
                } else {
                    if (value)
                        Sys_Printf ("%2d ", value);
                    else
                        Sys_Printf (" . ");
                }
                /* depth size */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_DEPTH_SIZE, &value);
                if (Success != ret) {
                    Sys_Printf (" ? ");
                } else {
                    if (value)
                        Sys_Printf ("%2d ", value);
                    else
                        Sys_Printf (" . ");
                }
                /* stencil size */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_STENCIL_SIZE, &value);
                if (Success != ret) {
                    Sys_Printf (" ? | ");
                } else {
                    if (value)
                        Sys_Printf ("%2d | ", value);
                    else
                        Sys_Printf (" . | ");
                }
                /* accum red size */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_ACCUM_RED_SIZE, &value);
                if (Success != ret) {
                    Sys_Printf (" ? ");
                } else {
                    if (value)
                        Sys_Printf ("%2d ", value);
                    else
                        Sys_Printf (" . ");
                }
                /* accum green size */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_ACCUM_GREEN_SIZE, &value);
                if (Success != ret) {
                    Sys_Printf (" ? ");
                } else {
                    if (value)
                        Sys_Printf ("%2d ", value);
                    else
                        Sys_Printf (" . ");
                }
                /* accum blue size */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_ACCUM_BLUE_SIZE, &value);
                if (Success != ret) {
                    Sys_Printf (" ? ");
                } else {
                    if (value)
                        Sys_Printf ("%2d ", value);
                    else
                        Sys_Printf (" . ");
                }
                /* accum alpha size */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_ACCUM_ALPHA_SIZE, &value);
                if (Success != ret) {
                    Sys_Printf (" ? | ");
                } else {
                    if (value)
                        Sys_Printf ("%2d | ", value);
                    else
                        Sys_Printf (" . | ");
                }
                /* multisample */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_SAMPLES, &value);
                if (Success != ret) {
                    Sys_Printf (" ? ");
                } else {
                    Sys_Printf ("%2d ", value);
                }
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_SAMPLE_BUFFERS, &value);
                if (Success != ret) {
                    Sys_Printf (" ? | ");
                } else {
                    Sys_Printf ("%2d | ", value);
                }
                /* caveat */
                ret = glXGetFBConfigAttrib (dpy, fbc[i], GLX_CONFIG_CAVEAT, &value);
                if (Success != ret) {
                    Sys_Printf ("???? |");
                } else {
                    if (GLX_NONE == value)
                        Sys_Printf ("none |\n");
                    else if (GLX_SLOW_CONFIG == value)
                        Sys_Printf ("slow |\n");
                    else if (GLX_NON_CONFORMANT_CONFIG == value)
                        Sys_Printf ("ncft |\n");
                    else
                        Sys_Printf ("???? |\n");
                }
            }
            /* print table footer */
            Sys_Printf (" +-----+-------------------------+-----------------+----------+-------------+-------+------+\n");
            Sys_Printf (" |  id | tp xr cl fm db st lv xp |  sz  r  g  b  a | bf th cl | r  g  b  a  | ns  b |  eat |\n");
            Sys_Printf (" |     |        visual           |      color      | ax dp st |    accum    |   ms  |  cav |\n");
            Sys_Printf (" +-----+-------------------------+-----------------+----------+-------------+-------+------+\n");
        }
    }
}
#endif
