/*
	$RCSfile: snd_dma.c,v $

	Copyright (C) 1996-1997  Id Software, Inc.

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

	See the GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to:
	
		Free Software Foundation, Inc.
		59 Temple Place - Suite 330
		Boston, MA  02111-1307, USA

*/
static const char rcsid[] =
    "$Id: snd_dma.c,v 1.38 2003/06/14 23:53:07 warp Exp $";

#include "twiconfig.h"

#include <stdlib.h>	/* for rand() */

#include "SDL.h"

#include "quakedef.h"
#include "cmd.h"
#include "client.h"
#include "cvar.h"
#include "model.h"
#include "host.h"
#include "mathlib.h"
#include "sound.h"
#include "strlib.h"
#include "sys.h"

static void S_Play (void);
static void S_PlayVol (void);
static void S_SoundList (void);
static void S_Update_ ();
void        S_StopAllSounds (qboolean clear);
static void S_StopAllSoundsC (void);
static void S_ClearBuffer (void);

// =======================================================================
// Internal sound data & structures
// =======================================================================

memzone_t	*snd_zone;
channel_t   channels[MAX_CHANNELS];
int         total_channels;

static int         snd_blocked = 0;
static qboolean    snd_initialized = false;

// pointer should go away
dma_t *shm = 0;

static vec3_t      listener_origin;
static vec3_t      listener_forward;
static vec3_t      listener_right;
static vec3_t      listener_up;
static vec_t       sound_nominal_clip_dist = 1000.0;

int         soundtime;					// sample PAIRS
int         paintedtime;				// sample PAIRS


#define	MAX_SFX		512
static sfx_t      *known_sfx;					// hunk allocated [MAX_SFX]
static int         num_sfx;

static sfx_t      *ambient_sfx[NUM_AMBIENTS];

static int         sound_started = 0;

cvar_t     *volume;

static cvar_t     *nosound;
static cvar_t     *precache;
cvar_t     *loadas8bit;
static cvar_t     *ambient_level;
static cvar_t     *ambient_fade;
static cvar_t     *snd_noextraupdate;
static cvar_t     *snd_show;
static cvar_t     *_snd_mixahead;


// ====================================================================
// User-setable variables
// ====================================================================


//
// Fake dma is a synchronous faking of the DMA progress used for
// isolating performance in the renderer.
//

static qboolean    fakedma = false;


static void
S_SoundInfo_f (void)
{
	if (!sound_started || !shm) {
		Com_Printf ("sound system not started\n");
		return;
	}

	Com_Printf ("%5d channels\n", shm->channels);
	Com_Printf ("%5d samples\n", shm->samples);
	Com_Printf ("%5d samplepos\n", shm->samplepos);
	Com_Printf ("%5d samplebits\n", shm->samplebits);
	Com_Printf ("%5d speed\n", shm->speed);
	Com_Printf ("0x%x dma buffer\n", shm->buffer);
	Com_Printf ("%5d total_channels\n", total_channels);
}


static void
S_Startup (void)
{
	int         rc;

	if (!snd_initialized)
		return;

	if (!fakedma) {
		rc = SNDDMA_Init ();

		if (!rc) {
			sound_started = 0;
			return;
		}
	}

	sound_started = 1;
}

void
S_Init_Cvars (void)
{
	volume = Cvar_Get ("volume", "0.7", CVAR_ARCHIVE, NULL);

	ambient_level = Cvar_Get ("ambient_level", "0.3", CVAR_NONE, NULL);
	ambient_fade = Cvar_Get ("ambient_fade", "100", CVAR_NONE, NULL);

	nosound = Cvar_Get ("nosound", "0", CVAR_NONE, NULL);
	precache = Cvar_Get ("precache", "1", CVAR_NONE, NULL);
	loadas8bit = Cvar_Get ("loadas8bit", "0", CVAR_NONE, NULL);
	snd_noextraupdate = Cvar_Get ("snd_noextraupdate", "0", CVAR_NONE, NULL);
	snd_show = Cvar_Get ("snd_show", "0", CVAR_NONE, NULL);
	_snd_mixahead = Cvar_Get ("_snd_mixahead", "0.05", CVAR_ARCHIVE, NULL);
}

void
S_Init (void)
{

//  Com_Printf("\nSound Initialization\n");

	if (COM_CheckParm ("-nosound"))
		return;

	if (COM_CheckParm ("-simsound"))
		fakedma = true;

	Cmd_AddCommand ("play", S_Play);
	Cmd_AddCommand ("playvol", S_PlayVol);
	Cmd_AddCommand ("stopsound", S_StopAllSoundsC);
	Cmd_AddCommand ("soundlist", S_SoundList);
	Cmd_AddCommand ("soundinfo", S_SoundInfo_f);

	snd_initialized = true;

	S_Startup ();

	SND_InitScaletable ();

	snd_zone = Zone_AllocZone ("sound");
	known_sfx = Zone_Alloc (snd_zone, MAX_SFX * sizeof (sfx_t));
	num_sfx = 0;

// create a piece of DMA memory

	if (fakedma) {
		shm = Zone_Alloc (snd_zone, sizeof (*shm));
		shm->samplebits = 16;
		shm->speed = 22050;
		shm->channels = 2;
		shm->samples = 32768;
		shm->samplepos = 0;
		shm->buffer = Zone_Alloc (snd_zone, 1 << 16);
	}
//  Com_Printf ("Sound sampling rate: %i\n", shm->speed);

	// provides a tick sound until washed clean

//  if (shm->buffer)
//      shm->buffer[4] = shm->buffer[5] = 0x7f; // force a pop for debugging

	ambient_sfx[AMBIENT_WATER] = S_PrecacheSound ("ambience/water1.wav");
	ambient_sfx[AMBIENT_SKY] = S_PrecacheSound ("ambience/wind2.wav");

	S_StopAllSounds (true);
}


// =======================================================================
// Shutdown sound engine
// =======================================================================

void
S_Shutdown (void)
{

	if (!sound_started)
		return;

	shm = 0;
	sound_started = 0;

	if (!fakedma) {
		SNDDMA_Shutdown ();
	}
}


// =======================================================================
// Load a sound
// =======================================================================

static sfx_t      *
S_FindName (char *name)
{
	int         i;
	sfx_t      *sfx;

	if (!name)
		Sys_Error ("S_FindName: NULL\n");

	if (strlen (name) >= MAX_QPATH)
		Sys_Error ("Sound name too long: %s", name);

// see if already loaded
	for (i = 0; i < num_sfx; i++)
		if (!strcmp (known_sfx[i].name, name)) {
			return &known_sfx[i];
		}

	if (num_sfx == MAX_SFX)
		Sys_Error ("S_FindName: out of sfx_t");

	sfx = &known_sfx[i];
	strlcpy_s (sfx->name, name);

	num_sfx++;

	return sfx;
}


sfx_t      *
S_PrecacheSound (char *name)
{
	sfx_t      *sfx;

	if (!sound_started || nosound->ivalue)
		return NULL;

	sfx = S_FindName (name);

// cache it in
	if (precache->ivalue)
		S_LoadSound (sfx);

	return sfx;
}


//=============================================================================

static channel_t  *
SND_PickChannel (int entnum, int entchannel)
{
	int         ch_idx;
	int         first_to_die;
	int         life_left;

// Check for replacement sound, or find the best one to replace
	first_to_die = -1;
	life_left = 0x7fffffff;
	for (ch_idx = NUM_AMBIENTS; ch_idx < NUM_AMBIENTS + MAX_DYNAMIC_CHANNELS;
		 ch_idx++) {
		if (entchannel != 0				// channel 0 never overrides
			&& channels[ch_idx].entnum == entnum
			&& (channels[ch_idx].entchannel == entchannel
				|| entchannel == -1)) {	// always override sound
										// from same entity
			first_to_die = ch_idx;
			break;
		}
		// don't let monster sounds override player sounds
		if (channels[ch_idx].entnum == cl.viewentity && entnum != cl.viewentity
			&& channels[ch_idx].sfx)
			continue;

		if (channels[ch_idx].end - paintedtime < life_left) {
			life_left = channels[ch_idx].end - paintedtime;
			first_to_die = ch_idx;
		}
	}

	if (first_to_die == -1)
		return NULL;

	if (channels[first_to_die].sfx)
		channels[first_to_die].sfx = NULL;

	return &channels[first_to_die];
}

static void
SND_Spatialize (channel_t *ch)
{
	vec_t       dot;
	vec_t       dist;
	vec_t       lscale, rscale, scale;
	vec3_t      source_vec;
	sfx_t      *snd;

// anything coming from the view entity will always be full volume
	if (ch->entnum == cl.viewentity) {
		ch->leftvol = ch->master_vol;
		ch->rightvol = ch->master_vol;
		return;
	}
// calculate stereo seperation and distance attenuation

	snd = ch->sfx;
	VectorSubtract (ch->origin, listener_origin, source_vec);

	dist = VectorNormalize (source_vec) * ch->dist_mult;

	dot = DotProduct (listener_right, source_vec);

	if (shm->channels == 1) {
		rscale = 1.0;
		lscale = 1.0;
	} else {
		rscale = 1.0 + dot;
		lscale = 1.0 - dot;
	}

// add in distance effect
	scale = (1.0 - dist) * rscale;
	ch->rightvol = (int) (ch->master_vol * scale);
	if (ch->rightvol < 0)
		ch->rightvol = 0;

	scale = (1.0 - dist) * lscale;
	ch->leftvol = (int) (ch->master_vol * scale);
	if (ch->leftvol < 0)
		ch->leftvol = 0;
}


// =======================================================================
// Start a sound effect
// =======================================================================

void
S_StartSound (int entnum, int entchannel, sfx_t *sfx, vec3_t origin, float fvol,
			  float attenuation)
{
	channel_t	*target_chan, *check;
	int			 vol;
	int			 ch_idx;
	int			 skip;

	if (!sound_started)
		return;

	if (!sfx)
		return;

	if (nosound->ivalue)
		return;

	vol = fvol * 255;

// pick a channel to play on
	target_chan = SND_PickChannel (entnum, entchannel);
	if (!target_chan)
		return;

// spatialize
	memset (target_chan, 0, sizeof (*target_chan));
	VectorCopy (origin, target_chan->origin);
	target_chan->dist_mult = attenuation / sound_nominal_clip_dist;
	target_chan->master_vol = vol;
	target_chan->entnum = entnum;
	target_chan->entchannel = entchannel;
	SND_Spatialize (target_chan);

	if (!target_chan->leftvol && !target_chan->rightvol)
		return;							// not audible at all

// new channel
	S_LoadSound (sfx);
	if (!sfx->loaded) {
		target_chan->sfx = NULL;
		return;							// couldn't load the sound's data
	}

	target_chan->sfx = sfx;
	target_chan->pos = 0.0;
	target_chan->end = paintedtime + sfx->length;

// if an identical sound has also been started this frame, offset the pos
// a bit to keep it from just making the first one louder
	check = &channels[NUM_AMBIENTS];
	for (ch_idx = NUM_AMBIENTS; ch_idx < NUM_AMBIENTS + MAX_DYNAMIC_CHANNELS;
		 ch_idx++, check++) {
		if (check == target_chan)
			continue;
		if (check->sfx == sfx && !check->pos) {
			skip = rand () % (int) (0.1 * shm->speed);
			if (skip >= target_chan->end)
				skip = target_chan->end - 1;
			target_chan->pos += skip;
			target_chan->end -= skip;
			break;
		}

	}
}

void
S_StopSound (int entnum, int entchannel)
{
	int         i;

	for (i = 0; i < MAX_DYNAMIC_CHANNELS; i++) {
		if (channels[i].entnum == entnum
			&& channels[i].entchannel == entchannel) {
			channels[i].end = 0;
			channels[i].sfx = NULL;
			return;
		}
	}
}

void
S_StopAllSounds (qboolean clear)
{
	int         i;

	if (!sound_started)
		return;

	total_channels = MAX_DYNAMIC_CHANNELS + NUM_AMBIENTS;	// no statics

	for (i = 0; i < MAX_CHANNELS; i++)
		if (channels[i].sfx)
			channels[i].sfx = NULL;

	memset (channels, 0, MAX_CHANNELS * sizeof (channel_t));

	if (clear)
		S_ClearBuffer ();
}

static void
S_StopAllSoundsC (void)
{
	S_StopAllSounds (true);
}

static void
S_ClearBuffer (void)
{
	int         clear;

	if (!sound_started || !shm || !shm->buffer)
		return;

	if (shm->samplebits == 8)
		clear = 0x80;
	else
		clear = 0;
	
	memset (shm->buffer, clear, shm->samples * shm->samplebits / 8);
}


void
S_StaticSound (sfx_t *sfx, vec3_t origin, float vol, float attenuation)
{
	channel_t  *ss;

	if (!sfx)
		return;

	if (total_channels == MAX_CHANNELS) {
		Com_Printf ("total_channels == MAX_CHANNELS\n");
		return;
	}

	ss = &channels[total_channels];
	total_channels++;

	S_LoadSound (sfx);
	if (!sfx->loaded)
		return;

	if (sfx->loopstart == -1) {
		Com_Printf ("Sound %s not looped\n", sfx->name);
		return;
	}

	ss->sfx = sfx;
	VectorCopy (origin, ss->origin);
	ss->master_vol = vol;
	ss->dist_mult = (attenuation / 64) / sound_nominal_clip_dist;
	ss->end = paintedtime + sfx->length;

	SND_Spatialize (ss);
}


//=============================================================================

static void
S_UpdateAmbientSounds (void)
{
	mleaf_t    *l;
	float       vol;
	int         ambient_channel;
	channel_t  *chan;

// calc ambient sound levels
	if (!ccl.worldmodel)
		return;

	l = Mod_PointInLeaf (listener_origin, ccl.worldmodel);
	if (!l || !ambient_level->fvalue) {
		for (ambient_channel = 0; ambient_channel < NUM_AMBIENTS;
			 ambient_channel++)
			channels[ambient_channel].sfx = NULL;
		return;
	}

	for (ambient_channel = 0; ambient_channel < NUM_AMBIENTS; ambient_channel++) {
		chan = &channels[ambient_channel];
		chan->sfx = ambient_sfx[ambient_channel];

		vol = ambient_level->fvalue * l->ambient_sound_level[ambient_channel];
		if (vol < 8)
			vol = 0;

		// don't adjust volume too fast
		if (chan->master_vol < vol) {
			chan->master_vol += host_frametime * ambient_fade->fvalue;
			if (chan->master_vol > vol)
				chan->master_vol = vol;
		} else if (chan->master_vol > vol) {
			chan->master_vol -= host_frametime * ambient_fade->fvalue;
			if (chan->master_vol < vol)
				chan->master_vol = vol;
		}

		chan->leftvol = chan->rightvol = chan->master_vol;
	}
}


/*
============
Called once each time through the main loop
============
*/
void
S_Update (vec3_t origin, vec3_t forward, vec3_t right, vec3_t up)
{
	int         i, j;
	int         total;
	channel_t  *ch;
	channel_t  *combine;

	if (!sound_started || (snd_blocked > 0))
		return;

	VectorCopy (origin, listener_origin);
	VectorCopy (forward, listener_forward);
	VectorCopy (right, listener_right);
	VectorCopy (up, listener_up);

// update general area ambient sound sources
	S_UpdateAmbientSounds ();

	combine = NULL;

// update spatialization for static and dynamic sounds  
	ch = channels + NUM_AMBIENTS;
	for (i = NUM_AMBIENTS; i < total_channels; i++, ch++) {
		if (!ch->sfx)
			continue;
		SND_Spatialize (ch);			// respatialize channel
		if (!ch->leftvol && !ch->rightvol)
			continue;

		// try to combine static sounds with a previous channel of the same
		// sound effect so we don't mix five torches every frame

		if (i >= MAX_DYNAMIC_CHANNELS + NUM_AMBIENTS) {
			// see if it can just use the last one
			if (combine && combine->sfx == ch->sfx) {
				combine->leftvol += ch->leftvol;
				combine->rightvol += ch->rightvol;
				ch->leftvol = ch->rightvol = 0;
				continue;
			}
			// search for one
			combine = channels + MAX_DYNAMIC_CHANNELS + NUM_AMBIENTS;
			for (j = MAX_DYNAMIC_CHANNELS + NUM_AMBIENTS; j < i; j++, combine++)
				if (combine->sfx == ch->sfx)
					break;

			if (j == total_channels) {
				combine = NULL;
			} else {
				if (combine != ch) {
					combine->leftvol += ch->leftvol;
					combine->rightvol += ch->rightvol;
					ch->leftvol = ch->rightvol = 0;
				}
				continue;
			}
		}


	}

//
// debugging output
//
	if (snd_show->ivalue) {
		total = 0;
		ch = channels;
		for (i = 0; i < total_channels; i++, ch++)
			if (ch->sfx && (ch->leftvol || ch->rightvol)) {
				// Com_Printf ("%3i %3i %s\n", ch->leftvol, ch->rightvol,
				// ch->sfx->name);
				total++;
			}

		Com_Printf ("----(%i)----\n", total);
	}
// mix some sound
	S_Update_ ();
}

static void
GetSoundtime (void)
{
	int         samplepos;
	static int  buffers;
	static int  oldsamplepos;
	int         fullsamples;

	fullsamples = shm->samples / shm->channels;

// it is possible to miscount buffers if it has wrapped twice between
// calls to S_Update.  Oh well.
	samplepos = SNDDMA_GetDMAPos ();

	if (samplepos < oldsamplepos) {
		buffers++;						// buffer wrapped

		if (paintedtime > 0x40000000) {	// time to chop things off to avoid 32
										// bit limits
			buffers = 0;
			paintedtime = fullsamples;
			S_StopAllSounds (true);
		}
	}
	oldsamplepos = samplepos;

	soundtime = buffers * fullsamples + samplepos / shm->channels;
}

void
S_ExtraUpdate (void)
{
	if (snd_noextraupdate->ivalue)
		return;							// don't pollute timings
	S_Update_ ();
}


static void
S_Update_ (void)
{
	Uint32	endtime;
	int         samps;

	if (!sound_started || (snd_blocked > 0))
		return;

	SDL_LockAudio ();
	// Updates DMA time
	GetSoundtime ();

	// check to make sure that we haven't overshot
	if (paintedtime < soundtime) {
		// Com_Printf ("S_Update_ : overflow\n");
		paintedtime = soundtime;
	}
	// mix ahead of current position
	endtime = soundtime + _snd_mixahead->fvalue * shm->speed;
	samps = shm->samples >> (shm->channels - 1);
	if ((endtime - soundtime) > (Uint32)samps)
		endtime = soundtime + samps;

	S_PaintChannels (endtime);
	SDL_UnlockAudio ();
}

/*
===============================================================================

console functions

===============================================================================
*/

static void
S_Play (void)
{
	static int  hash = 345;
	int         i;
	char        name[256];
	sfx_t      *sfx;

	i = 1;
	while (i < Cmd_Argc ()) {
		if (!strrchr (Cmd_Argv (i), '.')) {
			strlcpy_s (name, Cmd_Argv (i));
			strlcat_s (name, ".wav");
		} else
			strlcpy_s (name, Cmd_Argv (i));
		sfx = S_PrecacheSound (name);
		S_StartSound (hash++, 0, sfx, listener_origin, 1.0, 1.0);
		i++;
	}
}

static void
S_PlayVol (void)
{
	static int  hash = 543;
	int         i;
	float       vol;
	char        name[256];
	sfx_t      *sfx;

	i = 1;
	while (i < Cmd_Argc ()) {
		if (!strrchr (Cmd_Argv (i), '.')) {
			strlcpy_s (name, Cmd_Argv (i));
			strlcat_s (name, ".wav");
		} else
			strlcpy_s (name, Cmd_Argv (i));
		sfx = S_PrecacheSound (name);
		vol = Q_atof (Cmd_Argv (i + 1));
		S_StartSound (hash++, 0, sfx, listener_origin, vol, 1.0);
		i += 2;
	}
}

static void
S_SoundList (void)
{
	int         i;
	sfx_t      *sfx;
	int         size, total;

	total = 0;
	for (sfx = known_sfx, i = 0; i < num_sfx; i++, sfx++) {
		if (!sfx->loaded)
			continue;
		size = sfx->length * sfx->width * sfx->channels;
		total += size;
		if (sfx->loopstart >= 0)
			Com_Printf ("L");
		else
			Com_Printf (" ");
		Com_Printf ("(%2db) %6i : %s\n", sfx->width * 8, size, sfx->name);
	}
	Com_Printf ("Total resident: %i\n", total);
}


void
S_LocalSound (char *sound)
{
	sfx_t      *sfx;

	if (nosound->ivalue)
		return;
	if (!sound_started)
		return;

	sfx = S_PrecacheSound (sound);
	if (!sfx) {
		Com_Printf ("S_LocalSound: can't cache %s\n", sound);
		return;
	}
	S_StartSound (cl.viewentity, -1, sfx, vec3_origin, 1, 1);
}
