//**************************************************************************
//**
//**	$Id: cm46_local.h 247 2011-02-16 21:43:57Z dj_jl $
//**
//**	Copyright (C) 1996-2005 Id Software, Inc.
//**	Copyright (C) 2010 Jānis Legzdiņš
//**
//**	This program is free software; you can redistribute it and/or
//**  modify it under the terms of the GNU General Public License
//**  as published by the Free Software Foundation; either version 2
//**  of the License, or (at your option) any later version.
//**
//**	This program is distributed in the hope that it will be useful,
//**  but WITHOUT ANY WARRANTY; without even the implied warranty of
//**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//**  GNU General Public License for more details.
//**
//**************************************************************************

#ifndef _CM46_LOCAL_H
#define _CM46_LOCAL_H

#define MAX_FACETS			1024
#define MAX_PATCH_PLANES	2048
#define PLANE_TRI_EPSILON	0.1

#define SIDE_FRONT	0
#define SIDE_BACK	1
#define SIDE_ON		2
#define SIDE_CROSS	3

#define MAX_MAP_BOUNDS			65535

#define	MAX_SUBMODELS			256

struct cLeaf_t
{
	int			cluster;
	int			area;

	int			firstLeafBrush;
	int			numLeafBrushes;

	int			firstLeafSurface;
	int			numLeafSurfaces;
};

struct cArea_t
{
	int			floodnum;
	int			floodvalid;
};

struct cbrushside_t
{
	cplane_t*	plane;
	int			surfaceFlags;
	int			shaderNum;
};

struct cbrush_t
{
	int			shaderNum;		// the shader that determined the contents
	int			contents;
	vec3_t		bounds[2];
	int			numsides;
	cbrushside_t*	sides;
	int			checkcount;		// to avoid repeated testings
};

struct cNode_t
{
	cplane_t*	plane;
	int			children[2];		// negative numbers are leafs
};

struct patchPlane_t
{
	float		plane[4];
	int			signbits;		// signx + (signy<<1) + (signz<<2), used as lookup during collision
};

struct facet_t
{
	int			surfacePlane;
	int			numBorders;		// 3 or four + 6 axial bevels + 4 or 3 * 4 edge bevels
	int			borderPlanes[4 + 6 + 16];
	int			borderInward[4 + 6 + 16];
	qboolean	borderNoAdjust[4 + 6 + 16];
};

#define	MAX_GRID_SIZE	129

struct cGrid_t
{
	int			width;
	int			height;
	bool		wrapWidth;
	bool		wrapHeight;
	vec3_t		points[MAX_GRID_SIZE][MAX_GRID_SIZE];	// [width][height]

	void SetWrapWidth();
	void SubdivideColumns();
	void RemoveDegenerateColumns();
	void Transpose();
private:
	static bool NeedsSubdivision(vec3_t a, vec3_t b, vec3_t c);
	static void Subdivide(vec3_t a, vec3_t b, vec3_t c, vec3_t out1, vec3_t out2, vec3_t out3);
	static bool ComparePoints(const float* a, const float* b);
};

struct patchCollide_t
{
	vec3_t		bounds[2];
	int			numPlanes;			// surface planes plus edge planes
	patchPlane_t*	planes;
	int			numFacets;
	facet_t*	facets;

	void FromGrid(cGrid_t* grid);
private:
	static int FindPlane(float* p1, float* p2, float* p3);
	static int SignbitsForNormal(vec3_t normal);
	static int EdgePlaneNum(cGrid_t* grid, int gridPlanes[MAX_GRID_SIZE][MAX_GRID_SIZE][2], int i, int j, int k);
	static int GridPlane(int gridPlanes[MAX_GRID_SIZE][MAX_GRID_SIZE][2], int i, int j, int tri);
	static void SetBorderInward(facet_t* facet, cGrid_t* grid,
		int gridPlanes[MAX_GRID_SIZE][MAX_GRID_SIZE][2], int i, int j, int which);
	static int PointOnPlaneSide(const float* p, int planeNum);
	static bool ValidateFacet(facet_t* facet);
	static void AddFacetBevels(facet_t* facet);
	static bool PlaneEqual(patchPlane_t* p, float plane[4], int* flipped);
	static int FindPlane2(float plane[4], int* flipped);
	static void CM_SnapVector(vec3_t normal);
};

struct cPatch_t
{
	int			checkcount;				// to avoid repeated testings
	int			surfaceFlags;
	int			contents;
	patchCollide_t*	pc;
};

struct winding_t
{
	int			numpoints;
	vec3_t		p[4];		// variable sized
};

struct cmodel_t
{
	vec3_t		mins;
	vec3_t		maxs;
	cLeaf_t		leaf;			// submodels don't reference the main tree
};

class QClipMap46
{
private:
	static patchCollide_t* GeneratePatchCollide(int width, int height, vec3_t* points);

public:
	char		name[MAX_QPATH];
	unsigned	checksum;

	int			numShaders;
	bsp46_dshader_t*	shaders;

	int			numLeafs;
	cLeaf_t*	leafs;

	int			numLeafBrushes;
	int*		leafbrushes;

	int			numLeafSurfaces;
	int*		leafsurfaces;

	int			numPlanes;
	cplane_t	*planes;

	int			numBrushSides;
	cbrushside_t*	brushsides;

	int			numBrushes;
	cbrush_t	*brushes;

	int			numNodes;
	cNode_t*	nodes;

	int			numAreas;
	cArea_t*	areas;
	int*		areaPortals;	// [ numAreas*numAreas ] reference counts

	int			numClusters;
	int			clusterBytes;
	byte*		visibility;
	bool		vised;			// if false, visibility is just a single cluster of ffs

	int			numEntityChars;
	char*		entityString;

	int			numSurfaces;
	cPatch_t**	surfaces;			// non-patches will be NULL

	int			numSubModels;
	cmodel_t	*cmodels;

	int			floodvalid;
	int			checkcount;					// incremented on each trace

	QClipMap46()
	: checksum(0)
	, numShaders(0)
	, shaders(NULL)
	, numLeafs(0)
	, leafs(NULL)
	, numLeafBrushes(0)
	, leafbrushes(NULL)
	, numLeafSurfaces(0)
	, leafsurfaces(NULL)
	, numPlanes(0)
	, planes(NULL)
	, numBrushSides(0)
	, brushsides(NULL)
	, numBrushes(0)
	, brushes(NULL)
	, numNodes(0)
	, nodes(NULL)
	, numAreas(0)
	, areas(NULL)
	, areaPortals(NULL)
	, numClusters(0)
	, clusterBytes(0)
	, visibility(NULL)
	, vised(false)
	, numEntityChars(0)
	, entityString(NULL)
	, numSurfaces(0)
	, surfaces(NULL)
	, numSubModels(0)
	, cmodels(NULL)
	, floodvalid(0)
	, checkcount(0)
	{
		name[0] = 0;
	}
	~QClipMap46();

	void LoadMap(const char* name);
	void LoadShaders(const quint8* base, const bsp46_lump_t* l);
	void LoadLeafs(const quint8* base, const bsp46_lump_t* l);
	void LoadLeafBrushes(const quint8* base, const bsp46_lump_t* l);
	void LoadLeafSurfaces(const quint8* base, const bsp46_lump_t* l);
	void LoadPlanes(const quint8* base, const bsp46_lump_t* l);
	void LoadBrushSides(const quint8* base, const bsp46_lump_t* l);
	void LoadBrushes(const quint8* base, const bsp46_lump_t* l);
	void LoadNodes(const quint8* base, const bsp46_lump_t* l);
	void LoadVisibility(const quint8* base, const bsp46_lump_t* l);
	void LoadEntityString(const quint8* base, const bsp46_lump_t* l);
	void LoadPatches(const quint8* base, const bsp46_lump_t* surfs, const bsp46_lump_t* verts);
	void LoadSubmodels(const quint8* base, const bsp46_lump_t* l);
};

void CM46_FreeWinding(winding_t* w);
winding_t* CM46_BaseWindingForPlane(vec3_t normal, vec_t dist);
void CM46_ChopWindingInPlace(winding_t** inout, vec3_t normal, vec_t dist, vec_t epsilon);
// frees the original if clipped
void CM46_WindingBounds(winding_t* w, vec3_t mins, vec3_t maxs);
winding_t* CM46_CopyWinding(winding_t* w);

#endif
