/*
	DooM/HereTic/HeXen/StriFe INFO.c processors, by Frans P. de Vries.

Derived from:

	DooM MaP StaTistics, by Frans P. de Vries.

	You are allowed to use any parts of this code in another program, as
	long as you give credits to the authors in the documentation and in
	the program itself.  Read the file README for more information.

	This program comes with absolutely no warranty.

	PRINT.C - Common print routines
*/

#include "print.h"
#include "things.h"

#ifdef DMINFO
#include "../doom/things.h"
#include "p_pspr.h"

#elif HTINFO
#include "../heretic/things.h"

#elif HXINFO
#include "../hexen/things.h"
#include "levels.h"

#elif SFINFO
#include "../strife/things.h"
#include "p_pspr.h"
#endif

#include "maps.h"
#include "stub.h"
#include "wads.h"


/*
	the type definitions
*/
typedef struct monstdata_s {
	int objt;
	int pain;
	int hlth;
} monstdata_t;

typedef struct typedata_s {
	int objt;
	int tcls;
	int ttyp;
	char *name;
} typedata_t;

/*
	the global variables
*/
extern unsigned char rndtable[];

#define MAXFRM 40
int frames[MAXFRM];

monstdata_t **monstdata = NULL;
int nummonst = 0;

typedata_t **typedata = NULL;
int numtypes = 0;
int mobjtypes = 0;

/*
	the local function prototypes
*/
int SpriteCount( void);
char *SpriteFrames( int, Bool);
void AppendType( int, int);
void CheckTypes( void);
void ForgetTypes( void);
int SortMonst( const void *, const void *);
int SortTypes( const void *, const void *);
int SortClasses( const void *, const void *);
void PrintTypesTable( int );


void PrintSpawns( void)
{
	int i;

	for (i = 0; i < MAXSPAWN; i++)
		if (spawns[i] != -1)
		{
			if (i == 0)
				printf( "\nSpawns\t");
			else
				printf( "\n\t");
			printf( "\t%s", Obj2Str( spawns[i]));
		}
	printf( "\n");
}

void PrintSpeed( mobjinfo_t *MOBJ)
{
	int tics;

#ifdef SFINFO
	if (MOBJ->flags)
#else
	if (MOBJ->flags & MF_MISSILE)
#endif
		printf( "Speed\t\t%d / tic ; %d / sec\n",
		        MOBJ->speed / FRACUNIT, MOBJ->speed / FRACUNIT * 35);
	else
	{
		tics = FrameTics( MOBJ->seestate);
		if (tics != -1)
			printf( "Speed\t\t%d / frame ; %.1f / sec\n",
			        MOBJ->speed, MOBJ->speed * 35.0f / tics);
		else
			printf( "Speed\t\t%d\n", MOBJ->speed);
	}
}


void PrintSprite( statenum_t state, char *type)
{
	int num;
	char *frm;

	if (state == S_NULL)
		return;
	if (type != NULL)
		printf( "Sprite (%s)\t%s\n", type, sprnames[states[state].sprite]);
	else
		printf( "Sprite name\t%s\n", sprnames[states[state].sprite]);
	TallyFrames( state);
	num = SpriteCount();
	frm = SpriteFrames( num, FALSE);
	printf( "Frames\t\t%d (%s)\n", num, frm);
	FreeMemory( frm);
}

void TallyFrames( statenum_t state)
{
	statenum_t st = state;
	int f, num = 0;

	/* reset seen frames */
	for (f = 0; f < MAXFRM; f++)
		frames[f] = -1;
	/* ignore null state */
	if (state == S_NULL)
		return;

	for (;;)
	{
		/* tally this frame */
		f = states[st].frame & FF_FRAMEMASK;
		if (f >= MAXFRM)
			ProgError( "unable to count frames for state %d", state);
		frames[num++] = f;

		/* done if no next state, or same state, or looped around to earlier one */
		if (states[st].nextstate == S_NULL || states[st].nextstate <= st)
			break;
		st = states[st].nextstate;
		/* done if looped around to start state */
		if (st == state)
			break;

#ifndef SFINFO
		/* cease flashing */
		if (st == S_LIGHTDONE)
			break;
#endif

#ifdef DMINFO
		if (states[st].action.acp1)
			states[st].action.acp1(NULL);
#elif HTINFO
		if (states[st].action)
			states[st].action(NULL);
#elif HXINFO
		if (states[st].action)
			states[st].action(NULL);
#elif SFINFO
		if (states[st].action.acp1)
			states[st].action.acp1(NULL);
#endif
		else
			funcname[0] = '\0';

		/* cease firing */
#ifdef DMINFO
		if (strcmp( funcname, "A_ReFire") == 0)

#elif HTINFO
		if (strcmp( funcname, "A_ReFire") == 0 ||
		    strcmp( funcname, "A_VolcanoSet") == 0 ||     // S_VOLCANO9 loops to S_VOLCANO2, not 1
		    strcmp( funcname, "A_GenWizard") == 0 ||      // S_SOR2FX2_3 loops to S_SOR2FX2_2, not 1
		    strcmp( funcname, "A_WhirlwindSeek") == 0 ||  // S_HEADFX4_7 loops to S_HEADFX4_5, not 1
		    strcmp( funcname, "A_BeakReady") == 0)        // S_BEAKREADY loops to S_BEAKREADY

#elif HXINFO
		if (strcmp( funcname, "A_ReFire") == 0)

#elif SFINFO
		if (strcmp( funcname, "A_ReFire") == 0 ||
		    strcmp( funcname, "A_WeaponReady") == 0 ||
		    strcmp( funcname, "A_FloatWeave") == 0)
#endif
			break;
	}
}

int SpriteCount( void)
{
	int f, count = 0;

	for (f = 0; f < MAXFRM && frames[f] != -1; f++)
		count++;

	return count;
}

char *SpriteFrames( int count, Bool add)
{
	char *ret, frm[MAXFRM];
	int f;

	memset(frm, 0, MAXFRM);
	/* transpose char to actually used frame */
	for (f = 0; f < MAXFRM && frames[f] != -1; f++)
		frm[f] = 'A' + frames[f];

	/* append symbol for additional frames */
	if (add)
		if (f < MAXFRM)
			frm[f] = '+';
		else
			ProgError( "unable to append '+' after %d frames", count);

	if (strlen( frm) == 0)
		return strdup( "-");
	else
		return strdup( frm);
}


float PainPerc( int chance)
{
	int i, count = 0;

	for (i = 0; i < 256; i++)
		if (rndtable[i] < chance)
			count++;

	/* ensure x.xx50 is rounded up to x.xx+0.01 */
	return count * 100.0f / 256.0f + 0.0001f;
}

void PainChances( void)
{
	int i;

	printf( "\n");
	for (i = 0; i <= 256; i++)
		printf( "%3d\t%.2f%%\n", i, PainPerc( i));
}

int PainTics( statenum_t state)
{
	statenum_t st = state;
	int tics = 0;

	if (st != S_NULL)
		for (;;)
		{
			tics += states[st].tics;

#ifdef DMINFO
			if (states[st].action.acp1)
				states[st].action.acp1(NULL);
#elif HTINFO
			if (states[st].action)
				states[st].action(NULL);
#elif HXINFO
			if (states[st].action)
				states[st].action(NULL);
#elif SFINFO
			if (states[st].action.acp1)
				states[st].action.acp1(NULL);
#endif
			else
				funcname[0] = '\0';

#ifdef DMINFO
			if (strcmp( funcname, "A_Pain") == 0 ||
			    strcmp( funcname, "A_BrainPain") == 0)

#elif HTINFO
			if (strcmp( funcname, "A_Pain") == 0 ||
			    strcmp( funcname, "A_PodPain") == 0 ||
			    strcmp( funcname, "A_ChicPain") == 0 ||
			    strcmp( funcname, "A_Sor1Pain") == 0)

#elif HXINFO
			if (strcmp( funcname, "A_Pain") == 0 ||
			    strcmp( funcname, "A_PigPain") == 0 ||
			    strcmp( funcname, "A_DragonPain") == 0 ||
			    strcmp( funcname, "A_BishopPainBlur") == 0 ||
			    strcmp( funcname, "A_BounceCheck") == 0)

#elif SFINFO
			if (strcmp( funcname, "A_Pain") == 0 ||
			    strcmp( funcname, "A_MerchantPain") == 0 ||
			    strcmp( funcname, "A_ZombieInSpecialSector") == 0 ||
			    strcmp( funcname, "A_HideZombie") == 0 ||
			    strcmp( funcname, "A_CheckTargetVisible") == 0)
#endif
				break;

			st = states[st].nextstate;
			if (tics > 30)
				ProgError( "unable to count tics for painstate %d", state);
		}

	return tics;
}


void MonstersPcHp( void)
{
	mobjinfo_t *MOBJ;
	monstdata_t *mnst;

	for (Objt = 0; Objt < NUMMOBJTYPES; Objt++)
		/* skip player */
#ifdef DMINFO
		if (Objt != MT_PLAYER) {
#elif HTINFO
		if (Objt != MT_PLAYER && Objt != MT_CHICPLAYER) {
#elif HXINFO
		if (Objt != MT_PLAYER_FIGHTER && Objt != MT_PLAYER_CLERIC &&
		    Objt != MT_PLAYER_MAGE && Objt != MT_PIGPLAYER) {
#elif SFINFO
		if (Objt != MT_PLAYER) {
#endif
			MOBJ = &mobjinfo[Objt];
			if (MOBJ->painchance != 0

#ifdef DMINFO
			    /* add Explosive barrel */
			    || Objt == MT_BARREL

#elif HTINFO
			    /* add Pod */
			    || Objt == MT_POD

#elif HXINFO
			    /* add Destructible tree, Evergreen tree, Sitting corpse, Shrubs,
			       Pots, Suit of armor, Mushroom */
			    || Objt == MT_TREEDESTRUCTIBLE || Objt == MT_ZXMAS_TREE
			    || Objt == MT_MISC78 || Objt == MT_ZSHRUB1 || Objt == MT_ZSHRUB2
			    || Objt == MT_POTTERY1 || Objt == MT_POTTERY2 || Objt == MT_POTTERY3
			    || Objt == MT_ZSUITOFARMOR || Objt == MT_ZPOISONSHROOM

#elif SFINFO
			    /* add Ceiling turret, Exploding barrel, Wooden barrel */
			    || Objt == MT_TURRET || Objt == MT_MISC_06 || Objt == MT_MISC_05
#endif
			   )
			{
				mnst = (monstdata_t *) GetMemory( sizeof(monstdata_t));
				mnst->objt = Objt;
				mnst->pain = MOBJ->painchance;
				mnst->hlth = MOBJ->spawnhealth;
				monstdata = ResizeMemory( monstdata, (nummonst+1) * sizeof(mnst));
				monstdata[nummonst++] = mnst;
			}
		}

	qsort( monstdata, nummonst, sizeof(monstdata_t *), SortMonst);

	printf( "\n");
	for (Objt = 0; Objt < nummonst; Objt++)
	{
		printf( "%-20s\t%d (%.2f%%)\t%d\n",
		        Monst2Str( monstdata[Objt]->objt),
		        monstdata[Objt]->pain, PainPerc( monstdata[Objt]->pain),
		        monstdata[Objt]->hlth);
	}

	for (Objt = 0; Objt < nummonst; Objt++)
		if (monstdata[Objt] != NULL)
			FreeMemory( monstdata[Objt]);
	FreeMemory( monstdata);
	monstdata = NULL;
}

/* sort by decreasing pain chaince, decreasing hit points, alphabetic name */
int SortMonst( const void *p1, const void *p2)
{
	monstdata_t *m1 = * (monstdata_t * const *) p1,
	            *m2 = * (monstdata_t * const *) p2;

	if (m1->pain < m2->pain)
		return 1;
	else if (m1->pain > m2->pain)
		return -1;

	if (m1->hlth < m2->hlth)
		return 1;
	else if (m1->hlth > m2->hlth)
		return -1;

	return strcmp( Monst2Str( m1->objt),
	               Monst2Str( m2->objt));
}


void InitTypes( void)
{
	mobjinfo_t *MOBJ;
	typedata_t *type;

#ifdef DMINFO
	InitDoomThings();
#elif HTINFO
	InitHereticThings();
#elif HXINFO
	InitHexenThings();
#elif SFINFO
	InitStrifeThings();
#endif

	/* collect all mobjs with a Thing type */
	for (Objt = 0; Objt < NUMMOBJTYPES; Objt++)
	{
		MOBJ = &mobjinfo[Objt];
		if (MOBJ->doomednum != -1)
			AppendType( Objt, MOBJ->doomednum);
	}
	/* remember mobj count with doomednum */
	mobjtypes = numtypes;

	/* add various starting spots which are not in mobjinfo array */
#ifdef HXINFO
	AppendType( MT_PLAYER_FIGHTER, THING_PLAYER1);
	AppendType( MT_PLAYER_FIGHTER, THING_PLAYER2);
	AppendType( MT_PLAYER_FIGHTER, THING_PLAYER3);
	AppendType( MT_PLAYER_FIGHTER, THING_PLAYER4);
	AppendType( MT_PLAYER_FIGHTER, THING_PLAYER5X);
	AppendType( MT_PLAYER_FIGHTER, THING_PLAYER6X);
	AppendType( MT_PLAYER_FIGHTER, THING_PLAYER7X);
	AppendType( MT_PLAYER_FIGHTER, THING_PLAYER8X);
#else
	AppendType( MT_PLAYER, THING_PLAYER1);
	AppendType( MT_PLAYER, THING_PLAYER2);
	AppendType( MT_PLAYER, THING_PLAYER3);
	AppendType( MT_PLAYER, THING_PLAYER4);
#endif
	AppendType( MT_TELEPORTMAN, THING_DEATHMATCH);

#ifdef HTINFO
	/* add teleport spot */
	AppendType( MT_TELEPORTMAN, THING_DSPARILSPOT);

	/* add non-mobj sound things (besides Wind & Waterfall) */
	AppendType( MT_SOUNDWIND, THING_SNDBELSS);
	AppendType( MT_SOUNDWIND, THING_SNDDROPS);
	AppendType( MT_SOUNDWIND, THING_SNDFASTFOOT);
	AppendType( MT_SOUNDWIND, THING_SNDGROWL);
	AppendType( MT_SOUNDWIND, THING_SNDHEARTBEAT);
	AppendType( MT_SOUNDWIND, THING_SNDLAUGHTER);
	AppendType( MT_SOUNDWIND, THING_SNDMAGIC);
	AppendType( MT_SOUNDWIND, THING_SNDSCREAM);
	AppendType( MT_SOUNDWIND, THING_SNDSLOWFOOT);
	AppendType( MT_SOUNDWIND, THING_SNDSQUISH);
#endif

#ifdef HXINFO
	/* add polyobjects */
	AppendType( MT_MAPSPOT, THING_POLYANCHOR);
	AppendType( MT_MAPSPOT, THING_POLYSPAWN);
	AppendType( MT_MAPSPOT, THING_POLYSPAWNCR);

	/* add non-mobj sound things (besides Wind) */
	AppendType( MT_SOUNDWIND, THING_SNDCREAK);
	AppendType( MT_SOUNDWIND, THING_SNDEARTH);
	AppendType( MT_SOUNDWIND, THING_SNDHEAVY);
	AppendType( MT_SOUNDWIND, THING_SNDICE);
	AppendType( MT_SOUNDWIND, THING_SNDLAVA);
	AppendType( MT_SOUNDWIND, THING_SNDMETAL);
	AppendType( MT_SOUNDWIND, THING_SNDMETAL2);
	AppendType( MT_SOUNDWIND, THING_SNDSILENCE);
	AppendType( MT_SOUNDWIND, THING_SNDSTONE);
	AppendType( MT_SOUNDWIND, THING_SNDWATER);
#endif

#if defined(SFINFO) && defined(MOBJTRANSLATION)
	/* add Veteran Ed. spots */
	AppendType( MT_PLAYER, THING_TEAMBLUE);
	AppendType( MT_PLAYER, THING_TEAMRED);
#endif

	CheckTypes();
}

void AppendType( int objt, int dedn)
{
	typedata_t *type;

	type = (typedata_t *) GetMemory( sizeof(typedata_t));
	type->objt = objt;
	type->ttyp = dedn;
	type->tcls = LookupThingClass( dedn);
	type->name = LookupThing( dedn, FALSE);
	typedata = ResizeMemory( typedata, (numtypes+1) * sizeof(type));
	typedata[numtypes++] = type;
}

void CheckTypes( void)
{
	int i, j;
	Bool found;

	/* check that all Thing items have been included in mobj types list */
	for (i = 0; i < NumThItems; i++)
	{
		found = FALSE;
		for (j = 0; j < numtypes; j++)
			if (ThItems[i]->type == typedata[j]->ttyp)
			{
				found = TRUE;
				break;
			}
		if (!found)
			printf( "\tNo mobj found for Thing type: %d\n", ThItems[i]->type);
	}
}

void ForgetTypes( void)
{
	for (Objt = 0; Objt < numtypes; Objt++)
		if (typedata[Objt] != NULL)
			FreeMemory( typedata[Objt]);
	FreeMemory( typedata);
	typedata = NULL;
}


/* allocates the returned string which the caller must free */
char *LookupFlags( int flags, char *name)
{
	char flagstr[6];
	int flagidx = 0;

	memset(flagstr, 0, sizeof(flagstr));

	/* check for player/team starts, which have no flags */
	if ((strncmp(name, "Player ", strlen("Player ")) != 0
	     && strncmp(name, "Team ", strlen("Team ")) != 0) ||
	    strstr(name, " start") == NULL)
	{
		if (IsWeapon( name))
			flagstr[flagidx++] = 'W';

#if defined(DMINFO) || defined(HTINFO)
		if (flags & MF_COUNTITEM)
			flagstr[flagidx++] = 'A';
#endif

		if (flags & MF_SPECIAL
#ifdef SFINFO
		    /* enemies with special flag cannot be picked up */
		    && strcmp(name, "Entity") != 0
		    && strncmp(name, "Spectre", strlen( "Spectre")) != 0
		    /* VE flag spots with special flag cannot be picked up */
		    && strstr(name, "flag spot") == NULL
#endif
		   )
			flagstr[flagidx++] = 'P';

		if (flags & MF_COUNTKILL
#ifdef DMINFO
		    /* no count kill flag but still a monster */
		    || strcmp(name, "Lost soul") == 0
#endif
		   )
			flagstr[flagidx++] = 'M';

		if (flags & MF_SOLID)
			flagstr[flagidx++] = 'O';

		if (flags & MF_SHOOTABLE)
			flagstr[flagidx++] = '*';

		if ((flags & MF_SPAWNCEILING) || (flags & MF_FLOAT))
			flagstr[flagidx++] = '^';
	}

	return strdup( flagstr);
}


void ThingTypes( void)
{
	InitTypes();
	qsort( typedata, numtypes, sizeof(typedata_t *), SortTypes);

	PrintTypesTable( -1);

#if defined(SFINFO) && defined(MOBJTRANSLATION)
	/* add Veteran Ed. table */
	printf("===%s===\n", LookupClassName( ThClassSnds));
	PrintTypesTable( ThClassSnds);
#endif

	printf("Totals: '''%d''' editable things, out of %d mobjs and %d non-mobjs.\n",
	       numtypes, NUMMOBJTYPES, numtypes - mobjtypes);
	ForgetTypes();
	ForgetThings();
}

/* sort by increasing Thing type */
int SortTypes( const void *p1, const void *p2)
{
	typedata_t *t1 = * (typedata_t * const *) p1,
	           *t2 = * (typedata_t * const *) p2;

	if (t1->ttyp > t2->ttyp)
		return 1;
	else if (t1->ttyp < t2->ttyp)
		return -1;
	else // typ1 == typ2
		return 0; // this shouldn't happen anyway
}

void ClassTypes( void)
{
	int class;

	InitTypes();
	qsort( typedata, numtypes, sizeof(typedata_t *), SortClasses);

	/* process all classes */
	printf( "\n");
	for (class = 0; class < ThClass_End; class++)
#if defined(SFINFO) && defined(MOBJTRANSLATION)
		if (class != ThClassSnds) // skip Veteran Ed.
#endif
		{
			printf("===%s===\n", LookupClassName( class));
			PrintTypesTable( class);
		}

#if defined(SFINFO) && defined(MOBJTRANSLATION)
	/* add Veteran Ed. table */
	printf("===%s===\n", LookupClassName( ThClassSnds));
	PrintTypesTable( ThClassSnds);
#endif

	ForgetTypes();
	ForgetThings();
}

/* sort by Thing name */
int SortClasses( const void *p1, const void *p2)
{
	typedata_t *t1 = * (typedata_t * const *) p1,
	           *t2 = * (typedata_t * const *) p2;

	if (t1->tcls > t2->tcls)
		return 1;
	else if (t1->tcls < t2->tcls)
		return -1;
	else // cls1 == cls2
		return strcmp(t1->name, t2->name);
}


void PrintTypesTable( int class)
{
	typedata_t *type;
	statenum_t state;
	WadPtr curw;
	int num, lmp, wad, sprwad;
	char *frm, *flgs, *vers;
	Bool foundspr, foundlvl;

	/* output table header */
	printf( "\n{| {{prettySortable}}\n"
	        "! Decimal\n"
	        "! Hex\n"
	        "! Version\n"
	        "! Radius\n"
	        "! Height\n"
	        "! Sprite\n"
	        "! Sequence\n"
	        "! Class\n"
	        "! Description\n");

	/* output table with 8 columns */
	for (Objt = 0; Objt < numtypes; Objt++)
	{
		type = typedata[Objt];
		if (class != -1 && class != type->tcls)
			continue;

		state = mobjinfo[type->objt].spawnstate;

#ifdef HXINFO
		/* in Hexen's info.c, the Maulotaur mobjinfo entry specifies doomednum 9,
		   but the Maulotaur is only summonable and 9 is also used for a mossy rock */
		if (type->ttyp == THING_MOSSYROCKMED &&
		    strcmp("MNTR", sprnames[states[state].sprite]) == 0)
			continue;
		/* also, the Waterfall sound mobjinfo entry specifies doomednum 41, but this
		   is a Heretic remnant not used in Hexen, and 41 is also used for a mushroom */
		if (type->ttyp == THING_MEDIUMMUSHROOM &&
		    strcmp("TLGL", sprnames[states[state].sprite]) == 0)
			continue;
#endif

#ifdef SFINFO
		/* in Strife's info.c, a bloodied corpse mobjinfo entry specifies doomednum 54,
		   but that object is not placeable and 54 is also used for the Aztec pillar */
		if (type->ttyp == THING_AZTECPILLAR &&
		    strcmp("DEAD", sprnames[states[state].sprite]) == 0)
			continue;

#ifdef MOBJTRANSLATION
		/* skip Veteran Ed. if printing all types by number */
		if (class == -1 && type->tcls == ThClassSnds)
			continue;
#endif
#endif

		flgs = LookupFlags( mobjinfo[type->objt].flags, type->name);

		/* check for things not using a sprite at all */
		if (type->ttyp == THING_DEATHMATCH || type->ttyp == THING_TELEPORT

#ifdef DMINFO
		    || type->ttyp == THING_SPAWNSPOT || type->ttyp == THING_BOSSSHOOT

#elif HTINFO
		    || type->ttyp == THING_DSPARILSPOT || type->ttyp == THING_PODGENERATOR
		    || type->tcls == ThClassSnds

#elif HXINFO
		    || type->ttyp == THING_BATSPAWNER || type->ttyp == THING_FOGSPAWNER
		    || type->ttyp == THING_LEAFSPAWNER || type->ttyp == THING_MAPSPOT
		    || type->ttyp == THING_MAPSPOTGRAV || type->ttyp == THING_POLYANCHOR
		    || type->ttyp == THING_POLYSPAWN || type->ttyp == THING_POLYSPAWNCR
		    || type->tcls == ThClassSnds

#elif SFINFO
		    || type->ttyp == THING_ORESPAWNER || type->ttyp == THING_ZOMBIESPAWN
#endif
		   )
		{
			frm = NULL;
		}
		else
		{
			TallyFrames( state);
			num = SpriteCount();
			frm = SpriteFrames( num, FALSE);
		}

		/* count sprite prefix in each master directory (until found) and
		   thing in levels of each IWAD */
		wad = 0;
		sprwad = -1;
		foundspr = FALSE;
		curw = WadFileList;
		while (curw)
		{
			/* imply sprite-less thing is found */
			if (!frm)
			{
				foundspr = TRUE;
				sprwad = wad;
			}
			/* if sprite was already found in a more limited IWAD, it'll be
			   in the full IWAD too */
			else if (!foundspr)
			{
				/* search this IWAD's sprites section */
				lmp = CountSpriteLumps( curw->masterdir, sprnames[states[state].sprite]);
				if (lmp != 0)
				{
					foundspr = TRUE;
					sprwad = wad;
					if (Verb)
						fprintf( stderr, "%s:\t%5d\tsprite %s\n",
						         curw->filename, type->ttyp, sprnames[states[state].sprite]);

					/* check for additional sprites beyond default pattern */
					if (lmp > num
					    /* decorations are not interactive, but often use one out of a series */
					    && type->tcls != ThClassDeco

#ifdef HXINFO
					    /* these lit/unlit objects use one/multiple out of a series */
					    && type->ttyp != THING_CAULDRONLIT && type->ttyp != THING_CAULDRONULT
					    && type->ttyp != THING_MINOTSTATLIT && type->ttyp != THING_MINOTSTATULT
					    && type->ttyp != THING_TWINEDTORCHLIT && type->ttyp != THING_TWINEDTORCHULT

#elif SFINFO
					    /* plenty obstacles use one out of a series, but shootable barrels are animated */
					    && (type->tcls != ThClassObst ||
					        type->ttyp == THING_BARRELSTRIFE || type->ttyp == THING_WOODENBARREL)
					    /* (broken) power coupling use 1-2 out of 3 */
					    && type->ttyp != THING_POWERCOUPL && type->ttyp != THING_BRKNPWRCOUPL
					    /* some weapons use one/multiple out of a series */
					    && type->tcls != ThClassWeap
#endif
					   )
					{
						FreeMemory( frm);
						/* mark sequence as interactive with a '+' */
						frm = SpriteFrames( num, TRUE);
					}
				}
			}

			/* search this IWAD's levels */
			foundlvl = SearchLevels( curw,
				(curw->gamevers == 0x02 || curw->gamevers >= 0x20) ? "MAP##" : "E#M#",
				type->ttyp);

			/* done if sprite and thing are found in IWAD */
			if (foundspr && foundlvl)	
				break;

			wad++;
			curw = curw->next;
		}

		/* set IWAD version */
		switch (wad)
		{
#ifdef DMINFO
			case 0: vers = "S"; break; // shareware
			case 1: vers = "R"; break; // registered
			case 2: vers = "2"; break; // II

#elif HTINFO
			case 0: vers = "S"; break; // shareware
			case 1: vers = "R"; break; // registered
			case 2: vers = "X"; break; // SSR eXpansion: registered + more maps

#elif HXINFO
			case 0: vers = "D"; break; // demo
			case 1: vers = "C"; break; // commercial
			case 2: vers = "X"; break; // DDC eXpansion: no sprites, only maps

#elif SFINFO
			case 0: vers = "D"; break; // demo
			case 1: vers = "C"; break; // commercial
#ifdef MOBJTRANSLATION
			case 2: vers = "V"; break; // Veteran Ed: has only sprites not in commercial
#endif
#endif

			default:
				vers = "-";

#ifdef HXINFO
				/* present in demo but only in coop mode */
				if (type->ttyp == THING_MYSTAMBINC)
				{
					vers = "D<sup>#</sup>";
				}
				/* all sounds present in demo */
				if (type->tcls == ThClassSnds)
				{
					vers = "D<sup>#</sup>";
				}
				else
				{
					/* if sprite was found but thing not in levels, use sprite IWAD */
					if (foundspr && !foundlvl)
						switch (sprwad)
						{
							case 0: vers = "D<sup>#</sup>"; break;
							case 1: vers = "C<sup>#</sup>"; break;
							default: vers = "-";
						}
				}

#elif SFINFO
				/* all player, acolyte, peasant & misc. sprites present in demo */
				if (strcmp(sprnames[states[state].sprite], "PLAY") == 0 ||
				    strcmp(sprnames[states[state].sprite], "AGRD") == 0 ||
				    strcmp(sprnames[states[state].sprite], "PEAS") == 0 ||
				    type->ttyp == THING_ROCK1 || type->ttyp == THING_STALAGMSML ||
				    type->ttyp == THING_ROCK2 || type->ttyp == THING_BARRCOLUMN ||
				    type->ttyp == THING_ROCK4 || type->ttyp == THING_OUTSIDELAMP ||
				    type->ttyp == THING_RUBBLE6 || type->ttyp == THING_RUBBLE8 ||
				    type->ttyp == THING_TRAY)
				{
					vers = "D<sup>#</sup>";
				}
				/* enemies, weapons & keys are spawned if not present in levels */
				else if (type->tcls == ThClassMnst || type->tcls == ThClassWeap ||
				         type->tcls == ThClassKeys)
				{
					vers = "C";
				}
				/* obstacle & decoration sprites present in commercial */
				else
				{
					vers = "C<sup>#</sup>";
				}
#endif
		}

		printf( "|-\n"
		        "| align=\"right\" | %d\n"
		        "| align=\"right\" | %X\n"
		        "| align=\"center\" | %s\n"
		        "| align=\"right\" | %d\n"
		        "| align=\"right\" | %d\n"
		        "| %s\n"
		        "| %s\n"
		        "| %s\n"
		        "| %s\n",

			type->ttyp, type->ttyp, vers,
			mobjinfo[type->objt].radius / FRACUNIT,
			mobjinfo[type->objt].height / FRACUNIT,
			(!frm ? "''none''" : sprnames[states[state].sprite]),
			(!frm ? "-" : frm),
			flgs, LookupThing( type->ttyp, TRUE));

		if (frm)
			FreeMemory( frm);
		FreeMemory( flgs);
	}

	/* output table footer */
	printf( "|}\n\n");
}

/* vim:set noexpandtab tabstop=2 softtabstop=2 shiftwidth=2: */
