/*
	DooM MaP StaTistics, by Frans P. de Vries.

Derived from:

	DooM PostScript Maps Utility, by Frans P. de Vries.

And thus from:

	Doom Editor Utility, by Brendon Wyber and Raphaël Quinet.

	You are allowed to use any parts of this code in another program, as
	long as you give credits to the authors in the documentation and in
	the program itself.  Read the file README for more information.

	This program comes with absolutely no warranty.

	STATS.C - Statistics routines.
*/

#include "dmmpst.h"
#include "stats.h"
#include "levels.h"
#include "things.h"


/*
	map attribute definitions
*/
#define L_TELEP1 39        /* one-time teleport linedef; all in Doom format */
#define L_TELEPR 97        /* repeatable teleport linedef */
#define L_TELPM1 125       /* one-time monster-only teleport linedef */
#define L_TELPMR 126       /* repeatable monster-only teleport linedef */
#define S_SECRET 9         /* secret sector, not in Doom 64 */
#define D64TC_SECR 994     /* Doom 64 TC: secret linedef */
#define D64TC_SHRN 995     /* Doom 64 TC: shrine linedef */


/*
	the local variables
*/
Bool splitDM[ThClass_End];
Bool extraCP[ThClass_End];

/*
	the local function prototypes
*/
TMPL_varlist *tmpl_add_names( TMPL_varlist *, char *);
TMPL_varlist *tmpl_add_slot( TMPL_varlist *, BCINT, BCINT);
TMPL_varlist *tmpl_add_number( TMPL_varlist *, BCINT);
TMPL_varlist *tmpl_add_secrets( TMPL_varlist *, int, int *);
TMPL_varlist *tmpl_add_thgsecrets( TMPL_varlist *, int, int *);
TMPL_varlist *tmpl_add_linesecrets( TMPL_varlist *, int, int *, int, int *);
TMPL_varlist *tmpl_add_pscount( TMPL_varlist *, char *, BCINT);
TMPL_varlist *tmpl_add_psdir( TMPL_varlist *, char *, BCINT);
TMPL_varlist *tmpl_add_spawns( TMPL_varlist *);
TMPL_varlist *tmpl_add_stats( TMPL_varlist *);
TMPL_varlist *tmpl_add_things( TMPL_varlist *);
void SearchTriggering( int);


/*
	read level & analyze its statistics
*/
void AnalyzeLevel( BCINT episode, BCINT mission)
{
	TMPL_varlist *tpldata = NULL;
	char *wktpfile = NULL;
	int *secsecrets = NULL, *thgsecrets = NULL;
	int *d64tcsecrets = NULL, *d64tcshrines = NULL;
	int n, m, gensec, secretsectors = 0, secretthings = 0, telept = 0,
	    d64tcsecrcnt = 0, d64tcshrncnt = 0;
	BCINT lumpver;
	BCLNG k;

	/* read data of this level */
	lumpver = ReadLevelData( episode, mission);

	/* count secret Sectors & Things and teleporting LineDefs */
	secretsectors = CountSecrets( lumpver);
	if (!WD64TC && (UDMFmap || (GameVersion & 0x80) == 0x80)) // UDMF & Doom 64 formats
		secretthings = CountSecretThings();
	if ((GameVersion & 0x40) != 0x40) // skip for Hexen format
		for (n = 0; n < NumLineDefs; n++)
			if (LineDefs[n].type == L_TELEP1 || LineDefs[n].type == L_TELEPR)
				telept++;

	/* collect list of secret Sectors */
	if (secretsectors > 0)
	{
		if ((lumpver & 0x80) == 0x80) // Doom64 format
			gensec = D64_GENSEC;
		else if (UDMFmap || (lumpver & 0x40) == 0x40) // UDMF & Hexen formats
			gensec = B_GENSECH;
		else // Doom format
			gensec = B_GENSEC;
		secsecrets = (int *) GetMemory( secretsectors * sizeof(int));
		m = 0;
		for (n = 0; n < NumSectors; n++)
			if (Sectors[n].special == S_SECRET && (lumpver & 0x60) != 0x60 && // not Strife Hexen-format
			    (lumpver & 0x80) != 0x80 && !UDMFmap) // not Doom64 / UDMF
			{
				secsecrets[m] = n;
				m++;
			}
			else if (!WD64TC && (Sectors[n].special & gensec) == gensec) // or generalized but not Doom 64 TC
			{
				secsecrets[m] = -n;
				m++;
			}
	}

	/* collect list of Doom 64 secret Things */
	if (secretthings > 0)
	{
		thgsecrets = (int *) GetMemory( secretthings * sizeof(int));
		m = 0;
		for (n = 0; n < NumThings; n++)
			if ((UDMFmap && Things[n].secret) ||
			    (!UDMFmap && (Things[n].when & TF_D64SEC) != 0))
			{
				thgsecrets[m] = n;
				m++;
			}
	}

	/* collect list of Doom 64 TC secret/shrine Linedefs */
	if (WD64TC)
	{
		for (n = 0; n < NumLineDefs; n++)
			if (LineDefs[n].type == D64TC_SECR)
				d64tcsecrcnt++;
			else if (LineDefs[n].type == D64TC_SHRN)
				d64tcshrncnt++;

		if (d64tcsecrcnt > 0)
		{
			d64tcsecrets = (int *) GetMemory( d64tcsecrcnt * sizeof(int));
			m = 0;
			for (n = 0; n < NumLineDefs; n++)
				if (LineDefs[n].type == D64TC_SECR)
				{
					d64tcsecrets[m] = n;
					m++;
				}
		}

		if (d64tcshrncnt > 0)
		{
			d64tcshrines = (int *) GetMemory( d64tcshrncnt * sizeof(int));
			m = 0;
			for (n = 0; n < NumLineDefs; n++)
				if (LineDefs[n].type == D64TC_SHRN)
				{
					d64tcshrines[m] = n;
					m++;
				}
		}
	}

	/* check and warn if Hexen-format flags exist but aren't counted */
	if (lumpver != GameVersion && !WBoomMP)
		printf( "Warning: level's Things contain Hexen-format flags.  Use option '+b'.\n");
	/* check and warn if Boom MP flags exist but aren't counted */
	else if (CheckThingBoomFlags() && !WBoomMP && !WD64TC)
		printf( "Warning: level's Things contain Boom MP flags.  Use option '+b'.\n");

	/* initialize and analyze thing statistics */
	switch (GameVersion & 0xF0)
	{
		case 0x00: // fall thru
		case 0x80: InitDoomThings(); break;
		case 0x10: InitHereticThings(); break;
		case 0x20: InitStrifeThings( mission); break;
		case 0x40: InitHexenThings(); break;
	}
	if (CustomMap != NULL)
		ReadCustomThings();
	CheckUDMFall();
	AnalyzeThings( lumpver, TRUE);

	/* check for ZDoom secret trigger */
	for (n = 0; n < NumThings; n++)
		if (Things[n].type == THING_Z_SECTRIG)
			SearchTriggering( n);

	/* print verbose statistics */
	if (Verbose)
	{
		printf( "\n");
		if (GameVersion == 0x02 || GameVersion >= 0x20)
		{
			switch (GameVersion)
			{
				case 0x02: printf( "# DOOM II"); break;
				case 0x20: printf( "# Strife Demo"); break;
				case 0x22: printf( "# Strife Full"); break;
				case 0x40: printf( "# Hexen Demo"); break;
				case 0x42: printf( "# Hexen Full"); break;
				case 0x48: printf( "# Hexen: DotDC"); break;
				case 0x82: printf( "# Doom 64"); break;
				case 0x88: printf( "# Doom 64: The Lost Levels"); break;
				default: ProgError( "unsupported game version (%02x)", GameVersion);
			}
			printf( "  Map statistics for level %s%02d: %s\n\n",
			        ( UDMFmark != NULL ? UDMFmark : "MAP" ), mission,
			        ( UserLvlNm != NULL ? UserLvlNm : LevelName ));
		}
		else // < 0x20
		{
			switch (GameVersion)
			{
				case 0x00: printf( "# Shareware DOOM"); break;
				case 0x01: printf( "# Registered DOOM"); break;
				case 0x04: printf( "# Ultimate DOOM"); break;
				case 0x10: printf( "# Shareware Heretic"); break;
				case 0x11: printf( "# Registered Heretic"); break;
				case 0x14: printf( "# Heretic: SotSR"); break;
				default: ProgError( "unsupported game version (%02x)", GameVersion);
			}
			if (UDMFmark != NULL)
				printf( "  Map statistics for level %s%d: %s\n\n", UDMFmark, mission,
				        ( UserLvlNm != NULL ? UserLvlNm : LevelName ));
			else
				printf( "  Map statistics for level E%dM%d: %s\n\n", episode, mission,
				        ( UserLvlNm != NULL ? UserLvlNm : LevelName ));
		}

		printf( "MapDim:\t  MaxX\t  MinX\t SizeX\t  MaxY\t  MinY\t SizeY\n");
		printf( "\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\n",
		        MapMaxX, MapMinX, MapMaxX - MapMinX, MapMaxY, MapMinY, MapMaxY - MapMinY);

		printf( "\n");
		printf( "Struct:\tThings\tVertxs\tBefore\tLinDfs\tTelePt\tSidDfs\tSectrs\tSecret\n");
		printf( "\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\n",
		        NumThings, TotVertexes, NumVertexes, NumLineDefs, telept, NumSideDefs, NumSectors,
		        (WD64TC ? d64tcsecrcnt : secretsectors));
		printf( "\n");

		/* print list of secret sectors */
		if (secretsectors > 0)
		{
			printf( "Secrets:");
			for (n = 0; n < secretsectors; n++)
			{
				if (secsecrets[n] >= 0)
					printf( "%6d\t", secsecrets[n]);
				else
					if ((GameVersion & 0x80) == 0x80)
						printf( "%6d\t", -secsecrets[n]);
					else
						printf( "%4d:B\t", -secsecrets[n]);
				if (((n+1) % 8) == 0 && (n+1) < secretsectors)
					printf( "\n\t");
			}
			if (secretthings > 0)
				printf( "\n");
			else
				printf( "\n\n");
		}

		/* print list of secret things */
		if (secretthings > 0)
		{
			printf( "SecThgs:");
			for (n = 0; n < secretthings; n++)
			{
				printf( "%6d\t", thgsecrets[n]);
				if (((n+1) % 8) == 0 && (n+1) < secretthings)
					printf( "\n\t");
			}
			printf( "\n\n");
		}

		/* print list of Doom 64 TC secret linedefs */
		if (d64tcsecrcnt > 0)
		{
			printf( "SecrLns:");
			for (n = 0; n < d64tcsecrcnt; n++)
			{
				printf( "%6d\t", d64tcsecrets[n]);
				if (((n+1) % 8) == 0 && (n+1) < d64tcsecrcnt)
					printf( "\n\t");
			}
			if (d64tcshrncnt > 0)
				printf( "\n");
			else
				printf( "\n\n");
		}

		/* print list of Doom 64 TC shrine linedefs */
		if (d64tcshrncnt > 0)
		{
			printf( "ShrnLns:");
			for (n = 0; n < d64tcshrncnt; n++)
			{
				printf( "%6d\t", d64tcshrines[n]);
				if (((n+1) % 8) == 0 && (n+1) < d64tcshrncnt)
					printf( "\n\t");
			}
			printf( "\n\n");
		}
	}

	/* collect classes for split DM table */
	if (WSplitDM > 0)
	{
		for (n = 0; n < ThClass_End; n++)
			splitDM[n] = FALSE;
		for (k = WSplitDM; k > 0; k = k / 10)
		{
			splitDM[k % 10 - 1] = TRUE;
		}
		//for (n = 0; n < ThClass_End; n++)
		//	printf( "class %d: %s\n", n, (splitDM[n] ? "2nd" : "1st"));
	}

	/* output DoomWiki file */
	if (WkFile != NULL)
	{
		/* process aggregate values */
		AggregateSpecials();

		if (WSkeleton)
		{
			/* add map attributes */
			tpldata = tmpl_add_names( tpldata, (UserLvlNm != NULL ? UserLvlNm : LevelName));
			tpldata = tmpl_add_slot( tpldata, episode, mission);
			tpldata = tmpl_add_number( tpldata, mission);

			/* add secrets.tpl values */
			if (secretsectors > 0)
				tpldata = tmpl_add_secrets( tpldata, secretsectors, secsecrets);
			if (secretthings > 0)
				tpldata = tmpl_add_thgsecrets( tpldata, secretthings, thgsecrets);
			if (d64tcsecrcnt > 0 || d64tcshrncnt > 0)
				tpldata = tmpl_add_linesecrets( tpldata, d64tcsecrcnt, d64tcsecrets, d64tcshrncnt, d64tcshrines);
			if (secretsectors == 0 && secretthings == 0 && d64tcsecrcnt == 0 && d64tcshrncnt == 0)
				tpldata = TMPL_add_var( tpldata, "NOSECRETS", "NO", NULL);

			/* add dmspawns.tpl/ctfspawns.tpl, mapdata.tpl & things.tpl values */
			tpldata = tmpl_add_spawns( tpldata);
			tpldata = tmpl_add_stats( tpldata);
			tpldata = tmpl_add_things( tpldata);

			/* define and fill DoomWiki template */
			if (asprintf( &wktpfile, "%s/%s", (WkTpPath != NULL ? WkTpPath : "."), "skeleton.tpl") == -1)
				ProgError( "unable to concatenate template path");
			TMPL_write( wktpfile, NULL, NULL, tpldata, WkFile, stderr);
			TMPL_free_varlist( tpldata);
			tpldata = NULL;
			FreeMemory( wktpfile);
			wktpfile = NULL;
		}

		else if (WSecrets)
		{
			/* add secrets.tpl values */
			if (secretsectors > 0)
				tpldata = tmpl_add_secrets( tpldata, secretsectors, secsecrets);
			if (secretthings > 0)
				tpldata = tmpl_add_thgsecrets( tpldata, secretthings, thgsecrets);
			if (d64tcsecrcnt > 0 || d64tcshrncnt > 0)
				tpldata = tmpl_add_linesecrets( tpldata, d64tcsecrcnt, d64tcsecrets, d64tcshrncnt, d64tcshrines);
			if (secretsectors == 0 && secretthings == 0 && d64tcsecrcnt == 0 && d64tcshrncnt == 0)
				tpldata = TMPL_add_var( tpldata, "NOSECRETS", "NO", NULL);

			/* define and fill DoomWiki template */
			if (asprintf( &wktpfile, "%s/%s", (WkTpPath != NULL ? WkTpPath : "."), "secrets.tpl") == -1)
				ProgError( "unable to concatenate template path");
			TMPL_write( wktpfile, NULL, NULL, tpldata, WkFile, stderr);
			TMPL_free_varlist( tpldata);
			tpldata = NULL;
			FreeMemory( wktpfile);
			wktpfile = NULL;
		}

		else if (WDmSpawns)
		{
			/* add dmspawns.tpl/ctfspawns.tpl values */
			tpldata = tmpl_add_spawns( tpldata);

			/* define and fill DoomWiki template */
			if (asprintf( &wktpfile, "%s/%s", (WkTpPath != NULL ? WkTpPath : "."), "dmspawns.tpl") == -1)
				ProgError( "unable to concatenate template path");
			TMPL_write( wktpfile, NULL, NULL, tpldata, WkFile, stderr);
			TMPL_free_varlist( tpldata);
			tpldata = NULL;
			FreeMemory( wktpfile);
			wktpfile = NULL;
		}

		else if (WCtfSpawns)
		{
			/* add dmspawns.tpl/ctfspawns.tpl values */
			tpldata = tmpl_add_spawns( tpldata);

			/* define and fill DoomWiki template */
			if (asprintf( &wktpfile, "%s/%s", (WkTpPath != NULL ? WkTpPath : "."), "ctfspawns.tpl") == -1)
				ProgError( "unable to concatenate template path");
			TMPL_write( wktpfile, NULL, NULL, tpldata, WkFile, stderr);
			TMPL_free_varlist( tpldata);
			tpldata = NULL;
			FreeMemory( wktpfile);
			wktpfile = NULL;
		}

		else
		{
			if (WStatistics)
				fprintf( WkFile, "== Statistics ==\n");
			if (WStatistics || WMapdata)
			{
				/* add mapdata.tpl values */
				tpldata = tmpl_add_stats( tpldata);

				/* define and fill DoomWiki template */
				if (asprintf( &wktpfile, "%s/%s", (WkTpPath != NULL ? WkTpPath : "."), "mapdata.tpl") == -1)
					ProgError( "unable to concatenate template path");
				TMPL_write( wktpfile, NULL, NULL, tpldata, WkFile, stderr);
				TMPL_free_varlist( tpldata);
				tpldata = NULL;
				FreeMemory( wktpfile);
				wktpfile = NULL;
			}

			if (WStatistics)
				fprintf( WkFile, "\n");
			if (WStatistics || WThings)
			{
				/* add things.tpl values */
				tpldata = tmpl_add_things( tpldata);

				/* define and fill DoomWiki template */
				if (asprintf( &wktpfile, "%s/%s", (WkTpPath != NULL ? WkTpPath : "."), "things.tpl") == -1)
					ProgError( "unable to concatenate template path");
				TMPL_write( wktpfile, NULL, NULL, tpldata, WkFile, stderr);
				TMPL_free_varlist( tpldata);
				tpldata = NULL;
				FreeMemory( wktpfile);
				wktpfile = NULL;
			}
		}
	}

	/* clean up & free space */
	if (secretsectors > 0)
		FreeMemory( secsecrets);
	if (secretthings > 0)
		FreeMemory( thgsecrets);
	if (d64tcsecrcnt > 0)
		FreeMemory( d64tcsecrets);
	if (d64tcshrncnt > 0)
		FreeMemory( d64tcshrines);

	ForgetLevelData();
	ForgetThings();
}


/*
	add complete and sortable map names to template variables
*/
TMPL_varlist *tmpl_add_names( TMPL_varlist *tpl, char *name)
{
	TMPL_varlist *t = tpl;
	char *n = name;

	if (strlen( name) == 0)
		return t;
	t = TMPL_add_var( t, "MAPNAME", name, NULL);

	/* check if name starts with an article */
	if (strncmp( name, "A ", 2) == 0)
		n = &name[2];
	if (strncmp( name, "An ", 3) == 0)
		n = &name[3];
	if (strncmp( name, "The ", 4) == 0)
		n = &name[4];
	/* add sortable name without article */
	if (n != name)
		t = TMPL_add_var( t, "MAPSORT", n, NULL);

	return t;
}

/*
	add map slot & game to template variables
*/
TMPL_varlist *tmpl_add_slot( TMPL_varlist *tpl, BCINT epi, BCINT mis)
{
	TMPL_varlist *t = tpl;
	char strbuf[15];

	memset( strbuf, 0, sizeof(strbuf));
	if (GameVersion == 0x02 || GameVersion >= 0x20)
	{
		if (UDMFmark != NULL)
			snprintf( strbuf, sizeof(strbuf), "%s%02d", UDMFmark, mis);
		else
			snprintf( strbuf, sizeof(strbuf), "MAP%02d", mis);
		t = TMPL_add_var( t, "MAPSLOT", strbuf, NULL);
		if (UserGamNm != NULL)
		{
			t = TMPL_add_var( t, "MAPGAME", UserGamNm, NULL);
		}
		else
		{
			if (GameVersion == 0x02)
				t = TMPL_add_var( t, "MAPGAME", "Doom II", NULL);
			else if ((GameVersion & 0x20) == 0x20)
				t = TMPL_add_var( t, "MAPGAME", "Strife", NULL);
			else if ((GameVersion & 0x40) == 0x40)
				t = TMPL_add_var( t, "MAPGAME", "Hexen", NULL);
			else // == 0x80
				t = TMPL_add_var( t, "MAPGAME", "Doom 64", NULL);
		}
	}
	else // < 0x20
	{
		if (UDMFmark != NULL)
			snprintf( strbuf, sizeof(strbuf), "%s%d", UDMFmark, mis);
		else
			snprintf( strbuf, sizeof(strbuf), "E%dM%d", epi, mis);
		t = TMPL_add_var( t, "MAPSLOT", strbuf, NULL);
		if (UserGamNm != NULL)
		{
			t = TMPL_add_var( t, "MAPGAME", UserGamNm, NULL);
		}
		else
		{
			t = TMPL_add_var( t, "MAPGAME",
			                  ((GameVersion & 0x10) != 0x00 ? "Heretic": "Doom"), NULL);
		}
	}

	return t;
}

/*
	add secret sector values to template variables
*/
TMPL_varlist *tmpl_add_secrets( TMPL_varlist *tpl, int secret, int *secrets)
{
	TMPL_loop *tplloop = NULL;
	int n;

	for (n = 0; n < secret; n++)
		tplloop = TMPL_add_varlist( tplloop,
			tmpl_add_int( NULL, "SECRET", abs(secrets[n])));

	return TMPL_add_loop( tpl, "SECRETS", tplloop);
}

/*
	add secret thing values to template variables
*/
TMPL_varlist *tmpl_add_thgsecrets( TMPL_varlist *tpl, int secret, int *secrets)
{
	TMPL_loop *tplloop = NULL;
	int n, i, j;
	Bool first;

	for (n = 0; n < secret; n++)
	{
		/* find item for this thing */
		first = TRUE;
		j = -1;
		for (i = 0; i < NumThItems; i++)
			if (ThItems[i]->type == Things[abs(secrets[n])].type)
			{
				/* kludge for Doom 64 to match THING_64_LASER / THING_64_TRIGGER instead */
				if (first &&
				    (Things[abs(secrets[n])].type == THING_WOLF3DSS ||
				     Things[abs(secrets[n])].type == THING_BOSSSHOOT))
				{
					first = FALSE;
					continue;
				}
				j = i;
				break;
			}
		tplloop = TMPL_add_varlist( tplloop,
			TMPL_add_var(
				tmpl_add_int( NULL, "THGSECRET", abs(secrets[n])),
				"THGNAME", (j == -1 ? "<custom>" : ThItems[j]->wikiname), NULL));
	}

	return TMPL_add_loop( tpl, "THGSECRETS", tplloop);
}

/*
	add Doom 64 TC secret/shrine linedef values to template variables
*/
TMPL_varlist *tmpl_add_linesecrets( TMPL_varlist *tpl, int secret, int *secrets, int shrine, int *shrines)
{
	TMPL_loop *tplloop = NULL;
	int n;

	for (n = 0; n < secret; n++)
		tplloop = TMPL_add_varlist( tplloop,
			TMPL_add_var(
				tmpl_add_int( NULL, "LINSECRET", secrets[n]),
				"LINNAME", "", NULL));

	for (n = 0; n < shrine; n++)
		tplloop = TMPL_add_varlist( tplloop,
			TMPL_add_var(
				tmpl_add_int( NULL, "LINSECRET", shrines[n]),
				"LINNAME", "Shrine", NULL));

	return TMPL_add_loop( tpl, "LINSECRETS", tplloop);
}

/*
	add map number'th in words to template variables
*/
TMPL_varlist *tmpl_add_number( TMPL_varlist *tpl, BCINT mis)
{
	char strbuf[40];

	memset( strbuf, 0, sizeof(strbuf));
	if (GameVersion == 0x02 || GameVersion >= 0x80 ||  // Doom II & 64
	    ((GameVersion & 0x40) == 0x40 && PatchWads != NULL))  // Hexen w/ PWAD
	{
		switch (mis % 100)
		{
			case  0: sprintf( strbuf, "zeroth"); break;
			case  1: sprintf( strbuf, "first"); break;
			case  2: sprintf( strbuf, "second"); break;
			case  3: sprintf( strbuf, "third"); break;
			case  4: sprintf( strbuf, "fourth"); break;
			case  5: sprintf( strbuf, "fifth"); break;
			case  6: sprintf( strbuf, "sixth"); break;
			case  7: sprintf( strbuf, "seventh"); break;
			case  8: sprintf( strbuf, "eighth"); break;
			case  9: sprintf( strbuf, "ninth"); break;
			case 10: sprintf( strbuf, "tenth"); break;
			case 11: sprintf( strbuf, "eleventh"); break;
			case 12: sprintf( strbuf, "twelfth"); break;
			case 13: sprintf( strbuf, "thirteenth"); break;
			case 14: sprintf( strbuf, "fourteenth"); break;
			case 15: sprintf( strbuf, "fifteenth"); break;
			case 16: sprintf( strbuf, "sixteenth"); break;
			case 17: sprintf( strbuf, "seventeenth"); break;
			case 18: sprintf( strbuf, "eighteenth"); break;
			case 19: sprintf( strbuf, "nineteenth"); break;
			case 20: sprintf( strbuf, "twentieth"); break;
			case 21: sprintf( strbuf, "twenty-first"); break;
			case 22: sprintf( strbuf, "twenty-second"); break;
			case 23: sprintf( strbuf, "twenty-third"); break;
			case 24: sprintf( strbuf, "twenty-fourth"); break;
			case 25: sprintf( strbuf, "twenty-fifth"); break;
			case 26: sprintf( strbuf, "twenty-sixth"); break;
			case 27: sprintf( strbuf, "twenty-seventh"); break;
			case 28: sprintf( strbuf, "twenty-eighth"); break;
			case 29: sprintf( strbuf, "twenty-ninth"); break;
			case 30: sprintf( strbuf, "thirtieth and final"); break;
			case 31: sprintf( strbuf, "first secret"); break;
			case 32: sprintf( strbuf, "second secret"); break;
			case 33: sprintf( strbuf, "thirty-third"); break;
			case 34: sprintf( strbuf, "thirty-fourth"); break;
			case 35: sprintf( strbuf, "thirty-fifth"); break;
			case 36: sprintf( strbuf, "thirty-sixth"); break;
			case 37: sprintf( strbuf, "thirty-seventh"); break;
			case 38: sprintf( strbuf, "thirty-eighth"); break;
			case 39: sprintf( strbuf, "thirty-ninth"); break;
			case 40: sprintf( strbuf, "fortieth"); break;
			case 41: sprintf( strbuf, "forty-first"); break;
			case 42: sprintf( strbuf, "forty-second"); break;
			case 43: sprintf( strbuf, "forty-third"); break;
			case 44: sprintf( strbuf, "forty-fourth"); break;
			case 45: sprintf( strbuf, "forty-fifth"); break;
			case 46: sprintf( strbuf, "forty-sixth"); break;
			case 47: sprintf( strbuf, "forty-seventh"); break;
			case 48: sprintf( strbuf, "forty-eighth"); break;
			case 49: sprintf( strbuf, "forty-ninth"); break;
			case 50: sprintf( strbuf, "fiftieth"); break;
			case 51: sprintf( strbuf, "fifty-first"); break;
			case 52: sprintf( strbuf, "fifty-second"); break;
			case 53: sprintf( strbuf, "fifty-third"); break;
			case 54: sprintf( strbuf, "fifty-fourth"); break;
			case 55: sprintf( strbuf, "fifty-fifth"); break;
			case 56: sprintf( strbuf, "fifty-sixth"); break;
			case 57: sprintf( strbuf, "fifty-seventh"); break;
			case 58: sprintf( strbuf, "fifty-eighth"); break;
			case 59: sprintf( strbuf, "fifty-ninth"); break;
			case 60: sprintf( strbuf, "sixtieth"); break;
			case 61: sprintf( strbuf, "sixty-first"); break;
			case 62: sprintf( strbuf, "sixty-second"); break;
			case 63: sprintf( strbuf, "sixty-third"); break;
			case 64: sprintf( strbuf, "sixty-fourth"); break;
			case 65: sprintf( strbuf, "sixty-fifth"); break;
			case 66: sprintf( strbuf, "sixty-sixth"); break;
			case 67: sprintf( strbuf, "sixty-seventh"); break;
			case 68: sprintf( strbuf, "sixty-eighth"); break;
			case 69: sprintf( strbuf, "sixty-ninth"); break;
			case 70: sprintf( strbuf, "seventieth"); break;
			case 71: sprintf( strbuf, "seventy-first"); break;
			case 72: sprintf( strbuf, "seventy-second"); break;
			case 73: sprintf( strbuf, "seventy-third"); break;
			case 74: sprintf( strbuf, "seventy-fourth"); break;
			case 75: sprintf( strbuf, "seventy-fifth"); break;
			case 76: sprintf( strbuf, "seventy-sixth"); break;
			case 77: sprintf( strbuf, "seventy-seventh"); break;
			case 78: sprintf( strbuf, "seventy-eighth"); break;
			case 79: sprintf( strbuf, "seventy-ninth"); break;
			case 80: sprintf( strbuf, "eightieth"); break;
			case 81: sprintf( strbuf, "eighty-first"); break;
			case 82: sprintf( strbuf, "eighty-second"); break;
			case 83: sprintf( strbuf, "eighty-third"); break;
			case 84: sprintf( strbuf, "eighty-fourth"); break;
			case 85: sprintf( strbuf, "eighty-fifth"); break;
			case 86: sprintf( strbuf, "eighty-sixth"); break;
			case 87: sprintf( strbuf, "eighty-seventh"); break;
			case 88: sprintf( strbuf, "eighty-eighth"); break;
			case 89: sprintf( strbuf, "eighty-ninth"); break;
			case 90: sprintf( strbuf, "ninetieth"); break;
			case 91: sprintf( strbuf, "ninety-first"); break;
			case 92: sprintf( strbuf, "ninety-second"); break;
			case 93: sprintf( strbuf, "ninety-third"); break;
			case 94: sprintf( strbuf, "ninety-fourth"); break;
			case 95: sprintf( strbuf, "ninety-fifth"); break;
			case 96: sprintf( strbuf, "ninety-sixth"); break;
			case 97: sprintf( strbuf, "ninety-seventh"); break;
			case 98: sprintf( strbuf, "ninety-eighth"); break;
			case 99: sprintf( strbuf, "ninety-ninth"); break;
		}
		// prepend hundred count
		mis /= 100;
		if (mis == 1)
			strprep( strbuf, "one hundred ");
		else if (mis == 2)
			strprep( strbuf, "two hundred ");
	}
	else if (GameVersion >= 0x20)
		; // no number'th convention in Hexen and Strife
	else // < 0x20
	{
		switch (mis)
		{
			case 1: sprintf( strbuf, "first"); break;
			case 2: sprintf( strbuf, "second"); break;
			case 3: sprintf( strbuf, "third"); break;
			case 4: sprintf( strbuf, "fourth"); break;
			case 5: sprintf( strbuf, "fifth"); break;
			case 6: sprintf( strbuf, "sixth"); break;
			case 7: sprintf( strbuf, "seventh"); break;
			case 8: sprintf( strbuf, "eighth and final"); break;
			case 9: sprintf( strbuf, "ninth and secret"); break;
			case 10: sprintf( strbuf, "tenth"); break;
			case 11: sprintf( strbuf, "eleventh"); break;
			case 12: sprintf( strbuf, "twelfth"); break;
			case 13: sprintf( strbuf, "thirteenth"); break;
			case 14: sprintf( strbuf, "fourteenth"); break;
			case 15: sprintf( strbuf, "fifteenth"); break;
			case 16: sprintf( strbuf, "sixteenth"); break;
			case 17: sprintf( strbuf, "seventeenth"); break;
			case 18: sprintf( strbuf, "eighteenth"); break;
			case 19: sprintf( strbuf, "nineteenth"); break;
		}
	}

	return TMPL_add_var( tpl, "MAPNUMTH", strbuf, NULL);
}

/*
	add player spawns count in words to template variables
*/
TMPL_varlist *tmpl_add_pscount( TMPL_varlist *tpl, char *tvar, BCINT count)
{
	char strbuf[24];

	memset( strbuf, 0, sizeof(strbuf));
	switch (count)
	{
		case  1: sprintf( strbuf, "one"); break;
		case  2: sprintf( strbuf, "two"); break;
		case  3: sprintf( strbuf, "three"); break;
		case  4: sprintf( strbuf, "four"); break;
		case  5: sprintf( strbuf, "five"); break;
		case  6: sprintf( strbuf, "six"); break;
		case  7: sprintf( strbuf, "seven"); break;
		case  8: sprintf( strbuf, "eight"); break;
		case  9: sprintf( strbuf, "nine"); break;
		case 10: sprintf( strbuf, "ten"); break;
		case 11: sprintf( strbuf, "eleven"); break;
		case 12: sprintf( strbuf, "twelve"); break;
		case 13: sprintf( strbuf, "thirteen"); break;
		case 14: sprintf( strbuf, "fourteen"); break;
		case 15: sprintf( strbuf, "fifteen"); break;
		case 16: sprintf( strbuf, "sixteen"); break;
		case 17: sprintf( strbuf, "seventeen"); break;
		case 18: sprintf( strbuf, "eighteen"); break;
		case 19: sprintf( strbuf, "nineteen"); break;
		case 20: sprintf( strbuf, "twenty"); break;
		case 21: sprintf( strbuf, "twenty-one"); break;
		case 22: sprintf( strbuf, "twenty-two"); break;
		case 23: sprintf( strbuf, "twenty-three"); break;
		case 24: sprintf( strbuf, "twenty-four"); break;
		case 25: sprintf( strbuf, "twenty-five"); break;
		case 26: sprintf( strbuf, "twenty-six"); break;
		case 27: sprintf( strbuf, "twenty-seven"); break;
		case 28: sprintf( strbuf, "twenty-eight"); break;
		case 29: sprintf( strbuf, "twenty-nine"); break;
		case 30: sprintf( strbuf, "thirty"); break;
		case 31: sprintf( strbuf, "thirty-one"); break;
		case 32: sprintf( strbuf, "thirty-two"); break;
		case 33: sprintf( strbuf, "thirty-three"); break;
		case 34: sprintf( strbuf, "thirty-four"); break;
		case 35: sprintf( strbuf, "thirty-five"); break;
		case 36: sprintf( strbuf, "thirty-six"); break;
		case 37: sprintf( strbuf, "thirty-seven"); break;
		case 38: sprintf( strbuf, "thirty-eight"); break;
		case 39: sprintf( strbuf, "thirty-nine"); break;
		case 40: sprintf( strbuf, "forty"); break;
		default: sprintf( strbuf, "%d", count); break;
	}

	return TMPL_add_var( tpl, tvar, strbuf, NULL);
}

/*
	add player spawn direction in words to template variables
*/
TMPL_varlist *tmpl_add_psdir( TMPL_varlist *tpl, char *tvar, BCINT angle)
{
	char strbuf[12];

	memset( strbuf, 0, sizeof(strbuf));
	// wrap negative angles
	switch ((angle+360) % 360)
	{
		case   0: sprintf( strbuf, "east"); break;
		case  45: sprintf( strbuf, "north-east"); break;
		case  90: sprintf( strbuf, "north"); break;
		case 135: sprintf( strbuf, "north-west"); break;
		case 180: sprintf( strbuf, "west"); break;
		case 225: sprintf( strbuf, "south-west"); break;
		case 270: sprintf( strbuf, "south"); break;
		case 315: sprintf( strbuf, "south-east"); break;
		default : sprintf( strbuf, "%d&deg;", angle); break;
	}

	return TMPL_add_var( tpl, tvar, strbuf, NULL);
}

/*
	add DM/CTF player spawns to template variables
*/
TMPL_varlist *tmpl_add_spawns( TMPL_varlist *tpl)
{
	TMPL_varlist *t = tpl, *titem;
	TMPL_loop    *tplloop;
	int n, pscount;

	/* add DM player spawn values */
	pscount = 0;
	tplloop = NULL;
	for (n = 0; n < NumThings; n++)
		if (Things[n].type == THING_DEATHMATCH)
		{
			titem = tmpl_add_psdir(NULL, "DMDIR", Things[n].angle);
			titem = tmpl_add_int( titem, "DMSPAWN", n);
			tplloop = TMPL_add_varlist( tplloop, titem);
			pscount++;
		}
	t = TMPL_add_loop( t, "DMSPAWNS", tplloop);
	t = tmpl_add_pscount( t, "DMSCOUNT", pscount);

	/* add CTF Blue player spawn values */
	pscount = 0;
	tplloop = NULL;
	for (n = 0; n < NumThings; n++)
		if (Things[n].type == THING_TEAMBLUE)
		{
			titem = tmpl_add_psdir(NULL, "TBDIR", Things[n].angle);
			titem = tmpl_add_int( titem, "TBSPAWN", n);
			tplloop = TMPL_add_varlist( tplloop, titem);
			pscount++;
		}
	t = TMPL_add_loop( t, "TBSPAWNS", tplloop);
	t = tmpl_add_pscount( t, "TBSCOUNT", pscount);

	/* add CTF Red player spawn values */
	pscount = 0;
	tplloop = NULL;
	for (n = 0; n < NumThings; n++)
		if (Things[n].type == THING_TEAMRED)
		{
			titem = tmpl_add_psdir(NULL, "TRDIR", Things[n].angle);
			titem = tmpl_add_int( titem, "TRSPAWN", n);
			tplloop = TMPL_add_varlist( tplloop, titem);
			pscount++;
		}
	t = TMPL_add_loop( t, "TRSPAWNS", tplloop);
	t = tmpl_add_pscount( t, "TRSCOUNT", pscount);

	/* add CTF Green player spawn values */
	pscount = 0;
	tplloop = NULL;
	for (n = 0; n < NumThings; n++)
		if (Things[n].type == THING_TEAMGREEN)
		{
			titem = tmpl_add_psdir(NULL, "TGDIR", Things[n].angle);
			titem = tmpl_add_int( titem, "TGSPAWN", n);
			tplloop = TMPL_add_varlist( tplloop, titem);
			pscount++;
		}
	t = TMPL_add_loop( t, "TGSPAWNS", tplloop);
	t = tmpl_add_pscount( t, "TGSCOUNT", pscount);

	/* add CTF Gold player spawn values */
	pscount = 0;
	tplloop = NULL;
	for (n = 0; n < NumThings; n++)
		if (Things[n].type == THING_TEAMGREEN)
		{
			titem = tmpl_add_psdir(NULL, "TDDIR", Things[n].angle);
			titem = tmpl_add_int( titem, "TDSPAWN", n);
			tplloop = TMPL_add_varlist( tplloop, titem);
			pscount++;
		}
	t = TMPL_add_loop( t, "TDSPAWNS", tplloop);
	t = tmpl_add_pscount( t, "TDSCOUNT", pscount);

	return t;
}

/*
	add map statistics to template variables
*/
TMPL_varlist *tmpl_add_stats( TMPL_varlist *tpl)
{
	TMPL_varlist *t = tpl;

	t = tmpl_add_int( t, "NUMTHNG", NumThings);
	t = tmpl_add_int( t, "NUMVERT", TotVertexes);
	t = tmpl_add_int( t, "NUMLINE", NumLineDefs);
	t = tmpl_add_int( t, "NUMSIDE", NumSideDefs);
	t = tmpl_add_int( t, "NUMSECT", NumSectors);
	if (!UDMFmap)
		t = tmpl_add_int( t, "NUMVBEF", NumVertexes);

	return t;
}

/*
	add map things to template variables
*/
TMPL_varlist *tmpl_add_things( TMPL_varlist *tpl)
{
	TMPL_loop    *tlmodes = NULL, *tlclasses, *tlitems;
	TMPL_varlist *tmode, *tclass, *titem;
	int m, c, i, class, modecount = 0;
	Bool secondDM = FALSE;

	/* add game-specific skills anchor */
	tpl = tmpl_add_skill( tpl);

	/* process all relevant modes */
	for (m = 0; m < ThMode_End; m++)
	{
		if ((GameVersion & 0x80) == 0x80) // Doom64 format
		{
			/* collect items for this mode:
			   0 = SP */
			if (m != ThModeSP)
				continue;
		}
		else if ((GameVersion & 0x40) == 0x40) // Hexen format
		{
			/* collect items for these modes:
			   1 = Cleric, 2 = Fighter, 3 = Mage, 5 = CooP, 6 = DM */
			if (m == ThModeSP || m == ThModeMP)
				continue;
		}
		else // Doom format
		{
			/* collect items for these modes:
			   0 = SP, 4 = MP (if !WBoomMP), 5 = CooP, 6 = DM (both if WBoomMP) */
			if (WBoomMP && m == ThModeMP)
				continue;
			if (!WBoomMP && (m == ThModeCP || m == ThModeDM))
				continue;
			if (m == ThModeXC || m == ThModeXF || m == ThModeXM)
				continue;
		}

		/* if excluding SP/CP tables, collect only 4 = MP, 6 = DM */
		if (WXcludeSCp &&
		    (m == ThModeSP || m == ThModeCP || m == ThModeXC || m == ThModeXF || m == ThModeXM))
			continue;
		/* if excluding SP tables, collect only 4 = MP, 5 = CP, 6 = DM */
		if (WXcludeSip &&
		    (m == ThModeSP || m == ThModeXC || m == ThModeXF || m == ThModeXM))
			continue;
		/* if excluding SP/MP|DM tables, collect only 5 = CP */
		if (WXcludeSMD &&
		    (m == ThModeSP || m == ThModeMP || m == ThModeDM || m == ThModeXC || m == ThModeXF || m == ThModeXM))
			continue;
		/* if excluding MP|CP/DM tables, collect only 0 = SP & Hexen classes */
		if (WXcludeMCD &&
		    (m == ThModeMP || m == ThModeCP || m == ThModeDM))
			continue;
		/* if excluding MP|DM tables, collect only 0 = SP, 5 = CP & Hexen classes */
		if (WXcludeMDm &&
		    (m == ThModeMP || m == ThModeDM))
			continue;

		/* if no Coop starts, skip Coop table */
		if (!CheckThingTypes( THING_COOPSTARTS, ThModeCP) &&
		    !CheckThingTypes( THING_TEAMBLUE, ThModeCP) &&
		    !CheckThingTypes( THING_TEAMRED, ThModeCP) && m == ThModeCP)
			continue;
		/* if no DM/CTF starts, skip DM table */
		if (!CheckThingTypes( THING_DEATHMATCH, ThModeDM) &&
		    !CheckThingTypes( THING_TEAMBLUE, ThModeDM) &&
		    !CheckThingTypes( THING_TEAMRED, ThModeDM) && m == ThModeDM)
			continue;
		/* if neither starts, skip MP table */
		if (!CheckThingTypes( THING_COOPSTARTS, ThModeMP) &&
		    !CheckThingTypes( THING_DEATHMATCH, ThModeMP) && m == ThModeMP)
			continue;

		/* process all thing classes */
		tlclasses = NULL;
		for (c = 0; c < NumThClasses; c++)
		{
			class = ThClasses[c]->class;
			/* special check in DM */
			if (m == ThModeDM)
			{
				/* skip Keys class */
				if (class == ThClassKeys)
					continue;
			}
			/* special check in MP-only/DM */
			if ((m == ThModeMP && WXcludeSCp) || m == ThModeDM)
			{
				/* skip classes for 1st or 2nd DM column */
				if ((splitDM[class] && !secondDM) || (!splitDM[class] && secondDM))
					continue;
			}

			/* collect item(s) for this class */
			tlitems = NULL;
			for (i = 0; i < NumThItems; i++)
			{
				/* collect item if it exists and should be included in this mode */
				if (class == ThItems[i]->class && ThingInMode( ThItems[i]->type, m) &&
				    TotalThingSkills( ThItems[i], m) > 0)
				{
					titem = TMPL_add_var( NULL, "THITEM",
					                      UsedWiki( ThItems[i]->wikiname,
					                                ThItems[i]->wikilink,
					                                ThItems[i]->wikistr), NULL);
					/* check UDMF flag(s) */
					if (UDMFmap)
						titem = tmpl_add_int( titem, "THSKILL1", ThItems[i]->skill[0][m]);
					titem = tmpl_add_int( titem, "THSKILL2", ThItems[i]->skill[1][m]);
					titem = tmpl_add_int( titem, "THSKILL3", ThItems[i]->skill[2][m]);
					titem = tmpl_add_int( titem, "THSKILL4", ThItems[i]->skill[3][m]);
					if (UDMFmap)
					{
						titem = tmpl_add_int( titem, "THSKILL5", ThItems[i]->skill[4][m]);
						if (UDMFall)
						{
							titem = tmpl_add_int( titem, "THSKILL6", ThItems[i]->skill[5][m]);
							titem = tmpl_add_int( titem, "THSKILL7", ThItems[i]->skill[6][m]);
							titem = tmpl_add_int( titem, "THSKILL8", ThItems[i]->skill[7][m]);
						}
					}
					/* check whether to collate row into one value */
					if (ThItems[i]->skill[1][m] == ThItems[i]->skill[2][m] &&
					    ThItems[i]->skill[1][m] == ThItems[i]->skill[3][m] &&
					    (!UDMFmap ||
					     (ThItems[i]->skill[1][m] == ThItems[i]->skill[0][m] &&
					      ThItems[i]->skill[1][m] == ThItems[i]->skill[4][m] &&
					      (!UDMFall ||
					       (ThItems[i]->skill[1][m] == ThItems[i]->skill[5][m] &&
					        ThItems[i]->skill[1][m] == ThItems[i]->skill[6][m] &&
					        ThItems[i]->skill[1][m] == ThItems[i]->skill[7][m])))))
						titem = TMPL_add_var( titem, "THEQUAL", "equal", NULL);
					tlitems = TMPL_add_varlist( tlitems, titem);
				}
			}
			/* add class section if it contains item(s) */
			if (tlitems != NULL)
			{
				tclass = TMPL_add_var( NULL, "THCLASS",
				                       UsedWiki( ThClasses[c]->wikiname,
				                                 ThClasses[c]->wikilink,
				                                 ThClasses[c]->wikistr), NULL);
				tclass = TMPL_add_loop( tclass, "THITEMS", tlitems);
				tlclasses = TMPL_add_varlist( tlclasses, tclass);
			}
		}
		/* add mode section if it contains class(es) */
		if (tlclasses != NULL)
		{
			switch (m)
			{
				case ThModeSP: tmode = TMPL_add_var( NULL, "THMODE", "Single-player", NULL); break;
				case ThModeXC: tmode = TMPL_add_var( NULL, "THMODE", "Cleric single-player", NULL); break;
				case ThModeXF: tmode = TMPL_add_var( NULL, "THMODE", "Fighter single-player", NULL); break;
				case ThModeXM: tmode = TMPL_add_var( NULL, "THMODE", "Mage single-player", NULL); break;
				case ThModeMP: tmode = TMPL_add_var( NULL, "THMODE", (secondDM ? "Multiplayer (cont.)" : "Multiplayer"), NULL); break;
				case ThModeCP: tmode = TMPL_add_var( NULL, "THMODE", "Cooperative", NULL); break;
				case ThModeDM: tmode = TMPL_add_var( NULL, "THMODE", (secondDM ? "Deathmatch (cont.)" : "Deathmatch"), NULL); break;
			}
			/* control mode columns layout */
			if (modecount % 3 == 0)
			{
				/* check if only SP mode section */
				if (WBoomMP || (GameVersion & 0x40) == 0x40) // Boom (& UDMF) & Hexen format
				{
					if (CheckThingTypes( THING_COOPSTARTS, ThModeCP) ||
					    CheckThingTypes( THING_DEATHMATCH, ThModeDM))
						tmode = TMPL_add_var( tmode, "THFIRST", "first", NULL);
				}
				else // Doom format
				{
					if (CheckThingTypes( THING_COOPSTARTS, ThModeMP) ||
					    CheckThingTypes( THING_DEATHMATCH, ThModeMP))
						tmode = TMPL_add_var( tmode, "THFIRST", "first", NULL);
				}
				if (modecount > 0)
					tmode = TMPL_add_var( tmode, "THROW", "new", NULL);
			}
			modecount++;

			/* add UDMF flag(s) */
			if (UDMFmap)
				tmode = TMPL_add_var( tmode, "THUDMF", "true", NULL);
			if (UDMFall)
				tmode = TMPL_add_var( tmode, "THUDALL", "true", NULL);

			tmode = TMPL_add_loop( tmode, "THCLASSES", tlclasses);
			tlmodes = TMPL_add_varlist( tlmodes, tmode);
		}

		/* check whether splitting MP-only/DM table & repeat for 2nd column */
		if (((m == ThModeMP && WXcludeSCp) || m == ThModeDM)
		    && WSplitDM > 0 && !secondDM)
		{
			secondDM = TRUE;
			m--; // fool the modes loop through MP-only/DM again
		}
	}
	return TMPL_add_loop( tpl, "THMODES", tlmodes);
}

/*
	add integer as string to template variables
*/
TMPL_varlist *tmpl_add_int( TMPL_varlist *tpl, char *name, BCLNG value)
{
	char strbuf[12];

	memset( strbuf, 0, sizeof(strbuf));
	snprintf( strbuf, sizeof(strbuf), "%u", value);

	return TMPL_add_var( tpl, name, strbuf, NULL);
}

/*
	add skill anchor as string to template variables
*/
TMPL_varlist *tmpl_add_skill( TMPL_varlist *tpl)
{
	char *skill;

	switch (GameVersion & 0xF0)
	{
		case 0x00: skill = "Doom and Doom II skill levels"; break;
		case 0x10: skill = "Heretic skill levels"; break;
		case 0x20: skill = "Strife skill levels"; break;
		case 0x40: skill = "Hexen skill levels"; break;
		case 0x80: skill = "Doom 64 skill levels"; break;
	}
	return TMPL_add_var( tpl, "THSKILL", skill, NULL);
}


/*
	count secret sectors
*/
int CountSecrets( BCINT lumpver)
{
	int n, gensec, secret = 0;

	if ((lumpver & 0x80) == 0x80) // Doom64 format
		gensec = D64_GENSEC;
	else if (UDMFmap || (lumpver & 0x40) == 0x40) // UDMF & Hexen formats
		gensec = B_GENSECH;
	else // Doom format
		gensec = B_GENSEC;
	for (n = 0; n < NumSectors; n++)
		if ((Sectors[n].special == S_SECRET && (lumpver & 0x60) != 0x60 && // not Strife Hexen-format
		    (lumpver & 0x80) != 0x80 && !UDMFmap) || // not Doom64 / UDMF
		    (!WD64TC && (Sectors[n].special & gensec) == gensec)) // or generalized but not Doom 64 TC
			secret++;

	return secret;
}


/*
	read level & list extra Coop things
*/
void ExtraCPLevel( BCINT episode, BCINT mission)
{
	int c, i, class, coop, totsp, totcp;
	BCINT lumpver;
	BCLNG k;
	Bool header;

	/* read data of this level */
	lumpver = ReadLevelData( episode, mission);

	/* check and warn if Hexen-format flags exist but aren't counted */
	if (lumpver != GameVersion && !WBoomMP)
		printf( "Warning: level's Things contain Hexen-format flags.  Use option '+b'.\n");
	/* check and warn if Boom MP flags exist but aren't counted */
	else if (CheckThingBoomFlags() && !WBoomMP && !WD64TC)
		printf( "Warning: level's Things contain Boom MP flags.  Use option '+b'.\n");

	/* initialize and analyze thing statistics */
	switch (GameVersion & 0xF0)
	{
		case 0x00: // fall thru
		case 0x80: InitDoomThings(); break;
		case 0x10: InitHereticThings(); break;
		case 0x20: InitStrifeThings( mission); break;
		case 0x40: InitHexenThings(); break;
	}
	if (CustomMap != NULL)
		ReadCustomThings();
	AnalyzeThings( lumpver, TRUE);

	/* determine comparison mode */
	if ((GameVersion & 0x40) == 0x40) // Hexen format
		coop = ThModeCP;
	else // Doom format
		if (WBoomMP)
			coop = ThModeCP;
		else
			coop = ThModeMP;

	/* process aggregate values */
	AggregateSpecials();

	/* find coop starts item */
	c = -1;
	for (i = 0; i < NumThItems; i++)
		if (ThItems[i]->type == THING_COOPSTARTS)
		{
			c = i;
			break;
		}

	/* check for extra things if Coop starts exist */
	if (TotalThingSkills( ThItems[c], coop) > 0)
	{
		/* collect classes for extra Coop things */
		for (c = 0; c < ThClass_End; c++)
			extraCP[c] = FALSE;
		for (k = WExtraCP; k > 0; k = k / 10)
		{
			extraCP[k % 10 - 1] = TRUE;
		}
		//for (c = 0; c < ThClass_End; c++)
		//	printf( "class %d: %s\n", c, (extraCP[c] ? "show" : "skip"));

		/* process all thing classes */
		for (c = 0; c < NumThClasses; c++)
		{
			/* check for requested class */
			class = ThClasses[c]->class;
			if (extraCP[class])
			{
				header = FALSE;
				totsp = totcp = 0;
				/* tally item(s) in this class */
				for (i = 0; i < NumThItems; i++)
				{
					if (class != ThItems[i]->class)
						continue;
					if (TotalThingSkills( ThItems[i], ThModeSP) > 0)
						totsp++;
					if (TotalThingSkills( ThItems[i], coop) > 0)
						totcp++;
				}
				/* process item(s) for this class */
				for (i = 0; i < NumThItems; i++)
				{
					if (class != ThItems[i]->class)
						continue;
					/* check for more Coop than SP things */
					if (TotalThingSkills( ThItems[i], coop) > TotalThingSkills( ThItems[i], ThModeSP))
					{
						/* list class */
						if (!header)
						{
							header = TRUE;
							if (totsp > totcp)
								printf("\n%s:\t\t\t\t\t\t\t(%d > %d)\n", ThClasses[c]->wikiname, totsp, totcp);
							else
								printf("\n%s:\n", ThClasses[c]->wikiname);
						}
						/* list extra things per skill */
						printf("+ %2d %2d %2d\t%s\n",
						       ThItems[i]->skill[1][coop] - ThItems[i]->skill[1][ThModeSP],
						       ThItems[i]->skill[2][coop] - ThItems[i]->skill[2][ThModeSP],
						       ThItems[i]->skill[3][coop] - ThItems[i]->skill[3][ThModeSP],
						       ThItems[i]->wikiname);
					}
				}
			}
		}
	}

	ForgetLevelData();
	ForgetThings();
}

/*
	search things and linedefs for matching TID of SecretTrigger thing
*/
void SearchTriggering( int secret)
{
	int n, i;
	BCINT tid = Things[secret].tid;
	Bool first = TRUE, found = FALSE;

	printf( "Note: SecretTrigger Thing %d with TID %d found\n", secret, tid);

	/* check things for matching TID */
	for (n = 0; n < NumThings; n++)
		if (Things[n].special == THING_Z_ACTIVATE && Things[n].arg1 == tid)
		{
			printf( "      Matching Thing %d (", n);
			for (i = 0; i < NumThItems; i++)
				if (ThItems[i]->type == Things[n].type)
				{
					printf( "%s, ", ThItems[i]->wikiname);
					break;
				}
			printf( "type %d)\n", Things[n].type);
			found = TRUE;
			break;
		}

	/* check linedefs for matching TID */
	for (n = 0; n < NumLineDefs; n++)
		if (LineDefs[n].special == THING_Z_ACTIVATE && LineDefs[n].arg1 == tid)
		{
			if (first)
			{
				printf( "      Matching LineDef(s) %d", n);
				first = FALSE;
			}
			else
				printf( ", %d", n);
			found = TRUE;
		}
	if (found && !first)
		printf( "\n");

	if (!found)
		printf( "      No match found, check Scripts\n");
}

/* vim:set noexpandtab tabstop=2 softtabstop=2 shiftwidth=2: */
