/*
	DooM MaP StaTistics, by Frans P. de Vries.

Derived from:

	DooM PostScript Maps Utility, by Frans P. de Vries.

And thus from:

	Doom Editor Utility, by Brendon Wyber and Raphaël Quinet.

	You are allowed to use any parts of this code in another program, as
	long as you give credits to the authors in the documentation and in
	the program itself.  Read the file README for more information.

	This program comes with absolutely no warranty.

	WADS.C - WAD files routines.
*/

#include "dmmpst.h"
#include "wads.h"


/*
	the global variables
*/
WadPtr  WadFileList = NULL;  /* linked list of Wad files */
MDirPtr MasterDir   = NULL;  /* the master directory */

/*
	the local function prototypes
*/
MDirPtr AppendMasterEntry( void);
MDirPtr AppendLevelEntry( MDirPtr);
Bool IsLevelStart( char *);
Bool IsLevelLump( char *);
Bool Exists( char *);
void ListMasterDirectory( void);


/*
	swap routines to handle big/little endian machines
*/
#ifdef DM_BIG_ENDIAN
/* big endian machines (eg. Sun, SGI) need to swap bytes within int's/long's */

void swapint( BCINT *i)
{
	BCINT t;

	((char *) &t)[0] = ((char *) i)[1];
	((char *) &t)[1] = ((char *) i)[0];
	*i = t;
}

void swaplong( BCLNG *l)
{
	BCLNG t;

	((char *) &t)[0] = ((char *) l)[3];
	((char *) &t)[1] = ((char *) l)[2];
	((char *) &t)[2] = ((char *) l)[1];
	((char *) &t)[3] = ((char *) l)[0];
	*l = t;
}

#else
/* little endian machines (eg. IBM PC) need to do nothing */
#endif


/*
	open the main WAD file, read in its directory and create the master directory
*/
void OpenMainWad( char *filename)
{
	MDirPtr lastp, newp;
	WadPtr wad;
	BCLNG n;

	/* open the WAD file */
	printf( "Loading main WAD file: %s...\n", filename);
	wad = BasicWadOpen( filename);
	if (strncmp( wad->type, "IWAD", 4) != 0)
		ProgError( "\"%s\" is not the main WAD file", filename);

	/* create the master directory */
	lastp = NULL;
	for (n = 0; n < wad->dirsize; n++)
	{
		newp = (MDirPtr) GetMemory( sizeof(struct MasterDirectory));
		newp->next = NULL;
		newp->wadfile = wad;
		memcpy( &(newp->dir), &(wad->directory[n]), sizeof(struct Directory));
		if (MasterDir)
			lastp->next = newp;
		else
			MasterDir = newp;
		lastp = newp;
	}

	/* check for expansion/ultimate/commercial/registered version */
	/* if you change this, bad things will happen to you... */
	if (FindMasterDir( MasterDir, "MAP58") != NULL) // Hexen DDC
		GameVersion = 0x08;
	else if (FindMasterDir( MasterDir, "DENDBACK") != NULL) // Strife VE
		GameVersion = 0x08;
	else if (FindMasterDir( MasterDir, "ENDOFWAD") != NULL &&
	         FindMasterDir( MasterDir, "MAP40") != NULL) // Doom 64 tLL
		GameVersion = 0x08;
	else if (FindMasterDir( MasterDir, "E4M8") != NULL) // Doom ult, Heretic SSR
		GameVersion = 0x04;
	else if (FindMasterDir( MasterDir, "MAP28") != NULL) // Doom II, Strife, Hexen, Doom 64
		GameVersion = 0x02;
	else if (FindMasterDir( MasterDir, "E2M8") != NULL) // Doom reg, Heretic reg
		GameVersion = 0x01;
	else /* shareware, demo */
		GameVersion = 0x00;

	/* check for Doom end screen */
	if (FindMasterDir( MasterDir, "ENDOOM") != NULL)
		GameVersion += 0x00;
	/* check for Heretic end screen */
	else if (FindMasterDir( MasterDir, "ENDTEXT") != NULL)
		GameVersion += 0x10;
	/* check for Strife end screen */
	else if (FindMasterDir( MasterDir, "ENDSTRF") != NULL)
		GameVersion += 0x20;
	/* check for Strife VE end screen */
	else if (FindMasterDir( MasterDir, "STRBACK") != NULL)
		GameVersion += 0x20;
	/* check for Hexen map lumps */
	else if (FindMasterDir( MasterDir, "BEHAVIOR") != NULL)
		GameVersion += 0x40;
	/* check for Doom 64 wad end marker */
	else if (FindMasterDir( MasterDir, "ENDOFWAD") != NULL)
		GameVersion += 0x80;
	else
		ProgError( "\"%s\" is an unsupported main WAD file", filename);

	if (Verbose)
		printf( "Main WAD version: 0x%02x.\n", GameVersion);
}

/*
	open a patch WAD file, read in its directory and alter the master directory
*/
void OpenPatchWad( char *filename)
{
	MDirPtr mdir, ldir = NULL;
	WadPtr wad;
	BCINT n;
	char entryname[9];
	Bool append;

	/* ignore the file if it doesn't exist */
	if (!Exists( filename))
	{
		printf( "Warning: patch WAD file \"%s\" doesn't exist.  Ignored.\n", filename);
		return;
	}

	/* open the WAD file */
	printf( "Loading patch WAD file: %s...\n", filename);
	wad = BasicWadOpen( filename);
	if (strncmp( wad->type, "PWAD", 4) != 0)
		ProgError( "\"%s\" is not a patch WAD file", filename);

	/* alter the master directory */
	for (n = 0; n < wad->dirsize; n++)
	{
		strncpy( entryname, wad->directory[n].name, 8);
		entryname[8] = '\0';
		/* check for level-related lumps */
		if (IsLevelStart( entryname))
		{
			mdir = FindMasterDir( MasterDir, entryname);
			/* check for level in master directory */
			if (mdir)
			{
				if (Verbose)
					printf( "   [Updating  level %s]\n", entryname);
			}
			else
			{
				if (Verbose)
					printf( "   [Appending level %s]\n", entryname);
				mdir = AppendMasterEntry();
			}
			ldir = mdir;
		}
		else if (IsLevelLump( entryname) && ldir)
		{
			mdir = FindLevelDir( ldir, entryname);
			if (!mdir)
				mdir = AppendLevelEntry( ldir);
		}
		else // non-level lump
		{
			mdir = FindMasterDir( MasterDir, entryname);
			/* check for entry in master directory */
			if (mdir)
			{
				if (Verbose)
					printf( "   [Updating  entry %s]\n", entryname);
			}
			else
			{
				if (Verbose)
					printf( "   [Appending entry %s]\n", entryname);
				mdir = AppendMasterEntry();
			}
			ldir = NULL;
		}
		/* copy WAD name and directory info into existing or new entry */
		mdir->wadfile = wad;
		memcpy( &(mdir->dir), &(wad->directory[n]), sizeof(struct Directory));
	}
	//ListMasterDirectory();
}

/*
	append new entry to master directory
*/
MDirPtr AppendMasterEntry( void)
{
	MDirPtr mdir = MasterDir;

	/* find last lump */
	while (mdir->next)
		mdir = mdir->next;
	/* append new lump */
	mdir->next = (MDirPtr) GetMemory( sizeof(struct MasterDirectory));
	mdir = mdir->next;
	mdir->next = NULL;

	return mdir;
}

/*
	append new entry to level section
*/
MDirPtr AppendLevelEntry( MDirPtr ldir)
{
	MDirPtr mdir = ldir, tdir;

	/* find last level lump */
	while (mdir->next && !IsLevelLump( mdir->next->dir.name))
		mdir = mdir->next;
	/* append new lump */
	tdir = mdir->next;
	mdir->next = (MDirPtr) GetMemory( sizeof(struct MasterDirectory));
	mdir = mdir->next;
	mdir->next = tdir;

	return mdir;
}

/*
	close all the Wad files, deallocating the data structures
*/
void CloseWadFiles( void)
{
	MDirPtr curd, nextd;
	WadPtr curw, nextw;

	/* close the Wad files */
	curw = WadFileList;
	while (curw)
	{
		nextw = curw->next;
		fclose( curw->fileinfo);
		FreeMemory( curw->directory);
		FreeMemory( curw);
		curw = nextw;
	}
	WadFileList = NULL;

	/* delete the master directory */
	curd = MasterDir;
	while (curd)
	{
		nextd = curd->next;
		FreeMemory( curd);
		curd = nextd;
	}
	MasterDir = NULL;
}

/*
	forget unused patch Wad files
*/
void CloseUnusedWadFiles( void)
{
	MDirPtr mdir;
	WadPtr curw, prevw;

	prevw = NULL;
	curw = WadFileList;
	while (curw)
	{
		/* check if the Wad file is used by a directory entry */
		mdir = MasterDir;
		while (mdir && mdir->wadfile != curw)
			mdir = mdir->next;
		if (mdir)
			prevw = curw;
		else
		{
			/* if this Wad file is never used, close it */
			if (prevw)
				prevw->next = curw->next;
			else
				WadFileList = curw->next;
			fclose( curw->fileinfo);
			FreeMemory( curw->directory);
			FreeMemory( curw);
		}
		curw = prevw->next;
	}
}


/*
	basic opening of WAD file and creation of node in Wad linked list
*/
WadPtr BasicWadOpen( char *filename)
{
	WadPtr curw, prevw;
	BCLNG i;

	/* find the WAD file in the Wad file list */
	prevw = WadFileList;
	if (prevw)
	{
		curw = prevw->next;
		while (curw && strcmp( filename, curw->filename) != 0)
		{
			prevw = curw;
			curw = prevw->next;
		}
	}
	else
		curw = NULL;

	/* if this entry doesn't exist, add it to the WadFileList */
	if (!curw)
	{
		curw = (WadPtr) GetMemory( sizeof(struct WadFileInfo));
		if (!prevw)
			WadFileList = curw;
		else
			prevw->next = curw;
		curw->next = NULL;
		curw->filename = filename;
	}

	/* open the file */
	if ((curw->fileinfo = fopen( filename, "rb")) == NULL)
	{
		if (!prevw)
			WadFileList = NULL;
		else
			prevw->next = curw->next;
		FreeMemory( curw);
		ProgError( "error opening \"%s\"", filename);
	}

	/* read in the WAD directory info */
	BasicWadRead( curw, curw->type, 4);
	if (strncmp( curw->type, "IWAD", 4) != 0 && strncmp( curw->type, "PWAD", 4) != 0)
		ProgError( "\"%s\" is not a valid WAD file", filename);
	BasicWadRead( curw, &curw->dirsize, sizeof(curw->dirsize));
	swaplong( &(curw->dirsize));
	BasicWadRead( curw, &curw->dirstart, sizeof(curw->dirstart));
	swaplong( &(curw->dirstart));

	/* read in the WAD directory itself */
	curw->directory = (DirPtr) GetMemory( curw->dirsize * sizeof(struct Directory));
	BasicWadSeek( curw, curw->dirstart);
	BasicWadRead( curw, curw->directory, curw->dirsize * sizeof(struct Directory));
#ifdef DM_BIG_ENDIAN
	for (i = 0; i < curw->dirsize; i++)
	{
		DirPtr d = &(curw->directory[i]);
		swaplong( &(d->start));
		swaplong( &(d->size));
	}
#endif

	return curw;
}


/*
	read bytes from a file and store it into an address with error checking
*/
void BasicWadRead( WadPtr wadfile, void *addr, BCLNG size)
{
	if (fread( addr, 1, size, wadfile->fileinfo) != size)
		ProgError( "error reading from \"%s\"", wadfile->filename);
}

/*
	go to offset of a file with error checking
*/
void BasicWadSeek( WadPtr wadfile, BCLNG offset)
{
	if (fseek( wadfile->fileinfo, offset, 0))
		ProgError( "error reading from \"%s\"", wadfile->filename);
}


/*
	find an entry in the master directory
*/
MDirPtr FindMasterDir( MDirPtr from, char *name)
{
	while (from)
	{
		if (strncmp( from->dir.name, name, 8) == 0)
			break;
		from = from->next;
	}
	return from;
}

/*
	find an entry in this level's directory section
*/
MDirPtr FindLevelDir( MDirPtr from, char *name)
{
	/* start from lump after level marker */
	from = from->next;
	while (from)
	{
		if (!IsLevelLump( from->dir.name))
			return NULL;
		/* check for desired lump type */
		if (strncmp( from->dir.name, name, 8) == 0)
			break;
		from = from->next;
	}
	return from;
}

/*
	check whether name is level-start lump
*/
Bool IsLevelStart( char *name)
{
	int i;

	/* check for UDMF marker */
	if (UDMFmark != NULL)
	{
		for (i = 0; i < strlen( UDMFmark) && name[i] == UDMFmark[i]; i++)
			;
		if (i < strlen( UDMFmark))
			return FALSE;
		else
		{
			if (name[i] < '0' || name[i] > '9')
				return FALSE;
			i++;
			if (name[i] < '0' || name[i] > '9')
				return FALSE;
		}
		return TRUE;
	}
	/* check UDMF title marker */
	if (strcmp( name, "TITLEMAP") == 0)
		return TRUE;

	/* check Doom II, Hexen or Strife */
	if (GameVersion == 0x02 || GameVersion >= 0x20)
		if (name[0] == 'M' &&
		    name[1] == 'A' &&
		    name[2] == 'P' &&
		    name[3] >= '0' && name[3] <= '9' &&
		    name[4] >= '0' && name[4] <= '9' &&
		    (name[5] == '\0' || name[6] == '\0'))
			return TRUE;
		else
			return FALSE;
	else // Doom or Heretic
		if (name[0] == 'E' &&
		    name[1] >= '0' && name[1] <= '6' &&
		    name[2] == 'M' &&
		    name[3] >= '0' && name[3] <= '9' &&
		    (name[4] == '\0' || name[5] == '\0'))
			return TRUE;
		else
			return FALSE;
}

#define NUMLUMPS 19

/*
	check whether name is level-related lump
*/
Bool IsLevelLump( char *name)
{
	BCINT n;
	char *llumps[NUMLUMPS] = {
		"THINGS",
		"VERTEXES",
		"LINEDEFS",
		"SIDEDEFS",
		"SECTORS",
		"SEGS",
		"SSECTORS",
		"NODES",
		"REJECT",
		"BLOCKMAP",
		"BEHAVIOR",
		"SCRIPTS",
		"LEAFS",
		"LIGHTS",
		"MACROS",
		"TEXTMAP",
		"ZNODES",
		"DIALOGUE",
		"ENDMAP"
	};

	/* check known level lump types */
	for (n = 0; n < NUMLUMPS; n++)
		if (strncmp( name, llumps[n], 8) == 0)
			return TRUE;

	return FALSE;
}

/*
	check if a file exists and is readable
*/
Bool Exists( char *filename)
{
	FILE *test;

	if (!(test = fopen( filename, "rb")))
		return FALSE;
	fclose( test);
	return TRUE;
}

/*
	list the master directory
*/
void ListMasterDirectory( void)
{
	MDirPtr dir;
	char dataname[ 9];

	dataname[ 8] = '\0';
	printf( "\nThe Master Directory\n");
	printf( "====================\n\n");
	printf( "NAME____  FILE________________  SIZE__  START____\n");
	for (dir = MasterDir; dir; dir = dir->next)
	{
		strncpy( dataname, dir->dir.name, 8);
		printf( "%-8s  %-20s  %6d  x%08x\n", dataname,
		        dir->wadfile->filename, dir->dir.size, dir->dir.start);
	}
}

/* vim:set noexpandtab tabstop=2 softtabstop=2 shiftwidth=2: */
