/*
	DooM MaP StaTistics, by Frans P. de Vries.

Derived from:

	DooM PostScript Maps Utility, by Frans P. de Vries.

And thus from:

	Doom Editor Utility, by Brendon Wyber and Raphaël Quinet.

	You are allowed to use any parts of this code in another program, as
	long as you give credits to the authors in the documentation and in
	the program itself.  Read the file README for more information.

	This program comes with absolutely no warranty.

	APPEAR.C - Appearance stats & maps list routines.
*/

#include "dmmpst.h"
#include "appear.h"
#include "stats.h"
#include "levels.h"
#include "things.h"


/*
	names for all IWADs
*/
char *IWADNames[13] = {
/* 0 */ "Doom (registered)",
/* 1 */ "Ultimate Doom",
/* 2 */ "Doom II",
/* 3 */ "TNT: Evilution",
/* 4 */ "The Plutonia Experiment",
/* 5 */ "Heretic (registered)",
/* 6 */ "Shadow of the Serpent Riders",
/* 7 */ "Hexen",
/* 8 */ "Deathkings of the Dark Citadel",
/* 9 */ "Strife",
/* 10 */ "Strife: Veteran Edition",
/* 11 */ "Doom 64",
/* 12 */ "The Lost Levels",
};

/*
	official names for all Ultimate Doom episodes
*/
char *EpiNames1[4] = {
	"Knee-Deep in the Dead",
	"The Shores of Hell",
	"Inferno",
	"Thy Flesh Consumed",
};

/*
	official names for all Heretic episodes
*/
char *EpiNamesH[6] = {
	"City of the Damned",
	"Hell's Maw",
	"The Dome of D'Sparil",
	"The Ossuary",
	"The Stagnant Demesne",
	"Fate's Path",
};

/*
	official names for all Hexen & Deathkings hubs
*/
char *HubNamesX[5] = {
	"Hub 1: Seven Portals",
	"Hub 2: Shadow Wood",
	"Hub 3: Heresiarch's Seminary",
	"Hub 4: Castle of Grief",
	"Hub 5: Necropolis",
};

char *HubNamesD[4] = {
	"Expansion Hub 1: Blight",
	"Expansion Hub 2: Constable's Gate",
	"Expansion Hub 3: Nave",
	"Expansion Hub 4: Transit",
};


/*
	the appearance types and global variables
*/
typedef struct {
	BCINT iwad;
	BCINT epihub;
	UBCINT easy[ThMode_End];
	UBCINT medm[ThMode_End];
	UBCINT hard[ThMode_End];
	BCINT epeasy[ThMode_End];
	BCINT mseasy[ThMode_End];
	BCINT epmedm[ThMode_End];
	BCINT msmedm[ThMode_End];
	BCINT ephard[ThMode_End];
	BCINT mshard[ThMode_End];
} ApTally;

UBCINT NumApTallies = 0;
ApTally **ApTallies = NULL;

UBCINT NumUsedApprs = 0;
char    **UsedApprs = NULL;

/*
	the maps list types and global variables
*/
typedef struct {
	BCINT iwad;
	BCINT epis;
	BCINT miss;
	UBCINT thngs;
	UBCINT verts;
	UBCINT vertb;
	UBCINT lines;
	UBCINT sides;
	UBCINT sects;
	UBCINT sizex;
	UBCINT sizey;
	BCLNG  area;
	UBCINT scrts;
	UBCINT lands;
} MLTally;

UBCINT NumMLTallies = 0;
MLTally **MLTallies = NULL;


/*
	the local variables
*/
BCINT FirstVersion = -1; /* initial game and version */
char *ThingName = NULL; /* wiki name of the thing */

/*
	the local function prototypes
*/
ApTally *InitAppearances( BCINT, BCINT);
void Version2IWAD( BCINT *, BCINT *, BCINT *);
void IWAD2Lump( BCINT, BCINT, BCINT, Bool, BCINT *, BCINT *);
char *Lump2Name( BCINT, BCINT, BCINT);
void LevelAppearances( BCINT, BCINT, BCINT, ApTally *);
char *UsedAppearance( char *);

MLTally *InitMapsList( BCINT);
void LevelStatistics( BCINT, BCINT, BCINT, MLTally *);


/*
	read IWAD(s) & analyze thing appearances
*/
void AnalyzeAppearances( void)
{
	ApTally *aptr;
	BCINT iwad, episodes, missions;
	Bool found;
	int e, m, i;

	/* check WAD version & compatible games */
	if (FirstVersion == -1)
		FirstVersion = GameVersion;
	else
		if ((FirstVersion & 0xF0) != (GameVersion & 0xF0))
			ProgError( "additional WAD is incompatible with initial WAD");

	/* obtain IWAD id & episode/mission totals */
	Version2IWAD( &iwad, &episodes, &missions);

	/* check for episodic IWAD */
	if (episodes > 1)
	{
		/* tally thing type per episode */
		for (e = 0; e < episodes; e++)
		{
			/* check whether registered episode was already tallied */
			if (iwad == 1 || iwad == 6) // Ultimate Doom or Heretic: SSR
			{
				found = FALSE;
				for (i = 0; i < NumApTallies; i++)
					if (ApTallies[i]->iwad == iwad-1 && ApTallies[i]->epihub == e)
					{
						found = TRUE;
						break;
					}
				if (found)
					continue;
			}

			/* initialize episode appearances */
			aptr = InitAppearances( iwad, e);

			/* tally thing type on episode's levels */
			for (m = 0; m < missions; m++)
				LevelAppearances( iwad, e, m, aptr);

			/* append tally to global array */
			ApTallies = (ApTally **) ResizeMemory( ApTallies, (NumApTallies+1) * sizeof(aptr));
			ApTallies[NumApTallies++] = aptr;
		}
	}

	/* initialize IWAD appearances */
	aptr = InitAppearances( iwad, -1);

	/* tally thing type on all levels */
	for (e = 0; e < episodes; e++)
		for (m = 0; m < missions; m++)
			LevelAppearances( iwad, e, m, aptr);

	/* append tally to global array */
	ApTallies = (ApTally **) ResizeMemory( ApTallies, (NumApTallies+1) * sizeof(aptr));
	ApTallies[NumApTallies++] = aptr;
}


/*
	initialise an appearance tally
*/
ApTally *InitAppearances( BCINT iwad, BCINT epis)
{
	ApTally *aptr;
	int i;

	aptr = (ApTally *) GetMemory( sizeof(ApTally));
	aptr->iwad = iwad;
	aptr->epihub = epis;
	for (i = 0; i < ThMode_End; i++)
	{
		aptr->easy[i] = aptr->medm[i] = aptr->hard[i] = 0;
		aptr->epeasy[i] = aptr->epmedm[i] = aptr->ephard[i] = -1;
		aptr->mseasy[i] = aptr->msmedm[i] = aptr->mshard[i] = -1;
	}

	return aptr;
}

/*
	forget all appearance data
*/
void ForgetAppearances( void)
{
	int n;

	for (n = 0; n < NumApTallies; n++)
		if (ApTallies[n] != NULL)
			FreeMemory( ApTallies[n]);

	FreeMemory( ApTallies);
	ApTallies = NULL;
	NumApTallies = 0;

	for (n = 0; n < NumUsedApprs; n++)
		if (UsedApprs[n] != NULL)
			FreeMemory( UsedApprs[n]);

	FreeMemory( UsedApprs);
	UsedApprs = NULL;
	NumUsedApprs = 0;

	FreeMemory( ThingName);
	ThingName = NULL;
}


/*
	convert version to IWAD id & episode/mission totals
*/
void Version2IWAD( BCINT *iwad, BCINT *epis, BCINT *miss)
{
	MDirPtr lump;

	switch (GameVersion)
	{
		case 0x01: // Doom
			*iwad = 0;
			*epis = 3;
			*miss = 9;
			break;
		case 0x04: // Ultimate Doom
			*iwad = 1;
			*epis = 4;
			*miss = 9;
			break;
		case 0x02: // check Doom II/Final Doom
			lump = FindMasterDir( MasterDir, "PNAMES");
			if (lump->dir.size == 5332)
				*iwad = 3; // TNT
			else if (lump->dir.size == 4380)
				*iwad = 4; // Plutonia
			else // 3756
				*iwad = 2; // II
			*epis = 1;
			*miss = 32;
			break;
		case 0x11: // Heretic
			*iwad = 5;
			*epis = 3;
			*miss = 9;
			break;
		case 0x14: // SSR
			*iwad = 6;
			*epis = 6;
			*miss = 9;
			break;
		case 0x42: // Hexen
			*iwad = 7;
			*epis = 5;
			*miss = 7;
			break;
		case 0x48: // DDC
			*iwad = 8;
			*epis = 4;
			*miss = 7;
			break;
		case 0x22: // Strife
			*iwad = 9;
			*epis = 1;
			*miss = 34;
			break;
		case 0x28: // Strife: VE
			*iwad = 10;
			*epis = 1;
			*miss = 38;
			break;
		case 0x82: // Doom 64
			*iwad = 11;
			*epis = 1;
			*miss = 33;
			break;
		case 0x88: // The Lost Levels
			*iwad = 12;
			*epis = 1;
			*miss = 7;
			break;
		default:
			ProgError( "unsupported shareware/demo version (%02x)", GameVersion);
	}
	if (Verbose)
		printf( "Main WAD name: %s.\n", IWADNames[*iwad]);
}

/*
	convert IWAD level to lump numbers for appearances & map lists
*/
void IWAD2Lump( BCINT iwad, BCINT e, BCINT m, Bool list, BCINT *epis, BCINT *miss)
{
	/* map lists for all Ultimate Doom episodes */
	BCINT EpiLists1[6][9] = {
		{  1, 2, 3, 9, 4, 5, 6, 7, 8 },
		{  1, 2, 3, 4, 5, 9, 6, 7, 8 },
		{  1, 2, 3, 4, 5, 6, 9, 7, 8 },
		{  1, 2, 9, 3, 4, 5, 6, 7, 8 },
	};

	/* map list for Doom II/Final Doom */
	BCINT MapList2[32] = {
		 1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 31,
		32, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30
	};

	/* map lists for all Heretic: SSR episodes */
	BCINT EpiListsH[6][9] = {
		{  1, 2, 3, 4, 5, 6, 9, 7, 8 },
		{  1, 2, 3, 4, 9, 5, 6, 7, 8 },
		{  1, 2, 3, 4, 9, 5, 6, 7, 8 },
		{  1, 2, 3, 4, 9, 5, 6, 7, 8 },
		{  1, 2, 3, 9, 4, 5, 6, 7, 8 },
		{  1, 2, 3,-1,-1,-1,-1,-1,-1 },
	};

	/* map lists for all Hexen & Deathkings hubs */
	BCINT HubListsX[5][7] = {
		{  1,  2,  3,  4,  5,  6, -1 },
		{ 13,  8,  9, 10, 11, 12, -1 },
		{ 27, 32, 33, 34, 28, 30, 31 },
		{ 22, 21, 23, 24, 25, 26, -1 },
		{ 35, 36, 37, 38, 39, 40, -1 },
	};
	BCINT HubListsD[4][7] = {
		{ 41, 42, 45, 46, 43, 44, 47 },
		{ 48, 50, 52, 51, 49, 53, -1 },
		{ 54, 55, 57, 58, 56, 59, 60 },
		{ 33, 34, 35, 36, 37, 38, -1 },
	};

	/* map list for Doom 64 */
	BCINT MapList64[33] = {
		 1, 32,  2,  3,  4, 29,  5,  6,  7,  8,  9, 10, 11, 12, 30, 13,
		14, 15, 16, 17, 18, 31, 19, 20, 21, 22, 23, 24, 28, 25, 26, 27, -1
	};

	/* map list for The Lost Levels */
	BCINT MapList64L[7] = {
		34, 35, 36, 37, 38, 39, 40
	};

	switch (iwad)
	{
		case 0: // Doom (reg)
		case 1: // Ultimate Doom
			*epis = e+1;
			*miss = (list ? m+1 : EpiLists1[e][m]);
			break;
		case 2: // Doom II
		case 3: // TNT
		case 4: // Plutonia
			*epis = e;
			*miss = (list ? m+1 : MapList2[m]);
			break;
		case 5: // Heretic
		case 6: // SSR
			*epis = e+1;
			*miss = (list ? m+1 : EpiListsH[e][m]);
			/* check for partial episode */
			if (list && *epis == 6 && *miss > 3)
				*miss = -1;
			break;
		case 7: // Hexen
			*epis = e;
			*miss = HubListsX[e][m];
			break;
		case 8: // DDC
			*epis = e;
			*miss = HubListsD[e][m];
			break;
		case 9: // Strife
		case 10: // VE
			*epis = e;
			*miss = m+1;
			break;
		case 11: // Doom64
			*epis = e;
			*miss = (list ? m+1 : MapList64[m]);
			break;
		case 12: // The Lost Levels
			*epis = e;
			*miss = MapList64L[m];
			break;
	}
}


/*
	load level and tally thing type total & first occurrence
*/
void LevelAppearances( BCINT iwad, BCINT e, BCINT m, ApTally *aptr)
{
	BCINT lumpver, epis, miss;
	int i, item;
	Bool first = TRUE;

	/* convert level to lump numbers and read data */
	IWAD2Lump( iwad, e, m, FALSE, &epis, &miss);
	if (miss == -1) // skip non-existent level
		return;
	lumpver = ReadLevelData( epis, miss);

	/* initialize and analyze thing statistics */
	switch (GameVersion & 0xF0)
	{
		case 0x00: // fall thru
		case 0x80: InitDoomThings(); break;
		case 0x10: InitHereticThings(); break;
		case 0x20: InitStrifeThings( 0); break;
		case 0x40: InitHexenThings(); break;
	}
	AnalyzeThings( lumpver, FALSE);

	/* find item for this thing */
	item = -1;
	for (i = 0; i < NumThItems; i++)
		if (AppearType == ThItems[i]->type)
		{
			/* kludge for Doom 64 to match THING_64_LASER instead */
			if (first && (lumpver & 0xF0) == 0x80 && AppearType == THING_WOLF3DSS)
			{
				first = FALSE;
				continue;
			}
			item = i;
			break;
		}
	if (item == -1)
		ProgError( "unsupported thing type (%d)", AppearType);

	/* remember thing name */
	if (ThingName == NULL)
		ThingName = strdup( ThItems[item]->wikiname);

	/* check all thing modes */
	for (i = 0; i < ThMode_End; i++)
	{
		/* tally item for this level */
		aptr->easy[i] += ThItems[item]->skill[1][i];
		aptr->medm[i] += ThItems[item]->skill[2][i];
		aptr->hard[i] += ThItems[item]->skill[3][i];

		/* check first appearance for this item */
		if (ThItems[item]->skill[1][i] > 0 && aptr->mseasy[i] == -1)
		{
			aptr->epeasy[i] = epis;
			aptr->mseasy[i] = miss;
		}
		if (ThItems[item]->skill[2][i] > 0 && aptr->msmedm[i] == -1)
		{
			aptr->epmedm[i] = epis;
			aptr->msmedm[i] = miss;
		}
		if (ThItems[item]->skill[3][i] > 0 && aptr->mshard[i] == -1)
		{
			aptr->ephard[i] = epis;
			aptr->mshard[i] = miss;
		}
	}

	/* forget data of this level */
	ForgetThings();
	ForgetLevelData();
}


/*
	write the appearance data to template file
*/
void WriteAppearances( void)
{
	char *wktpfile = NULL, *mapname, *wikiname;
	int m, n, mode, iwads = 0;
	TMPL_varlist *tpldata = NULL, *tmode, *agroup;
	TMPL_loop    *tlmodes = NULL, *algroups;

	if (Verbose)
		printf( "Thing name: %s.\n", ThingName);

	/* count IWADs & episodes/hubs in appearances */
	for (n = 0; n < NumApTallies; n++)
		if (ApTallies[n]->epihub == -1)
			iwads++;
	if (iwads > 1)
		tpldata = TMPL_add_var( tpldata, "APPLURAL", "s", NULL);

	/* add thing name & game-specific skills anchor */
	tpldata = TMPL_add_var( tpldata, "THITEM", strlwr( ThingName), NULL);
	tpldata = tmpl_add_skill( tpldata);

	/* process all relevant modes */
	for (m = 0; m < ThMode_End; m++)
	{
		/* check for MP-only/DM */
		if (WXcludeSCP)
		{
			/* process only deathmatch or multiplayer modes */
			if ((GameVersion & 0xF0) == 0x40) // Hexen format
			{
				if (m != ThModeDM)
					continue;
			}
			else // Doom format
			{
				if (m != ThModeMP)
					continue;
			}
		}
		else
		{
			if ((GameVersion & 0xF0) == 0x80) // Doom 64 format
			{
				/* process only single-player mode */
				if (m != ThModeSP)
					continue;
			}
			else // other formats
			{
				/* process only single- and multiplayer/cooperative modes */
				if (m != ThModeSP && m != ThModeMP) // or ThModeCP for Hexen
					continue;
			}
		}
		tmode = NULL;
		/* control mode columns layout */
		if (m == 0)
			tmode = TMPL_add_var( tmode, "THFIRST", "first", NULL);

		/* process all appearance groups */
		algroups = NULL;
		for (n = 0; n < NumApTallies; n++)
		{
			/* add mode header */
			mode = m;
			if (mode == ThModeMP)
			{
				/* combine Hexen/DDC's ThModeCP into ThModeMP */
				if (ApTallies[n]->iwad == 7 || ApTallies[n]->iwad == 8)
				{
					mode = ThModeCP;
					tmode = TMPL_add_var( tmode, "THMODE", "Cooperative", NULL);
				}
				else
					tmode = TMPL_add_var( tmode, "THMODE", "Multiplayer", NULL);
			}
			else if (mode == ThModeDM)
				tmode = TMPL_add_var( tmode, "THMODE", "Deathmatch", NULL);
			else // mode == ThModeSP
				tmode = TMPL_add_var( tmode, "THMODE", "Single-player", NULL);

			/* add (sub)group name */
			if (ApTallies[n]->epihub == -1) // do IWAD
			{
				wikiname = UsedAppearance( strdup( IWADNames[ApTallies[n]->iwad]));
				agroup = TMPL_add_var( NULL, "APGROUP", wikiname, NULL);
				FreeMemory( wikiname);
				/* don't do first occurrence for episodic/hub-based IWAD */
				if (ApTallies[n]->iwad != 0 && ApTallies[n]->iwad != 1 && // Doom reg/ult
				    ApTallies[n]->iwad != 5 && ApTallies[n]->iwad != 6 && // Heretic/SSR
				    ApTallies[n]->iwad != 7 && ApTallies[n]->iwad != 8)   // Hexen/DDC
					agroup = TMPL_add_var( agroup, "APFIRST", "do", NULL);
			}
			else // do episode/hub
			{
				switch (ApTallies[n]->iwad)
				{
					case 0:
					case 1:
						wikiname = UsedAppearance( strdup( EpiNames1[ApTallies[n]->epihub]));
						agroup = TMPL_add_var( NULL, "APGROUP", wikiname, NULL);
						FreeMemory( wikiname);
						tmode = TMPL_add_var( tmode, "APSECT", "e", NULL);
						break;
					case 5:
					case 6:
						wikiname = UsedAppearance( strdup( EpiNamesH[ApTallies[n]->epihub]));
						agroup = TMPL_add_var( NULL, "APGROUP", wikiname, NULL);
						FreeMemory( wikiname);
						tmode = TMPL_add_var( tmode, "APSECT", "e", NULL);
						break;
					case 7:
						wikiname = UsedAppearance( strdup( HubNamesX[ApTallies[n]->epihub]));
						agroup = TMPL_add_var( NULL, "APGROUP", wikiname, NULL);
						FreeMemory( wikiname);
						tmode = TMPL_add_var( tmode, "APSECT", "h", NULL);
						break;
					case 8:
						wikiname = UsedAppearance( strdup( HubNamesD[ApTallies[n]->epihub]));
						agroup = TMPL_add_var( NULL, "APGROUP", wikiname, NULL);
						FreeMemory( wikiname);
						tmode = TMPL_add_var( tmode, "APSECT", "h", NULL);
						break;
				}
				/* always do first occurrence */
				agroup = TMPL_add_var( agroup, "APFIRST", "do", NULL);
				/* set italic flag for section (episode/hub) name */
				agroup = TMPL_add_var( agroup, "APFORMAT", "i", NULL);
			}

			/* add tallies for this group */
			agroup = tmpl_add_int( agroup, "THEASY", ApTallies[n]->easy[mode]);
			agroup = tmpl_add_int( agroup, "THMEDM", ApTallies[n]->medm[mode]);
			agroup = tmpl_add_int( agroup, "THHARD", ApTallies[n]->hard[mode]);

			/* add firsts for this group */
			mapname = Lump2Name( ApTallies[n]->iwad, ApTallies[n]->epeasy[mode], ApTallies[n]->mseasy[mode]);
			wikiname = UsedAppearance( mapname);
			agroup = TMPL_add_var( agroup, "MAPEASY", wikiname, NULL);
			FreeMemory( wikiname);
			mapname = Lump2Name( ApTallies[n]->iwad, ApTallies[n]->epmedm[mode], ApTallies[n]->msmedm[mode]);
			wikiname = UsedAppearance( mapname);
			agroup = TMPL_add_var( agroup, "MAPMEDM", wikiname, NULL);
			FreeMemory( wikiname);
			mapname = Lump2Name( ApTallies[n]->iwad, ApTallies[n]->ephard[mode], ApTallies[n]->mshard[mode]);
			wikiname = UsedAppearance( mapname);
			agroup = TMPL_add_var( agroup, "MAPHARD", wikiname, NULL);
			FreeMemory( wikiname);

			algroups = TMPL_add_varlist( algroups, agroup);
		}
		tmode = TMPL_add_loop( tmode, "APGROUPS", algroups);
		tlmodes = TMPL_add_varlist( tlmodes, tmode);
	}
	tpldata = TMPL_add_loop( tpldata, "THMODES", tlmodes);

	/* define and fill DoomWiki template */
	if (asprintf( &wktpfile, "%s/%s", (WkTpPath != NULL ? WkTpPath : "."), "appear.tpl") == -1)
		ProgError( "unable to concatenate template path");
	TMPL_write( wktpfile, NULL, NULL, tpldata, WkFile, stderr);

	TMPL_free_varlist( tpldata);
	tpldata = NULL;
	FreeMemory( wktpfile);
	wktpfile = NULL;
}

/*
	convert IWAD & lump numbers to map name
	allocates the returned string which the caller must free
*/
char *Lump2Name( BCINT iwad, BCINT e, BCINT m)
{
	char *name = NULL;
	int i;

	/* check for no map */
	if (m == -1)
		return strdup( "-");
	switch (iwad)
	{
		case 0:
		case 1:
			if (asprintf( &name, "E%dM%d: %s", e, m, LevelNames1[e-1][m-1]) == -1)
				ProgError( "unable to concatenate map name");
			return name;
		case 2:
			if (asprintf( &name, "MAP%02d: %s", m, LevelNames2[m-1]) == -1)
				ProgError( "unable to concatenate map name");
			return name;
		case 3:
			if (asprintf( &name, "MAP%02d: %s", m, LevelNamesT[m-1]) == -1)
				ProgError( "unable to concatenate map name");
			return name;
		case 4:
			if (asprintf( &name, "MAP%02d: %s", m, LevelNamesP[m-1]) == -1)
				ProgError( "unable to concatenate map name");
			return name;
		case 5:
		case 6:
			if (asprintf( &name, "E%dM%d: %s", e, m, LevelNamesH[e-1][m-1]) == -1)
				ProgError( "unable to concatenate map name");
			return name;
		case 7:
			/* check for map name same as hub name & add postfix */
			for (i = 0; i < 5; i++)
				if (strcmp( LevelNamesX[m-1], HubNamesX[i]) == 0)
				{
					if (asprintf( &name, "%s (map)", LevelNamesX[m-1]) == -1)
						ProgError( "unable to add postfix to map name");
					return name;
				}
			return strdup( LevelNamesX[m-1]);
		case 8:
			return strdup( LevelNamesD[m-31]);
		case 9:
		case 10:
			if (asprintf( &name, "MAP%02d: %s", m, LevelNamesS[m-1]) == -1)
				ProgError( "unable to concatenate map name");
			return name;
		case 11:
		case 12:
			if (asprintf( &name, "MAP%02d: %s", m, LevelNames64[m-1]) == -1)
				ProgError( "unable to concatenate map name");
			return name;
	}
}

/*
	use linked wiki name for the first wiki link, and then wiki name
	allocates the returned string which the caller must free
*/
char *UsedAppearance( char *wikiname)
{
	char *wikistr = NULL, *postfix;
	int i;

	/* check for and trim off " (registered)" */
	if ((postfix = strstr( wikiname, " (registered)")) != NULL)
	{
		if (asprintf( &wikistr, "[[%.*s|%s]]", (int) (postfix-wikiname), wikiname, wikiname) == -1)
			ProgError( "unable to concatenate trimmed appearance string");
	}
	else
	{
		if (asprintf( &wikistr, (strcmp( wikiname, "-") == 0 ? "%s" : "[[%s]]"), wikiname) == -1)
			ProgError( "unable to concatenate appearance string");
	}
	/* return name if already used */
	for (i = 0; i < NumUsedApprs; i++)
		if (strcmp( wikiname, UsedApprs[i]) == 0)
		{
			FreeMemory( wikistr);
			return strdup( wikiname);
		}

	/* append name to global array */
	UsedApprs = (char **) ResizeMemory( UsedApprs, (NumUsedApprs+1) * sizeof(char *));
	UsedApprs[NumUsedApprs++] = wikiname;

	/* return linked name */
	return wikistr;
}


/*
	read IWAD(s) & analyze map statistics
*/
void AnalyzeMapsList( void)
{
	MLTally *lptr;
	BCINT iwad, episodes, missions;
	Bool found;
	int e, m, i;

	/* check WAD version & compatible games */
	if (FirstVersion == -1)
		FirstVersion = GameVersion;
	else
		if ((FirstVersion & 0xF0) != (GameVersion & 0xF0))
			ProgError( "additional WAD is incompatible with initial WAD");

	/* obtain IWAD id & episode/mission totals */
	Version2IWAD( &iwad, &episodes, &missions);

	/* init & tally statistics on all levels */
	for (e = 0; e < episodes; e++)
		for (m = 0; m < missions; m++)
		{
			lptr = InitMapsList( iwad);
			LevelStatistics( iwad, e, m, lptr);

			/* append tally to global array */
			MLTallies = (MLTally **) ResizeMemory( MLTallies, (NumMLTallies+1) * sizeof(lptr));
			MLTallies[NumMLTallies++] = lptr;
		}
}


/*
	initialise a maps list tally
*/
MLTally *InitMapsList( BCINT iwad)
{
	MLTally *lptr;
	int i;

	lptr = (MLTally *) GetMemory( sizeof(MLTally));
	lptr->iwad = iwad;
	lptr->epis = lptr->miss = 0;
	lptr->thngs = lptr->verts = lptr->vertb = 0;
	lptr->lines = lptr->sides = lptr->sects = 0;
	lptr->sizex = lptr->sizex = lptr->scrts = lptr->lands = 0;

	return lptr;
}

/*
	forget all maps list data
*/
void ForgetMapsList( void)
{
	int n;

	for (n = 0; n < NumMLTallies; n++)
		if (MLTallies[n] != NULL)
			FreeMemory( MLTallies[n]);

	FreeMemory( MLTallies);
	MLTallies = NULL;
	NumMLTallies = 0;
}


/*
	load level and tally statistics
*/
void LevelStatistics( BCINT iwad, BCINT e, BCINT m, MLTally *lptr)
{
	BCINT lumpver;
	int i, item;

	/* convert level to lump numbers and read data */
	IWAD2Lump( iwad, e, m, TRUE, &lptr->epis, &lptr->miss);
	if (lptr->miss == -1) // skip non-existent level
		return;
	lumpver = ReadLevelData( lptr->epis, lptr->miss);

	/* collect relevant statistics */
	lptr->thngs = NumThings;
	lptr->verts = TotVertexes;
	lptr->vertb = NumVertexes;
	lptr->lines = NumLineDefs;
	lptr->sides = NumSideDefs;
	lptr->sects = NumSectors;
	lptr->sizex = MapMaxX - MapMinX;
	lptr->sizey = MapMaxY - MapMinY;
	lptr->area  = lptr->sizex * lptr->sizey;

	lptr->scrts = CountSecrets( lumpver);
	lptr->lands = CountThings( THING_TELEPORT);
/*
	printf("%d %d: T: %d V: %d B: %d L: %d S: %d S: %d X: %d Y: %d S: %d T: %d\n",
	       lptr->epis, lptr->miss,
	       lptr->thngs, lptr->verts, lptr->vertb, lptr->lines, lptr->sides, lptr->sects,
	       lptr->sizex, lptr->sizey, lptr->scrts, lptr->lands);
*/
	/* forget data of this level */
	ForgetLevelData();
}


/*
	write the maps list data to template file
*/
void WriteMapsList( void)
{
	char *wktpfile = NULL, *mapname;
	int n;
	TMPL_varlist *tpldata = NULL, *tplmap;
	TMPL_loop    *tplloop = NULL;

	/* process all maps */
	for (n = 0; n < NumMLTallies; n++)
	{
		if (MLTallies[n]->miss == -1) // skip non-existent level
			continue;
		/* add name for this map */
		mapname = Lump2Name( MLTallies[n]->iwad, MLTallies[n]->epis, MLTallies[n]->miss);
		tplmap = TMPL_add_var( NULL, "MAPNAME", mapname, NULL);
		FreeMemory( mapname);

		/* add stats for this map */
		tplmap = tmpl_add_int( tplmap, "NUMTHNG", MLTallies[n]->thngs);
		tplmap = tmpl_add_int( tplmap, "NUMVERT", MLTallies[n]->verts);
		tplmap = tmpl_add_int( tplmap, "NUMVBEF", MLTallies[n]->vertb);
		tplmap = tmpl_add_int( tplmap, "NUMLINE", MLTallies[n]->lines);
		tplmap = tmpl_add_int( tplmap, "NUMSIDE", MLTallies[n]->sides);
		tplmap = tmpl_add_int( tplmap, "NUMSECT", MLTallies[n]->sects);
		tplmap = tmpl_add_int( tplmap, "SIZEX",   MLTallies[n]->sizex);
		tplmap = tmpl_add_int( tplmap, "SIZEY",   MLTallies[n]->sizey);
		tplmap = tmpl_add_int( tplmap, "MAPAREA", MLTallies[n]->area);
		tplmap = tmpl_add_int( tplmap, "NUMSECR", MLTallies[n]->scrts);
		tplmap = tmpl_add_int( tplmap, "NUMTELE", MLTallies[n]->lands);

		tplloop = TMPL_add_varlist( tplloop, tplmap);
	}
	tpldata = TMPL_add_loop( NULL, "MAPLIST", tplloop);

	/* define and fill DoomWiki template */
	if (asprintf( &wktpfile, "%s/%s", (WkTpPath != NULL ? WkTpPath : "."), "mapslist.tpl") == -1)
		ProgError( "unable to concatenate template path");
	TMPL_write( wktpfile, NULL, NULL, tpldata, WkFile, stderr);

	TMPL_free_varlist( tpldata);
	tpldata = NULL;
	FreeMemory( wktpfile);
	wktpfile = NULL;
}

/* vim:set noexpandtab tabstop=2 softtabstop=2 shiftwidth=2: */
