
#include "bdefs.h"
#include "model.h"
#include "model_load.h"
#include "model_ops.h"
#include "dstream.h"
#include "conparse.h"


unsigned long g_ModelGeometryMemory=0;
unsigned long g_ModelAnimationMemory=0;


extern void* dalloc(unsigned long);
extern void dfree(void *);


#define ML_MIN(a,b) ((a) < (b) ? (a) : (b))
#define ML_MAX(a,b) ((a) > (b) ? (a) : (b))



//------------------------------------------------------------------ //
// Loading functions.
//------------------------------------------------------------------ //

static void model_GetSkinCoordinates(Model *pModel, DWORD iAnim, DWORD iKeyFrame)
{
	DWORD i;
	ModelAnim *pAnim;
	AnimKeyFrame *pKeyFrame;
	DVector tVerts[3];
	ModelTri *pCurTri, *pEndTri;
	BOOL bFrontFacing;
	UVPair *pCurUVPair;
	ModelVert *pVert;
	float tu, tv;
	DVector cross, v1, v2;
	DMatrix *pMat;
	

	ASSERT(iAnim < pModel->m_nAnims);
	pAnim = &pModel->m_Anims[iAnim];

	ASSERT(iKeyFrame < pAnim->m_nKeyFrames);
	pKeyFrame = &pAnim->m_KeyFrames[iKeyFrame];

	model_CreateTransforms(pModel, pAnim, pAnim, iKeyFrame, iKeyFrame, 0.0f, pModel->m_Transforms);

	// Go thru all the tris.
	pCurUVPair = pModel->m_SkinCoords;
	pCurTri = pModel->m_ModelTris;
	pEndTri = pCurTri + pModel->m_nModelTris;
	while(pCurTri < pEndTri)
	{
		for(i=0; i < 3; i++)
		{
			pVert = &pModel->m_Verts[pCurTri->m_Indices[i]];
			
			pMat = &pModel->m_Transforms[pVert->m_TransformIndex];
			MatVMul(&tVerts[i], pMat, &pVert->m_Vec);
		}

		// Find out which way it faces.
		VEC_SUB(v1, tVerts[2], tVerts[0]);
		VEC_SUB(v2, tVerts[1], tVerts[0]);
		VEC_CROSS(cross, v1, v2);
		bFrontFacing = cross.z > 0.0f;
		
		for(i=0; i < 3; i++)
		{
			// Here is where it FINALLY gets the coordinates.
			tu = (tVerts[i].x - pKeyFrame->m_BoundMin.x) / 
				(pKeyFrame->m_BoundMax.x - pKeyFrame->m_BoundMin.x);
			
			tv = (tVerts[i].y - pKeyFrame->m_BoundMin.y) / 
				(pKeyFrame->m_BoundMax.y - pKeyFrame->m_BoundMin.y);

			tu *= 0.5f;

			if(!bFrontFacing)
			{
				tu += 0.5f;
			}

			pCurUVPair->tu = (BYTE)(tu * 255.0f);
			pCurUVPair->tv = (BYTE)(tv * 255.0f);

			++pCurUVPair;
		}
	
		++pCurTri;
	}
}

static void model_LoadGeometry(Model *pModel, DStream *pStream)
{
	DWORD i, j, size;
	ModelTri *pCurTri;
	ModelVert *pCurVert;
	UVPair *pCurUV;

	
	STREAM_READ(pModel->m_NumberOfLODs);

	// Read the vertex start numbers..
	size = sizeof(unsigned short) * (pModel->m_NumberOfLODs + 1);
	pModel->m_VertexStartNums = (unsigned short*)dalloc(size);
	pModel->m_GeometryMemory += size;
	pStream->Read(pStream, pModel->m_VertexStartNums, size);

	// Load the tris and UV coordinates.
	STREAM_READ(pModel->m_nModelTris);

	pModel->m_ModelTris = (ModelTri*)dalloc(sizeof(ModelTri)*pModel->m_nModelTris);
	pModel->m_GeometryMemory += sizeof(ModelTri)*pModel->m_nModelTris;

	size = sizeof(UVPair) * pModel->m_nModelTris * 3;
	pModel->m_UVCoords = (UVPair*)dalloc(size);
	
	pCurTri = pModel->m_ModelTris;
	pCurUV = pModel->m_UVCoords;
	for(i=0; i < pModel->m_nModelTris; i++)
	{
		for(j=0; j < 3; j++)
		{
			STREAM_READ(pCurUV->tu);
			STREAM_READ(pCurUV->tv);
			
			pCurUV->tu = DCLAMP(pCurUV->tu, 0.0001f, 0.9999f);
			pCurUV->tv = DCLAMP(pCurUV->tv, 0.0001f, 0.9999f);

			++pCurUV;
		}

		STREAM_READ(pCurTri->m_Indices[0]);
		STREAM_READ(pCurTri->m_Indices[1]);
		STREAM_READ(pCurTri->m_Indices[2]);
		STREAM_READ(pCurTri->m_Normal[0]);
		STREAM_READ(pCurTri->m_Normal[1]);
		STREAM_READ(pCurTri->m_Normal[2]);

		++pCurTri;
	}

	// Load the vertices.
	STREAM_READ(pModel->m_nVerts);
	STREAM_READ(pModel->m_nNormalVerts);
	pModel->m_Verts = (ModelVert*)dalloc(sizeof(ModelVert)*pModel->m_nVerts);
	pModel->m_GeometryMemory += sizeof(ModelVert)*pModel->m_nVerts;
	
	pCurVert = pModel->m_Verts;
	for(i=0; i < pModel->m_nVerts; i++)
	{
		STREAM_READ(pCurVert->m_Vec);
		STREAM_READ(pCurVert->m_Normal[0]);
		STREAM_READ(pCurVert->m_Normal[1]);
		STREAM_READ(pCurVert->m_Normal[2]);
		STREAM_READ(pCurVert->m_TransformIndex);
		STREAM_READ(pCurVert->m_Replacements[0]);
		STREAM_READ(pCurVert->m_Replacements[1]);
		++pCurVert;
	}

	// Fill in the transform indices for the triangles.
	pCurTri = pModel->m_ModelTris;
	for(i=pModel->m_nModelTris; i > 0; i--)
	{
		pCurTri->m_TransformIndex = pModel->m_Verts[pCurTri->m_Indices[0]].m_TransformIndex;
		++pCurTri;
	}
}

static void model_LoadNode(Model *pModel, ModelNode *pNode, DStream *pStream)
{
	DWORD i, size;
	char tempStr[500];

	// Load all the misc. data.
	STREAM_READ(pNode->m_BoundMin);
	STREAM_READ(pNode->m_BoundMax);
	
	// Read the node name.
	pStream->ReadString(pStream, tempStr, 500);
	pNode->m_pName = model_AddToStringList(pModel, tempStr);

	STREAM_READ(pNode->m_TransformIndex);
	STREAM_READ(pNode->m_Flags);

	// Read the indices.
	STREAM_READ(pNode->m_nIndices);
	pNode->m_Indices = (unsigned short*)dalloc(sizeof(unsigned short) * pNode->m_nIndices);
	pStream->Read(pStream, pNode->m_Indices, sizeof(unsigned short) * pNode->m_nIndices);

	// Load the children nodes.
	STREAM_READ(pNode->m_nChildren);
	
	size = sizeof(ModelNode) * pNode->m_nChildren;
	pNode->m_Children = (ModelNode*)dalloc(size);
	pModel->m_GeometryMemory += size;
	memset(pNode->m_Children, 0, size);

	for(i=0; i < pNode->m_nChildren; i++)
	{
		model_LoadNode(pModel, &pNode->m_Children[i], pStream);
	}
}

static void model_LoadAnimNode(Model *pModel, ModelAnim *pAnim, AnimNode *pAnimNodeParent,
	AnimNode *pAnimNode, ModelNode *pNode, DStream *pStream, AnimNode **pNextAnimNode )
{
	DWORD i, size;

	pAnimNode->m_pAnim = pAnim;
	pAnimNode->m_pNode = pNode;
	pAnimNode->m_pParentNode = pAnimNodeParent;

	// Read the keyframes.
	pAnimNode->m_KeyFrames = (NodeKeyFrame*)dalloc(sizeof(NodeKeyFrame) * pAnim->m_nKeyFrames);

	for(i=0; i < pAnim->m_nKeyFrames; i++)
	{
		pStream->Read(pStream, &pAnimNode->m_KeyFrames[i], sizeof(NodeKeyFrame));
	}

	size = sizeof(DefVertex)*pAnimNode->m_pAnim->m_nKeyFrames*pAnimNode->m_pNode->m_nIndices;
	pAnimNode->m_DefVertices = (DefVertex*)dalloc(size);
	pStream->Read(pStream, pAnimNode->m_DefVertices, size);

	STREAM_READ(pAnimNode->m_Scale);
	STREAM_READ(pAnimNode->m_Shift);

	pModel->m_AnimationMemory += sizeof(NodeKeyFrame)*pAnim->m_nKeyFrames;
	pModel->m_AnimationMemory += size; // Size of the DefVertex's.

	// Add the children...
	if( pNode->m_nChildren )
	{
		size = sizeof( AnimNode * ) * pNode->m_nChildren;
		pAnimNode->m_Children = ( AnimNode ** )dalloc( size );
		pModel->m_AnimationMemory += size;

		// Read the children.
		for(i=0; i < pNode->m_nChildren; i++)
		{
			pAnimNode->m_Children[i] = *pNextAnimNode;
			*pNextAnimNode += 1;
			model_LoadAnimNode(pModel, pAnim, pAnimNode, pAnimNode->m_Children[i], &pNode->m_Children[i], pStream, pNextAnimNode );
		}
	}
}

static void model_LoadAnim(Model *pModel, ModelAnim *pAnim, DStream *pStream)
{
	AnimKeyFrame *pKeyFrame;
	DWORD i;
	char tempStr[501];
	DWORD size;
	AnimNode *pNextAnimNode;

	// Load misc info.
	pStream->ReadString(pStream, tempStr, 500);
	pAnim->m_pName = model_AddToStringList(pModel, tempStr);

	STREAM_READ(pAnim->m_msTime);
	STREAM_READ(pAnim->m_BoundMin);
	STREAM_READ(pAnim->m_BoundMax);
	
	// Figure out the bounding radius.
	if(VEC_MAGSQR(pAnim->m_BoundMin) > VEC_MAGSQR(pAnim->m_BoundMax))
	{
		pAnim->m_BoundRadius = VEC_MAG(pAnim->m_BoundMin);
	}
	else
	{
		pAnim->m_BoundRadius = VEC_MAG(pAnim->m_BoundMax);
	}

	// Read the keyframes in.
	STREAM_READ(pAnim->m_nKeyFrames);
	size = sizeof(AnimKeyFrame) * pAnim->m_nKeyFrames;
	pAnim->m_KeyFrames = (AnimKeyFrame*)dalloc( size );
	pModel->m_AnimationMemory += size;

	for(i=0; i < pAnim->m_nKeyFrames; i++)
	{
		pKeyFrame = &pAnim->m_KeyFrames[i];

		STREAM_READ(pKeyFrame->m_Time);
		STREAM_READ(pKeyFrame->m_BoundMin);
		STREAM_READ(pKeyFrame->m_BoundMax);

		pStream->ReadString(pStream, tempStr, 500);
		pKeyFrame->m_pString = model_AddToStringList(pModel, tempStr);

		pKeyFrame->m_KeyType = KEYTYPE_POSITION;
	}

	// Default its m_vDims to the boundaries of its first keyframe.
	pAnim->m_vDims.x = DMAX(-pAnim->m_KeyFrames[0].m_BoundMin.x, pAnim->m_KeyFrames[0].m_BoundMax.x);
	pAnim->m_vDims.y = DMAX(-pAnim->m_KeyFrames[0].m_BoundMin.y, pAnim->m_KeyFrames[0].m_BoundMax.y);
	pAnim->m_vDims.z = DMAX(-pAnim->m_KeyFrames[0].m_BoundMin.z, pAnim->m_KeyFrames[0].m_BoundMax.z);

	// Allocate all the child anim nodes...
	size = sizeof( AnimNode ) * pModel->m_nNodes;
	pAnim->m_AnimNodes = ( AnimNode * )dalloc_z( size );
	pModel->m_AnimationMemory += size;

	// Load the animation nodes.
	pNextAnimNode = &pAnim->m_AnimNodes[1];
	model_LoadAnimNode(pModel, pAnim, NULL, &pAnim->m_AnimNodes[0], &pModel->m_RootNode, pStream, &pNextAnimNode );
}


static BOOL model_SeekToSection(DStream *pStream, char *pSectionName)
{
	DWORD nextSectionPos;
	char sectionName[256];

	pStream->SeekTo(pStream, 0);
	while(1)
	{
		pStream->ReadString(pStream, sectionName, 256);
		STREAM_READ(nextSectionPos);
		if(strcmp(pSectionName, sectionName) == 0)
			return TRUE;

		if(nextSectionPos == (DWORD)-1)
			return FALSE;

		pStream->SeekTo(pStream, nextSectionPos);
	}

	return FALSE;
}

// {BP 1/20/98}
// Moved this code to this function because it was used in a couple places...
void model_GetDimsFromBounding( DVector *pvBoundMax, DVector *pvBoundMin, DVector *pvDims )
{
	DVector vDimsMax;

	// Set the model dims from the first frame of the first animation bounding box...
	// Not sure if BoundMin is always negative and BoundMax is always positive, 
	// so got to do some absolute values...
	pvDims->x = ( pvBoundMin->x > 0.0f ) ? pvBoundMin->x : -pvBoundMin->x;
	pvDims->y = ( pvBoundMin->y > 0.0f ) ? pvBoundMin->y : -pvBoundMin->y;
	pvDims->z = ( pvBoundMin->z > 0.0f ) ? pvBoundMin->z : -pvBoundMin->z;
	vDimsMax.x = ( pvBoundMax->x > 0.0f ) ? pvBoundMax->x : -pvBoundMax->x;
	vDimsMax.y = ( pvBoundMax->y > 0.0f ) ? pvBoundMax->y : -pvBoundMax->y;
	vDimsMax.z = ( pvBoundMax->z > 0.0f ) ? pvBoundMax->z : -pvBoundMax->z;
	if( pvDims->x < vDimsMax.x )
		pvDims->x = vDimsMax.x;
	if( pvDims->y < vDimsMax.y )
		pvDims->y = vDimsMax.y;
	if( pvDims->z < vDimsMax.z )
		pvDims->z = vDimsMax.z;
}


void model_SetDefaultTCoords(Model *pModel)
{
	DWORD i, j;
	ModelTri *pTri;
	UVPair *pCurUV;

	pCurUV = pModel->m_UVCoords;
	for(i=0; i < pModel->m_nModelTris; i++)
	{
		pTri = &pModel->m_ModelTris[i];
		
		for(j=0; j < 3; j++)
		{
			pModel->m_Verts[pTri->m_Indices[j]].tu = pCurUV->tu;
			pModel->m_Verts[pTri->m_Indices[j]].tv = pCurUV->tv;
		
			++pCurUV;
		}
	}

	// Setup the extra LOD verts with some valid texture coordinates.
	for(i=pModel->m_nNormalVerts; i < pModel->m_nVerts; i++)
	{
		pModel->m_Verts[i].tu = pModel->m_Verts[pModel->m_Verts[i].m_Replacements[0]].tu;
		pModel->m_Verts[i].tv = pModel->m_Verts[pModel->m_Verts[i].m_Replacements[0]].tv;
	}
}


static BOOL model_InternalLoad(Model *pModel, DStream *pStream)
{
	DWORD i;
	WORD index;
	char header[100], tempStr[500];

	// Read the header.
	if(!model_SeekToSection(pStream, "Header"))
		return FALSE;

	pStream->ReadString(pStream, header, 100);
	if(strcmp(header, MODELFILE_TOKEN) != 0)
		return FALSE;

	// Read the command string.
	pStream->ReadString(pStream, tempStr, 500);
	pModel->m_CommandString = model_AddToStringList(pModel, tempStr);
	model_ParseCommandString( pModel );

	// Read the bounding box and geometry.
	if(!model_SeekToSection(pStream, "Geometry"))
		return FALSE;

	// {MD 1/16/98: These are actually overwritten below.. due to a bug in the
	// model code, most of these are screwed so it generates them when
	// it loads the model now}.
	STREAM_READ(pModel->m_BoundMin);
	STREAM_READ(pModel->m_BoundMax);

	model_LoadGeometry(pModel, pStream);


	// Load the node tree.
	if(!model_SeekToSection(pStream, "Nodes"))
		return FALSE;

	model_LoadNode(pModel, &pModel->m_RootNode, pStream);


	// Build lists.
	model_BuildFlatNodeList(pModel);
	model_BuildDNodeList(pModel);
	model_BuildTransformList(pModel);


	// Load the animation.
	if(!model_SeekToSection(pStream, "Animation"))
		return FALSE;

	STREAM_READ(pModel->m_nAnims);
	pModel->m_Anims = (ModelAnim*)dalloc(sizeof(ModelAnim)*pModel->m_nAnims);
	memset(pModel->m_Anims, 0, sizeof(ModelAnim)*pModel->m_nAnims);
	pModel->m_AnimationMemory += sizeof(ModelAnim)*pModel->m_nAnims;

	for(i=0; i < pModel->m_nAnims; i++)
	{
		model_LoadAnim(pModel, &pModel->m_Anims[i], pStream);
		model_BuildAnimDNodeList(pModel, &pModel->m_Anims[i]);
	}


	// Initialize the skin coordinates last...
	pModel->m_SkinCoords = (UVPair*)dalloc(sizeof(UVPair) * pModel->m_nModelTris * 3);
	pModel->m_GeometryMemory += sizeof(UVPair)*pModel->m_nModelTris*3;

	model_GetSkinCoordinates(pModel, 0, 0);

	// {MD 1/16/98: Generate the correct min and max dims}.
	VEC_SET(pModel->m_BoundMin, 200.0f, 200.0f, 200.0f);
	VEC_NEGATE(pModel->m_BoundMax, pModel->m_BoundMin);
	
	for(i=0; i < pModel->m_nAnims; i++)
	{
		pModel->m_BoundMin.x = ML_MIN(pModel->m_BoundMin.x, pModel->m_Anims[i].m_BoundMin.x);
		pModel->m_BoundMin.y = ML_MIN(pModel->m_BoundMin.y, pModel->m_Anims[i].m_BoundMin.y);
		pModel->m_BoundMin.z = ML_MIN(pModel->m_BoundMin.z, pModel->m_Anims[i].m_BoundMin.z);

		pModel->m_BoundMax.x = ML_MAX(pModel->m_BoundMax.x, pModel->m_Anims[i].m_BoundMax.x);
		pModel->m_BoundMax.y = ML_MAX(pModel->m_BoundMax.y, pModel->m_Anims[i].m_BoundMax.y);
		pModel->m_BoundMax.z = ML_MAX(pModel->m_BoundMax.z, pModel->m_Anims[i].m_BoundMax.z);
	}

	if(VEC_MAGSQR(pModel->m_BoundMin) > VEC_MAGSQR(pModel->m_BoundMax))
	{
		pModel->m_BoundRadius = VEC_MAG(pModel->m_BoundMin);
	}
	else
	{
		pModel->m_BoundRadius = VEC_MAG(pModel->m_BoundMax);
	}

	// Read in animation dims if the section exists.
	if( model_SeekToSection( pStream, "AnimDims" ))
	{
		for(i=0; i < pModel->m_nAnims; i++)
		{
			STREAM_READ(pModel->m_Anims[i].m_vDims);
		}
	}

	if(model_SeekToSection(pStream, "TriStrips"))
	{
		STREAM_READ(pModel->m_nTriStripIndices);
		pModel->m_TriStripIndices = (unsigned short*)dalloc(sizeof(unsigned short)*pModel->m_nTriStripIndices);
		pStream->Read(pStream, pModel->m_TriStripIndices, sizeof(unsigned short)*pModel->m_nTriStripIndices);

		STREAM_READ(pModel->m_nTriStrips);
		pModel->m_TriStrips = (ModelTriStrip*)dalloc(sizeof(ModelTriStrip)*pModel->m_nTriStrips);
		pStream->Read(pStream, pModel->m_TriStrips, sizeof(ModelTriStrip)*pModel->m_nTriStrips);

		STREAM_READ(pModel->m_nUnstrippedTris);
		pModel->m_UnstrippedTris = (ModelTri**)dalloc(sizeof(ModelTri*)*pModel->m_nUnstrippedTris);
		for(i=0; i < pModel->m_nUnstrippedTris; i++)
		{
			STREAM_READ(index);
			pModel->m_UnstrippedTris[i] = &pModel->m_ModelTris[index];
		}
	
		for(i=0; i < pModel->m_nNormalVerts; i++)
		{
			STREAM_READ(pModel->m_Verts[i].tu);
			STREAM_READ(pModel->m_Verts[i].tv);
		}
	}

	model_SetDefaultTCoords(pModel);
	return TRUE;
}


//------------------------------------------------------------------ //
// Deleting functions.
//------------------------------------------------------------------ //

static void model_RecurseAndDeleteNode(ModelNode *pNode)
{
	DWORD i;

	for(i=0; i < pNode->m_nChildren; i++)
	{
		model_RecurseAndDeleteNode(&pNode->m_Children[i]);
	}

	if(pNode->m_Indices)
	{
		dfree(pNode->m_Indices);
		pNode->m_Indices = NULL;
	}

	pNode->m_nIndices = 0;

	if(pNode->m_Children)
		dfree(pNode->m_Children);
}


//------------------------------------------------------------------ //
// Main interface functions.
//------------------------------------------------------------------ //

Model* model_Load(DStream *pStream)
{
	Model *pModel;

	// Setup the model.
	pModel = model_Create();

	// Load it up.
	if(!model_InternalLoad(pModel, pStream))
	{
		model_Destroy(pModel);
		return NULL;
	}

	if(pStream->ErrorStatus(pStream) == 1)
	{
		model_Destroy(pModel);
		return NULL;
	}

	g_ModelGeometryMemory += pModel->m_GeometryMemory;
	g_ModelAnimationMemory += pModel->m_AnimationMemory;

	return pModel;
}

short model_LoadDimensions( struct DStream_t *pStream, DVector *pvDims )
{
	unsigned long nTemp;
	DVector	vBoundMin, vBoundMax;
	char tempStr[500];

	if( !pStream || !pvDims )
		return 0;

	// {BP 1/20/98}
	// Check to see if a Dimensions section exists.  If it doesn't, then get dims from 
	// first frame of first anim...
	if( model_SeekToSection( pStream, "Dimensions" ))
	{
		STREAM_READ( pvDims->x );
		STREAM_READ( pvDims->y );
		STREAM_READ( pvDims->z );
		return 1;
	}

	// Load the animation.
	if(!model_SeekToSection(pStream, "Animation"))
		return 0;

	// Skip the number of anims
	STREAM_READ(nTemp);

	if( nTemp == 0 )
		return 0;

	// Skip the temp string.
	pStream->ReadString(pStream, tempStr, 500);

	// skip the animation data time and bounding boxes..
	STREAM_READ(nTemp);
	STREAM_READ(vBoundMin);
	STREAM_READ(vBoundMin);
	
	// skip the number of key frames...
	STREAM_READ(nTemp);
	if( nTemp == 0 )
		return 0;
	
	// Skip the time...
	STREAM_READ(nTemp);

	// Get the bounding boxes...
	STREAM_READ(vBoundMin);
	STREAM_READ(vBoundMax);

	model_GetDimsFromBounding( &vBoundMin, &vBoundMax, pvDims );

	return 1;
}

Model* model_Create()
{
	Model *pRet;

	pRet = (Model*)dalloc(sizeof(Model));
	memset(pRet, 0, sizeof(Model));
	pRet->m_GeometryMemory += sizeof(Model);

	pRet->m_pFilename = "";
	pRet->m_CommandString = "";
	pRet->m_Link.m_pData = pRet;

	VEC_SET(pRet->m_BoundMin, 200.0f, 200.0f, 200.0f);
	VEC_NEGATE(pRet->m_BoundMax, pRet->m_BoundMin);

	// Set defaults.
	pRet->m_fLodStartDist = 200.0f;
	pRet->m_fLodIncrement = 7.0f;
	pRet->m_fLodMaxDist = 200000.0f;
	pRet->m_MipmapDistAdd = 0.0f;
	pRet->m_DrawIndexedDist = 200.0f;

	nexus_Init(&pRet->m_Nexus, pRet);
	return pRet;
}


void model_Destroy(Model *pModel)
{
	DWORD i;
	ModelString *pCurString, *pNextString;


	nexus_Term(&pModel->m_Nexus);

	for(i=0; i < pModel->m_nAnims; i++)
		model_DestroyAnim(&pModel->m_Anims[i]);

	model_RecurseAndDeleteNode(&pModel->m_RootNode);

	// dfree all the strings.
	pCurString = pModel->m_StringList;
	while(pCurString)
	{
		pNextString = pCurString->m_pNext;
		dfree(pCurString);
		pCurString = pNextString;
	}

	if(pModel->m_Anims)
		dfree(pModel->m_Anims);

	if(pModel->m_UVCoords)
		dfree(pModel->m_UVCoords);

	if(pModel->m_SkinCoords)
		dfree(pModel->m_SkinCoords);

	if(pModel->m_Verts)
		dfree(pModel->m_Verts);

	if(pModel->m_FlatNodeList)
		dfree(pModel->m_FlatNodeList);

	if(pModel->m_DNodes)
		dfree(pModel->m_DNodes);

	if(pModel->m_ModelTris)
		dfree(pModel->m_ModelTris);

	if(pModel->m_Transforms)
		dfree(pModel->m_Transforms);

	if(pModel->m_VertexStartNums)
		dfree(pModel->m_VertexStartNums);

	if(pModel->m_TriStripIndices)
		dfree(pModel->m_TriStripIndices);

	if(pModel->m_TriStrips)
		dfree(pModel->m_TriStrips);

	if(pModel->m_UnstrippedTris)
		dfree(pModel->m_UnstrippedTris);

	g_ModelGeometryMemory -= pModel->m_GeometryMemory;
	g_ModelAnimationMemory -= pModel->m_AnimationMemory;
	
	dfree(pModel);
}




									  