
#include "bdefs.h"
#include <string.h>
#include <malloc.h>
#include <stdlib.h>
#include "model.h"
#include "model_ops.h"
#include "quatbase.h"
#include "conparse.h"


static int g_iCreateTransformKeys[2];
static float g_CreateTransformPercent;

extern void* dalloc(unsigned long size);
extern void* dalloc_z(unsigned long size);
extern void dfree(void *ptr);


#define MDL_FIXED_SHIFT 14
#define MDL_FIXED_ONE_F (1 << MDL_FIXED_SHIFT)
#define MDL_FIXED_ONE_INV (1.0f / (1 << MDL_FIXED_SHIFT))


static void model_DestroyAnimNode(AnimNode *pNode)
{
	unsigned long i;

	if(pNode->m_KeyFrames)
	{
		dfree(pNode->m_KeyFrames);
		pNode->m_KeyFrames = 0;
	}

	if(pNode->m_DefVertices)
	{
		dfree(pNode->m_DefVertices);
		pNode->m_DefVertices = 0;
	}

	if(pNode->m_Children)
	{	
		for(i=0; i < pNode->m_pNode->m_nChildren; i++)
			model_DestroyAnimNode(pNode->m_Children[i]);
	
		dfree(pNode->m_Children);
		pNode->m_Children = 0;
	}
}


static inline DMatrix* model_RecurseAndCreateTransforms(AnimNode *pNode1, AnimNode *pNode2,
	DMatrix *pTransform, DMatrix *pCurTransform)
{
	unsigned long i;
	DMatrix mat, *pStartTransform;
	
	static DMatrix mTemp;
	static NodeKeyFrame *pKey1, *pKey2;
	static float quat[4];


	if((pNode1->m_pNode->m_Flags & MNODE_DEFORMATION) && pNode1 != pNode2)
	{
		pKey1 = pKey2 = &pNode1->m_KeyFrames[g_iCreateTransformKeys[0]];
	}
	else
	{
		pKey1 = &pNode1->m_KeyFrames[g_iCreateTransformKeys[0]];
		pKey2 = &pNode2->m_KeyFrames[g_iCreateTransformKeys[1]];
	}

	
	quat_Slerp(quat, (float*)pKey1->m_Quaternion, 
		(float*)pKey2->m_Quaternion, g_CreateTransformPercent);

	quat_ConvertToMatrixTransposed(quat, mat.m);

	mat.m[0][3] = pKey1->m_Translation.x + (pKey2->m_Translation.x - pKey1->m_Translation.x) * g_CreateTransformPercent;
	mat.m[1][3] = pKey1->m_Translation.y + (pKey2->m_Translation.y - pKey1->m_Translation.y) * g_CreateTransformPercent;
	mat.m[2][3] = pKey1->m_Translation.z + (pKey2->m_Translation.z - pKey1->m_Translation.z) * g_CreateTransformPercent;

	// Compute the final transform.
	pStartTransform = pCurTransform;
	MatMul(pCurTransform, pTransform, &mat);
	++pCurTransform;

	// Do the children..
	for(i=0; i < pNode1->m_pNode->m_nChildren; i++)
	{
		pCurTransform = model_RecurseAndCreateTransforms(
			pNode1->m_Children[i], pNode2->m_Children[i],
			pStartTransform, pCurTransform);
	}


	// Change our transformation if we're a deformation node.
	if(pNode1->m_pNode->m_Flags & MNODE_DEFORMATION)
	{
		MAT_SET(mat, 
			pNode2->m_Scale.x, 0, 0, pNode2->m_Shift.x,
			0, pNode2->m_Scale.y, 0, pNode2->m_Shift.y,
			0, 0, pNode2->m_Scale.z, pNode2->m_Shift.z,
			0, 0, 0, 1);

		MAT_COPY(mTemp, *pStartTransform);
		MatMul(pStartTransform, &mTemp, &mat);
	}

	return pCurTransform;
}


static int model_CalcNumNodes(ModelNode *pNode, BOOL bDeformationOnly)
{
	int total;
	unsigned long i;

	total = 1;
	if(bDeformationOnly && !(pNode->m_Flags & MNODE_DEFORMATION))
		total = 0;

	for(i=0; i < pNode->m_nChildren; i++)
	{
		total += model_CalcNumNodes(&pNode->m_Children[i], bDeformationOnly);
	}

	return total;
}


static void model_FillNodeList(Model *pModel, ModelNode *pNode, int *curNodeIndex)
{
	unsigned long i;

	pNode->m_TransformIndex = (unsigned short)*curNodeIndex;
	pModel->m_FlatNodeList[*curNodeIndex] = pNode;
	++(*curNodeIndex);

	for(i=0; i < pNode->m_nChildren; i++)
		model_FillNodeList(pModel, &pNode->m_Children[i], curNodeIndex);
}


static void model_FillDNodeList(Model *pModel, ModelNode *pNode, int *curNodeIndex)
{
	unsigned long i;

	if(pNode->m_Flags & MNODE_DEFORMATION)
	{
		pModel->m_DNodes[*curNodeIndex] = pNode;
		++(*curNodeIndex);
	}

	for(i=0; i < pNode->m_nChildren; i++)
		model_FillDNodeList(pModel, &pNode->m_Children[i], curNodeIndex);
}


static void model_FillAnimDNodeList(ModelAnim *pAnim, AnimNode *pNode, int *curNodeIndex)
{
	unsigned long i;

	if(pNode->m_pNode->m_Flags & MNODE_DEFORMATION)
	{
		pAnim->m_DNodes[*curNodeIndex] = pNode;
		++(*curNodeIndex);
	}

	for(i=0; i < pNode->m_pNode->m_nChildren; i++)
		model_FillAnimDNodeList(pAnim, pNode->m_Children[i], curNodeIndex);
}


void model_BuildFlatNodeList(Model *pModel)
{
	int curNodeIndex;

	// Figure out how many there are.
	pModel->m_nNodes = model_CalcNumNodes(&pModel->m_RootNode, FALSE);
	pModel->m_FlatNodeList = (ModelNode**)dalloc(sizeof(ModelNode*) * pModel->m_nNodes);
	
	// Fill in the list.
	curNodeIndex = 0;
	model_FillNodeList(pModel, &pModel->m_RootNode, &curNodeIndex);

	// Precalculate m_nDWordNodes;
	pModel->m_nDWordNodes = pModel->m_nNodes >> 2;
	if(pModel->m_nNodes % 4 != 0)
		++pModel->m_nDWordNodes;
}


void model_BuildDNodeList(Model *pModel)
{
	int curNodeIndex;

	pModel->m_nDNodes = model_CalcNumNodes(&pModel->m_RootNode, TRUE);
	pModel->m_DNodes = (ModelNode**)dalloc(sizeof(ModelNode*) * pModel->m_nNodes);
	
	curNodeIndex = 0;
	model_FillDNodeList(pModel, &pModel->m_RootNode, &curNodeIndex);
}


void model_BuildAnimDNodeList(Model *pModel, ModelAnim *pAnim)
{
	int curNodeIndex;

	pAnim->m_DNodes = (AnimNode**)dalloc(sizeof(AnimNode*) * pModel->m_nDNodes);
	curNodeIndex = 0;
	model_FillAnimDNodeList(pAnim, pAnim->m_AnimNodes, &curNodeIndex);
}

void model_BuildTransformList(Model *pModel)
{
	// Setup all the vertices to point at the right transform.
	pModel->m_nTransforms = pModel->m_nNodes;
	pModel->m_Transforms = (DMatrix*)dalloc(sizeof(DMatrix) * pModel->m_nTransforms);
}


void model_CreateTransforms(Model *pModel, 
	ModelAnim *pAnim1, ModelAnim *pAnim2, unsigned long iKey1, unsigned long iKey2, float percent,
	DMatrix *pTransformList)
{
	DMatrix transform;

	Mat_Identity(&transform);

	g_iCreateTransformKeys[0] = iKey1;
	g_iCreateTransformKeys[1] = iKey2;
	g_CreateTransformPercent = percent;

	model_RecurseAndCreateTransforms(&pAnim1->m_AnimNodes[0], &pAnim2->m_AnimNodes[0],
		&transform, pTransformList);
}


void model_CreateTransforms2(Model *pModel, 
	ModelAnim *pAnim1, ModelAnim *pAnim2, unsigned long iKey1, unsigned long iKey2, float percent,
	DMatrix *pStartMat, DMatrix *pTransformList)
{
	g_iCreateTransformKeys[0] = iKey1;
	g_iCreateTransformKeys[1] = iKey2;
	g_CreateTransformPercent = percent;
	
	model_RecurseAndCreateTransforms(&pAnim1->m_AnimNodes[0], &pAnim2->m_AnimNodes[0],
		pStartMat, pTransformList);
}


void model_GetNodeTransform(Model *pModel, ModelAnim *pAnim1, ModelAnim *pAnim2, 
	unsigned long iKey1, unsigned long iKey2, float percent, DWORD dwNodeIndex,
	DMatrix *pStartTransform, DMatrix *pTransform)
{
	AnimNode *pNode1, *pNode2;
	DMatrix mat, mCurTransform, mTemp;
	
	static NodeKeyFrame *pKey1, *pKey2;
	static float quat[4];

	g_iCreateTransformKeys[0] = iKey1;
	g_iCreateTransformKeys[1] = iKey2;
	g_CreateTransformPercent = percent;

	pNode1 = &pAnim1->m_AnimNodes[ dwNodeIndex ];
	pNode2 = &pAnim2->m_AnimNodes[ dwNodeIndex ];

	// Change our transformation if we're a deformation node.
	if(pNode1->m_pNode->m_Flags & MNODE_DEFORMATION)
	{
		MAT_SET( mCurTransform, 
			pNode2->m_Scale.x, 0, 0, pNode2->m_Shift.x,
			0, pNode2->m_Scale.y, 0, pNode2->m_Shift.y,
			0, 0, pNode2->m_Scale.z, pNode2->m_Shift.z,
			0, 0, 0, 1);
	}
	else
	{
		Mat_Identity( &mCurTransform );
	}

	while( pNode1 && pNode2 )
	{
		if((pNode1->m_pNode->m_Flags & MNODE_DEFORMATION) && pNode1 != pNode2)
		{
			pKey1 = pKey2 = &pNode1->m_KeyFrames[g_iCreateTransformKeys[0]];
		}
		else
		{
			pKey1 = &pNode1->m_KeyFrames[g_iCreateTransformKeys[0]];
			pKey2 = &pNode2->m_KeyFrames[g_iCreateTransformKeys[1]];
		}
		
		quat_Slerp(quat, (float*)pKey1->m_Quaternion, 
			(float*)pKey2->m_Quaternion, g_CreateTransformPercent);

		quat_ConvertToMatrixTransposed(quat, mat.m);

		mat.m[0][3] = pKey1->m_Translation.x + (pKey2->m_Translation.x - pKey1->m_Translation.x) * g_CreateTransformPercent;
		mat.m[1][3] = pKey1->m_Translation.y + (pKey2->m_Translation.y - pKey1->m_Translation.y) * g_CreateTransformPercent;
		mat.m[2][3] = pKey1->m_Translation.z + (pKey2->m_Translation.z - pKey1->m_Translation.z) * g_CreateTransformPercent;

		// Compute the final transform.
		MatMul( &mTemp, &mat, &mCurTransform );
		MAT_COPY( mCurTransform, mTemp );

		pNode1 = pNode1->m_pParentNode;
		pNode2 = pNode2->m_pParentNode;
	}

	MatMul( pTransform, pStartTransform, &mCurTransform );
}


void model_InterpolateDVertices(Model *pModel, ModelAnim *pAnim1, ModelAnim *pAnim2,
	DWORD iKey1, DWORD iKey2, float percent)
{
	DefVertex *pCur1, *pCur2;
	DWORD i, count;
	WORD *pIndex;
	ModelVert *pOut;
	AnimNode *pAnimNode1, *pAnimNode2;
	SDWORD fixedPercent, out[3];


	if(pAnim1 == pAnim2)
	{
		for(i=0; i < pModel->m_nDNodes; i++)
		{
			pAnimNode1 = pAnim1->m_DNodes[i];
			pAnimNode2 = pAnim2->m_DNodes[i];
			
			pCur1 = &pAnimNode1->m_DefVertices[iKey1*pAnimNode1->m_pNode->m_nIndices];
			pCur2 = &pAnimNode2->m_DefVertices[iKey2*pAnimNode2->m_pNode->m_nIndices];
			pIndex = pAnimNode1->m_pNode->m_Indices;

			fixedPercent = (SDWORD)(percent * MDL_FIXED_ONE_F);
			
			count = pAnimNode1->m_pNode->m_nIndices;
			while(count--)
			{
				pOut = &pModel->m_Verts[*pIndex++];

				// Fixed point interpolation (should be MUCH faster).
				out[0] = ((SDWORD)pCur1->m_Pos[0] << MDL_FIXED_SHIFT) + (SDWORD)(pCur2->m_Pos[0] - pCur1->m_Pos[0]) * fixedPercent;
				out[1] = ((SDWORD)pCur1->m_Pos[1] << MDL_FIXED_SHIFT) + (SDWORD)(pCur2->m_Pos[1] - pCur1->m_Pos[1]) * fixedPercent;
				out[2] = ((SDWORD)pCur1->m_Pos[2] << MDL_FIXED_SHIFT) + (SDWORD)(pCur2->m_Pos[2] - pCur1->m_Pos[2]) * fixedPercent;

				// Convert to floating point with most precision.. requires FP multiplication.  This reduces snapping
				// a lot so it looks way better.
				pOut->m_Vec.x = (float)out[0] * MDL_FIXED_ONE_INV;
				pOut->m_Vec.y = (float)out[1] * MDL_FIXED_ONE_INV;
				pOut->m_Vec.z = (float)out[2] * MDL_FIXED_ONE_INV;

				// Convert without FP multiplication and snap to 1 pixel grid.
				//pOut->m_Vec.x = (float)(out[0] >> MDL_FIXED_SHIFT);
				//pOut->m_Vec.y = (float)(out[1] >> MDL_FIXED_SHIFT);
				//pOut->m_Vec.z = (float)(out[2] >> MDL_FIXED_SHIFT);

				// Just for reference.. this is how it would interpolate positions if model_RecurseAndCreateTransforms
				// wasn't rigging the matrix.
				// DVector pos1, pos2;
				// pos1.x = (float)pCur1->m_Pos[0];
				// pos1.y = (float)pCur1->m_Pos[1];
				// pos1.z = (float)pCur1->m_Pos[2];
				// pos2.x = (float)pCur2->m_Pos[0];
				// pos2.y = (float)pCur2->m_Pos[1];
				// pos2.z = (float)pCur2->m_Pos[2];
				// pOut->m_Vec.x = pos1.x + (pos2.x - pos1.x) * percent;
				// pOut->m_Vec.y = pos1.y + (pos2.y - pos1.y) * percent;
				// pOut->m_Vec.z = pos1.z + (pos2.z - pos1.z) * percent;

				// pOut->m_Vec.x = pCur1->m_Pos[0] + (pCur2->m_Pos[0] - pCur1->m_Pos[0]) * percent;
				// pOut->m_Vec.y = pCur1->m_Pos[1] + (pCur2->m_Pos[1] - pCur1->m_Pos[1]) * percent;
				// pOut->m_Vec.z = pCur1->m_Pos[2] + (pCur2->m_Pos[2] - pCur1->m_Pos[2]) * percent;

				++pCur1;
				++pCur2;
			}
		}
	}
	else
	{	
		for(i=0; i < pModel->m_nDNodes; i++)
		{
			pAnimNode2 = pAnim2->m_DNodes[i];
			
			pCur2 = &pAnimNode2->m_DefVertices[iKey2*pAnimNode2->m_pNode->m_nIndices];
			pIndex = pAnimNode2->m_pNode->m_Indices;
			
			count = pAnimNode2->m_pNode->m_nIndices;
			while(count--)
			{
				pOut = &pModel->m_Verts[*pIndex++];

				pOut->m_Vec.x = (float)pCur2->m_Pos[0];
				pOut->m_Vec.y = (float)pCur2->m_Pos[1];
				pOut->m_Vec.z = (float)pCur2->m_Pos[2];

				++pCur2;
			}
		}
	}
}


void model_UpdateBoundingBox(Model *pModel, ModelAnim *pAnim, DWORD iKey1, DWORD iKey2, float percent,
	DVector *pMin, DVector *pMax)
{
	unsigned long i;
	DVector transformedPt;
	ModelVert *pCurVert;
	DMatrix *pMat;

	// Setup the transformations and deformation vertices.
	model_CreateTransforms(pModel, pAnim, pAnim, iKey1, iKey2, percent, pModel->m_Transforms);
	model_InterpolateDVertices(pModel, pAnim, pAnim, iKey1, iKey2, percent);

	pCurVert = pModel->m_Verts;
	for(i=0; i < pModel->m_nVerts; i++)
	{
		pMat = &pModel->m_Transforms[pCurVert->m_TransformIndex];
		MatVMul(&transformedPt, pMat, &pCurVert->m_Vec);

		VEC_MIN(*pMin, *pMin, transformedPt);
		VEC_MAX(*pMax, *pMax, transformedPt);

		++pCurVert;
	}
}


void model_GetAnimBox(Model *pModel, ModelAnim *pAnim, DVector *pMin, DVector *pMax)
{
	unsigned long i;
	float percent, percentAdd;

	VEC_SET(*pMin, 100000.0f, 100000.0f, 100000.0f);
	VEC_NEGATE(*pMax, *pMin);

	if(pAnim->m_nKeyFrames == 1)
	{
		model_UpdateBoundingBox(pModel, pAnim, 0, 0, 0.0f, pMin, pMax);
	}
	else
	{
		percentAdd = 1.0f / 200.0f;
		for(i=0; i < (pAnim->m_nKeyFrames-1); i++)
		{
			for(percent=0.0f; percent < 1.0f; percent += percentAdd)
			{
				model_UpdateBoundingBox(pModel, pAnim, i, i+1, percent, pMin, pMax);
			}
		}
	}
}


char* model_AddToStringList(Model *pModel, char *pString)
{
	ModelString *pCur, *pRet;
	int len;

	pCur = pModel->m_StringList;
	while(pCur)
	{
		if(strcmp(pCur->m_String, pString) == 0)
			return pCur->m_String;
	
		pCur = pCur->m_pNext;
	}

	len = strlen(pString) + 1;
	pRet = (ModelString*)dalloc(sizeof(ModelString) - 1 + len);
	pRet->m_pNext = pModel->m_StringList;
	pModel->m_StringList = pRet;

	strcpy(pRet->m_String, pString);
	return pRet->m_String;
}


ModelAnim* model_FindAnim(Model *pModel, char *pName)
{
	for(unsigned long i=0; i < pModel->m_nAnims; i++)
		if(stricmp(pModel->m_Anims[i].m_pName, pName) == 0)
			return &pModel->m_Anims[i];

	return NULL;
}


void model_SetAnimTrans(Model *pModel, ModelAnim *pAnim, DVector *pNewTrans)
{
	DWORD i;

	// Translate it.
	for(i=0; i < pAnim->m_nKeyFrames; i++)
	{
		VEC_COPY(pAnim->m_AnimNodes[0].m_KeyFrames[i].m_Translation, *pNewTrans);
	}

	// Update bounding box info.
	model_GetAnimBox(pModel, pAnim, &pAnim->m_BoundMin, &pAnim->m_BoundMax);

	VEC_MIN(pModel->m_BoundMin, pModel->m_BoundMin, pAnim->m_BoundMin);
	VEC_MAX(pModel->m_BoundMax, pModel->m_BoundMax, pAnim->m_BoundMax);
}


void model_RecurseAndRemoveAnimKeyFrame(Model *pModel, AnimNode *pRoot, unsigned long iKeyFrame)
{
	unsigned long i, nIndices;

	if(pRoot->m_DefVertices)
	{
		nIndices = pRoot->m_pNode->m_nIndices;
		memmove(&pRoot->m_DefVertices[iKeyFrame*nIndices], &pRoot->m_DefVertices[(iKeyFrame+1)*nIndices],
			sizeof(DefVertex)*((pRoot->m_pAnim->m_nKeyFrames-1)-iKeyFrame)*nIndices);
	}

	memmove(&pRoot->m_KeyFrames[iKeyFrame], &pRoot->m_KeyFrames[iKeyFrame+1],
		sizeof(NodeKeyFrame)*((pRoot->m_pAnim->m_nKeyFrames-1)-iKeyFrame));

	for(i=0; i < pRoot->m_pNode->m_nChildren; i++)
		model_RecurseAndRemoveAnimKeyFrame(pModel, pRoot->m_Children[i], iKeyFrame);
}


void model_RemoveKeyFrame(Model *pModel, unsigned long iAnim, unsigned long iKeyFrame)
{
	ModelAnim *pAnim;
	AnimKeyFrame *pKeyFrame;

	if(iAnim >= pModel->m_nAnims)
		return;

	pAnim = &pModel->m_Anims[iAnim];
	if(iKeyFrame >= pAnim->m_nKeyFrames)
		return;

	// Remove the AnimKeyFrame.
	pKeyFrame = &pAnim->m_KeyFrames[iKeyFrame];

	memmove(&pAnim->m_KeyFrames[iKeyFrame], &pAnim->m_KeyFrames[iKeyFrame+1],
		sizeof(AnimKeyFrame)*((pAnim->m_nKeyFrames-1)-iKeyFrame));

	// Recurse thru the AnimNode tree and remove the keyframe from each one..
	model_RecurseAndRemoveAnimKeyFrame(pModel, pAnim->m_AnimNodes, iKeyFrame);

	--pAnim->m_nKeyFrames;
}


AnimKeyFrame* model_AllocKeyFrames(int nKeyFrames)
{
	int i;
	AnimKeyFrame *pRet;

	pRet = (AnimKeyFrame*)dalloc(sizeof(AnimKeyFrame) * nKeyFrames);
	memset(pRet, 0, sizeof(AnimKeyFrame) * nKeyFrames);

	for(i=0; i < nKeyFrames; i++)
		pRet[i].m_pString = "";

	return pRet;
}


ModelNode* model_AllocNodes(int nNodes)
{
	ModelNode *pNodes;
	int i;

	pNodes = (ModelNode*)dalloc(sizeof(ModelNode) * nNodes);
	memset(pNodes, 0, sizeof(ModelNode) * nNodes);

	for(i=0; i < nNodes; i++)
		pNodes[i].m_pName = "";

	return pNodes;
}


ModelAnim* model_AllocAnims(int nAnims)
{
	ModelAnim *pAnims;
	int i;

	pAnims = (ModelAnim*)dalloc(sizeof(ModelAnim) * nAnims);
	memset(pAnims, 0, sizeof(ModelAnim) * nAnims);

	for(i=0; i < nAnims; i++)
		pAnims[i].m_pName = "";

	return pAnims;
}

void model_DestroyAnim(ModelAnim *pAnim)
{
	if(pAnim->m_KeyFrames)
	{
		dfree(pAnim->m_KeyFrames);
		pAnim->m_KeyFrames = NULL;
	}

	if(pAnim->m_DNodes)
	{
		dfree(pAnim->m_DNodes);
		pAnim->m_DNodes = NULL;
	}

	model_DestroyAnimNode(pAnim->m_AnimNodes);

	if( pAnim->m_AnimNodes )
	{
		dfree( pAnim->m_AnimNodes );
		pAnim->m_AnimNodes = NULL;
	}
}


// {BP 1/6/97}
void model_ParseCommandString( Model *pModel )
{
	char tokenSpace[PARSE_MAXARGS*PARSE_MAXARGLEN];
	char *pTokens[PARSE_MAXARGS], *pCommandPos, *pCommand;
	int nTokens, bMore;

	if( !pModel || !pModel->m_CommandString )
		return;

	// Parse for level of detail settings...
	pCommandPos = pModel->m_CommandString;
	bMore = 1;
	while(bMore)
	{
		pCommand = pCommandPos;
		bMore = cp_Parse(pCommand, &pCommandPos, tokenSpace, pTokens, &nTokens);
	
		if(nTokens == 2)
		{
			if( stricmp( "LODSTARTDIST", pTokens[0] ) == 0 )
			{
				pModel->m_fLodStartDist = ( float )atof( pTokens[1] );
			}
			else if( stricmp( "LODINCREMENT", pTokens[0] ) == 0 )
			{
				pModel->m_fLodIncrement = ( float )atof( pTokens[1] );
			}
			else if( stricmp( "LODMAXDIST", pTokens[0] ) == 0 )
			{
				pModel->m_fLodMaxDist = ( float )atof( pTokens[1] );
			}
			else if( stricmp( "MIPMAPDISTADD", pTokens[0] ) == 0 )
			{
				pModel->m_MipmapDistAdd = (float)atof(pTokens[1]);
			}
			else if(stricmp("DrawIndexedDist", pTokens[0]) == 0)
			{
				pModel->m_DrawIndexedDist = (float)atof(pTokens[1]);
			}
		}
	}
}


short model_CopyUVCoords( Model *pDest, Model *pSrc )
{
	unsigned long nUV, i;

	if( pDest->m_nModelTris != pSrc->m_nModelTris )
		return 0;

	nUV = pSrc->m_nModelTris * 3;
	for( i=0; i < nUV; i++)
	{
		pDest->m_UVCoords[i] = pSrc->m_UVCoords[i];
	}
	return 1;
}


int model_CompareVerts(DVector *pVec1, DVector *pVec2)
{
	return (fabs(pVec1->x - pVec2->x) < 0.001f) && (fabs(pVec1->y - pVec2->y) < 0.001f) &&
		(fabs(pVec1->z - pVec2->z) < 0.001f);
}


#define CV(index, vert) model_CompareVerts(&pVerts[pCurTri->m_Indices[(index)%3]], vert)

ModelTri* model_FindMatchingTri(Model *pModel, DVector *pVerts, DVector *pVert1, DVector *pVert2, DVector *pVert3,
	int *pStartingVert)
{
	unsigned long i;
	int j;
	ModelTri *pCurTri;

	for(i=0; i < pModel->m_nModelTris; i++)
	{
		pCurTri = &pModel->m_ModelTris[i];

		for(j=0; j < 3; j++)
		{
			if(CV(j, pVert1) && CV(j+1, pVert2) && CV(j+2, pVert3))
			{
				*pStartingVert = j;
				return pCurTri;
			}
		}
	}

	return 0;
}


int model_CopyUVCoords2(Model *pDest, Model *pSrc, ModelAnim *pDestAnim, ModelAnim *pSrcAnim)
{
	unsigned long i, srcTriIndex;
	DVector *pDestVerts, *pSrcVerts;
	ModelTri *pCurTri, *pSrcTri;
	int nTrisImported, sv;

	// Setup transforms for the given animations.
	model_CreateTransforms(pDest, pDestAnim, pDestAnim, 0, 0, 0.0f, pDest->m_Transforms);
	model_CreateTransforms(pSrc, pSrcAnim, pSrcAnim, 0, 0, 0.0f, pSrc->m_Transforms);

	nTrisImported = 0;

	if(pDestVerts = (DVector*)dalloc(sizeof(DVector) * pDest->m_nNormalVerts))
	{
		// Transform the dest verts.
		for(i=0; i < pDest->m_nNormalVerts; i++)
		{
			MatVMul_H(&pDestVerts[i], &pDest->m_Transforms[pDest->m_Verts[i].m_TransformIndex], &pDest->m_Verts[i].m_Vec);
		}
		
		if(pSrcVerts = (DVector*)dalloc(sizeof(DVector) * pSrc->m_nNormalVerts))
		{
			// Transform the source verts.
			for(i=0; i < pSrc->m_nNormalVerts; i++)
			{
				MatVMul_H(&pSrcVerts[i], &pSrc->m_Transforms[pSrc->m_Verts[i].m_TransformIndex], &pSrc->m_Verts[i].m_Vec);
			}			

			for(i=0; i < pDest->m_nModelTris; i++)
			{
				pCurTri = &pDest->m_ModelTris[i];

				pSrcTri = model_FindMatchingTri(pSrc, pSrcVerts, 
					&pDestVerts[pCurTri->m_Indices[0]], &pDestVerts[pCurTri->m_Indices[1]], &pDestVerts[pCurTri->m_Indices[2]],
					&sv);

				if(pSrcTri)
				{
					srcTriIndex = pSrcTri - pSrc->m_ModelTris;

					pDest->m_UVCoords[i*3+0] = pSrc->m_UVCoords[srcTriIndex*3+((sv+0)%3)];
					pDest->m_UVCoords[i*3+1] = pSrc->m_UVCoords[srcTriIndex*3+((sv+1)%3)];
					pDest->m_UVCoords[i*3+2] = pSrc->m_UVCoords[srcTriIndex*3+((sv+2)%3)];

					++nTrisImported;
				}
			}
		
			dfree(pSrcVerts);
		}

		dfree(pDestVerts);
	}

	return nTrisImported;
}


void model_SetAnimFramerate(Model *pModel, unsigned long iAnim, float framerate)
{
	unsigned long i, curMarkerTime, deltaTime;
	ModelAnim *pAnim;
	AnimKeyFrame *pKeyFrame;

	deltaTime = (unsigned long)(framerate * 1000.0f);
	if(deltaTime == 0)
		return;

	pAnim = &pModel->m_Anims[iAnim];

	if(pAnim->m_nKeyFrames > 0)
	{
		curMarkerTime = pAnim->m_KeyFrames[0].m_Time;

		// Remove all keyframes until we encounter one with delta time > deltaTime.
		for(i=1; i < (pAnim->m_nKeyFrames-1); i++)
		{
			pKeyFrame = &pAnim->m_KeyFrames[i];

			if(pKeyFrame->m_Time < (curMarkerTime + deltaTime))
			{
				// Get rid of this keyframe..
				model_RemoveKeyFrame(pModel, iAnim, i);

				--i;
			}
			else
			{
				curMarkerTime = pKeyFrame->m_Time;
			}
		}
	}
}


void model_Flip(Model *pModel)
{
}


// Remove all LOD info from the model.
void model_ClearLODInfo(Model *pModel)
{
	ModelVert *pVerts;

	if(pModel->m_nVerts != pModel->m_nNormalVerts)
	{
		pVerts = (ModelVert*)dalloc(sizeof(ModelVert) * pModel->m_nNormalVerts);
		memcpy(pVerts, pModel->m_Verts, sizeof(ModelVert)*pModel->m_nNormalVerts);
		dfree(pModel->m_Verts);
		pModel->m_Verts = pVerts;
		pModel->m_nVerts = pModel->m_nNormalVerts;
	}

	pModel->m_NumberOfLODs = 0;
}


typedef struct
{
	ModelAnim	*m_pAnim;
	AnimNode	*m_Nodes[1]; // Nodes in preorder traversal.
} TempAnim;


// Returns 1 if the two keyframes are the same.
static int CompareKeyFrames(NodeKeyFrame *pFrame1, NodeKeyFrame *pFrame2)
{
	int i;

	for(i=0; i < 4; i++)
	{
		if(fabs(pFrame1->m_Quaternion[i] - pFrame2->m_Quaternion[i]) > 0.001f)	return 0;
	}

//	if(fabs(pFrame1->m_Translation.x - pFrame2->m_Translation.x) > 0.1f)	return 0;
//	if(fabs(pFrame1->m_Translation.y - pFrame2->m_Translation.y) > 0.1f)	return 0;
//	if(fabs(pFrame1->m_Translation.z - pFrame2->m_Translation.z) > 0.1f)	return 0;

	return 1;
}


// Returns 1 if all the animation for the two nodes is the same.
static int IsAnimationSame(Model *pModel, TempAnim **pAnims, 
	unsigned long iNode1, unsigned long iNode2)
{
	unsigned long i, j;
	TempAnim *pAnim;

	for(i=0; i < pModel->m_nAnims; i++)
	{
		pAnim = pAnims[i];

		for(j=0; j < pAnim->m_pAnim->m_nKeyFrames; j++)
		{
			if(!CompareKeyFrames(&pAnim->m_Nodes[iNode1]->m_KeyFrames[j], &pAnim->m_Nodes[iNode2]->m_KeyFrames[j]))
				return 0;
		}
	}

	return 1;
}


static void FillAnimNodeList(TempAnim *pAnim, AnimNode *pNode, int &curNodeIndex)
{
	unsigned long i;

	pAnim->m_Nodes[curNodeIndex] = pNode;
	++curNodeIndex;

	for(i=0; i < pNode->m_pNode->m_nChildren; i++)
		FillAnimNodeList(pAnim, pNode->m_Children[i], curNodeIndex);
}


static BOOL model_CopyAnimNodes(Model *pDestModel, ModelNode *pDestModelNode, ModelAnim *pDestModelAnim,
	AnimNode *pDestParentNode, AnimNode *pDestNode, AnimNode *pSrcNode, AnimNode **pNextAnimNode )
{
	DWORD i, size;
	
	memcpy(pDestNode, pSrcNode, sizeof(AnimNode));
	pDestNode->m_pNode = pDestModelNode;
	pDestNode->m_pAnim = pDestModelAnim;
	pDestNode->m_pParentNode = pDestParentNode;

	size = sizeof(DefVertex) * pDestModelAnim->m_nKeyFrames * pDestModelNode->m_nIndices;
	pDestNode->m_DefVertices = (DefVertex*)dalloc(size);
	memcpy(pDestNode->m_DefVertices, pSrcNode->m_DefVertices, size);

	size = sizeof(NodeKeyFrame) * pDestModelAnim->m_nKeyFrames;
	pDestNode->m_KeyFrames = (NodeKeyFrame*)dalloc(size);
	memcpy(pDestNode->m_KeyFrames, pSrcNode->m_KeyFrames, size);

	ASSERT(pDestModelNode->m_nChildren == pSrcNode->m_pNode->m_nChildren);

	if( pSrcNode->m_pNode->m_nChildren )
	{
		pDestNode->m_Children = (AnimNode**)dalloc_z(sizeof(AnimNode) * pSrcNode->m_pNode->m_nChildren);
		for(i=0; i < pSrcNode->m_pNode->m_nChildren; i++)
		{
			pDestNode->m_Children[i] = *pNextAnimNode;
			*pNextAnimNode += 1;
			if( !model_CopyAnimNodes(pDestModel, &pDestModelNode->m_Children[i], pDestModelAnim,
					pDestNode, pDestNode->m_Children[i], pSrcNode->m_Children[i], pNextAnimNode ))
			{
				return FALSE;
			}
		}
	}

	return TRUE;
}


BOOL model_CopyAnim(Model *pDestModel, ModelAnim *pDest, ModelAnim *pSrc)
{
	DWORD i;
	AnimNode *pNextAnimNode;

	model_DestroyAnim(pDest);

	pDest->m_pName = model_AddToStringList(pDestModel, pSrc->m_pName);
	pDest->m_msTime = pSrc->m_msTime;

	// Copy the keyframes over.
	pDest->m_KeyFrames = (AnimKeyFrame*)dalloc(sizeof(AnimKeyFrame) * pSrc->m_nKeyFrames);
	pDest->m_nKeyFrames = pSrc->m_nKeyFrames;

	for(i=0; i < pSrc->m_nKeyFrames; i++)
	{
		memcpy(&pDest->m_KeyFrames[i], &pSrc->m_KeyFrames[i], sizeof(AnimKeyFrame));
		pDest->m_KeyFrames[i].m_pString = model_AddToStringList(pDestModel, pSrc->m_KeyFrames[i].m_pString);
	}

	VEC_COPY(pDest->m_BoundMin, pSrc->m_BoundMin);
	VEC_COPY(pDest->m_BoundMax, pSrc->m_BoundMax);
	VEC_COPY(pDest->m_vDims, pSrc->m_vDims);

	pDest->m_AnimNodes = ( AnimNode * )dalloc_z( sizeof( AnimNode ) * pDestModel->m_nNodes );
	pNextAnimNode = &pDest->m_AnimNodes[1];

	if(model_CopyAnimNodes(pDestModel, &pDestModel->m_RootNode, pDest, NULL, pDest->m_AnimNodes, pSrc->m_AnimNodes, &pNextAnimNode))
	{
		model_BuildAnimDNodeList(pDestModel, pDest);
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}


void model_SetAnimNodeAnimPointers(ModelAnim *pAnim, AnimNode *pRoot)
{
	DWORD i;

	pRoot->m_pAnim = pAnim;
	for(i=0; i < pRoot->m_pNode->m_nChildren; i++)
	{
		model_SetAnimNodeAnimPointers(pAnim, pRoot->m_Children[i]);
	}
}


// This routine uses the LOD information to squash down the LODs.  By doing this
// instead of using the LOD at runtime, it can potentially save vertices, but it
// turns out that you don't save very many AT ALL (like 25 vertices from 150 LODs).
void model_SquashLODs(Model *pModel, DWORD iLOD)
{
	DWORD *vertIndexMap, i, counter, iCurVert;
	DWORD nUsedVerts, lastUsedIndex, *squashedIndexMap;
	ModelVert *pCurVert;
	BYTE *vertUseMap;
	ModelTri *pCurTri;


	if(iLOD > pModel->m_NumberOfLODs)
		iLOD = pModel->m_NumberOfLODs;

	if(iLOD == 0)
		return;


	vertIndexMap = (DWORD*)dalloc(sizeof(DWORD) * pModel->m_nVerts);

	for(i=0; i < pModel->m_nVerts; i++)
		vertIndexMap[i] = i;

	// Do the remapping.
	iCurVert = (pModel->m_nNormalVerts + iLOD) - 1;
	pCurVert = &pModel->m_Verts[iCurVert];
	counter = iLOD;
	while(counter--)
	{
		vertIndexMap[pCurVert->m_Replacements[0]] = iCurVert;
		vertIndexMap[pCurVert->m_Replacements[1]] = iCurVert;
		--iCurVert;
		--pCurVert;
	}


	// Remap the triangle indices.
	pCurTri = pModel->m_ModelTris;
	counter = pModel->m_nModelTris;
	while(counter--)
	{
		for(i=0; i < 3; i++)
		{
			ASSERT(pCurTri->m_Indices[i] < pModel->m_nNormalVerts);
			pCurTri->m_Indices[i] = (WORD)vertIndexMap[pCurTri->m_Indices[i]];
		}

		++pCurTri;
	}

	// Mark which vertices are used by the triangles.
	vertUseMap = (BYTE*)dalloc_z((pModel->m_nVerts >> 3) + 1);
	
	pCurTri = &pModel->m_ModelTris[iLOD << 1];
	counter = pModel->m_nModelTris - (iLOD << 1);
	while(counter--)
	{
		for(i=0; i < 3; i++)
		{
			vertUseMap[pCurTri->m_Indices[i]>>3] |= 1 << (pCurTri->m_Indices[i] & 7);
		}
	
		++pCurTri;
	}

	// All the vertices used by other LODs need to be preserved.
	for(i=(pModel->m_nNormalVerts+iLOD); i < pModel->m_nVerts; i++)
		vertUseMap[i>>3] |= (1 << (i & 7));

	// Pack the vertices in.
	squashedIndexMap = (DWORD*)dalloc(sizeof(DWORD) * pModel->m_nVerts);
	nUsedVerts = 0;
	lastUsedIndex = 0;
	for(i=0; i < pModel->m_nVerts; i++)
	{
		if(vertUseMap[i>>3] & (1 << (i&7)))
		{
			lastUsedIndex = nUsedVerts;
			pModel->m_Verts[nUsedVerts] = pModel->m_Verts[i];
			nUsedVerts++;
		}
		
		squashedIndexMap[i] = lastUsedIndex;
	}
	pModel->m_nVerts = nUsedVerts;

	// Pack the triangles in.
	pModel->m_nModelTris -= iLOD << 1;
	memmove(pModel->m_ModelTris, &pModel->m_ModelTris[iLOD<<1], sizeof(ModelTri)*pModel->m_nModelTris);
	
	if(pModel->m_UVCoords)
		memmove(pModel->m_UVCoords, &pModel->m_UVCoords[(iLOD<<1)*3], sizeof(UVPair)*pModel->m_nModelTris*3);

	if(pModel->m_SkinCoords)
		memmove(pModel->m_SkinCoords, &pModel->m_SkinCoords[(iLOD<<1)*3], sizeof(UVPair)*pModel->m_nModelTris*3);

	// Trim the vertex start number list and remap it.
	memmove(pModel->m_VertexStartNums, &pModel->m_VertexStartNums[iLOD], pModel->m_NumberOfLODs-iLOD);
	for(i=0; i < (pModel->m_NumberOfLODs-iLOD); i++)
	{
		pModel->m_VertexStartNums[i] = (WORD)squashedIndexMap[pModel->m_VertexStartNums[i]];
	}

	pModel->m_NumberOfLODs -= iLOD;
	pModel->m_nNormalVerts = pModel->m_nVerts - pModel->m_NumberOfLODs;

	// Remap the remappers..
	for(i=pModel->m_nNormalVerts; i < pModel->m_nVerts; i++)
	{
		pCurVert = &pModel->m_Verts[i];
		pCurVert->m_Replacements[0] = (WORD)squashedIndexMap[vertIndexMap[pCurVert->m_Replacements[0]]];
		pCurVert->m_Replacements[1] = (WORD)squashedIndexMap[vertIndexMap[pCurVert->m_Replacements[1]]];
	}

	// Remap the triangle indices.
	pCurTri = pModel->m_ModelTris;
	counter = pModel->m_nModelTris;
	while(counter--)
	{
		for(i=0; i < 3; i++)
		{
			ASSERT(squashedIndexMap[pCurTri->m_Indices[i]] <= pCurTri->m_Indices[i]);
			pCurTri->m_Indices[i] = (WORD)squashedIndexMap[pCurTri->m_Indices[i]];
			ASSERT(pCurTri->m_Indices[i] < pModel->m_nNormalVerts);
		}

		++pCurTri;
	}
	

	// Cleanup!
	dfree(vertUseMap);
	dfree(vertIndexMap);
}





