
#include "stdlith.h"
#include "model.h"
#include "model_save.h"



//------------------------------------------------------------------ //
// Saving functions.
//------------------------------------------------------------------ //

static void model_InternalUpdateBoundingBox(Model *pModel)
{
	DWORD i;
	ModelAnim *pAnim;

	VEC_SET(pModel->m_BoundMin, 500000.0f, 500000.0f, 500000.0f);
	VEC_NEGATE(pModel->m_BoundMax, pModel->m_BoundMin);
	
	for(i = 0; i < pModel->m_nAnims; i++)
	{
		pAnim = &pModel->m_Anims[i];

		if(pAnim->m_BoundMin.x < pModel->m_BoundMin.x) pModel->m_BoundMin.x = pAnim->m_BoundMin.x;
		if(pAnim->m_BoundMin.y < pModel->m_BoundMin.y) pModel->m_BoundMin.y = pAnim->m_BoundMin.y;
		if(pAnim->m_BoundMin.z < pModel->m_BoundMin.z) pModel->m_BoundMin.z = pAnim->m_BoundMin.z;

		if(pAnim->m_BoundMax.x > pModel->m_BoundMax.x) pModel->m_BoundMax.x = pAnim->m_BoundMax.x;
		if(pAnim->m_BoundMax.y > pModel->m_BoundMax.y) pModel->m_BoundMax.y = pAnim->m_BoundMax.y;
		if(pAnim->m_BoundMax.z > pModel->m_BoundMax.z) pModel->m_BoundMax.z = pAnim->m_BoundMax.z;
	}
}

static void model_SaveGeometry(Model *pModel, CAbstractIO &file)
{
	DWORD i, j;
	ModelTri *pCurTri;
	ModelVert *pCurVert;
	UVPair *pCurUV;

	file << pModel->m_NumberOfLODs;

	file.Write(pModel->m_VertexStartNums, sizeof(WORD)*(pModel->m_NumberOfLODs+1));

	// Save the tris.
	file << pModel->m_nModelTris;
	
	pCurTri = pModel->m_ModelTris;
	pCurUV = pModel->m_UVCoords;
	for(i=0; i < pModel->m_nModelTris; i++)
	{
		for(j=0; j < 3; j++)
		{
			file << pCurUV->tu;
			file << pCurUV->tv;
			++pCurUV;
		}

		file << pCurTri->m_Indices[0] << pCurTri->m_Indices[1] << pCurTri->m_Indices[2];
		file << pCurTri->m_Normal[0] << pCurTri->m_Normal[1] << pCurTri->m_Normal[2];
		++pCurTri;
	}

	// Save the vertices.
	file << pModel->m_nVerts << pModel->m_nNormalVerts;
	pCurVert = pModel->m_Verts;
	for(i=0; i < pModel->m_nVerts; i++)
	{
		file << pCurVert->m_Vec.x << pCurVert->m_Vec.y << pCurVert->m_Vec.z;
		file << pCurVert->m_Normal[0] << pCurVert->m_Normal[1] << pCurVert->m_Normal[2];
		file << pCurVert->m_TransformIndex;
		file << pCurVert->m_Replacements[0] << pCurVert->m_Replacements[1];
		++pCurVert;
	}
}

static void model_SaveNode(ModelNode *pNode, CAbstractIO &file)
{
	DWORD i;

	// Save all the misc. data.
	file << pNode->m_BoundMin.x << pNode->m_BoundMin.y << pNode->m_BoundMin.z;
	file << pNode->m_BoundMax.x << pNode->m_BoundMax.y << pNode->m_BoundMax.z;
	
	file.WriteString(pNode->m_pName);

	file << pNode->m_TransformIndex;
	file << pNode->m_Flags;

	// Save its indices.
	file << pNode->m_nIndices;
	file.Write(pNode->m_Indices, sizeof(unsigned short) * pNode->m_nIndices);

	// Save the child nodes.
	file << pNode->m_nChildren;
	for(i=0; i < pNode->m_nChildren; i++)
	{
		model_SaveNode(&pNode->m_Children[i], file);
	}
}

static void model_SaveAnimNode(AnimNode *pNode, CAbstractIO &file)
{
	DWORD		i;

	for(i=0; i < pNode->m_pAnim->m_nKeyFrames; i++)
		file.Write(&pNode->m_KeyFrames[i], sizeof(NodeKeyFrame));

	file.Write(pNode->m_DefVertices, sizeof(DefVertex)*pNode->m_pAnim->m_nKeyFrames*pNode->m_pNode->m_nIndices);
	file << pNode->m_Scale.x << pNode->m_Scale.y << pNode->m_Scale.z;
	file << pNode->m_Shift.x << pNode->m_Shift.y << pNode->m_Shift.z;

	for(i=0; i < pNode->m_pNode->m_nChildren; i++)
		model_SaveAnimNode(&pNode->m_Children[i], file);
}

static void model_SaveAnim(ModelAnim *pAnim, CAbstractIO &file)
{
	DWORD i;
	AnimKeyFrame *pKeyFrame;

	
	// Save misc info.
	file.WriteString(pAnim->m_pName);
	file << pAnim->m_msTime;
	file << pAnim->m_BoundMin.x << pAnim->m_BoundMin.y << pAnim->m_BoundMin.z;
	file << pAnim->m_BoundMax.x << pAnim->m_BoundMax.y << pAnim->m_BoundMax.z;
	
	// Write the keyframes out.
	file << pAnim->m_nKeyFrames;
	for(i=0; i < pAnim->m_nKeyFrames; i++)
	{
		pKeyFrame = &pAnim->m_KeyFrames[i];

		file << (DWORD)pKeyFrame->m_Time;
		file << pKeyFrame->m_BoundMin.x << pKeyFrame->m_BoundMin.y << pKeyFrame->m_BoundMin.z;
		file << pKeyFrame->m_BoundMax.x << pKeyFrame->m_BoundMax.y << pKeyFrame->m_BoundMax.z;

		file.WriteString(pKeyFrame->m_pString);
	}

	// Write the transform tree.
	model_SaveAnimNode(&pAnim->m_RootNode, file);
}

static DWORD model_StartSection(CAbstractIO &file, char *pSectionName, DWORD prevSectionPos)
{
	DWORD curPos, retVal;

	// Go back and link the previous section to this one.
	if(prevSectionPos != (DWORD)-1)
	{
		curPos = file.GetCurPos();
		file.SeekTo(prevSectionPos);
		file << curPos;
		file.SeekTo(curPos);
	}

	// Write the section header with no next section..
	file.WriteString(pSectionName);
	retVal = file.GetCurPos();
	file << (DWORD)-1;

	return retVal;
}


//------------------------------------------------------------------ //
// Interface functions.
//------------------------------------------------------------------ //

void model_Save(Model *pModel, CAbstractIO &file)
{
	DWORD i;
	DWORD prevSection = (DWORD)-1;


	prevSection = model_StartSection(file, "Header", prevSection);
	file.WriteString(MODELFILE_TOKEN);

	// Save the command string.
	ASSERT(pModel->m_CommandString);
	file.WriteString(pModel->m_CommandString);

	// Save the full bounding box.
	prevSection = model_StartSection(file, "Geometry", prevSection);
	model_InternalUpdateBoundingBox(pModel);
	file << pModel->m_BoundMin.x << pModel->m_BoundMin.y << pModel->m_BoundMin.z;
	file << pModel->m_BoundMax.x << pModel->m_BoundMax.y << pModel->m_BoundMax.z;

	// Save the geometry.
	model_SaveGeometry(pModel, file);

	// Save the node tree.
	prevSection = model_StartSection(file, "Nodes", prevSection);
	model_SaveNode(&pModel->m_RootNode, file);

	// Save all the animation.
	prevSection = model_StartSection(file, "Animation", prevSection);
	file << pModel->m_nAnims;
	for(i=0; i < pModel->m_nAnims; i++)
		model_SaveAnim(&pModel->m_Anims[i], file);

	// Save the dimensions.
	prevSection = model_StartSection(file, "AnimDims", prevSection);
	for(i=0; i < pModel->m_nAnims; i++)
	{
		file << pModel->m_Anims[i].m_vDims.x;
		file << pModel->m_Anims[i].m_vDims.y;
		file << pModel->m_Anims[i].m_vDims.z;
	}

	// Save the tristrips.
	if(pModel->m_TriStrips)
	{
		prevSection = model_StartSection(file, "TriStrips", prevSection);
		
		file << pModel->m_nTriStripIndices;
		file.Write(pModel->m_TriStripIndices, sizeof(unsigned short)*pModel->m_nTriStripIndices);

		file << pModel->m_nTriStrips;
		file.Write(pModel->m_TriStrips, sizeof(ModelTriStrip)*pModel->m_nTriStrips);
	
		file << pModel->m_nUnstrippedTris;
		for(i=0; i < pModel->m_nUnstrippedTris; i++)
			file << (WORD)(pModel->m_UnstrippedTris[i] - pModel->m_ModelTris);

		// Save the vertex texture coordinates.
		for(i=0; i < pModel->m_nNormalVerts; i++)
		{
			file << pModel->m_Verts[i].tu;
			file << pModel->m_Verts[i].tv;
		}
	}
}


