/*
Copyright (C) 2000 Jason Wilkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "shaders.h"
#include "shader_internal.h"



static sh_message_func_t _message_func   = sh_message_stderr_func;
static int                   _message_type   = SH_MSG_ALL|SH_MSG_DEBUG;
static const char*           _message_prefix = NULL;
static const char*           _message_file   = NULL;

int _sh_error_count;
int _sh_warning_count;
int _sh_compatibility_count;



/* sh_message_set_func:
 ***************************************************************************/
sh_message_func_t sh_message_set_func(sh_message_func_t f)
{
   sh_message_func_t o;
   
   o = _message_func;
   _message_func = f;

   return o;
}



/* sh_message_stderr_func:
 ***************************************************************************/
void sh_message_stderr_func(unsigned type, const char* msg)
{
   (void)type;

   fprintf(stdout, "%s\n", msg);
}



/* sh_message_set_types:
 ***************************************************************************/
unsigned sh_message_set_types(unsigned t)
{
   unsigned o;
   
   o = _message_type;
   _message_type = t;

   return o;
}



/* sh_message_get_counts:
 ***************************************************************************/
void sh_message_get_counts(int* ecount, int* wcount, int* ccount)
{
   if (ecount) *ecount = _sh_error_count;

   if (ecount) *wcount = _sh_warning_count;

   if (ecount) *ccount = _sh_compatibility_count;
}



/* sh_message_set_types:
 ***************************************************************************/
void sh_message_reset_counts(unsigned types)
{
   if (types & SH_MSG_ERROR) _sh_error_count = 0;

   if (types & SH_MSG_WARNING) _sh_warning_count = 0;

   if (types & SH_MSG_COMPATIBILITY) _sh_compatibility_count = 0;
}



/* sh_message_set_prefix:
 ***************************************************************************/
const char* sh_message_set_prefix(const char* p)
{
   const char* o;

   o = _message_prefix;
   _message_prefix = p;

   return o;
}



/* sh_message_set_file:
 ***************************************************************************/
const char* sh_message_set_file(const char* f)
{
   const char* o;

   o = _message_file;
   _message_file = f;

   return o;
}



/* sh_message:
 ***************************************************************************/
void sh_message(unsigned type, const char* format, ...)
{
   if (_message_func && (type & _message_type)) {
      int         len;
      va_list     argptr;
      static char string[1024];

      string[0] = '\0';

      if (_message_file) {
         strcat(string, _message_file);
         strcat(string, ":");
      }

      if (_sh_line_count > 0) {
         char buffer[64];

         sprintf(buffer, "%d:", _sh_line_count);
         strcat(string, buffer);
      }

      switch (type) {
         case SH_MSG_ERROR:
            strcat(string, "error:");
            _sh_error_count++;
            break;

         case SH_MSG_WARNING:
            strcat(string, "warning:");
            _sh_warning_count++;
            break;

         case SH_MSG_COMPATIBILITY:
            strcat(string, "compatibility:");
            _sh_compatibility_count++;
            break;

         case SH_MSG_VERBOSE:
         default:
            /*nothing*/
            break;
      }

      if (_message_prefix) {
         strcat(string, _message_prefix);
         strcat(string, ": ");
      }

      len = strlen(string);

      va_start(argptr, format);
      vsprintf(string+len, format, argptr);
      va_end(argptr);

      _message_func(type, string);
   }
}


