/*
Copyright (C) 2000 Jason Wilkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "shaders.h"
#include "shader_internal.h"



static void report_skyparms(shader_t* shader)
{
   if ((shader->flags & SH_FAR_BOX) ||
       (shader->flags & SH_SKYGEN)  ||
       (shader->flags & SH_NEAR_BOX)) {
      printf("\nskyparms:\n\n");

      printf("\tfar box basename: ");

      if (shader->flags & SH_FAR_BOX) {
         printf("%s\n", shader->far_box_basename);
      }
      else {
         printf("-\n");
      }

      if (shader->flags & SH_SKYGEN) {
         printf("\tcloudheight: %s\n", shader->cloud_height.string);
      }
      else {
         printf("no cloud layer\n");
      }

      printf("\tnear box basename: ");

      if (shader->flags & SH_NEAR_BOX) {
         printf("%s\n", shader->near_box_basename);
      }
      else {
         printf("-\n");
      }
   }
   else {
      printf("\nskyparms: <none>\n");
   }
}



static void report_cull(shader_t* shader)
{
   printf("\ncull: %s\n", sh_cull_to_string(shader->cull_face));
}



static void report_vxmod(sh_vxmod_t* vxmod)
{
   printf("%s\n", sh_vxmod_to_string(vxmod->func));

   switch(vxmod->func) {
      case SH_VXMOD_WAVE:
         printf("\n");
         printf("\t\tdiv:       %s\n", vxmod->args.wave.div.string);
         printf("\t\twaveform:  %s\n", sh_waveform_to_string(vxmod->args.wave.form));
         printf("\t\tbase:      %s\n", vxmod->args.wave.base.string);
         printf("\t\tamplitude: %s\n", vxmod->args.wave.amp.string);
         printf("\t\tphase:     %s\n", vxmod->args.wave.phase.string);
         printf("\t\tfrequency: %s\n", vxmod->args.wave.freq.string);
         break;

      case SH_VXMOD_NORMAL:
         printf("\n");
         printf("\t\tamplitude: %s\n", vxmod->args.normal.amp.string);
         printf("\t\tfrequency: %s\n", vxmod->args.normal.freq.string);
         break;

      case SH_VXMOD_BULGE:
         printf("\n");
         printf("\t\twidth:     %s\n", vxmod->args.bulge.width.string);
         printf("\t\theight:    %s\n", vxmod->args.bulge.height.string);
         printf("\t\tspeed:     %s\n", vxmod->args.bulge.speed.string);
         break;

      case SH_VXMOD_MOVE:
         printf("\n");
         printf("\t\tvector:    <%s, %s, %s>\n", vxmod->args.move.x.string, vxmod->args.move.y.string, vxmod->args.move.z.string);
         printf("\t\twaveform:  %s\n", sh_waveform_to_string(vxmod->args.move.form));
         printf("\t\tbase:      %s\n", vxmod->args.move.base.string);
         printf("\t\tamplitude: %s\n", vxmod->args.move.amp.string);
         printf("\t\tphase:     %s\n", vxmod->args.move.phase.string);
         printf("\t\tfrequency: %s\n", vxmod->args.move.freq.string);
         break;

      case SH_VXMOD_AUTOSPRITE:
      case SH_VXMOD_AUTOSPRITE2:
      case SH_VXMOD_PROJECTION_SHADOW:
      case SH_VXMOD_TEXT0:
      case SH_VXMOD_TEXT1:

#ifdef SH_PHOENIX_QUAKE1_EXT
      case SH_VXMOD_STENCIL_SHADOW_EXT:
#endif

         break;

      default:
         break;
   }
}



static void report_vxmods(shader_t* shader)
{
   int i;

   if (shader->vxmod_count > 0) {
      printf("\ndeformVertexes:\n\n");
      printf("\tcount: %d\n", shader->vxmod_count);

      for (i = 0; i < shader->vxmod_count; i++) {
         printf("\n");
         printf("\tdeformVertexes #%d: ", i+1);
         report_vxmod(&shader->vxmods[i]);
      }
   }
   else {
      printf("\ndeformVertexes: <none>\n");
   }
}



static void report_fogparms(shader_t* shader)
{
   if (shader->flags & SH_FOGPARMS) {
      printf("\nfogparms:\n");
      printf("\tcolor:              <%s, %s, %s>\n", shader->fog_color[0].string, shader->fog_color[1].string, shader->fog_color[2].string);
      printf("\tdistance to opaque: %s\n", shader->fog_distance_to_opaque.string);
   }
   else {
      printf("\nfogparms: <none>\n");
   }
}



static void report_nopicmip(shader_t* shader)
{
   printf("\nnopicmip: ");

   if (shader->flags & SH_NOPICMIP) {
      printf("true");
   }
   else {
      printf("false");
   }

   printf("\n");
}



static void report_nomipmaps(shader_t* shader)
{
   printf("\nnomipmaps: ");

   if (shader->flags & SH_NOMIPMAPS) {
      printf("true");
   }
   else {
      printf("false");
   }

   printf("\n");
}



static void report_polygon_offset(shader_t* shader)
{
   printf("polygon offset: ");

   if (shader->flags & SH_POLYGON_OFFSET) {
      printf("true");
   }
   else {
      printf("false");
   }

   printf("\n");
}



static void report_sort(shader_t* shader)
{
   printf("\nsort priority: %s\n", sh_sort_to_string(shader->sort_priority, false));
}



static void report_entity_mergable(shader_t* shader)
{
   printf("\nentityMergable: ");

   if (shader->flags & SH_ENTITY_MERGABLE) {
      printf("true");
   }
   else {
      printf("false");
   }

   printf("\n");
}


static void report_map(sh_stage_t* stage)
{
   if (stage->anim_map_count == 1) {
      if (stage->maps[0].flags & SH_MAP_CLAMPMAP) {
         printf("\t\tclampmap: ");
      }
      else {
         printf("\t\tmap: ");
      }

      printf("%s\n", stage->maps[9].name);
   }
   else if (stage->anim_map_count > 1) {
      int i;

      printf("\t\tanimMap:\n\n");

      printf("\t\t\tfps:   %s\n", stage->anim_map_fps.string);
      printf("\t\t\tcount: %d\n\n", stage->anim_map_count);

      for (i = 0; i < stage->anim_map_count; i++) {
         printf("\t\t\tframe %d: %s\n", i+1, stage->maps[i].name);
      }
   }
   else {
      printf("\n\t\tmap: <none>\n");
   }
}



static void report_blendfunc(sh_stage_t* stage)
{
   if (stage->flags & SH_STAGE_BLENDFUNC) {
      printf("\n\t\tblendfunc: ");
      
      if (sh_is_blend_filter(stage)) {
         printf("filter\n");
      }
      else if (sh_is_blend_blend(stage)) {
         printf("blend\n");
      }
      else if (sh_is_blend_add(stage)) {
         printf("add\n");
      }
      else {
         printf("\n\n");
         printf("\t\t\tsource factor:      %s\n", sh_blend_to_string(stage->blend_src));
         printf("\t\t\tdestination factor: %s\n", sh_blend_to_string(stage->blend_dst));
      }
   }
   else {
      printf("\n\t\tblendFunc: <none>\n");
   }
}



static void report_rgbagen(sh_rgbagen_t* rgbagen, bool is_rgbgen)
{
   printf("%s\n", sh_rgbagen_to_string(rgbagen->func));

   switch(rgbagen->func) {
      case SH_RGBAGEN_WAVE:
         printf("\n");
         printf("\t\t\twaveform:  %s\n", sh_waveform_to_string(rgbagen->args.wave.form));
         printf("\t\t\tbase:      %s\n", rgbagen->args.wave.base.string);
         printf("\t\t\tamplitude: %s\n", rgbagen->args.wave.amp.string);
         printf("\t\t\tphase:     %s\n", rgbagen->args.wave.phase.string);
         printf("\t\t\tfrequency: %s\n", rgbagen->args.wave.freq.string);
         break;

#ifdef SH_PHOENIX_QUAKE1_EXT
      case SH_RGBAGEN_CONST_EXT:
         printf("\n\t\t\t");

         if (is_rgbgen) {
             printf("color: <%s, %s, %s>\n", rgbagen->args.constant.color[0], rgbagen->args.constant.color[1], rgbagen->args.constant.color[2]);
         }
         else {
             printf("alpha: %s\n", rgbagen->args.constant.color[0]);
         }

         break;

      case SH_RGBAGEN_ONE_MINUS_PORTAL_EXT:
#else
      (void)is_rgbgen;
#endif

      case SH_RGBAGEN_IDENTITY_LIGHTING:
      case SH_RGBAGEN_IDENTITY:
      case SH_RGBAGEN_ENTITY:
      case SH_RGBAGEN_ONE_MINUS_ENTITY:
      case SH_RGBAGEN_VERTEX:
      case SH_RGBAGEN_ONE_MINUS_VERTEX:
      case SH_RGBAGEN_PORTAL:
      case SH_RGBAGEN_LIGHTING_DIFFUSE:
      case SH_RGBAGEN_LIGHTING_SPECULAR:
      default:
         break;
   }
}



static void report_rgbgen(sh_stage_t* stage)
{
   printf("\n\t\trgbGen: ");
   report_rgbagen(&stage->rgbgen, true);
}



static void report_alphagen(sh_stage_t* stage)
{
   printf("\n\t\talphaGen: ");
   report_rgbagen(&stage->alphagen, false);
}



static void report_tcgen(sh_stage_t* stage)
{
   printf("\n\t\ttcGen: %s\n", sh_tcgen_to_string(stage->tcgen.func));

   switch(stage->tcgen.func) {
      case SH_TCGEN_VECTOR:
         printf("\n");
         printf("\t\t\ts vector:  <%s, %s, %s>\n", stage->tcgen.args.vector.sx.string, stage->tcgen.args.vector.sy.string, stage->tcgen.args.vector.sz.string);
         printf("\t\t\tt vector:  <%s, %s, %s>\n", stage->tcgen.args.vector.tx.string, stage->tcgen.args.vector.ty.string, stage->tcgen.args.vector.tz.string);
         break;

#ifdef SH_PHOENIX_QUAKE1_EXT
      case SH_TCGEN_SKY_EXT:
         printf("\n");
         printf("\t\t\tcloudheight:  %s\n", stage->tcgen.args.sky.cloud_height.string);
         break;

      case SH_TCGEN_Q1TURB_EXT:
         printf("\n");
         printf("\t\t\tdiv:       %s\n", stage->tcgen.args.q1turb.div.string);
         printf("\t\t\twaveform:  %s\n", sh_waveform_to_string(stage->tcgen.args.q1turb.form));
         printf("\t\t\tbase:      %s\n", stage->tcgen.args.q1turb.base.string);
         printf("\t\t\tamplitude: %s\n", stage->tcgen.args.q1turb.amp.string);
         printf("\t\t\tphase:     %s\n", stage->tcgen.args.q1turb.phase.string);
         printf("\t\t\tfrequency: %s\n", stage->tcgen.args.q1turb.freq.string);
         break;
#endif

      case SH_TCGEN_BASE:
      case SH_TCGEN_LIGHTMAP:
      case SH_TCGEN_ENVIRONMENT:
      default:
         break;
   }
}



static void report_tcmod(sh_tcmod_t* tcmod)
{
   printf("%s\n", sh_tcmod_to_string(tcmod->func));

   switch (tcmod->func) {
      case SH_TCMOD_ROTATE:
         printf("\n");
         printf("\t\t\t\tspeed: %s\n", tcmod->args.rotate.speed.string);
         break;

      case SH_TCMOD_SCALE:
         printf("\n");
         printf("\t\t\t\ts: %s\n", tcmod->args.scale.s.string);
         printf("\t\t\t\tt: %s\n", tcmod->args.scale.t.string);
         break;

      case SH_TCMOD_SCROLL:
         printf("\n");
         printf("\t\t\t\ts: %s\n", tcmod->args.scroll.s.string);
         printf("\t\t\t\tt: %s\n", tcmod->args.scroll.t.string);
         break;

      case SH_TCMOD_STRETCH:
         printf("\n");
         printf("\t\t\t\twaveform:  %s\n", sh_waveform_to_string(tcmod->args.stretch.form));
         printf("\t\t\t\tbase:      %s\n", tcmod->args.stretch.base.string);
         printf("\t\t\t\tamplitude: %s\n", tcmod->args.stretch.amp.string);
         printf("\t\t\t\tphase:     %s\n", tcmod->args.stretch.phase.string);
         printf("\t\t\t\tfrequency: %s\n", tcmod->args.stretch.freq.string);
         break;

      case SH_TCMOD_TRANSFORM:
         printf("\n");
         printf("\t\t\t\tm00: %s\n", tcmod->args.transform.m00.string);
         printf("\t\t\t\tm01: %s\n", tcmod->args.transform.m01.string);
         printf("\t\t\t\tm10: %s\n", tcmod->args.transform.m10.string);
         printf("\t\t\t\tm11: %s\n", tcmod->args.transform.m11.string);
         printf("\t\t\t\tt0:  %s\n", tcmod->args.transform.t0.string);
         printf("\t\t\t\tt1:  %s\n", tcmod->args.transform.t1.string);
         break;

      case SH_TCMOD_TURB:
         printf("\n");
         printf("\t\t\t\tbase:      %s\n", tcmod->args.turb.base.string);
         printf("\t\t\t\tamplitude: %s\n", tcmod->args.turb.amp.string);
         printf("\t\t\t\tphase:     %s\n", tcmod->args.turb.phase.string);
         printf("\t\t\t\tfrequency: %s\n", tcmod->args.turb.freq.string);
         break;
   }
}



static void report_tcmods(sh_stage_t* stage)
{
   if (stage->tcmod_count > 0) {
      int i;

      printf("\n\t\ttcMods:\n\n");
      printf("\t\t\tcount: %d\n", stage->tcmod_count);

      for (i = 0; i < stage->tcmod_count; i++) {
         printf("\n");
         printf("\t\t\ttcMod #%d: ", i+1);
         report_tcmod(&stage->tcmods[i]);
      }
   }
   else {
      printf("\n\t\ttcMods: <none>\n");
   }
}



static void report_depthfunc(sh_stage_t* stage)
{
   printf("\n\t\tdepthfunc: %s\n", sh_depthfunc_to_string(stage->depthfunc));
}



static void report_depthwrite(sh_stage_t* stage)
{
   printf("\n\t\tdepthWrite: ");

   if (stage->flags & SH_STAGE_DEPTHWRITE) {
      printf("true");
   }
   else {
      printf("false");
   }

   printf("\n");
}



static void report_detail(sh_stage_t* stage)
{
   printf("\n\t\tdetail: ");

   if (stage->flags & SH_STAGE_DETAIL) {
      printf("true");
   }
   else {
      printf("false");
   }

   printf("\n");
}



static void report_alphafunc(sh_stage_t* stage)
{
   printf("\n\t\talphafunc: ");

   if (stage->flags & SH_STAGE_ALPHAFUNC) {
      printf("%s\n", sh_alphafunc_to_string(stage->alphafunc));
   }
   else {
      printf("<none>\n");
   }
}



static void report_stage(sh_stage_t* stage)
{
   report_map(stage);
   report_blendfunc(stage);
   report_rgbgen(stage);
   report_alphagen(stage);
   report_tcgen(stage);
   report_tcmods(stage);
   report_depthfunc(stage);
   report_depthwrite(stage);
   report_detail(stage);
   report_alphafunc(stage);
}



static void report_stages(shader_t* shader)
{
   if (shader->stage_count > 0) {
      int i;

      printf("\nstages:\n\n");
      printf("\tcount: %d\n", shader->stage_count);

      for (i = 0; i < shader->stage_count; i++) {
         printf("\n");
         printf("\tstage #%d: \n\n", i+1);
         report_stage(&shader->stages[i]);
      }
   }
   else {
      printf("\nno stages\n");
   }
}



void shader_report(shader_t* shader)
{
   printf("------------------------\n");
   printf("name: %s\n", shader->name);
   report_skyparms(shader);
   report_cull(shader);
   report_vxmods(shader);
   report_fogparms(shader);
   report_nopicmip(shader);
   report_nomipmaps(shader);
   report_polygon_offset(shader);
   report_sort(shader);
   report_entity_mergable(shader);
   report_stages(shader);
   printf("------------------------\n");
}


