/*
Copyright (C) 2000 Jason Wilkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "shaders.h"
#include "sh_internal.h"



static sh_message_func_t message_func        = sh_message_file_func;
static int               message_type        = SH_MSG_DEFAULT;
static const char*       message_context     = NULL;
static const char*       message_subcontext  = NULL;
static const char*       message_type_string = NULL;
static const char*       message_message     = NULL;
static const char*       message_format      = SH_DEFAULT_MESSAGE_FORMAT;

int _sh_count_error;
int _sh_count_warning;
int _sh_count_compatibility;



/* sh_message_set_func:
 ***************************************************************************/
sh_message_func_t sh_message_set_func(sh_message_func_t f)
{
   sh_message_func_t o;
   
   o = message_func;
   message_func = f;

   return o;
}



/* sh_message_set_types:
 ***************************************************************************/
unsigned sh_message_set_types(unsigned t)
{
   unsigned o;
   
   o = message_type;
   message_type = t;

   return o;
}



/* sh_message_get_counts:
 ***************************************************************************/
void sh_message_get_counts(int* ecount, int* wcount, int* ccount)
{
   if (ecount) *ecount = _sh_count_error;

   if (ecount) *wcount = _sh_count_warning;

   if (ecount) *ccount = _sh_count_compatibility;
}



/* sh_message_set_types:
 ***************************************************************************/
void sh_message_reset_counts(unsigned types)
{
   if (types & SH_MSG_ERROR) _sh_count_error = 0;

   if (types & SH_MSG_WARNING) _sh_count_warning = 0;

   if (types & SH_MSG_COMPATIBILITY) _sh_count_compatibility = 0;
}



/* sh_message_set_subcontext:
 ***************************************************************************/
const char* sh_message_set_subcontext(const char* p)
{
   const char* o;

   o = message_subcontext;
   message_subcontext = p;

   return o;
}



/* sh_message_set_context:
 ***************************************************************************/
const char* sh_message_set_context(const char* f)
{
   const char* o;

   o = message_context;
   message_context = f;

   return o;
}




/* sh_message_set_format:
 ***************************************************************************/
const char* sh_message_set_format(const char* f)
{
   const char* o;

   o = message_format;
   message_format = f;

   return o;
}



/* format_message:
 ***************************************************************************/
static void format_message(const char* in, char* out)
{
   size_t len;

   while (*in) {
      if (*in == '%') {
         in++;

         len = 0;

         switch (*in) {
            case 'F':
            case 'f':
               if (message_context) {
                  strcpy(out, message_context);
                  len = strlen(message_context);
               }

               break;

            case 'L':
            case 'l':
               if (_sh_line == 0) {
                  char buffer[33];

                  sprintf(buffer, "%d", _sh_line);
                  strcpy(out, buffer);
                  len = strlen(buffer);
               }

               break;

            case 'T':
            case 't':
               if (message_type_string) {
                  strcpy(out, message_type_string);
                  len = strlen(message_type_string);
               }

               break;

            case 'C':
            case 'c':
               if (message_subcontext) {
                  strcpy(out, message_subcontext);
                  len = strlen(message_subcontext);
               }

               break;

            case 'M':
            case 'm':
               if (message_message) {
                  strcpy(out, message_message);
                  len = strlen(message_message);
               }

               break;

            case '%':
               *out = '%';
               len = 1;
               break;
 
            case '\0':
               *out = '\0';
               return;

            default:
               *out = *in;
               len = 1;
               break;
         }

         out += len;

         if ((len != 0) && isupper(*in)) {
            *out++ = ':';
         }

         in++;
      }
      else {
         *out++ = *in++;
      }
   }

   *out = '\0';
}



/* sh_message:
 ***************************************************************************/
void sh_message(unsigned type, const char* format, ...)
{
#if 1
   if (message_func && (type & message_type)) {
      size_t  len;
      char*   out;
      va_list argptr;

      switch (type) {
         case SH_MSG_ERROR:
            message_type_string = "error";
            break;

         case SH_MSG_WARNING:
            message_type_string = "warning";
            break;

         case SH_MSG_COMPATIBILITY:
            message_type_string = "compatibility";
            break;

         case SH_MSG_VERBOSE:
         default:
            message_type_string = NULL;
            break;
      }

      va_start(argptr, format);
      vsprintf(_sh_scratch, format, argptr);
      va_end(argptr);

      message_message = _sh_scratch;
      len = strlen(_sh_scratch);
      out = _sh_scratch + len + 1;
      format_message(message_format, out);
      message_func(type, out);
   }
#else
   if (message_func && (type & message_type)) {
      int     len;
      va_list argptr;

      _sh_scratch[0] = '\0';

      if (message_context) {
         strcat(_sh_scratch, message_context);
         strcat(_sh_scratch, ":");
      }

      if (_sh_line > 0) {
         char buffer[64];

         sprintf(buffer, "%d:", _sh_line);
         strcat(_sh_scratch, buffer);
      }

      switch (type) {
         case SH_MSG_ERROR:
            strcat(_sh_scratch, "error:");
            _sh_count_error++;
            break;

         case SH_MSG_WARNING:
            strcat(_sh_scratch, "warning:");
            _sh_count_warning++;
            break;

         case SH_MSG_COMPATIBILITY:
            strcat(_sh_scratch, "compatibility:");
            _sh_count_compatibility++;
            break;

         case SH_MSG_VERBOSE:
         default:
            /*nothing*/
            break;
      }

      if (message_subcontext) {
         strcat(_sh_scratch, message_subcontext);
         strcat(_sh_scratch, ": ");
      }

      len = strlen(_sh_scratch);

      va_start(argptr, format);
      vsprintf(_sh_scratch+len, format, argptr);
      va_end(argptr);

      message_func(type, _sh_scratch);
   }
#endif
}




/***************************************************************************\

string messages

\***************************************************************************/

static char* curpos;
static char* endpos;



/* sh_message_string_func:
 ***************************************************************************/
void sh_message_string_func(const char* string)
{
   size_t len  = strlen(string);
   size_t size = endpos - curpos;

   strncpy(curpos, string, size);

   if (len < size) {
      curpos += len;
   }
   else {
      curpos += size;
   }
}



/* sh_message_set_string:
 ***************************************************************************/
void sh_message_set_string(char* string, size_t size)
{
   curpos = string;
   endpos = string + size;
}



/***************************************************************************\

standard file messages

\***************************************************************************/

static FILE* message_file = stdout;



/* sh_message_file_func:
 ***************************************************************************/
void sh_message_file_func(unsigned type, const char* msg)
{
   int len = strlen(msg);
   fwrite(msg, len, 1, message_file);
   (void)type;
}



/* sh_message_set_file:
 ***************************************************************************/
FILE* sh_message_set_file(FILE* f)
{
   FILE* o;
   
   o = message_file;
   message_file = f;

   return o;
}


