/*
    Copyright (C) 1999 by Seth Galbraith

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "sysdef.h"
#include "squawk.h"

#include "csengine/world.h"
#include "csengine/sector.h"
#include "csengine/camera.h"
#include "csengine/csview.h"
#include "csengine/light.h"
#include "csengine/polygon.h"
#include "csengine/cssprite.h"

#include "csobject/nameobj.h"
#include "csparser/csloader.h"
#include "cssys/common/system.h"
#include "cstools/simpcons.h"
#include "csutil/inifile.h"
#include "csutil/vfs.h"

#include "igraph2d.h"
#include "igraph3d.h"
#include "itxtmgr.h"

// increment by one with every public release
#define SQK_VERSION 15

// size of the sky box
// @@@ the sky box should be practically infinite in size ...
// @@@ orthographically projected with maximum z values
#define SQK_SKYBOX_SIZE 50

sqkSystem* Sys = NULL;

sqkSystem::sqkSystem ()
{
  frameTime = 0;
  world = NULL;
  view[0] = NULL;
  view[1] = NULL;
  view[2] = NULL;
  view[3] = NULL;
  sprite[0] = NULL;
}

sqkSystem::~sqkSystem ()
{
  delete world;
  delete view[0];
  delete view[1];
  delete view[2];
  delete view[3];
  delete sprite[0];
}

void cleanup ()
{
  pprintf ("Cleaning up...\n");
  delete Sys;
  pprintf_close();
}

void debug_dump ()
{
}

void sqkSystem::AddViewVertex (int v, float x, float y)
{
  view[v]->AddViewVertex ((int)(x * FrameWidth), (int)(y * FrameHeight));
}

void sqkSystem::InitApp ()
{
  if (!Open ("Squawk Crystal Space Application"))
  {
    Printf (MSG_FATAL_ERROR, "Error opening system!\n");
    cleanup ();
    exit (1);
  }

  ITextureManager* txtmgr;
  piG3D->GetTextureManager (&txtmgr);
  txtmgr->Initialize ();
  txtmgr->Prepare ();
  txtmgr->AllocPalette ();

  csSimpleConsole* con = new csSimpleConsole (Sys->Config);
  con->SetupColors (txtmgr);
  con->SetTransparent (0);
  con->SetMaxLines (1000);       // Some arbitrarily high value.
  Sys->Console = con;

  Sys->DemoReady = true;

  Printf (MSG_INITIALIZATION, "Squawk Crystal Space Application version %d\n",
    SQK_VERSION);

  world->EnableLightingCache (false);
  world->Initialize (GetISystemFromSystem (this), piG3D, NULL, Sys->Vfs);

  // Change to virtual Squawk's directory
  Sys->Vfs->ChDir (Sys->Config->GetStr ("Squawk", "DATA", "/data/squawk"));

  // load sky box textures

  csTextureHandle* tm_up    = csLoader::LoadTexture (world, "up",    "sqk_u.jpg");
  csTextureHandle* tm_down  = csLoader::LoadTexture (world, "down",  "sqk_d.jpg");
  csTextureHandle* tm_left  = csLoader::LoadTexture (world, "left",  "sqk_l.jpg");
  csTextureHandle* tm_right = csLoader::LoadTexture (world, "right", "sqk_r.jpg");
  csTextureHandle* tm_front = csLoader::LoadTexture (world, "front", "sqk_f.jpg");
  csTextureHandle* tm_back  = csLoader::LoadTexture (world, "back",  "sqk_b.jpg");


  // create the sky box
  // @@@ create a Squawk and/or Crystal Space skybox class

  csSector* skybox = world->NewSector ();
  csNameObject::AddName (*skybox, "skybox");

  csPolygon3D* p;

  // (up) ceiling
  p = skybox->NewPolygon (tm_up);
  p->AddVertex (-SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->SetTextureSpace (p->Vobj (0), p->Vobj (1), SQK_SKYBOX_SIZE * 2);
  p->SetFlags (CS_POLY_MIPMAP | CS_POLY_LIGHTING, 0);

  // (down) floor
  p = skybox->NewPolygon (tm_down);
  p->AddVertex (-SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->SetTextureSpace (p->Vobj (0), p->Vobj (1), SQK_SKYBOX_SIZE * 2);
  p->SetFlags (CS_POLY_MIPMAP | CS_POLY_LIGHTING, 0);

  // (left) negative x wall
  p = skybox->NewPolygon (tm_left);
  p->AddVertex (-SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->SetTextureSpace (p->Vobj (0), p->Vobj (1), SQK_SKYBOX_SIZE * 2);
  p->SetFlags (CS_POLY_MIPMAP | CS_POLY_LIGHTING, 0);

  // (right) positive x wall 
  p = skybox->NewPolygon (tm_right);
  p->AddVertex ( SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->SetTextureSpace (p->Vobj (0), p->Vobj (1), SQK_SKYBOX_SIZE * 2);
  p->SetFlags (CS_POLY_MIPMAP | CS_POLY_LIGHTING, 0);

  // (front) positive z wall
  p = skybox->NewPolygon (tm_front);
  p->AddVertex (-SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE);
  p->SetTextureSpace (p->Vobj (0), p->Vobj (1), SQK_SKYBOX_SIZE * 2);
  p->SetFlags (CS_POLY_MIPMAP | CS_POLY_LIGHTING, 0);

  // (back) negative z wall
  p = skybox->NewPolygon (tm_back);
  p->AddVertex ( SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE,  SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex (-SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->AddVertex ( SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE, -SQK_SKYBOX_SIZE);
  p->SetTextureSpace (p->Vobj (0), p->Vobj (1), SQK_SKYBOX_SIZE * 2);
  p->SetFlags (CS_POLY_MIPMAP | CS_POLY_LIGHTING, 0);

  // load sprite

  csLoader::LoadTexture (world, "player1", "player1.png");
  csLoader::LoadLibraryFile (world, "library");

  sprite[0] = world -> GetSpriteTemplate ("player1") -> NewSprite ();
  sprite[0] -> MoveToSector (skybox);
  sprite[0] -> MoveTo (0, -0.4, 0);

  // add light

  csStatLight* light;
  light = new csStatLight ( 0, 0,  0, SQK_SKYBOX_SIZE * 2, 1, 1, 1, false);
  skybox->AddLight (light);

  // prepare world

  world->Prepare (piG3D);

  txtmgr->AllocPalette ();
  con->SetupColors (txtmgr);

  if (SUCCEEDED (piG2D->BeginDraw()))
  {
    piG2D->ClearAll (0);
    piG2D->FinishDraw ();
  }

  // create views

  // upper left
  view[0] = new csView (world, piG3D);
  view[0]->SetSector (skybox);
  view[0]->GetCamera ()->SetPosition (csVector3 (0, 0, 0));
  AddViewVertex (0, 0.006, 0.992);  AddViewVertex (0, 0.594, 0.992);
  AddViewVertex (0, 0.594, 0.708);  AddViewVertex (0, 0.444, 0.508);
  AddViewVertex (0, 0.006, 0.508);

  // upper right
  view[1] = new csView (world, piG3D);
  view[1]->SetSector (skybox);
  view[1]->GetCamera ()->SetPosition (csVector3 (0, 0, 0));
  AddViewVertex (1, 0.594, 0.992);  AddViewVertex (1, 0.994, 0.992);
  AddViewVertex (1, 0.994, 0.708);  AddViewVertex (1, 0.594, 0.708);

  // lower left
  view[2] = new csView (world, piG3D);
  view[2]->SetSector (skybox);
  view[2]->GetCamera ()->SetPosition (csVector3 (0, 0, 0));
  AddViewVertex (2, 0.006, 0.492);  AddViewVertex (2, 0.444, 0.492);
  AddViewVertex (2, 0.444, 0.008);  AddViewVertex (2, 0.006, 0.008);

  // lower right
  view[3] = new csView (world, piG3D);
  view[3]->SetSector (skybox);
  view[3]->GetCamera ()->SetPosition (csVector3 (0, 0, 0));
  AddViewVertex (3, 0.602, 0.692);  AddViewVertex (3, 0.994, 0.692);
  AddViewVertex (3, 0.995, 0.008);  AddViewVertex (3, 0.456, 0.008);
  AddViewVertex (3, 0.456, 0.495);

  view[0]->SetPerspectiveCenter (FrameWidth * 0.50, FrameHeight * 0.80);
  view[1]->SetPerspectiveCenter (FrameWidth * 0.50, FrameHeight * 0.80);
  view[2]->SetPerspectiveCenter (FrameWidth * 0.22, FrameHeight * 0.25);
  view[3]->SetPerspectiveCenter (FrameWidth * 0.75, FrameHeight * 0.35);

/*
  view[0]->GetCamera ()->SetFOV (FrameHeight);
  view[1]->GetCamera ()->SetFOV (FrameHeight);
@@@ when I set the FOV to FrameHeight / 2 and look at the sprite ...
@@@ I get a Floating Point Exception. (djgpp, nasm, optimize)
  view[2]->GetCamera ()->SetFOV (FrameHeight / 2);
  view[3]->GetCamera ()->SetFOV (FrameHeight / 2);
*/
}

void sqkSystem::RotateView (int i, const csVector3& v, float speed)
{
  if (view[i]->GetCamera()->GetW2C().Col2().y < 0) // upside down (-:
  {
    if (v == VEC_ROT_LEFT)
      {view[i]->GetCamera()->RotateWorld(VEC_ROT_RIGHT,speed); return;}
    if (v == VEC_ROT_RIGHT)
      {view[i]->GetCamera()->RotateWorld(VEC_ROT_LEFT,speed); return;}
  }
  if (v == VEC_ROT_LEFT || v == VEC_ROT_RIGHT)
    view[i]->GetCamera()->RotateWorld(v,speed);
  else
    view[i]->GetCamera()->Rotate(v,speed);
}

void sqkSystem::eatkeypress (int key, bool shift, bool alt, bool ctrl)
{
  (void)shift; (void)alt; (void)ctrl;

  switch (key)
  {
    case CSKEY_ESC : Shutdown = true;          break;
    case '1' : sprite[0]->SetAction("run");    break;
    case '2' : sprite[0]->SetAction("stand");  break;
    case '3' : sprite[0]->SetAction("pain");   break;
    case '4' : sprite[0]->SetAction("death");  break;
    case '5' : sprite[0]->SetAction("attack"); break;
    case '6' : sprite[0]->SetAction("jump");   break;

    case CSKEY_RIGHT :
      RotateView (0,VEC_ROT_RIGHT,0.03);
      RotateView (1,VEC_ROT_RIGHT,0.03);
      break;
    case CSKEY_LEFT  :
      RotateView (0,VEC_ROT_LEFT,0.03);
      RotateView (1,VEC_ROT_LEFT,0.03);
      break;
    case CSKEY_UP    :
      RotateView (0,VEC_TILT_UP,0.03);
      RotateView (1,VEC_TILT_UP,0.03);
      break;
    case CSKEY_DOWN  :
      RotateView (0,VEC_TILT_DOWN,0.03);
      RotateView (1,VEC_TILT_DOWN,0.03);
      break;

    case 'd' : RotateView (2,VEC_ROT_RIGHT,0.03); break;
    case 'a' : RotateView (2,VEC_ROT_LEFT,0.03);  break;
    case 'w' : RotateView (2,VEC_TILT_UP,0.03);   break;
    case 's' : RotateView (2,VEC_TILT_DOWN,0.03); break;

    case 'l' : RotateView (3,VEC_ROT_RIGHT,0.03); break;
    case 'j' : RotateView (3,VEC_ROT_LEFT,0.03);  break;
    case 'i' : RotateView (3,VEC_TILT_UP,0.03);   break;
    case 'k' : RotateView (3,VEC_TILT_DOWN,0.03); break;
  }
}

void sqkSystem::NextFrame (long elapsed_time, long current_time)
{
  SysSystemDriver::NextFrame (elapsed_time, current_time);

  // Handle all events.
  csEvent *Event;
  while ((Event = EventQueue->Get ()))
  {
    switch (Event->Type)
    {
      case csevKeyDown:
        eatkeypress (Event->Key.Code,
          Event->Key.ShiftKeys & CSMASK_SHIFT,
          Event->Key.ShiftKeys & CSMASK_ALT,
          Event->Key.ShiftKeys & CSMASK_CTRL);
        break;
      case csevMouseMove: break;
      case csevMouseDown: break;
      case csevMouseUp:   break;
    }
    delete Event;
  }

  // animate sprite
  if (frameTime == 0)  frameTime = current_time + 100;
  if (current_time > frameTime)
  {
    int frame = sprite[0]->GetCurFrame ();
    if (frame >= sprite[0]->GetNumFrames () - 1) sprite[0]->SetFrame (0);
    //if (frame >= 5) sprite[0]->SetFrame (0);
    else sprite[0]->SetFrame (frame + 1);
    frameTime = current_time + 100;
  }

  // Draw the views
  if (piG3D->BeginDraw (CSDRAW_3DGRAPHICS) != S_OK) return;
  view[0]->Draw ();
  view[1]->Draw ();
  view[2]->Draw ();
  view[3]->Draw ();
  piG3D->FinishDraw ();
  piG3D->Print (NULL);
}

  /////////////
 // M A I N //
/////////////

int main (int argc, char* argv[])
{
  Sys = new sqkSystem ();
  Sys->world = new csWorld ();
  if (!Sys->Initialize (argc, argv, "squawk.cfg", "VFS.cfg", Sys->world->GetEngineConfigCOM ()))
  {
    Sys->Printf (MSG_FATAL_ERROR, "Error initializing system!\n");
    cleanup ();
    exit (1);
  }

  Sys->InitApp ();
  Sys->Loop ();
  cleanup ();
  return 0;
}
