// QCameraView.cpp : implementation file
//

#include "stdafx.h"
#include "QuakeEdit.h"
#include "QCameraView.h"
#include "math.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

CameraView *cameraview_i = NULL;

typedef struct
{
	vec3_t	trans;
	int		clipflags;
	vec3_t	screen;			// only valid if clipflags == 0
} campt_t;

#define	CLIP_RIGHT	1
#define	CLIP_LEFT	2
#define	CLIP_TOP	4
#define	CLIP_BOTTOM	8
#define	CLIP_FRONT	16

int		cam_cur;
campt_t	campts[2];

vec3_t	r_matrix[3];
vec3_t	r_origin;
float	mid_x, mid_y;
float	topscale = (240.0/3)/160;
float	bottomscale = (240.0*2/3)/160;

extern	plane_t	frustum[5];

/////////////////////////////////////////////////////////////////////////////
// CameraView

void XYmoveto(vec3_t);
void XYlineto(vec3_t);

IMPLEMENT_DYNCREATE(CameraView,CQuakeView)

CameraView::CameraView()
{
  cameraview_i = this;
	
	xa = ya = za = 0;
	
	matrixFromAngles();
	
	origin[0] = 64;
	origin[1] = 64;
	origin[2] = 48;
	
	move = 16;

  m_pGL = new CGLMap;

  drawmode = dr_wire;
}

CameraView::~CameraView()
{
  if (m_pGL)
    delete m_pGL;
}


BEGIN_MESSAGE_MAP(CameraView, CQuakeView)
	//{{AFX_MSG_MAP(CameraView)
	ON_WM_LBUTTONDOWN()
 	ON_WM_CREATE()
	ON_WM_SIZE()
	ON_WM_ERASEBKGND()
	ON_WM_DESTROY()
	ON_WM_PALETTECHANGED()
	ON_WM_QUERYNEWPALETTE()
	ON_WM_NCCREATE()
	ON_WM_RBUTTONDOWN()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CameraView drawing


void CameraView::OnDraw(CDC* pDC)
{
  SetMatrices();

  switch(drawmode)
  {
    case dr_texture:
    case dr_flat:
       VectorCopy (origin, r_origin); 
       r_drawflat = (drawmode != dr_texture);
       m_pGL->SetRenderMode(CGLMap::render);
       m_pGL->Render();
       break;
    default:
    {
        m_pGL->SetRenderMode(CGLMap::wire);
        m_pGL->Render();
    }
  }
}

/////////////////////////////////////////////////////////////////////////////
// CameraView diagnostics

#ifdef _DEBUG
void CameraView::AssertValid() const
{
	CQuakeView::AssertValid();
}

void CameraView::Dump(CDumpContext& dc) const
{
	CQuakeView::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CameraView message handlers

void CameraView::setXYOrigin(CPoint *pt)
{
	origin[0] = pt->x;
	origin[1] = pt->y;
}

void CameraView::setZOrigin(float pt)
{
	origin[2] = pt;
}

void CameraView::setOrigin(vec3_t org, float angle)
{
	VectorCopy (org, origin);
	ya = angle;
	matrixFromAngles();
}

void CameraView::getOrigin(vec3_t org)
{
	VectorCopy (origin, org);
}

float CameraView::yawAngle()
{
	return ya;
}

void CameraView::setDrawMode(drawmode_t mode)
{
	drawmode = mode;
	quakeed_i->updateCamera();
}

void CameraView::matrixFromAngles()
{
	if (xa > M_PI*0.4)
		xa = M_PI*0.4;
	if (xa < -M_PI*0.4)
		xa = -M_PI*0.4;
		
// vpn
	matrix[2][0] = cos(xa)*cos(ya);
	matrix[2][1] = cos(xa)*sin(ya);
	matrix[2][2] = sin(xa);

// vup	
	matrix[1][0] = cos(xa+M_PI/2)*cos(ya);
	matrix[1][1] = cos(xa+M_PI/2)*sin(ya);
	matrix[1][2] = sin(xa+M_PI/2);

// vright
	CrossProduct (matrix[2], matrix[1], matrix[0]);

}


void CameraView::inverseTransform(vec_t *invec,vec_t *outvec)
{
	vec3_t		inverse[3];
	vec3_t		temp;
	int			i,j;
	
	for (i=0 ; i<3 ; i++)
		for (j=0 ; j<3 ; j++)
			inverse[i][j] = matrix[j][i];
	
	temp[0] = DotProduct(invec, inverse[0]);
	temp[1] = DotProduct(invec, inverse[1]);
	temp[2] = DotProduct(invec, inverse[2]);		

	VectorAdd (temp, origin, outvec);
}




void CameraMoveto(vec3_t p)
{
  glBegin(GL_LINE_STRIP);
  glVertex3f(p[0],p[1],p[2]);
}


void CameraLineto(vec3_t p)
{
  glVertex3f(p[0],p[1],p[2]);
}

void CameraEndLine()
{
  glEnd();
}

void CameraSetColor(COLORREF r)
{
  glColor3f(GetRValue(r) / 255.0,
            GetGValue(r) / 255.0,
            GetBValue(r) / 255.0);

}

void CameraView::drawWire()
{
/*
  map_i->makeUnselectedPerform(BRUSH_CAMERADRAWSELF);
  map_i->makeSelectedPerform(BRUSH_CAMERADRAWSELF);
  clipper_i->cameraDrawSelf();
*/
}

void CameraView::XYDrawSelf()
{
	SetColor(COLOR_CAMERA);
	
  POINT points[6];
  
  points[0].x = origin[0] - 16;
  points[0].y = origin[1];
  points[1].x = points[0].x + 16;
  points[1].y = points[0].y + 8;
  points[2].x = points[1].x + 16;
  points[2].y = points[1].y - 8;
  points[3].x = points[2].x - 16;
  points[3].y = points[2].y - 8;
  points[4].x = points[3].x - 16;
  points[4].y = points[3].y + 8;
  points[5].x = points[4].x + 32;
  points[5].y = points[4].y;
	Polyline(points,6);

  points[0].x = origin[0];
  points[0].y = origin[1];
  points[1].x = points[0].x + 64*cos(ya+M_PI/4);
  points[1].y = points[0].y + 64*sin(ya+M_PI/4);
  Polyline(points,2);

  points[0].x = origin[0];
  points[0].y = origin[1];
  points[1].x = points[0].x + 64*cos(ya-M_PI/4);
  points[1].y = points[0].y + 64*sin(ya-M_PI/4);
  Polyline(points,2);
}

void CameraView::ZDrawSelf()
{
	SetColor(COLOR_CAMERA);

  POINT points[6];
  points[0].x = - 16;
  points[0].y = origin[2];
  points[1].x = points[0].x + 16;
  points[1].y = points[0].y + 8;
  points[2].x = points[1].x + 16;
  points[2].y = points[1].y - 8;
  points[3].x = points[2].x - 16;
  points[3].y = points[2].y - 8;
  points[4].x = points[3].x - 16;
  points[4].y = points[3].y + 8;
  points[5].x = points[4].x + 32;
  points[5].y = points[4].y;

  Polyline(points,6);

  points[0].x = - 15;
  points[0].y = origin[2] - 47;
  points[1].x = points[0].x + 29;
  points[1].y = points[0].y;
  points[2].x = points[1].x;
  points[2].y = points[1].y + 54;
  points[3].x = points[2].x - 29;
  points[3].y = points[2].y;
  points[4].x = points[3].x;
  points[4].y = points[3].y - 54;

  Polyline(points,5);
}

/*
================
modalMoveLoop
================
*/
void CameraView::modalMoveLoop(CPoint *basept,vec3_t movemod, CQuakeView *converter)
{
	vec3_t		originbase;
	MSG       msg;
	CPoint		newpt;
	vec3_t		delta;
	int			i;
	
	qprintf ("moving camera position");

	VectorCopy (origin, originbase);	
		
//
// modal event loop using instance drawing
//
	
  
  converter->SetCapture();
  quakeed_i->updateCameraInstance();
  
  goto drawentry;

	while (msg.message != WM_LBUTTONUP && msg.message != WM_RBUTTONUP)
	{
		//
		// calculate new point
		//
    if (msg.message != WM_MOUSEMOVE)
      continue;
		
    newpt = CPoint(LOWORD(msg.lParam),HIWORD(msg.lParam));

		converter->WindowToMap(newpt);
		
    quakeed_i->updateCameraInstance();
    
		delta[0] = newpt.x-basept->x;
		delta[1] = newpt.y-basept->y;
		delta[2] = delta[1];		// height change
		
		for (i=0 ; i<3 ; i++)
			origin[i] = originbase[i]+movemod[i]*delta[i];
					
drawentry:
		//
		// instance draw new frame
		//
		quakeed_i->updateCameraInstance();

		GetMessage(&msg,converter->m_hWnd,WM_MOUSEFIRST,WM_MOUSELAST);
	}
  ReleaseCapture();
}

//============================================================================

/*
===============
XYmouseDown
===============
*/
BOOL CameraView::XYmouseDown(CPoint *pt, UINT nFlags)	// return YES if brush handled
{	
	vec3_t		movemod;
	
	if (fabs(pt->x - origin[0]) > 16
	|| fabs(pt->y - origin[1]) > 16 )
		return NO;
	
#if 0	
	if (flags & NX_ALTERNATEMASK)
	{	// up / down drag
		movemod[0] = 0;
		movemod[1] = 0;
		movemod[2] = 1;
	}
	else
#endif
	{	
		movemod[0] = 1;
		movemod[1] = 1;
		movemod[2] = 0;
	}
	
	modalMoveLoop(pt,movemod,xyview_i);
	
  quakeed_i->updateXY();

	return YES;
}


/*
===============
ZmouseDown
===============
*/
BOOL CameraView::ZmouseDown(CPoint *pt,UINT nFlags)	// return YES if brush handled
{	
	vec3_t		movemod;
	
	if (fabs(pt->y - origin[2]) > 16
	|| pt->x < -8 || pt->x > 8 )
		return NO;
		
	movemod[0] = 0;
	movemod[1] = 0;
	movemod[2] = 1;
	
	modalMoveLoop(pt,movemod,zview_i);

  quakeed_i->updateZ();

	return YES;
}


//=============================================================================

/*
===================
viewDrag:
===================
*/
void CameraView::viewDrag(CPoint *pt)
{
	
  float	dx,dy;
	MSG       msg;
	CPoint		newpt;
	
//
// modal event loop using instance drawing
//
	quakeed_i->updateCameraInstance();
  
  goto drawentry;

	while (msg.message != WM_RBUTTONUP)
	{
		//
		// calculate new point
		//
		newpt = CPoint(LOWORD(msg.lParam),HIWORD(msg.lParam));
//		WindowToMap(newpt);
    
    quakeed_i->updateCameraInstance();
		
    dx = newpt.x - pt->x;
		dy = newpt.y - pt->y;
		*pt = newpt;
	
		ya -= dx/m_ClientRect.Size().cx*M_PI/2 * 4;
		xa += dy/m_ClientRect.Size().cx*M_PI/2 * 4;
	
		matrixFromAngles();
		
drawentry:
    quakeed_i->updateCameraInstance();		
    GetMessage(&msg,m_hWnd,WM_MOUSEFIRST,WM_MOUSELAST);
	}
}

void CameraView::SetMatrices()
{
  m_pGL->Resize(m_ClientRect.Width(),m_ClientRect.Height());
  m_pGL->MakeCurrent();
  
  glMatrixMode(GL_MODELVIEW); 	
	
	// Translate and rotate the axis.
	glLoadIdentity();

  vec3_t origin, target;
  float distance = 200;
  float angle = yawAngle();
  double scale = GetScale();
  
  getOrigin(origin);
  
  target[0] = origin[0] + distance * cos(angle);
  target[1] = origin[1] + distance * sin(angle);
  target[2] = origin[2] + distance * sin(xa);

  gluLookAt(origin[0], origin[1],origin[2],
            target[0],target[1],target[2],0,0,1);

  glScaled(scale,scale,scale);

}

void CameraView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	vec3_t			p1, p2;
	
  SetMatrices();

  VectorCopy (origin, p1);
  
  point.y = m_ClientRect.Size().cy - point.y;
  WindowToMap(point,p2);

	nFlags = nFlags & (MK_SHIFT | MK_CONTROL);
  BOOL bAltDown = (GetKeyState(VK_MENU) & 0x8000) != 0;

//
// bare click to select a texture
//
	if (nFlags == 0 && !bAltDown)
	{
		map_i->getTextureRay(p1,p2);
		return;
	}
	
//
// shift click to select / deselect a brush from the world
//
	if (nFlags == MK_SHIFT && !bAltDown)
	{		
		map_i->selectRay(p1,p2,NO);
		return;
	}
	
//
// cmd-shift click to set a target/targetname entity connection
//
	if (nFlags == MK_SHIFT|MK_CONTROL && !bAltDown)
	{
		map_i->entityConnect(p1,p2);
		return;
	}

//
// alt click = set entire brush texture
//
	if (nFlags == 0 && bAltDown)
	{
		if (drawmode != dr_texture)
		{
			qprintf ("No texture setting except in texture mode!");
			MessageBeep(MB_ICONEXCLAMATION);
			return;
		}		
		map_i->setTextureRay(p1,p2,YES);
		quakeed_i->updateAll();
		return;
	}
		
//
// ctrl-alt click = set single face texture
//
	if (nFlags == MK_CONTROL && bAltDown)
	{
		if (drawmode != dr_texture)
		{
			qprintf ("No texture setting except in texture mode!");
			MessageBeep(MB_ICONEXCLAMATION);
			return;
		}
		map_i->setTextureRay(p1,p2,NO);
		quakeed_i->updateAll();
		return;
	}
	qprintf ("bad flags for click");
  MessageBeep(MB_ICONEXCLAMATION);	
}


BOOL CameraView::FlipY()
{
  return TRUE;
}

void CameraView::upFloor()
{
	sb_floor_dir = 1;
	sb_floor_dist = 99999;
	map_i->makeAllPerform(BRUSH_FEET_TO_FLOOR);
	if (sb_floor_dist == 99999)
	{
		qprintf ("already on top floor");
		return;
	}
	qprintf ("up floor");
	origin[2] += sb_floor_dist;
	quakeed_i->updateCamera();
}

void CameraView::downFloor()
{
	sb_floor_dir = -1;
	sb_floor_dist = -99999;
	map_i->makeAllPerform(BRUSH_FEET_TO_FLOOR);
	if (sb_floor_dist == -99999)
	{
		qprintf ("already on bottom floor");
		return;
	}
	qprintf ("down floor");
	origin[2] += sb_floor_dist;
	quakeed_i->updateCamera();
}

void CameraView::lockFocus(BOOL bXor)
{
  m_pGL->SetRenderMode(CGLMap::selection);
  m_pGL->BeginRender();
}

void CameraView::unlockFocus()
{
  m_pGL->EndRender();
}

void CameraView::keyDown(UINT nChar,UINT nRep,UINT nFlags)
{
	switch (nChar)
	{
	case 'a':
	case 'A':
		xa += M_PI/16.0;
		matrixFromAngles();
		quakeed_i->updateCamera();
		return;
		
	case 'z':
	case 'Z':
		xa -= M_PI/16.0;
		matrixFromAngles();
		quakeed_i->updateCamera();
		return;
		
	case VK_RIGHT:
		ya -= M_PI*move/(64*2);
		matrixFromAngles();
		quakeed_i->updateCamera();
		break;
		
	case VK_LEFT:
		ya += M_PI*move/(64*2);
		matrixFromAngles();
		quakeed_i->updateCamera();
		break;
		
	case VK_UP:
		origin[0] += move*cos(ya);
		origin[1] += move*sin(ya);
		quakeed_i->updateCamera();
		break;
		
	case VK_DOWN:
		origin[0] -= move*cos(ya);
		origin[1] -= move*sin(ya);
		quakeed_i->updateCamera();
		break;
		
	case VK_PRIOR:
  case 190:
  
		origin[0] += move*cos(ya-M_PI_2);
		origin[1] += move*sin(ya-M_PI_2);
		quakeed_i->updateCamera();
		break;
		
	case VK_NEXT:
  case 188:
		origin[0] -= move*cos(ya-M_PI_2);
		origin[1] -= move*sin(ya-M_PI_2);
		quakeed_i->updateCamera();
		break;
		
	case 'd':
	case 'D':
		origin[2] += move;
		quakeed_i->updateCamera();
		break;
		
	case 'c':
	case 'C':
		origin[2] -= move;
		quakeed_i->updateCamera();
		break;
	}
}

BOOL CameraView::OnEraseBkgnd(CDC* pDC) 
{
//	if (drawmode == dr_texture)
    return TRUE ;

//  return CQuakeView::OnEraseBkgnd(pDC);
}

//
// PreCreateWindow: Clip siblings and children.
//
BOOL CameraView::PreCreateWindow(CREATESTRUCT& cs) 
{
	TRACE0("PreCreateWindow\r\n") ;
  	cs.style |= WS_CLIPSIBLINGS | WS_CLIPCHILDREN; 
	BOOL bRet = CQuakeView::PreCreateWindow(cs);
    cs.style &= ~(WS_VSCROLL | WS_HSCROLL);
  return bRet;
}

//
// OnNcCreate: Setup the m_pGL pointer so we can use virtual function calls.
//
BOOL CameraView::OnNcCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (!CQuakeView::OnNcCreate(lpCreateStruct))
		return FALSE;
	TRACE0("OnNcCreate\r\n") ;
	
	ASSERT(m_pGL) ;
	
	return TRUE;
}

//
// OnCreate: Create a CGL object.
//
int CameraView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CQuakeView::OnCreate(lpCreateStruct) == -1)
		return -1;
	TRACE0("OnCreate\r\n") ;

	BOOL bResult = m_pGL->Create(this,PFD_TYPE_RGBA,PFD_SUPPORT_OPENGL |          // Use OpenGL
	               					                        PFD_DRAW_TO_WINDOW) ;
	if (bResult)
		return 0 ;
	else
		return -1;
}

//
// OnSize: Resize the CGL object.
//
void CameraView::OnSize(UINT nType, int cx, int cy) 
{
	CQuakeView::OnSize(nType, cx, cy);	
	TRACE0("OnSize\r\n") ;

	m_pGL->Resize(cx,cy) ;	
}

//
// OnInitialUpate: Initialize the CGL object.
//
void CameraView::OnInitialUpdate() 
{
	TRACE0("OnInitialUpdate\r\n") ;	
	
	m_pGL->Init() ;
	CQuakeView::OnInitialUpdate();

  ShowScrollBar(SB_BOTH,FALSE);
}

//
// OnDestory: Destory the CGL object.
//
void CameraView::OnDestroy() 
{
	m_pGL->Destroy() ;
	CQuakeView::OnDestroy();
}

void CameraView::OnPaletteChanged(CWnd* pFocusWnd) 
{
	//CQuakeView::OnPaletteChanged(pFocusWnd);
	TRACE("OnPaletteChanged\r\n") ;
    // See if the change was caused by us and ignore it if not.
    if (pFocusWnd != this) {
        OnQueryNewPalette();
    }
}

BOOL CameraView::OnQueryNewPalette() 
{
	//	return CQuakeView::OnQueryNewPalette();
	TRACE0("OnQueryNewPalette\r\n") ;
    // We are going active or the system palette has changed,
    // so realize our palette.
	CPalette* pPal = m_pGL->GetPalette() ;
    if (pPal) {   	
        CDC* pDC = GetDC();
        pDC->SelectPalette(pPal, FALSE);
        UINT u = pDC->RealizePalette();
        ReleaseDC(pDC);
       	if (u != 0) {
            // Some colors changed, so we need to do a repaint.
            InvalidateRect(NULL, TRUE); // Repaint the lot.
        }
    }
    return TRUE; // Windows ignores this.	
}

void CameraView::WindowToMap(CPoint& pt,vec3_t vec)
{
  // texture mode
  GLdouble modelMatrix[16];
  GLdouble projectionMatrix[16];
  GLint    viewport[4];
  GLdouble objX,objY,objZ;

  glGetDoublev(GL_MODELVIEW_MATRIX,modelMatrix);
  glGetDoublev(GL_PROJECTION_MATRIX,projectionMatrix);
  glGetIntegerv(GL_VIEWPORT, viewport);

  gluUnProject(pt.x,pt.y,1.0,modelMatrix,
               projectionMatrix,viewport,&objX,&objY,&objZ);

  vec[0] = objX;
  vec[1] = objY;
  vec[2] = objZ;
}

CGLMap *glmap_i = NULL;

CGLMap::CGLMap()
{
  glmap_i = this;
}

CGLMap::~CGLMap()
{
}


BOOL CGLMap::OnResize(int cx, int cy)
{
  //
  // Set up mapping 3-D coordinates to the 2-D screen
  //
  GLdouble gldAspect = (GLdouble) cx / (GLdouble)cy;
	glMatrixMode(GL_PROJECTION); 
  glLoadIdentity();
  gluPerspective(46.0, gldAspect, 1.0, 10000.0);
  glViewport(0, 0, cx, cy);
	
  return TRUE ;
}

//
//
//
BOOL CGLMap::OnInit()
{
	for (int i=0 ; i < tex_count ; i++)
    if (qtextures[i].glimage && !qtextures[i].alreadyScaled)
    { 
      qtextures[i].glimage->TexMapScalePow2(this);
      qtextures[i].alreadyScaled = TRUE; // avoid not neccessary actions !
  }
	return TRUE ;
}

//
//
//

#define FAST_RENDER

BOOL CGLMap::OnRender()
{
  // Enable depth calculations
  if(renMode == render) 
    glEnable(GL_DEPTH_TEST);
  else
    glDisable(GL_DEPTH_TEST);

	// Set the material color to follow the current color
#ifndef FAST_RENDER	
  glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE) ;
	if(renMode == render) 
    glEnable(GL_COLOR_MATERIAL) ;
  else 
    glDisable(GL_COLOR_MATERIAL) ;
#else
  glDisable(GL_COLOR_MATERIAL) ;
#endif

	// Enable lighting calculations

#ifndef FAST_RENDER
  if(renMode == render)
  {
    glEnable(GL_LIGHTING) ; 		
	  glEnable(GL_LIGHT0) ;			
  }
  else
#endif
  {
    glDisable(GL_LIGHTING) ; 		
	  glDisable(GL_LIGHT0) ;			
  }
	
   // Texturing
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

   // Textures repeat.
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

   // Set tecturing mode to Decal or Modulate
	glTexEnvi( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

  // Low quality, but faster speed.
  if(preferences_i->getFastRender())
  {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  }
  else
  {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  }
  
  if(renMode == render)
    glEnable(GL_TEXTURE_2D);
  else
    glDisable(GL_TEXTURE_2D);
  
	if(renMode != selection)
  {
    // Set the color to clear the screen
	  glClearColor(1.0f, 1.0f, 1.0f, 0.0f) ;

    // Clear the color and depth buffers
	  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  }

#ifdef FAST_RENDER
  glShadeModel(GL_FLAT);
#else
  if(renMode != render)
    glShadeModel(GL_FLAT);
  else
    glShadeModel(GL_SMOOTH);
#endif

  switch(renMode)
  {
    
    case render:map_i->enableAbort(TRUE); 
                map_i->makeAllPerform(BRUSH_CAMERA_RENDERSELF); 
                map_i->enableAbort(FALSE); 
                clipper_i->cameraDrawSelf();
                break;
    case wire: map_i->makeAllPerform(BRUSH_CAMERADRAWSELF); 
               clipper_i->cameraDrawSelf();
               break;
  }
  
  //
  // Flush the drawing pipeline.
  //
  if(renMode != selection)
    glFlush();

  return TRUE ;
}

void CGLMap::BeginRender()
{
	// Make the HGLRC current
	MakeCurrent() ;

	// Draw	
	OnRender() ;
}

void CGLMap::EndRender()
{
   // Glush OpenGL 
	glFlush() ;

   // Flush GDI
   GdiFlush() ;

	// Swap Buffers
	if (m_bDoubleBuffer)
   {
		  SwapBuffers(m_pdc->m_hDC) ;
      GdiFlush() ;
   }
}

void CameraView::homeView()
{
	xa = za = 0;
	
  quakeed_i->updateAll();

	qprintf ("homed view angle");
}

void CameraView::OnRButtonDown(UINT nFlags, CPoint point) 
{
//  WindowToMap(point);

	qprintf ("looking");
	viewDrag(&point);
}
