/*
======================================
Simple Client-side listener app that waits for 
commands from the game server and plays midis


Wed, May26th, 1999
Gaz "Skid" Iqbal
skid@planetquake.com
======================================
*/

#include "gclient.h"
#include "gmci.h"

//Can't use DirectMusic until i find out how to play DirectMusic in a global soundbuffer
//right now it'll only play when the genmidi.exe has focus.
#ifdef DXM						
#include "gdxmusic.h"
#endif


#include <winsock.h>

//Message codes

//INBOUND MESSAGE IDS 
//These get ack'ed back to the server

#define MSG_PLAY		'a'		
#define	MSG_PAUSE		'b'
#define MSG_RESUME		'c'
#define MSG_STOP		'd'
#define MSG_ERROR		'e'
#define MSG_DPLAYINTERMISSION		'i'		

//incoming and outgoing buffers
#define BUF_INSIZE		10
#define BUF_OUTSIZE		2

//Windows message id
#define SOCK_MSG	(WM_USER+1)

//Max log size for console window
#define LOG_SIZE 8184

//Socket states
enum SockSt
{
	SOCK_NOTREADY,
	SOCK_CONNECTING,
	SOCK_CONNECTED
};

SockSt		sstate;				//socket state

//Socket and addresses
SOCKET		sock;				//the socket
SOCKADDR_IN	saddr;				//our addr info
SOCKADDR_IN r_addr;				//remote address

//Incoming outgoing buffers
char bufin[BUF_INSIZE];
char bufout[BUF_OUTSIZE];


char searchdir[_MAX_PATH];		//DirectMusic searches here
char loopmidiname[16];			//complete name of the midi to loop

FILE *fp_log;					//file pointer for Log file
bool directmusic = false;		//are we using directmusic ?

#ifdef DXM
CDirectMusic	g_dxm;			//our DirectMusic class Object
#endif

CMidi			g_midi;			//our MCI interface class Object


//Keywork list for Config File
const char * configcmds[] =
{
	"logfile",
	"port",
	0
};

//user controlled vars
int port=28000;
bool logfile=false;

//Windows stuff
HWND g_hLog;
HWND g_hEdit;
HWND g_hWnd;
HICON	appicon;


/*
======================================
Utility Funcs
======================================
*/

char* ConvertString (char *buf)
{
	static	char	buf2[LOG_SIZE];
	int		i, l;
	char	*out;

	l = strlen(buf);
	out = buf2;

	if(strlen(buf2) >= LOG_SIZE)
		memset(buf2,0,LOG_SIZE);

	for (i=0 ; i<l ; i++)
	{
		if (buf[i] == '\n')
		{
			*out++ = '\r';
			*out++ = '\n';
		}
		else
			*out++ = buf[i];
	}
	*out++ = 0;

	return buf2;
}

void AddMsg(char *text)
{
	char *out = ConvertString (text);
	SendMessage(g_hLog, EM_REPLACESEL, 0, (LPARAM)out);
	//SendMessage(g_hLog,WM_SETTEXT ,0,(LPARAM)out);
}


/*
======================================
Our print func
======================================
*/

void dprintf(char *string, ...)
{
	char textBuffer[LOG_SIZE];

	va_list args;
	va_start(args, string);
	vsprintf(textBuffer, string, args);
	va_end(args);
	AddMsg(textBuffer);

	if(fp_log && logfile)
		fputs(textBuffer,fp_log);
}

/*
======================================
throw a default message box
======================================
*/

void messagebox(char *string)
{
	char Buffer[1024];

	va_list args;
	va_start(args, string);
	vsprintf(Buffer, string, args);
	va_end(args);

	MessageBox(0,Buffer, "Error",0);
}


/*
======================================
Util function - prints out Winsock errors
======================================
*/

void PrintSockError()
{
	char error[128];
	strcpy(error,"Socket Error :");

	switch(WSAGetLastError())
	{
	case 10013:strcat(error,"WSAEACCES - error in accessing socket"); break;
	case 10048:strcat(error,"WSAEADDRINUSE - address is in use"); break;
	case 10049:strcat(error,"WSAEADDRNOTAVAIL - address is not valid in context"); break;
	case 10047:strcat(error,"WSAEAFNOSUPPORT - address family not supported by protocol"); break;
	case 10037:strcat(error,"WSAEALREADY - operation already in progress"); break;
	case 10053:strcat(error,"WSACONNABORTED - software caused connection aborted"); break;
	case 10061:strcat(error,"WSAECONNREFUSED - connection refused"); break;
	case 10054:strcat(error,"WSAECONNRESET - connection reset by peer"); break;
	case 10039:strcat(error,"WSAEDESTADDRREQ - destination address required"); break;
	case 10014:strcat(error,"WSAEFAULT - bad address"); break;
	case 10064:strcat(error,"WSAEHOSTDOWN - host is down"); break;
	case 10065:strcat(error,"WSAEHOSTUNREACH - no route to host"); break;
	case 10036:strcat(error,"WSAEINPROGRESS - operation now in progress"); break;
	case 10004:strcat(error,"WSAEINTR - interrupted function call"); break;
	case 10022:strcat(error,"WSAEINVAL - invalid argument"); break;
	case 10056:strcat(error,"WSAEISCONN - socket is already connected"); break;
	case 10024:strcat(error,"WSAEMFILE - too many open files"); break;
	case 10040:strcat(error,"WSAEMSGSIZE - message to long"); break;
	case 10050:strcat(error,"WSAENETDOWN - network is down"); break;
	case 10052:strcat(error,"WSAENETRESET - network dropped connection on reset"); break;
	case 10051:strcat(error,"WSAENETUNREACH - network is unreachable"); break;
	case 10055:strcat(error,"WSAENOBUFS - no buffer space available"); break;
	case 10042:strcat(error,"WSAENOPROTOOPT - bad protocol option"); break;
	case 10057:strcat(error,"WSAENOTCONN - socket is not connected"); break;
	case 10038:strcat(error,"WSAENOTSOCK - socket operation on non-socket"); break;
	case 10045:strcat(error,"WSAEOPNOTSUPP - operation not supported"); break;
	case 10046:strcat(error,"WSAEPFNOSUPPORT - protocol family not supported"); break;
	case 10067:strcat(error,"WSAEPROCLIM - too many processes"); break;
	case 10043:strcat(error,"WSAEPROTONOSUPPORT - protocol not supported"); break;
	case 10041:strcat(error,"WSAEPROTOTYPE - protocol wrong type for socket"); break;
	case 10058:strcat(error,"WSAESHUTDOWN - cannot send after socket shutdown"); break;
	case 10044:strcat(error,"WSAESOCKTNOSUPPORT - socket type not supported"); break;
	case 10060:strcat(error,"WSAETIMEDOUT - connection timed out"); break;
	case 10035:strcat(error,"WSAEWOULDBLOCK - resource temporarily unavailable"); break;
	case 11001:strcat(error,"WSAHOST_NOT_FOUND - host not found"); break;
	case 10093:strcat(error,"WSANOTINITIALISED - WSAStartup not yet performed"); break;
	case 11004:strcat(error,"WSANO_DATA - valid name, no data record of requested type"); break;
	case 11003:strcat(error,"WSANO_RECOVERY - non-recoverable error"); break;
	case 10091:strcat(error,"WSASYSNOTREADY - network subsystem is unavailable"); break;
	case 11002:strcat(error,"WSATRY_AGAIN - non-authoritative host not found"); break;
	case 10092:strcat(error,"WSAVERNOTSUPPORTED - winsock.dll verison out of range"); break;
	case 10094:strcat(error,"WSAEDISCON - graceful shutdown in progress"); break;
	}
	dprintf("%s\n",error);
}


void PrintErrorString(int nerror)
{
	char error[128];
	strcpy(error,"Socket Error :");

	switch(nerror)
	{
	case 10013:strcat(error,"WSAEACCES - error in accessing socket"); break;
	case 10048:strcat(error,"WSAEADDRINUSE - address is in use"); break;
	case 10049:strcat(error,"WSAEADDRNOTAVAIL - address is not valid in context"); break;
	case 10047:strcat(error,"WSAEAFNOSUPPORT - address family not supported by protocol"); break;
	case 10037:strcat(error,"WSAEALREADY - operation already in progress"); break;
	case 10053:strcat(error,"WSACONNABORTED - software caused connection aborted"); break;
	case 10061:strcat(error,"WSAECONNREFUSED - connection refused"); break;
	case 10054:strcat(error,"WSAECONNRESET - connection reset by peer"); break;
	case 10039:strcat(error,"WSAEDESTADDRREQ - destination address required"); break;
	case 10014:strcat(error,"WSAEFAULT - bad address"); break;
	case 10064:strcat(error,"WSAEHOSTDOWN - host is down"); break;
	case 10065:strcat(error,"WSAEHOSTUNREACH - no route to host"); break;
	case 10036:strcat(error,"WSAEINPROGRESS - operation now in progress"); break;
	case 10004:strcat(error,"WSAEINTR - interrupted function call"); break;
	case 10022:strcat(error,"WSAEINVAL - invalid argument"); break;
	case 10056:strcat(error,"WSAEISCONN - socket is already connected"); break;
	case 10024:strcat(error,"WSAEMFILE - too many open files"); break;
	case 10040:strcat(error,"WSAEMSGSIZE - message to long"); break;
	case 10050:strcat(error,"WSAENETDOWN - network is down"); break;
	case 10052:strcat(error,"WSAENETRESET - network dropped connection on reset"); break;
	case 10051:strcat(error,"WSAENETUNREACH - network is unreachable"); break;
	case 10055:strcat(error,"WSAENOBUFS - no buffer space available"); break;
	case 10042:strcat(error,"WSAENOPROTOOPT - bad protocol option"); break;
	case 10057:strcat(error,"WSAENOTCONN - socket is not connected"); break;
	case 10038:strcat(error,"WSAENOTSOCK - socket operation on non-socket"); break;
	case 10045:strcat(error,"WSAEOPNOTSUPP - operation not supported"); break;
	case 10046:strcat(error,"WSAEPFNOSUPPORT - protocol family not supported"); break;
	case 10067:strcat(error,"WSAEPROCLIM - too many processes"); break;
	case 10043:strcat(error,"WSAEPROTONOSUPPORT - protocol not supported"); break;
	case 10041:strcat(error,"WSAEPROTOTYPE - protocol wrong type for socket"); break;
	case 10058:strcat(error,"WSAESHUTDOWN - cannot send after socket shutdown"); break;
	case 10044:strcat(error,"WSAESOCKTNOSUPPORT - socket type not supported"); break;
	case 10060:strcat(error,"WSAETIMEDOUT - connection timed out"); break;
	case 10035:strcat(error,"WSAEWOULDBLOCK - resource temporarily unavailable"); break;
	case 11001:strcat(error,"WSAHOST_NOT_FOUND - host not found"); break;
	case 10093:strcat(error,"WSANOTINITIALISED - WSAStartup not yet performed"); break;
	case 11004:strcat(error,"WSANO_DATA - valid name, no data record of requested type"); break;
	case 11003:strcat(error,"WSANO_RECOVERY - non-recoverable error"); break;
	case 10091:strcat(error,"WSASYSNOTREADY - network subsystem is unavailable"); break;
	case 11002:strcat(error,"WSATRY_AGAIN - non-authoritative host not found"); break;
	case 10092:strcat(error,"WSAVERNOTSUPPORTED - winsock.dll verison out of range"); break;
	case 10094:strcat(error,"WSAEDISCON - graceful shutdown in progress"); break;
	}
	dprintf("%s\n",error);
}




/*
======================================
Init Winsock2 
======================================
*/

bool GenInitWinsock(void)
{
	WORD wVersionRequested;
	WSADATA wsaData;
	int err;
	
	sstate = SOCK_NOTREADY;

	//Init Winsock
	wVersionRequested = MAKEWORD( 2, 0 ); 
	err = WSAStartup( wVersionRequested, &wsaData );
	
	if ( err != 0 ) 
	{
		/* Tell the user that we couldn't find a usable 
		   WinSock DLL.                                 */    
		dprintf("GenInitSocks::Could not find a usable Winsock dll\n");
		return false;
	} 
	
		/* Confirm that the WinSock DLL supports 2.0.		 */
		/* Note that if the DLL supports versions greater    */
		/* than 2.0 in addition to 2.0, it will still return */
		/* 2.0 in wVersion since that is the version we      */
		/* requested.                                        */ 

	if ( LOBYTE( wsaData.wVersion ) != 2 ||
         HIBYTE( wsaData.wVersion ) != 0 ) 
	{
		/* Tell the user that we couldn't find a usable 
		   WinSock DLL.                                 */    
		WSACleanup();
		dprintf("GenInitSocks::Could not find Winsock 2.0\n");
		return false;
	}  
	// The WinSock DLL is acceptable. Proceed. 

	memset(bufin,0,sizeof(bufin));
	memset(bufout,0,sizeof(bufout));
	return true;
}


/*
======================================
Init our listener Socket
======================================

WSA Events
FD_READ  Want to receive notification of readiness for reading 
FD_WRITE  Want to receive notification of readiness for writing 
FD_OOB  Want to receive notification of the arrival of out-of-band data 
FD_ACCEPT  Want to receive notification of incoming connections 
FD_CONNECT  Want to receive notification of completed connection 
FD_CLOSE  Want to receive notification of socket closure 

*/

bool GenInitSock()
{
	char   myname[256];
	struct hostent *hp;
	
	//Init Local Address Info
	memset(&saddr, 0, sizeof(struct sockaddr_in));	
	gethostname(myname, sizeof(myname));				
	hp = gethostbyname(myname);							
  
	if (hp == NULL)										
	{
		printf("GenInitSocks::Bad Host Address\n");
		return false;
	}

	saddr.sin_family = hp->h_addrtype; 
	saddr.sin_port = htons(port);				

	//Init Socket
	sock = socket(AF_INET, SOCK_STREAM, 0);		
	if (sock == INVALID_SOCKET)
	{
		dprintf("Invalid Listener Socket Created\n");
		return false;
	}

	//Setup the Listener Socket
	if(!setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,"1",sizeof(int)))
		dprintf("== Socket Options Set ==\n");
	else
		dprintf("GenInitSocks::Error setting Socket Options:%d\n",WSAGetLastError());

	// bind the socket to the internet address 
	if (bind(sock, (struct sockaddr *)&saddr, 
					   sizeof(struct sockaddr_in)) == SOCKET_ERROR) 
	{
		closesocket(sock);
		dprintf("GenInitSocks::Could not bind socket\n");
		return false;
	}

	//place socket in listening state
	if(listen(sock,1)== SOCKET_ERROR)
	{
		dprintf("Listen() failed\n");
		PrintSockError();
		closesocket(sock);
		return false;
	}

	if(WSAAsyncSelect(sock,g_hWnd,SOCK_MSG,(FD_READ|FD_CLOSE|FD_ACCEPT))== SOCKET_ERROR)
	{
		dprintf("WSAAsynchSelect() failed\n");
		PrintSockError();
		closesocket(sock);
		return false;
	}

	sstate = SOCK_CONNECTING;
	dprintf("== Socket listening ==\n");
	return true;
}



/*
======================================
Shutdown Socket
======================================
*/

bool GenShutdownSockets()
{
	if(sock != INVALID_SOCKET)
		closesocket(sock);
	WSACleanup();

//Close all open sockets
	return true;
}


/*
======================================
GenWriteSock
======================================
*/

bool GenWriteSock()
{
	int rc;

	dprintf("Sending [%s] to server\n",bufout);
	rc = send(sock,bufout,sizeof(bufout), 0);
	memset(bufout,0,sizeof(bufout));

	if (rc == SOCKET_ERROR) 
	{
	  dprintf("GenWriteSock:: Error sending data\n");
	  PrintSockError();
	  return false;
	}
	
	if (rc ==0) 
	{
		dprintf("GenWriteSock:: Connection Closed by Server\n");
		return false;
	}
	return true;
}

/*
======================================
GenReadSock
======================================
*/

//bleh
void GenSendError()
{
	strcpy(bufout,"e");
	GenWriteSock();
}

bool GenReadSock()
{
	int rc;
	static char midiname[12];
	rc = recv(sock,bufin,sizeof(bufin),0);

	if (rc == SOCKET_ERROR) 
	{
		//printf("recv() failed with error %d\n",WSAGetLastError());	
		PrintSockError();
	  	return false;
	}
	
	if (rc ==0) 
	{
		dprintf("Connection closed by server\n");
		return false;
	}
	dprintf("Received [%s] from server\n",bufin);

	//do whatever with the command received from the server
	switch(bufin[0])
	{
	case MSG_PLAY:
		{
			char *c = bufin;
			c++;
			if(strlen(bufin) <2)
			{
				dprintf("Message discarded, too small\n",bufin);
				break;
			}
			strcpy(midiname,c);
			strcat(midiname,".mid");
			
			dprintf("Trying to play : %s\n",midiname);
			if(
#ifdef DXM
				((directmusic==true) && g_dxm.Play(midiname)) ||
#endif
				(g_midi.Play(midiname)))
			{
				strcpy(loopmidiname,midiname);
				strcpy(bufout,"a");
				GenWriteSock();
			}
			else
				GenSendError();
			break;
		}
	case MSG_STOP:
		{
			if(
#ifdef DXM
				((directmusic == true) && g_dxm.Stop()) ||
#endif
				(g_midi.Stop()))
			{
				strcpy(bufout,"d");
				GenWriteSock();
			}
			else
				GenSendError();
			break;
		}
	case MSG_PAUSE:
		{
			if(
#ifdef DXM
				((directmusic == true) && g_dxm.Pause()) ||
#endif
				(g_midi.Pause()))
			{
				strcpy(bufout,"b");
				GenWriteSock();
			}
			else
				GenSendError();
			break;
		}
	case MSG_RESUME:
		{
			if(
#ifdef DXM
				((directmusic == true) && g_dxm.Resume()) ||	
#endif
				(g_midi.Resume()))
			{
				strcpy(bufout,"c");
				GenWriteSock();
			}
			else
				GenSendError();
			break;
		}
	case MSG_DPLAYINTERMISSION:
		{
			if(g_midi.Play("Dm2int.mid"))
			{
				strcpy(bufout,"i");
				GenWriteSock();
			}
			else
				GenSendError();
			break;
		}
	}
	memset(bufin,0,sizeof(bufin));
	return true;
}



/*===============================================
	MainWndProc
===============================================*/
LRESULT CALLBACK MainWndProc(HWND hWnd, UINT msg, 
				    	     WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_CREATE:
		{
			
			//Create logging window
			g_hLog= CreateWindow("EDIT",
							   "",
								WS_CHILD|WS_VISIBLE|WS_VSCROLL|ES_READONLY|ES_MULTILINE|ES_OEMCONVERT,  //|ES_AUTOVSCROLL
								14,
								10,
								325,
								360,
								hWnd,
								0,
								0,
								0
								);

			//Create Edit Box
#if 0
			g_hEdit= CreateWindow("EDIT",
							   "",
								WS_CHILD|WS_VISIBLE,
								14,
								380,
								312,
								25,
								hWnd,
								0,
								0,
								0
								);
#endif
			break;
		}
	//Asynch Socket Event
	case SOCK_MSG:
		{
			LPARAM para = lParam;
			unsigned int sevent = WSAGETSELECTEVENT(para);
			unsigned int serror = WSAGETSELECTERROR(para);
			
			if(serror)
				PrintErrorString(serror);

			switch(sevent)
			{
			case FD_READ:		//incoming data
				{
					GenReadSock();
					break;
				}
			case FD_CLOSE:		//connection closed
				{
					dprintf("Connection Closed\n");
					WSAAsyncSelect(sock,hWnd,0,0);
					closesocket(sock);
					sstate = SOCK_NOTREADY;
					if(GenInitSock())
					{
						dprintf("== Socket Reintialized ==\n");
					}
					else
						dprintf("== Failed to reintialize socket ==\n");
					break;
				}
			case FD_ACCEPT:		//got a connection request
				{
					int len = sizeof(r_addr);
					dprintf("Incoming connection request\n");
					sock = accept(sock, (struct sockaddr *)&r_addr, &len);
					if(sock != INVALID_SOCKET)
					{
						dprintf("Connected to %s\n",inet_ntoa(r_addr.sin_addr));
						sstate = SOCK_CONNECTED;
					}
					else
					{
						dprintf("Failed to Connect\n");
						PrintSockError();
					}
					break;
				}
			}
			break;
		}
	case MM_MCINOTIFY:
		{
			if(wParam == MCI_NOTIFY_SUCCESSFUL)
			{
				dprintf("Received MCI notification message\n");
				if(g_midi.SeekToStart())
				{
					g_midi.Play(loopmidiname);
					dprintf("Restarting midi\n");
				}
			}
			//else
			//	dprintf("Error Receiving MCI notification message\n");
			break;
		}
	case WM_CLOSE:
	case WM_DESTROY:
	case WM_QUIT:
		{
			g_midi.Shutdown();
#ifdef DXM
			g_dxm.Shutdown();
#endif 
			GenShutdownSockets();
			if(fp_log)
				fclose(fp_log);
			exit(1);
			break; 
		}
	}
	return(DefWindowProc(hWnd, msg, wParam, lParam));
}


/*===============================================
	Windows Entry Point Function
===============================================*/
int WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrevInst,
				   LPSTR lpCmdLine, int nCmdShow)
{
	MSG msg;
	FILE *fp_cfg;
	static char wndclassname[] = "Genmidi";
	WNDCLASSEX	wcl;

	appicon = LoadIcon(hInst, MAKEINTRESOURCE(IDI_APPLICATION));//LoadIcon(hInst, MAKEINTRESOURCE(IDI_ICON1));	


	wcl.cbSize = sizeof(WNDCLASSEX);
	wcl.style = CS_DBLCLKS|CS_BYTEALIGNCLIENT|CS_BYTEALIGNWINDOW | CS_HREDRAW|CS_VREDRAW;
	wcl.hIcon = appicon;
	wcl.hIconSm = appicon; 
	wcl.hCursor = LoadCursor(NULL, IDC_ARROW);
	wcl.lpszMenuName = NULL;
	wcl.cbClsExtra = 0;
	wcl.cbWndExtra = 0;
	wcl.hInstance = hInst;
	wcl.lpszClassName = wndclassname;
	wcl.lpfnWndProc = MainWndProc;
	wcl.hbrBackground = (HBRUSH)GetStockObject(GRAY_BRUSH);

	if (!RegisterClassEx(&wcl))
	{
		MessageBox(GetForegroundWindow(),"Couldnt register Window", "Error",0);
		exit(1);
	} 


	g_hWnd= CreateWindowEx(WS_EX_LEFT,
						wndclassname,
						"Generations Midi Client",
						WS_CLIPCHILDREN|WS_MINIMIZEBOX|WS_CAPTION|WS_OVERLAPPED|WS_SYSMENU|WS_VISIBLE, 
						CW_USEDEFAULT,
						CW_USEDEFAULT,
#if 0
						360,
						450,
#else
						360,
						405,
#endif
						0,
						0,
						hInst,
						&wcl
						);

    // Display the window we just created.
    ShowWindow(g_hWnd, nCmdShow);
    UpdateWindow(g_hWnd);

	//Open config file
	if((fp_cfg = fopen("genmidi.cfg","r")) != NULL)
	{
		char c;
		char line[80];
		int i=0;
		int t=0;
		int keyword=0;
						
		//read the damn file
		do
		{
			//read a damn line
			c=fgetc(fp_cfg);
			memset(line,0,sizeof(line));
			i=0;
			
			//Tokenize the line into damn tokens ;)
			while((c != EOF) && (c != '\n') && (c!= ' ') && (c!= '\t') && (i<80))
			{
				line[i] = c;
			    c=fgetc(fp_cfg);
				i++;
			}

			//Process the tokens
			if(((t%2)==0) || t==0) //if the number of the token is even, match with our keywork list
			{
				for(int j=0;configcmds[j] != 0; j++)
				{
					if(!strcmp(configcmds[j],line)) //we have a match, save the id of the keyword
					{
						keyword = j;
						break;
					}
					keyword= -1;
				}
			}
			else	//odd token, SHOULD be a value corresponding to a keyword
			{
				//do we have a keyword id 
				if(keyword >= 0)
				{
					int a;
					if(sscanf(line,"%d",&a))
					{
						dprintf("%s is %d\n",configcmds[keyword],a);
						
						//add any more vars here
						switch(keyword)
						{
						case 0:
							if(a > 0)
								logfile = true;
							break;
						case 1:
							if(a > 0 && a < 65535)
								port = a;
							else
								dprintf("Incorrect Port specified in Cfg\n");
							break;
						}
					}
				}
			}
			t++;
		}while(c != EOF);
	}
	else
	{
		dprintf("couldnt read config file, using defaults\n");
	}

	//Open up the log file
	if(logfile)
	if((fp_log = fopen("genmidi.log","w")) == NULL)
	{
		messagebox("Couldnt open logfile\n");
	}

	//get current directory
	if(_getcwd(searchdir,_MAX_PATH)==NULL)
	{	dprintf("Error getting current directory\n");
	}
	else
	{
		char *temp = _strlwr(searchdir);
		strcpy(searchdir,temp);
		dprintf("Current Directory: %s\n", searchdir);
	}

	//throw a message if we arent in the quake2\gen dir
	char *subs;
	subs = strstr(searchdir,"quake2" );
	if(subs == NULL)
		dprintf("WARNING - need to run this from quake2\\gen\n");
	subs = strstr(searchdir,"gen" );
	if(subs == NULL)
		dprintf("WARNING-need to run this from quake2\\gen\n");

	strcat(searchdir,"\\midi");

#ifdef DXM
	if(!g_dxm.Init(searchdir))
	{	dprintf("Failed Direct Music init\n");
		directmusic=false;
	}
	else 
	{	dprintf("== Init Direct Music ==\n");
		directmusic=true;
	}
#else
	directmusic = false;
#endif

	
	if(!g_midi.Init())
	{	dprintf("Failed MCI init\n");
	}
	else
	{	dprintf("==Init MCI==\n");
	}

	
	//Start up Winsock
	if(!GenInitWinsock())
	{
			MessageBox(0,"Couldnt Initialize Winsock\n", "Error",0);
			PostMessage(g_hWnd,	// handle of destination window 
			WM_QUIT,			// message to post 
			0,					// first message parameter 
			0);
	}
	dprintf("== Initialized Winsock ==\n");
	
	//Create our listener socket
	if(!GenInitSock())
	{
			MessageBox(0,"Couldnt Initialize listener socket\n", "Error",0);
			PostMessage(g_hWnd,	// handle of destination window 
			WM_QUIT,			// message to post 
			0,					// first message parameter 
			0);
	}

	//the application message loop
	while (1)
	{
		if(PeekMessage(&msg,g_hWnd,NULL, 0, PM_REMOVE)) 
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}
	return 0;
}











