/**********************************************************
Quake2 Cluster Project
An addition and modification of iD Software's Quake2 shared 
library sources which enable Quake2 servers running this 
library to interconnect other servers which speak the 
protocol developed under the Quake2 Cluster Project.
Copyright (c) 1998 Justin Randall and Todd Bliss

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; version 2
of the License.

This program is distributed in the hope that it will be 
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more 
details. You should have received a copy of the GNU General 
Public License along with this program; if not, write to 
the Free Software Foundation, Inc., 
59 Temple Place - Suite 330, 
Boston, MA  02111-1307, USA.
**********************************************************/
/**********************************************************
*	$Log: cluster_protocol.c,v $
*	Revision 1.5  1998/01/31 09:53:59  logic
*	Added RULES_ClientUserInfoChanged for access to client variables.
*	Fixed the password bug
*	Fixed the looping respawn/telefrag bug
*	Added Invincibility functions to RULES
*	Added ClientThink2 to RULES
*	Added RULES_ClientCmd
*	Added cmd invincible to RULES
*	Added ()'s around player names on broadcast messages
*	Added MOTD motd-mapname.txt
*	Changed invincibility fx
*
*	Revision 1.4  1998/01/29 23:26:33  logic
*	Fixed gibbable head bug.
*
*	Revision 1.3  1998/01/28 04:03:10  logic
*	I think the Big Fucking Memory Leak bug has been irradicated.
*
*	Fixed:
*	  ClusterListen, a persistant thread, was spinning ClusterMiniServer,
*	  a temporary worker thread, which would also spin some temporary and
*	  some semi-persistant threads. If those threads died, and ClusterMiniServer
*	  was no longer available, the allocated system resources would not clean
*	  up --hence the hosed servers in the morning.
*
*	  ToDo:
*	  Fix ClusterExit relinks from an entrance that is still sending keepalives.
*	   if the exit sends a keepalive to an exit that is not linked to it, then
*	   the exit should return a NACK to tell the damned thing to shutup and quit
*	   spamming the network.
*
*	  Fix the serverlist generation: it's sloppy, piggish, and doesn't take
*	    dead systems very well. Also slow and nowhere near real time. Flooding
*	    is a bad idea. I need to devise some link-state status update, perhaps
*	    by arrangning some peering capabilities --something like the "directly
*	    connected host" concept of OSPF --if a directly connect host dies, his
*	    peer(s) check their other directly connected hosts to see if they too
*	    are having problems, then a link-state advertisement (heh, LSA) can be
*	    propogated to the rest of the network advising the rest of the cluster
*	    servers that the system is not available. Likewise, when a peering
*	    arrangement is established, another LSA can advise the cluster that
*	    a new server is available. (Who said network geeks make bad programmers?)
*
*	  Add some RULES callbacks for cluster mod authors
*
*	  Add ClusterPlayerLocate() function to identify the location of a player
*	    in the cluster. (perhaps tracking his last known exit rather than
*	    flooding for his location?)
*
*	Revision 1.2  1998/01/26 23:49:53  logic
*	Autolog commenting added to original source files.
*
*	Found bugs but not yet fixed:
*	Windows95 system resources are consumed over a period of time.
*	Links die after a period of time ( > 12 hours)
*
**********************************************************/

#include "udp.h"
#include "windows.h"
#include <stdlib.h>
#include "g_local.h"
#include "cluster_globs.h"
#include "memory.h"


/**********************************************************
* $Log: cluster_protocol.c,v $
* Revision 1.5  1998/01/31 09:53:59  logic
* Added RULES_ClientUserInfoChanged for access to client variables.
* Fixed the password bug
* Fixed the looping respawn/telefrag bug
* Added Invincibility functions to RULES
* Added ClientThink2 to RULES
* Added RULES_ClientCmd
* Added cmd invincible to RULES
* Added ()'s around player names on broadcast messages
* Added MOTD motd-mapname.txt
* Changed invincibility fx
*
* Revision 1.4  1998/01/29 23:26:33  logic
* Fixed gibbable head bug.
*
**********************************************************/

/**********************************************************
	Cluster Messaging Protocol
	The next few functions are for generating and
	obfuscating packet checksums. A low-utilization hack
	to "sign" packets bound for another host.


	Packets with bad signatures are ignored.
**********************************************************/


unsigned char ClusterByteFromLong(unsigned long data);
unsigned int ClusterIntFromLong(unsigned long data);
unsigned char ClusterByteSignData(unsigned char *data, long dataLen, unsigned char *key);
void ClusterIntToBytes(int i, unsigned char *bytes);
void ClusterMiniServer(RecvMSG *Param);

/**********************************************************
My broken and hacked cheap-o checksum routines. I ought to
just throw a decent output feedback stream cipher in here
and be done with the whole security issue.
**********************************************************/
unsigned long ClusterChecksum(unsigned char *data, long dataLen) {
	unsigned long checksum;
	long i;
	
	checksum = 0;
	
	for(i=0;i<dataLen;i++) {
		checksum = checksum + data[i];
	}
	return checksum;
};

unsigned long ClusterKeyedChecksum(unsigned char *data, long dataLen, unsigned char *key, long keyLen) {
	unsigned long checksum;
	checksum = ClusterChecksum(data, dataLen) * ClusterChecksum(key, keyLen);
	return checksum;
};


unsigned char ClusterByteFromLong(unsigned long data) {
   unsigned char byte;
   unsigned long max;

   max = (256 * 256 * 256);
   byte = data % max;
   return byte;
}

unsigned int ClusterIntFromLong(unsigned long data) {	
   unsigned int retval;
   unsigned long max;

   max = (256 * 256);
   retval = data % max;
   return retval;
}
/**********************************************************
	ClusterByteSignData()
	This is the main function to call for packet 
	signatures.
**********************************************************/
unsigned char ClusterByteSignData(unsigned char *data, long dataLen, unsigned char *key) {
	unsigned long checksum;
	unsigned char byte;
	int err;
	
	checksum = 0;
	err = 0;
	checksum = ClusterKeyedChecksum(data, dataLen, key, strlen(key));
	byte = ClusterByteFromLong(checksum);

	return byte;
}


void ClusterSpinNetServer(void) {
/**********************************************************
	Spins this server's comm subsystem.
**********************************************************/
	HANDLE	hListen;
	long lThreadId;

	
	hListen = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE) thread_ClusterListen, NULL, CREATE_SUSPENDED, &lThreadId);
	if(hListen) {
		G_ActiveThreads++;
		if(G_ActiveThreads < 254)
			strcpy(ThreadList[G_ActiveThreads].ThreadName, "thread_ClusterListen");
		
		SetThreadPriority(hListen, THREAD_PRIORITY_NORMAL);
		ResumeThread(hListen);
	} else {
		printf("cluster_protocol.c:ClusterSpinNetServer:CREATE THREAD FAILURE\n");
	}
}

long thread_ClusterListen(void) {
/**********************************************************
	Starts a listening comm process. Each server has
	it's own comm subsystem, and registers with this
	machine's well-known referal.

**********************************************************/
	SOCKET sock;
	struct sockaddr_in MyAddr;
	int ssize, i, flood_interval;
	RecvMSG LocalMSG;
	long ltime, list_timer;
	RecvMSG	buf[255];

	ssize = sizeof(struct sockaddr);
	i = 0;
	flood_interval = 2;
	
	sock = soListen(0);
	if(sock < 1) {
		printf("cluster_protocol.c:thread_ClusterListen:SOCKET ALLOC FAILURE!\n");
		ExitThread(-1);
	}
	
	getsockname(sock, (struct sockaddr *)&MyAddr, &ssize);
	G_LocalComPort = MyAddr.sin_port;
	closesocket(sock);
	
	sock = soListen(G_LocalComPort);
	if(sock < 1) {
		printf("cluster_protocol.c:thread_ClusterListen:SOCKET ALLOC #2 FAILURE!\n");
		ExitThread(-1);
	}

	printf("LOCAL:MiniServer listening on port %d\n", G_LocalComPort);
	_sleep(2000);
	ClusterRegServer();
	list_timer = 0;
	while(1) {
		time(&ltime);
		if(ltime - flood_interval > list_timer) {
			time(&list_timer);
			flood_interval = flood_interval * 2;
			if(flood_interval > 720)
				flood_interval = 720;
			ClusterServerListRequest();
		}
		GetMSG(sock, &LocalMSG);
		memcpy(&buf[i], &LocalMSG, sizeof(RecvMSG));	// Leave struct for new thread.. otherwise GetMSG kills data
		ClusterMiniServer(&buf[i]);
		i++;
		if(i > 254)
			i = 0;
	}
	
	closesocket(sock);
	return 0;
}

void ClusterRegServer(void) {
	ServerRegPacket	LocalInfo;
	char	addr[64];

	strcpy(addr, inet_ntoa(G_ServerAddr));
	LocalInfo.packet_id = PKT_SERVER_REG;
	ClusterIntToBytes(G_ServerPort, LocalInfo.port);
	ClusterIntToBytes(G_LocalComPort, LocalInfo.comport);
	LocalInfo.max_clients = game.maxclients;
	LocalInfo.clients = level.players;

	ClusterSendMSG(addr, G_ReferalPort, (unsigned char *)&LocalInfo, sizeof(LocalInfo), G_ClusterKey);
};

void ClusterSendMSG(char *dest, unsigned short iPort, unsigned char *msg, long Len, unsigned char *key) {
	unsigned char SignedMSG[MAX_MSGLEN];
	unsigned char byte;
		
	memcpy(SignedMSG, msg, Len);
	
	// Minimum packet size is 4, pad the packet before checksum
	if(Len < 4) {
		memset(&SignedMSG[Len], 0, 4 - Len);
		Len = 4;
	}
	
	if(key == NULL) {
		byte = ClusterByteSignData(SignedMSG, Len, G_ClusterKey);
	} else {
		byte = ClusterByteSignData(SignedMSG, Len, key);
	}
	memcpy(&SignedMSG[Len+1], &byte, 1);
	SendMSG(dest, iPort, SignedMSG, Len + 1);
}

int ClusterCheckSig(unsigned char *data, unsigned char *key, long dataLen) {
	unsigned char *junk;
	unsigned char set, match;
	int retval;

	retval = 0;
	junk = (unsigned char *)malloc(dataLen -1);
	if(junk == NULL) {
		printf("cluster_protocol.c:ClusterCheckSig:MALLOC FAILURE!\n");
		return 0;
	}
	memcpy(junk, data, dataLen - 1);
	set = data[dataLen];
	match = ClusterByteSignData(junk, dataLen, key);
	if(set != match) {
		retval = 0;
	} else {
		retval = 1;
	}
	free(junk);
	return retval;
}

void ClusterIntToBytes(int i, unsigned char *bytes) {
	bytes[0] = i / 256;
	bytes[1] = i % 256;
	return;
}

unsigned short ClusterCharToUS(unsigned char *bytes) {
	return (bytes[0] * 256) + bytes[1];
}
int ClusterCharToINT(unsigned char *bytes) {
	return ((char)bytes[0] * 256) + (char)bytes[1];
}
/**********************************************************
	ClusterMiniServer()

	When a packet is received, it is passed to this thread
	for processing.

	The miniserver interperets incoming messages and 
	performs functions based on the packet_id. messages
	sent to this server specifically are handled here,
	whereas db registrations are handled by the machine
	db server.
**********************************************************/
void ClusterMiniServer(RecvMSG *InComing) {
	// New thread for each packet received
	HANDLE	hNewThread;
	long lThreadId;

	
	switch ( InComing->cData[0] ) {
		case PKT_ENTER_GET_NACK:
			break;
		case PKT_ENTER_REQ_HB:
			// Spin a heartbeat thread
			hNewThread = CreateThread (NULL, 0, (LPTHREAD_START_ROUTINE) ClusterEnterHeartBeat, InComing, CREATE_SUSPENDED, &lThreadId);
			if(hNewThread) {
				G_ActiveThreads++;
				if(G_ActiveThreads < 254)
					strcpy(ThreadList[G_ActiveThreads].ThreadName, "ClusterEnterHeartBeat");
				SetThreadPriority(hNewThread, THREAD_PRIORITY_NORMAL);
				ResumeThread(hNewThread);
			} else {
				printf("cluster_protocol.c:ClusterMiniServer:ClusterEnterHeartBeat:CREATE THREAD FAILURE\n");
			}
				
			break;
		case PKT_ENTER_KEEPALIVE:
			ClusterReceiveEnterKeepAlive(InComing);
			
			break;
		case PKT_CLIENT_SET:
			hNewThread = CreateThread (NULL, 0, (LPTHREAD_START_ROUTINE) ClusterRecvPlayer, InComing, CREATE_SUSPENDED, &lThreadId);
			if(hNewThread) {
				G_ActiveThreads++;
				if(G_ActiveThreads < 254)
					strcpy(ThreadList[G_ActiveThreads].ThreadName, "ClusterRecvPlayer");
				SetThreadPriority(hNewThread, THREAD_PRIORITY_NORMAL);
				ResumeThread(hNewThread);
			} else {
				printf("cluster_protocol.c:ClusterMiniServer:ClusterRecvPlayer:CREATE THREAD FAILURE\n");
			}
				
			break;
		case PKT_EDICT_SET:
			// Received an edict value packet
			// spin a thread for each of these received
			// have the thread loop until a timeout value or until the data is received
			// then if data received set edict value and leave or just leave
			break;
		case PKT_ITEM_SEND:
			// Receving some inventory for a player
			// memcpy(&G_ExitMSG, &Packet, sizeof(RecvMSG));
			hNewThread = CreateThread (NULL, 0, (LPTHREAD_START_ROUTINE) ClusterRecvPlayerItem, InComing, CREATE_SUSPENDED, &lThreadId);
			if(hNewThread) {
				G_ActiveThreads++;
				if(G_ActiveThreads < 254)
					strcpy(ThreadList[G_ActiveThreads].ThreadName, "ClusterRecvPlayerItem");
				SetThreadPriority(hNewThread, THREAD_PRIORITY_NORMAL);
				ResumeThread(hNewThread);
			} else {
				printf("cluster_protocol.c:ClusterMiniServer:ClusterRecvPlayerItem:CREATE THREAD FAILURE\n");
			}
			break;
		case PKT_SERVER_LIST_REQ:
			hNewThread = CreateThread (NULL, 0, (LPTHREAD_START_ROUTINE) ClusterServerListResponse, InComing, CREATE_SUSPENDED, &lThreadId);
			if(hNewThread) {
				G_ActiveThreads++;
				if(G_ActiveThreads < 254)
					strcpy(ThreadList[G_ActiveThreads].ThreadName, "ClusterServerListResponse");
				SetThreadPriority(hNewThread, THREAD_PRIORITY_NORMAL);
				ResumeThread(hNewThread);
			} else {
				printf("cluster_protocol.c:ClusterMiniServer:ClusterServerListResponse:CREATE THREAD FAILURE\n");
			}
			break;

		case PKT_SERVER_LIST_RESP:
			hNewThread = CreateThread (NULL, 0, (LPTHREAD_START_ROUTINE) ClusterRecvListResponse, InComing, CREATE_SUSPENDED, &lThreadId);
			if(hNewThread) {
				G_ActiveThreads++;
				if(G_ActiveThreads < 254)
					strcpy(ThreadList[G_ActiveThreads].ThreadName, "ClusterRecvListResponse");
				SetThreadPriority(hNewThread, THREAD_PRIORITY_NORMAL);
				ResumeThread(hNewThread);
			} else {
				printf("cluster_protocol.c:ClusterMiniServer:ClusterRecvListResponse:CREATE THREAD FAILURE\n");
			}
			break;
		case PKT_BPRINTF:
			ClusterRecvBPrintf(InComing);
			break;
		default:
			break;
	}

}

/**********************************************************
	ClusterBytesToDDIP()

	Converts 4 unsigned chars representing each number
	in an address to a Dotted Decimal IP Address --used
	for packing and unpacking addresses

	bytes should contain 4 unsigned chars
	addr should point to an array of 17 chars

	I really don't like trusting the OS with this stuff.
**********************************************************/
void ClusterBytesToDDIP(unsigned char *bytes, char *addr) {

	sprintf(addr, "%d.%d.%d.%d", bytes[0], bytes[1], bytes[2], bytes[3]);
}
void ClusterDDIPToBytes(char *addr, unsigned char *bytes) {
	char *token;

	token = (char *)malloc(17);
	if(token == NULL) {
		bytes = NULL;
	} else {
		token = strtok(addr, ".");
		if(token != NULL) {
			bytes[0] = strtol (token, NULL, 10);
			token = strtok(NULL, ".");
			if(token != NULL) {
				bytes[1] = strtol (token, NULL, 10);
				token = strtok(NULL, ".");
				if(token != NULL) {
					bytes[2] = strtol(token, NULL, 10);
					token = strtok(NULL, ".");
					if(token != NULL) {
						bytes[3] = strtol(token, NULL, 10);
					} else {
						bytes = NULL;
					}
				} else {
					bytes = NULL;
				}
			} else {
				bytes = NULL;
			}
		} else {
			bytes = NULL;
		}
	}
	return;
}

/**********************************************************
	ClusterPackAddr(<IP Address>, <MiniServer Port>, <Q2 Port> <Return Buffer>)

	This takes an FQDN or Dotted Decimal IP Address and 2 unsigned
	shorts, then packs them into an 8 byte structure for use in
	packets that designate a cluster server.
**********************************************************/
void ClusterPackAddr(char *addr, unsigned short ms_port, unsigned short q2_port, ClusterPackedAddr *qaddr) {
	HOSTENT *heAddr;
	struct sockaddr_in saAddr;
	int i, err;
	char junk[17];
	err = 0;

	for(i=0;i<(int)strlen(addr); i++) {
		if(isalpha(addr[i])) {
			err = 1;
			break;
		}
	}
	
	if(err) {
		heAddr = gethostbyname(addr);
		if(heAddr == NULL) {
			qaddr = NULL;
			return;
		} else {
			memcpy(&saAddr.sin_addr, heAddr->h_addr, heAddr->h_length);
			strcpy(junk, inet_ntoa(saAddr.sin_addr));
			ClusterDDIPToBytes(junk, qaddr->addr);
		}
	}
	ClusterIntToBytes(ms_port, qaddr->ms_port);
	ClusterIntToBytes(q2_port, qaddr->q2_port);
}


void ClusterSendPlayer(edict_t *player, edict_t *exit) {
	// Send player data to destination
	// then send player to destination
	ClusterSendPlayerPacket Packet;
	int i;
	char cmd[64];


	Packet.packet_id = PKT_CLIENT_SET;
	ClusterIntToBytes(G_LocalComPort, Packet.ms_port);
	strcpy(Packet.netname, player->client->pers.netname);
	ClusterIntToBytes(player->health, Packet.health);
	ClusterIntToBytes(player->max_health, Packet.max_health);
	ClusterIntToBytes(player->client->resp.score, Packet.score);
	strcpy(Packet.weapon, player->client->pers.weapon->pickup_name);

	// Give the player his items on the other side of the link
	for(i=0;i<=MAX_ITEMS;i++) {
		if(player->client->pers.inventory[i]) {
			ClusterSendPlayerItem(exit->target, exit->remote_ms_port, player->client->pers.netname, i, player->client->pers.inventory[i]);
		}
	}
	// Send data
	ClusterSendMSG(exit->target, exit->remote_ms_port, (unsigned char *)&Packet, sizeof(ClusterSendPlayerPacket), G_ClusterKey);
	// Send Player
	sprintf(cmd, "set exitname %s u\n", exit->targetname);
	stuffcmd(player, cmd);
	sprintf(cmd, "connect %s:%d\n", exit->target, exit->remote_q2_port);
	stuffcmd(player, cmd);		
	return;
}

void ClusterRecvPlayer(RecvMSG *ParentData) {
	ClusterSendPlayerPacket Player;
	RecvMSG Packet;
	long ltime, timeout, loadtime;
	edict_t *ent;
	int ready;

	ready = 0;
	ent = NULL;
	memcpy(&Packet, ParentData, sizeof(RecvMSG));
	memcpy(&Player, &Packet.cData, sizeof(ClusterSendPlayerPacket));
	time(&timeout);

	while(!ready) {	// Loop until timeout or player arrival
		time(&ltime);
		while((ent = G_Find(ent, FOFS(classname), "player")) != NULL) {
			time(&ltime);
			if(stricmp(ent->client->pers.netname, Player.netname) == 0) {
				// Player is in the game now, set attributes and break from loop
				time(&loadtime);
				while(ltime < loadtime + 5) {
					ent->health = ClusterCharToINT(Player.health);
					if(ent->health < 0) 
						ent->health = 1;
					ent->max_health = ClusterCharToINT(Player.max_health);
					ent->client->resp.score = ClusterCharToINT(Player.score);
					if(stricmp(ent->client->pers.weapon->pickup_name, Player.weapon)) {
						ent->client->newweapon = FindItem(Player.weapon);
						ChangeWeapon (ent);
					}
					time(&ltime);
					if(ent->solid != SOLID_NOT) {
						ent->s.effects |= EF_BFG;
						ent->timeout = ltime + 5;
						ent->solid = SOLID_NOT;
						ent->takedamage = DAMAGE_NO;
					}
					_sleep(250);
				}
				ready = 1;
				ent->s.effects = 0;
				ent->solid = SOLID_BBOX;
				ent->takedamage = DAMAGE_AIM;

				break;
			}
			if(ltime - 60 > timeout) {
				ready = 1;
				break;
			}
		}
		_sleep(500);
	}
	G_ActiveThreads--;
	ExitThread(0);
}

void ClusterSendPlayerItem(char *dest_addr, unsigned short dest_msport, char *playername, int inv_index, int value) {
	ClusterSendItemPacket Item;
	int err;
	err = 0;
	if(inv_index == 0) {
		// There is a game bug with this for some reason, just return
		err = 1;
	}
	if(!err) {
		Item.packet_id = PKT_ITEM_SEND;
		Item.inv_index = inv_index;
		Item.inv_value = value;
		strcpy(Item.player_name, playername);
		ClusterSendMSG(dest_addr, dest_msport, (unsigned char *)&Item, sizeof(ClusterSendItemPacket), G_ClusterKey);
	}
	return;
}

void ClusterRecvPlayerItem(RecvMSG *ParentData) {
	ClusterSendItemPacket Item;
	RecvMSG Packet;
	long ltime, timeout;
	edict_t *ent;
	int ready, err;

	err = 0;
	ready = 0;
	ent = NULL;
	memcpy(&Packet, ParentData, sizeof(RecvMSG));
	memcpy(&Item, &Packet.cData, sizeof(ClusterSendItemPacket));
	if(Item.inv_index == 0) {
		// Bad, crashes server, don't process, just bail
		err = 1;
		
	}
	if(!err) {
		time(&timeout);
		timeout = timeout + 60;
		while(!ready) { // loop until timeout or player arrival
			time(&ltime);
			while((ent = G_Find(ent, FOFS(classname), "player")) != NULL) {
				if(stricmp(ent->client->pers.netname, Item.player_name) == 0) {
					ent->client->pers.inventory[Item.inv_index] = Item.inv_value;
					ready = 1;
					break;
				}
			}
			if(ltime > timeout) {
				ready = 1;
				break;
			}
			_sleep(500);
		}		
	} else {
	}
	G_ActiveThreads--;
	ExitThread(0);
}

/**********************************************************
	The ClusterServerListxxx functions are my early evil
	attempts to maintain a list of servers participating
	in the cluster. This **WILL** be replaced with a more
	efficient Link State Advertisement protocol modelled 
	after OSPF. I *HATE* that I am flooding for this 
	information right now. I also don't want to hold up
	testing to check for other bugs before I really go
	to town on the API.
**********************************************************/
void ClusterServerListRequest(void) {
	ClusterServerListReqPacket	Request;
	char	addr[64];
	edict_t	*peer;

	G_LastRequestID++;
	if(G_LastRequestID > 254)
		G_LastRequestID = 1;
	peer = NULL;
	Request.packet_id = PKT_SERVER_LIST_REQ;
	Request.request_id = G_LastRequestID;
	strcpy(addr, inet_ntoa(G_ServerAddr));
	ClusterDDIPToBytes(addr, Request.addr);
	ClusterIntToBytes(G_ServerPort, Request.q2_port);
	ClusterIntToBytes(G_LocalComPort, Request.ms_port);
	// OK, the packet is built, now send it off to our peers
	while((peer = G_Find(peer, FOFS(classname), "cluster_exit")) != NULL) {	
		// we have a peer
		if(peer->target) {
			ClusterSendMSG(peer->target, peer->remote_ms_port, (unsigned char *)&Request, sizeof(ClusterServerListReqPacket), G_ClusterKey);
		}
	}
}

void ClusterServerListResponse(RecvMSG *Packet) {
	ClusterServerListRespPacket	Response;
	ClusterServerListReqPacket Request;
	char addr[17];
	char req_addr[17];
	char que_addr[17];
	int i, err;
	unsigned short port;
	edict_t *peer;

	peer = NULL;
	err = 0;
	memcpy(&Request, Packet->cData, sizeof(ClusterServerListReqPacket));
	strcpy(addr, inet_ntoa(G_ServerAddr));
	ClusterBytesToDDIP(Request.addr, req_addr);
	// Check to see if we have alread responded to this request
	for(i=0;i<CLUSTER_MAX_REQ_QUEUE;i++) {
		ClusterBytesToDDIP(G_ServerListHistory[i].addr, que_addr);
		if(G_ServerListHistory[i].request_id == Request.request_id && !stricmp(req_addr, que_addr) && ClusterCharToUS(Request.ms_port) == ClusterCharToUS(G_ServerListHistory[i].ms_port)) {
			// We have alread processed this
			err = 1;
		}
	}
	if(!err) {
		G_QueueIndex ++;
		if(G_QueueIndex >= CLUSTER_MAX_REQ_QUEUE)
			G_QueueIndex = 0;
		Response.packet_id = PKT_SERVER_LIST_RESP;
		ClusterDDIPToBytes(addr, Response.addr);
		ClusterIntToBytes(G_ServerPort, Response.q2_port);
		ClusterIntToBytes(G_LocalComPort, Response.ms_port);
		Response.clients = level.players;
		// OK, Packet is built, lets respond
		ClusterBytesToDDIP(Request.addr, addr);
		port = ClusterCharToUS(Request.ms_port);
		ClusterSendMSG(addr, port, (unsigned char *)&Response, sizeof(ClusterServerListRespPacket), G_ClusterKey);
		G_ServerListHistory[G_QueueIndex] = Request;
		// Now forward the request to my peers
		while((peer = G_Find(peer, FOFS(classname), "cluster_exit")) != NULL) {	
		// we have a peer
			if(peer->target) {
				ClusterSendMSG(peer->target, peer->remote_ms_port, (unsigned char *)&Request, sizeof(ClusterServerListReqPacket), G_ClusterKey);
			}
		}
	
	} 
	G_ActiveThreads--;
	ExitThread(0);
}

void ClusterRecvListResponse(RecvMSG *Packet) {
	ClusterServerListRecord Server;
	ClusterServerListRespPacket Response;
	int i, idx_tag_zero, idx_update;
	
	i = 0;
	idx_tag_zero = -1;
	idx_update = -1;

	memcpy(&Response, Packet->cData, sizeof(ClusterServerListRespPacket));
	Server.addr[0] = Response.addr[0];
	Server.addr[1] = Response.addr[1];
	Server.addr[2] = Response.addr[2];
	Server.addr[3] = Response.addr[3];
	Server.ms_port[0] = Response.ms_port[0];	
	Server.ms_port[1] = Response.ms_port[1];	
	Server.q2_port[0] = Response.q2_port[0];
	Server.q2_port[1] = Response.q2_port[1];
	Server.clients = Response.clients;

	for(i=0; i < CLUSTER_MAX_SERVERS; i++) {
		if(idx_tag_zero == -1 && G_ServerList[i].q2_port[0] == 0 && G_ServerList[i].q2_port[1] == 0) {
			idx_tag_zero = i;
		}
		if(!memcmp(&G_ServerList[i].addr, &Server.addr, 4) && !memcmp(&G_ServerList[i].q2_port, Server.q2_port, 2)) {
			idx_update = i;
			break;
		}
	}
	
	if(idx_update > -1) {
		// We have an update to make
		memcpy(&G_ServerList[i], &Server, sizeof(ClusterServerListRecord));
	} else {
		if(idx_tag_zero > -1) {
			// We are adding a new server, use the first blank in the array
			memcpy(&G_ServerList[idx_tag_zero], &Server, sizeof(ClusterServerListRecord));
		}
	}
	G_ActiveThreads--;
	ExitThread(0);
}

void ClusterSendBPrintf(char *message) {
	int i;
	char	addr[17];
	unsigned short	msport;
	ClusterBPrintfPacket	Packet;

	Packet.packet_id = PKT_BPRINTF;
	strcpy(Packet.message, message);

	for(i = 0; i < CLUSTER_MAX_SERVERS; i++) {
		if(ClusterCharToUS(G_ServerList[i].ms_port) > 0) {
			// Valid Q2 Server
			ClusterBytesToDDIP(G_ServerList[i].addr, addr);
			msport = ClusterCharToUS(G_ServerList[i].ms_port);
			ClusterSendMSG(addr, msport, (unsigned char *)&Packet, sizeof(ClusterBPrintfPacket), G_ClusterKey);
		}
	}
}

void ClusterRecvBPrintf(RecvMSG *Packet) {
	ClusterBPrintfPacket	Message;
	
	memcpy(&Message, Packet->cData, sizeof(ClusterBPrintfPacket));
	printf("Received message %s\n", Message.message);
	gi.bprintf(PRINT_CHAT, "%s\n", Message.message);
}
