/**********************************************************
Quake2 Cluster Project
An addition and modification of iD Software's Quake2 shared 
library sources which enable Quake2 servers running this 
library to interconnect other servers which speak the 
protocol developed under the Quake2 Cluster Project.
Copyright (c) 1998 Justin Randall and Todd Bliss

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; version 2
of the License.

This program is distributed in the hope that it will be 
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more 
details. You should have received a copy of the GNU General 
Public License along with this program; if not, write to 
the Free Software Foundation, Inc., 
59 Temple Place - Suite 330, 
Boston, MA  02111-1307, USA.
**********************************************************/


/**********************************************************
	rules.c

	Callback to cluster and id source for various game
	rules. This is in place to eventually support a dll
	abstraction, hopefully permitting additive mods for
	a cluster --e.g. plug-in dll's for a server.

  If you are developing a new mod, and need to perform
  some function that is not supported here, PLEASE
  contact logic@logic.columbus.oh.us so we can work together
  to build a useable API. 

	$Log: rules.c,v $
	Revision 1.3  1998/02/01 16:51:49  logic
	Moved RULES_SetInvinc to RULES_PutClientInServer2

	Revision 1.2  1998/02/01 15:07:21  logic
	Release 0.3

	Revision 1.1  1998/01/31 09:54:01  logic
	Added RULES_ClientUserInfoChanged for access to client variables.
	Fixed the password bug
	Fixed the looping respawn/telefrag bug
	Added Invincibility functions to RULES
	Added ClientThink2 to RULES
	Added RULES_ClientCmd
	Added cmd invincible to RULES
	Added ()'s around player names on broadcast messages
	Added MOTD motd-mapname.txt
	Changed invincibility fx

**********************************************************/
#include "udp.h"
#include "windows.h"
#include <stdlib.h>
#include "g_local.h"
#include "cluster_globs.h"


/**********************************************************
RULES_ClientUserInfoChanged

  This is called whenever a client changes one of his
  vars (e.g. "password mynewpassword")

  It is also called when the client connects to the server,
  so you can save some data between server moves using
  StuffCMD("set something value") on the entity. It will
  be reloaded via this function.
**********************************************************/
void RULES_ClientUserinfoChanged (edict_t *ent, char *userinfo) {
	char	*s;

	s = Info_ValueForKey(userinfo, "password");
	if(strlen(s)) {
		strcpy(ent->client->pers.password, s);
	}
	s = Info_ValueForKey(userinfo, "exitname");
	if(strlen(s)) {
		strcpy(ent->client->pers.exitname, s);
	}
}

/**********************************************************
RULES_CheckInvincibility(edict_t *invincible_entity)

  If RULE_RESPAWN_INVINC_TIME + ltime is greater than
  ent->timeout (set on respawn), then stop
  invincibility.
**********************************************************/
int RULES_CheckInvincibility(edict_t *ent) {
	long ltime;
	int retval;

	retval = 0;
	time(&ltime);
	if (ent->rules.RespawnInvTimeOut > ltime || ent->rules.RespawnInvTimeOut == -1) {
		if(ent->takedamage != DAMAGE_NO) {
			if(ent->solid != SOLID_NOT) {
				if(ent->rules.RespawnInvTimeOut == -1) {
					ent->s.renderfx = RF_BEAM|RF_TRANSLUCENT;
				} else {
					ent->s.effects |= EF_COLOR_SHELL;
					ent->s.renderfx |= (RF_SHELL_RED|RF_SHELL_GREEN|RF_SHELL_BLUE);
				}
				ent->solid = SOLID_NOT;
				ent->takedamage = DAMAGE_NO;
			}
		}
		retval = 1;
	} else {
		if(ent->takedamage == DAMAGE_NO && ent->deadflag == DEAD_NO) {
			ent->s.effects = 0;
			ent->s.renderfx = 0;
			ent->solid = SOLID_BBOX;
			ent->takedamage = DAMAGE_AIM;
			retval = 0;
		}
	}
	return retval;
}

/**********************************************************
RULES_SetInvincibility

  Changes the invincibility value of a player
  When called as RULES_SetInvincibility(client, -1)
  it will make the client invincible until it respawns
**********************************************************/
void RULES_SetInvincibility(edict_t *client, int seconds) {
	long timeout;

	time(&timeout);

	if(seconds > 0) {
		client->rules.RespawnInvTimeOut = timeout + seconds;
	} else {
		client->rules.RespawnInvTimeOut = seconds;
	}
}

/**********************************************************
RULES_ClientThink2

  Called at the end of id's ClientThink() function (in 
  p_client.c) 

  edict_t *ent is the client that is thinking
**********************************************************/
void RULES_ClientThink2(edict_t *ent) {
	long ltime;

	time(&ltime);
	// Check to see if client should be invincible
	RULES_CheckInvincibility(ent);

	// Display MOTD if not timed out
	if(ent->rules.motd_time > ltime && ent->rules.motd_msg != NULL && ent->rules.motd_lasttime < ltime - 1) {
		gi.centerprintf(ent, ent->rules.motd_msg);
		ent->rules.motd_lasttime = ltime;
	}


};


void RULES_CmdInvinc(edict_t *ent) {
	int breakval, timeopt;
	edict_t *target;
	
	target = NULL;
	breakval = 0;
	if(!ValidateAdmin(ent)) {
		gi.cprintf(ent, PRINT_HIGH, "You are not authorized to make players invincible\n");
		return;
	}
	if(gi.argc() > 1) {
		if(!stricmp(gi.argv(1), "all")) {
			// setting invinc for all
			while((target = G_Find(target, FOFS(classname), "player")) != NULL && !breakval) {
				if(gi.argc() > 2) {
					// Also a timelimit
					timeopt = atoi(gi.argv(2));
					if(timeopt > 0) {
						gi.cprintf(ent, PRINT_HIGH, "%s is invincible for %d seconds\n", target->client->pers.netname, timeopt);
						gi.cprintf(target, PRINT_HIGH, "You have received %d seconds of invincibility from %s\n", timeopt, ent->client->pers.netname);	
					} else if(timeopt == 0 && !target->takedamage) {
						gi.cprintf(ent, PRINT_HIGH, "%s is mortal again\n", target->client->pers.netname);
						gi.cprintf(target, PRINT_HIGH, "%s has made you are mortal again\n", ent->client->pers.netname);
					}
				} else {
					timeopt = -1;
					if(!target->takedamage) {
						gi.cprintf(ent, PRINT_HIGH, "%s is permanently invincible\n", target->client->pers.netname);
						gi.cprintf(target, PRINT_HIGH, "%s grants you permanent invincibility\n", ent->client->pers.netname);
					}
						
				}
				target->s.renderfx = 0;
				target->s.effects = 0;
				RULES_SetInvincibility(target, timeopt);
			}
		} else {
			while((target = G_Find(target, FOFS(classname), "player")) != NULL && !breakval) {
				if(!stricmp(target->client->pers.netname, gi.argv(1))) {
					// We have a player
					if(gi.argc() > 2) {
						// Also a timelimit
						timeopt = atoi(gi.argv(2));
						if(timeopt > 0) {
							gi.cprintf(ent, PRINT_HIGH, "%s is invincible for %d seconds\n", target->client->pers.netname, timeopt);
							gi.cprintf(target, PRINT_HIGH, "%s grants you invincibility for %d seconds\n", ent->client->pers.netname, timeopt);
						} else if(timeopt == 0) {
							gi.cprintf(ent, PRINT_HIGH, "%s is mortal again\n", target->client->pers.netname);
							gi.cprintf(target, PRINT_HIGH, "%s returns your mortality\n", ent->client->pers.netname);
						}
					} else {
						timeopt = -1;
						gi.cprintf(ent, PRINT_HIGH, "%s is permanently invincible\n", target->client->pers.netname);
						gi.cprintf(target, PRINT_HIGH, "%s has granted you permanent immortality\n", ent->client->pers.netname);
					}
					target->s.effects = 0;
					target->s.renderfx = 0;
					RULES_SetInvincibility(target, timeopt);
					
					breakval = 1;
				}
			}	
		}
	}
}

/**********************************************************
RULES_ClientCMD

  Called when a client enters cmd <command> [<options>]
**********************************************************/
int RULES_ClientCMD(char *cmd, edict_t *ent) {
	int retval;
	retval = 0;
	if(!Q_stricmp(cmd, "invinc")) {
		RULES_CmdInvinc(ent);
		retval = 1;
	}
	return retval;

}

void RULES_CmdHelp(edict_t *ent) {
	int retval;

	retval = 1;
	if(!ValidateAdmin(ent)) {
		retval = 0;
	}

	if(retval == 1) {
		gi.cprintf(ent, PRINT_HIGH, "cmd invinc <player name | all> [<seconds>]\n");
	}
}

void RULES_MOTD(edict_t *ent) {
	char filename[64] = {"\0"};
	FILE *motd;
	char map[32] = {"\0"};
	char buf[1024] = {"\0"};
	char msg[1024] = {"\0"};
	int err;
	long bytesread, ltime;

	err = 0;
	bytesread = 0;
	if(ent->rules.motd_time > 0)
		err = 1;
	else
		err = 0;

	if(level.mapname != NULL) {
		strcpy(map, level.mapname);
	} else {
		err = 1;
	}
	if(gi.cvar("gamedir", NULL, 0) != NULL && !err) {
		sprintf(filename, "%s/motd-%s.txt", gi.cvar("gamedir", NULL, 0)->string, map);
	} else {
		err = 1;
	}
	if(!err)
		motd = fopen(filename, "r");
	if(motd != NULL && !err) {
		while(!feof(motd)) {
			fgets(buf, 80, motd);
			if(buf != NULL && !feof(motd)) {
				bytesread += strlen(buf);
				if(bytesread < 1024) {	
					strcat(msg, buf);
				}
			}
		}
		fclose(motd);
		strcpy(ent->rules.motd_msg, msg);
		time(&ltime);
		ent->rules.motd_time = ltime + MOTD_TIME;
	}
	return;
}

/**********************************************************
RULES_PutClientInServer2

  One of the last things run after id's own PutClientInServer
  is run.

  Extend it with whatever you like :)
**********************************************************/
void RULES_PutClientInServer2(edict_t *ent) {
	// Cluster MOD: Set Invincibility
	RULES_SetInvincibility(ent, RULE_RESPAWN_INVINC_TIME);

	RULES_MOTD(ent);
}

/**********************************************************
RULES_ClientBegin

  Called when the client connects to the server the first
  time. MOTD lives here. 
***********************************************************/
void RULES_ClientBegin(edict_t *ent) {
}
void RULES_Cmd_Kill_f(edict_t *ent) {
	long ltime;

	time(&ltime);	// Get current system time
	if(ent->rules.killtime > ltime) {
		ent->rules.killcount++;
		if(ent->rules.killcount > 3) {
			// We have a problem user here, kick him off and let him spam
			// his own PC
			stuffcmd(ent, "quit\n");
			gi.bprintf(PRINT_CHAT, "%s left the fray to play with himself.\n", ent->client->pers.netname);
			
		} else {
			// OK, MAYBE this guy really isn't spamming...
			// Just send a warning
			gi.cprintf(ent, PRINT_HIGH, "Don't spam the server, wait a few moments before suiciding again\n");
			ent->rules.killtime = ltime + 3;	// Reset the timer on him
		}
	} else {
		// First time suiciding, just do the normal KILL cmd
		ent->flags &= ~FL_GODMODE;
		ent->health = 0;
		ent->rules.killtime = ltime + 3;	// Reset the timer on him
		ent->rules.killcount = 0;		// reset killcount
		player_die (ent, ent, ent, 100000, vec3_origin);
		// don't even bother waiting for death frames
		ent->deadflag = DEAD_DEAD;
	}
	return;
}

/**********************************************************
RULES_CmdClusterSay

  CmdClusterSay will check the RULES before sendint a 
  broadcast message. If it returns true, then message is
  OK. If false, it doesn't transmit.

  This example thwarts broadcast spam messages
**********************************************************/
int RULES_CmdClusterSay(edict_t *ent) {
	long ltime;
	int	retval;
	retval = 0;	// default to false..

	time(&ltime);
	if(ent->rules.last_cluster_say > ltime) {
		// don't send message
		gi.cprintf(ent, PRINT_HIGH, "Please, don't spam the cluster. Messages take bandwidth you know.\n");
		retval = 0;
	} else {
		retval = 1;
	}
	ent->rules.last_cluster_say = ltime + 2;
	return retval;
}