/*

    GAF2TEX.CPP -- Dump contents of a GAF texture archive used by Total Annihilation

    Thanks!! to Saruman for providing the initial decoding info for the .GAF files!

    Version 0.90  April 16, 1998 (c) Chris Wallace

    Future enhancements:
      - output compressed images
      - greater flexibility in specifying input/output files and paths
          
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <windows.h>

// These structures represent the physical, unprocessed version of the GAF file

typedef struct tagGafHeader {
  long iVersion;
  long nEntries;
  long iUnknown;
} GafHeader;

typedef struct tagGafInfo {
  short nFrames;
  short iUnknown1;
  long  iUnknown2;
  char  sName[32];
  long  iOffsetFrames;
  long  iUnknown3;
} GafInfo;

typedef struct tagGafFrameHeader {
  short  width;
  short  height;
  short  xOffset;
  short  yOffset;
  long   iUnknown1;
  long   iUnknown2;
  long   iOffsetBmp;  
} GafFrameHeader;


// These structures represent the final, massaged, software-friendly version of the GAF file

typedef struct tagGafFrame {
  short  width;
  short  height;
  short  xOffset;
  short  yOffset;
  unsigned char  *pImage;
} GafFrame;

typedef struct tagGafEntry {
  short     nFrames;
  char      sName[32];
  GafFrame  *pGafFrame;
} GafEntry;

typedef struct tagGaf {
  char     szGafFilename[128];
  long     iVersion;
  long     nEntries;
  GafEntry  *pGafEntry;
} Gaf;


void Read_Gaf (char *szFilename, Gaf *pGaf)
{
  GafHeader       header;
  GafInfo         info;
  GafFrameHeader  frame;
  long           iOffsetInfo;
  GafEntry       *pEntry;
  GafFrame       *pFrame;
  
  // Open file
  FILE *fGaf = fopen (szFilename, "rb");

  if (!fGaf) {
    fprintf (stderr, "ERROR reading %s\n\n", szFilename);
    exit (-1);
  }
  
  // Read header
  fread (&header, sizeof(GafHeader), 1, fGaf);

  //strcpy (pGaf->szGafFilename, szFilename);
  strcpy (pGaf->szGafFilename, "test");
  pGaf->iVersion = header.iVersion;
  pGaf->nEntries = header.nEntries;
  
  // Allocate entries and fill in header info for each entry
  pGaf->pGafEntry = (GafEntry *) calloc (sizeof(GafEntry), pGaf->nEntries);

  // Read in the table of offset pointers
  long *pOffsetInfo = (long *) calloc (sizeof(long), pGaf->nEntries);
  fread (pOffsetInfo, sizeof(long), pGaf->nEntries, fGaf);
      
  // For each entry, read the info structure
  for (int iEntry=0; iEntry<pGaf->nEntries; iEntry++) {
    // Find the index of the GafInfo record for this entry
    iOffsetInfo = pOffsetInfo[iEntry];
        
    // Read the GafInfo record and propagate the fields
    fseek (fGaf, iOffsetInfo, SEEK_SET);
    fread (&info, sizeof(GafInfo), 1, fGaf);

    pEntry = &(pGaf->pGafEntry[iEntry]);
    pEntry->nFrames = info.nFrames;
    strcpy (pEntry->sName, info.sName);

    // Allocate storage for the frames
    pEntry->pGafFrame = (GafFrame *) calloc (sizeof(GafFrame), pEntry->nFrames);

    // For each frame, load the relevant info
    /**** NOTE -- Currently only frame 0 is supported.  Multiple animation frames will be ignored. */
    fseek (fGaf, info.iOffsetFrames, SEEK_SET);
    fread (&frame, sizeof(GafFrameHeader), 1, fGaf);
    pFrame = &(pEntry->pGafFrame[0]);
    pFrame->width = frame.width;
    pFrame->height = frame.height;
    pFrame->xOffset = frame.xOffset;
    pFrame->yOffset = frame.yOffset;
    int iBitmapSize = frame.width * frame.height;
    pFrame->pImage = (unsigned char *) malloc (iBitmapSize);
    // Read the image bitmap
    fseek (fGaf, frame.iOffsetBmp, SEEK_SET);
    fread (pFrame->pImage, sizeof(unsigned char), iBitmapSize, fGaf);
  }

  fclose (fGaf);
}


void Print_Gaf (FILE *fDest, Gaf *pGaf)
{
  GafEntry  *pEntry;
  GafFrame  *pFrame;
    
  fprintf (fDest, ".GAF File : %s\n", pGaf->szGafFilename);
  fprintf (fDest, "nEntries  : %d\n", pGaf->nEntries);

  for (int iEntry=0; iEntry<pGaf->nEntries; iEntry++) {
    pEntry = &(pGaf->pGafEntry[iEntry]);
    fprintf (fDest, "  Entry (%03d) : %s\n", iEntry, pEntry->sName);
    fprintf (fDest, "    nFrames : %d\n", pEntry->nFrames);
    for (int iFrame=0; iFrame<pEntry->nFrames; iFrame++) {
      pFrame = &(pEntry->pGafFrame[iFrame]);
      fprintf (fDest, "      Frame (%03d)\n", iFrame);
      fprintf (fDest, "        Width  : %d\n", pFrame->width);
      fprintf (fDest, "        Height : %d\n", pFrame->height);
      fprintf (fDest, "        xOffset  : %d\n", pFrame->xOffset);
      fprintf (fDest, "        yOffset  : %d\n", pFrame->yOffset);
    }
  }
}


char pTaPalette[256][3] = {
  {0, 0, 0},
  {128, 0, 0},
  {0, 128, 0},
  {128, 128, 0},
  {0, 0, 128},
  {128, 0, 128},
  {0, 128, 128},
  {192, 192, 192},
  {79, 101, 125},
  {131, 153, 177},
  {0, 0, 0},
  {0, 0, 0},
  {0, 0, 0},
  {0, 0, 0},
  {0, 0, 0},
  {0, 0, 0},
  {255, 235, 243},
  {235, 199, 211},
  {215, 163, 179},
  {195, 135, 151},
  {175, 111, 127},
  {155, 91, 99},
  {139, 71, 79},
  {123, 59, 71},
  {111, 51, 59},
  {99, 43, 51},
  {87, 35, 43},
  {75, 27, 39},
  {59, 23, 31},
  {47, 15, 23},
  {35, 11, 15},
  {23, 7, 11},
  {115, 255, 223},
  {87, 231, 191},
  {67, 207, 159},
  {47, 183, 131},
  {31, 159, 103},
  {19, 139, 79},
  {15, 119, 63},
  {11, 107, 55},
  {7, 95, 47},
  {7, 83, 43},
  {0, 71, 39},
  {0, 63, 35},
  {0, 51, 27},
  {0, 39, 23},
  {0, 27, 15},
  {0, 19, 11},
  {227, 239, 255},
  {199, 223, 231},
  {175, 207, 203},
  {147, 183, 167},
  {127, 159, 131},
  {107, 135, 103},
  {95, 111, 83},
  {95, 99, 71},
  {91, 87, 59},
  {83, 67, 51},
  {71, 59, 43},
  {59, 51, 35},
  {47, 43, 27},
  {35, 31, 19},
  {23 ,19, 15},
  {11, 11, 7},
  {251, 251, 215},
  {223, 223, 183},
  {195, 195, 155},
  {171, 171, 131},
  {147, 147, 111},
  {119, 119, 87},
  {99, 99, 67},
  {83, 83, 51},
  {67, 67 ,35},
  {51, 51, 23},
  {35, 35, 15},
  {27, 27, 7},
  {23, 23, 7},
  {19, 19, 0},
  {15, 15, 0},
  {11, 11, 0},
  {251, 251, 251},
  {235, 235, 235},
  {219, 219, 219},
  {203, 203, 203},
  {187, 187, 187},
  {171, 171, 171},
  {155, 155, 155},
  {139, 139, 139},
  {123, 123, 123},
  {107, 107, 107},
  {91, 91, 91},
  {75, 75, 75},
  {59, 59, 59},
  {43, 43, 43},
  {31, 31, 31},
  {15, 15, 15},
  {235, 243, 255},
  {203, 227, 255},
  {175, 207, 255},
  {151, 179, 255},
  {123, 151, 255},
  {103, 127, 255},
  {83, 107, 239},
  {63, 91, 227},
  {51, 75, 215},
  {35, 59, 203},
  {23, 47, 175},
  {15, 39, 151},
  {7, 31, 123},
  {7, 23, 99},
  {0, 15, 71},
  {0, 11, 47},
  {227, 247, 255},
  {191, 219, 231},
  {159, 191, 207},
  {131, 167, 183},
  {107, 143, 163},
  {83, 119, 139},
  {63, 95, 115},
  {47, 75, 95},
  {39, 63, 87},
  {35, 55, 79},
  {31, 47, 71},
  {27, 39, 63},
  {23, 31, 55},
  {19, 27, 47},
  {15, 19, 39},
  {11, 15, 31},
  {215, 239, 255},
  {187, 227, 239},
  {155, 203, 223},
  {131, 183, 207},
  {107, 163, 195},
  {83, 143, 179},
  {63, 123, 163},
  {47, 107, 151},
  {35, 91, 135},
  {27, 75, 119},
  {19, 63, 103},
  {11, 51, 87},
  {7, 39, 71},
  {0, 27, 55},
  {0, 19, 39},
  {0, 11, 27},
  {255, 231, 255},
  {231, 199, 235},
  {211, 171, 215},
  {187, 147, 195},
  {167, 123, 179},
  {143, 99, 159},
  {119, 75, 143},
  {99, 59, 127},
  {79, 43, 111},
  {67, 31, 99},
  {55, 23, 87},
  {43, 15, 71},
  {31, 7, 59},
  {19, 0, 43},
  {11, 0, 31},
  {7, 0, 19},
  {215, 255, 167},
  {171, 231, 127},
  {131, 211, 91},
  {103, 191, 63},
  {75, 171, 43},
  {67, 151, 43},
  {55, 135, 39},
  {47, 119, 27},
  {43, 103, 19},
  {35, 91, 15},
  {31, 79, 11},
  {27, 67, 7},
  {23, 51, 0},
  {15, 39, 0},
  {11, 27, 0},
  {7, 15, 0},
  {255, 227, 159},
  {227, 199, 115},
  {203, 175, 83},
  {179, 151, 63},
  {155, 131, 47},
  {131, 111, 35},
  {107, 91, 23},
  {83, 71, 15},
  {75, 59, 11},
  {67, 51, 7},
  {59, 43, 7},
  {55, 35, 0},
  {47, 27, 0},
  {39, 19, 0},
  {31, 15, 0},
  {27, 11, 0},
  {255, 255, 163},
  {251, 243, 131},
  {247, 227, 103},
  {243, 211, 79},
  {239, 187, 51},
  {239, 167, 27},
  {235, 143, 19},
  {231, 123, 15},
  {223, 79, 7},
  {215, 35, 0},
  {191, 31, 0},
  {167, 27, 0},
  {147, 23, 0},
  {123, 19, 0},
  {99, 19, 0},
  {79, 15, 0},
  {255, 255, 0},
  {255, 191, 0},
  {255, 131, 0},
  {255, 71, 0},
  {211, 43, 0},
  {171, 23, 0},
  {127, 7, 0},
  {87, 0, 0},
  {223, 203, 255},
  {187, 159, 223},
  {155, 119, 191},
  {127, 87, 159},
  {103, 59, 127},
  {75, 35, 95},
  {51, 19, 63},
  {27, 7, 31},
  {211, 219, 255},
  {135, 159, 247},
  {67, 111, 239},
  {23, 71, 231},
  {11, 43, 187},
  {7, 23, 143},
  {0, 7, 99},
  {0, 0, 55},
  {123, 255, 119},
  {83, 223, 79},
  {51, 191, 43},
  {27, 159, 19},
  {27, 127, 11},
  {23, 95, 7},
  {19, 63, 0},
  {11, 31, 0},
  {0, 0, 0},
  {0, 0, 0},
  {0, 0, 0},
  {0, 0, 0},
  {0, 0, 0},
  {0, 0, 0},
  {193, 204, 217},
  {160, 160, 164},
  {128, 128, 128},
  {255, 0, 0},
  {0, 255, 0},
  {255, 255, 0},
  {0, 0, 255},
  {255, 0, 255},
  {0, 255, 255},
  {255, 255, 255}
};


void Bmp_Frame (char *szBmpFilename, GafFrame *pFrame) {

  BITMAPFILEHEADER  bmfh;
  BITMAPINFOHEADER  bmih;

  int iBytesPerRow = (pFrame->width * 3);
  if ((iBytesPerRow % 4) != 0) {
    iBytesPerRow += 4 - (iBytesPerRow % 4);
  }
  int iImageSize = pFrame->height * iBytesPerRow;

  // Initialize bitmap file header structure
  bmfh.bfType      =  0x4d42;     //  'BM'
  bmfh.bfSize      =  sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER) + iImageSize;
  bmfh.bfReserved1 =  0;
  bmfh.bfReserved2 =  0;
  bmfh.bfOffBits   =  sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER);

  // Initialize the bitmap info header structure
  bmih.biSize = sizeof (BITMAPINFOHEADER);
  bmih.biWidth = pFrame->width;
  bmih.biHeight = pFrame->height;
  bmih.biPlanes = 1;
  bmih.biBitCount =  24;               // 24-bit image
  bmih.biCompression = BI_RGB;         // No compression
  bmih.biSizeImage = iImageSize;
  bmih.biXPelsPerMeter = 0;
  bmih.biYPelsPerMeter = 0;
  bmih.biClrUsed = 0;                  // Not required for 24-bit image
  bmih.biClrImportant = 0;             // Not required for 24-bit image
  
  FILE *fBmp = fopen (szBmpFilename, "wb");

  // Output BMP header
  fwrite (&bmfh, sizeof(BITMAPFILEHEADER), 1, fBmp);
  fwrite (&bmih, sizeof(BITMAPINFOHEADER), 1, fBmp);

  // Output bitmap information.

  int nRows = pFrame->height;
  int nCols = pFrame->width;
  for (int iRow=(nRows-1); iRow>=0; iRow--) {
    int iRowBytes = 0;
    for (int iCol=0; iCol<nCols; iCol++) {

      // Output palette index as R, G and B values
      char iIndex = pFrame->pImage[(iRow * nCols) + iCol];
      char iRed = pTaPalette[iIndex][0];
      char iGreen = pTaPalette[iIndex][1];
      char iBlue = pTaPalette[iIndex][2];
      fwrite (&iBlue, 1, 1, fBmp);
      fwrite (&iGreen, 1, 1, fBmp);
      fwrite (&iRed, 1, 1, fBmp);

      iRowBytes += 3;
    }

    // Pad the row to a longword boundary
    char iZero = 0;
    while ((iRowBytes % 4) != 0) {
      fwrite (&iZero, 1, 1, fBmp);
      iRowBytes++;
    }    
  }
 
  fclose (fBmp);
}

typedef struct tagTgaHeader {
  short  iVersionMinor;
  short  iVersionMajor;
  long   iUnknown1;
  long   iUnknown2;
  short  iWidth;
  short  iHeight;
  short  iDepth;
} TgaHeader;

  
void Tga_Frame (char *szTgaFilename, GafFrame *pFrame) {

  TgaHeader  tgah;
  
  // Initialize bitmap file header structure
  tgah.iVersionMinor  = 0;
  tgah.iVersionMajor  = 2;
  tgah.iUnknown1      = 0;
  tgah.iUnknown2      = 0;
  tgah.iWidth         = pFrame->width;
  tgah.iHeight        = pFrame->height;
  tgah.iDepth         = 24;

  FILE *fTga = fopen (szTgaFilename, "wb");

  // Output TGA header
  fwrite (&tgah, sizeof(TgaHeader), 1, fTga);

  // Output bitmap information.

  int nRows = pFrame->height;
  int nCols = pFrame->width;
  for (int iRow=(nRows-1); iRow>=0; iRow--) {
    for (int iCol=1; iCol<=nCols; iCol++) {

      // Output palette index as R, G and B values
      char iIndex = pFrame->pImage[(iRow * nCols) + iCol];
      char iRed = pTaPalette[iIndex][0];
      char iGreen = pTaPalette[iIndex][1];
      char iBlue = pTaPalette[iIndex][2];
      fwrite (&iRed, 1, 1, fTga);
      fwrite (&iBlue, 1, 1, fTga);
      fwrite (&iGreen, 1, 1, fTga);
    }
  }
  
  fclose (fTga);
}


void print_help (void) {
  fprintf (stdout, "GAF2TEX -- Convert Total Annihilation .GAF files into individual texture images\n");
  fprintf (stdout, "Version 0.90 (c) 1998 Chris Wallace\n");
  fprintf (stdout, "Syntax :   gaf2tex [-b] <filename.gaf>\n");
  fprintf (stdout, "                    -b will generate .BMP files instead of the default .TGA\n\n");
  exit (0);
}


#define   OUTPUT_TGA    0
#define   OUTPUT_BMP    1


int main (int argc, char **argv)
{
  char szGafFilename[128];
  int  iOutputFormat = OUTPUT_TGA;
 
  // Parse arguments
  if (argc == 2) {
    // Only argument was the .gaf filename
    strcpy (szGafFilename, argv[1]);
    
  } else if (argc == 3) {
    // First argument should be -b switch, second argument the .gaf filename
    if (strcmp(argv[1], "-b") != 0) {
      print_help ();
    } else {
      iOutputFormat = OUTPUT_BMP;
      strcpy (szGafFilename, argv[2]);
    }
  } else {
    // Invalid number of arguments
    print_help();
  }

  Gaf  theGaf;
  Read_Gaf (szGafFilename, &theGaf);

  // For each entry, generate a BMP file of the texture.
  for (int iEntry =0; iEntry<theGaf.nEntries; iEntry++) {
    GafEntry  *pEntry = &(theGaf.pGafEntry[iEntry]);
    char szOutFilename[128];
    strcpy (szOutFilename, pEntry->sName);

    switch (iOutputFormat) {
      case OUTPUT_TGA:
        strcat (szOutFilename, ".tga");
        Tga_Frame (szOutFilename, pEntry->pGafFrame);
        break;
        
      case OUTPUT_BMP:
        strcat (szOutFilename, ".bmp");
        Bmp_Frame (szOutFilename, pEntry->pGafFrame);
        break;
    }
  }
  return 0;
}

