/*
=============================================================================
Module Information
------------------
Name:			obj_barrett.cpp
Author:			Rich Whitehouse
Description:	server logic object: barrett
=============================================================================
*/

#include "main.h"
#include "ai.h"

typedef enum
{
	BARANIM_SHOOT = NUM_HUMAN_ANIMS,
	BARANIM_CIN_IDLE_1,
	BARANIM_CIN_WORKIT,
	BARANIM_CIN_GETANGRY,
	BARANIM_CIN_VIBRATE,
	BARANIM_CIN_GETUNANGRY,
	NUM_BARRETT_ANIMS
} barrettAnims_e;

static gameAnim_t g_barrettAnims[NUM_BARRETT_ANIMS] =
{
	{0, 19, 68.0f, true},			//HUMANIM_IDLE
	{92, 97, 168.0f, true},			//HUMANIM_WALK
	{92, 97, 118.0f, true},			//HUMANIM_RUNSLOW
	{92, 97, 88.0f, true},			//HUMANIM_RUN
	{0, 2, 68.0f, false},			//HUMANIM_JUMP
	{0, 2, 68.0f, false},			//HUMANIM_FALL
	{0, 2, 68.0f, false},			//HUMANIM_LAND
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_HIGH1
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_HIGH2
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_LOW1
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_LOW2
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_AIR
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_POPUP
	{30, 32, 168.0f, false},		//HUMANIM_GETUP_BACK
	{30, 32, 168.0f, false},		//HUMANIM_GETUP_FRONT
	{196, 196, 68.0f, false},		//HUMANIM_FLYBACK
	{196, 196, 68.0f, false},		//HUMANIM_FLYBACK2
	{19, 22, 168.0f, false},		//HUMANIM_HIT_FALLFORWARD
	{19, 22, 68.0f, false},			//HUMANIM_FALL_LAND
	{19, 22, 68.0f, false},			//HUMANIM_POPUP_LAND
	{0, 19, 1568.0f, false},		//HUMANIM_DEATH
	{481, 482, 68.0f, false},		//BARANIM_SHOOT 
	{483, 484, 1068.0f, true},		//BARANIM_CIN_IDLE_1 
	{490, 491, 468.0f, true},		//BARANIM_CIN_WORKIT
	{485, 486, 218.0f, false},		//BARANIM_CIN_GETANGRY 
	{486, 487, 48.0f, true},		//BARANIM_CIN_VIBRATE 
	{487, 489, 318.0f, false}		//BARANIM_CIN_GETUNANGRY 
};

static scriptableAnim_t g_barScriptAnims[] =
{
	DEF_SCRIPT_ANIM(HUMANIM_IDLE),
	DEF_SCRIPT_ANIM(BARANIM_CIN_IDLE_1),
	DEF_SCRIPT_ANIM(BARANIM_CIN_WORKIT),
	DEF_SCRIPT_ANIM(BARANIM_CIN_GETANGRY),
	DEF_SCRIPT_ANIM(BARANIM_CIN_VIBRATE),
	DEF_SCRIPT_ANIM(BARANIM_CIN_GETUNANGRY),
	{0, NULL}
};

//think
void ObjBarrett_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	if (obj->curAnim >= BARANIM_CIN_IDLE_1)
	{
		Util_AnimateObject(obj, timeMod);
		return;
	}

	AI_StandardGoals(obj, timeMod);
	AI_GenericThink(obj, timeMod);
	AI_GetToGoal(obj, timeMod);

	if (obj->aiObj->noMoveTime >= g_curTime)
	{ //barrett always looks
		float angleBlendScale = 0.3f;
		obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], obj->aiObj->lookYaw, timeMod*angleBlendScale);
	}

	obj->aiObj->combatRange = 768.0f;
	if (!AI_NonAnglingAnim(obj) && obj->aiObj->enemy && AI_FacingEnemy(obj, obj->aiObj->enemy, 0.7f))
	{
		if (obj->aiObj->attackTime < g_curTime)
		{
			obj->aiObj->combatRange = 1024.0f;
			//if (obj->aiObj->distToEnemy > 400.0f)
			{
				AI_StartAnim(obj, BARANIM_SHOOT, true);
				obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%3000));
				obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 500);
			}
		}
		else
		{
			obj->aiObj->combatRange = 1500.0f;
		}
	}
}

//pick which anim to be in
void ObjBarrett_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->curAnim >= BARANIM_CIN_IDLE_1)
	{ //script will break out
		return;
	}
	else if (obj->curAnim == BARANIM_SHOOT)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			if (rand()%10 <= 8 && obj->aiObj->enemy &&
				AI_ValidateEnemy(obj, obj->aiObj->enemy) &&
				AI_FacingEnemy(obj, obj->aiObj->enemy, 0.9f) &&
				obj->aiObj->distToEnemy > 400.0f)
			{
				obj->animRestart = true;
			}
			else
			{
				AI_StartAnim(obj, HUMANIM_IDLE, true);
			}
		}
	}
	else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//frame tick
void ObjBarrett_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case HUMANIM_WALK:
	case HUMANIM_RUNSLOW:
	case HUMANIM_RUN:
		if ((obj->net.frame == 92 || obj->net.frame == 95) && obj->onGround)
		{
			int snd = g_soundFootsteps[rand()%NUM_SOUNDS_FOOTSTEPS];
			ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
		}
		break;
	case BARANIM_SHOOT:
		if (oldFrame == 481 || oldFrame == 482)
		{
			float pos[3] = {0.0f, -116.0f, -450.0f};
			float ang[3] = {0.0f, 0.0f, -1.0f};
			modelMatrix_t boneMat;
			if (g_sharedFn->Coll_GetModelBoneMatrix(obj->rcColModel, "b12", &boneMat))
			{
				float t[3];
				Math_VecCopy(pos, t);
				Math_TransformPointByMatrix(&boneMat, t, pos);

				if (obj->aiObj->enemy && obj->aiObj->enemy->inuse && AI_FacingEnemy(obj, obj->aiObj->enemy, 0.9f))
				{ //fire at them
					float c[3];
					Util_GameObjectCenter(obj->aiObj->enemy, c);
					Math_VecSub(c, boneMat.o, ang);

					//add some spread
					float maxSpread = 0.05f;
					float r;
					r = (float)(rand()%16384)/16384.0f;
					float spreadX = -maxSpread + r*maxSpread*2.0f;
					r = (float)(rand()%16384)/16384.0f;
					float spreadY =  -maxSpread + r*maxSpread*2.0f;
					Math_VecNorm(ang);
					float right[3], up[3];
					Math_AxisForNormal(ang, 0, right, up);
					Math_VecScale(right, spreadX);
					Math_VecScale(up, spreadY);
					Math_VecAdd(ang, right, ang);
					Math_VecAdd(ang, up, ang);
				}
				else
				{ //fire where the barrel is pointed
					Math_VecCopy(ang, t);
					Math_TransformPointByMatrix(&boneMat, t, ang);
					Math_VecSub(ang, boneMat.o, ang);
				}
			}
			Math_VecNorm(ang);
			Util_FireBullet(obj, pos, ang, NULL);

			Math_VecToAngles(ang, ang);
			ObjParticles_Create("barrettfire", pos, ang, -1);

			ObjSound_Create(obj->net.pos, "assets/sound/cin/shot02.wav", 1.0f, -1);
			obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(200+rand()%400));
			obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 500);
		}
		break;
	default:
		break;
	}
}

//spawn
void ObjBarrett_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	g_sharedFn->Common_ServerString("$assets/sound/cin/shot02.wav");
	g_sharedFn->Common_ServerString("&&barrettfire");

	AI_GenericSpawn(obj, b, args, numArgs);
	obj->death = AI_ImmortalDeath;
	obj->localFlags &= ~LFL_ENEMY;

	obj->net.aiDescIndex = AIDESC_BARRETT;
	obj->aiObj->maxHealth = obj->health;

	obj->aiObj->moveSpeed = 400.0f;
	obj->animhandler = ObjBarrett_PickAnim;
	obj->animframetick = ObjBarrett_FrameTick;
	obj->animTable = g_barrettAnims;
	obj->scriptAnims = g_barScriptAnims;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjBarrett_Think;
}
