/*
=============================================================================
Module Information
------------------
Name:			obj_cloud.cpp
Author:			Rich Whitehouse
Description:	server logic object: cloud
=============================================================================
*/

#include "main.h"
#include "ai.h"

typedef enum
{
	CLDANIM_SLASH01 = NUM_HUMAN_ANIMS,
	CLDANIM_SLASH02,
	CLDANIM_SLASH03,
	CLDANIM_BLOCK,
	CLDANIM_CIN_IDLE_1,
	CLDANIM_CIN_CALM_OUT,
	CLDANIM_CIN_CALM_IN,
	NUM_CLOUD_ANIMS
} cloudAnims_e;

static gameAnim_t g_cloudAnims[NUM_CLOUD_ANIMS] =
{
	{0, 19, 68.0f, true},			//HUMANIM_IDLE
	{122, 127, 168.0f, true},		//HUMANIM_WALK
	{122, 127, 118.0f, true},		//HUMANIM_RUNSLOW
	{122, 127, 88.0f, true},		//HUMANIM_RUN
	{0, 2, 68.0f, false},			//HUMANIM_JUMP
	{0, 2, 68.0f, false},			//HUMANIM_FALL
	{0, 2, 68.0f, false},			//HUMANIM_LAND
	{223, 231, 68.0f, false},		//HUMANIM_PAIN_HIGH1
	{223, 231, 68.0f, false},		//HUMANIM_PAIN_HIGH2
	{223, 231, 68.0f, false},		//HUMANIM_PAIN_LOW1
	{223, 231, 68.0f, false},		//HUMANIM_PAIN_LOW2
	{223, 231, 68.0f, false},		//HUMANIM_PAIN_AIR
	{223, 231, 68.0f, false},		//HUMANIM_PAIN_POPUP
	{37, 40, 118.0f, false},		//HUMANIM_GETUP_BACK
	{37, 40, 118.0f, false},		//HUMANIM_GETUP_FRONT
	{211, 211, 68.0f, false},		//HUMANIM_FLYBACK
	{211, 211, 68.0f, false},		//HUMANIM_FLYBACK2
	{223, 231, 68.0f, false},		//HUMANIM_HIT_FALLFORWARD
	{19, 22, 68.0f, false},			//HUMANIM_FALL_LAND
	{19, 22, 68.0f, false},			//HUMANIM_POPUP_LAND
	{0, 19, 1568.0f, false},		//HUMANIM_DEATH
	{374, 379, 138.0f, false},		//CLDANIM_SLASH01
	{380, 387, 118.0f, false},		//CLDANIM_SLASH02
	{388, 394, 148.0f, false},		//CLDANIM_SLASH03
	{116, 117, 168.0f, false},		//CLDANIM_BLOCK
	{490, 491, 1068.0f, true},		//CLDANIM_CIN_IDLE_1
	{492, 492, 668.0f, false},		//CLDANIM_CIN_CALM_OUT
	{493, 493, 668.0f, false}		//CLDANIM_CIN_CALM_IN
};

static scriptableAnim_t g_cldScriptAnims[] =
{
	DEF_SCRIPT_ANIM(HUMANIM_IDLE),
	DEF_SCRIPT_ANIM(CLDANIM_CIN_IDLE_1),
	DEF_SCRIPT_ANIM(CLDANIM_CIN_CALM_OUT),
	DEF_SCRIPT_ANIM(CLDANIM_CIN_CALM_IN),
	{0, NULL}
};

#define NUM_CLOUD_DAMAGE_BONES	1
static damageBone_t g_cloudDamageBones[NUM_CLOUD_DAMAGE_BONES] =
{
	{"b13", "b13", {0.0f, -800.0f, 0.0f}, 64.0f}			//DMGBONE_FIST_R
};

//returns damage bone or -1 if not in attack
int ObjCloud_InAttack(gameObject_t *obj)
{
	switch (obj->curAnim)
	{
	case CLDANIM_SLASH01:
		if (obj->net.frame >= 375 && obj->net.frame <= 376)
		{
			return DMGBONE_FIST_R;
		}
		break;
	case CLDANIM_SLASH02:
		if (obj->net.frame == 382)
		{
			return DMGBONE_FIST_R;
		}
		break;
	case CLDANIM_SLASH03:
		if (obj->net.frame >= 390 && obj->net.frame <= 393)
		{
			return DMGBONE_FIST_R;
		}
		break;
	default:
		break;
	}

	return -1;
}

//think
void ObjCloud_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	if (obj->curAnim >= CLDANIM_CIN_IDLE_1)
	{
		Util_AnimateObject(obj, timeMod);
		return;
	}

	AI_StandardGoals(obj, timeMod);
	AI_GenericThink(obj, timeMod);
	AI_GetToGoal(obj, timeMod);

	dmgBoneLine_t dmgPos[NUM_CLOUD_DAMAGE_BONES];
	AI_TransformDamageBones(obj, g_cloudDamageBones, dmgPos, NUM_CLOUD_DAMAGE_BONES);

	int dmgBone = ObjCloud_InAttack(obj);
	if (obj->aiObj->hasLastDmg && dmgBone >= 0)
	{
		AI_RunDamageBone(obj, &g_cloudDamageBones[dmgBone], &obj->aiObj->lastDmg[dmgBone],
			&dmgPos[dmgBone], 30);
	}

	if (obj->curAnim == CLDANIM_SLASH01 ||
		obj->curAnim == CLDANIM_SLASH02 ||
		obj->curAnim == CLDANIM_SLASH03)
	{
		obj->net.renderEffects |= FXFL_BLADETRAIL;
		obj->net.effectLen = 500.0f;
		obj->net.strIndexC = g_sharedFn->Common_ServerString(g_cloudDamageBones[0].boneName);
		obj->net.strIndexD = obj->net.strIndexC;
	}
	else
	{
		obj->net.renderEffects &= ~FXFL_BLADETRAIL;
	}

	memcpy(obj->aiObj->lastDmg, dmgPos, sizeof(dmgPos));
	obj->aiObj->hasLastDmg = true;

	obj->aiObj->combatRange = 512.0f;
	if (!AI_NonAnglingAnim(obj) && obj->aiObj->enemy && AI_FacingEnemy(obj, obj->aiObj->enemy))
	{
		if (obj->aiObj->attackTime < g_curTime)
		{
			obj->aiObj->combatRange = 128.0f;
			if (obj->aiObj->distToEnemy < 450.0f)
			{
				AI_StartAnim(obj, CLDANIM_SLASH01, true);
				ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
				obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
				obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 1000);
			}
		}
		else
		{
			obj->aiObj->combatRange = 768.0f;
		}
	}
}

//pick which anim to be in
void ObjCloud_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->curAnim >= CLDANIM_CIN_IDLE_1)
	{ //script will break out
		return;
	}
	else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		if (obj->curAnim == CLDANIM_SLASH01 && AI_ValidateEnemy(obj, obj->aiObj->enemy) &&
			obj->aiObj->distToEnemy < 450.0f)
		{
			AI_StartAnim(obj, CLDANIM_SLASH02, true);
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
			obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
			obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 1000);
		}
		else if (obj->curAnim == CLDANIM_SLASH02 && AI_ValidateEnemy(obj, obj->aiObj->enemy) &&
			obj->aiObj->distToEnemy < 450.0f)
		{
			AI_StartAnim(obj, CLDANIM_SLASH03, true);
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
			obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
			obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 1000);
		}
		else
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
	}
}

//see if it hit the buster
bool ObjCloud_AttackBlock(gameObject_t *obj, gameObject_t *hurter, int dmg, collObj_t *col,
							float *impactPos, float *impactAng)
{
	if (dmg > 999)
	{
		return false;
	}

	if (col->hitTex)
	{
		const char *tn = g_sharedFn->Coll_GetHitTexName(col);
		if (tn && !stricmp(tn, "assets/textures/cloudffdclr17"))
		{
			ObjParticles_Create("melee/impact", impactPos, impactAng, -1);
			ObjSound_Create(impactPos, "assets/sound/cb/hitmetal.wav", 1.0f, -1);
			if (!AI_NonMovingAnim(obj))
			{
				AI_StartAnim(obj, CLDANIM_BLOCK, true);
			}
			return true;
		}
	}

	return false;
}

//frame tick
void ObjCloud_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case HUMANIM_WALK:
	case HUMANIM_RUNSLOW:
	case HUMANIM_RUN:
		if ((obj->net.frame == 123 || obj->net.frame == 126) && obj->onGround)
		{
			int snd = g_soundFootsteps[rand()%NUM_SOUNDS_FOOTSTEPS];
			ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
		}
		break;
	default:
		break;
	}
}

//spawn
void ObjCloud_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	static int dmgSounds[3];
	dmgSounds[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/buscut01.wav");
	dmgSounds[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/buscut02.wav");
	dmgSounds[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/buscut03.wav");

	AI_GenericSpawn(obj, b, args, numArgs);
	obj->death = AI_ImmortalDeath;
	obj->localFlags &= ~LFL_ENEMY;

	obj->net.aiDescIndex = AIDESC_CLOUD;
	obj->aiObj->maxHealth = obj->health;

	obj->aiObj->dmgEffect = "melee/bigslash";
	obj->aiObj->dmgSounds = dmgSounds;
	obj->aiObj->numDmgSounds = 3;
	obj->aiObj->moveSpeed = 400.0f;
	obj->animhandler = ObjCloud_PickAnim;
	obj->animframetick = ObjCloud_FrameTick;
	obj->animTable = g_cloudAnims;
	obj->scriptAnims = g_cldScriptAnims;
	obj->attackblock = ObjCloud_AttackBlock;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjCloud_Think;
}
