/*
=============================================================================
Module Information
------------------
Name:			obj_gigas.cpp
Author:			Rich Whitehouse
Description:	server logic object: gigas
=============================================================================
*/

#include "main.h"
#include "ai.h"

//just to make things a bit more readable
#define debounceFastLook		debounce5
#define debounceAura			debounce4

typedef enum
{
	GIGANIM_POWERSMASH = NUM_HUMAN_ANIMS,
	GIGANIM_FISTBLOW,
	GIGANIM_POWERUP,
	NUM_GIG_ANIMS
} gigAnims_e;

static gameAnim_t g_gigAnims[NUM_GIG_ANIMS] =
{
	{0, 35, 68.0f, true},			//HUMANIM_IDLE
	{378, 383, 168.0f, true},		//HUMANIM_WALK
	{378, 383, 168.0f, true},		//HUMANIM_RUNSLOW
	{378, 383, 168.0f, true},		//HUMANIM_RUN
	{0, 4, 68.0f, false},			//HUMANIM_JUMP
	{0, 4, 68.0f, false},			//HUMANIM_FALL
	{0, 4, 68.0f, false},			//HUMANIM_LAND
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_HIGH1
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_HIGH2
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_LOW1
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_LOW2
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_AIR
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_POPUP
	{36, 41, 118.0f, false},		//HUMANIM_GETUP_BACK
	{36, 41, 118.0f, false},		//HUMANIM_GETUP_FRONT
	{36, 41, 68.0f, false},			//HUMANIM_FLYBACK
	{36, 41, 68.0f, false},			//HUMANIM_FLYBACK2
	{36, 41, 68.0f, false},			//HUMANIM_HIT_FALLFORWARD
	{0, 4, 68.0f, false},			//HUMANIM_FALL_LAND
	{0, 4, 68.0f, false},			//HUMANIM_POPUP_LAND
	{106, 106, 1568.0f, false},		//HUMANIM_DEATH
	{64, 116, 68.0f, false},		//GIGANIM_POWERSMASH
	{/*196*/218, 243, 68.0f, false},		//GIGANIM_FISTBLOW
	{274, 309, 68.0f, false}		//GIGANIM_POWERUP
};

#define NUM_GIG_DAMAGE_BONES	2
static damageBone_t g_gigDamageBones[NUM_GIG_DAMAGE_BONES] =
{
	{"b23", "b22", {0.0f, 0.0f, 0.0f}, 200.0f},			//DMGBONE_FIST_R
	{"b15", "b14", {0.0f, 0.0f, 0.0f}, 200.0f}			//DMGBONE_FIST_L
};

static damageBone_t g_gigDamageBones2[NUM_GIG_DAMAGE_BONES] =
{
	{"b17", "b16", {0.0f, 0.0f, 0.0f}, 200.0f},			//DMGBONE_FIST_R
	{"b9", "b8", {0.0f, 0.0f, 0.0f}, 200.0f}			//DMGBONE_FIST_L
};

//returns damage bone or -1 if not in attack
int ObjGig_InAttack(gameObject_t *obj, int *dmgBones)
{
	if (obj->curAnim == GIGANIM_POWERSMASH)
	{
		if (obj->net.frame >= 85 && obj->net.frame <= 92)
		{
			dmgBones[0] = DMGBONE_FIST_L;
			return 1;
		}
	}
	else if (obj->curAnim == GIGANIM_FISTBLOW)
	{
		if (obj->net.frame >= 223 && obj->net.frame <= 229)
		{
			dmgBones[0] = DMGBONE_FIST_R;
			return 1;
		}
	}

	return -1;
}

//occupied?
bool ObjGig_CanAct(gameObject_t *obj)
{
	if (!obj->onGround)
	{
		return false;
	}
	if (obj->curAnim >= GIGANIM_POWERSMASH)
	{
		return false;
	}
	if (AI_NonMovingAnim(obj))
	{
		return false;
	}
	if (obj->aiObj->noMoveTime >= g_curTime)
	{
		return false;
	}
	return true;
}

//can i dodge?
bool ObjGig_CanDodge(gameObject_t *obj)
{
	if (obj->generalFlag >= 0)
	{
		return false;
	}
	if (!ObjGig_CanAct(obj))
	{
		return false;
	}

	return true;
}

//dodge (gigs currently don't dodge)
bool ObjGig_DodgeEnemy(gameObject_t *obj, gameObject_t *enemy)
{
	return false;
}

//think
void ObjGig_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	if (obj->debounceAura > g_curTime && obj->health > 0)
	{
		obj->net.renderEffects |= FXFL_AURA2;
	}
	else
	{
		obj->net.renderEffects &= ~FXFL_AURA2;
	}

	//special case, set move speed based on walk frame
	if (obj->net.frame == 379 || obj->net.frame == 382)
	{
		obj->aiObj->moveSpeed = 100.0f;
	}
	else if (obj->net.frame == 380 || obj->net.frame == 383)
	{
		obj->aiObj->moveSpeed = 50.0f;
	}
	else if (obj->net.frame == 381 || obj->net.frame == 378)
	{
		obj->aiObj->moveSpeed = 200.0f;
	}
	else
	{
		obj->aiObj->moveSpeed = 50.0f;
	}

	AI_StandardGoals(obj, timeMod);
	if (!obj->onGround && obj->aiObj->noMoveTime < g_curTime+500)
	{
		obj->aiObj->noMoveTime = g_curTime+500;
	}
	if ((!obj->aiObj->enemy || obj->aiObj->distToEnemy > 1400.0f) && Math_VecLen(obj->net.vel) > 5.0f && obj->onGround)
	{ //look where i'm going
		float ang[3];
		Math_VecToAngles(obj->net.vel, ang);
		obj->aiObj->lookYaw = ang[YAW];
	}
	AI_GenericThink(obj, timeMod);
	AI_GetToGoal(obj, timeMod);

	if (obj->debounceFastLook >= g_curTime)
	{
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 45.0f);
	}

	damageBone_t *useDmgBones = (obj->generalFlag <= -2) ? g_gigDamageBones2 : g_gigDamageBones;

	dmgBoneLine_t dmgPos[NUM_GIG_DAMAGE_BONES];
	AI_TransformDamageBones(obj, useDmgBones, dmgPos, NUM_GIG_DAMAGE_BONES);
	int dmgBones[MAX_AI_DAMAGE_BONES];
	int numDmgBones = ObjGig_InAttack(obj, dmgBones);
	if (obj->aiObj->hasLastDmg && numDmgBones >= 0)
	{
		int dmg = AI_LevelDamage(obj, 80);
		for (int i = 0; i < numDmgBones; i++)
		{
			AI_RunDamageBone(obj, &useDmgBones[dmgBones[i]], &obj->aiObj->lastDmg[dmgBones[i]],
				&dmgPos[dmgBones[i]], dmg);
		}
	}

	if (obj->curAnim == GIGANIM_FISTBLOW)
	{
		obj->atkType = ATKTYPE_KNOCKBACK;
	}
	else
	{
		obj->atkType = ATKTYPE_NORMAL;
	}

	memcpy(obj->aiObj->lastDmg, dmgPos, sizeof(dmgPos));
	obj->aiObj->hasLastDmg = true;

	obj->aiObj->combatRange = 700.0f;
	if (!AI_NonAnglingAnim(obj) && obj->aiObj->enemy && AI_FacingEnemy(obj, obj->aiObj->enemy))
	{
		if (obj->aiObj->enemy->net.index < MAX_NET_CLIENTS &&
			ObjGig_CanDodge(obj) &&
			ObjPlayer_InDodgeableAttack(obj->aiObj->enemy) &&
			ObjGig_DodgeEnemy(obj, obj->aiObj->enemy))
		{ //dodge
			obj->aiObj->noMoveTime = g_curTime+100;
		}
		else
		{
			bool shouldCloseIn = AI_ShouldCloseIn(obj);
			if (obj->generalFlag <= -1 && obj->aiObj->distToEnemy > 1600.0f && obj->debounceAura < g_curTime && ObjGig_CanAct(obj))
			{
				AI_StartAnim(obj, GIGANIM_POWERUP, true);
				obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 3000);
				obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(3000+rand()%2000));
			}
			else if (obj->aiObj->attackTime < g_curTime && shouldCloseIn && ObjGig_CanAct(obj))
			{
				obj->aiObj->combatRange = 400.0f;
				float closeAttackDist = 700.0f;
				if (obj->aiObj->distToEnemy < closeAttackDist)
				{
					serverTime_t atkDel;
					if (rand()%10 < 5)
					{
						AI_StartAnim(obj, GIGANIM_POWERSMASH, true);
						atkDel = 3000;
					}
					else
					{
						AI_StartAnim(obj, GIGANIM_FISTBLOW, true);
						atkDel = 2000;
					}
					if (obj->generalFlag <= -1)
					{
						obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(atkDel+rand()%2000));
					}
					else
					{
						obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(atkDel+1000+rand()%4000));
					}
					obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, atkDel);
				}
			}
			else
			{
				float closeDist = (obj->generalFlag <= 0) ? 600.0f : 400.0f;
				float r = (shouldCloseIn) ? closeDist : 800.0f;
				obj->aiObj->combatRange = g_ai.playerCloseDist+r;
			}
		}
	}
}

//pick which anim to be in
void ObjGig_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//frame tick
void ObjGig_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case HUMANIM_WALK:
	case HUMANIM_RUNSLOW:
	case HUMANIM_RUN:
		if (obj->onGround && (obj->net.frame == 381 || obj->net.frame == 378))
		{
			char *stepSounds[3] =
			{
				"assets/sound/cb/bigfoot1.wav",
				"assets/sound/cb/bigfoot2.wav",
				"assets/sound/cb/bigfoot3.wav"
			};
			ObjSound_Create(obj->net.pos, stepSounds[rand()%3], 1.0f, -1);
			float shakeRad = (obj->generalFlag2 <= -2) ? 3000.0f : 4096.0f;
			Util_ProximityShake(obj->net.pos, shakeRad, 5.0f, 0.5f);
		}
		break;
	case GIGANIM_POWERSMASH:
		if (obj->net.frame == 86)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
		}
		else if (obj->net.frame == 87 && obj->onGround)
		{
			float p[3];
			Math_VecCopy(obj->net.pos, p);
			float fwd[3], right[3];
			Math_AngleVectors(obj->net.ang, fwd, right, 0);
			if (obj->generalFlag <= -2)
			{
				Math_VecMA(p, 350.0f, fwd, p);
				Math_VecMA(p, -100.0f, right, p);
			}
			else
			{
				Math_VecMA(p, 130.0f, fwd, p);
				Math_VecMA(p, -300.0f, right, p);
			}
			p[2] += 8.0f;
			float ang[3] = {-90.0f, 0.0f, 0.0f};
			ObjParticles_Create("impacts/gigascrush", p, ang, -1);
			ObjSound_Create(p, "assets/sound/cb/groundsmash.wav", 1.0f, -1);
			int dmg = AI_LevelDamage(obj, 200);
			Util_RadiusDamage(obj, dmg, 2048.0f, p, false);
			Util_ProximityShake(obj->net.pos, 4096.0f, 12.0f, 1.0f);
		}
		break;
	case GIGANIM_FISTBLOW:
		if (obj->net.frame == 225)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
		}
		break;
	case GIGANIM_POWERUP:
		if (obj->net.frame == 300)
		{
			float p[3];
			float fwd[3], up[3];
			Math_AngleVectors(obj->net.ang, fwd, 0, up);
			Math_VecCopy(obj->net.pos, p);
			if (obj->generalFlag <= -2)
			{
				Math_VecMA(p, 400.0f, fwd, p);
				Math_VecMA(p, 450.0f, up, p);
			}
			else
			{
				Math_VecMA(p, 450.0f, fwd, p);
				Math_VecMA(p, 650.0f, up, p);
			}
			ObjSound_Create(p, "assets/sound/cb/fistpower.wav", 1.0f, -1);
			ObjParticles_Create("impacts/gigaspow", p, obj->net.ang, -1);
			Util_ProximityShake(p, 4096.0f, 12.0f, 1.0f);
			obj->debounceAura = g_curTime + 20000;
			obj->net.renderEffects |= FXFL_AURA2;
		}
		break;
	default:
		break;
	}
}

//touch
void ObjGig_Touch(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	AI_GenericTouch(obj, other, col);
}

//gigs currently can't block
bool ObjGig_AttackBlock(gameObject_t *obj, gameObject_t *hurter, int dmg, collObj_t *col,
							float *impactPos, float *impactAng)
{
	return false;
}

//called when inflicting damage
void ObjGig_OnHurt(gameObject_t *obj, gameObject_t *victim, int dmg, const collObj_t *col)
{
	if (obj->debounceAura > g_curTime && col)
	{
		float impactPos[3];
		float ang[3];
		Math_VecCopy((float *)col->endPos, impactPos);
		Math_VecToAngles((float *)col->endNormal, ang);
		ObjParticles_Create("melee/impactexplosive", impactPos, ang, -1);
		ObjSound_Create(impactPos, "assets/sound/cb/hitaura.wav", 1.0f, -1);
		Util_RadiusDamage(obj, dmg, 900.0f, impactPos, false);
	}
}

//spawn
void ObjGig_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	AI_GenericSpawn(obj, b, args, numArgs);

	obj->touch = ObjGig_Touch;
	obj->animframetick = ObjGig_FrameTick;
	obj->onhurt = ObjGig_OnHurt;
	obj->aiObj->painChance = 5;
	obj->aiObj->forceGravity = 300.0f;

	const char *elvalue = Common_GetValForKey(b, "elvalue");
	if (elvalue && elvalue[0])
	{
		obj->generalFlag = atoi(elvalue);
	}

	if (obj->generalFlag == -2)
	{
		obj->aiObj->makoValue = 100 + rand()%30;
		obj->net.aiDescIndex = AIDESC_GIG2;
		obj->aiObj->painChance = 25;
	}
	else if (obj->generalFlag == -1)
	{
		obj->aiObj->makoValue = 120 + rand()%100;
		obj->net.aiDescIndex = AIDESC_GIG3;
	}
	else
	{
		obj->aiObj->makoValue = 60 + rand()%10;
		obj->net.aiDescIndex = AIDESC_GIG1;
	}

	obj->aiObj->dropChances[INVITEM_EARTHDRUM] = 90;
	if (obj->generalFlag <= -1)
	{
		obj->aiObj->dropChances[INVITEM_EARTHDRUM] = 100;
		obj->aiObj->dropChances[INVITEM_STARDUST] = 30;
	}
	obj->aiObj->maxHealth = obj->health;

	obj->aiObj->dmgEffect = "melee/impacten";
	obj->aiObj->dmgSounds = g_soundHitHeavy;
	obj->aiObj->numDmgSounds = NUM_SOUNDS_HITHEAVY;
	obj->aiObj->moveSpeed = 150.0f;
	obj->animhandler = ObjGig_PickAnim;
	obj->animTable = g_gigAnims;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjGig_Think;
	obj->attackblock = ObjGig_AttackBlock;
}
