/*
=============================================================================
Module Information
------------------
Name:			obj_guarddog.cpp
Author:			Rich Whitehouse
Description:	server logic object: guard dog
=============================================================================
*/

#include "main.h"
#include "ai.h"

//just to make things a bit more readable
#define debounceLungeTime		debounce2
#define debounceDrainTime		debounce3
#define debounceFastLunge		debounce4
#define debounceFastLook		debounce5

typedef enum
{
	DOGANIM_HEADLUNGE = NUM_HUMAN_ANIMS,
	DOGANIM_SPINEATTACK,
	NUM_DOG_ANIMS
} dogAnims_e;

static gameAnim_t g_dogAnims[NUM_DOG_ANIMS] =
{
	{0, 29, 68.0f, true},			//HUMANIM_IDLE
	{43, 46, 188.0f, true},			//HUMANIM_WALK
	{43, 46, 138.0f, true},			//HUMANIM_RUNSLOW
	{43, 46, 98.0f, true},			//HUMANIM_RUN
	{45, 46, 68.0f, false},			//HUMANIM_JUMP
	{43, 43, 68.0f, false},			//HUMANIM_FALL
	{45, 46, 68.0f, false},			//HUMANIM_LAND
	{30, 38, 68.0f, false},			//HUMANIM_PAIN_HIGH1
	{30, 38, 68.0f, false},			//HUMANIM_PAIN_HIGH2
	{30, 38, 68.0f, false},			//HUMANIM_PAIN_LOW1
	{30, 38, 68.0f, false},			//HUMANIM_PAIN_LOW2
	{171, 174, 68.0f, false},		//HUMANIM_PAIN_AIR
	{171, 174, 68.0f, false},		//HUMANIM_PAIN_POPUP
	{175, 177, 118.0f, false},		//HUMANIM_GETUP_BACK
	{175, 177, 118.0f, false},		//HUMANIM_GETUP_FRONT
	{171, 174, 68.0f, false},		//HUMANIM_FLYBACK
	{171, 174, 68.0f, false},		//HUMANIM_FLYBACK2
	{172, 175, 68.0f, false},		//HUMANIM_HIT_FALLFORWARD
	{175, 175, 68.0f, false},		//HUMANIM_FALL_LAND
	{175, 175, 68.0f, false},		//HUMANIM_POPUP_LAND
	{171, 171, 1568.0f, false},		//HUMANIM_DEATH
	{140, 150, 68.0f, false},		//DOGANIM_HEADLUNGE
	{90, 104, 68.0f, false}			//DOGANIM_SPINEATTACK
};

#define NUM_DOG_DAMAGE_BONES	7
static damageBone_t g_dogDamageBones[NUM_DOG_DAMAGE_BONES] =
{
	{"b6", "b5", {0.0f, 0.0f, 0.0f}, 140.0f},			//DMGBONE_FIST_R
	{"b9", "b8", {0.0f, 0.0f, 0.0f}, 128.0f},
	{"b10", "b9", {0.0f, 0.0f, 0.0f}, 128.0f},
	{"b11", "b10", {0.0f, 0.0f, 0.0f}, 128.0f},
	{"b12", "b11", {0.0f, 0.0f, 0.0f}, 128.0f},
	{"b13", "b12", {0.0f, 0.0f, 0.0f}, 128.0f},
	{"b14", "b13", {0.0f, 0.0f, 0.0f}, 128.0f}
};

//returns damage bone or -1 if not in attack
int ObjDog_InAttack(gameObject_t *obj, int *dmgBones)
{
	if (obj->curAnim == DOGANIM_HEADLUNGE)
	{
		if (obj->net.frame >= 144 && obj->net.frame <= 150)
		{
			dmgBones[0] = DMGBONE_FIST_R;
			return 1;
		}
	}
	else if (obj->curAnim == DOGANIM_SPINEATTACK)
	{
		if (obj->net.frame >= 94 && obj->net.frame <= 102)
		{
			dmgBones[0] = 1;
			dmgBones[1] = 2;
			dmgBones[2] = 3;
			dmgBones[3] = 4;
			dmgBones[4] = 5;
			dmgBones[5] = 6;
			return 6;
		}
	}

	return -1;
}

//occupied?
bool ObjDog_CanAct(gameObject_t *obj)
{
	if (!obj->onGround)
	{
		return false;
	}
	if (obj->curAnim >= DOGANIM_HEADLUNGE)
	{
		return false;
	}
	if (AI_NonMovingAnim(obj))
	{
		return false;
	}
	if (obj->aiObj->noMoveTime >= g_curTime)
	{
		return false;
	}
	return true;
}

//can i dodge?
bool ObjDog_CanDodge(gameObject_t *obj)
{
	if (obj->generalFlag >= 0)
	{
		return false;
	}
	if (!ObjDog_CanAct(obj))
	{
		return false;
	}

	return true;
}

//dodge (dogs currently don't dodge)
bool ObjDog_DodgeEnemy(gameObject_t *obj, gameObject_t *enemy)
{
	return false;
}

//pick which lunge to do
int ObjDog_PickLungeAnim(gameObject_t *obj)
{
	return DOGANIM_HEADLUNGE;
}

//think
void ObjDog_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	if (obj->debounceDrainTime >= g_curTime)
	{
		obj->net.renderEffects2 |= FXFL2_DRAINAURA;
	}
	else
	{
		obj->net.renderEffects2 &= ~FXFL2_DRAINAURA;
	}

	AI_StandardGoals(obj, timeMod);
	if ((!obj->aiObj->enemy || obj->aiObj->distToEnemy > 1000.0f) && Math_VecLen(obj->net.vel) > 5.0f && obj->onGround)
	{ //look where i'm going
		float ang[3];
		Math_VecToAngles(obj->net.vel, ang);
		obj->aiObj->lookYaw = ang[YAW];
	}
	AI_GenericThink(obj, timeMod);
	AI_GetToGoal(obj, timeMod);

	if (obj->debounceFastLook >= g_curTime)
	{
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 45.0f);
	}

	if (obj->net.frame >= 142 && obj->net.frame <= 150)
	{ //lunge
		float fwd[3], f;
		f = (obj->debounceFastLunge >= g_curTime) ? 325.0f : 300.0f;
		if (obj->generalFlag <= -1)
		{
			f *= 1.25f;
		}
		Math_AngleVectors(obj->net.ang, fwd, 0, 0);
		obj->net.vel[0] += fwd[0]*f;
		obj->net.vel[1] += fwd[1]*f;
		obj->net.vel[2] += fwd[2]*f;
	}

	dmgBoneLine_t dmgPos[NUM_DOG_DAMAGE_BONES];
	AI_TransformDamageBones(obj, g_dogDamageBones, dmgPos, NUM_DOG_DAMAGE_BONES);
	int dmgBones[MAX_AI_DAMAGE_BONES];
	int numDmgBones = ObjDog_InAttack(obj, dmgBones);
	if (obj->aiObj->hasLastDmg && numDmgBones >= 0)
	{
		int dmg = AI_LevelDamage(obj, 60);
		for (int i = 0; i < numDmgBones; i++)
		{
			AI_RunDamageBone(obj, &g_dogDamageBones[dmgBones[i]], &obj->aiObj->lastDmg[dmgBones[i]],
				&dmgPos[dmgBones[i]], dmg);
		}
	}

	if (obj->curAnim == DOGANIM_HEADLUNGE)
	{
		obj->atkType = ATKTYPE_KNOCKBACK2;
	}
	else
	{
		obj->atkType = ATKTYPE_NORMAL;
	}

	if (obj->curAnim == DOGANIM_HEADLUNGE)
	{
		obj->atkType = ATKTYPE_KNOCKBACK;
		if (obj->curAnim == DOGANIM_HEADLUNGE)
		{
			obj->net.renderEffects |= FXFL_MODELTRAIL;
		}
	}
	memcpy(obj->aiObj->lastDmg, dmgPos, sizeof(dmgPos));
	obj->aiObj->hasLastDmg = true;

	obj->aiObj->combatRange = 700.0f;
	if (!AI_NonAnglingAnim(obj) && obj->aiObj->enemy && AI_FacingEnemy(obj, obj->aiObj->enemy))
	{
		if (obj->aiObj->enemy->net.index < MAX_NET_CLIENTS &&
			ObjDog_CanDodge(obj) &&
			ObjPlayer_InDodgeableAttack(obj->aiObj->enemy) &&
			ObjDog_DodgeEnemy(obj, obj->aiObj->enemy))
		{ //dodge
			obj->aiObj->noMoveTime = g_curTime+100;
		}
		else if (obj->aiObj->enemy->net.index < MAX_NET_CLIENTS &&
			obj->generalFlag <= 0 &&
			ObjDog_CanAct(obj) &&
			ObjPlayer_InVulnerableState(obj->aiObj->enemy) &&
			obj->aiObj->distToEnemy < 1200.0f)
		{ //lunge when they're vulnerable
			AI_StartAnim(obj, ObjDog_PickLungeAnim(obj), true);
			obj->debounceLungeTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%8000));
			obj->debounceFastLunge = g_curTime + Util_LocalDelay(obj, 1000);
			obj->debounceFastLook = g_curTime + Util_LocalDelay(obj, 300);
		}
		else
		{
			bool shouldCloseIn = AI_ShouldCloseIn(obj);
			if (obj->aiObj->attackTime < g_curTime && shouldCloseIn && ObjDog_CanAct(obj))
			{
				bool okToAttack = ObjPlayer_OkayToAttack(obj, obj->aiObj->enemy);
				obj->aiObj->combatRange = (okToAttack) ? 300.0f : Util_RandFloat(400.0f, 1300.0f);
				float closeAttackDist = 700.0f;
				if (obj->aiObj->distToEnemy < closeAttackDist &&
					okToAttack)
				{
					AI_StartAnim(obj, DOGANIM_SPINEATTACK, true);
					if (obj->generalFlag <= -1)
					{
						obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(1500+rand()%2000));
					}
					else
					{
						obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
					}
					if (obj->aiObj->aiLevel < 15 && ObjPlayer_IsWeakSauce(obj->aiObj->enemy))
					{
						obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 3000);
					}
					else
					{
						obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 1000);
					}
				}
				else if (obj->generalFlag <= 0 && obj->debounceLungeTime < g_curTime &&
					obj->aiObj->distToEnemy < 2000.0f /*&& (obj->aiObj->distToEnemy > 850.0f)*/ &&
					AI_CanReachPoint(obj, obj->aiObj->goalPos, obj->aiObj->enemy) &&
					ObjPlayer_OkayToAttack(obj, obj->aiObj->enemy))
				{
					if (g_ai.weakAI && obj->aiObj->aiLevel < 15 && ObjPlayer_IsWeakSauce(obj->aiObj->enemy))
					{ //don't lunge right now
						obj->debounceLungeTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(4000+rand()%6000));
					}
					else
					{
						obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 90.0f);
						AI_StartAnim(obj, ObjDog_PickLungeAnim(obj), true);
						obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
						obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 1000);
						obj->debounceLungeTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%8000));
					}
				}
			}
			else
			{
				float closeDist = (obj->generalFlag <= 0) ? 600.0f : 400.0f;
				float r = (shouldCloseIn) ? closeDist : 800.0f;
				obj->aiObj->combatRange = g_ai.playerCloseDist+r;
			}
		}
	}
}

//pick which anim to be in
void ObjDog_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//frame tick
void ObjDog_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case HUMANIM_WALK:
	case HUMANIM_RUNSLOW:
	case HUMANIM_RUN:
		if (obj->onGround && (obj->net.frame == 44 || obj->net.frame == 46))
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chocostep.wav", 1.0f, -1);
		}
		break;
	case DOGANIM_HEADLUNGE:
		if (obj->net.frame == 142)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/lunge.wav", 1.0f, -1);
		}
		break;
	case DOGANIM_SPINEATTACK:
		if (obj->net.frame == 94)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
		}
		else if (obj->net.frame == 101)
		{
			float fwd[3];
			float f = 1024.0f;
			Math_AngleVectors(obj->net.ang, fwd, 0, 0);
			obj->net.vel[0] -= fwd[0]*f;
			obj->net.vel[1] -= fwd[1]*f;
			obj->net.vel[2] = 384.0f;
		}
		break;
	default:
		break;
	}
}

//touch
void ObjDog_Touch(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	if (!col || col->containsSolid)
	{
		return;
	}

	if (obj->curAnim == DOGANIM_HEADLUNGE)
	{
		if (other && other->hurtable && Util_ValidEnemies(obj, other))
		{
			if (Util_SequenceDamage(obj, other))
			{
				float ang[3];
				Math_VecToAngles((float *)col->endNormal, ang);
				Util_SetSequenceDamage(obj, other);
				float impactPos[3];
				Math_VecCopy((float *)col->endPos, impactPos);
				impactPos[2] += obj->net.mins[2]*0.5f;
				ObjSound_CreateFromIndex(impactPos, g_soundHitLight[rand()%NUM_SOUNDS_HITLIGHT], 1.0f, -1);
				ObjParticles_Create("melee/impacten", impactPos, ang, -1);
				collObj_t nCol = *col;
				Math_VecCopy(impactPos, nCol.endPos);
				int dmg = AI_LevelDamage(obj, 60);
				Util_DamageObject(obj, other, dmg, &nCol);
			}
		}
	}

	AI_GenericTouch(obj, other, col);
}

//dogs currently can't block
bool ObjDog_AttackBlock(gameObject_t *obj, gameObject_t *hurter, int dmg, collObj_t *col,
							float *impactPos, float *impactAng)
{
	return false;
}

//called when inflicting damage
void ObjDog_OnHurt(gameObject_t *obj, gameObject_t *victim, int dmg, const collObj_t *col)
{
	if (victim && victim->hurtable && obj->generalFlag <= 0 && Util_ValidEnemies(obj, victim))
	{
		if (obj->generalFlag <= -1)
		{
			obj->health += dmg;
		}
		else
		{
			obj->health += dmg/2;
		}
		if (obj->health > obj->aiObj->maxHealth)
		{
			obj->health = obj->aiObj->maxHealth;
		}
		ObjSound_Create(obj->net.pos, "assets/sound/cb/healstep.wav", 1.0f, -1);
		obj->net.effectLen = 500.0f;
		obj->net.renderEffects2 |= FXFL2_DRAINAURA;
		obj->debounceDrainTime = g_curTime + Util_LocalDelay(obj, 700);
	}
}

//spawn
void ObjDog_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	AI_GenericSpawn(obj, b, args, numArgs);

	obj->touch = ObjDog_Touch;
	obj->animframetick = ObjDog_FrameTick;
	obj->onhurt = ObjDog_OnHurt;

	const char *elvalue = Common_GetValForKey(b, "elvalue");
	if (elvalue && elvalue[0])
	{
		obj->generalFlag = atoi(elvalue);
	}

	if (obj->generalFlag == -1)
	{
		obj->aiObj->makoValue = 60 + rand()%10;
		obj->net.aiDescIndex = AIDESC_DOG3;
	}
	else if (obj->generalFlag == 0)
	{
		obj->aiObj->makoValue = 30 + rand()%10;
		obj->net.aiDescIndex = AIDESC_DOG2;
	}
	else
	{
		obj->aiObj->makoValue = 12 + rand()%6;
		obj->net.aiDescIndex = AIDESC_DOG1;
	}

	obj->aiObj->dropChances[INVITEM_POTION] = 100;
	if (obj->generalFlag == 0)
	{
		obj->aiObj->dropChances[INVITEM_VAMPIREFANG] = 20;
	}
	else if (obj->generalFlag == -1)
	{
		obj->aiObj->dropChances[INVITEM_HIPOTION] = 100;
		obj->aiObj->dropChances[INVITEM_VAMPIREFANG] = 50;
	}
	obj->aiObj->maxHealth = obj->health;

	obj->aiObj->dmgEffect = "melee/impacten";
	obj->aiObj->dmgSounds = g_soundHitHeavy;
	obj->aiObj->numDmgSounds = NUM_SOUNDS_HITHEAVY;
	if (obj->generalFlag == -1)
	{
		obj->aiObj->moveSpeed = 400.0f;
	}
	else if (obj->generalFlag == 0)
	{
		obj->aiObj->moveSpeed = 330.0f;
	}
	else
	{
		obj->aiObj->moveSpeed = 200.0f;
	}
	obj->animhandler = ObjDog_PickAnim;
	obj->animTable = g_dogAnims;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjDog_Think;
	obj->attackblock = ObjDog_AttackBlock;
}
