/*
=============================================================================
Module Information
------------------
Name:			rscript_rop.cpp
Author:			Rich Whitehouse
Description:	R-Script R-Op node handlers
=============================================================================
*/

#include "main.h"
#include "rscript.h"

typedef struct ropObjOffset_s
{
	char			*parmName;
	unsigned int	ofs;
	unsigned int	argSize;
} ropObjOffset_t;

#pragma warning(disable : 4311)
#define ROP_OBJ_OFS(a)		{#a, (unsigned int)&((gameObject_t *)0)->a, (unsigned int)sizeof(((gameObject_t *)0)->a)}
static ropObjOffset_t ropOffsets[] =
{
	ROP_OBJ_OFS(hurtable),
	ROP_OBJ_OFS(health),
	ROP_OBJ_OFS(numAnims),
	ROP_OBJ_OFS(curAnim),
	ROP_OBJ_OFS(isOnFire),
	ROP_OBJ_OFS(alwaysSend),
	ROP_OBJ_OFS(canPush),
	ROP_OBJ_OFS(canBePushed),
	ROP_OBJ_OFS(localFlags),
	ROP_OBJ_OFS(net.index),
	ROP_OBJ_OFS(net.type),
	ROP_OBJ_OFS(net.pos[0]),
	ROP_OBJ_OFS(net.pos[1]),
	ROP_OBJ_OFS(net.pos[2]),
	ROP_OBJ_OFS(net.ang[0]),
	ROP_OBJ_OFS(net.ang[1]),
	ROP_OBJ_OFS(net.ang[2]),
	ROP_OBJ_OFS(net.vel[0]),
	ROP_OBJ_OFS(net.vel[1]),
	ROP_OBJ_OFS(net.vel[2]),
	ROP_OBJ_OFS(net.mins[0]),
	ROP_OBJ_OFS(net.mins[1]),
	ROP_OBJ_OFS(net.mins[2]),
	ROP_OBJ_OFS(net.maxs[0]),
	ROP_OBJ_OFS(net.maxs[1]),
	ROP_OBJ_OFS(net.maxs[2]),
	ROP_OBJ_OFS(net.renderEffects),
	ROP_OBJ_OFS(net.renderEffects2),
	ROP_OBJ_OFS(net.effectLen),
	ROP_OBJ_OFS(net.solid),
	ROP_OBJ_OFS(net.owner),
	ROP_OBJ_OFS(net.frame),
	ROP_OBJ_OFS(net.lerpDuration),
	{NULL, 0}
};
#pragma warning(default : 4311)

//get register index
static int RScr_ROP_GetRegister(rscriptNode_t *node, int argIdx)
{
	rscriptArg_t *arg = &node->menuArgs[argIdx];
	if (arg->p[0] != 'r')
	{
		return -1;
	}

	return atoi(&arg->p[1]);
}

//get byte offset for an object parm
static ropObjOffset_t *RScr_ROP_GetObjectOffset(char *objParm)
{
	for (int i = 0; ropOffsets[i].parmName; i++)
	{
		ropObjOffset_t *ofs = &ropOffsets[i];
		if (!strcmp(ofs->parmName, objParm))
		{
			return ofs;
		}
	}
	return NULL;
}

//"rop_jmp" (jump) command
bool RSHan_ROP_JMP(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	char *targ = RScr_GetArgStr(node, rscript, 0);
	rscriptLabel_t *lab = RScr_GetLabel(targ);
	if (!lab)
	{
		Util_ErrorMessage("rop_jmp: bad target: %s", targ);
		return true;
	}

	rscript->retNode = node->next;
	rscript->curNode = lab->node;
	return true;
}

//"rop_jr" (jump return) command
bool RSHan_ROP_JR(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (!rscript->retNode)
	{
		Util_ErrorMessage("rop_jr: no return node");
		return true;
	}

	rscript->curNode = rscript->retNode;
	return true;
}

//"rop_je" (jump if equal) command
bool RSHan_ROP_JE(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	unsigned int c = rscript->ropIntRegs[ROP_INTREG_CMP];
	if (c & ROP_CMPBIT_EQUAL)
	{
		return RSHan_ROP_JMP(node, rscript, timeMod);
	}
	return true;
}

//"rop_jne" (jump if not equal) command
bool RSHan_ROP_JNE(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	unsigned int c = rscript->ropIntRegs[ROP_INTREG_CMP];
	if (!(c & ROP_CMPBIT_EQUAL))
	{
		return RSHan_ROP_JMP(node, rscript, timeMod);
	}
	return true;
}

//"rop_jge" (jump if greater or equal) command
bool RSHan_ROP_JGE(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	unsigned int c = rscript->ropIntRegs[ROP_INTREG_CMP];
	if (c & (ROP_CMPBIT_EQUAL|ROP_CMPBIT_GREATER))
	{
		return RSHan_ROP_JMP(node, rscript, timeMod);
	}
	return true;
}

//"rop_jle" (jump if less or equal) command
bool RSHan_ROP_JLE(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	unsigned int c = rscript->ropIntRegs[ROP_INTREG_CMP];
	if (c & (ROP_CMPBIT_EQUAL|ROP_CMPBIT_LESS))
	{
		return RSHan_ROP_JMP(node, rscript, timeMod);
	}
	return true;
}

//"rop_jg" (jump if greater) command
bool RSHan_ROP_JG(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	unsigned int c = rscript->ropIntRegs[ROP_INTREG_CMP];
	if (c & ROP_CMPBIT_GREATER)
	{
		return RSHan_ROP_JMP(node, rscript, timeMod);
	}
	return true;
}

//"rop_jl" (jump if less) command
bool RSHan_ROP_JL(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	unsigned int c = rscript->ropIntRegs[ROP_INTREG_CMP];
	if (c & ROP_CMPBIT_LESS)
	{
		return RSHan_ROP_JMP(node, rscript, timeMod);
	}
	return true;
}

//"rop_push" (push string) command
bool RSHan_ROP_PUSH(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	char *str = RScr_GetArgStr(node, rscript, 0);
	if (rscript->ropStackPtr <= 0)
	{
		Util_ErrorMessage("rop_push: stack overflow pushing %s", str);
		return true;
	}
	rscript->ropStack[rscript->ropStackPtr] = str;
	rscript->ropStackPtr--;
	return true;
}

//"rop_pop" (pop string) command
bool RSHan_ROP_POP(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (rscript->ropStackPtr >= MAX_RSCRIPT_STACK-1)
	{
		Util_ErrorMessage("rop_pop: stack underflow");
		return true;
	}
	rscript->ropStackPtr++;
	return true;
}

//"rop_popgvar" (pop string to gvar) command
bool RSHan_ROP_POPGVAR(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (rscript->ropStackPtr >= MAX_RSCRIPT_STACK-1)
	{
		Util_ErrorMessage("rop_pop: stack underflow");
		return true;
	}
	rscript->ropStackPtr++;
	GVar_SetValue(RScr_GetArgStr(node, rscript, 0), rscript->ropStack[rscript->ropStackPtr]);
	return true;
}

//"rop_ldri" (load register immediate int) command
bool RSHan_ROP_LDRI(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_ldr: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		rscript->ropRegs[regNum] = atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		rscript->ropRegs[regNum] = rscript->ropRegs[srcReg];
	}

	return true;
}

//"rop_ldru" (load register immediate unsigned int) command
bool RSHan_ROP_LDRU(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_ldr: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		rscript->ropRegs[regNum] = (unsigned int)atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		rscript->ropRegs[regNum] = rscript->ropRegs[srcReg];
	}

	return true;
}

//"rop_ldrf" (load register immediate float) command
bool RSHan_ROP_LDRF(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_ldr: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		float *fdst = (float *)&rscript->ropRegs[regNum];
		(*fdst) = (float)atof(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		rscript->ropRegs[regNum] = rscript->ropRegs[srcReg];
	}

	return true;
}

//"rop_ldro" (load register object parameter) command
bool RSHan_ROP_LDRO(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_ldro: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	char *targ = RScr_GetArgStr(node, rscript, 2);
	ropObjOffset_t *ofs = RScr_ROP_GetObjectOffset(targ);
	if (!ofs)
	{
		Util_ErrorMessage("rop_ldro: bad offset name: %s", targ);
		return true;
	}

	char *objTarg = RScr_GetArgStr(node, rscript, 1);
	gameObject_t *obj = Util_FindTargetObject(objTarg);
	if (!obj)
	{
		rscript->ropRegs[regNum] = 0;
		return true;
	}

	BYTE *rawMem = ((BYTE *)obj)+ofs->ofs;
	unsigned int val = *((unsigned int *)rawMem);
	unsigned int argBitMask = (0xFFFFFFFF>>(32-ofs->argSize*8));
	rscript->ropRegs[regNum] = (val & argBitMask);
	return true;
}

//"rop_ldrinv" (load register inventory) command
bool RSHan_ROP_LDRINV(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_ldrinv: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	char *itemName = RScr_GetArgStr(node, rscript, 1);
	int itemIndex = -1;
	for (int i = 0; i < NUM_INVENTORY_ITEM_DEFS; i++)
	{
		const invItemDef_t *item = &g_invItems[i];
		if (!stricmp(item->name, itemName))
		{
			itemIndex = i;
			break;
		}
	}
	if (itemIndex == -1)
	{
		return true;
	}

	gameObject_t *obj = &g_gameObjects[0];
	if (!obj->inuse || !obj->plObj)
	{
		return true;
	}

	rscript->ropRegs[regNum] = 0;
	for (int i = 0; i < MAX_PLAYER_INVENTORY_ITEMS; i++)
	{
		playerInvItem_t *plItem = &obj->plObj->plData.inventory[i];
		if (plItem->itemIndex == itemIndex)
		{
			rscript->ropRegs[regNum] = plItem->itemQuantity;
			break;
		}
	}
	return true;
}

//"rop_stro" (store register object parameter) command
bool RSHan_ROP_STRO(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_stro: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	char *targ = RScr_GetArgStr(node, rscript, 2);
	ropObjOffset_t *ofs = RScr_ROP_GetObjectOffset(targ);
	if (!ofs)
	{
		Util_ErrorMessage("rop_stro: bad offset name: %s", targ);
		return true;
	}

	char *objTarg = RScr_GetArgStr(node, rscript, 1);
	gameObject_t *obj = Util_FindTargetObject(objTarg);
	if (!obj)
	{
		rscript->ropRegs[regNum] = 0;
		return true;
	}

	BYTE *rawMem = ((BYTE *)obj)+ofs->ofs;
	unsigned int argBitMask = (0xFFFFFFFF>>(32-ofs->argSize*8));
	switch (ofs->argSize)
	{
	case 1:
		*rawMem = (BYTE)(rscript->ropRegs[regNum] & argBitMask);
		break;
	case 2:
		*((WORD *)rawMem) = (WORD)(rscript->ropRegs[regNum] & argBitMask);
		break;
	case 4:
		*((DWORD *)rawMem) = (DWORD)(rscript->ropRegs[regNum] & argBitMask);
		break;
	default:
		Util_ErrorMessage("rop_stro: bad size: %i", ofs->argSize);
		break;
	}

	return true;
}

//"rop_strgi" (store register gvar int) command
bool RSHan_ROP_STRGI(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_strgi: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int *val = (int *)&rscript->ropRegs[regNum];
	GVar_SetInt(RScr_GetArgStr(node, rscript, 1), *val);
	return true;
}

//"rop_strgf" (store register gvar float) command
bool RSHan_ROP_STRGF(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_strgf: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	float *val = (float *)&rscript->ropRegs[regNum];
	GVar_SetFloat(RScr_GetArgStr(node, rscript, 1), *val);
	return true;
}

//"rop_addi" (add integer) command
bool RSHan_ROP_ADDI(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_addi: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int *i = (int *)&rscript->ropRegs[regNum];
	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		(*i) += atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		(*i) += (int)rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_addu" (add unsigned integer) command
bool RSHan_ROP_ADDU(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_addu: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		rscript->ropRegs[regNum] += (unsigned int)atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		rscript->ropRegs[regNum] += rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_addf" (add float) command
bool RSHan_ROP_ADDF(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_addf: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	float *f = (float *)&rscript->ropRegs[regNum];
	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		(*f) += (float)atof(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		float *fsrc = (float *)&rscript->ropRegs[srcReg];
		(*f) += (*fsrc);
	}
	return true;
}

//"rop_subi" (sub integer) command
bool RSHan_ROP_SUBI(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_subi: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int *i = (int *)&rscript->ropRegs[regNum];
	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		(*i) -= atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		(*i) -= (int)rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_subu" (sub unsigned integer) command
bool RSHan_ROP_SUBU(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_subu: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		rscript->ropRegs[regNum] -= (unsigned int)atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		rscript->ropRegs[regNum] -= rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_subf" (sub float) command
bool RSHan_ROP_SUBF(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_subf: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	float *f = (float *)&rscript->ropRegs[regNum];
	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		(*f) -= (float)atof(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		float *fsrc = (float *)&rscript->ropRegs[srcReg];
		(*f) -= (*fsrc);
	}
	return true;
}

//"rop_muli" (mul integer) command
bool RSHan_ROP_MULI(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_muli: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int *i = (int *)&rscript->ropRegs[regNum];
	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		(*i) *= atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		(*i) *= (int)rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_mulu" (mul unsigned integer) command
bool RSHan_ROP_MULU(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_mulu: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		rscript->ropRegs[regNum] *= (unsigned int)atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		rscript->ropRegs[regNum] *= rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_mulf" (mul float) command
bool RSHan_ROP_MULF(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_mulf: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	float *f = (float *)&rscript->ropRegs[regNum];
	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		(*f) *= (float)atof(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		float *fsrc = (float *)&rscript->ropRegs[srcReg];
		(*f) *= (*fsrc);
	}
	return true;
}

//"rop_divi" (div integer) command
bool RSHan_ROP_DIVI(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_muli: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int *i = (int *)&rscript->ropRegs[regNum];
	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		(*i) /= atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		(*i) /= (int)rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_divu" (div unsigned integer) command
bool RSHan_ROP_DIVU(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_divu: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		rscript->ropRegs[regNum] /= (unsigned int)atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		rscript->ropRegs[regNum] /= rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_divf" (div float) command
bool RSHan_ROP_DIVF(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_divf: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	float *f = (float *)&rscript->ropRegs[regNum];
	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		(*f) /= (float)atof(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		float *fsrc = (float *)&rscript->ropRegs[srcReg];
		(*f) /= (*fsrc);
	}
	return true;
}
//"rop_shl" (shift left) command
bool RSHan_ROP_SHL(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_shl: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		rscript->ropRegs[regNum] <<= (unsigned int)atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		rscript->ropRegs[regNum] <<= rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_shr" (shift right) command
bool RSHan_ROP_SHR(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_shr: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		rscript->ropRegs[regNum] >>= (unsigned int)atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		rscript->ropRegs[regNum] >>= rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_and" (bitwise and) command
bool RSHan_ROP_AND(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_and: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		rscript->ropRegs[regNum] &= (unsigned int)atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		rscript->ropRegs[regNum] &= rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_or" (bitwise or) command
bool RSHan_ROP_OR(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_or: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		rscript->ropRegs[regNum] |= (unsigned int)atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		rscript->ropRegs[regNum] |= rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_xor" (exclusive or) command
bool RSHan_ROP_XOR(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_xor: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		rscript->ropRegs[regNum] ^= (unsigned int)atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		rscript->ropRegs[regNum] ^= rscript->ropRegs[srcReg];
	}
	return true;
}

//"rop_cmpi" (compare integer) command
bool RSHan_ROP_CMPI(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_cmpi: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	int a = *((int *)&rscript->ropRegs[regNum]);
	int b;
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		b = atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		b = *((int *)&rscript->ropRegs[srcReg]);
	}

	if (a == b)
	{
		rscript->ropIntRegs[ROP_INTREG_CMP] = ROP_CMPBIT_EQUAL;
	}
	else if (a > b)
	{
		rscript->ropIntRegs[ROP_INTREG_CMP] = ROP_CMPBIT_GREATER;
	}
	else
	{
		rscript->ropIntRegs[ROP_INTREG_CMP] = ROP_CMPBIT_LESS;
	}

	return true;
}

//"rop_cmpu" (compare unsigned integer) command
bool RSHan_ROP_CMPU(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_cmpu: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	unsigned int a = rscript->ropRegs[regNum];
	unsigned int b;
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		b = (unsigned int)atoi(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		b = rscript->ropRegs[srcReg];
	}

	if (a == b)
	{
		rscript->ropIntRegs[ROP_INTREG_CMP] = ROP_CMPBIT_EQUAL;
	}
	else if (a > b)
	{
		rscript->ropIntRegs[ROP_INTREG_CMP] = ROP_CMPBIT_GREATER;
	}
	else
	{
		rscript->ropIntRegs[ROP_INTREG_CMP] = ROP_CMPBIT_LESS;
	}

	return true;
}

//"rop_cmpf" (compare float) command
bool RSHan_ROP_CMPF(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int regNum = RScr_ROP_GetRegister(node, 0);
	if (regNum < 0 || regNum >= MAX_RSCRIPT_REGISTERS)
	{
		Util_ErrorMessage("rop_cmpf: bad target: %s", node->menuArgs[0].p);
		return true;
	}

	int srcReg = RScr_ROP_GetRegister(node, 1);
	float a = *((float *)&rscript->ropRegs[regNum]);
	float b;
	if (srcReg < 0 || srcReg >= MAX_RSCRIPT_REGISTERS)
	{ //immediate value
		b = (float)atof(RScr_GetArgStr(node, rscript, 1));
	}
	else
	{
		b = *((float *)&rscript->ropRegs[srcReg]);
	}

	if (a == b)
	{
		rscript->ropIntRegs[ROP_INTREG_CMP] = ROP_CMPBIT_EQUAL;
	}
	else if (a > b)
	{
		rscript->ropIntRegs[ROP_INTREG_CMP] = ROP_CMPBIT_GREATER;
	}
	else
	{
		rscript->ropIntRegs[ROP_INTREG_CMP] = ROP_CMPBIT_LESS;
	}

	return true;
}

//"rop_cmps" (compare string) command
bool RSHan_ROP_CMPS(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	char *s1 = RScr_GetArgStr(node, rscript, 0);
	char *s2 = RScr_GetArgStr(node, rscript, 1);
	int r = strcmp(s1, s2);
	if (r == 0)
	{
		rscript->ropIntRegs[ROP_INTREG_CMP] = ROP_CMPBIT_EQUAL;
	}
	else if (r > 0)
	{
		rscript->ropIntRegs[ROP_INTREG_CMP] = ROP_CMPBIT_GREATER;
	}
	else
	{
		rscript->ropIntRegs[ROP_INTREG_CMP] = ROP_CMPBIT_LESS;
	}
	return true;
}
