/*
=============================================================================
Module Information
------------------
Name:			ai_combat.cpp
Author:			Rich Whitehouse
Description:	ai combat routines
=============================================================================
*/

#include "main.h"
#include "ai.h"

//modify damage for enemy level
int AI_LevelDamage(gameObject_t *obj, int dmg)
{
	if (!obj->aiObj || !obj->aiObj->aiLevel)
	{
		return dmg;
	}

	float f = (float)obj->aiObj->aiLevel/8.0f;
	dmg = (int)((float)dmg * f);

	return dmg;
}

//see if a damage collision is reasonable
bool AI_ValidDamageCollision(gameObject_t *obj, collObj_t *col)
{
	if (!col->hit || col->hitObjectIndex < 0 || col->hitObjectIndex >= MAX_GAME_OBJECTS)
	{
		return false;
	}

	gameObject_t *other = &g_gameObjects[col->hitObjectIndex];
	if (!other->inuse || !other->hurtable)
	{
		return false;
	}

	if (!Util_ValidEnemies(obj, other))
	{
		return false;
	}

	if (!Util_SequenceDamage(obj, other))
	{ //already hit this thing in this sequence
		return false;
	}

	modelMatrix_t boneMat;
	if (g_sharedFn->Coll_GetModelBoneMatrix(obj->rcColModel, "b0", &boneMat))
	{ //if it has a standard root bone, make sure the enemy is roughly in front of the model
		float fwd[3] = {0.0f, 0.0f, -1.0f};
		float posFwd[3];
		Math_TransformPointByMatrix(&boneMat, fwd, posFwd);

		float dirFwd[3], dirOther[3];
		Math_VecSub(other->net.pos, obj->net.pos, dirOther);
		Math_VecSub(posFwd, boneMat.o, dirFwd);
		Math_VecNorm(dirOther);
		Math_VecNorm(dirFwd);
		float dp = Math_DotProduct(dirFwd, dirOther);
		if (dp < -0.25f)
		{
			return false;
		}
	}

	return true;
}

//get damage bone positions
void AI_TransformDamageBones(gameObject_t *obj, damageBone_t *damageBones, dmgBoneLine_t *dmgPos, int numDamageBones)
{
	for (int i = 0; i < numDamageBones; i++)
	{
		damageBone_t *b = damageBones+i;
		modelMatrix_t boneMat;
		if (!b->boneName || !b->boneName[0])
		{
			gameObject_t *sword = NULL;
			char *boneName = "b32";
			float ofs[3] = {0.0f, 0.0f, -90.0f};
			if (obj->plObj)
			{
				sword = obj->plObj->sword;
			}
			else if (obj->chain && obj->chain->swordObj)
			{
				if (obj->aiObj && obj->aiObj->swordBone)
				{
					boneName = obj->aiObj->swordBone;
					Math_VecCopy(obj->aiObj->swordOfs, ofs);
				}
				sword = obj->chain;
			}
			dmgPos[i].parPos[0] = obj->net.pos[0];
			dmgPos[i].parPos[1] = obj->net.pos[1];
			dmgPos[i].parPos[2] = obj->net.pos[2];
			dmgPos[i].pos[0] = obj->net.pos[0];
			dmgPos[i].pos[1] = obj->net.pos[1];
			dmgPos[i].pos[2] = obj->net.pos[2];
			if (sword)
			{ //sword transform
				float hiltLen = sword->spareVec[0];
				float bladeLen = sword->spareVec[1];
				if (Util_GetObjBolt(obj, boneName, &boneMat, ofs))
				{
					Math_VecCopy(boneMat.o, dmgPos[i].parPos);
					dmgPos[i].parPos[0] += boneMat.x1[0]*b->offset[0]*hiltLen;
					dmgPos[i].parPos[1] += boneMat.x1[1]*b->offset[0]*hiltLen;
					dmgPos[i].parPos[2] += boneMat.x1[2]*b->offset[0]*hiltLen;
					dmgPos[i].parPos[0] += boneMat.x2[0]*b->offset[1]*hiltLen;
					dmgPos[i].parPos[1] += boneMat.x2[1]*b->offset[1]*hiltLen;
					dmgPos[i].parPos[2] += boneMat.x2[2]*b->offset[1]*hiltLen;
					dmgPos[i].parPos[0] += boneMat.x3[0]*b->offset[2]*hiltLen;
					dmgPos[i].parPos[1] += boneMat.x3[1]*b->offset[2]*hiltLen;
					dmgPos[i].parPos[2] += boneMat.x3[2]*b->offset[2]*hiltLen;
					Math_VecCopy(dmgPos[i].parPos, dmgPos[i].pos);
					dmgPos[i].pos[0] += boneMat.x1[0]*b->offset[0]*bladeLen;
					dmgPos[i].pos[1] += boneMat.x1[1]*b->offset[0]*bladeLen;
					dmgPos[i].pos[2] += boneMat.x1[2]*b->offset[0]*bladeLen;
					dmgPos[i].pos[0] += boneMat.x2[0]*b->offset[1]*bladeLen;
					dmgPos[i].pos[1] += boneMat.x2[1]*b->offset[1]*bladeLen;
					dmgPos[i].pos[2] += boneMat.x2[2]*b->offset[1]*bladeLen;
					dmgPos[i].pos[0] += boneMat.x3[0]*b->offset[2]*bladeLen;
					dmgPos[i].pos[1] += boneMat.x3[1]*b->offset[2]*bladeLen;
					dmgPos[i].pos[2] += boneMat.x3[2]*b->offset[2]*bladeLen;
				}
			}
			continue;
		}
		if (g_sharedFn->Coll_GetModelBoneMatrix(obj->rcColModel, b->boneName, &boneMat))
		{
			Math_TransformPointByMatrix(&boneMat, b->offset, dmgPos[i].pos);
		}
		else
		{
			Math_VecCopy(obj->net.pos, dmgPos[i].pos);
		}

		if (g_sharedFn->Coll_GetModelBoneMatrix(obj->rcColModel, b->parentName, &boneMat))
		{
			Math_VecCopy(boneMat.o, dmgPos[i].parPos);
		}
		else
		{
			Math_VecCopy(obj->net.pos, dmgPos[i].parPos);
		}
	}
}

//returns index of object hit, -1 if none
int AI_RunDamageBone(gameObject_t *obj, damageBone_t *dmgBone, dmgBoneLine_t *linePrev, dmgBoneLine_t *lineCur, int dmg)
{
	collObj_t col;
	float start[3], end[3];
	float boneRadius = dmgBone->radius;
	bool isSword = false;
	gameObject_t *sword = NULL;
	if (!dmgBone->boneName || !dmgBone->boneName[0])
	{
		if (obj->plObj)
		{
			sword = obj->plObj->sword;
		}
		else if (obj->chain && obj->chain->swordObj)
		{
			sword = obj->chain;
		}
		if (sword)
		{
			sword->swordTime = g_gameTime;
			isSword = true;
			boneRadius = sword->spareVec[2];
			if (!obj->aiObj || obj->net.aiDescIndex != AIDESC_SEPHFIN)
			{ //slight hack
				dmg = sword->generalFlag;
			}
			if (sword->generalFlag2 && obj->plObj)
			{ //an evolving blade
				int bladeEvo = (int)obj->plObj->plData.statStr + (int)obj->plObj->plData.statDef +
					(int)obj->plObj->plData.statDex + (int)obj->plObj->plData.statLuck;
				dmg = (int)((float)dmg * ((float)bladeEvo / 100.0f));
				if (dmg < 1)
				{
					dmg = 1;
				}
			}
		}
	}
	Math_VecCopy(lineCur->parPos, start);
	Math_VecCopy(lineCur->pos, end);
	g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, start, end, boneRadius, NULL, false);
	if (!AI_ValidDamageCollision(obj, &col))
	{ //try previous pos to current
		Math_VecCopy(linePrev->pos, start);
		Math_VecCopy(lineCur->pos, end);
		g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, start, end, boneRadius, NULL, false);
		if (!AI_ValidDamageCollision(obj, &col))
		{ //try previous parent pos to current
			Math_VecCopy(linePrev->parPos, start);
			Math_VecCopy(lineCur->pos, end);
			g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, start, end, boneRadius, NULL, false);
		}
	}

	if (!AI_ValidDamageCollision(obj, &col))
	{
		return -1;
	}

	collObj_t preCol = col;
	collObj_t closerCol;
	float d[3];
	Math_VecSub(end, start, d);
	Math_VecScale(d, 2.0f);
	Math_VecAdd(end, d, end);
	//Util_DebugLine(start, end);
	g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &closerCol, start, end, 1.0f, g_gameObjects[col.hitObjectIndex].rcColModel, false);
	if (closerCol.hit && closerCol.hitObjectIndex == col.hitObjectIndex)
	{
		col = closerCol;
	}
	else
	{
		Math_VecCopy(lineCur->parPos, start);
		Util_GameObjectCenter(&g_gameObjects[col.hitObjectIndex], end);
		end[2] = lineCur->pos[2];
		g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &closerCol, start, end, 1.0f, g_gameObjects[col.hitObjectIndex].rcColModel, false);
		if (closerCol.hit && closerCol.hitObjectIndex == col.hitObjectIndex)
		{
			col = closerCol;
		}
		else
		{
			Util_GameObjectCenter(&g_gameObjects[col.hitObjectIndex], end);
			Math_VecSub(end, start, d);
			Math_VecScale(d, 2.0f);
			start[0] -= d[0];
			start[1] -= d[1];
			start[2] -= d[2];
			end[0] += d[0];
			end[1] += d[1];
			end[2] += d[2];
			end[2] += g_gameObjects[col.hitObjectIndex].net.maxs[2]*0.5f;
			g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &closerCol, start, end, 1.0f, g_gameObjects[col.hitObjectIndex].rcColModel, false);
			if (closerCol.hit && closerCol.hitObjectIndex == col.hitObjectIndex)
			{
				col = closerCol;
			}
		}
	}

	bool explosiveBlows = false;
	if (obj->plObj)
	{
		if (obj->plObj->explosiveBlows)
		{
			explosiveBlows = true;
		}
		else
		{
			const invItemDef_t *item = ObjPlayer_GetEquippedItem(obj);
			if (item && item->itemBehavior == INVITEM_BEHAVIOR_FINALHEAVEN)
			{
				explosiveBlows = true;
			}
		}
	}

	gameObject_t *other = &g_gameObjects[col.hitObjectIndex];
	Util_SetSequenceDamage(obj, other);

	float ang[3], impactPos[3];
	Math_VecSub(col.endPos, lineCur->pos, d);
	Math_VecScale(d, 0.5f);
	Math_VecAdd(lineCur->pos, d, impactPos);
	Math_VecToAngles(col.endNormal, ang);

	if (other && other->attackblock && other->attackblock(other, obj, dmg, &preCol, impactPos, ang))
	{
		if (explosiveBlows)
		{
			ObjParticles_Create("melee/impactexplosive", impactPos, ang, -1);
			ObjSound_Create(impactPos, "assets/sound/cb/hitaura.wav", 1.0f, -1);
			Util_RadiusDamage(obj, dmg, 900.0f);
		}
		return col.hitObjectIndex;
	}

	if (other && other->aiObj)
	{
		other->aiObj->boneDmgTime = g_gameTime;
		Math_VecSub(lineCur->pos, linePrev->pos, other->aiObj->boneDmgDir);
		if (other->aiObj->boneDmgDir[0] == 0.0f &&
			other->aiObj->boneDmgDir[1] == 0.0f &&
			other->aiObj->boneDmgDir[2] == 0.0f)
		{
			Math_VecSub(lineCur->pos, lineCur->parPos, other->aiObj->boneDmgDir);
		}
	}

	int snd;
	if (obj->aiObj && obj->aiObj->dmgSounds)
	{
		snd = obj->aiObj->dmgSounds[rand()%obj->aiObj->numDmgSounds];
	}
	else if (isSword)
	{
		if (sword && sword->swordObj &&
			sword->swordObj->numHitSounds > 0)
		{
			int numHitSounds = sword->swordObj->numHitSounds;
			snd = sword->swordObj->hitSounds[rand()%numHitSounds];
		}
		else
		{
			char *s[3] =
			{
				"$assets/sound/cb/buscut01.wav",
				"$assets/sound/cb/buscut02.wav",
				"$assets/sound/cb/buscut03.wav"
			};
			snd = g_sharedFn->Common_ServerString(s[rand()%3]);
		}
	}
	else if (dmg >= 35 || (obj->atkType > ATKTYPE_NORMAL && obj->atkType != ATKTYPE_INAIR) || explosiveBlows)
	{
		snd = g_soundHitHeavy[rand()%NUM_SOUNDS_HITHEAVY];
	}
	else
	{
		snd = g_soundHitLight[rand()%NUM_SOUNDS_HITLIGHT];
	}

	ObjSound_CreateFromIndex(impactPos, snd, 1.0f, -1);

	if (explosiveBlows)
	{
		ObjParticles_Create("melee/impactexplosive", impactPos, ang, -1);
		ObjSound_Create(impactPos, "assets/sound/cb/hitaura.wav", 1.0f, -1);
		Util_DamageObject(obj, other, ObjPlayer_StrMod(obj, dmg*2), &col);
		int oldHurt = other->hurtable;
		other->hurtable = 0;
		Util_RadiusDamage(obj, dmg, 900.0f);
		other->hurtable = oldHurt;
	}
	else if (obj->aiObj && obj->aiObj->dmgEffect)
	{
		ObjParticles_Create(obj->aiObj->dmgEffect, impactPos, ang, -1);
		Util_DamageObject(obj, other, ObjPlayer_StrMod(obj, dmg), &col);
	}
	else if (isSword)
	{
		if (sword &&
			sword->swordObj && sword->swordObj->numHitFX > 0)
		{
			int fxIdx = sword->swordObj->hitFX[rand()%sword->swordObj->numHitFX];
			ObjParticles_CreateFromIndex(fxIdx, impactPos, ang, -1);
		}
		else
		{
			ObjParticles_Create("melee/bigslash", impactPos, ang, -1);
		}
		Util_DamageObject(obj, other, ObjPlayer_StrMod(obj, dmg), &col);
	}
	else
	{
		ObjParticles_Create("melee/impact", impactPos, ang, -1);
		Util_DamageObject(obj, other, ObjPlayer_StrMod(obj, dmg), &col);
	}

	return col.hitObjectIndex;
}
