/*
=============================================================================
Module Information
------------------
Name:			ai_logic.cpp
Author:			Rich Whitehouse
Description:	ai general logic routines
=============================================================================
*/

#include "main.h"
#include "ai.h"

static serverTime_t g_pathTime = 0;
aiGlobals_t g_ai;

//ai logic which can be shared by all ai throughout the frame
void AI_GlobalFrame(void)
{
	g_ai.playerCloseDist = 700.0f;

	float closestDist = 0.0f;
	g_ai.closestToPlayer = NULL;
	g_ai.numActiveAI = 0;
	g_ai.numActiveEnemyAI = 0;
	g_ai.numCloseToPlayer = 0;
	g_ai.numWantClose = 2+rand()%6;//3;
	gameObject_t *pl = &g_gameObjects[0];
	if (pl->inuse && pl->plObj)
	{
		for (int i = 0; i < g_gameObjectSlots; i++)
		{
			gameObject_t *obj = &g_gameObjects[i];
			if (!obj->inuse || !obj->aiObj || obj->aiObj->nonActive)
			{
				continue;
			}
			g_ai.numActiveAI++;
			if (obj->localFlags & LFL_ENEMY)
			{
				g_ai.numActiveEnemyAI++;
			}
			float d[3];
			Math_VecSub(obj->net.pos, pl->net.pos, d);
			float l = Math_VecLen(d);
			if (l <= g_ai.playerCloseDist)
			{
				g_ai.numCloseToPlayer++;
			}
			if (!g_ai.closestToPlayer || l < closestDist)
			{
				g_ai.closestToPlayer = obj;
				closestDist = l;
			}
		}
	}
}

//given global considerations, should i close in for attack?
bool AI_ShouldCloseIn(gameObject_t *obj)
{
	if (obj->aiObj->enemy != &g_gameObjects[0])
	{
		return true;
	}
	int closeVal = g_ai.numWantClose;//(obj->aiObj->aiLevel < 15) ? g_ai.numWantCloseWeak : g_ai.numWantClose;
	if (g_ai.numCloseToPlayer < closeVal)
	{
		return true;
	}

	if (obj->aiObj->distToEnemy <= g_ai.playerCloseDist)
	{
		return true;
	}
	return false;
}

//see if ai is facing enemy
bool AI_FacingEnemy(gameObject_t *obj, gameObject_t *other, float minDot)
{
	if (!other || !other->inuse)
	{
		return false;
	}

	float c1[3], c2[3];
	Util_GameObjectCenter(obj, c1);
	Util_GameObjectCenter(other, c2);

	float d1[3], d2[3];
	Math_VecSub(c2, c1, d1);
	Math_VecNorm(d1);
	Math_AngleVectors(obj->net.ang, d2, 0, 0);
	if (Math_DotProduct(d1, d2) > minDot)
	{
		return true;
	}

	return false;
}

//if true, will take more expensive pathing options
bool AI_HighPriorityGoal(gameObject_t *obj)
{
	if (obj->localFlags & LFL_ENEMY)
	{
		return false;
	}

	if (obj->aiObj->scriptGoal && obj->aiObj->scriptGoal->inuse)
	{
		return true;
	}
	return false;
}

//handle charting paths and moving to the goal position
void AI_GetToGoal(gameObject_t *obj, float timeMod)
{
	if (obj->aiObj->goalObj == obj)
	{
		return;
	}

	float d[3];
	bool goalObstructed = true;
	Math_VecSub(obj->aiObj->goalPos, obj->net.pos, d);
	obj->aiObj->distToGoal = Math_VecLen(d);
	if (obj->aiObj->obstructTime < g_gameTime)
	{
		if (obj->aiObj->distToGoal < 4096.0f && AI_CanReachPoint(obj, obj->aiObj->goalPos, obj->aiObj->goalObj))
		{
			goalObstructed = false;
		}
		else if (!AI_ShouldCloseIn(obj))
		{
			goalObstructed = false;
		}
	}

	if (!obj->aiObj->path && goalObstructed)
	{
		if (AI_HighPriorityGoal(obj) || (g_pathTime < g_gameTime && rand()%15 < 2))
		{
			if (obj->aiObj->obstructTime > g_gameTime)
			{
				float rp[3];
				rp[0] = obj->aiObj->goalPos[0] + Util_RandFloat(-2000.0f, 2000.0f);
				rp[1] = obj->aiObj->goalPos[1] + Util_RandFloat(-2000.0f, 2000.0f);
				rp[2] = obj->aiObj->goalPos[2];
				obj->aiObj->path = AI_ChartPath(obj, rp);
			}
			else
			{
				obj->aiObj->path = AI_ChartPath(obj, obj->aiObj->goalPos);
			}
			obj->aiObj->pathNode = obj->aiObj->path->nodes;
			g_pathTime = g_gameTime+100;
		}
		else
		{ //don't want to path again yet, try to steal someone else's path
			obj->aiObj->path = AI_BorrowPath(obj, obj->aiObj->goalPos);
			if (obj->aiObj->path)
			{
				obj->aiObj->pathNode = AI_ClosestPathNode(obj, obj->aiObj->goalPos, obj->aiObj->path);
				if (obj->aiObj->pathNode && !AI_CanReachPoint(obj, obj->aiObj->pathNode->pos, NULL))
				{ //nevermind
					AI_FreePath(obj->aiObj->path);
					obj->aiObj->path = NULL;
					obj->aiObj->pathNode = NULL;
				}
			}
		}
	}

	if (obj->aiObj->path && obj->aiObj->pathCheckTime < g_gameTime)
	{ //check path obstructions
		if (obj->aiObj->pathNode)
		{
			if (!AI_CanReachPoint(obj, obj->aiObj->pathNode->pos, NULL))
			{
				if (!goalObstructed)
				{ //if it can get right to the goal, don't care about the path
					AI_FreePath(obj->aiObj->path);
					obj->aiObj->path = NULL;
					obj->aiObj->pathNode = NULL;
				}
				else
				{
					if (AI_HighPriorityGoal(obj) || (g_pathTime < g_gameTime && rand()%7 < 2))
					{
						AI_FreePath(obj->aiObj->path);
						obj->aiObj->path = AI_ChartPath(obj, obj->aiObj->goalPos);
						obj->aiObj->pathNode = obj->aiObj->path->nodes;
						g_pathTime = g_gameTime+100;
					}
					else
					{
						aiPathNode_t *b = AI_ClosestPathNode(obj, obj->aiObj->goalPos, obj->aiObj->path);
						if (b)
						{
							obj->aiObj->pathNode = b;
						}
					}
				}
			}
			obj->aiObj->pathCheckTime = g_gameTime+1000;
		}
	}

	Math_VecSub(obj->aiObj->goalPos, obj->net.pos, d);
	float distToGoalXY = sqrtf(d[0]*d[0] + d[1]*d[1]);

	if (obj->aiObj->goalObj && obj->aiObj->enemy == obj->aiObj->goalObj &&
		!obj->aiObj->goalObj->onGround && distToGoalXY < 512.0f)
	{ //back off when the enemy is going to land on me
		Math_VecNorm(d);
		d[0] = -d[0];
		d[1] = -d[1];
		AI_Move(obj, timeMod, d);
	}
	else if (obj->aiObj->pathNode)
	{
		if (g_ai.showPaths)
		{
			for (aiPathNode_t *pn = obj->aiObj->path->nodes; pn; pn = pn->next)
			{
				if (pn->next)
				{
					Util_DebugLine(pn->pos, pn->next->pos);
				}
			}
		}
		Math_VecSub(obj->aiObj->pathNode->pos, obj->net.pos, d);
		d[2] = 0.0f;
		float nodeDist = Math_VecNorm(d);
		if (nodeDist < obj->aiObj->moveSpeed*0.25f)
		{
			obj->aiObj->pathNode = obj->aiObj->pathNode->next;
		}

		if (obj->aiObj->pathNode)
		{
			if (goalObstructed || distToGoalXY > obj->aiObj->goalRange)
			{
				AI_Move(obj, timeMod, d);
			}
		}
		else
		{ //done with the path
			AI_FreePath(obj->aiObj->path);
			obj->aiObj->path = NULL;
		}
	}
	else
	{ //no path, go straight to goal
		if (distToGoalXY > obj->aiObj->goalRange)
		{
			if (AI_ShouldCloseIn(obj))
			{
				Math_VecNorm(d);
				AI_Move(obj, timeMod, d, true);
			}
		}
		else if (distToGoalXY < (obj->aiObj->goalRange*0.75f))
		{
			Math_VecNorm(d);
			d[0] = -d[0];
			d[1] = -d[1];
			AI_Move(obj, timeMod, d, true);
		}
	}
}

//validate the current enemy
bool AI_ValidateEnemy(gameObject_t *obj, gameObject_t *other)
{
	if (!other)
	{
		return false;
	}

	if (!other->inuse || !other->hurtable || other->health <= 0)
	{
		return false;
	}
	if (!Util_ValidEnemies(obj, other))
	{ //same side
		return false;
	}

	if (other->plObj && other->plObj->noTarget)
	{
		return false;
	}

	float c1[3], c2[3];
	Util_GameObjectCenter(obj, c1);
	Util_GameObjectCenter(other, c2);

	if (!g_sharedFn->Coll_GeneralVisibilityWithBounds(c1, c2, obj->radius, obj->spawnMins, obj->spawnMaxs))
	{ //not in the same view cluster
		return false;
	}

	collObj_t col;
	collOpt_t opt;
	memset(&opt, 0, sizeof(opt));
	opt.ignoreBoxModels = true;
	g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, c1, c2, 0.0f, NULL, &opt);
	if (col.hit && col.hitObjectIndex != other->net.index)
	{ //no direct line of sight
		return false;
	}

	if (obj->aiObj && other != obj->aiObj->enemy)
	{
		if (obj->aiObj->aiFov)
		{
			float d[3];
			d[0] = c2[0]-c1[0];
			d[1] = c2[1]-c1[1];
			d[2] = 0.0f;
			Math_VecNorm(d);
			float fwd[3];
			Math_AngleVectors(obj->net.ang, fwd, 0, 0);
			float dp = 1.0f-Math_DotProduct(d, fwd);
			if (dp*180.0f > obj->aiObj->aiFov)
			{
				return false;
			}
		}
		if (obj->aiObj->aiRange)
		{
			float d[3];
			Math_VecSub(c2, c1, d);
			if (Math_VecLen(d) > obj->aiObj->aiRange)
			{
				return false;
			}
		}
	}

	return true;
}

//find a suitable enemy
gameObject_t *AI_GetEnemy(gameObject_t *obj)
{
	float d[3];
	float c1[3], c2[3];
	Util_GameObjectCenter(obj, c1);
	const float enemyChangeDist = 500.0f;
	float bestEnemyDist;
	float bestEnemyCenter[3];
	gameObject_t *bestEnemy = obj->aiObj->enemy;
	if (bestEnemy)
	{
		Util_GameObjectCenter(bestEnemy, c2);
		Math_VecSub(c1, c2, d);
		bestEnemyDist = Math_VecLen(d);
	}
	for (int i = 0; i < g_gameObjectSlots; i++)
	{
		gameObject_t *other = &g_gameObjects[i];
		if (!AI_ValidateEnemy(obj, other))
		{
			continue;
		}

		Util_GameObjectCenter(other, c2);
		Math_VecSub(c1, c2, d);
		float dist = Math_VecLen(d);
		if (bestEnemy && dist > (bestEnemyDist+enemyChangeDist))
		{ //not any closer than the current best
			continue;
		}

		//it's visible and closer than anything so far
		Math_VecCopy(c2, bestEnemyCenter);
		bestEnemy = other;
		bestEnemyDist = dist;
	}

	if (bestEnemy)
	{
		Math_VecSub(c1, bestEnemyCenter, d);
		d[2] = 0.0f;
		float v1 = c1[2]+obj->net.maxs[2];
		float v2 = bestEnemyCenter[2]+bestEnemy->net.mins[2];
		if (v1 < v2)
		{
			d[2] += (v2-v1);
		}
		v1 = c1[2]+obj->net.mins[2];
		v2 = bestEnemyCenter[2]+bestEnemy->net.maxs[2];
		if (v2 < v1)
		{
			d[2] += (v1-v2);
		}
		obj->aiObj->distToEnemy = Math_VecLen(d);
	}
	else
	{
		obj->aiObj->distToEnemy = 99999.0f;
	}
	return bestEnemy;
}

//standard enemy-based goal logic
void AI_StandardGoals(gameObject_t *obj, float timeMod)
{
	if (!AI_ValidateEnemy(obj, obj->aiObj->enemy))
	{
		obj->aiObj->enemy = NULL;
	}

	float lookPos[3];
	bool changeLook = false;
	obj->aiObj->enemy = AI_GetEnemy(obj);
	if (obj->aiObj->enemy)
	{
		g_battleTime = g_gameTime+2000;
	}

	if (obj->aiObj->scriptGoal)
	{
		if (obj->aiObj->distToGoal <= obj->aiObj->scriptGoalRange &&
			obj->aiObj->scriptGoal->target)
		{ //chained goals
			obj->aiObj->scriptGoal = obj->aiObj->scriptGoal->target;
		}

		if (!obj->aiObj->scriptGoal->inuse)
		{
			obj->aiObj->scriptGoal = NULL;
		}
		else
		{
			obj->aiObj->goalRange = obj->aiObj->scriptGoalRange;
			obj->aiObj->goalObj = obj->aiObj->scriptGoal;
			Math_VecCopy(obj->aiObj->goalObj->net.pos, obj->aiObj->goalPos);
			if (Math_VecLen(obj->net.vel) > 32.0f)
			{ //look in the direction i'm headed
				Math_VecAdd(obj->net.pos, obj->net.vel, lookPos);
				changeLook = true;
			}
			else
			{
				if (obj->aiObj->scriptLookGoal && obj->aiObj->scriptLookGoal->inuse)
				{
					Math_VecCopy(obj->aiObj->scriptLookGoal->net.pos, lookPos);
				}
				else
				{
					Math_VecCopy(obj->aiObj->goalObj->net.pos, lookPos);
				}
				changeLook = true;
			}
		}
	}
	else if (obj->aiObj->enemy)
	{
		obj->aiObj->goalRange = obj->aiObj->combatRange;
		obj->aiObj->goalObj = obj->aiObj->enemy;
		Math_VecCopy(obj->aiObj->enemy->net.pos, obj->aiObj->goalPos);
		Math_VecCopy(obj->aiObj->goalPos, lookPos);
		changeLook = true;
	}
	else
	{ //if no enemy, just try to seek toward the last enemy/goal pos we had, if it's valid
		if (!(obj->localFlags & LFL_ENEMY))
		{ //have friendly ai try to find the player
			obj->aiObj->goalObj = &g_gameObjects[0];
			obj->aiObj->goalRange = 1024.0f;
			if (Math_VecLen(obj->net.vel) > 32.0f)
			{ //look in the direction i'm headed
				Math_VecAdd(obj->net.pos, obj->net.vel, lookPos);
				changeLook = true;
			}
			else if (obj->aiObj->goalObj->inuse)
			{
				Math_VecCopy(obj->aiObj->goalObj->net.pos, lookPos);
				changeLook = true;
			}
		}
		else
		{
			obj->aiObj->goalRange = obj->aiObj->moveSpeed;
			if (Math_VecLen(obj->net.vel) > 32.0f)
			{ //look in the direction i'm headed
				Math_VecAdd(obj->net.pos, obj->net.vel, lookPos);
				changeLook = true;
			}
		}

		if (obj->aiObj->goalObj && !obj->aiObj->goalObj->inuse)
		{
			obj->aiObj->goalObj = NULL;
		}
		if (obj->aiObj->goalObj)
		{
			Math_VecCopy(obj->aiObj->goalObj->net.pos, obj->aiObj->goalPos);
		}
	}

	if (changeLook)
	{
		float a[3];
		Math_VecSub(lookPos, obj->net.pos, a);
		Math_VecToAngles(a, a);
		obj->aiObj->lookYaw = a[YAW];
		obj->aiObj->lookPitch = (obj->aiObj->fly) ? a[PITCH] : 0.0f;
	}
}
